<copyright> Runtime class.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1995-1998 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: Runtime.t,v 1.71 1998/10/08 12:27:15 tiggr Exp $</id>  </copyright>

<c>
/* You do not want to see this.  Good thing the interface and
   documentation generators skip it.  */
#include <trt/trt.h>
#include <config/config.h>
</c>

<doc> The Runtime class provides an interface to the functionality in the
    runtime library and other process related information.

    Most variables of the {Runtime} class are not {public}, since they can
    be accessed by simply inheriting {Runtime}.  </doc>
implementation class
Runtime: Conditions, Constants, stdio
{
  /******************** general ********************/

  <doc> The name of the host on which this program is running.  </doc>
  static String hostname;

  <doc> The name under which this program was invoked (i.e. the basename
      of argv[0] in C).  </doc>
  public static String program_name;

  <doc> The long program name (i.e. argv[0] in C).  </doc>
  static String long_program_name;

  <doc> The arguments as passed to {main}.  </doc>
  static Array arguments;

  <doc> All the arguments, before any {load} method modified it.  </doc>
  static Array all_arguments;

  <doc> The environment.  </doc>
  static Mapped environment;

  <doc> The directory holding the main resources, at least including the
      character encodings.  </doc>
  private static ByteString main_resource_dir;

  <doc> The dictionary of classes.  Mapping from name to array of classes
      with that name.  Since this is created upon request and it is reset
      by dynamic loading, it is not publicly available.  Access it through
      the {classes_by_name} method.  </doc>
  private static MutableMapped classes_by_name;

  /******************** signals ********************/

  // Maybe the information on signals should reside in a {Signal} class...
  <doc> Iff !0, quit ({SIGINT}, the user interrupt signal) is inhibited.
      </doc>
  private static int quit_inhibit;

  <doc> Iff !0, a {signal-int} will be raised when {quit_inhibit} and
      {panic_mode} again reach 0.  </doc>
  static boolean quit_pending;

  <doc> Iff !0, any signal received (excluding the interrupt signal),
    other condition signaled or raised, or any object thrown will simply
    cause an abort.  This is used to protect critical sections in the
    runtime, such as during garbage collection or object allocation.  If
    {panic_mode} is set, {quit_inhibit} is implicitly set too.  </doc>
  static int panic_mode;

  <doc> Iff {TRUE}, a corefile will be produced on fatal errors, such as
      uncaught condition raises.  </doc>
  static boolean core_on_fatal;

  /******************** garbage collection statistics ********************/

  <doc> The number of objects allocated since the last partial garbage
      collection run.  In this respect, a partial run completing a full
      run is still considered a partial run.  </doc>
  static int gc_alloc_since_partial;

  <doc> The number of objects allocated since the previous completed run.
      This excludes the objects counted by {gc_alloc_since_partial}; it is
      adjusted after a run is initiated, before the run is actually
      started.  </doc>
  static int gc_alloc_since_total;

  <doc> Iff {TRUE} statistics on memory usage and the garbage collector
    will be emitted upon exit.  </doc>
  static boolean gc_stat_at_exit;

  <doc> Iff {TRUE} output statistics on the runtime structures at exit.
      </doc>
  static boolean rt_stat_at_exit;

  <doc> Iff {TRUE} output the number of live instances of each class at
      exit.  </doc>
  static boolean rt_num_inst_at_exit;

  <doc> Statistics on the garbage collector and allocator, in order: the
    number of gc runs; the number of runs which complete a full run; the
    number of object allocated; the number of objects deallocated; the
    (real, i.e. elapsed) time spent protecting, marking, and sweeping; and
    the time spent in all of gc (this is the sum of the previous three,
    plus overhead).  </doc>
  static int gc_num_runs, gc_num_complete;
  static long gc_num_alloc, gc_num_dealloc;
  static double gc_total_protect, gc_total_mark, gc_total_sweep, gc_total_all;

  /******************** malloc statistics ********************/

  <doc> These numbers are only maintained if the runtime library was not
      instructed to not do so at compile time.  </doc>
  static int malloc_cur_bytes, malloc_max_bytes, malloc_cum_bytes;
  static int malloc_cur_items, malloc_max_items, malloc_cum_items;

  /******************** garbage collection parameters ********************/

  <doc> Iff !0, garbage collection won't be run.  This is important
      during, for instance, enumerating a Container, since (most)
      enumerators can not handle the collection changing while they are
      enumerating.  </doc>
  static int gc_inhibit;

  <doc> Iff {TRUE} (the default), garbage collection will run atomically,
      irrespective of the time constraint argument to {garbageCollect}.
      When running with atomic garbage collection, new objects are white
      (presumed dead) whereas with non-atomic garbage collection, new
      objects are gray (presumably alive).

      A program using atomic garbage collection needs less memory, since
      only one run is needed to reclaim a dead object, instead of two runs.
      It also means that, for example, in a multi-threaded program, the
      thread doing garbage collection will block all other threads.  </doc>
  private static boolean gc_atomic;

  <doc> The desired value of {gc_atomic}, which will take effect after the
      next GC run.  Default is whatever the value of {gc_atomic} was at
      startup.  </doc>
  static boolean gc_atomic_next;

  <doc> Iff {TRUE} all garbage will be cleaned upon exit.  This is a
    debugging tool mostly.  </doc>
  static boolean gc_full_at_exit;

  // Output to STDERR is wrong, but the result of the split of the
  // standard environment in two units (tom and too).  Maybe classes
  // needed by the runtime (time, streams, &c) should reside in tom too.
  //
  // The split has been undone, so this can be fixed.
  // Tue May 21 15:55:54 1996, tiggr@tom.es.ele.tue.nl
  <doc> The level of debugging garbage output by the garbage collector.
      Information is output to {stderr} stream provided by the C library.
      No information will be output if {gc_debug} is 0 or if the runtime
      was not compiled with the appropriate flags.  </doc>
  static int gc_debug;

  <doc> Threshold for {gc_alloc_since_partial} before a garbage collection
      run will be initiated.  If {gc_partial_threshold} is 0, garbage
      collection is never run implicitly.  The default value is 25000, or
      the value passed as {:gc-pth} on the command line.  </doc>
  static int gc_partial_threshold;

  <doc> The time allowed for a partial garbage collection run when initiated
      by {gc_alloc_since_partial} exceeding {gc_partial_threshold}.  The
      default is 0, implying no time limit.  </doc>
  static double gc_partial_time_limit;

  <doc> When a partial garbage collection run is initiated and
      {gc_alloc_since_total} exceeds {gc_total_threshold}, the
      {gc_partial_time_limit} is ignored and instead the
      {gc_total_time_limit} is used.  If {gc_total_threshold} is 0, it is
      ignored.  </doc>
  static int gc_total_threshold;

  <doc> The time limit used in case the condition described for
      {gc_total_threshold} applies.  </doc>
  static double gc_total_time_limit;

  /******************** general parameters ********************/

  <doc> Iff {TRUE}, preconditions are checked.  </doc>
  static boolean preconditions_enabled;

  <doc> Iff {TRUE}, postconditions are checked.  </doc>
  static boolean postconditions_enabled;

  <doc> Iff {TRUE}, unhandled signals are printed on {[stdio err]}.  This
      is for debugging purposes.  </doc>
  static boolean rt_print_signals;
}

/********** entry and exit **********/

<doc> This method is invoked by the runtime library.  Its main
    responsibility is to invoke the real main method, which is identified
    by the {sel} and {object}.  </doc>
int
      start (All, selector) (object, sel)
  arguments Array arguments
{
  out = [BufferedStream with out];

  if (rt_print_signals)
    {
      ConditionClass cc = condition;

      bind ((cc, {
		   [self unhandledSignal condition];
		   condition;
		 }))
	[self exit [object perform sel with arguments]];
    }
  else
    [self exit [object perform sel with arguments]];
}

<doc> Normal level exit.  Cleaning up will be performed.  </doc>
void
  exit int rc
{
  [self willExit rc];
  [self fastExit rc];
}

<doc> Low level exit.  Usual functionality for cleaning up is avoided.  </doc>
extern void 
  fastExit int rc;

<doc> Output information on the unhandled signal {condition} on {[stdio
    err]}.  </doc>
void
  unhandledSignal Condition condition
{
  [[err print ("unhandled signal: ", condition)] nl];
}

<doc> Perform all things necessary for a clean exit.  This runs the
    garbage collector if specified by a {:gc-exit}, dumps gc statistics if
    specified by {:gc-stat}, number of instances if specified by
    {:rt-inst}, and memory overhead information if specified by
    {:rt-stat}.  </doc>
void
  willExit int rc
{
  if (gc_full_at_exit)
    {
      if (!gc_atomic && (!!gc_alloc_since_partial || !!gc_alloc_since_total))
	[self garbageCollect];
      [self garbageCollect];
    }

  if (gc_stat_at_exit)
    {
      OutputStream s = [stdio err];

      [[[s print ("gc: runs: ", gc_num_runs)]
	print (" completed: ", gc_num_complete)] nl];
      [[[[s print ("total: alloc: ", gc_num_alloc)]
	 print (" dealloc: ", gc_num_dealloc)]
	print (" remaining: ", gc_num_alloc - gc_num_dealloc)] nl];
      [[[[[s print ("time: ", gc_total_all)]
	  print (" protect: ", gc_total_protect)]
	 print (" mark: ", gc_total_mark)]
	print (" sweep: ", gc_total_sweep)] nl];

      [[[[s print ("malloc (cur, max, cum): (", malloc_cur_items, ", ",
		   malloc_cur_bytes)]
	  print (") (", malloc_max_items, ", ", malloc_max_bytes)]
	 print (") (", malloc_cum_items, ", ", malloc_cum_bytes, ")")] nl];
    }

  if (rt_stat_at_exit)
    [self runtimeStatistics [stdio err]];

  if (rt_num_inst_at_exit)
    [self reportNumInstances [stdio err]];
}

<doc> Output help information about the facilities (most notably `:'
    arguments) offered by the receiving class, on the {OutputStream} s.

    Any implementation should add itself to the set {done}, and check for
    presence before outputting anything, to avoid generating the same
    output for every subclass not overriding this method.  </doc>
OutputStream
  help OutputStream s
  done MutableKeyed done
{
  class (Runtime) cls = [Runtime self];

  if (!done[cls])
    {
      [done add cls];

      s = [[s print "tom.Runtime
 control options:
  :cc-pre               enable precondition checking
  :cc-post              enable postcondition checking
  :gc-pth <n>           set allocation threshold for a partial gc run to <n>
  :gc-ptl <n>           set gc partial run time limit to <n> milliseconds
                        (induces non-atomic garbage collection.)
  :rt-resource-dir <d>  denote the main resource directory (for charmaps)
  :rt-core              dump core on fatal errors (gdb: breakpoint in abort; bt)
 verbosity options:
  :gc-debug <n>	        output gc debugging info (level = <n>) to [stdio err]
  :gc-exit              perform a full gc run (possibly 2) upon exit
  :gc-stat              output allocation and gc statistics upon exit
  :rt-inst              output number of live instances of every class upon exit
  :rt-signals           trace signaled conditions that are not handled
  :rt-stat              output runtime structure statistics upon exit"] nl];
    }
  = s;
}

<doc> Scan the arguments to the program for something telling us whether
    or how to do certain things.

    See the output of {:help} of any TOM program for short information on
    the options.  </doc>
void
  load MutableArray arguments
{
  boolean desired_gc_atomic = TRUE;
  int i, n = [arguments length];

  gc_partial_threshold = 25000;

  for (i = 0; i < n;)
    {
      ByteString a = arguments[i];
      String s;

      if ([a length] > 0 && a[0] == ':')
	{
	  int hit;

	  if ([":cc-pre" equal a])
	    {
	      preconditions_enabled = TRUE;
	      hit = 1;
	    }
	  else if ([":cc-post" equal a])
	    {
	      postconditions_enabled = TRUE;
	      hit = 1;
	    }
	  else if ([":gc-debug" equal a])
	    {
	      gc_debug = [arguments[i + 1] unsignedIntValue];
	      hit = 2;
	    }
	  else if ([":gc-pth" equal a] && i < n - 1)
	    {
	      gc_partial_threshold = [arguments[i + 1] unsignedIntValue];
	      hit = 2;
	    }
	  else if ([":gc-ptl" equal a] && i < n - 1)
	    {
	      gc_partial_time_limit
		= 0.001 * double ([arguments[i + 1] unsignedIntValue]);
	      desired_gc_atomic = FALSE;
	      hit = 2;
	    }
	  else if ([":gc-exit" equal a])
	    {
	      gc_full_at_exit = TRUE;
	      hit = 1;
	    }
	  else if ([":gc-stat" equal a])
	    {
	      gc_stat_at_exit = TRUE;
	      hit = 1;
	    }
	  else if ([":rt-core" equal a])
	    {
	      core_on_fatal = YES;
	      hit = 1;
	    }
	  else if ([":rt-inst" equal a])
	    {
	      rt_num_inst_at_exit = YES;
	      hit = 1;
	    }
	  else if ([":rt-signals" equal a])
	    {
	      rt_print_signals = TRUE;
	      hit = 1;
	    }
	  else if ([":rt-stat" equal a])
	    {
	      rt_stat_at_exit = TRUE;
	      hit = 1;
	    }

	  if (hit != 0)
	    {
	      [arguments removeElements (i, hit)];
	      n -= hit;
	    }
	  else
	    i++;
	}
      else
	i++;
    }

  gc_atomic = gc_atomic_next = desired_gc_atomic;
}

<doc> Invoked by the runtime library before the first load is invoked.

    This method is needed for two occasions: first is to check for
    {:help}.  The reason this is not done in {load} is to be able to get
    some help before any negative side effects of any {load} method.  The
    second reason is for finding {:rt-resource-dir}, which must be done
    before {ByteString}'s {load} method can play with its encoding.
    </doc>
void
  preload MutableArray arguments
{
  int i, n = [arguments length];

  main_resource_dir = TOM_RESOURCES;

  /* Get a short name, `c', for the ConditionClass.  */
  class (ConditionClass) c = [ConditionClass self];

  condition = [c with nil name "condition"];
  warning = [c with condition name "warning"];
  unimplemented = [c with condition name "unimplemented"];
  encoding-condition = [c with condition name "encoding-condition"];
  serious-condition = [c with condition name "serious-condition"];
  runtime-condition = [c with serious-condition name "runtime-condition"];
  runtime-fatal = [c with runtime-condition name "runtime-fatal"];
  nil-receiver = [c with runtime-condition name "nil-receiver"];
  unrecognized-selector = [c with runtime-condition
			     name "unrecognized-selector"];
  uncaught-throw = [c with runtime-condition name "uncaught-throw"];
  program-condition = [c with serious-condition name "program-condition"];
  unknown-class-condition = [c with program-condition
			       name "unknown-class-condition"];
  coding-condition = [c with program-condition name "coding-condition"];
  lock-condition = [c with program-condition name "lock-condition"];
  condition-condition = [c with program-condition name "condition-condition"];
  type-condition = [c with program-condition name "type-condition"];
  error = [c with serious-condition name "error"];
  file-error = [c with error name "file-error"];
  stream-error = [c with error name "stream-error"];
  stream-eos = [c with error name "stream-eos"];
  signal-condition = [c with condition name "signal-condition"];
  signal-hup = [c with signal-condition name "signal-hup"];
  signal-int = [c with signal-condition name "signal-int"];
  signal-bus = [c with signal-condition name "signal-bus"];
  signal-segv = [c with signal-condition name "signal-segv"];

  while (i < n)
    {
      String a = arguments[i];

      if ([a equal ":help"])
	{
	  /* This is ugly, but needed to get some performance for
             outputting the help information.  */
	  [BufferedStream load nil];

	  OutputStream s = [BufferedStream with [stdio out]];
	  Enumerator e = [[self classes] enumerator];
	  MutableEqSet done = [MutableEqSet new];
	  class (State) c;
	  boolean b;

	  /* This is extremely expensive since, when dynamically resolved,
	     this will build the dispatch tables of all class objects...  */
	  while ({(b, c) = [e next]; b;})
	    if ([c respondsTo selector (OutputStream help OutputStream
					done MutableKeyed)])
	      s = [c perform selector (OutputStream help OutputStream
				       done MutableKeyed)
		     with (s, done)];

	  [self fastExit 0];
	}
      else if ([a equal ":rt-resource-dir"])
	{
	  main_resource_dir = arguments[i + 1];
	  [arguments removeElements (i, 2)];
	  n -= 2;
	}
      else
	i++;
    }
}

<doc> Output to the stream {s} the number of live instances for each
    class.  If the optional {zeroes} is {TRUE}, classes with zero
    instances are included in the report.  This includes deferred classes,
    as they cannot have any instances.  </doc>
void
  reportNumInstances OutputStream s
      includeZeroes: boolean zeroes = FALSE
{
  Indexed all_classes = [self classes];
  int i, n = [all_classes length];

  for (i = 0; i < n; i++)
    {
      class (State) cls = all_classes[i];
      int num = [cls num_instances];

      if (num != 0 || zeroes)
	[[s print ([[cls unit] name], '.', [cls name], '\t', num)] nl];
    }
}

<doc> Output the actual statistics for {:rt-stat} to the stream {s}.
    </doc>
protected extern void
  runtimeStatistics OutputStream s;

/********** classes **********/
<doc> <h4>Classes</h4>  </doc>

<doc> Return the array of all class objects.  </doc>
extern Indexed
  classes;

<doc> Return the, possibly created upon request, mapped collection of
    classes keyed on their name.  </doc>
Mapped
  classes_by_name
{
  if (!classes_by_name)
    {
      /* This is very expensive, since it messages every class in
	 existence, causing the dispatch table to be installed for
	 every class object...  */
      Enumerator e = [[self classes] enumerator];
      class (State) class_object;
      boolean b;

      classes_by_name = [MutableDictionary new];
      while ({(b, class_object) = [e next]; b;})
	{
	  String n = [class_object name];
	  MutableArray a = classes_by_name[n];

	  if (!a)
	    {
	      a = [MutableObjectArray withCapacity 1];
	      classes_by_name[n] = a;
	    }

	  [a add class_object];
	}
    }

  = classes_by_name;
}

<doc> Return the class with the {name}.

    Name may be unqualified, as in {"Runtime"}, which will return the
    single class with that name, or {nil} in case such a class does not
    exist, or if more than one class with that name exists in multiple
    units.

    The {name} may be qualified, as in {"tom.Runtime"}, in which case the
    {Runtime} class of the {tom} unit will be returned, if that unit and
    class within that unit exist.  </doc>
class (State) (class_object)
  classNamed String name
{
  Array un = [name componentsSeparatedBy '.' limit: 2];

  if ([un length] == 2)
    {
      Unit unit = [Unit named un[0]];

      if (unit != nil)
	class_object = [unit classNamed un[1]];
    }
  else
    {
      if (!classes_by_name)
	[self classes_by_name];

      Array found = classes_by_name[name];
      class_object = !found || [found length] != 1 ? nil : found[0];
    }
}

/********** selectors **********/
<doc> <h4>Selectors</h4>  </doc>

<doc> Return the existing selector known by the {name}.  </doc>
extern selector
  selectorNamed String name;

// CCC This should be functionality offered by the compiler for the `=='
// operator.
// Tue Jan 21 22:14:29 1997, tiggr@tricky.es.ele.tue.nl
<doc> Return {TRUE} iff the selectors {s1} and {s2} denote the same
    selector.  </doc>
extern boolean
  selector selector s1
    equals selector s2;

<doc> Return the name of the {selector}.  </doc>
extern String
  nameOfSelector selector sel;

// Shouldn't this be expressible within the language?
// Mon Mar 31 20:08:44 1997, tiggr@tricky.es.ele.tue.nl
<doc> Return the invalid selector.  </doc>
extern selector
  nullSelector;

/********** garbage collection **********/
<doc> <h4>Garbage collection</h4>  </doc>

<doc> Run the garbage collector to the end of a full garbage collection
    run.  </doc>
void
  garbageCollect
{
<c>
  tgc_collect_garbage (0);
</c>
}

<doc> Run the garbage collector for at most {time} seconds.  </doc>
void
  garbageCollect double time
pre
  gc_atomic -> !time
{
<c>
  tgc_collect_garbage (time);
</c>
}

<doc> Increase the {gc_inhibit}.  This invocation should be matched by an
      invocation of {enableGC}.  </doc>
void
  disableGC
pre
  gc_inhibit >= 0
post
  gc_inhibit == old gc_inhibit + 1
{
  gc_inhibit += 1;
}

<doc> Decrease the {gc_inhibit}.  </doc>
void
  enableGC
pre
  gc_inhibit > 0
post
  gc_inhibit == old gc_inhibit - 1
{
  gc_inhibit -= 1;
}

/********** process information **********/
<doc> <h4>Process information</h4>  </doc>

<doc> Return the dictionary holding the process environment.  The
    dictionary is filled upon the first request, thread-safely.  </doc>
extern Mapped
  environment;

<doc> Return the {main_resource_dir}.  </doc>
ByteString
  main_resource_dir
{
  = main_resource_dir;
}

<doc> Set the {value} of the {environment_variable}, thread-safely.  </doc>
extern void
  setenv (String, String) (environment_variable, value);

<doc> Return the {hostname} of this machine.  If the class variable is not
    set, it is set once from gethostname(2).  </doc>
extern String
  hostname;

<doc> Return the directory in which all TOM stuff has been installed.
    This returns the value of {TOM_PREFIX} in the {Constants} class.
    </doc>
String
  tom_prefix
{
  = TOM_PREFIX;
}

/********** signals **********/
<doc> <h4>Signal and condition handler</h4>  </doc>

<doc> Construct a {Condition} for the {object} with the {condition_class}
    and a message created from the (optional) prefix, plus the information
    available from the (ANSI C) {errno} variable.  If {not_signal} is
    {TRUE}, the new condition is raised; otherwise it is signaled and the
    result is returned (if a return is allowed).  </doc>
Any
  perror String prefix
     for All object
   class ConditionClass condition_class
   raise boolean not_signal
{
  String suffix;

  // This should not be here.  In fact the whole error reporting should be
  // in a separate interface unit, with this code in a (UNIX, machine
  // dependent) implementation thereof.  Ahem, with variants for
  // single-threaded v. multi-threaded libraries...
  // Fri Dec  5 13:18:20 1997, tiggr@natlab.research.philips.com
<c>
  suffix = byte_string_with_c_string (strerror (errno));
</c>

  MutableByteString message = [MutableByteString new];

  if (prefix != nil)
    [message print (prefix, ": ")];
  [message print suffix];
  Condition c = [Condition for object class condition_class message message];
  if (not_signal)
    [c raise];
  else
    = [c signal];
}

<doc> Accessor method for {quit_inhibit} which is private to the {Runtime}
    class to protect it against being mutated by subclasses but which can
    be freely read, hence this method.  </doc>
int
  quit_inhibit
{
  = quit_inhibit;
}

<doc> Increase the {quit_inhibit} flag.  Any increase should be
    accompanied later on by the corresponding decrease.  </doc>
void
  quit_disable
{
  quit_inhibit++;
}

<doc> Decrease the {quit_inhibit} flag, raising a postponsed {signal-int}
    if indicated by {quit_pending}.  </doc>
void
  quit_enable
pre
  quit_inhibit > 0
{
  if (!--quit_inhibit && quit_pending)
    {
      quit_pending = FALSE;
      [[Condition for self class signal-int
		  message "postponed interrupt"] raise];
    }
}

<doc> Accessor method for {panic_mode} which is private to the {Runtime}
    class to protect it against being mutated by subclasses but which can
    be freely read, hence this method.  </doc>
int
  panic_mode
{
  = panic_mode;
}

<doc> Increase the {panic_mode} flag.  Any increase should be accompanied
    later on by the corresponding decrease.  </doc>
void
  panic_enable
{
  panic_mode++;
}

<doc> Decrease the {panic_mode} flag, raising a {signal-int} if requested
    by {quit_pending}.  </doc>
void
  panic_disable
pre
  panic_mode > 0
{
  if (!--panic_mode && quit_pending && !quit_inhibit)
    {
      quit_pending = FALSE;
      [[Condition for self class signal-int
		  message "postponed interrupt"] raise];
    }
}

end;

<doc> The {Runtime} instance is totally empty.  </doc>
implementation instance Runtime end;
