<copyright> Trie class.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1996, 1997 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: Trie.t,v 1.9 1998/01/05 01:07:22 tiggr Exp $</id>
    </copyright>

<doc> The {Trie} is a class providing the mechanism to store information
    in a trie on {char} strings.  It does not by itself store any
    information, subclasses should be created to hold the information.
    The {Trie} is accompanied by its subclass {ObjectTrie} which can store
    objects in a trie.  </doc>
implementation class
Trie: State,
	Constants

end;

implementation instance
Trie
{
  <doc> The offset to the first element in {next}, i.e. the element with
      numeric value {start} resides at index 0 in {next}.  </doc>
  int start;

  <doc> The value of the first element beyond the last element in {next}.
      </doc>
  int beyond;

  <doc> If {start == beyond}, this is the suffix {String} which leads up
      to the value this node holds (if any).  Otherwise, {start > beyond}
      and this is a {MutableObjectArray} pointing to the next nodes, and
      which is to be indexed with offset {start}.  </doc>
  Any next;
}

<doc> Return {YES} iff we can hold a value, i.e. if we do not yet hold a
    value.  </doc>
deferred boolean
  isEmpty;

<doc> Create the node for that part of the string {str} starting at {s},
    and ending at {e}.  The {TRIE_LOOKUP_PREFIX} option is ignored.  If
    this node already exists, it is returned.

    When the {options} include {TRIE_FOLD_CASE}, the {str} is inserted in
    lower case.  </doc>
protected id
  createNode String str
       start int s
         end int e
     options int options
{
  /* If we have a suffix, push it one level further.  */
  if (start == beyond && next != nil)
    [self pushSuffix options];

  /* If the remains of the string are empty, we're it.  */
  if (s == e)
    return self;

  /* If we're empty, we can store the suffix in us.  */
  if (start == beyond && [self isEmpty])
    {
      String suffix = [str substring (s, e - s)];
      if (options & TRIE_FOLD_CASE != 0)
	suffix = [suffix downcase];

      next = Any (suffix);
      return self;
    }

  /* We can't handle this with a suffix.  Face it like a trie.  */
  char c = options & TRIE_REVERSED != 0 ? str[--e] : str[s++];
  MutableObjectArray n = next;

  if (options & TRIE_FOLD_CASE != 0 && [str isUpper c])
    c = [str toLower c];
    
  if (start == beyond)
    {
      start = c;
      beyond = int (c) + 1;
      n = [MutableObjectArray withCapacity 1];
      next = Any (n);
      [next resize 1];
    }
  else if (c >= beyond)
    {
      beyond = int (c) + 1;
      [n resize beyond - start];
    }
  else if (c < start)
    {
      [n resize (0, start - c)];
      start = c;
    }

  id next_node = n[c - start];
  if (!next_node)
    n[c - start] = next_node = [isa new];

  = [next_node createNode str start s end e options options];
}

<doc> Find the node for that part of the string {str} starting at {s}, and
    ending at {e}.  Iff a prefix match is desired, the node returned is
    the longest prefix match.  </doc>
protected id
  findNode String str
     start int s
       end int e
   options int options
{
  id node;

  /* If we have a suffix, we're supposed to be at the end of that string.  */
  if (start == beyond && next != nil)
    if (s == e)
      {
	/* We can't match if we're here for a non-empty suffix.  */
	return nil;
      }
    else
      {
	String suffix = next;
	int i, j, sl = [suffix length];

	if (options & TRIE_LOOKUP_PREFIX != 0
	    ? e - s < sl : e - s != sl)
	  return nil;

	for ({i = 0; j = options & TRIE_REVERSED != 0 ? e - sl : s;};
	     i < sl; {i++; j++;})
	  {
	    char c1 = str[j], c2 = suffix[i];

	    if (c1 != c2 && !(options & TRIE_FOLD_CASE != 0
			      && [str toLower c2] == [str toLower c1]))
	      return nil;
	  }

	return ![self isEmpty] ? self : nil;
      }
  else if (s == e)
    return self;
  else
    {
      char c = options & TRIE_REVERSED != 0 ? str[--e] : str[s++];
      if (options & TRIE_FOLD_CASE != 0 && [str isUpper c])
	c = [str toLower c];

      if (c >= start && c < beyond)
	{
	  MutableObjectArray n = next;
	  id node = n[c - start];

	  if (node != nil)
	    {
	      node = [node findNode str start s end e options options];
	      if (node != nil)
		return node;
	    }
	}
    }

  = options & TRIE_LOOKUP_PREFIX != 0 && ![self isEmpty] ? self : nil;
}

<doc> Push our suffix one node down.  </doc>
protected void
  pushSuffix int options
{
  String str = next;
  int s = 0, e = [str length];
  char c = options & TRIE_REVERSED != 0 ? str[--e] : str[s++];

  start = c;
  beyond = c + 1;

  MutableObjectArray n = [MutableObjectArray withCapacity 1];
  id next_node = [isa new];
  n[0] = next_node;
  next = Any (n);

  [next_node createNode str start s end e options options];
}

OutputStream
  write OutputStream s
{
  s = [[[[s print ('(', [isa name])] print (' ', start)]
	 print (' ', beyond)] print (' ', next)];
  s = [self writeValue s];
  = [s print ')'];
}

deferred OutputStream
  writeValue OutputStream s;

end;

<doc> An {ObjectTrie} is a {Trie} which can hold an object.  </doc>
implementation class
ObjectTrie: Trie

end;

implementation instance
ObjectTrie
{
  <doc> Our value.  </doc>
  public mutable Any value;
}

boolean
  isEmpty
{
  = !value;
}

<doc> Move our value with the suffix.  </doc>
void
  pushSuffix int options
{
  [super pushSuffix options];

  MutableObjectArray n = next;
  [n[0] set_value value];
  value = nil;
}

Any
       at String key
  options int options
{
  id node = [self findNode key start 0 end [key length] options options];

  = !node ? nil : [node value];
}

void
      set All object
       at String key
  options int options
{
  id node = [self createNode key start 0 end [key length] options options];

  [node set_value Any (object)];
}

OutputStream
  writeValue OutputStream s
{
  = [s print (' ', value)];
}

/******************** MutableMapped ********************/
<doc> <h4>MutableMapped collection</h4> </doc>

Any
  at String key
{
  = [self at key options TRIE_PLAIN];
}

void
  set All object
   at String key
{
  [self set object at key options TRIE_PLAIN];
}

end;
