/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: OTMBasic.m,v 1.47 1999/06/13 11:46:59 tiggr Exp $  */

#import "OTMBasic.h"
#import "OTMClass.h"
#import "OTMExtension.h"
#import "OTMInstance.h"
#import "OTMMeta.h"
#import "OTMMethod.h"
#import "OTMType.h"
#import "global.h"

OTMBasic *basic_type[BT_NUM];

static id <TLString> basic_name[] =
{
  @"void",
  @"boolean",
  @"byte",
  @"char",
  @"int",
  @"long",
  @"float",
  @"double",
  @"pointer",
  @"selector",
  @"id",
  @"class (id)",
  @"instance (id)",
};

static id <TLString> basic_te[] =
{
  @"TRT_TE_VOID",
  @"TRT_TE_BOOLEAN",
  @"TRT_TE_BYTE",
  @"TRT_TE_CHAR",
  @"TRT_TE_INT",
  @"TRT_TE_LONG",
  @"TRT_TE_FLOAT",
  @"TRT_TE_DOUBLE",
  @"TRT_TE_POINTER",
  @"TRT_TE_SELECTOR",
  @"TRT_TE_REFERENCE",
  @"TRT_TE_REFERENCE",
  @"TRT_TE_REFERENCE",
};

static const char *basic_output[] =
{
  "void",
  "tom_byte",
  "tom_byte",
  "tom_char",
  "tom_int",
  "tom_long",
  "tom_float",
  "tom_double",
  "void *",
  "selector",
  "xxx tom_object",
  "xxx tom_object",
  "xxx tom_object",
};

static id <TLString> basic_frobnicated[] =
{
  @"v",
  @"o",
  @"b",
  @"c",
  @"i",
  @"l",
  @"f",
  @"d",
  @"p",
  @"s",
  @"r",
  @"r",
  @"r",
};

char basic_cast[BT_NUM][BT_NUM] =
{
	/* v   b   b   c   i   l   f   d   p   s   r   rc  ri */
  /* v */ {1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
  /* b */ {0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0},
  /* b */ {0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0},
  /* c */ {0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0},
  /* i */ {0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0},
  /* l */ {0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0},
  /* f */ {0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0},
  /* d */ {0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0},
  /* p */ {0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0},
  /* s */ {0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0},
  /* r */ {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0},
 /* rc */ {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0},
 /* ri */ {0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1},
};

@implementation OTMBasic

+(void) init;
{
  int i;

  for (i = BT_VOID; i < BT_NUM; i++)
    {
      basic_type[i] = [[self alloc] initWithName: basic_name[i] type: i];
      [basic_type[i] gcLock];
    }
}

-(id) actualSelf: (OTMMeta *) context
{
  switch (type)
    {
    case BT_RECV:
      return context;
    case BT_RECV_CLASS:
      return context ? [[[context structure] itsClass] semantics] : nil;
    case BT_RECV_INST:
      return context ? [[[context structure] instance] semantics] : nil;
    default:
      return self;
    }
}

-(id) actualSelfNonPosing: (id) context
{
  return [self actualSelf: context];
}

-(BOOL) allowedTypeForArgumentRedeclaration: (OTMType *) t
				 inSubclass: (BOOL) subclass_p
{
  return t == self;
}

-(enum basic_type) basicKind
{
  return type;
}

-itsClass
{
  if (self == basic_type[BT_RECV])
    return basic_type[BT_RECV_CLASS];
  if (self == basic_type[BT_RECV_INST])
    return basic_type[BT_RECV];

  if (self == basic_type[BT_RECV_CLASS])
    /* Well...  */
    error (@"nonsense to ask for a class' class");
  else
    error (@"basic types (`%@') do not have a class object",
	   [self typeName]);

  return self;
}

-(BOOL) declaredp
{
  return YES;
}

-(BOOL) definedp
{
  return YES;
}

-(void) description: (id <TLMutableStream>) stream
{
  [super description: stream];

  if (basic_type[type] == self)
    [@" unique" print: stream quoted: NO];
  formac (stream, @" %@", basic_name[type]);
}

-(id <TLString>) flatFrobnicatedName
{
  return type == BT_VOID ? @"" : basic_frobnicated[type];
}

-(id <TLString>) frobnicatedName
{
  return basic_frobnicated[type];
}

-initWithName: (TLString *) n type: (enum basic_type) t
{
  [super initWithName: n];

  type = t;

  return self;
}

-instance
{
  if (self == basic_type[BT_RECV])
    return basic_type[BT_RECV_INST];
  if (self == basic_type[BT_RECV_CLASS])
    return basic_type[BT_RECV];

  if (self == basic_type[BT_RECV_INST])
    error (@"an instance does not define a class of objects");
  else
    error (@"a basic type instance (%@) does not define a class of objects",
	   [self typeName]);

  return self;
}

-(BOOL) isObjectType
{
  return type >= BT_RECV;
}

-(BOOL) matches: (OTMType *) t
{
  if (type < BT_RECV)
    {
      if (self == (id) t
	  || ([t isKindOf: isa] && [(OTMBasic *) t basicKind] == type))
	return YES;
    }
  else if ([t isObjectType])
    return YES;

  return NO;
}

-(int) matchesConvertibly: (OTMType *) t
{
  BOOL sk, identical;

  t = [t tupleSingleElement];

  sk = [(OTMBasic *) t isKindOf: isa];
  identical = self == (id) t || (sk && [(OTMBasic *) t basicKind] == type);

  if (identical)
    return 0;

  if (type >= BT_RECV && [t isObjectType])
    return 0;

  if (sk && (type > BT_BOOLEAN && type < BT_SELECTOR))
    {
      enum basic_type ot = [(OTMBasic *) t basicKind];

      if (ot > BT_BOOLEAN && ot < BT_SELECTOR)
	{
	  if ((type >= BT_BYTE && type <= BT_LONG
	       && ot >= BT_BYTE && ot <= BT_LONG)
	      || (type >= BT_FLOAT && type <= BT_DOUBLE
		  && ot >= BT_FLOAT && ot <= BT_DOUBLE))
	    {
	      if (type >= ot)
		return type - ot;
	      else
		{
		  /* Do not allow narrowing.  */
		  return -1;
		}
	    }
	}
    }

  return -1;
}

-(TLCons *) methodsNamed: (TLVector *) name_parts
		  sender: (OTMMeta *) sender
		   super: (BOOL) super_p
		confined: (OTMMeta *) confined
{
  error (@"basic types can not be messaged");
  return nil;
}

-(int) minimumAlignment
{
  switch (type)
    {
    case BT_BOOLEAN:
    case BT_BYTE:
      return 8;
    case BT_CHAR:
      return 16;
    case BT_INT:
    case BT_FLOAT:
      return 32;
    case BT_LONG:
    case BT_DOUBLE:
      return 64;
    case BT_VOID:
      abort ();
    default:
      return 8 * sizeof (void *);
    }
}

-(id <TLString>) outputFunctionTypeForType
{
  if (type < BT_RECV)
    return (id) formac (nil, @"%@_imp", basic_name[type]);

  return @"reference_imp";
}

-(void) compileDeclaration
{
  if (type >= BT_RECV)
    {
      LTTMeta *m = output_current_context;

      if (type == BT_RECV_CLASS)
	m = [m itsClass];
      else if (type == BT_RECV_INST)
	m = [m instance];

      [[m semantics] compileDeclaration];
    }
}

-(id <TLString>) outputTypeName
{
  if (type < BT_RECV)
    return TLSTRING (basic_output[type]);

  {
    LTTMeta *m = output_current_context;

    if (type == BT_RECV_CLASS)
      m = [m itsClass];
    else if (type == BT_RECV_INST)
      m = [m instance];

    return [m outputTypeName];
  }
}

-(id <TLString>) outputTypeEncoding
{
  return basic_te[type];
}

-(id <TLString>) typeInfo
{
  return name;
}

-(BOOL) validCastTo: (OTMType *) t
{
  OTMBasic *b = (id) t;

  return (b == self
	  || ([b isKindOf: isa] && basic_cast[type][[b basicKind]]));
}

@end
