/* C routines for the too unit.
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: glue.c,v 1.22 1998/05/07 21:20:16 tiggr Exp $  */

#include "too-r.h"
#include <trt/trt.h>

#if HAVE_LIBC_H
#include <libc.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <netdb.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>

/* Return the ADDRESS and the PORT according to the SIN.  */
static void
address_and_port (struct sockaddr_in *sin, tom_object *address, tom_int *port)
{
  tom_object host = sin->sin_addr.s_addr ? 0 : c_too_InetHost_local_host_any;

  *address = TRT_SEND ((tom_object (*) (void *, selector,
					tom_object, void *, tom_int)),
		       _mr_c_too_InetAddress, SEL (r_with__rpi_),
		       host, memcpy (xmalloc (sizeof (sin->sin_addr)),
				     &sin->sin_addr, sizeof (sin->sin_addr)),
		       sizeof (sin->sin_addr));
  *port = ntohs (sin->sin_port);
}

/* Return the ADDRESS and the PORT of the socket FD.  Return 0 on success.  */
static int
get_socket_name (int fd, tom_object *address, tom_int *port)
{
  struct sockaddr_in sin;
  int l = sizeof (sin);

  if (getsockname (fd, (void *) &sin, &l))
    return -1;

  address_and_port (&sin, address, port);

  return 0;
}

/* Return the ADDRESS and the PORT of the socket peer of the FD.  Return 0
   on success.  */
static int
get_peer_name (int fd, tom_object *address, tom_int *port)
{
  struct sockaddr_in sin;
  int l = sizeof (sin);

  if (getpeername (fd, (void *) &sin, &l))
    return -1;

  address_and_port (&sin, address, port);

  return 0;
}

/* Return a newly created InetHost (from SELF, obviously) given the
   hostent HE.  */
static tom_object
host_with_host_ent (tom_object self, struct hostent *he)
{
  tom_object host, address, addresses, name, names;
  tom_int n_addr, n_name;
  int i;

  /* Create and fill the array of names.  */
  for (n_name = 0; he->h_aliases[n_name]; n_name++);
  names = TRT_SEND ((reference_imp), _mr_c_tom_MutableObjectArray,
		    SEL (r_withCapacity_i), n_name + 1);
  TRT_SEND (, names, SEL (v_add_r), byte_string_with_c_string (he->h_name));
  for (i = 0; i < n_name; i++)
    {
      name = byte_string_with_c_string (he->h_aliases[i]);
      TRT_SEND (, names, SEL (v_add_r), name);
    }

  /* Create the array which is to hold the addresses.  */
  for (n_addr = 0; he->h_addr_list[n_addr]; n_addr++);
  addresses = TRT_SEND ((reference_imp), _mr_c_tom_MutableObjectArray,
			SEL (r_withCapacity_i), n_addr);

  /* Make the host.  */
  host = TRT_SEND ((reference_imp), self, SEL (r_with__rr_),
		   names, addresses);

  /* Create the addresses and add them to the host (which is already
     initialized... not too clean...).  */
  for (i = 0; i < n_addr; i++)
    {
      address = TRT_SEND ((reference_imp), _mr_c_too_InetAddress,
			  SEL (r_with__rpi_), host,
			  memcpy (xmalloc (he->h_length),
				  he->h_addr_list[i], he->h_length),
			  he->h_length);
      TRT_SEND (, addresses, SEL (v_add_r), address);
    }

  return host;
}

void
i_too_InetAddress_v_dealloc (tom_object self, selector cmd)
{
  struct _es_i_too_InetAddress *this;

  this = trt_ext_address (self, _ei_i_too_InetAddress);

  xfree (this->address);
}

tom_object
i_too_InetAddress_r_write_r (tom_object self, selector cmd, tom_object out)
{
  struct _es_i_too_InetAddress *this;
  int i;

  this = trt_ext_address (self, _ei_i_too_InetAddress);
  for (i = 0; i < this->address_length; i++)
    {
      if (i)
	TRT_SEND (, out, USEL (tom, r_print_b), '.');
      TRT_SEND (, out, USEL (tom, r_print_i), ((tom_byte *) this->address)[i]);
    }

  return out;
}

tom_byte
i_too_InetAddress_o_equal_r (tom_object self, selector cmd, tom_object other)
{
  if (other != self)
    {
      struct _es_i_too_InetAddress *this;
      tom_byte *oa;
      tom_int ol;
      int i;

      this = trt_ext_address (self, _ei_i_too_InetAddress);
      oa = TRT_SEND ((pointer_imp), other, SEL (_pi__osAddress), &ol);

      if (ol != this->address_length)
	return 0;

      for (i = 0; i < ol; i++)
	if (((tom_byte *) this->address)[i] != oa[i])
	  return 0;
    }

  return 1;
}

tom_int
i_too_InetAddress_i_hash (tom_object self, selector cmd)
{
  struct _es_i_too_InetAddress *this;
  tom_int h;
  int i;

  this = trt_ext_address (self, _ei_i_too_InetAddress);
  for (i = h = 0; i < this->address_length; i++)
    h = (h << 3) ^ ((tom_byte *) this->address)[i];

  return h;
}

tom_object
c_too_InetHost_r_hostWithAddress_r (tom_object self, selector cmd,
				    tom_object addr)
{
  struct hostent *he;
  tom_byte *oa;
  tom_int ol;

  /* XXX This is synchronous.  And only works for the InetAddress class
     for ADDR.  */
  oa = TRT_SEND ((pointer_imp), addr, SEL (_pi__osAddress), &ol);
  he = gethostbyaddr (oa, ol, AF_INET);

  return he ? host_with_host_ent (self, he) : 0;
}

tom_object
c_too_InetHost_r_hostWithName_r (tom_object self, selector cmd, tom_object name)
{
  struct hostent *he;
  int len;
  char *s;

  s = TRT_SEND ((pointer_imp), name, USEL (tom, _pi__byteStringContents), &len);
  s = memcpy (alloca (1 + len), s, len);
  s[len] = 0;

  /* XXX This is synchronous.  */
  he = gethostbyname (s);

  return he ? host_with_host_ent (self, he) : 0;
}

tom_object
i_too_ConnectedInetPort_r_initWithPort_i_at_r (tom_object self, selector cmd,
					       tom_int port, tom_object address)
{
  struct _es_i_too_ConnectedInetPort *cip;
  int fd, fail = 0;

  /* XXX Should do getprotobyname...  */
  fd = socket (AF_INET, SOCK_STREAM, 6);
  if (fd == -1)
    fail = 1;
  else
    {
      struct sockaddr_in sin;
      tom_byte *aa;
      tom_int al;

      aa = TRT_SEND ((pointer_imp), address, SEL (_pi__osAddress), &al);

      sin.sin_family = AF_INET;
      sin.sin_port = htons (port);
      
      if (al != sizeof (struct in_addr))
	ABORT ();

      memcpy (&sin.sin_addr, aa, al);

      if (connect (fd, (void *) &sin, sizeof (sin)))
	fail = 1;
    }

  cip = trt_ext_address (self, _ei_i_too_ConnectedInetPort);
  cip->server = (void *) TRT_SEND ((reference_imp), _mr_c_too_InetPort,
				   SEL (r_with_i_at_r), port, address);

  if (fail || get_socket_name (fd, &address, &port))
    {
      if (fd != -1)
	close (fd);
      TRT_RAISE (self, c_tom_Conditions_error,
		 byte_string_with_c_string (OS_ERROR_MSG));
    }
  else
    {
      struct _es_i_too_InetPort *ip;

      /* XXX Should super's init.  */
      ip = trt_ext_address (self, _ei_i_too_InetPort);

      /* XXX This is in an init method, so we should be safe by assuming
         we're gray to the garbage collector.  (This should be checked
         throughout.)  */
      ip->address = (void *) address;
      ip->port = port;

      /* XXX Should send super.  */
      TRT_SEND ((reference_imp), self, SEL (r_init_i), fd);
    }

  return self;
}

tom_object
i_too_ConnectedInetPort_r_peer (tom_object self, selector cmd)
{
  struct _es_i_too_ConnectedInetPort *cip;

  cip = trt_ext_address (self, _ei_i_too_ConnectedInetPort);
  if (!cip->peer)
    {
      struct _es_i_tom_Descriptor *bs;
      tom_object address;
      tom_int port;

      bs = trt_ext_address (self, _ei_i_tom_Descriptor);
      if (get_peer_name (bs->descriptor, &address, &port))
	TRT_RAISE (self, c_tom_Conditions_error,
		   byte_string_with_c_string (OS_ERROR_MSG));
      cip->peer = trt_assign_object_var
	(self, (void *) TRT_SEND ((reference_imp), _mr_c_too_InetPort,
				  SEL (r_with_i_at_r), port, address));
    }

  return (void *) cip->peer;
}

tom_object
i_too_ServerInetPort_r_accept (tom_object self, selector cmd)
{
  tom_object address, address_self, peer, r;
  struct _es_i_tom_Descriptor *sip;
  tom_int port, port_self;
  struct sockaddr_in sin;
  int len = sizeof (sin);
  tom_int fd;

  sip = trt_ext_address (self, _ei_i_tom_Descriptor);

  fd = accept (sip->descriptor, (void *) &sin, &len);
  if (fd < 0 || get_socket_name (fd, &address_self, &port_self))
    TRT_RAISE (self, c_tom_Conditions_error,
	       byte_string_with_c_string (OS_ERROR_MSG));
  
  address_and_port (&sin, &address, &port);
  peer = (void *) TRT_SEND ((reference_imp), _mr_c_too_InetPort,
			    SEL (r_with_i_at_r), port, address);

  r = TRT_SEND ((reference_imp), _mr_c_too_ConnectedInetPort, SEL (r_alloc));
  return TRT_SEND ((reference_imp), r,
		   SEL (r_initWithPort_i_at_r_descriptor_i_server_r_peer_r),
		   port_self, address_self, fd, self, peer);
}

tom_object
i_too_ServerInetPort_r_initWithPort_i_at_r (tom_object self, selector cmd,
					    tom_int port, tom_object address)
{
  struct _es_i_too_InetPort *ip;
  int fd, fail = 0;

  /* XXX Should do getprotobyname...  */
  fd = socket (AF_INET, SOCK_STREAM, 6);
  if (fd == -1)
    fail = 1;
  else
    {
      struct sockaddr_in sin;

      if (address || port)
	{
	  BZERO (&sin, sizeof (sin));
	  sin.sin_family = AF_INET;
	  sin.sin_port = htons (port);
      
	  if (address)
	    {
	      tom_byte *aa;
	      tom_int al;

	      aa = TRT_SEND ((pointer_imp), address, SEL (_pi__osAddress), &al);

	      if (al != sizeof (struct in_addr))
		ABORT ();

	      memcpy (&sin.sin_addr, aa, al);
	    }

	  if (bind (fd, (void *) &sin, sizeof (sin)))
	    fail = 1;
	}

      if (!fail)
	if (listen (fd, 0)
	    || ((!address || !port)
		&& get_socket_name (fd, &address, &port)))
	  fail = 1;
    }

  /* Initialize the ADDRESS and PORT ivars _now_ so we contain some useful
     info when we raise a condition.  */
  /* XXX Should super's init.  */
  ip = trt_ext_address (self, _ei_i_too_InetPort);
  ip->address = (void *) address;
  ip->port = port;

  if (fail)
    {
      if (fd != -1)
	close (fd);
      TRT_RAISE (self, c_tom_Conditions_error,
		 byte_string_with_c_string (OS_ERROR_MSG));
    }
  else
    {
      struct _es_i_tom_Descriptor *sip
	= trt_ext_address (self, _ei_i_tom_Descriptor);
      sip->descriptor = fd;
    }

  return self;
}

void
i_too_DescriptorSet_v_remove_r (tom_object self, selector cmd,
				tom_object descriptor)
{
  tom_int fd = TRT_SEND ((int_imp), descriptor, USEL (tom, i_descriptor));
  struct _es_i_too_DescriptorSet *this;

  /* If it hasn't got a valid file descriptor, it also isn't present...
     On the other hand, let's be safe.  */
  if (fd < 0)
    TRT_RAISE (self, c_tom_Conditions_error,
	       byte_string_with_c_string ("invalid descriptor"));

  this = trt_ext_address (self, _ei_i_too_DescriptorSet);

  if (this->beyond_last > fd && FD_ISSET (fd, (fd_set *) this->set))
    {
      FD_CLR (fd, (fd_set *) this->set);

      if (!--this->num)
	this->beyond_last = 0;
      else if (fd == this->beyond_last - 1)
	do
	  {
	    this->beyond_last--;
	  } while (!FD_ISSET (this->beyond_last - 1, (fd_set *) this->set));
    }
}

void
i_too_DescriptorSet_v_set_r_at_r (tom_object self, selector cmd,
				  tom_object delegate, tom_object descriptor)
{
  tom_int fd = TRT_SEND ((int_imp), descriptor, USEL (tom, i_descriptor));
  struct _es_i_too_DescriptorSet *this;

  if (fd < 0)
    TRT_RAISE (self, c_tom_Conditions_error,
	       byte_string_with_c_string ("invalid descriptor"));

  this = trt_ext_address (self, _ei_i_too_DescriptorSet);
  if (fd > this->cap)
    {
      tom_int new_cap = this->cap;

      do
	{
	  new_cap += 8 * (sizeof (fd_set) ?: sizeof (tom_long));
	} while (fd >= new_cap);

      this->set = xrealloc (this->set, new_cap / 8);
      BZERO ((char *) this->set + this->cap / 8, (new_cap - this->cap) / 8);
      this->cap = new_cap;
    }

  FD_SET (fd, (fd_set *) this->set);
  if (fd >= this->beyond_last)
    {
      this->beyond_last = fd + 1;
      TRT_SEND (, this->descriptors, USEL (tom, v_resize_i), this->beyond_last);
      TRT_SEND (, this->delegates, USEL (tom, v_resize_i), this->beyond_last);
    }

  this->num++;
  TRT_SEND (, this->descriptors, SEL (v_set_r_at_i), descriptor, (tom_int) fd);
  TRT_SEND (, this->delegates, SEL (v_set_r_at_i), delegate, (tom_int) fd);
}

void
i_too_RunLoop_v_run (tom_object self, selector cmd)
{
  int i, n, end = 0, read_end, write_end, warp_enabled = 0;
  fd_set *in_read_set = 0, *in_write_set = 0;
  fd_set *var_read_set = 0, *var_write_set = 0;
  struct _es_i_too_RunLoop *this;
  tom_object next_timer = 0;
  tom_double next_time = 0;

  /* The size of the sets in bits.  */
  int set_size = 0;

  this = trt_ext_address (self, _ei_i_too_RunLoop);
  this->d_changed = this->t_changed = 1;
  for (;;)
    {
      if (this->delegate)
	TRT_SEND (, this->delegate, SEL (v_runLoopWillSelect_r), self);

      if (this->d_changed)
	{
	  void *read_set, *write_set;
	  tom_int cap;

	  read_set = TRT_SEND ((pointer_imp), this->read_set,
			       SEL (_pi__vitals), &read_end);
	  write_set = TRT_SEND ((pointer_imp), this->write_set,
				SEL (_pi__vitals), &write_end);

	  end = read_end > write_end ? read_end : write_end;
	  cap = ((end + 8 * sizeof (tom_long) - 1)
		 & ~(8 * sizeof (tom_long) - 1));
	  if (cap > set_size)
	    {
	      set_size = cap;

	      /* XXX This alloca sort-of is a leak, but it is not supposed
                 to happen that much, with things rounded up to 64 bits.  */
	      in_read_set = alloca (cap / 8);
	      in_write_set = alloca (cap / 8);

	      var_read_set = alloca (cap / 8);
	      var_write_set = alloca (cap / 8);
	    }

	  /* Clear the bits (from the previous time these were set beyond
             the current limit for each set).  */
	  BZERO (in_read_set, cap / 8);
	  BZERO (in_write_set, cap / 8);

	  /* XXX 63!  */
	  if (read_set)
	    memcpy (in_read_set, read_set, ((read_end + 63) & ~63) / 8);
	  if (write_set)
	    memcpy (in_write_set, write_set, ((write_end + 63) & ~63) / 8);

	  this->d_changed = 0;
	}

      if (this->t_changed)
	{
	  next_timer = TRT_SEND (_PI_, this->timers, USEL (tom, r_min));
	  if (next_timer)
	    next_time = TRT_SEND (_DI_, next_timer, SEL (d_fire_time));
	  this->t_changed = 0;
	}

      memcpy (var_read_set, in_read_set, set_size / 8);
      memcpy (var_write_set, in_write_set, set_size / 8);

      if (next_timer)
	{
	  tom_double d = TRT_SEND (_DI_, CREF (tom_Date),
				   SEL (d_relativeTimeIntervalSinceNow));
	  struct timeval timeout;

	  d = next_time - d;
	  if (d < 0)
	    {
	      timeout.tv_sec = timeout.tv_usec = 0;
	      if (!end)
		{
		  n = 0;
		  goto skip_select;
		}
	    }
	  else
	    {
	      timeout.tv_sec = d;
	      timeout.tv_usec = (d - timeout.tv_sec) * 1e6;
	    }
	  n = select (end, (void *) var_read_set, (void *) var_write_set,
		      0, &timeout);
	skip_select:
	}
      else
	n = select (end, (void *) var_read_set, (void *) var_write_set, 0, 0);

      /* Handle the descriptors.  */
      /* XXX Should use this thread's ERRNO.  */
      if (n < 0 && errno != EINTR)
	{
	  /* XXX This should signal instead of raise?
	     Fri Mar 14 18:30:09 1997, tiggr@akebono.ics.ele.tue.nl  */
	  TRT_RAISE (self, c_tom_Conditions_error,
		     byte_string_with_c_string (OS_ERROR_MSG));
	}
      else if (n > 0)
	{
	  for (i = 0; i < read_end; i++)
	    if (FD_ISSET (i, var_read_set))
	      {
		TRT_SEND (, this->read_set, SEL (v_readEvent_i), (tom_int) i);
		if (!--n)
		  break;
	      }

	  if (n)
	    {
	      for (i = 0; i < write_end; i++)
		if (FD_ISSET (i, var_write_set))
		  {
		    TRT_SEND (, this->write_set, SEL (v_writeEvent_i),
			      (tom_int) i);
		    if (!--n)
		      break;
		  }

	      if (n)
		TRT_RAISE (self, c_tom_Conditions_error,
			   byte_string_with_c_string (OS_ERROR_MSG));
	    }
	}

      /* Handle the timers.  */
      if (next_timer)
	{
	  tom_double d = TRT_SEND (_DI_, CREF (tom_Date),
				   SEL (d_relativeTimeIntervalSinceNow));

	  while (d >= next_time)
	    {
	      tom_object o = TRT_SEND (_PI_, this->timers, USEL (tom, r_min));

	      if (next_timer != o)
		next_timer = o;
	      else
		{
		  TRT_SEND (, this->timers, SEL (v_remove_r), next_timer);
		  this->t_changed = 1;

		  TRT_SEND (, next_timer, SEL (v_fire));

		  next_timer = TRT_SEND (_PI_, this->timers, USEL (tom, r_min));
		}
	      this->t_changed = 0;

	      if (next_timer)
		next_time = TRT_SEND (_DI_, next_timer, SEL (d_fire_time));
	      else
		break;
	    }
	}
    }
}

void
i_tom_All_RunLoop_v_perform_s_after_d_with_x (tom_object self, selector cmd,
					      selector sel, tom_double secs,
					      ...)
{
  tom_object invocation, timer;
  va_list ap;

  va_start (ap, secs);
  invocation = TRT_SEND (_PI_, CREF (tom_Invocation),
			 SEL (r_of_s_to__r_using_p), sel, self, &ap);
  va_end (ap);

  timer = TRT_SEND (_PI_, CREF (too_Timer),
		    SEL (r_withInterval_d_invocation_r_repeats__o),
		    secs, invocation, (tom_byte) 0);

  TRT_SEND (, timer, SEL (v_schedule));
}
