/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: array.c,v 1.72 1998/08/19 08:04:32 tiggr Exp $  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "trt.h"
#include <tom/tom-r.h>

#define CONTENTS(X)		((ET *) ((X)->contents))
#define VOID_CONTENTS(X)	((X)->contents)

__inline__ void
trt_array_resize (struct _es_i_tom_Array *a, struct _es_i_tom_MutableArray *m,
		  int elt_size, int num)
{
  if (m->capacity - a->length < num)
    {
      if (!m->capacity)
	m->capacity = 4;

      while (m->capacity - a->length < num)
	m->capacity *= 2;

      a->contents = xrealloc (a->contents, m->capacity * elt_size);
    }
}

/* Raise a condition iff the INDEX is equal to or larger than the LENGTH.  */
static __inline__ void
check_index (tom_object self, tom_int index, tom_int length)
{
  if ((unsigned) index >= (unsigned) length)
    trt_raise (0, self, 0, c_tom_Conditions_program_condition,
	       "bad index %d for array bound %d", (int) index, (int) length);
}

static void
array_remove_elements (tom_object self, tom_int start, tom_int length, int size)
{
  struct _es_i_tom_Array *this;
  struct _es_i_tom_MutableArray *that;

  this = trt_ext_address (self, _ei_i_tom_Array);
  that = trt_ext_address (self, _ei_i_tom_MutableArray);

  start = TRT_SEND (_II_, self, SEL (_ii__adjustRange__ii_),
		    start, length, &length);
  if (length)
    {
      memmove ((char *) this->contents + start * size,
	       (char *) this->contents + (start + length) * size,
	       (this->length - (start + length)) * size);
      this->length -= length;
    }
}

static __inline__ tom_object
array_with_elts (tom_object self, selector cmd, va_list ap,
		 enum trt_type_encoding te, int elt_size)
{
  int i, n = cmd->in->num;
  void *contents = xmalloc (n * elt_size);
  tom_object inst;

  for (i = 0; i < n; i++)
    switch (te)
      {
      case TRT_TE_REFERENCE:
	switch (cmd->in->args[i])
	  {
	  case TRT_TE_REFERENCE:
	    ((tom_object *) contents)[i] = va_arg (ap, tom_object);
	    break;

	  default:
	  fail_arg:
	    unimplemented (__FUNCTION__ ": unhandled arg type %d",
			   cmd->in->args[i]);
	    break;
	  }
	break;

      case TRT_TE_INT:
	switch (cmd->in->args[i])
	  {
	  case TRT_TE_INT:
	    ((tom_int *) contents)[i] = va_arg (ap, tom_int);
	    break;

	  default:
	    goto fail_arg;
	  }
	break;

      default:
	unimplemented (__FUNCTION__ ": unhandled array type %d", te);
	break;
      }

  inst = TRT_SEND ((reference_imp), self, SEL (r_alloc));
  inst = TRT_SEND (_PI_, inst, SEL (r_initWith_i_at_p), (tom_int) n, contents);

  return inst;
}

#define ARRAY Array

/* Element type.  */
#define ET	tom_byte
/* Element type description.  */
#define ETD	TRT_TE_BYTE
/* Element prefix.  */
#define EP	byte
/* Capitalized element prefix.  */
#define CP	Byte
#define F	b
#include "arraydef.c"
#undef ET
#undef ETD
#undef EP
#undef CP
#undef F

#define ET	tom_char
#define ETD	TRT_TE_CHAR
#define EP	char
#define CP	Char
#define F	c
#include "arraydef.c"
#undef ET
#undef ETD
#undef EP
#undef CP
#undef F

#undef STRING_METHODS

#define ET	tom_int
#define ETD	TRT_TE_INT
#define EP	int
#define CP	Int
#define F	i
#include "arraydef.c"
#undef ET
#undef ETD
#undef EP
#undef CP
#undef F

#define ET	tom_long
#define ETD	TRT_TE_LONG
#define EP	long
#define CP	Long
#define F	l
#include "arraydef.c"
#undef ET
#undef ETD
#undef EP
#undef CP
#undef F

#define ET	tom_float
#define ETD	TRT_TE_FLOAT
#define EP	float
#define CP	Float
#define F	f
#include "arraydef.c"
#undef ET
#undef ETD
#undef EP
#undef CP
#undef F

#define ET	tom_double
#define ETD	TRT_TE_DOUBLE
#define EP	double
#define CP	Double
#define F	d
#include "arraydef.c"
#undef ET
#undef ETD
#undef EP
#undef CP
#undef F

#define NO_INIT_WITH_ENUMERATOR
#define ET	selector
#define ETD	TRT_TE_SELECTOR
#define EP	selector
#define CP	Selector
#define F	s
#include "arraydef.c"
#undef ET
#undef ETD
#undef EP
#undef CP
#undef F

#define ET	tom_byte
#define ETD	TRT_TE_BOOLEAN
#define EP	boolean
#define CP	Boolean
#define F	o
#include "arraydef.c"
#undef ET
#undef ETD
#undef EP
#undef CP
#undef F
#undef NO_INIT_WITH_ENUMERATOR

#define ET	tom_object
#define ETD	TRT_TE_REFERENCE
#define EP	object
#define CP	Object
#define F	r
#define DO_ARRAY_METHODS
#include "arraydef.c"
#undef ETD
#undef EP
#undef CP
#undef F

/* Subsidiary for the Indexed elements and elements: methods.  */
static void
x_elements (tom_object self, selector cmd, builtin_return_type *rt,
	    tom_int start, tom_int num, tom_int length, va_list ap)
{
  int i;

  BZERO (rt, sizeof (*rt));

  if (num != cmd->out->num)
    fatal ("elements count mismatch return=%d actual=%d", cmd->out->num, num);

  if (start < 0 || start + num > length)
    fatal ("index range (%d, %d) does not fit array length %d",
	   start, num, length);

  for (i = 0; i < num; i++)
    {
      void *p = i ? va_arg (ap, void *) : 0;

      switch (cmd->out->args[i])
	{
	case TRT_TE_BOOLEAN:
	case TRT_TE_BYTE:
	  {
	    tom_byte v = TRT_SEND ((byte_imp), self, SEL (b_at_i), i + start);
	    if (i)
	      *((tom_byte *) p) = v;
	    else
	      rt->i.i = v;
	  }
	  break;

	case TRT_TE_CHAR:
	  {
	    tom_char v = TRT_SEND ((char_imp), self, SEL (c_at_i), i + start);
	    if (i)
	      *((tom_char *) p) = v;
	    else
	      rt->i.i = v;
	  }
	  break;

	case TRT_TE_INT:
	  {
	    tom_int v = TRT_SEND ((int_imp), self, SEL (i_at_i), i + start);
	    if (i)
	      *((tom_int *) p) = v;
	    else
	      rt->i.i = v;
	  }
	  break;

	case TRT_TE_LONG:
	  {

	    tom_long v = TRT_SEND ((long_imp), self, SEL (l_at_i), i + start);
	    if (i)
	      *((tom_long *) p) = v;
	    else
	      rt->l.l = v;
	  }
	  break;

	case TRT_TE_FLOAT:
	  {
	    tom_float v = TRT_SEND ((float_imp), self, SEL (f_at_i), i + start);
	    if (i)
	      *((tom_float *) p) = v;
	    else
	      RETURN_SET_FLOAT (rt, v);
	  }
	  break;

	case TRT_TE_DOUBLE:
	  {
	    tom_double v = TRT_SEND ((double_imp), self, SEL (d_at_i),
				     i + start);
	    if (i)
	      *((tom_double *) p) = v;
	    else
	      RETURN_SET_DOUBLE (rt, v);
	  }
	  break;

	default:
	  fatal ("elements: unhandled type %d", cmd->out->args[i]);
	  break;
	}
    }
}

void
i_tom_Array_v_dealloc (tom_object self, selector cmd)
{
  struct _es_i_tom_Array *this = trt_ext_address (self, _ei_i_tom_Array);

  xfree (this->contents);
}

GENERIC_RETURN_TYPE
i_tom_Indexed_x_elements (tom_object self, selector cmd, ...)
{
  tom_int length = TRT_SEND ((int_imp), self, SEL (i_length));
  builtin_return_type rt;
  va_list ap;
  
  va_start (ap, cmd);
  x_elements (self, cmd, &rt, 0, length, length, ap);
  va_end (ap);

  APPLY_ARGS_RETURN (&rt);
}

GENERIC_RETURN_TYPE
i_tom_Indexed_x_elements__ii_ (tom_object self, selector cmd,
			       tom_int start, tom_int num, ...)
{
  tom_int length = TRT_SEND ((int_imp), self, SEL (i_length));
  builtin_return_type rt;
  va_list ap;

  va_start (ap, num);
  x_elements (self, cmd, &rt, start, num, length, ap);
  va_end (ap);

  APPLY_ARGS_RETURN (&rt);
}

tom_char
i_tom_ByteArray_c_at_i (tom_object self, selector cmd, tom_int index)
{
  struct _es_i_tom_Array *this = trt_ext_address (self, _ei_i_tom_Array);

  if ((unsigned) index >= (unsigned) this->length)
    fatal ("bad index %d for array bound %d", (int) index, (int) this->length);

  return ((tom_byte *) VOID_CONTENTS (this))[index];
}

tom_byte
i_tom_CharArray_b_at_i (tom_object self, selector cmd, tom_int index)
{
  struct _es_i_tom_Array *this = trt_ext_address (self, _ei_i_tom_Array);

  if ((unsigned) index >= (unsigned) this->length)
    fatal ("bad index %d for array bound %d", (int) index, (int) this->length);

  return ((tom_char *) VOID_CONTENTS (this))[index];
}

tom_int
i_tom_ByteArray_i_hash (tom_object self, selector cmd)
{
  struct _es_i_tom_Array *this = trt_ext_address (self, _ei_i_tom_Array);
  int h = 0, i, n = this->length > 32 ? 32 : this->length;

  /* ZZZ A ByteString should hash in Unicode characters, according to its
     mapping.  */
  for (i = 0; i < n; i++)
    h = (h << 3) ^ ((tom_byte *) this->contents)[i];

  return h;
}

tom_int
i_tom_IntArray_i_hash (tom_object self, selector cmd)
{
  struct _es_i_tom_Array *this = trt_ext_address (self, _ei_i_tom_Array);
  int h = 0, i, n = this->length > 32 ? 32 : this->length;

  /* XXX Is this good hashing?  */
  for (i = 0; i < n; i++)
    h = (h << 3) ^ ((tom_int *) this->contents)[i];

  return h;
}

tom_int
i_tom_ByteString_i_hashRange__ii_ (tom_object self, selector cmd,
				   tom_int start, tom_int len)
{
  struct _es_i_tom_Array *this = trt_ext_address (self, _ei_i_tom_Array);
  int h = 0, i, n;

  start = TRT_SEND (, self, SEL (_ii__adjustRange__ii_), start, len, &len);

  n = len > 32 ? 32 : len;

  /* ZZZ A ByteString should hash in Unicode characters, according to its
     mapping.  */
  for (i = 0; i < n; i++)
    h = (h << 3) ^ ((tom_byte *) this->contents)[start + i];

  return h;
}

tom_int
i_tom_MutableByteArray_i_readRange__ii__fromByteArray_r_to_i
  (tom_object self, selector cmd, tom_int start, tom_int length,
   tom_object other, tom_int position)
{
  struct _es_i_tom_Array *a;
  struct _es_i_tom_MutableArray *m;
  void *elts;

  a = trt_ext_address (self, _ei_i_tom_Array);
  m = trt_ext_address (self, _ei_i_tom_MutableArray);

  if (position < 0 || position > a->length)
    ABORT ();

  elts = TRT_SEND ((pointer_imp), other, SEL (_pi__pointerToElements__ii_),
		   start, length, &length);

  if (position + length > m->capacity)
    trt_array_resize (a, m, sizeof (tom_byte), position + length - a->length);

  memcpy ((char *) a->contents + position, elts, length * sizeof (tom_byte));
  if (position + length > a->length)
    a->length = position + length;

  return length;
}

tom_object
byte_string_with_string (char *s, int n)
{
  tom_object str = TRT_SEND ((reference_imp), _mr_c_tom_ByteString,
			     SEL (r_alloc));

  s = memcpy (xmalloc (n), s, n);

  return TRT_SEND ((reference_imp), str, SEL (r_init__pi_), s, n);
}

tom_object
byte_string_with_c_string (char *s)
{
  return byte_string_with_string (s, strlen (s));
}

char *
c_string_with_length (char *buf, int len)
{
  char *s = xmalloc (len + 1);

  memcpy (s, buf, len);
  s[len] = 0;

  return s;
}

char *
c_string_with_tom_string (tom_object string)
{
  tom_int l;
  char *s;

  s = TRT_SEND ((pointer_imp), string, SEL (_pi__byteStringContents), &l);
  s = memcpy (xmalloc (1 + l), s, l);
  s[l] = 0;

  return s;
}

tom_object
c_tom_State_r_name (c_tom_State self, selector cmd)
{
  return byte_string_with_string (self->c_tom_State.info.name.s,
				  self->c_tom_State.info.name.len);
}

#if 0
/* XXX Is this ever used?  Should it?  Can it be removed?
   Sat May 18 22:37:05 1996, tiggr@tricky.es.ele.tue.nl  */
i_tom_ByteString
new_byte_string (struct _es_i_tom_Array **this, tom_int len)
{
  i_tom_ByteString str
    = (void *) TRT_SEND ((reference_imp), _mr_c_tom_ByteString, SEL (r_alloc));
  struct _es_i_tom_Array *that
    = trt_ext_address ((tom_object) str, _ei_i_tom_Array);

  that->length = len;
  that->contents = xmalloc (that->length);

  *this = that;

  return str;
}
#endif

tom_byte
i_tom_ByteString_o_equalByteString_r (tom_object self, selector cmd,
				      tom_object o)
{
  struct _es_i_tom_Array *this = trt_ext_address (self, _ei_i_tom_Array);
  tom_int len;
  void *s = TRT_SEND ((pointer_imp), o, SEL (_pi__byteStringContents), &len);

  return this->length == len && !memcmp (this->contents, s, len);
}

tom_byte
i_tom_CharString_o_equalCharString_r (tom_object self, selector cmd,
				      tom_object o)
{
  ABORT ();
}

tom_object
i_tom_MutableByteString_r_substring__ii_
  (tom_object self, selector cmd, tom_int start, tom_int len)
{
  tom_object str = TRT_SEND ((reference_imp), self->isa, SEL (r_alloc));
  struct _es_i_tom_Array *this;

  start = TRT_SEND ((int_imp), self, SEL (_ii__adjustRange__ii_),
		    start, len, &len);
  this = trt_ext_address (self, _ei_i_tom_Array);

  return TRT_SEND ((reference_imp), str, SEL (r_init__pi_),
		   memcpy (xmalloc (len * sizeof (tom_byte)),
			   (tom_byte *) this->contents + start, len), len);
}

tom_object
i_tom_CharString_r_substring__ii_
  (tom_object self, selector cmd, tom_int start, tom_int len)
{
  tom_object str = TRT_SEND ((reference_imp), self->isa, SEL (r_alloc));
  struct _es_i_tom_Array *this;

  start = TRT_SEND ((int_imp), self, SEL (_ii__adjustRange__ii_),
		    start, len, &len);
  this = trt_ext_address (self, _ei_i_tom_Array);

  return TRT_SEND ((reference_imp), str, SEL (r_init__pi_),
		   memcpy (xmalloc (len * sizeof (tom_char)),
			   (tom_char *) this->contents + start,
			   len * sizeof (tom_char)), len);
}

tom_object
i_tom_CharString_r_mutableSubstring__ii_
  (tom_object self, selector cmd, tom_int start, tom_int len)
{
  tom_object str = TRT_SEND ((reference_imp),
			     _mr_c_tom_MutableCharString, SEL (r_alloc));
  struct _es_i_tom_Array *this;

  start = TRT_SEND ((int_imp), self, SEL (_ii__adjustRange__ii_),
		    start, len, &len);
  this = trt_ext_address (self, _ei_i_tom_Array);

  return TRT_SEND ((reference_imp), str, SEL (r_init__pi_),
		   memcpy (xmalloc (len * sizeof (tom_char)),
			   (tom_char *) this->contents + start,
			   len * sizeof (tom_char)), len);
}

tom_object
i_tom_ObjectArray_r_deepen_i_mutably__o (tom_object self, selector cmd,
					 tom_int level, tom_byte mutably)
{
  struct _es_i_tom_Array *this = trt_ext_address (self, _ei_i_tom_Array);
  int i;

  if (!level)
    return self;

  for (i = 0; i < this->length; i++)
    {
      tom_object o = TRT_SEND ((reference_imp), ((void **) this->contents)[i],
			       mutably ? SEL (r_mutableCopy) : SEL (r_copy));

      ((void **) this->contents)[i] = TRT_SEND ((reference_imp), o,
						SEL (r_deepen_i_mutably__o),
						level - 1, mutably);
    }

  return self;
}

void
i_tom_MutableObjectArray_v_insert_r_at_i (tom_object self, selector cmd,
					  tom_object object, tom_int index)
{
  struct _es_i_tom_MutableArray *that;
  struct _es_i_tom_Array *this;

  this = trt_ext_address (self, _ei_i_tom_Array);
  that = trt_ext_address (self, _ei_i_tom_MutableArray);

  check_index (self, index, this->length + 1);
  trt_array_resize (this, that, sizeof (void *), 1);

  if (index != this->length)
    memmove ((void **) this->contents + index + 1,
	     (void **) this->contents + index,
	     sizeof (void *) * (this->length - index));
  ((void **) this->contents)[index] = trt_assign_object_var (self, object);
  this->length++;
}
