/* GtkEditor - a source editor widget for GTK
 * Copyright (C) 1998 Thomas Mailund.
 *
 * The editor widget was written by Thomas Mailund, so bugs should be
 * reported to <mailund@daimi.au.dk>, not the gtk ppl.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>

#include <vdkb/gtkeditor.h>
#include <vdkb/internal.h>
#include <vdkb/syntaxtable.h>
#include <vdkb/gtkeditor-regex.h>
#include <vdkb/parent.h>

#define TIME_DELAY    300	/* delay in milisec. before hilite */
#define INITIAL_SCREEN_SIZE 100	/* FIXME */

/* widget stuff */
static void   gtk_editor_class_init          (GtkEditorClass   *klass);
static void   gtk_editor_init                (GtkEditor        *editor);
static void   gtk_editor_destroy             (GtkObject        *object);

/* text property data */
static gboolean cmp_prop_data                (gpointer          d1,
					      gpointer          d2);
static gpointer clone_prop_data              (gpointer          d);
static void free_prop_data                   (gpointer          d);

/* search stuff */
static gint   search                         (GtkEditor        *editor,
					      guint             pos,
					      gchar            *string,
					      gboolean          casein);
static gint   search_back                    (GtkEditor        *editor,
					      guint             pos,
					      gchar            *string,
					      gboolean          casein);
/* hilit */
static void   hilite_when_idle             (GtkEditor        *editor);

/* indent */
static void   default_one_line_ind           (GtkEditor        *editor);

/* events */
static void   reset_idle_event               (GtkWidget        *editor);
static gint   idle_event                     (GtkWidget        *editor);
static gint   gtk_editor_key_press           (GtkWidget        *widget,
					      GdkEventKey      *event);
static void   gtk_editor_changed_pre         (GtkEditor        *editor);
static void   gtk_editor_changed_post        (GtkEditor        *editor);


/* --<local data>--------------------------------------------------------- */
static GtkWidgetClass *parent_class = NULL;

/* --<widget initialization, constructor and destructor>------------------ */
guint
gtk_editor_get_type ()
{
  static guint editor_type = 0;

  if (!editor_type)
    {
      GtkTypeInfo editor_info =
      {
	"GtkEditor",
	sizeof (GtkEditor),
	sizeof (GtkEditorClass),
	(GtkClassInitFunc) gtk_editor_class_init,
	(GtkObjectInitFunc) gtk_editor_init,
	(GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL,
      };

      editor_type = gtk_type_unique (gtk_sctext_get_type (), &editor_info);
    }

  return editor_type;
}

static void
gtk_editor_class_init (GtkEditorClass *class)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) class;

  re_set_syntax (RE_SYNTAX_EMACS);

  /* init local data */
  parent_class = gtk_type_class (gtk_sctext_get_type ());

  /* setup signals */
  object_class->destroy = gtk_editor_destroy;
}

static void
gtk_editor_init (GtkEditor *editor)
{
  GtkSCText *text = GTK_SCTEXT (editor);

  gtk_sctext_set_property_data_functions (text, cmp_prop_data,
					  clone_prop_data, free_prop_data);
  /* hilite */
  editor->stable = NULL;

  editor->hilite_when_idle = TRUE;
  editor->idle_time = 0;
  editor->screen_size = INITIAL_SCREEN_SIZE; /* FIXME */
  editor->changed_from = editor->changed_to = 0;

  /* idle stuff */
  gtk_signal_connect (GTK_OBJECT (editor), "key_press_event",
		      GTK_SIGNAL_FUNC (reset_idle_event),
		      NULL);
  gtk_signal_connect (GTK_OBJECT (editor), "button_press_event",
		      GTK_SIGNAL_FUNC (reset_idle_event),
		      NULL);

  /* to keep track of changes, for hilite_when_idle */
  gtk_signal_connect (GTK_OBJECT (editor), "insert_text",
		      GTK_SIGNAL_FUNC (gtk_editor_changed_pre),
		      NULL);
  gtk_signal_connect (GTK_OBJECT (editor), "delete_text",
		      GTK_SIGNAL_FUNC (gtk_editor_changed_pre),
		      NULL);
  gtk_signal_connect (GTK_OBJECT (editor), "changed",
		      GTK_SIGNAL_FUNC (gtk_editor_changed_post),
		      NULL);

  gtk_signal_connect (GTK_OBJECT (editor), "key_press_event",
		      GTK_SIGNAL_FUNC (gtk_editor_key_press),
		      NULL);

  /* indent */
  editor->one_line_ind = (GtkEditorIndFunc)default_one_line_ind;
  editor->ind_accel_key = GDK_Tab;
  editor->ind_accel_mod = 0;
  editor->indenter_data = NULL;

  /* HACK!!! */
  /* it would seem that the properties list in sctext doesn't get
   * initialized as it is supposed to unless we set the list to NULL
   * initially.  This should really be fixed in the text patch, but I
   * want to minimize the number of new patches, so... */
  text->text_properties = NULL;
}

GtkWidget*
gtk_editor_new (GtkAdjustment *hadj,
		GtkAdjustment *vadj)
{
  GtkWidget *editor;

  if (hadj)
    g_return_val_if_fail (GTK_IS_ADJUSTMENT (hadj), NULL);
  if (vadj)
    g_return_val_if_fail (GTK_IS_ADJUSTMENT (vadj), NULL);
  
  editor = gtk_widget_new (GTK_TYPE_EDITOR,
			   "hadjustment", hadj,
			   "vadjustment", vadj,
			   NULL);

  return GTK_WIDGET (editor);
}

static void
gtk_editor_destroy (GtkObject *object)
{
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_EDITOR (object));

  GTK_OBJECT_CLASS(parent_class)->destroy (object);
}

/* --<text property data>------------------------------------------------- */
/* These are really dummy functions right now, since we don't store any 
 * real data in the properties, but just a bit stating wether we're in a block
 * or not */
static gboolean
cmp_prop_data (gpointer p1, gpointer p2)
{
  return TRUE;
}

static gpointer
clone_prop_data (gpointer p)
{
  return p;
}

static void
free_prop_data (gpointer d)
{
  ;
}

/* --<search'n'stuff>----------------------------------------------------- */
/* NB! There is really no reason why all this searching stuff couldn't
 * be moved to the text widget...the only reason I can think of is
 * that, the both gtksctext and gtkeditor would have to deal with
 * regexps, but that doesn't seem too discouraging. We will have to
 * make them more 'public' though...the editor _needs_ some of the
 * static functions, so we can't hide them in the text widget. */

/* _gtk_editor_txteq -- compares 'text' at 'index' with 'string' */
#if 0
gboolean
_gtk_editor_txteq (GtkSCText *text, char *string, int index)
{
  int diff;
  int len = strlen (string);

  if ( (index < text->gap_position) ) {
    /* before gap */

    /* does the string overlap the gap? */
    if ((index + strlen (string)) > text->gap_position) {
      diff = text->gap_position - index;
      return ( (strncmp (text->text+index, string, diff) == 0)
	       && (strncmp (text->text+text->gap_position+text->gap_size,
			    string+diff, len-diff) == 0) );
      
    } else {
      return ( strncmp (text->text+index, string, len) == 0 );
    }

  } else {
    /* after gap ... adjust index */

    index += text->gap_size;
    diff = text->text_end - index;
    return ( strncmp (text->text+index, string, MIN (diff,len)) == 0 );
  }
}
#endif

/* search -- searches in editor, from pos and forward, for string.  If
 * string is found, returns position, otherwise returns -1.
 * If casein is true, search is case insensitive. */
/* FIXME: This could and should be optimised. There is no need to
 * get_text all the time, is there?  We just have to be carefull with
 * the gap. ... use a modified version of _gtk_editor_txteq! */
static gint
search (GtkEditor *editor, guint pos, gchar *string, gboolean casein)
{
  gchar *buf = NULL;
  gint match = -1;
  gint text_len = gtk_sctext_get_length (GTK_SCTEXT (editor));
  gint len = strlen (string);

  g_return_val_if_fail (pos <= text_len, -1);
  if (pos == text_len) return -1; /* not illegal, but nothing to do */

  gtk_sctext_freeze (GTK_SCTEXT (editor));
  for (; pos < (text_len - len) + 1; pos++) {
    buf = gtk_editable_get_chars (GTK_EDITABLE (editor), pos, pos + len);
    if (!buf) return match;	/* somethings wrong */
    if (casein) {
      if (strcasecmp (buf, string) == 0) {
	match = pos;
	break;
      }
    } else {
      if (strcmp (buf, string) == 0) {
	match = pos;
	break;
      }
    }
    g_free (buf);
  }
  g_free (buf);
  gtk_sctext_thaw (GTK_SCTEXT (editor));

  return match;
}

/* gtk_editor_search_from_point -- searches for string, if found, goto
 * and select match. If casein is true, search is case insensitive. If
 * string is found it returns TRUE, otherwise FALSE. */
gboolean
gtk_editor_search_from_point (GtkEditor *editor, gchar *string, gboolean casein)
{
  gint new_pos;

  g_return_val_if_fail (editor != NULL, FALSE);
  g_return_val_if_fail (string != NULL, FALSE);

  new_pos = search (editor, gtk_sctext_get_point (GTK_SCTEXT (editor)), string, casein);
  if (new_pos > -1) {
    gtk_editable_set_position (GTK_EDITABLE (editor), new_pos);
    gtk_editable_select_region (GTK_EDITABLE (editor), new_pos,
				new_pos + strlen (string));
    return TRUE;
  } else {
    return FALSE;
  }
}

/* search_back -- searches in editor, from pos and backward, for
 * string.  If string is found, returns position, otherwise returns
 * -1.  If casein is true, search is case insensitive. */
/* FIXME: This could and should be optimised. There is no need to
 * get_text all the time, is there?  We just have to be carefull with
 * the gap. */
static gint
search_back (GtkEditor *editor, guint pos, gchar *string, gboolean casein)
{
  gchar *buf = NULL;
  gint match = -1;
  gint text_len = gtk_sctext_get_length (GTK_SCTEXT (editor));
  gint len = strlen (string);

  if (pos == 0) return -1; /* not illegal, but nothing to do */
  if (pos > text_len - len) pos = text_len - len; /* we must be reasonable */

  gtk_sctext_freeze (GTK_SCTEXT (editor));
  for (; pos >= 0; pos--) {
    buf = gtk_editable_get_chars (GTK_EDITABLE (editor), pos, pos + len);
    if (!buf) return match;	/* somethings wrong */
    if (casein) {
      if (strcasecmp (buf, string) == 0) {
	match = pos;
	break;
      }
    } else {
      if (strcmp (buf, string) == 0) {
	match = pos;
	break;
      }
    }
    g_free (buf);
  }
  g_free (buf);
  gtk_sctext_thaw (GTK_SCTEXT (editor));

  return match;
}

/* gtk_editor_search_back_from_point -- searches backward for string,
 * if found, goto and select match. If casein is true, search is case
 * insensitive. If string is found it returns TRUE, otherwise it
 * returns FALSE. */
gboolean
gtk_editor_search_back_from_point (GtkEditor *editor, 
				   gchar *string,
				   gboolean casein)
{
  gint new_pos;

  g_return_val_if_fail (editor != NULL, FALSE);
  g_return_val_if_fail (string != NULL, FALSE);

  new_pos = search_back (editor, gtk_sctext_get_point (GTK_SCTEXT (editor)),
			 string, casein);
  if (new_pos > -1) {
    gtk_editable_set_position (GTK_EDITABLE (editor), new_pos);
    gtk_editable_select_region (GTK_EDITABLE (editor), new_pos,
				new_pos + strlen (string));
    return TRUE;
  } else {
    return FALSE;
  }
}

/* gtk_editor_search_regex_from_point -- searches for string matching
 * regex, if found, goto and select match.  If the regex doesn't
 * compile (or a serious error occurs) it returns -1, if a match is
 * found it returns 1, and if no match is found it returns 0. */
gint
gtk_editor_regex_search_from_point (GtkEditor     *editor,
				    gchar         *regex)
{
  return _gtk_editor_select_regex (editor, regex, TRUE);
}

/* gtk_editor_search_regex_back_from_point -- searches for string
 * matching regex, if found, goto and select match.  If the regex
 * doesn't compile (or a serious error occurs) it returns -1, if a
 * match is found it returns 1, and if no match is found it returns
 * 0. */
gint
gtk_editor_regex_search_back_from_point (GtkEditor     *editor,
					 gchar         *regex)
{
  return _gtk_editor_select_regex (editor, regex, FALSE);
}


/* --<hilite stuff>------------------------------------------------------- */
GdkFont*
_gtk_editor_fontdup (GdkFont *font)
{
  if (font) {
    gdk_font_ref (font);
  }
  return font;
}

GdkColor*
_gtk_editor_coldup (const GdkColor *color)
{
  GdkColor *new;
  if (color) {
    new = g_new (GdkColor, 1);
    memcpy (new, color, sizeof (GdkColor));
  } else {
    new = NULL;
  }
  return new;
}


/* gtk_editor_install_stable -- installs the syntax table consisting
 * of the lists entries in editor. */
void
gtk_editor_install_stable (GtkEditor *editor,
			   GList *entries)
{
  _gtk_editor_destroy_stable (editor->stable);
  editor->stable = _gtk_editor_new_stable (entries);
}

/* gtk_editor_install_patterns -- installs highlight patterns for the
 * editor */
void
gtk_editor_install_patterns (GtkEditor *editor,
			     GList *entries)
{
  _gtk_editor_destroy_patterns (editor->patterns);
  editor->patterns = _gtk_editor_new_patterns (entries);
}

/* hilite_interval -- yep! That's what it does. FIXME: add description
 * of *how* it does it. */
void
gtk_editor_hilite_interval (GtkEditor *editor, guint from, guint to)
{
  guint pos;

  g_return_if_fail (editor != NULL);
  g_return_if_fail (from <= to);
  g_return_if_fail (to <= gtk_sctext_get_length (GTK_SCTEXT (editor)));

  pos = gtk_editable_get_position (GTK_EDITABLE (editor));

  gtk_sctext_freeze (GTK_SCTEXT (editor));

  /* correct start position, so we start at the beginning of a text
   * property If we are on offset 1, then the state should really be
   * the previous state.  It's a wonder to me, why it shouldn't be
   * offset 0...but this works, it doesn't with offset 0.  It's
   * probably because this is to fix deleting the last char of a
   * state, and this can only be done with backsspace, which puts us
   * in offset 1, or something. */
  gtk_sctext_set_point (GTK_SCTEXT (editor), from);
  if (GTK_SCTEXT (editor)->point.offset == 1) {
    GList *prev = GTK_SCTEXT (editor)->point.property->prev;
    if (prev) {
      /* it's a little hacky to get the length of the previous prop. */
      gint len = gtk_sctext_get_property_length (GTK_SCTEXT (editor),
					       from - 2);
      /* goto previous frame...the 1 is for the offset==1 thingie */
      from -= len + 1;
      if ((gint)from < 0) from = 0;	/* not before beginning of text */
    } else {
      from = 0;
    }
  } else {
    from -= GTK_SCTEXT (editor)->point.offset; /* go to beginning of property. */
  }

  /* syntax table highlighting */
  _gtk_editor_sthilite_interval (editor, from, to);

  /*  gtk_editable_set_position (GTK_EDITABLE (editor), pos);
   * This doesn't seem to be necessary any longer. Hmm... 
   */
  gtk_sctext_thaw (GTK_SCTEXT (editor));
}

void
gtk_editor_hilite_buffer (GtkEditor *editor)
{
  gtk_editor_hilite_interval (editor, 0,
			      GTK_SCTEXT (editor)->text_end
			      - GTK_SCTEXT (editor)->gap_size);
  /* and just a little extra service */
  editor->changed_from = editor->changed_to = GTK_SCTEXT (editor)->point.index;
}

/* hiliting the entire buffer can be *very* slow...this guy can be used in stead.
 * Since I have no way of actually calculating the visible text, I make a guess...
 * FIXME: when the text API changes, this should be changed!
 */
void
gtk_editor_set_screen_size (GtkEditor *editor, guint size)
{
  editor->screen_size = size;
}

void
gtk_editor_hilite_screen (GtkEditor *editor)
{
  guint idx = GTK_SCTEXT (editor)->point.index;

  gtk_editor_hilite_interval (editor, MAX (0, (int)(idx - editor->screen_size)),
			      MIN (idx + editor->screen_size,
				   GTK_SCTEXT (editor)->text_end
				   - GTK_SCTEXT (editor)->gap_size));
  /* and just a little extra service */
  editor->changed_from = editor->changed_to = idx;
}


static void
hilite_when_idle (GtkEditor *editor)
{
#ifdef EDITOR_DEBUG
  g_print ("hilite_when_idle [%u,%u]\n", 
	   editor->changed_from, editor->changed_to);
#endif

  if (editor->changed_from < editor->changed_to) {
    gtk_editor_hilite_interval (editor, editor->changed_from, editor->changed_to);
    editor->changed_from = editor->changed_to = 
      gtk_editable_get_position (GTK_EDITABLE (editor));
  }
}

void
gtk_editor_hilite_when_idle (GtkEditor *editor, gboolean hwi)
{
  if (hwi) {
    editor->hilite_when_idle = 1;
  } else {
    editor->hilite_when_idle = 0;
  }
}

/* --<indent stuff>--------------------- */

/* FIXME: we should have some better api */

#include <stdio.h>
#include <ctype.h>
extern int GetEditorTab();
/*
patched by mario to have customizable
and smart tabs.
*/
static void
default_one_line_ind (GtkEditor *editor)
{
  int spaces = 4, z = 0, len=0;
  char* tabs = NULL,*buffer,*locbuff;
  // remeber actual position
  guint pos = gtk_editable_get_position (GTK_EDITABLE (editor));
  guint iPos;
  // move to beginning of present line
  // and set there inserting point
  gtk_signal_emit_by_name (GTK_OBJECT (editor), "move_to_column", 0);
  gtk_sctext_set_point (GTK_SCTEXT (editor),
       (iPos =  gtk_editable_get_position (GTK_EDITABLE (editor))));
  // get tabs from VDKBuilder ide defaults
  // comment  line below  using default = 4
  spaces = GetEditorTab();
  // gets all chars from iPos to pos
  locbuff = gtk_editable_get_chars(GTK_EDITABLE(editor),iPos,pos);
  len = pos-iPos;
  buffer = (char*) g_malloc(len+1);
  strcpy(buffer,locbuff);
  // counts how many white spaces there are from line beginning to pos
  while( (buffer[z]) && (isspace(buffer[z])) ) z++;
  g_free(buffer);
  // compute how many spaces to next tab
  spaces = (z > 0) ? spaces - (z%spaces): spaces;
  // inserts tabs only if needed
  if(spaces > 0)
    {
      tabs = (char*) malloc(spaces+1);
      memset(tabs,(char) 0, spaces+1);
      memset(tabs,(char) ' ',spaces);
      // we send a message instead using 
      // gtk_sctext_insert(), so we have insert_text signal
      // trapped by VDKBuilder
      // 
      gtk_signal_emit_by_name (GTK_OBJECT (editor), 
			       "insert_text", 
			       tabs, 
			       spaces, 
			       &iPos);
      free(tabs);
    }
  gtk_editable_set_position (GTK_EDITABLE (editor), pos + spaces); 
}


/*
Thomas Mailund original one
*/
/*
static void
default_one_line_ind (GtkEditor *editor)
{
guint pos = gtk_editable_get_position (GTK_EDITABLE (editor)); 
gtk_sctext_freeze (GTK_SCTEXT (editor));
// FIXME: change this when they change the interface to *_move_* 
gtk_signal_emit_by_name (GTK_OBJECT (editor), "move_to_column", 0);
gtk_sctext_set_point (GTK_SCTEXT (editor),
		      gtk_editable_get_position (GTK_EDITABLE (editor)));
gtk_sctext_insert (GTK_SCTEXT (editor), NULL, NULL, NULL, "\t", 1);
gtk_sctext_thaw (GTK_SCTEXT (editor));
gtk_editable_set_position (GTK_EDITABLE (editor), pos + 1); 
//old pos + tab 
} 
*/ 

void
gtk_editor_install_indenter (GtkEditor *editor, GtkEditorIndFunc ind,
			     guint accel_key, GdkModifierType accel_mod,
			     gpointer user_data)
{
  editor->one_line_ind = ind;
  editor->ind_accel_key = accel_key;
  editor->ind_accel_mod = accel_mod;
  editor->indenter_data = user_data;
}

/* --<event handles>------------------------------------------------------ */
/* reset_idle_event -- keeps track of whether stuff is going on, or whether
 * it's OK to highlight or parenmatch or whatever... */
static void
reset_idle_event (GtkWidget *widget)
{
  GtkEditor *editor = GTK_EDITOR (widget);

  if (editor->idle_time)
    gtk_timeout_remove (editor->idle_time);

  editor->idle_time = gtk_timeout_add (TIME_DELAY, (GtkFunction) idle_event,
				       (gpointer) editor);
}

static gint
idle_event (GtkWidget *widget)
{
  GtkEditor *editor = GTK_EDITOR (widget);

  if (editor->hilite_when_idle)
    hilite_when_idle (editor);

  gtk_editor_parenmatch (GTK_WIDGET(editor));

  editor->idle_time = 0;
  return FALSE;
}

static gint
gtk_editor_key_press (GtkWidget *widget, GdkEventKey *event)
{
  GtkEditor *editor;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_EDITOR (widget), FALSE);

  editor = GTK_EDITOR (widget);

  if (editor->one_line_ind) {
    if ( (!editor->ind_accel_mod || (event->state & editor->ind_accel_mod))
	 && (event->keyval == editor->ind_accel_key) ) {

      gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
      editor->one_line_ind (editor, editor->indenter_data);
      return TRUE;
    }
  }
  
  return FALSE;
}

/* To keep track of changes for hilite_when_idle */
static void
gtk_editor_changed_pre (GtkEditor *editor)
{
  if (editor->hilite_when_idle) {
    /* the first MIN looks silly, but think about deleting the last
     * char in a text, and you will understand. */
    editor->changed_from = MIN (MIN (editor->changed_from,
				     GTK_SCTEXT (editor)->point.index),
				gtk_sctext_get_length (GTK_SCTEXT (editor)) - 1);
  }
}

/* Keeps track of changes for hilite_when_idle, and installs hilite
 * function with a time delay. */
static void
gtk_editor_changed_post (GtkEditor *editor)
{
  if (editor->hilite_when_idle) {
    editor->changed_to = MIN (MAX (editor->changed_to,
				   GTK_SCTEXT (editor)->point.index),
			      gtk_sctext_get_length (GTK_SCTEXT (editor)));

#if 0
    /* something changed...re-highlight when idle */
    if (editor->hilite_time) gtk_timeout_remove (editor->hilite_time);
    editor->hilite_time = gtk_timeout_add (TIME_DELAY, (GtkFunction) hilite_when_idle,
					   (gpointer) editor);
#endif
  }
}
