(**
   A list gadget. Is prepared for handling multible columns. Can be
   used as multi-selection listgadget.
**)

MODULE VOList;

(*
    Implements a list gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VODisplay,
       E  := VOEvent,
       F  := VOFrame,
       G  := VOGUIObject,
       LM := VOListModel,
       O  := VOObject,
       P  := VOPrefs,
       S  := VOScroller,
       V  := VOValue;

CONST
  selectedMsg* = 0;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (P.PrefsDesc)
                 rFrame*,
                 wFrame* : LONGINT; (* the frame to use for the button *)
               END;

  List*         = POINTER TO ListDesc;
  ListDesc*     = RECORD (G.GadgetDesc)
                    prefs       : Prefs;
                    frame       : F.Frame;
                    model-      : LM.ListModel;
                    top,
                    bottom      : LM.ListEntry;
                    vis-,
                    total-,
                    topPos-     : V.ValueModel;

                    scroller    : S.Scroller;

                    multi       : BOOLEAN;
                    readonly    : BOOLEAN;    (* this is a read-only list *)
                    useScroller : BOOLEAN;    (* generate a scroller *)
                  END;

  (* Some messages *)

  SelectedMsg*      = POINTER TO SelectedMsgDesc;
  SelectedMsgDesc*  = RECORD (O.MessageDesc)
                        entry* : LM.ListEntry;
                      END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.rFrame:=F.double3DOut;
    p.wFrame:=F.double3DIn;
  END Init;

  PROCEDURE (l : List) Init*;

  BEGIN
    l.Init^;

    l.prefs:=prefs;

    INCL(l.flags,G.canFocus);

    l.model:=NIL;
    l.top:=NIL;
    l.bottom:=NIL;

    l.scroller:=NIL;

    l.multi:=FALSE;
    l.readonly:=FALSE;
    l.useScroller:=TRUE;

    NEW(l.frame);
    l.frame.Init;
    l.frame.SetFlags({G.horizontalFlex,G.verticalFlex});

    NEW(l.topPos);
    l.topPos.Init;
    l.topPos.SetLongint(1);
    l.AttachModel(l.topPos);

    NEW(l.vis);
    l.vis.Init;
    l.vis.SetLongint(1);

    NEW(l.total);
    l.total.Init;
    l.total.SetLongint(1);

    l.SetBackground(D.tableBackgroundColor);
  END Init;

  PROCEDURE (l : List) SetModel*(model : LM.ListModel);

  BEGIN
    IF l.model#NIL THEN
      l.UnattachModel(l.model);
    END;
    l.model:=model;
    l.top:=model.Get(1);
    l.topPos.SetLongint(1);
    l.AttachModel(model);
  END SetModel;

  PROCEDURE (l: List) SetReadOnly*(readonly : BOOLEAN);

  BEGIN
    l.readonly:=readonly;
  END SetReadOnly;

  PROCEDURE (l : List) UseScroller*(use : BOOLEAN);

  BEGIN
    l.useScroller:=use;
  END UseScroller;

  PROCEDURE (l : List) CalcSize*(display : D.Display);

  BEGIN
    IF l.readonly THEN
      l.SetBackground(D.backgroundColor);
      l.frame.SetInternalFrame(l.prefs.rFrame);
    ELSE
      l.SetBackground(D.tableBackgroundColor);
      l.frame.SetInternalFrame(l.prefs.wFrame);
    END;

    l.frame.CalcSize(display);
    l.width:=l.frame.leftBorder+l.frame.rightBorder;
    l.height:=l.frame.topBorder+l.frame.bottomBorder;

    IF l.useScroller THEN
      NEW(l.scroller);
      l.scroller.Init;
      l.scroller.SetFlags({G.verticalFlex});
      l.scroller.SetModel(l.topPos,l.vis,l.total);
      l.scroller.CalcSize(display);
      INC(l.width,G.MaxLong(3*display.spaceWidth,l.scroller.oWidth));
      INC(l.height,G.MaxLong(3*display.spaceHeight,l.scroller.oHeight));
    ELSE
      INC(l.width,3*display.spaceWidth);
      INC(l.height,3*display.spaceHeight);
    END;

    l.minWidth:=l.width;
    l.minHeight:=l.height;

    l.CalcSize^(display);
  END CalcSize;

  PROCEDURE (l : List) GetEntry(y : LONGINT):LM.ListEntry;

  VAR
    entry  : LM.ListEntry;
    object : G.Object;

  BEGIN
    IF l.top=NIL THEN
      RETURN NIL;
    END;

    entry:=l.top;
    WHILE entry#l.bottom.next DO
      object:=entry.GetObject(1);
      IF (y>=object.y) & (y<=object.y+object.height-1) THEN
        RETURN entry;
      END;
      entry:=entry.next;
    END;
    RETURN NIL;
  END GetEntry;

  PROCEDURE (l : List) GetFocus*(event : E.Event):G.Object;

  VAR
    entry    : LM.ListEntry;
    selected : SelectedMsg;

  BEGIN
    IF ~l.visible OR l.disabled OR l.readonly THEN
      IF l.useScroller THEN
        RETURN l.scroller.GetFocus(event);
      ELSE
        RETURN NIL;
      END;
    END;

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown)
      & l.PointIsIn(event.x,event.y) & (event.qualifier={}) & (event.button=E.button1) THEN
        IF l.useScroller & l.scroller.PointIsIn(event.x,event.y) THEN
          RETURN l.scroller.GetFocus(event);
        ELSE
          entry:=l.GetEntry(event.y);
          IF entry#NIL THEN
            IF l.multi THEN
              entry.Toggle;
              (* Action: State changed *)
            ELSE
              IF l.model.lastSelected#NIL THEN
                l.model.lastSelected.Deselect;
              END;
              entry.Select;
              (* Action: Selected *)
              NEW(selected);
              selected.entry:=entry;
              l.Send(selected,selectedMsg);
            END;
          END;
        END;
        RETURN l;
      END;
    ELSE
    END;

    IF l.useScroller THEN
      RETURN l.scroller.GetFocus(event);
    ELSE
      RETURN NIL;
    END;

  END GetFocus;


  PROCEDURE (l : List) HandleEvent*(event : E.Event):BOOLEAN;

  BEGIN
    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        RETURN ~l.readonly;
      END;
    ELSE
    END;

    RETURN FALSE;
  END HandleEvent;

  PROCEDURE (l : List) HandleFocusEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym : LONGINT;

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF keysym=E.up THEN
        IF l.topPos.GetLongint()>1 THEN
          l.topPos.Dec;
        END;
        RETURN TRUE;
      ELSIF keysym=E.down THEN
        IF l.topPos.GetLongint()<=l.total.GetLongint()-l.vis.GetLongint() THEN
          l.topPos.Inc;
        END;
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleFocusEvent;

  PROCEDURE (l : List) RedrawEntry(entry : LM.ListEntry);

  VAR
    help    : LM.ListEntry;
    current : G.Object;

  BEGIN
    IF ~l.visible THEN
      RETURN;
    END;

    help:=l.top;
    WHILE (help#NIL) & (help#l.bottom.next) DO
      IF help=entry THEN
        l.draw.InstallClip;

        IF l.useScroller THEN
          l.draw.AddRegion(l.x+l.frame.leftBorder,l.y+l.frame.topBorder,
                           l.width-l.frame.leftBorder-l.frame.rightBorder-l.scroller.oWidth,
                           l.height-l.frame.topBorder-l.frame.bottomBorder);
        ELSE
          l.draw.AddRegion(l.x+l.frame.leftBorder,l.y+l.frame.topBorder,
                           l.width-l.frame.leftBorder-l.frame.rightBorder,
                           l.height-l.frame.topBorder-l.frame.bottomBorder);
        END;
        current:=entry.GetObject(1);

        IF entry.selected THEN
          l.draw.mode:={D.selected};
        END;
        current.Draw(current.x,current.y,l.draw);
        l.draw.mode:={};
        l.draw.FreeLastClip;
        RETURN;
      END;
      help:=help.next;
    END;

    IF l.disabled THEN (* This is suboptimal *)
      l.DrawDisabled;
    END;
  END RedrawEntry;


  PROCEDURE (l : List) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    pos,vis : LONGINT;
    current : G.Object;
    entry   : LM.ListEntry;
    exit    : BOOLEAN;

  BEGIN
    l.Draw^(x,y,draw);

    l.frame.Resize(l.width,l.height);
    l.frame.Draw(l.x,l.y,draw);

    IF l.useScroller THEN
      l.scroller.Resize(-1,l.height-l.frame.topBorder-l.frame.bottomBorder);
      l.scroller.Draw(l.x+l.width-l.frame.rightBorder-l.scroller.oWidth,l.y+l.frame.topBorder,draw);
    END;

    l.draw.InstallClip;

    IF l.useScroller THEN
      l.draw.AddRegion(l.x+l.frame.leftBorder,l.y+l.frame.topBorder,
                       l.width-l.frame.leftBorder-l.frame.rightBorder-l.scroller.oWidth,
                       l.height-l.frame.topBorder-l.frame.bottomBorder);

      draw.PushForeground(l.background);
      draw.FillRectangle(l.x+l.frame.leftBorder,l.y+l.frame.topBorder,
                         l.width-l.frame.leftBorder-l.frame.rightBorder-l.scroller.oWidth,
                         l.height-l.frame.topBorder-l.frame.bottomBorder);
      draw.PopForeground;
    ELSE
      l.draw.AddRegion(l.x+l.frame.leftBorder,l.y+l.frame.topBorder,
                       l.width-l.frame.leftBorder-l.frame.rightBorder,
                       l.height-l.frame.topBorder-l.frame.bottomBorder);

      draw.PushForeground(l.background);
      draw.FillRectangle(l.x+l.frame.leftBorder,l.y+l.frame.topBorder,
                         l.width-l.frame.leftBorder-l.frame.rightBorder,
                         l.height-l.frame.topBorder-l.frame.bottomBorder);
      draw.PopForeground;
    END;

    l.top:=l.model.Get(l.topPos.GetLongint());
    l.total.SetLongint(l.model.size);

    pos:=l.frame.topBorder;
    entry:=l.top;
    l.bottom:=l.top;
    vis:=0;
    exit:=FALSE;
    WHILE ~exit DO
      IF entry#NIL THEN
        entry.CalcSize(l.display);
        current:=entry.GetObject(1);
        IF pos+entry.height<=l.height-l.frame.topBorder-l.frame.bottomBorder THEN
          IF l.useScroller THEN
            current.Resize(l.width-l.frame.leftBorder-l.frame.rightBorder-l.scroller.oWidth,-1);
          ELSE
            current.Resize(l.width-l.frame.leftBorder-l.frame.rightBorder,-1);
          END;
          IF entry.selected THEN
            draw.mode:={D.selected};
          END;
          l.CopyBackground(current);
          current.Draw(l.x+l.frame.leftBorder,l.y+pos,draw);
          draw.mode:={};
          INC(pos,entry.height);
          INC(vis);
          entry:=entry.next;
          IF entry#NIL THEN
            l.bottom:=entry;
          END;
        ELSE
          exit:=TRUE;
        END;

      ELSE
        exit:=TRUE;
      END;
    END;

    l.draw.FreeLastClip;

    l.vis.SetLongint(vis);

    IF l.disabled THEN
      l.DrawDisabled;
    END;

  END Draw;

  PROCEDURE (l : List) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF (l.model#NIL) THEN
      l.total.SetLongint(l.model.size);
      IF l.visible & ~l.disabled THEN
        IF msg#NIL THEN
          WITH
            msg : LM.EntryChgdMsg DO
              l.RedrawEntry(msg.entry);
              RETURN;
          ELSE

            l.Redraw;
          END;
        ELSE
          l.Redraw;
        END;
      END;
    END;

  END Resync;

  PROCEDURE (l : List) Hide*;

  BEGIN
    IF l.visible THEN
      l.DrawHide;
      IF l.useScroller THEN
        l.scroller.Hide;
      END;
      l.Hide^;
    END;
  END Hide;

  PROCEDURE (l : List) SetTop*(top : LONGINT);

  BEGIN
    IF (top>=1) & (top<=l.total.GetLongint()) THEN
      l.topPos.SetLongint(top);
    END;
  END SetTop;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOList.