(**
  Module for handling of the various preferences GUIs.
  Offers a flexible settings dialog.
**)

MODULE VOPrefsGUI;

(*
    Implements a unvisible tab gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT B   := VOButton,
       BR  := VOButtonRow,
       D   := VODisplay,
       F   := VOFrame,
       G   := VOGUIObject,
       K   := VOKeyHandler,
       L   := VOList,
       LM  := VOListModel,
       M   := VOMulti,
       O   := VOObject,
       P   := VOPanel,
       PL  := VOPlate,
       PP  := VOPrefsParser,
       PS  := VOPrefsScanner,
       Q   := VOQuickHelp,
       S   := VOSpace,
       T   := VOText,
       U   := VOUtil,
       V   := VOVecImage,
       W   := VOWindow,
       WG  := VOWindowGroup,

       b   := BinaryRider,
              Err,
       f   := Files,
              Channel,
              Rts,
       str := Strings,
       t   := TextRider;

CONST
  prefsNameSize* = 20;

  useId    = 1;
  saveId   = 2;
  closeId  = 3;

TYPE
  PrefsItem*     = POINTER TO PrefsItemDesc;
  PrefsItemDesc* = RECORD (O.MsgObjectDesc)
                     appName- : U.Text;
                     display- : D.Display;
                     window-  : W.Window;
                     next,
                     last     : PrefsItem;
                     name*    : ARRAY prefsNameSize OF CHAR;
                   END;

  Settings*      = POINTER TO SettingsDesc;
  SettingsDesc*  = RECORD ( W.WindowDesc)
                     listModel : LM.ListModel;
                     list      : L.List;
                     multi     : M.Multi;

                     itemList,
                     itemLast  : PrefsItem;

                     exit      : BOOLEAN;
                   END;

  AboutItem     = POINTER TO AboutItemDesc;
  AboutItemDesc = RECORD (PrefsItemDesc);
                  END;


  Sel2Set*     = POINTER TO Sel2SetDesc;
  Sel2SetDesc* = RECORD (O.HandlerDesc)
                 END;

VAR
  settings* : Settings;

  (**
    The name of the directory for all preferences
    files.
  **)

  home-    : ARRAY 256 OF CHAR;
  fileName : ARRAY 256 OF CHAR;


  PROCEDURE (i : PrefsItem) Init*;

  BEGIN
    i.Init^;
  END Init;

  (**
    The standard error handler for the preferences parser.
    It just prints out errors to stderr.
  **)

  PROCEDURE StdErrorProc* (n: INTEGER; pos: LONGINT);

    PROCEDURE Msg(string : ARRAY OF CHAR);

    BEGIN
      Err.String(string);
    END Msg;

  BEGIN
    Err.String(fileName);
    Err.String(":");
    Err.LongInt(pos,0);
    Err.String(":");
    Err.LongInt(n,0);
    Err.String(" Error: ");

    CASE n OF
    |0: Msg("EOF expected")
    |1: Msg("ident expected")
    |2: Msg("string expected")
    |3: Msg("number expected")
    |4: Msg("'BEGIN' expected")
    |5: Msg("';' expected")
    |6: Msg("'END' expected")
    |7: Msg("':=' expected")
    |8: Msg("',' expected")
    |9: Msg("??? expected")
    |10: Msg("this symbol not expected in Parameter")
    |11: Msg("this symbol not expected in Parameter")
    |12: Msg("invalid Parameter")
    |13: Msg("this symbol not expected in Block")
    |14: Msg("this symbol not expected in Block")
    ELSE
      Msg("Unknown error");
    END;
    Err.Ln;
    INC(PS.errors);
  END StdErrorProc;

  (**
    Load the preferences file with the given name. Don't start the
    filename with something like ~/.VisualOberon, VO will prepend that
    itself.

    If the file can be found and has been successfully parsed, it will
    return a pointe to the top node. Use VOPrefsParser.Item.GetEntry
    to get its contents.
  **)

  PROCEDURE (p : PrefsItem) LoadPrefsFile*(name,application : ARRAY OF CHAR):PP.Item;

  VAR
    reader   : b.Reader;
    res      : f.Result;
    length   : LONGINT;
    error    : ARRAY 256 OF CHAR;
    text     : U.Text;
    file     : f.File;

  BEGIN
    COPY(home,fileName);
    str.Append(name,fileName);
    str.Append(".res",fileName);
    file:=f.Old(fileName,{f.read},res);
    IF file#NIL THEN
      reader:=b.ConnectReader(file);
      IF reader#NIL THEN
        length:=reader.Available();
        IF length>0 THEN
          NEW(text,length+1);
          reader.ReadBytes(text^,0,length);
          text[length]:=0X;
          PS.Initialize(text,StdErrorProc);
          PP.Initialize(NIL);
          PP.Parse;
          IF PS.errors=0 THEN
            file.Close;
            file:=NIL;
            RETURN PP.top;
          END;
        END;
      ELSE
        Err.String("Error loading '"); Err.String(fileName); Err.String("': ");
        file.res.GetText(error); Err.String(error); Err.Ln;
      END;
      file.Close;
    ELSIF res.code#f.noSuchFile THEN
      Err.String("Error loading '"); Err.String(fileName); Err.String("': ");
      res.GetText(error); Err.String(error); Err.Ln;
      RETURN NIL;
    END;

    RETURN NIL;
  END LoadPrefsFile;

  PROCEDURE  (p : PrefsItem) ErrorWrongData*(name,value : ARRAY OF CHAR);

  BEGIN
    Err.String("Unknown value for '");
    Err.String(name);
    Err.String("': ");
    Err.String(value);
    Err.Ln;
  END ErrorWrongData;


  (**
    All preferences should have a load-method. The baseclass
    does nothing.
  **)

  PROCEDURE (p : PrefsItem) LoadPrefs*(appName : ARRAY OF CHAR);

  BEGIN
    NEW(p.appName,str.Length(appName)+1);
    COPY(appName,p.appName^);
  END LoadPrefs;

  PROCEDURE (i : PrefsItem) GetObject*():G.Object;

  BEGIN
    RETURN NIL;
  END GetObject;

  PROCEDURE (i : PrefsItem) SetData(window : W.Window; display : D.Display);

  BEGIN
    i.window:=window;
    i.display:=display;
  END SetData;

  PROCEDURE ( p : PrefsItem) SaveItems*(name : ARRAY OF CHAR; top : PP.Item);

  VAR
    file     : f.File;
    writer   : t.Writer;
    res      : f.Result;
    text     : ARRAY 256 OF CHAR;

  BEGIN
    COPY(home,fileName);
    str.Append(name,fileName);
    str.Append(".res",fileName);
    file:=f.New(fileName,{f.write},res);
    IF file#NIL THEN
      writer:=t.ConnectWriter(file);
      IF writer#NIL THEN
        WHILE top#NIL DO
          top.Print(writer,0);
          top:=top.next;
          IF top#NIL THEN
            writer.WriteLn;
          END;
        END;
        IF writer.res#Channel.done THEN
          writer.res.GetText(text);
          Err.String("Error saving '"); Err.String(fileName);
          Err.String("': "); Err.String(text); Err.Ln;
        END;
      ELSE
        file.res.GetText(text);
        Err.String("Error saving '"); Err.String(fileName);
        Err.String("': "); Err.String(text); Err.Ln;
      END;
      file.Close;
    ELSE
      res.GetText(text);
      Err.String("Error saving '"); Err.String(fileName);
      Err.String("': "); Err.String(text); Err.Ln;
    END;
  END SaveItems;

  PROCEDURE (o : PrefsItem) Apply*;

  BEGIN
  END Apply;

  PROCEDURE (o : PrefsItem) Save*;

  BEGIN
  END Save;

  PROCEDURE (o : PrefsItem) Refresh*;

  BEGIN
  END Refresh;

  PROCEDURE (s : Settings) AddItem*(item : PrefsItem);

  BEGIN
    IF s.itemList=NIL THEN
      s.itemList:=item;
    ELSE
      s.itemLast.next:=item;
      item.last:=s.itemLast;
    END;
    s.itemLast:=item;
  END AddItem;

  PROCEDURE (s : Settings) Init*;

  VAR
    about      : AboutItem;
    keyHandler : K.KeyHandler;

  BEGIN
    s.Init^;

    s.SetModal;
    s.SetTitle("VisualOberon - Settings");

    s.exit:=FALSE;

    NEW(keyHandler);
    keyHandler.Init;
    s.AddKeyHandler(keyHandler);


    s.itemList:=NIL;

    NEW(about);
    about.Init;
    s.AddItem(about);
  END Init;

  PROCEDURE (s : Settings) SetDisplay*(display : D.Display);

  VAR
    item : PrefsItem;

  BEGIN
    s.SetDisplay^(display);
    item:=s.itemList;
    WHILE item#NIL DO
      item.SetData(s,s.display);
      item:=item.next;
    END;
  END SetDisplay;

  (**
    If you want the dialog to quit the application, set this to true.

    This is usefull, if the configuration dialog is a standalone
    window.
  **)

  PROCEDURE (s : Settings) ExitOnClose*(exit : BOOLEAN);

  BEGIN
    s.exit:=exit;
  END ExitOnClose;

  PROCEDURE (s : Settings) PreInit*;

  VAR
    top,vert    : P.Panel;
    row         : BR.ButtonRow;
    button      : B.Button;
    msg2Close   : W.Msg2Close;
    space       : S.Space;
    item        : PrefsItem;
    listHandler : Sel2Set;
    wGroup      : WG.WindowGroup;

  BEGIN
    NEW(top);
    top.Init;
    top.SetFlags({G.horizontalFlex,G.verticalFlex});
    top.Set(P.horizontal);

      NEW(s.listModel);
      s.listModel.Init;

      NEW(s.list);
      s.list.Init;
      s.list.SetReadOnly(FALSE);
      s.list.SetFlags({G.verticalFlex});
      s.list.SetMinWidth(G.sizeFontRel,prefsNameSize);
      s.list.SetMinHeight(G.sizeFontRel,20);
      s.list.SetModel(s.listModel);
      s.keyHandler.AddFocusObject(s.list);

    top.Add(s.list);

      NEW(space);
      space.Init;
      space.Set(FALSE,S.normal);
    top.Add(space);

      NEW(vert);
      vert.Init;
      vert.SetFlags({G.horizontalFlex,G.verticalFlex});
      vert.Set(P.vertical);

        NEW(s.multi);
        s.multi.Init;
        s.multi.SetFlags({G.horizontalFlex,G.verticalFlex});
        NEW(listHandler);
        listHandler.destination:=s.multi;
        s.list.AddHandler(listHandler,L.selectedMsg);

        item:=s.itemList;
        WHILE item#NIL DO
          s.listModel.Append(LM.NewTextEntry(item.name));
          s.multi.Add(item.GetObject());
          item:=item.next;
        END;

        s.listModel.Select(1);
      vert.Add(s.multi);

        NEW(space);
        space.Init;
        space.Set(TRUE,S.normal);
      vert.Add(space);

        NEW(row);
        row.Init;
        row.SetFlags({G.horizontalFlex});


          NEW(button);
          button.Init;
          button.SetId(saveId);
          button.SetFlags({G.horizontalFlex});
          button.SetLabelText("_Save",s.keyHandler);
          button.Forward(B.pressedMsg,s);
          s.keyHandler.AddFocusObject(button);
        row.Add(button);

          NEW(button);
          button.Init;
          button.SetId(useId);
          button.SetFlags({G.horizontalFlex});
          button.SetLabelText("_Apply",s.keyHandler);
          button.Forward(B.pressedMsg,s);
          s.keyHandler.AddFocusObject(button);
        row.Add(button);

          NEW(button);
          button.Init;
          button.SetId(closeId);
          button.SetFlags({G.horizontalFlex});
          button.SetLabelText("_Close^",s.keyHandler);
          button.Forward(B.pressedMsg,s);
          NEW(msg2Close);
          msg2Close.destination:=s;
          button.AddHandler(msg2Close,B.pressedMsg);
          s.keyHandler.AddFocusObject(button);
        row.Add(button);

      vert.Add(row);
    top.Add(vert);

    NEW(wGroup);
    wGroup.Init;
    wGroup.Set(NIL,top,TRUE);

    s.SetTop(wGroup);

    NEW(msg2Close);
    msg2Close.destination:=s;
    s.AddHandler(msg2Close,W.closeMsg);

    s.PreInit^;
  END PreInit;

  PROCEDURE (s : Settings) Open*;

  VAR
    item : PrefsItem;

  BEGIN
    s.Open^;

    item:=s.itemList;
    WHILE item#NIL DO
      item.Refresh;
      item:=item.next;
    END;
  END Open;

  PROCEDURE (s : Settings) LoadPrefs*(appName : ARRAY OF CHAR);

  VAR
    item : PrefsItem;

  BEGIN
    item:=s.itemList;
    WHILE item#NIL DO
      item.LoadPrefs(appName);
      item:=item.next;
    END;
  END LoadPrefs;

  PROCEDURE (s : Settings) ApplyPrefs*;

  VAR
    item : PrefsItem;

  BEGIN
    item:=s.itemList;
    WHILE item#NIL DO
      item.Apply;
      item:=item.next;
    END;
  END ApplyPrefs;

  PROCEDURE (s : Settings) SavePrefs*;

  VAR
    item : PrefsItem;

  BEGIN
    item:=s.itemList;
    WHILE item#NIL DO
      item.Save;
      item:=item.next;
    END;
  END SavePrefs;

  PROCEDURE (s : Settings) Receive*(message : O.Message);

  BEGIN
    WITH
      message : B.PressedMsg DO
        CASE message.source.id OF
          closeId:
            IF s.exit THEN
              s.display.Exit;
            END;
        | useId:
            s.ApplyPrefs;
            s.display.ReinitWindows;
        | saveId:
            s.ApplyPrefs;
            s.display.ReinitWindows;
            s.SavePrefs;
        END;
    | message : W.CloseMsg DO
        IF s.exit THEN
          s.display.Exit;
        ELSE
          s.Close;
        END;
    ELSE
      s.Receive^(message);
    END;
  END Receive;

  PROCEDURE (a : AboutItem) Init*;

  BEGIN
    a.name:="About";
  END Init;


  PROCEDURE (a : AboutItem) GetObject*():G.Object;

  BEGIN
    RETURN T.MakeLeftText("\e\c\e\s\e\9VisualOberon\e\p\n\en\n\e\cCopyright 1997\nTim Teulings\n(rael@edge.ping.de)");
  END GetObject;


  PROCEDURE (h : Sel2Set) Convert*(message : O.Message):O.Message;

  VAR
    new : M.SetMsg;

  BEGIN
    NEW(new);
    new.pos:=message(L.SelectedMsg).entry.GetPos();
    RETURN new;
  END Convert;

  PROCEDURE PrefsCallback (appName : ARRAY OF CHAR; display : D.Display);

  BEGIN
    settings.SetDisplay(display);
    settings.LoadPrefs(appName);
  END PrefsCallback;

  PROCEDURE CreateFrameButton*(value : LONGINT; VAR frame : F.Frame; display : D.Display;
                               destination : O.MsgObject; keyHandler : K.KeyHandler;
                               tooltip : ARRAY OF CHAR):B.Button;

  VAR
    button    : B.Button;
    plate     : PL.Plate;
    msg2Open  : W.Msg2Open;

  BEGIN
    NEW(button);
    button.Init;

      NEW(plate);
      plate.Init;
        NEW(frame);
        frame.Init;
        frame.SetInternalFrame(value);
        frame.SetMinWidth(G.sizeFontRel,3);
        frame.SetMinHeight(G.sizeFontRel,3);
        plate.Add(frame);

        button.SetImage(plate);
        NEW(msg2Open);
        msg2Open.destination:=destination;
        button.AddHandler(msg2Open,B.pressedMsg);
        button.SetHelpObject(Q.InitHelp(display,T.MakeLeftText(tooltip),button));
        keyHandler.AddFocusObject(button);

   RETURN button;
 END CreateFrameButton;

 PROCEDURE CreateImageButton*(value : LONGINT; VAR image : V.VecImage; display : D.Display;
                               destination : O.MsgObject; keyHandler : K.KeyHandler;
                               tooltip : ARRAY OF CHAR):B.Button;

  VAR
    button    : B.Button;
    plate     : PL.Plate;
    msg2Open  : W.Msg2Open;

  BEGIN
    NEW(button);
    button.Init;

      NEW(plate);
      plate.Init;
        NEW(image);
        image.Init;
        image.Set(value);
        image.SetMinWidth(G.sizeFontRel,3);
        image.SetMinHeight(G.sizeFontRel,3);
      plate.Add(image);

      button.SetImage(plate);
      NEW(msg2Open);
      msg2Open.destination:=destination;
      button.AddHandler(msg2Open,B.pressedMsg);
      button.SetHelpObject(Q.InitHelp(display,T.MakeLeftText(tooltip),button));
      keyHandler.AddFocusObject(button);

   RETURN button;
 END CreateImageButton;


BEGIN
  Rts.GetUserHome(home,"");
  str.Append("/.VisualOberon/",home);


  NEW(settings);
  settings.Init;

  D.prefsCallback:=PrefsCallback;
END VOPrefsGUI.
