(**
  Implements a statusline group object.
**)

MODULE VOStatusLine;

(*
    Implements a statusline group object.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D := VODisplay,
       F := VOFrame,
       G := VOGUIObject,
       U := VOUtil;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 iFrame*,
                 oFrame*  : LONGINT;
               END;

  StatusLine*     = POINTER TO StatusLineDesc;
  StatusLineDesc* = RECORD (G.GroupDesc)
                      prefs  : Prefs;
                      frame,
                      frame2 : F.Frame;
                      xRest,
                      yRest  : LONGINT;
                    END;


VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.iFrame:=F.single3DIn;
    p.oFrame:=F.single3DOut;
  END Init;

  PROCEDURE (p : Prefs) SetPrefs(s : StatusLine);

  BEGIN
    s.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      s.SetBackgroundObject(p.background.Copy());
      s.backgroundObject.source:=s;
    END;
  END SetPrefs;

  PROCEDURE (s : StatusLine) Init*;

  BEGIN
    s.Init^;

    prefs.SetPrefs(s);

    s.frame:=NIL;
    s.frame2:=NIL;
  END Init;

  PROCEDURE (s : StatusLine) CalcSize*(display : D.Display);

  VAR
    object : G.Object;

  BEGIN
    s.width:=0;
    s.height:=0;
    s.minWidth:=0;
    s.minHeight:=0;

    object:=s.list;
    WHILE object#NIL DO
      s.CopyBackground(object);
      object.CalcSize(display);

      INC(s.width,object.oWidth);
      s.height:=G.MaxLong(s.height,object.oHeight);

      INC(s.minWidth,object.oMinWidth);
      s.minHeight:=G.MaxLong(s.minHeight,object.oMinHeight);

      object:=object.next;
    END;


    NEW(s.frame);
    s.frame.Init;
    s.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
    s.frame.SetInternalFrame(s.prefs.oFrame);
    s.frame.CalcSize(display);

    NEW(s.frame2);
    s.frame2.Init;
    s.frame2.SetFlags({G.horizontalFlex,G.verticalFlex});
    s.frame2.SetInternalFrame(s.prefs.iFrame);
    s.frame2.CalcSize(display);

    s.xRest:=s.frame.leftBorder+s.frame.rightBorder+
             s.count*(s.frame2.leftBorder+s.frame2.rightBorder)+
             s.count*display.spaceWidth+           (* Width/2 between object and frame *)
             (s.count+1)*display.spaceWidth DIV 2; (* space between objects *)
    s.yRest:=display.spaceHeight+
             s.frame.topBorder+s.frame.bottomBorder+
             s.frame2.topBorder+s.frame2.bottomBorder;
    INC(s.width,s.xRest);
    INC(s.minWidth,s.xRest);
    INC(s.height,s.yRest);
    INC(s.minHeight,s.yRest);

    s.CalcSize^(display);
  END CalcSize;

  PROCEDURE (s : StatusLine) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    pos,
    minSize,
    FCount,
    old      : LONGINT;
    object   : G.Object;

  BEGIN
    s.Draw^(x,y,draw);
    s.DrawBackground(s.x,s.y,s.width,s.height);

    (* Find min size in order direction *)

    minSize:=0;
    object:=s.list;
    WHILE object#NIL DO
      INC(minSize,object.oWidth);
      object:=object.next;
    END;

    LOOP
      FCount:=0;
      object:=s.list;
      WHILE object#NIL DO
        IF object.CanResize(s.width-s.xRest>minSize,TRUE) THEN
          INC(FCount);
        END;
        object:=object.next;
      END;

      IF FCount=0 THEN
        EXIT;
      END;

      object:=s.list;
      WHILE object#NIL DO
        IF object.CanResize(s.width-s.xRest>minSize,TRUE) THEN
          old:=object.oWidth;
          object.Resize(object.oWidth + U.UpDiv(s.width-s.xRest-minSize,FCount),s.height-s.yRest);
          INC(minSize,object.oWidth-old);
          DEC(FCount);
        END;
        object:=object.next;
      END;

      IF minSize=s.width-s.xRest THEN
        EXIT;
      END;
    END;

    s.frame.Resize(s.width,s.height);
    s.frame.Draw(s.x,s.y,draw);

    pos:=s.x+s.frame.leftBorder+s.display.spaceWidth DIV 2;

    object:=s.list;
    WHILE object#NIL DO
      s.frame2.Resize(object.oWidth+s.display.spaceWidth+
                      s.frame2.leftBorder+s.frame2.rightBorder,
                      s.height-s.display.spaceHeight);
      s.frame2.Draw(pos,s.y+(s.height-s.frame2.oHeight) DIV 2,draw);
      INC(pos,s.frame2.leftBorder+s.display.spaceWidth DIV 2);
      object.Draw(pos,s.y+(s.height-object.oHeight) DIV 2,draw);
      INC(pos,object.oWidth+s.frame2.rightBorder+s.display.spaceWidth+s.frame2.leftBorder);
      object:=object.next;
    END;
  END Draw;

  PROCEDURE (s : StatusLine) Refresh*(x,y,w,h : LONGINT);

  BEGIN
    (*
      We have to store all frames to get a properly working
      smarter refresh.

      Because we don't have, we simply redraw it all again.
    *)
    s.Draw(s.x,s.y,s.draw);
  END Refresh;

  PROCEDURE (s : StatusLine) Hide*;

  VAR
    help : G.Object;

  BEGIN
    IF s.visible THEN
      s.frame.Hide;
      help:=s.list;
      WHILE help#NIL DO
        help.Hide;
        help:=help.next;
      END;
      s.Hide^;
    END;
  END Hide;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOStatusLine.