/*
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Brian W. Barrett, Arun F. Rodrigues, Jeffrey M. Squyres,
 * 	 and Andrew Lumsdaine
 *
 * This file is part of XMPI
 *
 * You should have received a copy of the License Agreement for XMPI 
 * along with the software; see the file LICENSE.  If not, contact 
 * Office of Research, University of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 *
 * Additional copyrights may follow.

 *
 *	$Id: xmpi_ctl.c,v 1.4 1999/11/11 05:28:55 arodrig6 Exp $
 *
 *	Function:	- control panel for main window
 */

#define _NO_PROTO

#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/SelectioB.h>

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#include "xmpi.h"

#include "Pixmaps/wiper.xpm"
#include "Pixmaps/camera.xpm"
#include "Pixmaps/kiviat.xpm"
#include "Pixmaps/matrix.xpm"
#include "Pixmaps/trace.xpm"
#include "Pixmaps/rerun.xpm"

/*
 * global functions
 */
Widget			xmpi_ctl_build();
void			xmpi_ctl_clean();
void			xmpi_ctl_setinfo();
void			xmpi_ctl_resetinfo();
void			xmpi_ctl_setapp();
void			xmpi_ctl_setsensitive __ARGS((int , Boolean));

/*
 * external functions
 */
extern int		stoi();

/*
 * local functions
 */
static void		ctl_snap_cb();
static void		ctl_rerun_cb();
static void		ctl_trview_cb();
static void		ctl_trdump_cb();
static void		ctl_trexpress_cb();
static void		ctl_kiv_cb();
static void		ctl_clean_cb();
static void		ctl_matrix_cb();
static void		ctl_app_cb();
static void		ctl_trace_cb();
static void		ctl_opt_cb();
static void		ctl_help_cb();
static Widget		ctl_create_buttons();

/*
 * external variables
 */
extern int		xmpi_do_rerun;		/* re-running application? */
extern int		xmpi_is_express;	/* current appl. trace view? */
extern int		xmpi_do_express;	/* creating current app view */
extern int		xmpi_app_nprocs;	/* number of procs in appl. */
extern struct _gps	*xmpi_app_procs;	/* appl. GPS array */

/*
 * local variables
 */
static Widget		info_w;			/* information label */
static Widget		snap_b;			/* snapshot button */
static Widget		rerun_b;		/* rerun button */
static Widget		express_b;		/* trace express button */
static Widget		kiv_b;			/* kiviat view button */
static Widget		clean_b;		/* clean button */
static Widget		matrix_b;		/* matrix button */
static Widget		matrix_m;		/* matrix menu widget */
static Widget		snap_m;			/* snap menu widget */
static Widget		rerun_m;		/* rerun menu widget */
static Widget		dump_m;			/* trace dump menu widget */

static Widget		express_m;		/* trace express menu widget */
static Widget		kiv_m;			/* kiviat menu widget */
static Widget		clean_m;		/* clean menu widget */
static char		appschema[XMPI_PATHMAX]
				= "<None>";	/* application schema */

/*
 * button info messages
 */
static char		*snap_msg = "Take a system snap-shot";
static char		*rerun_msg = "Re-run current application";
static char		*exp_msg = "Dump and view trace information";
static char		*kiv_msg = "View Kiviat information";
static char		*clean_msg = "Clean the system";
static char		*matrix_msg = "View message sources";

/*
 *	xmpi_ctl_build
 *
 *	Function:	- creates widgets for xmpi control panel
 *	Accepts:	- parent widget
 *	Returns:	- created widget
 */
Widget
xmpi_ctl_build(parent_w)

Widget			parent_w;

{
	Widget		menubar_w;		/* menu bar */
	Widget		mgr_w;			/* control area manager */
	Widget		rc_w;			/* buttons row column */
	Widget		w;			/* favourite widget */
	XmString	str0, str1, str2, str3;	/* various strings */
	XmString	str4, str5;		/* various strings */
	
	mgr_w = XtVaCreateWidget("ctl_mgr", xmFormWidgetClass, parent_w, NULL);
/*
 * Create the menu bar.
 */
	str0 = XmStringCreateSimple("Application");
	str1 = XmStringCreateSimple("Trace");
	str2 = XmStringCreateSimple("Options");
	str3 = XmStringCreateSimple("Help");

	menubar_w = XmVaCreateSimpleMenuBar(mgr_w, "ctl_bar",
			XmVaCASCADEBUTTON, str0, 'A',
			XmVaCASCADEBUTTON, str1, 'T',
			XmVaCASCADEBUTTON, str2, 'O',
			XmVaCASCADEBUTTON, str3, 'H',
			XmNtopAttachment, XmATTACH_FORM,
			XmNleftAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			NULL);

	XmStringFree(str0);
	XmStringFree(str1);
	XmStringFree(str2);
	XmStringFree(str3);

	w = XtNameToWidget(menubar_w, "button_3");
	if (w) XtVaSetValues(menubar_w, XmNmenuHelpWidget, w, NULL);
/*
 * Create the pulldown menu for "Application".
 */
	str0 = XmStringCreateSimple("Build&Run...");
	str1 = XmStringCreateSimple("Browse&Run...");
	str2 = XmStringCreateSimple("Snapshot");
	str3 = XmStringCreateSimple("Rerun");
	str4 = XmStringCreateSimple("Clean");
	str5 = XmStringCreateSimple("Quit");
	
	w = XmVaCreateSimplePulldownMenu(menubar_w, "App_menu",
			0, ctl_app_cb,
			XmVaPUSHBUTTON, str0, 'B', NULL, NULL,
			XmVaPUSHBUTTON, str1, 'o', NULL, NULL,
			XmVaPUSHBUTTON, str2, 'S', NULL, NULL,
			XmVaPUSHBUTTON, str3, 'R', NULL, NULL,
			XmVaPUSHBUTTON, str4, 'C', NULL, NULL,
			XmVaSEPARATOR,
			XmVaPUSHBUTTON, str5, 'Q', NULL, NULL,
			NULL);
	snap_m = XtNameToWidget(w, "button_2");
	rerun_m = XtNameToWidget(w, "button_3");
	clean_m = XtNameToWidget(w, "button_4");
/*
 * XmVaSEPARATOR seems buggy.  It makes following buttons forget labels.
 */
	w = XtNameToWidget(w, "button_5");
	if (w) XtVaSetValues(w, XmNlabelString, str5, XmNmnemonic, 'Q', NULL);

	XmStringFree(str0);
	XmStringFree(str1);
	XmStringFree(str2);
	XmStringFree(str3);
	XmStringFree(str4);
	XmStringFree(str5);
/*
 * Create the pulldown menu for "Trace".
 */
	str0 = XmStringCreateSimple("Dump...");
	str1 = XmStringCreateSimple("View...");
	str2 = XmStringCreateSimple("Express");
	str3 = XmStringCreateSimple("Kiviat");
	str4 = XmStringCreateSimple("Matrix");

	w = XmVaCreateSimplePulldownMenu(menubar_w, "Trace_menu",
			1, ctl_trace_cb,
			XmVaPUSHBUTTON, str0, 'D', NULL, NULL,
			XmVaPUSHBUTTON, str1, 'V', NULL, NULL,
			XmVaPUSHBUTTON, str2, 'E', NULL, NULL,
			XmVaPUSHBUTTON, str3, 'K', NULL, NULL,
			XmVaPUSHBUTTON, str4, 'M', NULL, NULL,
			NULL);

	dump_m = XtNameToWidget(w, "button_0");
	XtNameToWidget(w, "button_1");
	express_m = XtNameToWidget(w, "button_2");
	kiv_m = XtNameToWidget(w, "button_3");
	matrix_m = XtNameToWidget(w, "button_4");

	XmStringFree(str0);
	XmStringFree(str1);
	XmStringFree(str2);
	XmStringFree(str3);
	XmStringFree(str4);
/*
 * Create the pulldown menu for "Options".
 */
	str0 = XmStringCreateSimple("Runtime...");

	XmVaCreateSimplePulldownMenu(menubar_w, "Opt_menu",
			2, ctl_opt_cb,
			XmVaPUSHBUTTON, str0, 'R', NULL, NULL,
			NULL);

	XmStringFree(str0);
/*
 * Create the pulldown menu for "Help".
 */
	str0 = XmStringCreateSimple("About XMPI");
	str1 = XmStringCreateSimple("Help");

	w = XmVaCreateSimplePulldownMenu(menubar_w, "Help_menu",
			3, ctl_help_cb,
			XmVaPUSHBUTTON, str0, 'A', NULL, NULL,
			XmVaPUSHBUTTON, str1, 'H', NULL, NULL,
			NULL);

	w = XtNameToWidget(w, "button_1");

	if (strlen(app_res.ap_helpcmd) == 0) {
		XtSetSensitive(w, False);
	}
		
	XmStringFree(str0);
	XmStringFree(str1);

	XtManageChild(menubar_w);
/*
 * Create the row of control buttons.
 */
	rc_w = ctl_create_buttons(mgr_w);

	XtVaSetValues(rc_w,
			XmNtopAttachment, XmATTACH_WIDGET,
			XmNtopWidget, menubar_w,
			XmNtopOffset, 2,
			XmNleftAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			NULL);

	xmpi_add_pophelp(snap_b, snap_msg);
	xmpi_add_pophelp(express_b, exp_msg);
	xmpi_add_pophelp(kiv_b, kiv_msg);
	xmpi_add_pophelp(matrix_b, matrix_msg);
	xmpi_add_pophelp(rerun_b, rerun_msg);
	xmpi_add_pophelp(clean_b, clean_msg);

        xmpi_ctl_setsensitive(XMPI_BSNAPSHOT, False);
	xmpi_ctl_setsensitive(XMPI_BTRACE, False);
	xmpi_ctl_setsensitive(XMPI_BKIVIAT, False);
	xmpi_ctl_setsensitive(XMPI_BMATRIX, False);
	xmpi_ctl_setsensitive(XMPI_BRERUN , False);

	XtManageChild(rc_w);
/*
 * Build the information area.
 */
	info_w = xmpi_mklabel(mgr_w, "info_lbl",
			XmALIGNMENT_BEGINNING, False);
	XtVaSetValues(info_w, XmNleftAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			XmNtopAttachment, XmATTACH_WIDGET,
			XmNtopWidget, rc_w,
			XmNbottomAttachment, XmATTACH_FORM,
			XmNbottomOffset, 2, NULL);

	xmpi_ctl_resetinfo();

	XtManageChild(mgr_w);

	return(mgr_w);
}

/*
 *	ctl_snap_cb
 *
 *	Function:	- takes a snapshot of current application state
 *			- invokes LAM functions
 */
static void
ctl_snap_cb()

{
	xmpi_vw_update(1);
}

/*
 *	ctl_rerun_cb
 *
 *	Function:	- rerun current application
 *	Accepts:	- widget
 */
static void
ctl_rerun_cb(w)

Widget			w;

{
	if (strcmp(appschema, "<None>") == 0) {
		errno = 0;
		xmpi_error(w, "No previous application.");
	} else if (strcmp(appschema, "<custom built>") == 0) {
		xmpi_run_rerun();
	} else {
		xmpi_do_rerun = 1;
		xmpi_run(appschema, (Widget) 0);
	}
}

/*
 *	ctl_trview_cb
 *
 *	Function:	- activate trace viewing
 *	Accepts:	- widget
 */
static void
ctl_trview_cb(w)

Widget			w;

{
	xmpi_do_express = 0;
	xmpi_tr_file(xmpi_wmshell(w));
}

/*
 *	ctl_trdump_cb
 *
 *	Function:	- activate trace dump
 *	Accepts:	- widget
 */
static void
ctl_trdump_cb(w)

Widget			w;

{
	xmpi_tr_dump(xmpi_wmshell(w));
}

/*
 *	ctl_trexpress_cb
 *
 *	Function:	- activate express trace dump and view
 *	Accepts:	- widget
 */
static void
ctl_trexpress_cb(w)

Widget			w;

{
	xmpi_do_express = 1;
	xmpi_tr_express(xmpi_wmshell(w));
}

/*
 *	ctl_kiv_cb
 *
 *	Function:	- activate kiviat viewing
 *	Accepts:	- widget
 */
static void
ctl_kiv_cb(w)

Widget			w;

{
	xmpi_kv_popup();
}

/*
 *	ctl_clean_cb
 *
 *	Function:	- clean system
 *	Accepts:	- widget
 */
static void
ctl_clean_cb(w)

Widget			w;

{
	xmpi_ctl_clean();
}

/*
 *	ctl_matrix_cb
 *
 *	Function:	- activate message matrix viewing
 *	Accepts:	- widget
 */
static void
ctl_matrix_cb(w)

Widget			w;

{
	xmpi_mat_popup();
}

/*
 *	ctl_app_cb
 *
 *	Function:	- invokes items from the Application menu
 *	Accepts:	- menu widget
 *			- selected menu item
 */
static void
ctl_app_cb(w, item)

Widget			w;
int			item;

{
	switch(item) {

	case 0:
#ifdef HPUX
		xmpi_hp_run_dialog(xmpi_wmshell(w));
#else		
		xmpi_run_dialog(xmpi_wmshell(w));
#endif		
		break;

	case 1:
		xmpi_aschema_dialog(xmpi_wmshell(w));
		break;

	case 2:
		ctl_snap_cb();
		break;

	case 3:
		ctl_rerun_cb(w);
		xmpi_ctl_resetinfo();
		break;

	case 4:
		ctl_clean_cb(w);
		break;
	
	case 5:
		xmpi_bail();
		break;
	}
}

/*
 *	ctl_trace_cb
 *
 *	Function:	- invokes items from the Trace menu
 *	Accepts:	- menu widget
 *			- selected menu item
 */
static void
ctl_trace_cb(w, item)

Widget			w;
int			item;

{
	switch(item) {

	case 0:
		ctl_trdump_cb(w);
		break;
		
	case 1:
		ctl_trview_cb(w);
		break;

	case 2:
		ctl_trexpress_cb(w);
		break;

	case 3:
		ctl_kiv_cb(w);
		break;
		
	case 4:
		ctl_matrix_cb(w);
		break;
	}
}

/*
 *	ctl_opt_cb
 *
 *	Function:	- invokes items from the Options menu
 *	Accepts:	- menu widget
 *			- selected menu item
 */
static void
ctl_opt_cb(w, item)

Widget			w;
int			item;

{
	switch(item) {

	case 0:
#ifdef HPUX		
		xmpi_hp_options_set(xmpi_wmshell(w));
#else
		xmpi_options_set(xmpi_wmshell(w));
#endif		
		break;
	}
}

/*
 *	ctl_help_cb
 *
 *	Function:	- invokes items from the Help menu
 */
static void
ctl_help_cb(w, item)

Widget			w;
int			item;

{
	switch(item) {

	case 0:
		xmpi_about(xmpi_wmshell(w));
		break;

	case 1:
		xmpi_help();
		break;
	}
}

/*
 *	xmpi_ctl_clean
 *
 *	Function:	- kills the application and cleans up the debris
 */
void
xmpi_ctl_clean()

{
/*
 * In database mode we don't clear the view window etc. but only
 * run lamclean to blow away traces and so on. 
 */
	xmpi_ctl_setinfo("Cleaning the system...");
	xmpi_flush();
	xmpi_busy();

	if (xmpi_sys_kill(xmpi_app_procs, xmpi_app_nprocs)) {
		xmpi_error(xmpi_shell, "Killing application");
	}

	if (!xmpi_vw_dbmode()) {
		if (!xmpi_do_rerun) {
			xmpi_ctl_setsensitive(XMPI_BSNAPSHOT, False);
			xmpi_ctl_setsensitive(XMPI_BMATRIX, False);
			xmpi_mat_destroy();
			xmpi_dt_clear();
			xmpi_vw_clear();
		}

	}

	if (xmpi_app_procs) {
		free((char *) xmpi_app_procs);
		xmpi_app_procs = 0;
		xmpi_app_nprocs = 0;
	}

	xmpi_ctl_setsensitive(XMPI_BTRACE, False);
	xmpi_ctl_resetinfo();
	xmpi_unbusy();
}

/*
 *	xmpi_ctl_setinfo
 *
 *	Function:	- set info label
 *	Accepts:	- string
 */
void
xmpi_ctl_setinfo(info)

char			*info;

{
	xmpi_setlabel(info_w, info);
	XmUpdateDisplay(info_w);
}

/*
 *	xmpi_ctl_resetinfo
 *
 *	Function:	- reset info label
 */
void
xmpi_ctl_resetinfo()

{
	char		fmtbuf[80];
	char		*app;
	char		*info = "<None>";
/*
 * Strip paths.
 */
	if (xmpi_info[0]) {
		info = strrchr(xmpi_info, STRDIR);
		info = info ? info + 1 : xmpi_info;
	}

	app = strrchr(appschema, STRDIR);
	app = app ? app + 1 : appschema;
	
	sprintf(fmtbuf, "%s: %s   App: %s", app_res.ap_infolbl, info, app);

	xmpi_setlabel(info_w, fmtbuf);
	XmUpdateDisplay(info_w);
}

/*
 *	xmpi_ctl_setapp
 *
 *	Function:	- set application schema
 *	Accepts:	- string
 */
void
xmpi_ctl_setapp(string)

char			*string;

{
	strcpy(appschema, string);
	xmpi_ctl_resetinfo();
}

/*
 *	xmpi_ctl_setsensitive
 *
 *	Function:	- set button/menu sensitivity for an action
 *	Accepts:	- action
 *			- state to set to
 */
#ifdef __STDC__
void
xmpi_ctl_setsensitive(int button, Boolean state)
#else
void
xmpi_ctl_setsensitive(button, state)

int			button;
Boolean			state;
#endif
{
	switch(button) {
		
	case XMPI_BSNAPSHOT:
		XtSetSensitive(snap_b, state);
		XtSetSensitive(snap_m, state);
		break;

	case XMPI_BRERUN:
		XtSetSensitive(rerun_b, state);
		XtSetSensitive(rerun_m, state); 
		break;

	case XMPI_BTRACE:
		XtSetSensitive(dump_m, state);
		XtSetSensitive(express_b, state);
		XtSetSensitive(express_m, state);
		break;

	case XMPI_BKIVIAT:
		XtSetSensitive(kiv_b, state);
		XtSetSensitive(kiv_m, state);
		break;

	case XMPI_BCLEAN:
		XtSetSensitive(clean_b, state);
		XtSetSensitive(clean_m, state);
		break;

	case XMPI_BMATRIX:
		XtSetSensitive(matrix_b, state);
		XtSetSensitive(matrix_m, state);
		break;
	}
}

/*
 *	free_button_pixmaps
 *
 *	Function:	- free buttons pixmap resources
 *	Accepts:	- button
 */
static void
free_button_pixmaps(button)

Widget			button;

{
	Pixmap		pix;			/* pixmap */
	Pixmap		ipix;			/* insensitive pixmap */

	if (button) {
		XtVaGetValues(button,
			XmNlabelPixmap, &pix,
			XmNlabelInsensitivePixmap, &ipix,
			NULL);

		if (pix) {
			XFreePixmap(XtDisplay(button), pix);
		}
		if (ipix) {
			XFreePixmap(XtDisplay(button), ipix);
		}
	}
}

/*
 *	ctl_create_buttons
 *
 *	Function:	- create control buttons
 *	Accepts:	- parent widget
 *	Returns:	- row column widget containing the buttons
 */
static Widget
ctl_create_buttons(parent_w)

Widget			parent_w;

{
	XmString	xstr;
	Widget		rc_w;

	rc_w = XtVaCreateWidget("ctl_brc",
			xmRowColumnWidgetClass, parent_w,
			XmNorientation, XmHORIZONTAL,
			XmNnumColumns, 1,
			NULL);

	snap_b = xmpi_mkcolpixbutton(rc_w, ctl_snap_cb, NULL,
			NULL, snap_msg, NULL, snap_msg,
			camera_xpm, True);

	express_b = xmpi_mkcolpixbutton(rc_w, ctl_trexpress_cb, NULL,
			NULL, exp_msg, NULL, exp_msg,
			trace_xpm, True);

	kiv_b = xmpi_mkcolpixbutton(rc_w, ctl_kiv_cb, NULL,
			NULL, kiv_msg, NULL, kiv_msg,
			kiviat_xpm, True);

	matrix_b = xmpi_mkcolpixbutton(rc_w, ctl_matrix_cb, NULL,
			NULL, matrix_msg, NULL, matrix_msg,
			matrix_xpm, True);

	rerun_b = xmpi_mkcolpixbutton(rc_w, ctl_rerun_cb, NULL,
			NULL, rerun_msg, NULL, rerun_msg,
			rerun_xpm, True);

	clean_b = xmpi_mkcolpixbutton(rc_w, ctl_clean_cb, NULL,
			NULL, clean_msg, NULL, clean_msg,
			wiper_xpm, True);

	if (snap_b && express_b && kiv_b && matrix_b && rerun_b && clean_b) {
		return(rc_w);
	}
/*
 * Couldn't create one of the pixmaps.  Free everything created above and
 * start over using text buttons. 
 */
	free_button_pixmaps(snap_b);
	free_button_pixmaps(express_b);
	free_button_pixmaps(kiv_b);
	free_button_pixmaps(matrix_b);
	free_button_pixmaps(rerun_b);
	free_button_pixmaps(clean_b);

	XtDestroyWidget(rc_w);

	rc_w = XtVaCreateWidget("ctl_brc",
			xmRowColumnWidgetClass, parent_w,
			XmNorientation, XmHORIZONTAL,
			XmNnumColumns, 1,
			NULL);

	snap_b = xmpi_mkpixbutton(rc_w, ctl_snap_cb, NULL,
			NULL, snap_msg, NULL, snap_msg,
			(void *) 0, 0, 0, True);

	express_b = xmpi_mkpixbutton(rc_w, ctl_trexpress_cb, NULL,
			NULL, exp_msg, NULL, exp_msg,
			(void *) 0, 0, 0, True);

	kiv_b = xmpi_mkpixbutton(rc_w, ctl_kiv_cb, NULL,
			NULL, kiv_msg, NULL, kiv_msg,
			(void *) 0, 0, 0, True);

	matrix_b = xmpi_mkpixbutton(rc_w, ctl_matrix_cb, NULL,
			NULL, matrix_msg, NULL, matrix_msg,
			(void *) 0, 0, 0, True);

	rerun_b = xmpi_mkpixbutton(rc_w, ctl_rerun_cb, NULL,
			NULL, rerun_msg, NULL, rerun_msg,
			(void *) 0, 0, 0, True);

	clean_b = xmpi_mkpixbutton(rc_w, ctl_clean_cb, NULL,
			NULL, clean_msg, NULL, clean_msg,
			(void *) 0, 0, 0, True);

	xstr = XmStringCreateSimple("Snapshot");
	XtVaSetValues(snap_b, XmNlabelString, xstr, NULL);
	XmStringFree(xstr);

	xstr = XmStringCreateSimple("Trace");
	XtVaSetValues(express_b, XmNlabelString, xstr, NULL);
	XmStringFree(xstr);

	xstr = XmStringCreateSimple("Kiviat");
	XtVaSetValues(kiv_b, XmNlabelString, xstr, NULL);
	XmStringFree(xstr);

	xstr = XmStringCreateSimple("Matrix");
	XtVaSetValues(matrix_b, XmNlabelString, xstr, NULL);
	XmStringFree(xstr);
	
	xstr = XmStringCreateSimple("Rerun");
	XtVaSetValues(rerun_b, XmNlabelString, xstr, NULL);
	XmStringFree(xstr);

	xstr = XmStringCreateSimple("Clean");
	XtVaSetValues(clean_b, XmNlabelString, xstr, NULL);
	XmStringFree(xstr);

	return(rc_w);
}
