/* filebrowser.c - draws an interactive file browser
   Copyright (C) 1996, 1997 Paul Sheer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.
 */

#include <config.h>
#include <stdio.h>
#include <my_string.h>
#include <stdlib.h>
#include <stdarg.h>

#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include "lkeysym.h"

#include "stringtools.h"
#include "app_glob.c"

#include "coolwidget.h"
#include "editcmddef.h"

#include "mad.h"

static char *mime_majors[3] =
{"url", "text", 0};

Window draw_file_browser (const char *identifier, Window parent, int x, int y,
		    const char *directory, const char *file, const char *label)
{
    CWidget * w;
    struct file_entry *filelist = 0, *directorylist = 0;
    char *dir;
    char *resolved_path, *p;
    int y2, x2, x3, y3;
    Window win;

    dir = (char *) strdup (directory);

    if (parent == CRoot)
	win = CDrawMainWindow (identifier, label);
    else
	win = CDrawHeadedDialog (identifier, parent, x, y, label);
    (CIdent (identifier))->options |= WINDOW_ALWAYS_RAISED;
    CHourGlass (CFirstWindow);
    for (;;) {
	filelist = get_file_entry_list (dir, FILELIST_FILES_ONLY, CLastInput (catstrs (identifier, ".filt", 0)));
	if (!filelist) {
	    char *s;
	    s = strrchr (dir, '/');
	    if (s) {
		*s = '\0';
		continue;
	    }
	}
	break;
    }
    CUnHourGlass (CFirstWindow);
    if (!filelist || !(directorylist = get_file_entry_list (dir, FILELIST_DIRECTORIES_ONLY, ""))) {
	CErrorDialog (parent, 20, 20, _(" File browser "), _(" Unable to read directory "));
	CDestroyWidget (identifier);
	goto error;
    }
    CGetHintPos (&x, &y);
    resolved_path = canonicalize_pathname (dir);
    p = resolved_path + strlen (resolved_path) - 1;
    if (*p != '/') {
	*++p = '/';
	*++p = '\0';
    }
    (CDrawText (catstrs (identifier, ".dir", 0), win, x, y, resolved_path))->position |= POSITION_FILL;
    free (resolved_path);
    CGetHintPos (0, &y);
    reset_hint_pos (x, y);
    y3 = y;
    (w = CDrawFilelist (catstrs (identifier, ".fbox", 0), win, x, y,
		  FONT_MEAN_WIDTH * 24 + 7, FONT_PIX_PER_LINE * 15 + 6, 0, 0, filelist, TEXTBOX_FILE_LIST))->position |= POSITION_WIDTH | POSITION_HEIGHT;
    xdnd_set_type_list (CDndClass, w->winid, xdnd_typelist_send[DndFiles]);
    (CIdent (catstrs (identifier, ".fbox", 0)))->options |= TEXTBOX_MARK_WHOLE_LINES;

/* the vertical scroll bar is named the same as the text box, with a ".vsc" at the end: */
    CSetMovement (catstrs (identifier, ".fbox.vsc", 0), POSITION_HEIGHT | POSITION_RIGHT);
    CSetMovement (catstrs (identifier, ".fbox.hsc", 0), POSITION_WIDTH | POSITION_BOTTOM);
    CGetHintPos (&x2, &y2);
    x3 = x2;
    (w = CDrawFilelist (catstrs (identifier, ".dbox", 0), win, x2, y + TICK_BUTTON_WIDTH + WIDGET_SPACING,
		  FONT_MEAN_WIDTH * 24 + 7, y2 - WIDGET_SPACING * 3 - 12 - y - TICK_BUTTON_WIDTH, 0, 0, directorylist, TEXTBOX_FILE_LIST))->position |= POSITION_HEIGHT | POSITION_RIGHT;
    xdnd_set_type_list (CDndClass, w->winid, xdnd_typelist_send[DndFiles]);

/* Toolhint */
    CSetToolHint (catstrs (identifier, ".dbox", 0), _("Double click to enter directories"));
    (CIdent (catstrs (identifier, ".dbox", 0)))->options |= TEXTBOX_MARK_WHOLE_LINES;
    CSetMovement (catstrs (identifier, ".dbox.vsc", 0), POSITION_HEIGHT | POSITION_RIGHT);
    CSetMovement (catstrs (identifier, ".dbox.hsc", 0), POSITION_RIGHT | POSITION_BOTTOM);
    CGetHintPos (&x2, &y2);
    (CDrawText (catstrs (identifier, ".msg", 0), win, x, y2, ""))->position |= POSITION_FILL | POSITION_BOTTOM;
    CGetHintPos (0, &y2);
    (w = CDrawTextInput (catstrs (identifier, ".finp", 0), win, x, y2,
		    WIDGET_SPACING * 2 - 2, AUTO_HEIGHT, 256, file))->position |= POSITION_FILL | POSITION_BOTTOM;
    xdnd_set_type_list (CDndClass, w->winid, xdnd_typelist_send[DndFile]);
    w->funcs->types = DndFile;
    w->funcs->mime_majors = mime_majors;

    CGetHintPos (0, &y2);
/* Label for file filter input line. For example, to list files matching '*.c' */
    (CDrawText (catstrs (identifier, ".filx", 0), win, x, y2, _("Filter : ")))->position |= POSITION_BOTTOM;
    CGetHintPos (&x, 0);
    (CDrawTextInput (catstrs (identifier, ".filt", 0), win, x, y2,
		    WIDGET_SPACING * 2 - 2, AUTO_HEIGHT, 256, TEXTINPUT_LAST_INPUT))->position |= POSITION_FILL | POSITION_BOTTOM;
/* Toolhint */
    CSetToolHint (catstrs (identifier, ".filt", 0), _("List only files matching this shell filter"));
    CSetToolHint (catstrs (identifier, ".filx", 0), _("List only files matching this shell filter"));
    (CDrawPixmapButton (catstrs (identifier, ".ok", 0), win, x3, y3,
		       PIXMAP_BUTTON_TICK))->position |= POSITION_RIGHT;
/* Toolhint */
    CSetToolHint (catstrs (identifier, ".ok", 0), _("Accept, Enter"));

    (CDrawPixmapButton (catstrs (identifier, ".cancel", 0), win, x2 - WIDGET_SPACING * 2 - TICK_BUTTON_WIDTH - 20, y3,
		       PIXMAP_BUTTON_CROSS))->position |= POSITION_RIGHT;
/* Toolhint */
    CSetToolHint (catstrs (identifier, ".cancel", 0), _("Abort this dialog, Escape"));
    CSetSizeHintPos (identifier);
    CMapDialog (identifier);
    y = (CIdent (identifier))->height;
    CSetWindowResizable (identifier, FONT_MEAN_WIDTH * 40, min (FONT_PIX_PER_LINE * 5 + 210, y), 1600, 1200);	/* minimum and maximum sizes */

  error:
    if (directorylist)
	free (directorylist);
    if (filelist)
	free (filelist);
    free (dir);
    return win;
}

/* returns 0 on fail */
static int goto_partial_file_name (CWidget * list, char *text)
{
    int i = 0;
    struct file_entry *fe = 0;
    char *e;
    for (;;) {
	if (!strlen (text))
	    break;
	if (list->kind == C_FIELDED_TEXTBOX_WIDGET) {
	    fe = CGetFilelistLine (list, i);
	    e = fe->name;
	} else {
	    e = CGetTextBoxLine (list, i);
	    if (e)
		while (*e == '/')
		    e++;
	}
	if (!e)
	    break;
	if (!strncmp (e, text, strlen (text))) {
	    CSetTextboxPos (list, TEXT_SET_CURSOR_LINE, i);
	    CSetTextboxPos (list, TEXT_SET_LINE, i);
	    return 1;
	    break;
	}
	if (list->kind == C_FIELDED_TEXTBOX_WIDGET) {
	    if (fe->options & FILELIST_LAST_ENTRY)
		break;
	} else {
	    if (i >= list->numlines - 1)
		break;
	}
	i++;
    }
    return 0;
}

/* options */
#define GETFILE_GET_DIRECTORY		1
#define GETFILE_GET_EXISTING_FILE	2
#define GETFILE_BROWSER			4

void input_insert (CWidget * w, int c);

static char *file_error (void)
{
    if (errno) {
	char *error_msg;
#ifdef HAVE_STRERROR
	error_msg = _ (strerror (errno));
#else
	extern int sys_nerr;
	extern char *sys_errlist[];
	if ((0 <= errno) && (errno < sys_nerr))
	    error_msg = _ (sys_errlist[errno]);
	else
/* The returned value, 'errno' has an unknown meaning */
	    error_msg = _ ("strange errno");
#endif
	return catstrs ("[", error_msg, "]", 0);
    }
    return "";
}


/*
   Returns "" on no file entered and NULL on exit (i.e. Cancel button pushed)
   else returns the file or directory. Result must be immediately copied.
   Result must not be free'd.
 */
char *handle_browser (const char *identifier, CEvent * cwevent, int options)
{
    struct stat st;
    char *q;
    char *idd = catstrs (identifier, ".dbox", 0);
    char *idf = catstrs (identifier, ".fbox", 0);
    static char estr[MAX_PATH_LEN + 1];
    CWidget *directory = CIdent (catstrs (identifier, ".dir", 0));
    CWidget *filelist = CIdent (idf);
    CWidget *directorylist = CIdent (idd);
    CWidget *textinput = CIdent (catstrs (identifier, ".finp", 0));
    CWidget *filterinput = CIdent (catstrs (identifier, ".filt", 0));

    CSetDndDirectory (directory->text);

    if (cwevent->type == ButtonPress || cwevent->type == KeyPress)
	CRedrawText (catstrs (identifier, ".msg", 0), "%s", file_error());

    if (!strcmp (cwevent->ident, filterinput->ident) && cwevent->command == CK_Enter) {
	struct file_entry *f;
	if (!(*(filterinput->text))) {
	    filterinput->text[0] = '*';
	    filterinput->text[1] = '\0';
	    CExpose (filterinput->ident);
	}
	CHourGlass (CFirstWindow);
	CRedrawFilelist (catstrs (identifier, ".fbox", 0),
			 f = get_file_entry_list (directory->text, FILELIST_FILES_ONLY, filterinput->text),
			 0);
	if (f)
	    free (f);
	CUnHourGlass (CFirstWindow);
	return "";
    }
    if (!strcmp (cwevent->ident, idf) && !(options & (GETFILE_GET_DIRECTORY | GETFILE_BROWSER))) {
	if (cwevent->button == Button1 || cwevent->command == CK_Enter) {
	    q = (CGetFilelistLine (filelist, filelist->cursor))->name;
	    CDrawTextInput (textinput->ident, CIdent (identifier)->winid, textinput->x, textinput->y,
			    textinput->width, textinput->height, 256, q);
	} else if (cwevent->insert > 0) {
	    input_insert (textinput, cwevent->insert);
	    if (goto_partial_file_name (filelist, textinput->text))
		CExpose (filelist->ident);
	    CExpose (textinput->ident);
	} else if (cwevent->command == CK_BackSpace && textinput->cursor > 0) {
	    textinput->text[--textinput->cursor] = '\0';
	    if (goto_partial_file_name (filelist, textinput->text))
		CExpose (filelist->ident);
	    CExpose (textinput->ident);
	}
    }
    if (!strcmp (cwevent->ident, idd)) {
	if (cwevent->button == Button1 || cwevent->command == CK_Enter) {
	    q = (CGetFilelistLine (directorylist, directorylist->cursor))->name;
	    CDrawTextInput (catstrs (identifier, ".finp", 0), CIdent (identifier)->winid, textinput->x, textinput->y,
			textinput->width, textinput->height, 256, q);
	} else if (cwevent->insert > 0) {
	    input_insert (textinput, cwevent->insert);
	    if (goto_partial_file_name (directorylist, textinput->text))
		CExpose (directorylist->ident);
	    CExpose (textinput->ident);
	} else if (cwevent->command == CK_BackSpace && textinput->cursor > 0) {
	    textinput->text[--textinput->cursor] = '\0';
	    if (goto_partial_file_name (directorylist, textinput->text))
		CExpose (directorylist->ident);
	    CExpose (textinput->ident);
	}
    }
    if (!strcmp (cwevent->ident, textinput->ident)) {
	switch ((int) cwevent->command) {
	case CK_Enter:
	    if (!(options & (GETFILE_GET_DIRECTORY | GETFILE_BROWSER)))
		if (!strncmp ((CGetFilelistLine (filelist, filelist->cursor))->name, textinput->text, strlen (textinput->text))) {
		    CFocus (filelist);
		    CExpose (filelist->ident);
		    return "";
		}
	    break;
	case CK_Down:
	    if (textinput->keypressed) {
		if (goto_partial_file_name (filelist, textinput->text)) {
		    CFocus (filelist);
		    CExpose (filelist->ident);
		    break;
		}
	    }
	    CFocus (filelist);
	    CSetTextboxPos (filelist, TEXT_SET_CURSOR_LINE, 0);
	    CExpose (filelist->ident);
	    break;
	case CK_Up:
	    if (textinput->keypressed) {
		if (goto_partial_file_name (filelist, textinput->text)) {
		    CFocus (filelist);
		    CExpose (filelist->ident);
		    break;
		}
	    }
	    CFocus (filelist);
	    CSetTextboxPos (filelist, TEXT_SET_CURSOR_LINE, 999999);
	    CExpose (filelist->ident);
	    break;
	case CK_Page_Down:
	    CFocus (filelist);
	    CSetTextboxPos (filelist, TEXT_SET_CURSOR_LINE, filelist->height / FONT_PIX_PER_LINE - 1);
	    CExpose (filelist->ident);
	    break;
	case CK_Page_Up:
	    CFocus (filelist);
	    CSetTextboxPos (filelist, TEXT_SET_CURSOR_LINE, filelist->numlines - filelist->height / FONT_PIX_PER_LINE + 1);
	    CExpose (filelist->ident);
	    break;
	default:
	    if (cwevent->insert > 0)
		if (goto_partial_file_name (filelist, textinput->text))
		    CExpose (filelist->ident);
	    break;
	}
    }
    if (cwevent->command == CK_Cancel || !strcmp (cwevent->ident, catstrs (identifier, ".cancel", 0)))
	return 0;

    if (options & GETFILE_GET_DIRECTORY) {
	if (!strcmp (cwevent->ident, catstrs (identifier, ".ok", 0))) {
	    char *resolved_path;
	    resolved_path = canonicalize_pathname (directory->text);
	    strcpy (estr, resolved_path);
	    free (resolved_path);
	    return estr;
	}
    }
    if (!strcmp (cwevent->ident, catstrs (identifier, ".ok", 0))
	|| cwevent->command == CK_Enter
	|| (cwevent->double_click && !strcmp (cwevent->ident, catstrs (identifier, ".finp", 0)))
	|| (cwevent->double_click && !strcmp (cwevent->ident, catstrs (identifier, ".dbox", 0)))
	|| (cwevent->double_click && !strcmp (cwevent->ident, catstrs (identifier, ".fbox", 0)))) {
	char *resolved_path;
	if (*textinput->text == '/' || *textinput->text == '~')
	    *estr = '\0';
	else {
	    strcpy (estr, directory->text);
	    strcat (estr, "/");
	}
	strcat (estr, textinput->text);
	resolved_path = canonicalize_pathname (estr);
	strcpy (estr, resolved_path);
	free (resolved_path);
	textinput->keypressed = 0;
	q = estr + strlen (estr) - 1;
	if (!estr[0])
	    return "";
	if (stat (estr, &st)) {
	    if (errno == ENOENT) {
/* The user wanted a directory, but typed in one that doesn't exist */
		if (*q != '/' && !(options & GETFILE_GET_EXISTING_FILE) && !(options & (GETFILE_GET_DIRECTORY | GETFILE_BROWSER)))
/* user wants a new file */
		    return estr;
	    }
/* ********* */
	    CRedrawText (catstrs (identifier, ".msg", 0), "%s", file_error());
	    return "";
	}
	if (S_ISDIR (st.st_mode)) {
	    struct file_entry *g = 0, *f = 0;
	    CHourGlass (CFirstWindow);
	    g = get_file_entry_list (estr, FILELIST_FILES_ONLY, filterinput->text);
	    CUnHourGlass (CFirstWindow);
	    if (g) {
		CRedrawFilelist (catstrs (identifier, ".fbox", 0), g, 0);
		CRedrawFilelist (catstrs (identifier, ".dbox", 0), f = get_file_entry_list (estr, FILELIST_DIRECTORIES_ONLY, ""), 0);
		if (*q != '/') {
		    *++q = '/';
		    *++q = '\0';
		}
		CRedrawText (catstrs (identifier, ".dir", 0), estr);
		if (options & GETFILE_BROWSER)
		    CAddToTextInputHistory (textinput->ident, estr);
	    }
	    if (g)
		free (g);
	    if (f)
		free (f);
	    return "";
	} else {
	    if (options & (GETFILE_GET_DIRECTORY | GETFILE_BROWSER)) {
		CRedrawText (catstrs (identifier, ".msg", 0), "%s", file_error());
		return "";
	    }
	    return estr;	/* entry exists and is a file */
	}
    }
    return "";
}

Window find_mapped_window (Window w);

/* result must be free'd */
char *get_file_or_dir (Window parent, int x, int y,
       const char *dir, const char *file, const char *label, int options)
{
    CEvent cwevent;
    XEvent xevent;
    CState s;
    CWidget *w;

    CBackupState (&s);
    CDisable ("*");
    CEnable ("_cfileBr*");

    parent = find_mapped_window (parent);
    if (!(x | y)) {
	x = 20;
	y = 20;
    }
    draw_file_browser ("CGetFile", parent, x, y, dir, file, label);

    CFocus (CIdent ("CGetFile.finp"));

    file = "";
    do {
	CNextEvent (&xevent, &cwevent);
	if (xevent.type == Expose || !xevent.type
	    || xevent.type == InternalExpose || xevent.type == TickEvent)
	    continue;
	if (!CIdent ("CGetFile")) {
	    file = 0;
	    break;
	}
	if (xevent.type == Expose || !xevent.type || xevent.type == AlarmEvent
	  || xevent.type == InternalExpose || xevent.type == TickEvent) {
	    file = "";
	    continue;
	}
	file = handle_browser ("CGetFile", &cwevent, options);
	if (!file)
	    break;
    } while (!(*file));

/* here we want to add the complete path to the text-input history: */
    w = CIdent ("CGetFile.finp");
    if (w) {
	if (w->text) {
	    free (w->text);
	    w->text = 0;
	}
	if (file)
	    w->text = (char *) strdup (file);
    }
    CDestroyWidget ("CGetFile");	/* text is added to history 
					   when text-input widget is destroyed */

    CRestoreState (&s);

    if (file)
	return (char *) strdup (file);
    else
	return 0;
}

int cb_browser (CWidget * w, XEvent * x, CEvent * c)
{
    char id[32], *s;
    strcpy (id, w->ident);
    s = strchr (id, '.');
    if (s)
	*s = 0;
    if (!handle_browser (id, c, GETFILE_BROWSER)) {
	w = CIdent (catstrs (id, ".finp", 0));
	if (w)
	    if (w->text) {
		free (w->text);
		w->text = 0;
	    }
	CDestroyWidget (id);
    }
    return 0;
}

void CDrawBrowser (const char *ident, Window parent, int x, int y,
		   const char *dir, const char *file, const char *label)
{
    if (!(parent | x | y)) {
	parent = CFirstWindow;
	x = 20;
	y = 20;
    }

    draw_file_browser (ident, parent, x, y, dir, file, label);

    CAddCallback (catstrs (ident, ".dbox", 0), cb_browser);
    CAddCallback (catstrs (ident, ".fbox", 0), cb_browser);
    CAddCallback (catstrs (ident, ".finp", 0), cb_browser);
    CAddCallback (catstrs (ident, ".filt", 0), cb_browser);
    CAddCallback (catstrs (ident, ".ok", 0), cb_browser);
    CAddCallback (catstrs (ident, ".cancel", 0), cb_browser);

    CFocus (CIdent (catstrs (ident, ".finp", 0)));
}

char *CGetFile (Window parent, int x, int y,
		const char *dir, const char *file, const char *label)
{
    return get_file_or_dir (parent, x, y, dir, file, label, 0);
}

char *CGetDirectory (Window parent, int x, int y,
		     const char *dir, const char *file, const char *label)
{
    return get_file_or_dir (parent, x, y, dir, file, label, GETFILE_GET_DIRECTORY);
}

char *CGetSaveFile (Window parent, int x, int y,
		    const char *dir, const char *file, const char *label)
{
    return get_file_or_dir (parent, x, y, dir, file, label, 0);
}

char *CGetLoadFile (Window parent, int x, int y,
		    const char *dir, const char *file, const char *label)
{
    return get_file_or_dir (parent, x, y, dir, file, label, GETFILE_GET_EXISTING_FILE);
}


