/* vi:set ts=8 sts=0 sw=8:
 * $Id: project.c,v 1.23 2000/01/19 17:29:06 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <gtk/gtk.h>
#include "win.h"
#include "doc.h"
#include "gtkefilesel.h"
#include "file.h"
#include "msgbar.h"
#include "msgbox.h"
#include "recent.h"
#include "prefs.h"
#include "prjbar.h"
#include "misc.h"
#include "dialog.h"
#include "project.h"
#include "gnpintl.h"

extern void	win_set_title(doc_t *d);

#ifdef WANT_PROJECT

/*** local types ***/
#define PRJ_HEADER	":GNP+ prj file:"


/*** local function prototypes ***/
static int prj_header_read(FILE *fp, prj_t *prj);
static void prj_open_ok(GtkWidget *wgt, gpointer cbdata);
static void prj_open_cancel(GtkWidget *wgt, gpointer cbdata);
static void prj_open_destroy(GtkWidget *wgt, gpointer cbdata);
static void prj_files_read(win_t *w, prj_t *prj, FILE *fp);
static void prj_list_destroy(GtkWidget *wgt, gpointer cbdata);
static void prj_list_add(win_t *w, char *name, int numfiles);
static int prj_list_select(GtkWidget *wgt, unsigned row, int column,
				GdkEventButton *event, gpointer cbdata);
static void prj_list_close_selected(GtkWidget *wgt, gpointer cbdata);
static void prj_free(win_t *w, prj_t *curprj);


typedef enum {
	PRJ_HDR_NO_ERROR,
	PRJ_HDR_ERR_HDR_INFO,
	PRJ_HDR_ERR_HDR_VERSION,
	PRJ_HDR_ERR_NAME,
	PRJ_HDR_ERR_NUM_FILES,
	PRJ_HDR_ERR_VERSION,
	PRJ_HDR_ERR_PATH
} prjhdr_err_t;


/*** global function defintions ***/
/*
 * PUBLIC: prj_close_cb
 *
 * close the current project
 */
void
prj_close_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)(cbdata);
	doc_t *d;
	GSList *dlp;
	int num;
	prj_t *prj;

	g_assert(w != NULL);
	gtk_signal_disconnect_by_func(GTK_OBJECT(w->nb),
				      GTK_SIGNAL_FUNC(doc_switch_page), w);
	if ((prj = prj_current(DOC_CURRENT(w))) == NULL) {
		do_dialog_ok(_("No project selected"),
			     _(" Please select a project to close "));
		prj_list_show(NULL, (void *)w);
		(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
					       GTK_SIGNAL_FUNC(doc_switch_page),
					       w);
		return;
	}

	/* close all the files associated with this project */
	g_warning("prj_close_cb: FIXME closing all files for **ANY** project");
	num = 0;
	dlp = w->doclist;
	while (dlp) {
		d = (doc_t *)dlp->data;
		if (d && d->prj) {
			/* change to the new document */
			gtk_notebook_set_page(GTK_NOTEBOOK(w->nb), num);
			w->curdoc = g_slist_nth_data(w->doclist, num);
			(void)doc_close_common(w, FALSE);
			dlp = w->doclist;
			num = 0;
		} else {
			dlp = dlp->next;
			num++;
		}
	}

	/* now free up the prj structure */
	prj_free(w, prj);

	(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
				       GTK_SIGNAL_FUNC(doc_switch_page), w);
	win_set_title(DOC_CURRENT(w));
	prjbar_update(w);
} /* prj_close_cb */


/*
 * PUBLIC: prj_free_all
 *
 * frees all projects
 */
void
prj_free_all(win_t *w)
{
	while (w->prjlist)
		prj_free(w, (prj_t *)w->prjlist->data);
} /* prj_free_all */


/*
 * PUBLIC: prj_list_show
 *
 * creates and displays the project list dialog box
 */
void
prj_list_show(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	GSList *pp;
	prj_t *prj;
	char *titles[] = { N_(" # "), N_(" # Files "), N_(" Project Name ") };

	if (misc_show_and_raise(w->prjlw))
		return;

	msgbar_printf(w, _("Creating project list..."));
	do_dialog_clist(&(w->prjlw), N_("Project List"),
			prefs.prjlist_h, prefs.prjlist_w,
			&(w->prjlw_data), 3, titles,
			prj_list_destroy, (void *)w,
			prj_list_select, NULL,
			prj_list_close_selected, NULL,
			prj_close_all_cb, NULL);

	gtk_clist_freeze(GTK_CLIST(w->prjlw_data));
	for (pp = w->prjlist; pp; pp = pp->next) {
		prj = (prj_t *)pp->data;
		prj_list_add(w, prj->name, prj->numfiles);
		gtk_main_iteration_do(FALSE);
	}
	gtk_clist_thaw(GTK_CLIST(w->prjlw_data));

	gtk_widget_show_all(w->prjlw);
} /* prj_list_show */


/*
 * PUBLIC: prj_close_all_cb
 *
 *
 */
void
prj_close_all_cb(GtkWidget *wgt, gpointer cbdata)
{
	g_warning("[prj_close_all_cb] TODO: implement this");
} /* prj_close_all_cb */


/*
 * PUBLIC: prj_open_cb
 *
 * creates a file open dialog box for opening projects
 */
void
prj_open_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	if (misc_show_and_raise(w->prjsel))
		return;

	g_assert(!w->prjsel);
	w->prjsel = misc_open_dialog_create((void *)w,
					    _("Open Project..."),
					    prj_open_ok,
					    prj_open_cancel,
					    prj_open_destroy);
	gtk_efilesel_set_mode(GTK_EFILESEL(w->prjsel), GTK_SELECTION_SINGLE);
	gtk_widget_show(w->prjsel);
} /* prj_open_cb */


/*
 * PRIVATE: prj_free
 *
 * frees memory for a specified project
 */
static void
prj_free(win_t *w, prj_t *curprj)
{
	GSList *pp, *fnp;
	prj_t *prj;
	char *fname;
	doc_t *d = DOC_CURRENT(w);

	if (!curprj || !w)
		return;

	pp = g_slist_find(w->prjlist, curprj);
	w->prjlist = g_slist_remove_link(w->prjlist, pp);
	prj = (prj_t *)pp->data;
	g_free(prj->name);
	g_free(prj->path);
	while (prj->files) {
		fnp = prj->files;
		prj->files = g_slist_remove_link(prj->files, fnp);
		fname = (char *)fnp->data;
		g_free(fname);
		g_slist_free_1(fnp);
	}
	g_free(prj);
	g_slist_free_1(pp);

	if (d)
		d->prj = (w->prjlist) ? (prj_t *)w->prjlist->data : NULL;
}


/*
 * PRIVATE: prj_list_add
 *
 * add a project name and the number of files in that project to the project
 * list dialog window
 */
static void
prj_list_add(win_t *w, char *name, int numfiles)
{
	char numstr[4], numdocstr[8];
	char *rownfo[3];
	GtkCList *clist;

	if (!w->prjlw_data)
		return;

	clist = GTK_CLIST(w->prjlw_data);
	g_snprintf(numstr, 4, "%d", clist->rows + 1);
	rownfo[0] = numstr;
	g_snprintf(numdocstr, 8, "%4d", numfiles);
	rownfo[1] = numdocstr;
	rownfo[2] = name;
	gtk_clist_freeze(clist);
	gtk_clist_append(clist, rownfo);
	gtk_clist_select_row(clist, clist->rows - 1, 0);
	gtk_clist_thaw(clist);
} /* prj_list_add */


/*
 * PRIVATE: prj_list_select
 *
 * callback invoked when a project is selected in the project list window
 */
static int
prj_list_select(GtkWidget *wgt, unsigned row, int column,
	GdkEventButton *event, gpointer cbdata)
{
	g_warning("[prj_list_select] TODO: implement this");
	return TRUE;
} /* win_list_select */


static void
prj_list_destroy(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	if (w->prjlw) {
		gtk_widget_destroy(w->prjlw);
		w->prjlw = NULL;
		w->prjlw_data = NULL;
	}
} /* prj_list_destroy */


static void
prj_list_close_selected(GtkWidget *wgt, gpointer cbdata)
{
	g_warning("[prj_list_close_selected] TODO: implement this");
} /* prj_list_close_selected */


static void
prj_open_destroy(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	w->prjsel = NULL;
} /* prj_open_destroy */


/*
 * PRIVATE: doc_open_cancel
 *
 * The "Cancel" button was hit.  Just hide the dialog box.
 */
static void
prj_open_cancel(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	gtk_widget_hide(w->prjsel);
} /* prj_open_cancel */


/*
 * PRIVATE: prj_open_ok
 *
 * reads a project (.prj) file and opens all files of the project.
 */
static void
prj_open_ok(GtkWidget *wgt, gpointer cbdata)
{
	FILE *fp;
	prj_t *prj;
	char *fname;
	int err;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	fname = gtk_efilesel_get_filename(GTK_EFILESEL(w->prjsel));
	if (fname == NULL || fname[0] == '\0')
		goto out;

	if ((fp = fopen(fname, "r")) == NULL) {
		printf("[prj_open] could not open '%s'", fname);
		goto out;
	}
	(void)file_lock(fname, fp, FALSE, FALSE, TRUE);

	prj = g_new0(prj_t, 1);

	if ((err = prj_header_read(fp, prj))) {
		g_warning("[prj_open] could not read header for '%s'", fname);
		g_free(prj);
		goto out;
	}
	w->prjlist = g_slist_prepend(w->prjlist, prj);

	/* now read in the files */
	prj_files_read(w, prj, fp);

	(void)file_unlock(fname, fp);
	(void)fclose(fp);

	msgbox_printf(_("Project '%s' opened"), prj->name);
	msgbar_printf(w, _("Project '%s' opened"), prj->name);
	prjbar_update(w);

out:
	gtk_widget_hide(w->prjsel);
} /* prj_open_ok */


static int
prj_header_read(FILE *fp, prj_t *prj)
{
	char line[MAXPATH], *p;
	char *title = _("Invalid project file");
	int ret = PRJ_HDR_NO_ERROR;

	if (fgets(line, MAXPATH, fp) == NULL) {
		ret = PRJ_HDR_ERR_HDR_INFO;
		goto err_out;
	}

	if ((p = strstr(line, PRJ_HEADER)) == NULL) {
		ret = PRJ_HDR_ERR_HDR_VERSION;
		goto err_out;
	}
	p += strlen(PRJ_HEADER);
	prj->_rev = atoi(p);

	if (fgets(line, MAXPATH, fp) == NULL) {
		ret = PRJ_HDR_ERR_NAME;
		goto err_out;
	}
	line[strlen(line) - 1] = '\0';	/* strip \n */
	for (p = line + 1; isspace((int)(*p)); p++)
		;
	prj->name = g_strdup(p);

	if (fgets(line, MAXPATH, fp) == NULL) {
		ret = PRJ_HDR_ERR_NUM_FILES;
		goto err_out;
	}
	p = line + 1;
	prj->numfiles = atoi(p);

	if (fgets(line, MAXPATH, fp) == NULL) {
		ret = PRJ_HDR_ERR_VERSION;
		goto err_out;
	}
	p = line + 1;
	prj->version = atoi(p);

	if (fgets(line, MAXPATH, fp) == NULL) {
		ret = PRJ_HDR_ERR_PATH;
		goto err_out;
	}
	for (p = line + 1; isspace((int)(*p)); p++)
		;
	prj->path = g_strdup(p);	 /* we want the extra char */
	prj->path[strlen(prj->path) - 1] = '\0';	/* strip \n */
	if (prj->path[strlen(prj->path) - 1] != '/')
		strcat(prj->path, "/");			/* append '/' */

	GNPDBG_PROJECT(("prj->_rev = %d\n", prj->_rev));
	GNPDBG_PROJECT(("prj->name = '%s'\n", prj->name));
	GNPDBG_PROJECT(("prj->numfiles = %d\n", prj->numfiles));
	GNPDBG_PROJECT(("prj->version = %d\n", prj->version));
	GNPDBG_PROJECT(("prj->path = '%s'\n", prj->path));

	return ret;

err_out:
	if (prj) {
		if (prj->name)
			g_free(prj->name);
		if (prj->path)
			g_free(prj->path);
	}

	switch (ret) {
	case PRJ_HDR_ERR_HDR_INFO:
		do_dialog_error(title, _(" Could not read version info "));
		break;
	case PRJ_HDR_ERR_HDR_VERSION:
		do_dialog_error(title, _(" Version number not found "));
		break;
	case PRJ_HDR_ERR_NAME:
		do_dialog_error(title, _(" Project name not found "));
		break;
	case PRJ_HDR_ERR_NUM_FILES:
		do_dialog_error(title, _(" Number of files not found "));
		break;
	case PRJ_HDR_ERR_VERSION:
		do_dialog_error(title, _(" Project version not found "));
		break;
	case PRJ_HDR_ERR_PATH:
		do_dialog_error(title, _(" Project path not found "));
		break;
	default:
		do_dialog_error(title, _(" Unknown error "));
	}

	return ret;
} /* prj_header_read */


static void
prj_files_read(win_t *w, prj_t *prj, FILE *fp)
{
	char line[MAXPATH], buf[MAXPATH];
	long flags;
	int cnt;
	doc_t *d;

	gtk_widget_hide(w->nb);
	gtk_signal_disconnect_by_func(GTK_OBJECT(w->nb),
				      GTK_SIGNAL_FUNC(doc_switch_page), w);
	cnt = 0;
	flags = 0;
	while (fgets(line, MAXPATH, fp)) {
		cnt++;
		if (cnt == prj->numfiles)
			flags = UPDATE_MSGBAR | DO_LOAD | UPDATE_TITLE;
		line[strlen(line) - 1] = '\0';	/* strip \n */
		if (line[0] != '/') {
			g_snprintf(buf, MAXPATH, "%s%s", prj->path, line);
			GNPDBG_PROJECT(("[prj_files_read] fname '%s'\n", buf));
			doc_new(w, buf, DocText, flags);
			if (prj->numfiles - cnt <= prefs.maxrecent)
				recent_list_add(w, buf);
		} else {
			GNPDBG_PROJECT(("[prj_files_read] fname '%s'\n", line));
			doc_new(w, line, DocText, flags);
			if (prj->numfiles - cnt <= prefs.maxrecent)
				recent_list_add(w, line);
		}

		d = DOC_CURRENT(w);
		d->prj = prj;
		prj->files = g_slist_prepend(prj->files, g_strdup(line));
	}

	(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
				       GTK_SIGNAL_FUNC(doc_switch_page), w);
	gtk_widget_show(w->nb);

	if (cnt < prj->numfiles) {
		g_warning("[prj_files_read] Loaded fewer files (%d) than value "
			  "specified in project (%d)\n", cnt, prj->numfiles);
		perror("[prj_files_read] errno value\n");
	}
} /* prj_files_read */


#endif	/* WANT_PROJECT */
/* the end */
