/* vi:set ts=8 sts=0 sw=8:
 * $Id: recent.c,v 1.23 2000/02/25 18:20:37 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "main.h"
#ifdef USE_RECENT
#include <stdio.h>
#include <string.h>
#include "doc.h"
#include "file.h"
#include "prefs.h"
#include "menu.h"
#include "recent.h"


/*** local variables ***/
static char *recentrc = NULL;


/*** local function prototypes ***/
static void recent_list_trim(win_t *w, bool_t remove_from_menu);


/*** global function definitions ***/
/*
 * PUBLIC: recent_list_add
 *
 * called from doc_open_ok(), app_init_open_files(), prj_open()
 */
void
recent_list_add(win_t *w, char *fname)
{
	doc_new_name_t *dnnp, *dnnp2;
	char *full;
	GSList *rlp;
	doc_t *d;

	d = DOC_CURRENT(w);
	g_assert(d != NULL);
	if (!d->fname)
		return;	/* don't add 'Untitled' */

	/* construct full pathname to file */
	full = file_full_pathname_make(fname);

	/*
	 * add doc to recent doc list, and add to recent doc menu.  note that
	 * dnnp gets free'd down below, when trimming the recent doc list.
	 */
	dnnp = g_new0(doc_new_name_t, 1);
	dnnp->w = w;
	dnnp->fname = g_strdup(full);

	/* if duplicate, remove it from recentlist, and remove menu entry */
	rlp = w->recentlist;
	while (rlp) {
		dnnp2 = (doc_new_name_t *)rlp->data;
		if (strcmp(dnnp2->fname, dnnp->fname) == 0) {
			w->recentlist = g_slist_remove_link(w->recentlist, rlp);
			menu_recent_doc_remove(rlp->data);
			g_slist_free_1(rlp);
			g_free(dnnp2->fname);
			g_free(dnnp2);
			break;
		}
		rlp = rlp->next;
	}

	/* now add to list and menu */
	w->recentlist = g_slist_append(w->recentlist, dnnp);
	menu_recent_doc_add(doc_by_name, dnnp);

	recent_list_trim(w, TRUE);

	if (full != fname)
		g_free(full);
	g_assert(g_slist_length(w->recentlist) <= prefs.maxrecent);
} /* recent_list_add */


/*
 * PUBLIC: recent_list_init
 *
 * called from win_new()
 */
void
recent_list_init(win_t *w)
{
	doc_new_name_t *dnnp;
	GSList *rlp;
	FILE *fp;
	char *buf;
	int ret;

	if (prefs.appdir == NULL)
		return;

	if (recentrc == NULL) {
		int len = strlen(prefs.appdir) + strlen("recentrc") + 2;
		recentrc = g_new(char, len);
		g_snprintf(recentrc, len, "%s/recentrc", prefs.appdir);
	}

	if ((fp = fopen(recentrc, "r")) == NULL) {
		/* not actually an error, since it's ok to not exist */
		return;
	}
	buf = g_new(char, MAXPATH);

	ret = file_lock(recentrc, fp, FALSE, FALSE, TRUE);
	while (fgets(buf, MAXPATH, fp)) {
		if (buf[strlen(buf) - 1] == '\n')
			buf[strlen(buf) - 1] = '\0';

		dnnp = g_new(doc_new_name_t, 1);
		dnnp->w = w;
		dnnp->fname = g_strdup(buf);
		w->recentlist = g_slist_append(w->recentlist, dnnp);
	}
	ret = file_unlock(recentrc, fp);
	fclose(fp);
	g_free(buf);

	recent_list_trim(w, FALSE);

	/* add to menu */
	rlp = w->recentlist;
	while (rlp) {
		menu_recent_doc_add(doc_by_name, rlp->data);
		rlp = rlp->next;
	}
} /* recent_list_init */


/*
 * PUBLIC: recent_list_write
 *
 * called from doc_close_cb(), doc_close_all_common(), doc_save_common()
 */
void
recent_list_write(win_t *w)
{
	doc_new_name_t *dnnp;
	GSList *rlp;
	FILE *fp;
	int ret;

	if (prefs.appdir == NULL || w->recentlist == NULL)
		return;

	if ((fp = fopen(recentrc, "w")) == NULL) {
		char *buf = g_new(char, strlen(recentrc) + 64);
		sprintf(buf, "recent_list_write: could not open "
			"'%s'", recentrc);
		perror(buf);
		g_free(buf);
		return;
	}

	GNPDBG_RECENT(("recent_list_write: opened '%s'\n", recentrc));
	ret = file_lock(recentrc, fp, TRUE, FALSE, TRUE);
	rlp = w->recentlist;
	while (rlp) {
		dnnp = (doc_new_name_t *)(rlp->data);
		fprintf(fp, "%s\n", dnnp->fname);
		GNPDBG_RECENT(("recent_list_write: wrote '%s'\n", dnnp->fname));
		rlp = rlp->next;
	}
	fflush(fp);
	ret = file_unlock(recentrc, fp);
	fclose(fp);
	GNPDBG_RECENT(("recent_list_write: closed '%s'\n", recentrc));
} /* recent_list_write */


void
recent_list_free(win_t *w)
{
	doc_new_name_t *dnnp;
	GSList *rlp;

	while (w->recentlist) {
		rlp = w->recentlist;
		w->recentlist = g_slist_remove_link(w->recentlist, rlp);
		dnnp = (doc_new_name_t *)(rlp->data);
		g_slist_free_1(rlp);
		g_free(dnnp->fname);
		g_free(dnnp);
	}
} /* recent_list_free */


/*** local function definitions ***/
/*
 * PRIVATE: recent_list_trim
 *
 * trim list if we have too many entries
 */
static void
recent_list_trim(win_t *w, bool_t remove_from_menu)
{
	doc_new_name_t *dnnp;
	GSList *rlp;

	if (g_slist_length(w->recentlist) <= prefs.maxrecent)
		return;

	while (g_slist_length(w->recentlist) > prefs.maxrecent) {
		rlp = w->recentlist;
		if (remove_from_menu)
			menu_recent_doc_remove(rlp->data);
		w->recentlist = g_slist_remove_link(w->recentlist, rlp);
		dnnp = (doc_new_name_t *)(rlp->data);
		g_slist_free_1(rlp);
		g_free(dnnp->fname);
		g_free(dnnp);
	}
} /* recent_list_trim */

#endif	/* USE_RECENT */

/* the end */
