#i linuxdoc.tm

#d slang \bf{S-lang}
#d kw#1 \tt{$1}
#d exmp#1 \tt{$1}
#d var#1 \tt{$1}
#d ldots ...
#d chapter#1 <chapt>$1<p>
#d preface <preface>
#d tag#1 <tag>$1</tag>

#d function#1 \section{<bf>$1</bf>\label{$1}}<descrip>
#d variable#1 \section{<bf>$1</bf>\label{$1}}<descrip>
#cd function#1 <p><bf>$1</bf>\label{$1}<p><descrip>
#d synopsis#1 <tag> Synopsis </tag> $1
#d keywords#1 <tag> Keywords </tag> $1
#d usage#1 <tag> Usage </tag> <tt>$1</tt>
#d description <tag> Description </tag>
#d example <tag> Example </tag>
#d notes <tag> Notes </tag>
#d seealso#1 <tag> See Also </tag> <tt>$1</tt>
#d documentstyle article
#d r#1 \ref{$1}{$1}
#d done </descrip><p>

#d user-manual \bf{A Guide to the S-Lang Language}

\linuxdoc
\begin{\documentstyle}

\title Jed Intrinsic Function Reference Manual
\author John E. Davis, \tt{davis@space.mit.edu}
\date \__today__

\toc


\function{setkey}
\synopsis{setkey}
\usage{Void setkey(String fun, String key);}
\description
  This function may be used to define a key sequence specified by the
  string \var{key} to the function \var{fun}.  \var{key} can contain the \exmp{^}
  character which denotes that the following character is to be
  interpreted as a control character, e.g.,
#v+
        setkey("bob", "^Kt");
#v-
  sets the key sequence \exmp{Ctrl-K t} to the function \var{bob}.

  The \var{fun} argument is usually the name of an internal or a user
  defined S-Lang function.  However, if may also be a sequence of
  functions or even another keysequence (a keyboard macro).  For
  example,
#v+
        setkey ("bol;insert(string(whatline()))", "^Kw");
#v-
  assigns the key sequence \exmp{Ctrl-K w} to move to the beginning of a line
  and insert the current line number.  For more information about this
  important function, see the JED User Manual.

  Note that \var{setkey} works on the "global" keymap.
\seealso{unsetkey, definekey}
\done

\function{_autoload}
\synopsis{_autoload}
\usage{Void _autoload (String fun, String fn, ..., Integer n);}
\description
  The \var{_autoload} function is like the \var{autoload} function except that
  it takes \var{n} pairs of function name (\var{fun}) / filename (\var{fn}) pairs.
  For example,
#v+
        _autoload ("fun_a", "file_a", "fun_b", "file_b", 2);
#v-
  is equivalent to
#v+
        autoload ("fun_a", "file_a");
        autoload ("fun_b", "file_b");
#v-
\seealso{autoload}
\done

\function{push_mark}
\synopsis{push_mark}
\usage{Void push_mark();}
\description
  This function marks the current position as the beginning of a region.
  and pushes other marks onto a stack.  A region is defined by this
  mark and the editing point.  The mark is removed from the stack only
  when the function \var{pop_mark} is called.
  For example,
#v+
        define mark_buffer ()
        {
          bob ();
          push_mark ();
          eob ();
        }
#v-
  marks the entire buffer as a region.
\seealso{pop_mark, push_spot, markp, dupmark, check_region}
\done

\function{bol}
\synopsis{bol}
\usage{Void bol();}
\description
  This function moves the current editing point to the beginning of the
  current line.  The function \var{bolp} may be used to see if one is already
  at the beginning of a line.
\seealso{eol, bob, eob, bolp}
\done

\function{insert}
\synopsis{insert}
\usage{Void insert (String str);}
\description
  Inserts string \var{str} into buffer at the current position.  The editing
  point is moved to the end of the of the string that was inserted.
\seealso{insert_char, del, insert_file, insbuf}
\done

\function{insert_char}
\synopsis{insert_char}
\description
  Undocumented
\done

\function{eol}
\synopsis{eol}
\usage{Void eol();}
\description
  Moves the current position to the end of the current line.  The function
  \var{eolp} may be used to see if one is at the end of a line or not.
\seealso{eolp, bol, bob, eob}
\done

\function{setbuf}
\synopsis{setbuf}
\usage{Void setbuf(String buf);}
\description
  Changes the default buffer to one named \var{buf}.  If the buffer does not
  exist, it will be created.
  Note: This change only lasts until top
  level of editor loop is reached at which point the the buffer
  associated with current window will be made the default.  That is this
  change should only be considered as temporary.  To make a long lasting
  change, use the function \var{sw2buf}.
\seealso{sw2buf, pop2buf, whatbuf, pop2buf_whatbuf}
\done

\function{_add_completion}
\synopsis{_add_completion}
\usage{Void _add_completion (String f1, String f2, ..., Integer n);}
\description
  The \var{_add_completion} function is like the \var{add_completion} function
  except that it takes \var{n} names \var{f1}, ... \var{fn}.
  For example,
#v+
        _add_completion ("fun_a", "fun_b", 2);
#v-
  is equivalent to
#v+
        add_completion ("fun_a");
        add_completion ("fun_b");
#v-
\seealso{add_completion}
\done

\function{del_region}
\synopsis{del_region}
\usage{Void del_region ();}
\description
  This function deletes the region defined by the mark and the current
  editing point.  For example,
#v+
        define delete_this_line ()
        {
          bol (); push_mark (); eol ();
          del_region ();
        }
#v-
  defines a function that deletes all characters on the current line
  from the beginning of the line until the end of the line.  It does not
  delete the line itself.
\seealso{push_mark, markp, check_region}
\done

\function{bufsubstr}
\synopsis{bufsubstr}
\usage{String bufsubstr ();}
\description
  This function returns a string that contains the characters in the
  region specified by a mark and the current editing point.
  If the region crosses lines, the string will contain newline
  characters.
\seealso{insbuf, push_mark}
\done

\function{right}
\synopsis{right}
\usage{Integer right(Integer n);}
\description
  This function moves the editing position forward forward \var{n}
  characters. It returns the number of characters actually moved.  The
  number returned will be smaller than \var{n} if the end of the buffer is
  reached.
\seealso{left, up, down, eol, eob}
\done

\function{definekey}
\synopsis{definekey}
\usage{Void definekey(String f, String key, String kmap);}
\description
  Unlike \var{setkey} which operates on the global keymap, this function is
  used for binding keys to functions in a specific keymap.  Here \var{f} is
  the function to be bound, \var{key} is a string of characters that make up
  the key sequence and \var{kmap} is the name of the keymap to be used.  See
  \var{setkey} for more information about the arguments.
\seealso{setkey, undefinekey, make_keymap, use_keymap}
\done

\function{left}
\synopsis{left}
\usage{Integer left(Integer n);}
\description
  \var{left} moves the editing point backward \var{n} characters and returns the
  number actually moved.  The number returned will be less than \var{n} only
  if the top of the buffer is reached.
\seealso{right, up, down, bol, bob}
\done

\function{whatbuf}
\synopsis{whatbuf}
\usage{String what_buffer();}
\description
  \var{whatbuf} returns the name of the current buffer.  It is usually used
  in functions when one wants to work with more than one buffer.  The
  function \var{setbuf_info} may be used to change the name of the buffer.
\seealso{getbuf_info, bufferp}
\done

\function{getbuf_info}
\synopsis{getbuf_info}
\usage{getbuf_info ();}
\description
  This function returns values to the stack.  The four values from the
  top are:
#v+
        Integer  % buffer flags
        String   % name of buffer
        String   % directory associated with buffer
        String   % name of file associated with buffer (if any).
#v-
  The integer that corresponds to the buffer flags are encoded as:
#v+
        bit 0: buffer modified
        bit 1: auto save mode
        bit 2: file on disk modified
        bit 3: read only bit
        bit 4: overwrite mode
        bit 5: undo enabled
        bit 6: buffer buried
        bit 7: Force save upon exit.
        bit 8: Do not backup
        bit 9: File associated with buffer is a binary file
        bit 10: Add CR to end of lines when writing buffer to disk.
        bit 11: Abbrev mode
#v-
  For example,
#v+
        (file,,,flags) = getbuf_info();
#v-
  returns the file and the flags associated with the current buffer.
\seealso{setbuf_info, whatbuf}
\done

\function{otherwindow}
\synopsis{otherwindow}
\usage{Void otherwindow ();}
\description
  This function will make the next window in the ring of windows as the
  default window. For example,
#v+
        define zoom_next_window ()
        {
          otherwindow (); onewindow ();
        }
#v-
  defines a function that moves to the next window and then makes it the
  only window on the screen.
\seealso{nwindows, onewindow}
\seealso{MINIBUFFER_ACTIVE}
\done

\function{is_internal}
\synopsis{is_internal}
\usage{Integer is_internal(String f);}
\description
  \var{is_internal} returns non-zero is function \var{f} is defined as an
  internal function or returns zero if not.  Internal functions not
  immediately accessable from S-Lang; rather, they must be called using
  the \var{call} function.  See also the related S-Lang function
  \var{is_defined} in the S-Lang Programmer's Reference.
\seealso{call}
\done

\function{setbuf_info}
\synopsis{setbuf_info}
\usage{Void setbuf_info(String file, String dir, String buf, Integer flags);}
\description
  This function may be used to change attributes regarding the current
  buffer.  It performs the opposite function of the related function
  \var{getbuf_info}.   Here \var{file} is the name of the file to be associated
  with the buffer; \var{dir} is the directory to be associated with the
  buffer; \var{buf} is the name to be assigned to the buffer, and \var{flags}
  describe the buffer attributes.  See \var{getbuf_info} for a discussion of
  \var{flags}.  Note that the actual file associated with the buffer is
  located in directory \var{dir} with the name \var{file}.
  For example, the function
#v+
        define set_overwrite_mode ()
        {
           variable dir, file, flags, name;
           (file, dir, name, flags) = getbuf_info ();
           flags = flags | (1 shl 4);
           setbuf_info (file, dir, name, flags);
        }
#v-
  may be used to turn on overwrite mode for the current buffer.  Note
  that it is better exploit the fact that S-Lang is a stack based language
  and simply write the above function as:
#v+
        define set_overwrite_mode ()
        {
           setbuf_info (getbuf_info () | 0x10);
        }
#v-
  Here, \exmp{(1 shl 4)} has been written as the hexidecimal number \exmp{0x10}.
\seealso{getbuf_info, setbuf, whatbuf}
\done

\function{up}
\synopsis{up}
\usage{Integer up(Integer n);}
\description
  This function moves the current point up \var{n} lines and returns the
  number of lines actually moved.  The number returned will be less than
  \var{n} only if the top of the buffer is reached.
\seealso{down, left, right}
\done

\function{down}
\synopsis{down}
\usage{Integer down(Integer n);}
\description
  The \var{down} function is used to move the editing point down a number of
  lines specified by the integer \var{n}.  It returns the number of lines
  actually moved.  The number returned will be less than \var{n} only if the
  last line of the buffer has been reached.  The editing point will be
  left at the beginning of the line if it succeeds in going down more
  than one line.
  Example: The function
#v+
        define trim_buffer
        {
          bob ();
          do
            {
               eol (); trim ();
            }
          while (down (1));
        }
#v-
  removes excess whitespace from the end of every line in the buffer.
\seealso{down, left, right, goto_line}
\done

\function{call}
\synopsis{call}
\usage{Void call(String f);}
\description
  The \var{call} function is used to execute an internal function which is
  not directly accessable to the S-Lang interpreter.
\seealso{is_internal}
\done

\function{eob}
\synopsis{eob}
\usage{Void eob();}
\description
  The \var{eob} function is used to move the current point to the end of the
  buffer.  The function \var{eobp} may be used to see if the current
  position is at the end of the buffer.
\seealso{eobp, bob, bol, eol}
\done

\function{unsetkey}
\synopsis{unsetkey}
\usage{Void unsetkey(String key);}
\description
  This function is used to remove the definition of the key sequence
  \var{key} from the "global" keymap.  This is sometimes necessary to bind
  new key sequences which conflict with other ones.  For example, the
  "global" keymap binds the keys \exmp{"^[[A"}, \exmp{"^[[B"}, \exmp{"^[[C"}, and
  \exmp{"^[[D"} to the character movement functions.  Using
  \exmp{unsetkey("^[[A")} will remove the binding of \exmp{"^[[A"} from the global
  keymap but the other three will remain.  However, \exmp{unsetkey("^[[")}
  will remove the definition of all the above keys.  This might be
  necessary to bind, say, \exmp{"^[["} to some function.
\seealso{setkey, undefinekey}
\done

\function{bob}
\synopsis{bob}
\usage{Void bob ();}
\description
  The function \var{bob} is used to move the current editing point to the
  beginning of the buffer.  The function \var{bobp} may be used to determine
  if the editing point is at the beginning of the buffer or not.
\seealso{bobp, eob, bol, eol}
\done

\function{looking_at}
\synopsis{looking_at}
\usage{Integer looking_at (String s);}
\description
  This function returns non-zero if the characters immediately following
  the current editing point match the string specified by \var{s}.  Whether
  the match is case-sensitive or not depends upon the value of the
  variable \var{CASE_SEARCH}.  The function returns zero if there is no match.
\seealso{ffind, fsearch, re_fsearch, bfind}
\done

\function{del}
\synopsis{del}
\usage{Void del ();}
\description
  The \var{del} function deletes the character at the current editing
  position.  If the position is at the end of the buffer, nothing happens.
  If the deletion occurs at the end of a line, the next line will be joined
  with the current one.
\seealso{eobp, erase_buffer, insert}
\done

\function{markp}
\synopsis{markp}
\usage{Void markp ();}
\description
  This function returns a non-zero value if the mark is set; otherwise, it
  returns zero.  If a mark is set, a region is defined.
\seealso{push_mark, pop_mark, check_region, push_spot}
\done

\function{nwindows}
\synopsis{nwindows}
\usage{Integer nwindows ();}
\description
  The \var{nwindows} function returns the number of windows currently visible.
  If the variable \var{MINIBUFFER_ACTIVE} is non-zero, the minibuffer is busy and
  contributes to the number of windows.
\seealso{splitwindow, onewindow, window_size}
\seealso{MINIBUFFER_ACTIVE}
\done

\function{add_completion}
\synopsis{add_completion}
\usage{Void add_completion(String f);}
\description
  The \var{add_completion} function adds the user defined S-Lang function
  with name specified by the string \var{f} to the list of functions that
  are eligible for mini-buffer completion.  The function specified by
  \var{f} must be already defined before this function is called.  The
  S-Lang function \var{is_defined} may be used to test whether or not the
  function is defined.
\seealso{read_with_completion, _add_completion}
\done

\function{what_column}
\synopsis{what_column}
\usage{Integer what_column ();}
\description
  The \var{what_column} function returns the current column number expanding
  tabs, control characters, etc...  The beginning of the line is at
  column number one.
\seealso{whatline, whatpos, goto_column, bolp, eolp}
\done

\function{eobp}
\synopsis{eobp}
\usage{Integer eobp ();}
\description
  The functio \var{eobp} is used to determine if the current position is at
  the end of the buffer or not.  It returns a non-zero value if at the
  end of the buffer or zero if not.
\seealso{eob, bolp, eolp}
\done

\function{set_mode}
\synopsis{set_mode}
\usage{ Void set_mode(String mode, Integer flags);}
\description
  This function sets buffer mode flags and status line mode name.  \var{mode}
  is a string which is displayed on the status line if the \exmp{%m} status
  line format specifier is used. The second argument, \var{flags} is an
  integer with the possible values:
#v+
        0 : no mode. Very generic
        1 : Wrap mode.  Lines are automatically wrapped at wrap column.
        2 : C mode.
        4 : Language mode.  Mode does not wrap but is useful for computer
            languages.
        8 : S-Lang mode
        16: Fortran mode highlighting
        32: TeX mode highlighting
#v-
\seealso{whatmode, getbuf_info, setbuf_info.}
\done

\function{buffer_visible}
\synopsis{buffer_visible}
\usage{Integer buffer_visible (String buf);}
\description
  This function is used to determine whether or not a buffer with name
  specified by the string \var{buf} is in a window or not.  More explicitly,
  it returns the number of windows containing \var{buf}.  This means that if
  \var{buf} does not occupy a window, it returns zero.  For Example,
#v+
        define find_buffer_in_window (buf)
        {
           !if (buffer_visible (buf)) return 0;
           pop2buf (buf);
           return 1;
        }
#v-
  is a function that moves to the window containing \var{buf} if \var{buf} is in
  a window.
\seealso{bufferp, nwindows}
\done

\function{exit_jed}
\synopsis{exit_jed}
\usage{Void exit_jed ();}
\description
  This function should be called to exit JED is a graceful and safe
  manner.  If any buffers have been modified but not saved, the user is
  queried about whether or not to save each one first.  \var{exit_jed} calls
  the S-Lang hook \var{exit_hook} if it is defined.  If \var{exit_hook} is
  defined, it must either call \var{quit_jed} or \var{exit_jed} to really exit
  the editor.  If \var{exit_jed} is called from \var{exit_hook}, \var{exit_hook} will
  not be called again.  For example:
#v+
        define exit_hook ()
        {
          flush ("Really Exit?");
       
          forever
            {
              switch (getkey () & 0x20)    % map to lowercase
               { case 'y': exit_jed (); }
               { case 'n': return; }
              beep ();
            }
        }
#v-
  may be used to prompt user for confirmation of exit.
\seealso{quit_jed, suspend, flush, getkey}
\seealso{BATCH}
\done

\function{set_color}
\synopsis{set_color}
\usage{Void set_color(String obj, String fg, String bg);}
\description
  This function sets the foreground and background colors of an object
  specified by the string \var{obj} to \var{fg} and \var{bg}.  The exact values of
  the strings \var{fg} and \var{bg} are system dependent.  For the X-Window
  system, they can be any string that the server understands, e.g.,
  \exmp{"SteelBlue"}.  For other systems, the color must be one of the
  following:
#v+
        "black"            "gray"
        "red"              "brightred"
        "green"            "brightgreen"
        "brown"            "yellow"
        "blue"             "brightblue"
        "magenta"          "brightmagenta"
        "cyan"             "brightcyan"
        "lightgray"        "white"
#v-
  One most terminals, the values in the second column have no affect
  when used as the background color.

  The valid names for \var{obj} are:
#v+
        "normal"      Default foreground/background
        "status"      The status window line
        "region"      Highlighted Regions
        "cursor"      Text Cursor (X-Windows)
        "menu"        The menu bar
        "error"       Error messages
        "message"     Other messages
        "dollar"      Color of the indicator that text extends beyond the
                       boundary of the window.
#v-
  If color syntax highlighting is enabled, the following object names
  are also meaningful:
#v+
        "number"      Numbers in C-mode and Equations in TeX-mode
        "delimiter"   Commas, semi-colons, etc...
        "keyword"     Language dependent
        "string"      Literal strings
        "comment"     Comments
        "operator"    Such as +, -, etc...
        "preprocess"  Preprocessor lines
#v-
  If line attributes are available, then you may also specifiy the color
  of the hidden line indicator:
#v+
        "..."         Hidden line indicator
#v-
  See a discussion in the JED User Manual for more discussion.
\seealso{set_color_esc, set_column_colors, set_color_object}
\seealso{WANT_SYNTAX_HIGHLIGHT, USE_ANSI_COLORS}
\done

\function{color_number}
\synopsis{color_number}
\usage{Integer color_number (String obj);}
\description
  This function returns the object number associated with the
  string \var{obj}.  Valid names for \var{obj} are as per \var{set_color}.
\seealso{set_color, set_column_colors}
\seealso{WANT_SYNTAX_HIGHLIGHT, USE_ANSI_COLORS}
\done

\function{set_color_esc}
\synopsis{set_color_esc}
\usage{Void set_color_esc (String object, String esc_seq);}
\description
  This function may be used to associate an escape sequence with an
  object.  The escape sequence will be sent to the terminal prior to
  sending updating the object.  It may be used on mono terminals to
  underline objects, etc...  The object names are the same names used by
  the \var{set_color} function.
  Note: Care should be exercised when using
  this function.  Also, one may need to experiment around a little to
  get escape sequences that work together.
\seealso{set_color}
\done

\function{extract_filename}
\synopsis{extract_filename}
\usage{String extract_filename (String filespec);}
\description
  This function may be used to separate the file name from the path of
  of a file specified by \var{filespec}.  For example, under Unix, the
  expression
#v+
        extract_filename ("/tmp/name");
#v-
  returns the string \exmp{"name"}.
\seealso{expand_filename}
\done

\function{trim}
\synopsis{trim}
\usage{Void trim ();}
\description
  The \var{trim} function removes all whitespace around the current editing
  point.  In this context, whitespace is considered to be any
  combination of tab and space characters.  In particular, it does not
  include the newline character.  This means that the \var{trim} function
  will not delete across lines.
\seealso{skip_chars, skip_white, del, del_region}
\done

\function{pop2buf}
\synopsis{pop2buf}
\usage{Void pop2buf (String buf);}
\description
  The \var{pop2buf} function will switch to another window and display the
  buffer specified by \var{buf} in it.  If \var{buf} does not exist, it will be
  created. If \var{buf} already exists in a window, the window containing
  \var{buf} will become the active one.  This function will create a new
  window if necessary.  All that is guaranteed is that the current
  window will continue to display the same buffer before and after the
  call to \var{pop2buf}.
\seealso{whatbuf, pop2buf_whatbuf, setbuf, sw2buf, nwindows}
\done

\function{pop2buf_whatbuf}
\synopsis{pop2buf_whatbuf}
\usage{String pop2buf_whatbuf (String buf);}
\description
  This function performs the same function as \var{pop2buf} except that the
  name of the buffer that \var{buf} replaced in the window is returned.
  This allows one to restore the buffer in window to what it was before
  the call to \var{pop2buf_whatbuf}.
\seealso{pop2buf, whatbuf}
\done

\function{copy_file}
\synopsis{copy_file}
\usage{Integer copy_file (String src, String dest);}
\description
  This function may be used to copy a file named \var{src} to a new file
  named \var{dest}.  It attempts to preserve the file access and modification
  times as well as the ownership and protection.

  It returns \exmp{0} upon success and \exmp{-1} upon failure.
\seealso{rename_file, file_status}
\done

\function{copy_region}
\synopsis{copy_region}
\usage{Void copy_region (String buf);}
\description
  This function may be used to copy a region defined by a mark and the
  current position to the buffered specified by the name \var{buf}. It does
  not delete the characters in region but it does pop the mark that
  determines the region.
\seealso{insbuf, bufsubstr, push_mark, pop_mark, bufferp}
\done

\function{insbuf}
\synopsis{insbuf}
\usage{Void insbuf (String buf);}
\description
  This function may be used to insert the contents of a buffer specified
  by the name \var{buf} into the current buffer.  The editing position is
  advanced to the end of the insertion.
\seealso{copy_region, narrow, narrow_to_region}
\done

\function{bolp}
\synopsis{bolp}
\usage{Integer bolp ();}
\description
  \var{bolp} is used to test if the current position is at the beginning of
  a line or not.  It returns non-zero if the position is at the
  beginning of a line or zero if not.
\seealso{bol, eolp, bobp, eobp}
\done

\function{beep}
\synopsis{beep}
\usage{Void beep ();}
\description
  The \var{beep} function causes the terminal to beep according to the value
  of the variable \var{IGNORE_BEEP}.
\seealso{tt_send}
\seealso{IGNORE_BEEP}
\done

\function{onewindow}
\synopsis{onewindow}
\usage{Void onewindow ();}
\description
  This function deletes all other windows except the current window and
  the mini-buffer window.
\seealso{nwindows, splitwindow, enlargewin}
\seealso{MINIBUFFER_ACTIVE}
\done

\function{pop_spot}
\synopsis{pop_spot}
\usage{Void pop_spot ();}
\description
  This function is used after a call to \var{push_spot} to return to the
  editing position at the last call to \var{push_spot} in the current buffer.
\seealso{push_spot, pop_mark}
\done

\function{push_spot}
\synopsis{push_spot}
\usage{Void push_spot ();}
\description
  \var{push_spot} pushes the location of the current buffer location onto a
  stack.  This function does not set the mark.  The function \var{push_mark}
  should be used for that purpose. The spot can be returned to using the
  function \var{pop_spot}.
  Note: Spots are local to each buffer.  It is not
  possible to call \var{push_spot} from one buffer and then subsequently
  call \var{pop_spot} from another buffer to return to the position in the
  first buffer.  For this purpose, one must use user marks instead.
\seealso{pop_spot, push_mark, create_user_mark}
\done

\function{sw2buf}
\synopsis{sw2buf}
\usage{Void sw2buf (String buf);}
\description
  This function is used to switch to another buffer whose name is
  specified by the parameter \var{buf}.  If the buffer specified by \var{buf}
  does not exist, one will be created.
  Note: Unlike \var{setbuf}, the change
  to the new buffer is more permanent in the sense that when control
  passed back out of S-Lang to the main editor loop, if the current
  buffer at that time is the buffer specified here, this buffer will be
  attached to the window.
\seealso{setbuf, pop2buf, bufferp}
\done

\function{tt_send}
\synopsis{tt_send}
\usage{Void tt_send (String s);}
\description
  This function may be used to send a string specified by \var{s} directly
  to the terminal with no interference by the editor.  One should
  exercise caution when using this routine since it may interfere with
  JED's screen management routines forcing one to redraw the screen.
  Nevertheless, it can serve a useful purpose.  For example, when run in
  an XTerm window, using
#v+
        tt_send ("\e[?9h");
#v-
  will enable sending mouse click information to JED encoded as
  keypresses.
\seealso{beep}
\done

\function{eolp}
\synopsis{eolp}
\usage{Integer eolp ();}
\description
  This function may be used to determine whether or not the current
  position is at the end of a line ot not.  If it is, the routine
  returns a non-zero value; otherwise it returns zero.
\seealso{eol, bolp, eobp, bobp}
\done

\function{what_keymap}
\synopsis{what_keymap}
\usage{String what_keymap ();}
\description
  This function returns the name of the keymap associated with the
  current buffer.
\seealso{create_keymap, keymap_p}
\done

\function{find_file}
\synopsis{find_file}
\usage{Integer find_file (String name);}
\description
  The \var{find_file} function switches to the buffer associated with the
  file specified by \var{name}.  If no such buffer exists, one is created
  and the file specified by \var{name} is read from the disk and associated
  with the new buffer.  The buffer will also become attached to the
  current window.  Use the \var{read_file} function to find a file but not
  associate it with the current window.
\seealso{read_file}
\done

\function{set_status_line}
\synopsis{set_status_line}
\usage{set_status_line (String format, Integer flag);}
\description
  This function may be used to customize the status line of the current
  window according to the string \var{format}.  If the second parameter
  \var{flag} is non-zero, \var{format} will apply to the global format string;
  otherwise it applies to current buffer only.  Newly created buffer
  inherit the global format string when they appear in a window.
  The format string may contain the following format specifiers:
#v+
        %b   buffer name
        %f   file name
        %v   JED version
        %t   current time --- only used if variable DISPLAY_TIME is non-zero
        %p   line number or percent string
        %%   literal '%' character
        %m   mode string
        %a   If abbrev mode, expands to "abbrev"
        %n   If buffer is narrowed, expands to "Narrow"
        %o   If overwrite mode, expands to "Ovwrt"
        %c   If the variable LINENUMBERS is 2, this expands to the current
               column number.
#v-
  For example, the default status line used by JED's EDT emulation uses
  the format string:
#v+
        "(Jed %v) EDT: %b   (%m%a%n%o)  %p,%c   Advance   %t"
#v-
\seealso{set_mode, narrow, whatbuf, getbuf_info}
\seealso{DISPLAY_TIME,LINENUMBERS}
\done

\function{bury_buffer}
\synopsis{bury_buffer}
\usage{Void bury_buffer (String name);}
\description
  The \var{bury_buffer} function may be used to make it unlikely for the
  buffer specified by the paramter \var{name} to appear in a window.
\seealso{sw2buf, getbuf_info}
\done

\function{dupmark}
\synopsis{dupmark}
\usage{Integer dupmark ();}
\description
  This function returns zero if the mark is not set or, if the mark is
  set, a duplicate of it is pushed onto the mark stack and a non-zero
  value is returned.
\seealso{push_mark, markp, pop_mark}
\done

\function{erase_buffer}
\synopsis{erase_buffer}
\usage{erase_buffer ();}
\description
  The \var{erase_buffer} function erases all text from the current buffer.
  However, it does not delete the buffer itself.

  Note: This function destroys all undo information associated with the
  buffer making it impossible to undo the result of this function.
\seealso{delbuf, del}
\done

\function{window_info}
\synopsis{window_info}
\usage{Integer window_info(Integer item);}
\description
  The \var{window_info} function returns information concerning the current
  window.  The actual information that is returned depends on the \var{item}
  parameter.  Acceptable values of \var{item} and the description of the
  information returned is given in the following table:
#v+
        'r'  : Number of rows
        'w'  : Width of window
        'c'  : Starting column (from 1)
        't'  : Screen line of top line of window (from 1)
\seealso{otherwindow, nwindows}
#v-
\seealso{SCREEN_HEIGHT,SCREEN_WIDTH}
\done

\function{copy_region_to_kill_array}
\synopsis{copy_region_to_kill_array}
\usage{Void copy_region_to_kill_array (Integer n);}
\description
  This function copies the currently defined region to the nth element,
  specified by \var{n}, of an internal array of character strings replacing
  what is currently there.

  Note: This function is not available on 16 bit systems.
\seealso{insert_from_kill_array, append_region_kill_array}
\seealso{KILL_ARRAY_SIZE}
\done

\function{append_region_to_kill_array}
\synopsis{append_region_to_kill_array}
\usage{Void append_region_to_kill_array (Integer n);}
\description
  This function appends the currently defined region to the contents of
  nth element, specified by \var{n}, of an internal array of character strings.

  Note: This function is not available on 16 bit systems.
\seealso{insert_from_kill_array, copy_region_to_kill_array}
\seealso{KILL_ARRAY_SIZE}
\done

\function{insert_from_kill_array}
\synopsis{insert_from_kill_array}
\usage{Void insert_from_kill_array (Integer n);}
\description
  This function inserts the contents of the nth element, specified by
  \var{n}, of an internal array of character strings.

  Note: This function is not available on 16 bit systems.
\seealso{insert_from_kill_array, copy_region_to_kill_array}
\seealso{KILL_ARRAY_SIZE}
\done

\function{goto_column}
\synopsis{goto_column}
\usage{Void goto_column (Integer n);}
\description
  This function moves the current editing point to the column specified
  by the parameter \var{n}.  It will insert a combination of spaces and tabs
  if necessary to achieve the goal.
  Note: The actual character number offset from the beginning of the
  line depends upon tab settings and the visual expansion of other
  control characters.
\seealso{goto_column_best_try, what_column, left, right, goto_line}
\seealso{TAB,TAB_DEFAULT,DISPLAY_EIGHT_BIT}
\done

\function{goto_column_best_try}
\synopsis{goto_column_best_try}
\usage{Integer goto_column_best_try (Integer c);}
\description
  This function is like \var{goto_column} except that it will not insert
  whitespace.  This means that it may fail to achieve the column number
  specified by the argument \var{c}.  It returns the current column number.
\seealso{goto_column, what_column}
\done

\function{goto_line}
\synopsis{goto_line}
\usage{Void goto_line (Integer n);}
\description
  The \var{goto_line} function may be used to move to a specific line number
  specified by the parameter \var{n}.
  Note: The actual column that the editing point will be left in is
  indeterminate.
\seealso{what_line, goto_column, down, up.}
\done

\function{pop_mark}
\synopsis{pop_mark}
\usage{pop_mark (Integer g);}
\description
  \var{pop_mark} pops the most recent mark pushed onto the mark stack.  If
  the argument \var{g} is non-zero, the editing position will be moved to
  the location of the mark.  However, if \var{g} is zero, the editing
  position will be unchanged.
\seealso{push_mark, pop_spot, markp, check_region, goto_user_mark}
\done

\function{read_mini}
\synopsis{read_mini}
\usage{String read_mini (String prompt, String dflt, String init);}
\description
  The \var{read_mini} function reads a line of input from the user in the
  mini-buffer.  The first parameter, \var{prompt}, is used to prompt the
  user.  The second parameter, \var{dflt}, is what is returned as a default
  value if the user simply presses the return key.  The final parameter,
  \var{init}, is stuffed into the mini-buffer for editing by the user.
  For example,
#v+
        define search_whole_buffer ()
        {
          variable str;
          str = read_mini ("Search for:", "", "");
          !if (strlen (str)) return;
          !if (fsearch (str))
             {
               push_mark (); bob ();
               if (fsearch (str)) pop_mark (0);
               else pop_mark (1);
                 {
                    pop_mark (1);
                    error ("Not found");
                 }
             }
        }
#v-
  reads a string from the user and then searches forward for it and if
  not found, it resumes the search from the beginning of the buffer.
  Note: If the user aborts the function \var{mini_read} by pressing the
  keyboard quit character (e.g., Ctrl-G), an error is signaled.  This
  error can be caught by an \var{ERROR_BLOCK} and the appropriate action
  taken. Also if the mini-buffer is already in use, this function should
  not be called.  The variable \var{MINIBUFFER_ACTIVE} may be checked to
  determine if this is the case or not.
\seealso{read_with_completion, getkey, input_pending}
\seealso{MINIBUFFER_ACTIVE}
\done

\function{file_status}
\synopsis{file_status}
\usage{Integer file_status (String filename);}
\description
  The \var{file_status} function returns information about a file specified
  by the name \var{filename}.  It returns an integer describing the file
  type:
   2     file is a directory
   1     file exists and is not a directory
   0     file does not exist.
  -1     no access.
  -2     path invalid
  -3     unknown error
\done

\function{skip_white}
\synopsis{skip_white}
\usage{Void skip_white ();}
\description
  The \var{skip_white} function moves the current point forward until it
  reaches a non-whitespace character or the end of the current line,
  whichever happens first.  In this context, whitespace is considered to
  be any combination of space and tab characters.  To skip newline
  characters as well, the function \var{skip_chars} may be used.
\seealso{bskip_chars, what_char, trim, right}
\done

\function{bobp}
\synopsis{bobp}
\usage{Integer bobp ();}
\description
  The \var{bolp} function is used to determine if the current position is at
  the beginning of the buffer or not.  If so, it returns a non-zero
  value.  However, if it is not, it returns zero.  This simple example,
#v+
        define is_buffer_empty ()
        {
          return bobp () and eobp ();
        }
#v-
  returns non-zero if the buffer is empty; otherwise, it returns zero.
\seealso{bob, eobp, bolp, eolp}
\done

\function{flush}
\synopsis{flush}
\usage{Void flush (String msg);}
\description
  The \var{flush} function behaves like \var{message} except that it immediately
  displays its argument \var{msg} as a message in the mini-buffer.  That is,
  it is not necessary to call \var{update} to see the message appear.
\seealso{message, error}
\done

\function{input_pending}
\synopsis{input_pending}
\usage{Integer input_pending (Integer tsecs);}
\description
  This function is used to see if keyboard input is available to be read
  or not. The paramter \var{tsecs} is the amount of time to wait for input
  before returning if input is not available.  The time unit for \var{tsecs}
  is one-tenth of a second.  That is, to wait up to one second, pass a
  value of ten to this routine.  It returns zero if no input is
  available, otherwise it returns non-zero.  As an example,
#v+
        define peek_key ()
        {
          variable ch;
          !if (input_pending (0)) return -1;
          ch = getkey ();
          ungetkey (ch);
          return ch;
        }
#v-
  returns the value of the next character to be read if one is
  available; otherwise, it returns -1.
\seealso{getkey, ungetkey}
\done

\function{usleep}
\synopsis{usleep}
\usage{Void usleep (Integer ms);}
\description
  A call to usleep will cause the editor to pause for \var{ms} milliseconds.
\seealso{input_psnding}
\done

\function{insert_file}
\synopsis{insert_file}
\usage{Integer insert_file (String f);}
\description
  This function may be used to insert the contents of a file named \var{f}
  into the buffer at the current position.  The current editing point
  will be placed at the end of the inserted text.  The function returns
  \exmp{-1} if the file was unable to be opened; otherwise it returns the
  number of lines inserted.  This number can be zero if the file is empty.
\seealso{read_file, find_file, insert}
\done

\function{keymap_p}
\synopsis{keymap_p}
\usage{Integer keymap_p (String kmap);}
\description
  The \var{keymap_p} function may be used to determine whether or not a
  keymap with name \var{kmap} exists.  If the keymap specified by \var{kmap}
  exists, the function returns non-zero.  It returns zero if the keymap
  does not exist.
\seealso{make_keymap, definekey}
\done

\function{what_char}
\synopsis{what_char}
\usage{Integer what_char ();}
\description
  The \var{what_char} function returns the value of the character at the
  current position as an integer in the range 0 to 256.  This simple
  example,
#v+
        while (not (eolp ()))
          {
             if (what_char () == '_')
               {
                  del (); insert ("\\_");
               }
          }
#v-
  has the effect of replacing all underscore characters on the current
  line with a backslash-underscore combination.
\seealso{looking_at}
\done

\function{recenter}
\synopsis{recenter}
\usage{Void recenter (Integer nth);}
\description
  This function may be used to scroll the window such that the \var{nth} line
  of the window contains the current line.  If \var{nth} is zero, the current
  line will be placed at the center of the window and the screen will be
  completely redrawn.
\seealso{nwindows, window_info}
\done

\function{bufferp}
\synopsis{bufferp}
\usage{Integer bufferp (String buf);}
\description
  This function is used to see if a buffer exists or not.  If a buffer with
  name \var{buf} exists, it returns a non-zero value.  If it does not exist,
  it returns zero.
\seealso{setbuf, getbuf_info}
\done

\function{get_key_function}
\synopsis{get_key_function}
\usage{String get_key_function ();}
\description
  \var{get_key_function} waits for a key to be pressed and returns a string
  that represents the binding of the key.  If the key has no binding
  the empty string is returned.  Otherwise, it also returns an integer
  that describes whether or not the function is an internal one.  If
  the function is internal, \var{1} will be returned; otherwise zero will
  be returned to indicate that the binding is either to an S-Lang
  function or a macro.  If it is a macro, the first character of the
  of returned string will be the \exmp{@} character.
\seealso{getkey, input_pending}
\done

\function{dump_bindings}
\synopsis{dump_bindings}
\usage{Void dump_bindings(String map);}
\description
  This functions inserts a formatted list of keybindings for the keymap
  specified by \var{map} into the buffer at the current point.
\seealso{get_key_function}
\done

\function{undefinekey}
\synopsis{undefinekey}
\usage{Void undefinekey (String key, String kmap);}
\description
  This function may be used to remove a keybinding from a specified
  keymap.  The key sequence is given by the parameter \var{key} and the
  keymap is specified by the second parameter \var{kmap}.
\seealso{unsetkey, definekey, what_keymap}
\done

\function{getpid}
\synopsis{getpid}
\usage{Integer getpid ();}
\description
  This function returns the process identification number for the current
  editor process.
\done

\function{update}
\synopsis{update}
\usage{Void update (Integer f);}
\description
  This function may be called to update the display.  If the parameter
  \var{f} is non-zero, the display will be updated even if there is input
  pending.  If \var{f} is zero, the display may only be partially updated if
  input is pending.
\seealso{input_pending, flush}
\done

\function{skip_word_chars}
\synopsis{skip_word_chars}
\usage{Void skip_word_chars ();}
\description
  This function moves the current editing point forward across all
  characters that constitute a word until a non-word character is
  encountered. Characters that make up a word are set by the
  \var{define_word} function.
\seealso{define_word, skip_non_word_chars, skip_chars, bskip_word_chars}
\done

\function{skip_non_word_chars}
\synopsis{skip_non_word_chars}
\usage{Void skip_non_word_chars ();}
\description
  This function moves the current editing point forward past all
  non-word characters until a word character is encountered.
  Characters that make up a word are set by the \var{define_word} function.
\seealso{define_word, skip_word_chars, skip_chars, bskip_non_word_chars}
\done

\function{bskip_word_chars}
\synopsis{bskip_word_chars}
\usage{Void bskip_word_chars ();}
\description
  This function moves the current editing point backward past all
  word characters until a non-word character is encountered.
  Characters that make up a word are set by the \var{define_word} function.
\seealso{define_word, skip_word_chars, bskip_chars, bskip_non_word_chars}
\done

\function{bskip_non_word_chars}
\synopsis{bskip_non_word_chars}
\usage{Void bskip_word_chars ();}
\description
  This function moves the current editing point backward past all
  non-word characters until a word character is encountered.
  Characters that make up a word are set by the \var{define_word} function.
\seealso{define_word, skip_non_word_chars, bskip_chars, bskip_word_chars}
\done

\function{which_key}
\synopsis{which_key}
\usage{Integer which_key (String f);}
\description
  The \var{which_key} function returns the the number of keys that are
  bound to the function \var{f} in the current keymap.  It also returns
  that number of key sequences with control characters expanded as the
  two character sequence \exmp{^} and the the whose ascii value is the
  control character + 64. For example,
#v+
        define insert_key_bindings (f)
        {
           variable n, key;
           n = which_key (f);
           loop (n)
             {
                 str = ();
                 insert (str);
                 insert ("\n");
             }
        }
#v-
  inserts into the buffer all the key sequences that are bound to the
  function \var{f}.
\seealso{get_key_function, setkey, what_keymap}
\done

\function{whitespace}
\synopsis{whitespace}
\usage{whitespace (Integer n);}
\description
  The \var{whitespace} function inserts white space of length \var{n} into the
  current buffer using a combination of spaces and tabs.  The actual
  combination of spaces and tabs used depends upon the buffer local
  variable \var{TAB}.  In particular, if \var{TAB} is zero, no tab characters
  will be used for the expansion.
\seealso{insert, trim, goto_column}
\seealso{TAB,TAB_DEFAULT}
\done

\function{enlargewin}
\synopsis{enlargewin}
\usage{Void enlargewin ();}
\description
  This function increases the size of the current window by one line by
  adjusting the size of the other windows accordingly.
\seealso{window_info, onewindow}
\done

\function{splitwindow}
\synopsis{splitwindow}
\usage{Void splitwindow ();}
\description
  This function splits the current window vertically creating another
  window that carries the current window's buffer.
\seealso{onewindow, enlargewin, window_info}
\done

\function{file_time_compare}
\synopsis{file_time_compare}
\usage{Integer file_time_cmp (String file1, String file2);}
\description
  This function compares the modification times of two files,
  \var{file1} and \var{file2}. It returns an integer that is either
  positive, negative, or zero integer for \exmp{file1 > file2},
  \exmp{file1 < file2}, or \exmp{file1 == file2}, respectively.  In
  this context, the comparison operators are comparing file
  modification times.  That is, the operator \exmp{>} should be read
  ``is more recent than''.  The convention adopted by this routine is
  that if a file does not exist, its modification time is taken to be
  at the beginning of time.  Thus, if \var{f} exists, but \var{g} does
  not, the \exmp{file_time_compare (f, g)} will return a positive
  number. 
\seealso{file_status, time}
\done

\function{xform_region}
\synopsis{xform_region}
\usage{Void xform_region (Integer how);}
\description
  This function changes the characters in the region in a way specified
  by the parameter \var{how}.  This is an integer that can be any of of the
  following:
#v+
        'u'       Upcase_region
        'd'       Downcase_region
        'c'       Capitalize region
#v-
  Anything else will change case of region.
\seealso{translate_region, define_case}
\done

\function{skip_chars}
\synopsis{skip_chars}
\usage{Void skip_chars(String s);}
\description
  This fnction may be used to move the editing point forward past all
  characters in string \var{s} which contains the chars to skip, or a range
  of characters.  A character range is denoted by two charcters
  separated by a hyphen.  If the first character of the string \var{s} is a
  \exmp{'^'} character, then the list of characters actually denotes the
  complement of the set of characters to be skipped.  To explicitly
  include the hyphen character in the list, it must be either the first
  or the second character of the string, depending upon whether or not
  the \exmp{'^'} character is present. So for example,
#v+
        skip_chars ("- \t0-9ai-o_");
#v-
  will skip the hyphen, space, tab, numerals \exmp{0} to \var{9}, the letter \var{a},
  the letters \var{i} to \var{o}, and underscore.  An example which illustrates
  the complement of a range is
#v+
        skip_chars("^A-Za-z");
#v-
  which skips all characters except the letters.
  Note: The backslash character may be used to escape only the first
  character in the string.  That is, \exmp{"\\\\^"} is to be used to skip over
  \exmp{^} characters.
\seealso{bskip_chars, skip_white}
\done

\function{set_file_translation}
\synopsis{set_file_translation}
\usage{set_file_translation (Integer n);}
\description
  This function affects only the way the next file is opened.  Its
  affect does not last beyond that.  If it the value of the parameter
  is 1, the next file will be opened in binary mode.  If the parameter is
  zero, the file will be opened in text mode.
\done

\function{pipe_region}
\synopsis{pipe_region}
\usage{Integer pipe_region (String cmd);}
\description
  The \var{pipe_region} function executes \var{cmd} in a separate process and
  sends the region of characters defined by the mark and the current
  point to the standard input of the process.  It successful, it returns
  the exit status of the process.  Upon failure it signals an error.
  Note: This function is only available for Unix and OS/2 systems.
\seealso{run_shell_cmd, push_mark}
\done

\function{run_shell_cmd}
\synopsis{run_shell_cmd}
\usage{Void run_shell_cmd (String cmd);}
\description
  The \var{run_shell_cmd} function may be used to run \var{cmd} in a separate
  process.  Any output generated by the process is inserted into the
  buffer at the current point.  It generates a S-Lang error if the
  process specified by \var{cmd} could not be opened.  Otherwise, it
  returns the exit status of the process.
\done

\function{append_region_to_file}
\synopsis{append_region_to_file}
\usage{Integer append_region_to_file (String file);}
\description
  Appends a marked region to \var{file} returning number of lines
  written or \-1 on error.  This does NOT modify a buffer visiting the
  file; however, it does flag the buffer as being changed on disk.
\done

\function{autosave}
\synopsis{autosave}
\usage{Void autosave ();}
\description
  The \var{autosave} function saves the current buffer in an auto save file
  if the buffer has been marked for the auto save operation.
\seealso{setbuf_info, autosaveall}
\seealso{MAX_HITS}
\done

\function{autosaveall}
\synopsis{autosaveall}
\usage{Void autosaveall ();}
\description
  This function is like \var{autosave} except that it causes all files
  marked for the auto save operation to be auto-saved.
\seealso{autosave, setbuf_info}
\seealso{MAX_HITS}
\done

\function{backward_paragraph}
\synopsis{backward_paragraph}
\usage{Void backward_paragraph ();}
\description
  This function moves the current editing point backward past the
  current paragraph to the line that is a paragraph separator.  Such a
  line is determined by the S-Lang hook \var{is_paragraph_separator}.  This
  hook can be modified on a buffer by buffer basis by using the
  function \var{set_buffer_hook}.
\seealso{forward_paragraph, set_buffer_hook}
\done

\function{blank_rect}
\synopsis{blank_rect}
\description
   The \var{blank_rect} function replaces all text in the rectangle defined by
  the current editing point and the mark by spaces.
\seealso{push_mark, kill_rect, insert_rect, copy_rect}
\done

\function{bskip_chars}
\synopsis{bskip_chars}
\usage{Void bskip_chars (String str);}
\description
  This function may be used to skip past all characters defined by the
  string \var{str}.  See \var{skip_chars} for the definition of \var{str}.
  The following example illustrates how to skip past all whitespace
  including newline characters:
#v+
        bskip_chars (" \t\n");
#v-
\seealso{skip_chars, left}
\done

\function{buffer_list}
\synopsis{buffer_list}
\usage{Integer buffer_list ();}
\description
  This function returns an integer indicating the number of buffers and
  leaves the names of the buffers on the stack.  For example, the
  following function displays the names of all buffers attached to
  files:
#v+
        define show_buffers ()
        {
           variable b, str = "", file;
           loop (buffer_list ())
             {
                 b = ();
                 setbuf (b);
                 (file,,,) = getbuf_info ();
                 if (strlen (file)) str = strcat (str, strcat (" ", b));
             }
           message (str);
        }
#v-
\seealso{getbuf_info, whatbuf}
\done

\function{check_region}
\synopsis{check_region}
\usage{Void check_region (Integer ps);}
\description
  This function checks to see if a region is defined and may exchange
  the current editing point and the mark to define a canonical region.
  If the mark is not set, it signals an S-Lang error.  A canonical
  region is one with the mark set earlier in the buffer than than the
  editing point.  Always call this if using a region which requires
  such a situation.

  If the argument \var{ps} is non-zero, \var{push_spot} will be called,
  otherwise, \var{ps} is zero and it will not be called.

  As an example, the following function counts the number of lines in
  a region:
#v+
        define count_lines_region ()
        {
           variable n;
           check_region (1);   % spot pushed
           narrow ();
           n = what_line ();
           widen ();
           pop_spot ();
           return n;
         }
#v-
\seealso{markp, push_mark}
\done

\function{copy_rect}
\synopsis{copy_rect}
\usage{Void copy_rect ();}
\description
  The \var{copy_rect} function is used to copy the contents of the
  currently defined rectangle to the rectangle buffer.  It overwrites
  the previous contents of the rectangle buffer.  A rectangle is defined
  by the diagonal formed by the mark and the current point.
\seealso{insert_rect, kill_rect, blank_rect}
\done

\function{define_word}
\synopsis{define_word}
\usage{Void define_word (String s);}
\description
  This function is used to define the set of characters that form a
  word. The string \var{s} consists of those characters or ranges of
  characters that define the word.  For example, to define only the
  characters \exmp{A-Z} and \exmp{a-z} as word characters, use:
#v+
        define_word ("A-Za-z");
#v-
  To include a hyphen as part of a word, it must be the first character
  of the control string \var{s}.  So for example,
#v+
        define_word ("-i-n");
#v-
  defines a word to consist only of the letters \var{i} to \var{n} and the
  hyphen character.
\done

\function{delbuf}
\synopsis{delbuf}
\usage{Void delbuf (String buf);}
\description
  \var{delbuf} may be used to delete a buffer with the name specified by
  \var{buf}.  If the buffer does not exist, a S-Lang error will be generated.
\seealso{whatbuf, bufferp, sw2buf}
\done

\function{delete_file}
\synopsis{delete_file}
\usage{Integer delete_file (String file);}
\description
  This function may be used to delete a file specified by the \var{file}
  parameter.  It returns non-zero if the file was sucessfully deleted
  or zero otherwise.
\seealso{rmdir}
\done

\function{directory}
\synopsis{directory}
\description
   returns number of files and list of files which match filename.
      On unix, this defaults to filename*.  It is primarily useful for
      DOS and VMS to expand wildcard filenames
\done

\function{evalbuffer}
\synopsis{evalbuffer}
\usage{Void evalbuffer ();}
\description
  This function causes the current buffer to be sent to the S-Lang
  interpreter for evaluation.  If an error is encountered while parsing
  the buffer, the cursor will be placed at the location of the error.
\seealso{evalfile}
\done

\function{expand_filename}
\synopsis{expand_filename}
\usage{String expand_filename (String file);}
\description
  The \var{expand_filename} function expands a file to a canonical form.
  For example, under Unix, if \var{file} has the value \exmp{"/a/b/../c/d"}, it
  returns \exmp{"/a/c/d"}.  Similarly, if \var{file} has the value
  \exmp{"/a/b/c//d/e"}, \exmp{"/d/e"} is returned.
\done

\function{file_changed_on_disk}
\synopsis{file_changed_on_disk}
\usage{Integer file_changed_on_disk (String fn);}
\description
  This function may be used to determine if the disk file specified by the
  parameter \var{fn} is more recent than the current buffer.
\seealso{file_time_compare, check_buffers}
\done

\function{forward_paragraph}
\synopsis{forward_paragraph}
\usage{Void forward_paragraph ();}
\description
  This function moves the current editing point forward past the end of
  the current paragraph.  Paragraph delimiters are defined through either
  a buffer hook or via the hook \var{is_paragraph_separator}.
\seealso{backward_paragraph, set_buffer_hook}
\done

\function{getkey}
\synopsis{getkey}
\usage{Integer getkey ();}
\description
  The \var{getkey} function may be used to read an input character from the
  keyboard.  It returns an integer in the range 0 to 256 which represents
  the ASCII or extended ASCII value of the character.
\seealso{input_pending, ungetkey}
\done

\function{indent_line}
\synopsis{indent_line}
\usage{Void indent_line ();}
\description
  The \var{indent_line} line function indents the current line in a manner
  which depends upon the current buffer.  The actual function that gets
  called is set via a prior call the \var{set_buffer_hook} to set the indent
  hook.  The default value is to indent the line to the indentation
  level of the previous line.
\seealso{set_buffer_hook}
\done

\function{insert_rect}
\synopsis{insert_rect}
\usage{insert_rect ();}
\description
  The \var{insert_rect} function inserts the contents of the rectangle buffer
  at the current editing point.  The rectangle buffer is not modified.
  Any text that the rectangle would overwrite is moved to the right by an
  amount that is equal to the width of the rectangle.
\seealso{kill_rect, blank_rect, copy_rect}
\done

\function{kill_rect}
\synopsis{kill_rect}
\usage{Void kill_rect ();}
\description
  This function deletes the rectangle defined by the mark and the current
  point.  The contents of the rectangle are saved in the rectangle buffer
  for later retrieval via the \var{insert_rect} function.  The previous
  contents of the rectangle buffer will be lost.
\seealso{insert_rect, blank_rect, copy_rect}
\done

\function{make_keymap}
\synopsis{make_keymap}
\usage{Void make_keymap (String km);}
\description
  The \var{make_keymap} function creates a keymap with a name specified by
  the \var{km} parameter.  The new keymap is an exact copy of the
  pre-defined \exmp{"global"} keymap.
\seealso{use_keymap, keymap_p, definekey, setkey}
\done

\function{map_input}
\synopsis{map_input}
\usage{Void map_input (Integer x, Integer y);}
\description
  The \var{map_input} function may be used to remap an input character with
  ascii value \var{x} from the keyboard to a different character with ascii
  value \var{y}.  This mapping can be quite useful because it takes place
  before the editor interprets the character. One simply use of this
  function is to swap the backspace and delete characters.  Since the
  backspace character has an ascii value of \var{8} and the delete character
  has ascii value \var{127}, the statement
#v+
        map_input (8, 127);
#v-
  maps the backspace character to a delete character and
#v+
        map_input (127, 8);
#v-
  maps the delete character to a backspace character.  Used together,
  these two statement effectively swap the delete and backspace keys.
\seealso{getkey}
\done

\function{narrow_to_region}
\synopsis{narrow_to_region}
\usage{Void narrow_to_region (void);}
\description
  The \var{narrow_to_region} function behaves like the \var{narrow} function
  that \var{narrow} operates on lines and \var{narrow_to_region} restricts
  editing to only characters within the region.
\seealso{widen_region, narrow.}
\done

\function{narrow}
\synopsis{narrow}
\usage{Void narrow ();}
\description
  This function may be used to restict editing to the region of lines
  between the mark and the editing point.  The region includes the line
  containing the mark as well as the line at the current point. All
  other lines outside this region are completely inacessable without
  first lifting the restriction using the \var{widen} function. As a simple
  example, suppose that there is a function called \var{print_buffer} that
  operates on the entire buffer.  Then the following function will work
  on a region of lines:
#v+
        define print_region ()
        {
           narrow ();
           print_buffer ();
           widen ();
        }
#v-
  The \var{narrow} function will signal an error if the mark is not set.
  Note also that the narrow function may be used recursively in the
  sense that a narrowed region may be further restricted using the
  \var{narrow} function.  For each narrow, the \var{widen} function must be called
  to lift each restriction.
\seealso{widen, narrow_to_region}
\done

\function{open_rect}
\synopsis{open_rect}
\usage{Void open_rect ();}
\description
  The \var{open_rect} function may be used to insert a blank rectangle whose
  size is determined by the mark and the current editing point.  Any text
  that lies in the region of the rectangle will be pushed to the right.
\seealso{insert_rect, kill_rect, copy_rect}
\done

\function{quit_jed}
\synopsis{quit_jed}
\usage{Void quit_jed ();}
\description
  This function quits the editor immediately.  No buffers are
  auto-saved and no hooks are called.  The function \var{exit_jed} should be
  called when it is desired to exit in a safe way.
\seealso{exit_jed}
\done

\function{read_file}
\synopsis{read_file}
\usage{Integer read_file (string fn);}
\description
  The \var{read_file} function may be used to read a file specified by \var{fn}
  into its own buffer.  It returns a non-zero value upon success and
  signals an error upon failure.  The hook \var{find_file_hook} is called
  after the file is read in.  Unlike the related function, \var{find_file},
  this function does not create a window for the newly created buffer.
\seealso{find_file, file_status, write_buffer}
\done

\function{read_with_completion}
\synopsis{read_with_completion}
\usage{Void read_with_completion (String prt, String dflt, String s, Integer type);}
\description
  This function may be used to read one of the objects specified by the
  last parameter \var{type}.  The first parameter, \var{prt}, is used as a
  prompt, the second parameter, \var{dflt}, is used to specify a default,
  and the third parameter, \var{s}, is used to initialize the string to
  be read.
  \var{type} is an integer with the following meanings:
#v+
        'f'   file name
        'b'   buffer name
        'F'   function name
        'V'   variable name.

#v-
  Finally, if \var{type} has the value \exmp{'s'}, then the set of completions
  will be defined by a zeroth parameter, \var{list}, to the function call.
  This parameter is simple a comma separated list of completions.
  For example,
#v+
        read_with_completion ("Larry,Curly,Moe", "Favorite Stooge:",
                              "Larry", "", 's');
#v-
  provides completion over the set of three stooges.
  The function returns the string read.
\seealso{read_mini}
\done

\function{set_abort_char}
\synopsis{set_abort_char}
\usage{Void set_abort_char (Integer ch);}
\description
  This function may be used to change the keyboard character that
  generates an S-Lang interrupt.  The parameter \var{ch} is the ASCII value
  of the character that will become the new abort character. The
  default abort character \exmp{Ctrl-G} corresponds to \exmp{ch=7}.
\done

\function{suspend}
\synopsis{suspend}
\usage{Void suspend ();}
\description
  The action of this command varies with the operating system.
  Under Unix, the editor will be suspended and control will pass to the
  parent process.  Under VMS and MSDOS, a new subprocess will be spawned.
  Before suspension, \var{suspend_hook} is called.  When the editor is
  resumed, \var{resume_hook} will be called.  These hooks are user-defined
  functions that take no arguments and return no values.
\done

\function{ungetkey}
\synopsis{ungetkey}
\usage{Void ungetkey (Integer ch);}
\description
  This function may be used to push a character \var{ch} represented by its
  ASCII value, onto the input stream.  This means that the next keyboard
  to be read will be \var{ch}.
\seealso{buffer_keystring, getkey, get_key_function}
\done

\function{buffer_keystring}
\synopsis{buffer_keystring}
\usage{Void buffer_keystring (String str);}
\description
  Append string \var{str} to the end of the input stream to be read by JED's
  getkey routines.
\seealso{ungetkey, getkey}
\done

\function{use_keymap}
\synopsis{use_keymap}
\usage{Void use_keymap (String km);}
\description
  This function may be used to dictate which keymap will be used by the
  current buffer.  \var{km} is a string value that corresponds to the name
  of a keymap.
\seealso{make_keymap, keymap_p, what_keymap}
\done

\function{w132}
\synopsis{w132}
\usage{Void w132 ();}
\description
  This function may be used to set the number of columns on a vtxxx
  compatable terminal to 132.
\seealso{w80, set_term_vtxxx}
\done

\function{w80}
\synopsis{w80}
\usage{Void w80 ();}
\description
  This function may be used to set the number of columns on a vtxxx
  compatable terminal to 80.
\seealso{w132, set_term_vtxxx}
\done

\function{what_mode}
\synopsis{what_mode}
\usage{(String name, Integer flags) = Integer what_mode ();}
\description
  This function may be used to obtain the mode flags and mode name of the
  current buffer.  See \var{set_mode} for more details.
\seealso{set_mode, getbuf_info, setbuf_info}
\done

\function{widen}
\synopsis{widen}
\usage{Void widen ();}
\description
  This function undoes the effect of \var{narrow}.  Consult the documentation
  for \var{narrow} for more information.
\seealso{widen_region, narrow}
\done

\function{widen_region}
\synopsis{widen_region}
\usage{Void widen_region ();}
\description
  This function undoes the effect of \var{narrow_to_region}.  Consult the
  documentation for \var{narrow_to_region} for more information.
\seealso{widen, narrow_to_region}
\done

\function{window_line}
\synopsis{window_line}
\usage{Integer window_line ();}
\description
  This function returns the number of rows from the top of the current
  window for the current line.  If the current line is the very first line
  in the window, a value of \var{1} will be returned, i.e., it is the first
  line of the window.
\seealso{window_info, nwindows}
\seealso{TOP_WINDOW_ROW}
\done

\function{write_buffer}
\synopsis{write_buffer}
\usage{Integer write_buffer (String filename);}
\description
  This function may be used to write the current buffer out to a file
  specified by \var{filename}.  The buffer will then become associated with
  that file.  The number of lines written to the file is returned.  An
  error condition will be signaled upon error.
\seealso{write_region_to_file, setbuf_info}
\done

\function{write_region_to_file}
\synopsis{write_region_to_file}
\usage{Integer write_region_to_file (String filename);}
\description
  This function may be used to write a region of the current buffer to
  the file specified by \var{filename}.  It returns the number of lines
  written to the file or signals an error upon failure.
\seealso{write_buffer, append_region_to_file, push_mark}
\done

\function{count_chars}
\synopsis{count_chars}
\usage{String count_chars ();}
\description
  This function returns information about the size of the current buffer
  and current position in the buffer.  The string returned is of the form:
#v+
        'h'=104/0x68/0150, point 90876 of 127057
#v-
\seealso{what_char}
\done

\function{get_yes_no}
\synopsis{get_yes_no}
\usage{Integer get_yes_no (String s);}
\description
  This function may be used to get a yes or no response from the
  user.  The string parameter \var{s} will be used to construct the prompt
  by concating the string \exmp{"? (yes/no)"} to \var{s}.
  It returns \var{1} if the answer is yes or \exmp{0} if the answer is no.
\seealso{getkey, flush, message}
\done

\function{rename_file}
\synopsis{rename_file}
\usage{Integer rename_file (String old_name, String new_name);}
\description
  This function may be used to change the name of a disk file from
  \var{old_name} to \var{new_name}.  Upon success, zero is returned.  Any other
  value indicates failure.
  Note: Both filenames must refer to the same file system.
\seealso{file_status, stat_file}
\done

\function{change_default_dir}
\synopsis{change_default_dir}
\usage{Integer change_default_dir (String new_dir);}
\description
  This function may be used to change the current working directory
  of the editor to \var{new_dir}.  It returns zero upon success or \exmp{-1} upon
  failure.
  Note: Each buffer has its own working directory.  This function does not
  change the working directory of the buffer.  Rather, it changes the
  working directory of the whole editor.  This has an effect on functions
  such as \var{rename_file} when such functions are passed relative filenames.
\seealso{setbuf_info, getbuf_info, rename_file}
\done

\function{prefix_argument}
\synopsis{prefix_argument}
\usage{Integer prefix_argument (Integer dflt);}
\description
  This function may be used to determine whether or not the user has entered
  a prefix argument from the keyboard.  If a prefix argument is present,
  its value is returned; otherwise, \var{dflt} is returned.  Calling this
  function cancels the prefix argument.
  For example,
#v+
        variable arg = prefix_argument (-9999);
        if (arg == -9999)
          message ("No Prefix Argument");
        else
          message (Sprintf ("Prefix argument: %d", arg, 1));
#v-
  displays the prefix argument in the message area.
  Note: This function is incapable of distinguishing between the case of
  no prefix argument and when the argument's value is \var{dflt}.  Currently,
  this is not a problem because the editor does not allow negative prefix
  arguments.
\seealso{set_prefix_argument}
\done

\function{set_prefix_argument}
\synopsis{Set the prefix argument}
\usage{Void set_prefix_argument (Int_Type n)}
\description
  This function may be used to set the prefix argument to the value
  specified by \var{n}.  If \var{n} is less than zero, then the prefix
  argument is cancelled.
\seealso{prefix_argument}
\done

\function{set_buffer_hook}
\synopsis{set_buffer_hook}
\usage{Void set_buffer_hook (String hook, String f);}
\description
  Set current buffer hook \var{hook} to function \var{f}. \var{f} is a user
  defined S-Lang function.  Currently, \var{hook} can be any one of:
#v+
         "par_sep"  -- returns zero if the current line does not
              constitute the beginning or end of a paragraph.
              It returns non-zero otherwise.  The default value of hook is
              is_paragraph_separator.
         "indent_hook" -- returns nothing.  It is called by the indent line
              routines.
         "wrap_hook"   hook that is called after a line is wrapped.  Returns
              nothing.
         "newline_indent_hook"  --- returns nothing.  If this hook is defined,
              it will be called instead of the internal function
              newline_and_indent is called.
         "bob_eob_error_hook"  --- returns nothing.  If this hook is defined,
              it will be called whenever an error one of the internal cursor
              movement functions would have generated an end of buffer or beginning of
              buffer error.  It is passed an integer that indicates which function
              would have generated the error.  Specifically:
       
                       -1  previous_line_cmd
                       -2  previous_char_cmd
                       -3  page_up
                        1  next_line_cmd
                        2  next_char_cmd
                        3  page_down
         "mouse_down", "mouse_up", "mouse_drag" "mouse_2click" "mouse_3click"
              These hooks are used to override default hooks defined by the
              mouse_set_default_hook function.
#v-
\seealso{unset_buffer_hook, mouse_set_default_hook}
\done

\function{insert_file_region}
\synopsis{insert_file_region}
\usage{Integer insert_file_region (String file, String beg, String end);}
\description
  This function may be used to insert a region specified by the strings
  \var{beg} and \var{end} of the file with name \var{file} into the current buffer.
  The file is scanned line by line until a line that begins with the
  string given by \var{beg} is encountered.  Then, that line and all
  successive lines up to the one that starts with the string specified
  by \var{end} is inserted into the buffer.  The line beginning with the
  value of \var{end} is not inserted although the one beginning with \var{beg} is.
  The function returns the number of lines inserted or \exmp{-1} upon failure
  to open the file.

  Note that a zero length \var{beg} corresponds to the first line
  and that a zero length \var{end} corresponds to the last line.
\seealso{insert_file}
\done

\function{search_file}
\synopsis{search_file}
\usage{Integer search_file (String filename, String re, Integer nmax);}
\description
  This function may be used to search for strings in a disk file
  matching the regular expression \var{re}.  The first argument \var{filename}
  specifies which file to search.  The last argument \var{nmax} specifies
  how many matches to return.  Each line that is matched is pushed onto
  the S-Lang stack.  The number of matches (limited by \var{nmax}) is returned.
  If the file contains no matches, zero is returned.
\done

\function{random}
\synopsis{random}
\usage{Integer random (Integer seed, Integer nmax);}
\description
  The \var{random} function returns a random number in the range 0 to, but
  not including, \var{nmax}.  If the first parameter \var{seed} is 0, the
  number generated depends on a previous seed.  If \var{seed} is -1, the
  current time and process id will be used to seed the random number
  generator; otherwise \var{seed} will be used.

  Example: generate 1000 random integers in the range 0-500 and insert
  them into buffer:
#v+
        () = random (-1, 0);  % seed generator usingtime and pid
        loop (1000)
          insert (Sprintf ("%d\n", random (0, 500), 1));
#v-
  Note: The random number is generated via the expression:
#v+
        r = r * 69069UL + 1013904243UL;
#v-
\done

\function{set_term_vtxxx}
\synopsis{set_term_vtxxx}
\description
   Set terminal display appropriate for a vtxxx terminal.  This function
  takes a single integer parameter.  If non-zero, the terminal type is set
  for a vt100.  This means the terminal lacks the ability to insert/delete
  lines and characters.  If the parameter is zero, the terminal is assumed
  to be vt102 compatable.  Unless you are using a VERY old terminal or
  a primitive emulator, use zero as the parameter.
\done

\function{mouse_get_event_info}
\synopsis{mouse_get_event_info}
\usage{(x, y, state) = mouse_get_event_info ();}
\description
  This function returns the position of the last processed
  mouse event, and the state of the mouse buttons and shift
  keys before the event.

  \var{x} and \var{y} represent the column and row, respectively, where
  the event took place. They are measured with relative to the
  top left corner of the editor's display.

  \var{state} is a bitmapped integer whose bits are defined as follows:
#v+
         1  Left button pressed
         2  Middle button pressed
         4  Right button pressed
         8  Shift key pressed
        16  Ctrl key pressed
#v-
  Other information such as the button that triggered the event is
  available when the mouse handler is called.  As a result, this information
  is not returned by \var{mouse_get_event_info}.
\seealso{mouse_set_default_hook, set_buffer_hook.}
\done

\function{mouse_set_current_window}
\synopsis{mouse_set_current_window}
\usage{Void mouse_set_current_window ();}
\description
  Use of this function results in changing windows to the window that
  was current at the time of the mouse event.
\seealso{mouse_set_default_hook}
\done

\function{mouse_set_default_hook}
\synopsis{mouse_set_default_hook}
\usage{Void set_default_mouse_hook (String name, String fun);}
\description
  This function associates a slang function \var{fun} with the mouse event
  specified by \var{name}.  The first parameter \var{name} must be one of the
  following:
#v+
            "mouse_up"          "mouse_status_up"
            "mouse_down"        "mouse_status_down"
            "mouse_drag"        "mouse_status_drag"
            "mouse_2click"      "mouse_status_2click"
            "mouse_3click"      "mouse_status_3click"
#v-
  The meaning of these names should be obvious.  The second parameter,
  \var{fun} must be defined as
#v+
           define fun (line, column, btn, shift)
#v-
  and it must return an integer.  The parameters \var{line} and
  \var{column} correspond to the line and column numbers in the
  buffer where the event took place. \var{btn} is an integer that
  corresonds to the button triggering the event.  It can take
  on values \var{1}, \var{2}, and \var{4} corresponding to the left,
  middle, and right buttons, respectively.  \var{shift} can take on
  values \exmp{0}, \var{1}, or \var{2} where \exmp{0} indicates that no modifier
  key was pressed, \var{1} indicates that the SHIFT key was
  pressed, and \var{2} indicates that the CTRL key was pressed.
  For more detailed information about the modifier keys, use
  the function \var{mouse_get_event_info}.

  When the hook is called, the editor will automatically change
  to the window where the event occured.  The return value of
  the hook is used to dictate whether or not hook handled the
  event or whether the editor should switch back to the window
  prior to the event.  Specifically, the return value is interpreted
  as follows:

#v+
          -1     Event not handled, pass to default hook.
           0     Event handled, return active window prior to event
           1     Event handled, stay in current window.
#v-
\seealso{mouse_get_event_info, mouse_set_current_window, set_buffer_hook}
\done

\function{mouse_map_buttons}
\synopsis{mouse_map_buttons}
\usage{Void mouse_map_buttons (Integer x, Integer y);}
\description
  This function may be used to map one mouse button to another.  The
  button represented by \var{x} will appear as \var{y}.
\done

\function{vms_get_help}
\synopsis{vms_get_help}
\usage{Void vms_get_help (String hlp_file, String hlp_topic);}
\description
  This function may be used on VMS systems to interact with the VMS help
  system from within the editor.  \var{hlp_file} is the name of the help file
  to use and \var{hlp_topic} is the topic for which help is desired.
\done

\function{vms_send_mail}
\synopsis{vms_send_mail}
\usage{Integer vms_send_mail (String recip_lis, String subj);}
\description
  This VMS specific function provides an interface to the VMS callable
  mail facility.  The first argument, \var{recip_lis}, is a comma separated list
  of email addresses and \var{subj} is a string that represents the subject of
  the email.  The current buffer will be emailed.  It returns \var{1} upon
  success and \exmp{0} upon failure.
\done

\function{enable_flow_control}
\synopsis{enable_flow_control}
\usage{Void enable_flow_control (Integer flag);}
\description
  This Unix specific function may be used to turn XON/XOFF flow control
  on or off.  If \var{flag} is non-zero, flow control is turned on; otherwise,
  it is turned off.
\done

\function{core_dump}
\synopsis{core_dump}
\usage{Void core_dump(String msg, Integer severity);}
\description
  \var{core_dump} will exit the editor dumping the state of some crucial
  variables. If \var{severity} is \var{1}, a core dump will result.  Immediately
  before dumping, \var{msg} will be displayed.
\seealso{exit_jed, quit_jed, message, error}
\done

\function{get_last_macro}
\synopsis{get_last_macro}
\usage{String get_last_macro ();}
\description
  This function returns characters composing the last keyboard macro.  The
  charactors that make up the macro are encoded as themselves except the
  following characters:
#v+
        '\n'    ---->   \J
        null    ---->   \@
         \      ---->   \\
         '"'    ---->   \"
#v-
\done

\function{IsHPFSFileSystem}
\synopsis{IsHPFSFileSystem}
\usage{Integer IsHPFSFileSystem(String path);}
\description
  Returns non-zero if drive of \var{path} (possibly the default drive) is
  HPFS.
\done

\function{msdos_fixup_dirspec}
\synopsis{msdos_fixup_dirspec}
\usage{String msdos_fixup_dirspec (String dir);}
\description
  The motivation behind this is that DOS does not like a trailing
  backslash except if it is for the root dir.  This function makes
  \var{dir} conform to that.
\done

\function{set_top_status_line}
\synopsis{set_top_status_line}
\usage{String set_top_status_line (String str);}
\description
  This functions sets the string to be displayed at the top of the
  display. It returns the value of the line that was previously
  displayed.
\seealso{enable_top_status_line}
\done

\function{enable_top_status_line}
\synopsis{enable_top_status_line}
\usage{Void enable_top_status_line (Integer x);}
\description
  If x is non-zero, the top status line is enabled.  If x is zero, the
  top status line is disabled and hidden.
\seealso{set_top_status_line}
\done

\function{create_line_mark}
\synopsis{create_line_mark}
\usage{User_Mark create_line_mark (Integer c);}
\description
  The function \var{create_line_mark} returns an object of the type
  \var{User_Mark}.  This object contains information regarding the current
  position and current buffer.  The parameter \var{c} is used to specify the
  color to use when the line is displayed.
\seealso{create_user_mark, set_color_object}
\done

\function{create_user_mark}
\synopsis{create_user_mark}
\usage{User_Mark create_user_mark ();}
\description
  The function \var{create_user_mark} returns an object of the type
  \var{User_Mark}. This object contains information regarding the current
  position and current buffer.
\seealso{move_user_mark, goto_user_mark, user_mark_buffer}
\done

\function{goto_user_mark}
\synopsis{goto_user_mark}
\usage{Void goto_user_mark (User_Mark mark);}
\description
  This function returns to the position of the User Mark \var{mark}.  Before
  this function may be called, the current buffer must be the buffer
  associated with the makr.
\seealso{move_user_mark, create_user_mark, user_mark_buffer}
\done

\function{move_user_mark}
\synopsis{move_user_mark}
\usage{Void move_user_mark (User_Mark mark);}
\description
  This function call takes a previously created User Mark, \var{mark}, and
  moves it to the current position and buffer.  This means that if one
  subsequently calls \var{goto_user_mark} with this mark as an argument, the
  the position will be set to what it is prior to the call to
  \var{move_user_mark}.
  Note: This function call is not equivalent to simply using
#v+
        mark = create_user_mark ();
#v-
  because independent copies of a User Mark are not created uponn
  assignment.  That is, if one has
#v+
        variable mark1, mark2;
        setbuf ("first");
        mark1 = create_user_mark ();
        mark2 = mark1;
        setbuf ("second");
#v-
  and then calls
#v+
        move_user_mark (mark1);
#v-
  both user marks, \var{mark1} and \var{mark2} will be moved since they refer to
  the same mark.
\seealso{goto_user_mark, create_user_mark, user_mark_buffer}
\done

\function{user_mark_buffer}
\synopsis{user_mark_buffer}
\usage{String user_mark_buffer (User_Mark m);}
\description
  This function returns the name of the buffer associated with the
  User Mark specified by \var{m}.
\seealso{goto_user_mark, create_user_mark, move_user_mark, is_user_mark_in_narrow}
\done

\function{is_user_mark_in_narrow}
\synopsis{is_user_mark_in_narrow}
\usage{Integer is_user_mark_in_narrow (User_Mark m);}
\description
  This function returns non-zero if the user mark \var{m} refers to a
  position that is within the current narrow restriction of the current
  buffer.  It returns zero if the mark lies outside the restriction.
  An error will be generated if \var{m} does not represent a mark for the current
  buffer.
\seealso{goto_user_mark, move_user_mark}
\done

\function{list_abbrev_tables}
\synopsis{list_abbrev_tables}
\usage{Integer list_abbrev_tables ();}
\description
  This function returns the names of all currently defined
  abbreviation tables.  The top item on the stack will be the number of
  tables followed by the names of the tables.
\done

\function{use_abbrev_table}
\synopsis{use_abbrev_table}
\usage{Void use_abbrev_table (String table);}
\description
  Use the abbreviation table named \var{table} as the abbreviation table for
  the current buffer.  By default, the "Global" table is used.
\done

\function{create_abbrev_table}
\synopsis{create_abbrev_table}
\usage{Void create_abbrev_table (String name, String word);}
\description
  Create an abbreviation table with name \var{name}.  The second parameter
  \var{word} is the list of characters used to represent a word for the
  table. If the empty string is passed for \var{word}, the characters that
  currently constitute a word are used.
\done

\function{define_abbrev}
\synopsis{define_abbrev}
\usage{Void define_abbrev (String tbl, String abbrv, String expans);}
\description
  This function is used to define an abbreviation \var{abbrv} that will be
  expanded to \var{expans}.  The definition will be placed in the table with
  name \var{tbl}.
\done

\function{abbrev_table_p}
\synopsis{abbrev_table_p}
\usage{Integer abbrev_table_p (String name);}
\description
  Returns non-zero if an abbreviation table with called \var{name} exists. If
  the table does not exist, it returns zero.
\done

\function{dump_abbrev_table}
\synopsis{dump_abbrev_table}
\usage{Void dump_abbrev_table (String name);}
\description
  This function inserts the contents of the abbreviation table called
  \var{name} into the current buffer.
\done

\function{what_abbrev_table}
\synopsis{what_abbrev_table}
\usage{(String, String) what_abbrev_table ();}
\description
  This functions returns both the name of the abbreviation table and the
  definition of the word for the table currently associated with the
  current buffer.  If none is defined it returns two empty strings.
\done

\function{delete_abbrev_table}
\synopsis{delete_abbrev_table}
\usage{Void delete_abbrev_table (String name);}
\description
  Delete the abbrev table specified by \var{name}.
\done

\function{set_column_colors}
\synopsis{set_column_colors}
\usage{Void set_column_colors (Integer color, Integer c0, Integer c1);}
\description
  This function associates a color with columns \exmp{c0} through \var{c1} in the
  current buffer.  That is, if there is no syntax highlighting already
  defined for the current buffer, when the current buffer is displayed,
  columns \exmp{c0} through \var{c1} will be displayed with the attributes of the
  \var{color} object.  The parameters \exmp{c0} and \var{c1} are restricted to the range
  1 through \var{SCREEN_WIDTH}.  Use the function \var{set_color_object} to assign
  attributes to the \var{color} object.
\seealso{set_color_object}
\done

\function{set_color_object}
\synopsis{set_color_object}
\usage{Void set_color_object (Integer obj, String fg, String bg);}
\description
  Associate colors fg and bg with object obj.  Valid values for \var{obj}
  are in the range 30 to 128.  All other values are reserved.  Values for
  the strings \var{fg} and \var{bg} are as given by the description for \var{set_color}.
\seealso{set_column_colors, set_color}
\done

\function{translate_region}
\synopsis{translate_region}
\usage{Void translate_region ();}
\description
  This function uses the global character array \var{TRANSLATE_ARRAY} to
  modify the characters in a region based on the mapping defined by the
  array.  The only restriction is that the newline character cannot be
  mapped.  This example
#v+
        define swap_a_and_b ()
        {
          variable i;
          _for (0; 255, 1)
            {
               i = ();
               TRANSLATE_ARRAY[i] = i;
            }
          TRANSLATE_ARRAY['a'] = 'b';
          TRANSLATE_ARRAY['b'] = 'a';
          bob (); push_mark (); eob ();
          translate_region ();
        }
#v-
  uses \var{translate_region} to swap the \exmp{'a'} and \exmp{'b'} characters in the
  current buffer.
\seealso{insert, delete, what_char}
\seealso{TRANSLATE_ARRAY}
\done

\function{set_current_kbd_command}
\synopsis{set_current_kbd_command}
\usage{Void set_current_kbd_command (String s);}
\description
  Undocumented
\done

\function{find_matching_delimiter}
\synopsis{find_matching_delimiter}
\usage{Integer find_matching_delimiter (Integer ch);}
\description
  This function scans either forward or backward looking for the
  delimiter that matches the character specified by \var{ch}.  The actual
  direction depends upon the syntax of the character \var{ch}.  The
  matching delimiter pair must be declared as such by a prior call to
  \var{define_syntax}.  This function returns one of the following values:
#v+
         1    Match found
         0    Match not found
        -1    A match was attempted from within a string.
        -2    A match was attempted from within a comment
         2    No information
#v-
  In addition, the current point is left either at the match or is left
  at the place where the routine either detected a mismatch or gave up.
  In the case of a comment or a string (return values of -2 or -1), the
  current point is left at the beginning of a comment.
  Note: If the of \var{ch} is zero, the character at the current point will be
  used.
\seealso{blink_match, create_syntax_table, define_syntax, parse_to_point}
\done

\function{blink_match}
\synopsis{blink_match}
\usage{Void blink_match ();}
\description
  This function will attempt to blink the matching delimiter immediately
  before the editing point.
\seealso{find_matching_delimiter, define_syntax}
\done

\function{parse_to_point}
\synopsis{parse_to_point}
\usage{Integer parse_to_point ();}
\description
  This function attempts to determine the syntactic context of the
  current editing point.  That is, it tries to determine whether or not
  the current point is in a comment, a string, or elsewhere.
  It returns:
#v+
        -2   In a comment
        -1   In a string or a character
         0   Neither of the above
#v-
  Note: This routine is rather simplistic since it makes the assumption
  that the character at the beginning of the current line is not in a
  comment nor is in a string.
\seealso{define_syntax, find_matching_delimiter}
\done

\function{set_syntax_flags}
\synopsis{set_syntax_flags}
\usage{Void set_syntax_flags (String table, Integer flag);}
\description
  This function may be used to set the flags in the syntax table
  specified by the \var{table} parameter.  The \var{flag} parameter may take
  any of the following values or any combination bitwise or-ed together:
#v+
      0x01     Keywords are case insensitive
      0x02     Comments are Fortran-like
      0x04     Comments are C-like
      0x08     Keywords are TeX-like
      0x20     Syntax highlight whole preprocessor line in same color
#v-
  A Fortran-like comment means that any line that begins with certain
  specified characters is considered to be a comment.  This special
  subset of characters must be specified via a call to the
  \var{set_fortran_comment_chars} function.

  C-like comment means that any line that starts with \exmp{"* "} preceeded
  by any ammount of whitespace is to be highlighted as a comment.

  A TeX-like keyword is any word that follows the quote character.
\seealso{define_syntax, set_fortran_comment_chars}
\done

\function{set_fortran_comment_chars}
\synopsis{Specify characters for fortran-like comments}
\usage{Void set_fortran_comment_chars (String_Type table, String_Type list}
\description
  This function may be used to specify the set of characters that
  denote fortran style comments.  The first parameter \var{table} is
  the name of a previously defined syntax table, and \var{list}
  denotes the set of characters that specify the fortran-style
  comment.
  
  The string \var{list} is simply a set of characters and may include
  character ranges.  If the first character of \var{list} is
  \var{'^'}, then the meaning is that only those characters that do
  not specify fortran sytle comments are included in the list.
\example
  Fortran mode uses the following:
#v+
     set_fortran_comment_chars ("FORTRAN", "^0-9 \t\n");
#v-
  This means that if any line that begins with any character
  except the characters \exmp{0} to \exmp{9}, the space, tab, and
  newline characters will denote a comment.
\notes
  The usefulness of this function is not limited to fortran modes.  In
  fact, many languages have fortran-style comments.  

  This function is meaningful only if the syntax table has
  fortran-style comments as specified via the \var{set_syntax_flags}
  function.
\seealso{define_syntax, set_syntax_flags}
\done


\function{define_syntax}
\synopsis{define_syntax}
\usage{Void define_syntax (..., Integer type, String name);}
\description
  This function adds a syntax entry to the table specified by the last
  parameter \var{name}.  The actual number of parameters vary according to
  the next to the last parameter \var{type}.

  If \var{type} is \exmp{'"'} or \exmp{'\\''}, a string or character delimiter syntax is
  defined. In this case, \var{define_syntax} only takes three parameters
  where the first parameter is an integer that represents the character
  for which the syntax is to be applied.

  Similarly, if \var{type} is \exmp{'\\\\'}, then a quote syntax is defined and
  again \var{define_syntax} only takes three parameters where the first
  parameter is an integer that represents the character for which the
  syntax is to be applied.  A quote character is one in which the
  syntax of the following character is not treated as special.

  If \var{type} is \exmp{'('}, then \var{define_syntax} takes four parameters where
  the first two parameters are strings that represent a matching set of
  delimiters.  The first string contains the set of opening delimiters
  and the second string specifies the set of closing delimiters that
  match the first set.  If a character from the closing set is entered
  into the buffer, the corresponding delimiter from the opening set
  will be blinked.  For example, if the C language syntax table is
  called \exmp{"C"}, then one would use
#v+
        define_syntax ("([{", ")]}", '(', "C");
#v-
  to declare the matching delimiter set.  Note that the order of the
  characters in the two strings must correspond.  That is, the above
  example says that \exmp{'('} matches \exmp{')'} and so on.

  If \var{type} is \exmp{'%'}, a comment syntax is defined.  As in the
  previous case, \var{define_syntax} takes four parameters where there
  first two parameters are strings that represent the begin and end
  comment delimiters.  If the comment syntax is such that the comment
  ends at the end of a line, the second string must either be the empty
  string, \exmp{""}, or a newline \exmp{"\\n"}.  In the current implementation, at
  most the begin and end comment strings can consist of at most two
  characters.

  If \var{type} is \exmp{'+'}, the first parameter is a string whose characters
  are given the operator syntax.  If type is \exmp{','}, the first parameter
  is a string composed of characters that are condered to be
  delimiters.  If type is '0', the first parameter is a string composed
  of characters that make up a number.

  If \var{type} is \exmp{<}, the first parameter is a string whose successive
  characters form begin and end keyword highlight directives.

  Finally, if \var{type} is \exmp{'#'}, the first parameter is an integer whose
  value corresponds to the character used to begin preprocessor lines.

  As an example, imagine a language in which the dollar sign character
  \exmp{$} is used as a string delimiter, the backward quote character \exmp{`}
  is used as a quote character, comments begin with a semi-colon and
  end at the end of a line, and the characters \exmp{'<'} and \exmp{'>'} form
  matching delimiters.  The one might use
#v+
        create_syntax_table ("strange");
        define_syntax ('$',        '"',  "strange");
        define_syntax ('`',        '\\', "strange");
        define_syntax (";", "",    '%',  "strange");
        define_syntax ("<", ">",   '(',  "strange");
#v-
  to create a syntax table called \exmp{"strange"} and define the
  syntax entries for appropriate this example.
\seealso{create_syntax_table, use_syntax_table, find_matching_delimiter}
\seealso{BLINK}
\done

\function{use_syntax_table}
\synopsis{use_syntax_table}
\usage{Void use_syntax_table (String n);}
\description
  This function associates the current buffer with the syntax table
  specified by the name \var{n}.  Until another syntax table is associated
  with the buffer, the syntax table named \var{n} will be used in all
  operations that require a syntax.  This includes parenthesis matching,
  indentation, etc.
\seealso{create_syntax_table, define_syntax}
\done

\function{create_syntax_table}
\synopsis{create_syntax_table}
\usage{Void create_syntax_table (String name);}
\description
  This the purpose of this function is to create a new syntax table
  with the name specified by \var{name}.  If the table already exists, this
  function does nothing.
\seealso{define_syntax, use_syntax_table, define_keywords}
\done

\function{define_keywords_n}
\synopsis{define_keywords_n}
\usage{String define_keywords_n (String table, String kws, Integer len, Integer n);}
\description
  This function is used to define a set of keywords that will be color
  syntax highlighted in the keyword color associated with the table
  specified by \var{n}.  The first parameter, \var{table}, specifies which
  syntax table is to be used for the definition. The second parameter,
  \var{kws}, is a string that is the concatenation of keywords of length
  specified by the last parameter \var{len}.  The list of keywords specified
  by \var{kws} must be in alphabetic order.  The function returns the
  previous list of keywords of length \var{len}. For example, C mode uses
  the statement
#v+
        () = define_keywords_n ("C", "asmforintnewtry", 3, 0);
#v-
  to define the four three-letter keywords \var{asm}, \var{for}, \var{int}, \var{new},
  and \var{try}.  Note that in the above example, the return value is not used.
\seealso{define_syntax, set_color}
\seealso{WANT_SYNTAX_HIGHLIGHT,USE_ANSI_COLORS}
\done

\function{set_highlight_cache_dir}
\synopsis{set_highlight_cache_dir}
\usage{Void set_highlight_cache_dir (String dir);}
\description
  This function sets the directory where the dfa syntax highlighting
  cache files are located.
  See also: \var{enable_highlight_cache}
\done

\function{define_highlight_rule}
\synopsis{define_highlight_rule}
\usage{Void define_highlight_rule (String rule, String color, String n);}
\description
  This function adds an enhanced highlighting rule to the
  syntax table specified by the name \var{n}. The rule is described
  as a regular expression by the string \var{rule}, and the
  associated color is given by the string \var{color}, in the same
  format as is passed to \var{set_color}. For example:
#v+
        create_syntax_table ("demo");
        define_highlight_rule ("[A-Za-z][A-Za-z0-9]*", "keyword", "demo");
        define_highlight_rule ("//.*$", "comment", "demo");
        build_highlight_table ("demo");
#v-
  causes a syntax table to be defined in which any string of
  alphanumeric characters beginning with an alphabetic is
  highlighted in keyword color, and anything after "//" on a
  line is highlighted in comment color.

  The regular expression syntax understands character classes
  like \exmp{[a-z]} and \exmp{[^a-z0-9]}, parentheses, \exmp{+}, \exmp{*}, \exmp{?}, \exmp{|}
  and \exmp{.}. Any metacharacter can be escaped using a backslash
  so that it can be used as a normal character, but beware that
  due to the syntax of S-Lang strings the backslash has to be
  doubled when specified as a string constant. For example:
#v+
        define_highlight_rule ("^[ \t]*\\*+[ \t].*$", "comment", "C");
#v-
  defines any line beginning with optional whitespace, then one
  or more asterisks, then more whitespace to be a comment. Note
  the doubled backslash before the \exmp{*}.

  Note also that \var{build_highlight_table} must be called before
  the syntax highlighting can take effect.
\seealso{create_syntax_table, use_syntax_table, build_highlight_table}
\seealso{WANT_SYNTAX_HIGHLIGHT,USE_ANSI_COLORS}
\done

\function{build_highlight_table}
\synopsis{build_highlight_table}
\usage{Void build_highlight_table (String n);}
\description
  This function builds a DFA table for the enhanced syntax
  highlighting scheme specified for the syntax table specified
  by the name \var{n}. This must be called before any syntax
  highlighting will be done for that syntax table.
\seealso{create_syntax_table, use_syntax_table, define_highlight_rule, enable_highlight_cache}
\seealso{WANT_SYNTAX_HIGHLIGHT,USE_ANSI_COLORS}
\done

\function{enable_highlight_cache}
\synopsis{enable_highlight_cache}
\usage{Void enable_highlight_cache (String file, String n);}
\description
  This function enables caching of the DFA table for the
  enhanced syntax highlighting scheme belonging to the syntax
  table specified by the name \var{n}. This should be called before
  any calls to \var{define_highlight_rule} or to
  \var{build_highlight_table}. The parameter \var{file}
  specifies the name of the file (stored in the directory set by the
  \var{set_highlight_cache_dir} function) which should be used as a cache.

  For example, in \exmp{cmode.sl} one might write
#v+
        enable_highlight_cache ("cmode.dfa", "C");
#v-
  to enable caching of the DFA. If caching were not enabled for
  C mode, the DFA would take possibly a couple of seconds to
  compute every time Jed was started.

  Transferring cache files between different computers is
  theoretically possible but not recommended. Transferring them
  between different versions of Jed is not guaranteed to work.
\seealso{create_syntax_table, use_syntax_table, define_highlight_rule, build_highlight_table}
\seealso{WANT_SYNTAX_HIGHLIGHT,USE_ANSI_COLORS}
\done

\function{set_expansion_hook}
\synopsis{set_expansion_hook}
\usage{Void set_expansion_hook (String fname);}
\description
  This function may be used to specify a function that will be called to
  expand a filename upon TAB completion.  The function \var{fname} must
  already be defined.  When \var{fname} is called, it is given a string to
  be expanded. If it changes the string, it must return a non-zero value
  and the modified string.  If the string is not modified, it must simply
  return zero.
\done

\function{get_passwd_info}
\synopsis{get_passwd_info}
\usage{(dir, shell, pwd, uid, gid) = get_passwd_info (String username);}
\description
  This function returns password information about the user with name
  \var{username}.  The returned variables have the following meaning:
#v+
        dir:     login directory
        shell:   login shell
        pwd:     encripted password
        uid:     user identification number
        gid:     group identification number
#v-
  If the user does not exist, or the system call fails, the function
  returns with \var{uid} and \var{gid} set to \exmp{-1}.
\done

\function{get_termcap_string}
\synopsis{get_termcap_string}
\usage{String get_termcap_string (String cap);}
\description
  This function may be used to extract the string associated with the
  termcap capability associated with \var{cap}.
  Note: This function is only available on Unix systems.
\done

\function{get_doc_string}
\synopsis{get_doc_string}
\usage{Integer get_doc_string (String obj, String filename);}
\description
  This function may be used to extract the documentation for a variable
  or function from a jed documentation file given by \var{filename}.
  If successful, it returns non-zero as well as the documentation string.
  It returns zero upon failure.  The first character of \var{obj} determines
  whether \var{obj} refers to a function or to a variable.  The rest of the
  characters specify the name of the object.
\done

\function{kill_process}
\synopsis{kill_process}
\usage{Void kill_process (Integer id);}
\description
  Kill the subprocess specified by the process handle \var{id}.
\done

\function{send_process}
\synopsis{send_process}
\usage{Void send_process (Integer id, String s);}
\description
  Undocumented
\done

\function{open_process}
\synopsis{open_process}
\usage{Integer open_process (name, argv1, argv2, ..., argvN, N);}
\description
  Returns id of process, -1 upon failure.
\done

\function{process_mark}
\synopsis{process_mark}
\usage{User_Mark process_mark (Integer id);}
\description
  This function returns the user mark that contains the position of the
  last output by the process.
\done

\function{set_process}
\synopsis{set_process}
\usage{Void set_process (Integer pid, String what, String value);}
\description
  \var{pid} is the process hendle returned by \var{open_process}.  The second
  parameter, \var{what}, specifies what to set.  It must be one of the
  strings:
#v+
        "signal" :  indicates that 'value' is the name of a function to call
                    when the process status changed.  The function specified
                    by 'value' must be declared to accept an argument list:
                    (pid, flags) where 'pid' has the same meaning as above and
                    flags is an integer with the meanings:
                      1: Process Running
                      2: Process Stopped
                      4: Process Exited Normally
                      8: Process Exited via Signal
                    To obtain more information about the process, e.g., exit_status,
                    use the function 'get_process_status'.
                    Note: when this function is called, the current buffer is
                    guaranteed to be the buffer associated with the process.
       
        output" :   This parameter determines how output from the process is
                    is processed.  If the 'value' is the empty string "", output
                    will go to the end of the buffer associated with the process
                    and the point will be left there.
                    If value is ".", output will go at the current buffer position.
                    If value is "@", output will go to the end of the buffer but
                    the point will not move.  Otherwise, 'value' is the name of
                    a slang function with arguments: (pid, data) where pid has
                    the above meaning and data is the output from the process.
#v-
\done

\function{send_process_eof}
\synopsis{send_process_eof}
\usage{send_process_eof (Integer pid);}
\description
  This function closes the \var{stdin} of the process specified by the
  handle \var{pid}.
\done

\function{get_process_input}
\synopsis{get_process_input}
\usage{Void get_process_input (Integer tsecs);}
\description
  Read all pending input by all subprocesses.  If no input is
  available, this function will wait for input until \var{tsecs} tenth of
  seconds have expired.
\done

\function{signal_process}
\synopsis{signal_process}
\usage{Void signal_process (Integer pid, Integer signum);}
\description
  This function may be used to send a signal to the process whose
  process handle is given by \var{pid}.  The \var{pid} must be a valid handle
  that was returned by \var{open_process}.
\seealso{open_process, kill_process, send_process_eof}
\done

\function{clear_message}
\synopsis{clear_message}
\usage{Void clear_message ();}
\description
  This function may be used to clear the message line of the display.
\seealso{message, update, error, flush}
\done

\function{flush_input}
\synopsis{flush_input}
\usage{Void flush_input ();}
\description
  This function may be used to remove all forms of queued input.
\seealso{input_pending, getkey}
\done

\function{set_buffer_umask}
\synopsis{set_buffer_umask}
\usage{Integer set_buffer_umask (Integer cmask);}
\description
  The function may be used to set the process file creation mask
  for the appropriate operations associated with the current
  buffer.  This makes it possible to have a buffer-dependent
  umask setting. The function takes the desired umask setting and
  returns the previous setting.  If \var{cmask} is zero, the default
  process umask setting will be used for operations while the buffer
  is current.  If \var{cmask} is -1, the umask associated with the buffer
  will not be changed.
\done

\function{fsearch}
\synopsis{fsearch}
\usage{Integer fsearch (String str);}
\description
  This function may be used to search forward in buffer looking for the
  string \var{str}.  If not found, this functions returns zero.  However,
  if found, the length of the string is returned and the current point
  is moved to the to the start of the match.  It respects the setting
  of the variable \var{CASE_SEARCH}.  If the string that one is searching
  for is known to be at the beginning of a line, the function
  \var{bol_fsearch} should be used instead.

  Note: This function cannot find a match that crosses lines.
\seealso{ffind, fsearch_char, bsearch, bol_fsearch, re_fsearch, looking_at}
\seealso{CASE_SEARCH}
\done

\function{bsearch}
\synopsis{bsearch}
\usage{Integer bsearch (String str);}
\description
  The \var{bsearch} function searches backward from the current position
  for the string \var{str}.  If \var{str} is found, this function will return
  the length of \var{str} and move the current position to the beginning of
  the matched text.  If a match is not found, zero will be returned and
  the position will not change.  It respects the value of the variable
  \var{CASE_SEARCH}.
\seealso{fsearch, bol_bsearch, re_bsearch}
\done

\function{bfind}
\synopsis{bfind}
\usage{Integer bfind (String str);}
\description
  \var{bfind} searches backward from the current position to the beginning
  of the line for the string \var{str}.  If a match is found, the length of
  \var{str} is returned and the current point is moved to the start of the
  match. If no match is found, zero is returned.
  Note: This function respects the setting of the \var{CASE_SEARCH} variable.
\seealso{bsearch, ffind, bol_bsearch, re_bsearch}
\seealso{CASE_SEARCH}
\done

\function{ffind}
\synopsis{ffind}
\usage{Integer ffind (String s);}
\description
  \var{ffind} searches forward from the current position to the end of the
  line for the string \var{str}.  If a match is found, the length of \var{str}
  is returned and the current point is moved to the start of the match.
  If no match is found, zero is returned.
  Note: This function respects the setting of the \var{CASE_SEARCH} variable.
  To perform a search that includes multiple lines, use the \var{fsearch}
  function.
\seealso{fsearch, bfind, re_fsearch, bol_fsearch}
\done

\function{bol_fsearch}
\synopsis{bol_fsearch}
\usage{Integer bol_fsearch (str);}
\description
  \var{bol_fsearch} searches forward from the current point until the end
  of the buffer for occurrences of the string \var{str} at the beginning of
  a line.  If a match is found, the length of \var{str} is returned and the
  current point is moved to the start of the match.  If no match is
  found, zero is returned.
  Note: \var{bol_fsearch} is much faster than using \var{re_fsearch} to perform
  a search that matches the beginning of a line.
\seealso{bol_bsearch, fsearch, ffind, re_fsearch}
\seealso{CASE_SEARCH}
\done

\function{bol_bsearch}
\synopsis{bol_bsearch}
\usage{Integer bol_bsearch (str);}
\description
  \var{bol_bsearch} searches backward from the current point until the
  beginning of the buffer for the occurrences of the string \var{str} at
  the beginning of a line.  If a match is found, the length of \var{str} is
  returned and the current point is moved to the start of the match. If
  no match is found, zero is returned.

  Note: \var{bol_bsearch} is much faster than using \var{re_bsearch} to perform
  a search that matches the beginning of a line.
\seealso{bol_fsearch, bsearch, bfind, re_bsearch}
\seealso{CASE_SEARCH}
\done

\function{bol_fsearch_char}
\synopsis{bol_fsearch_char}
\usage{Integer bol_fsearch_char (Integer ch);}
\description
  This function searches forward for a character \var{ch} at the beginning
  of a line.  If it is found, \var{1} is returned; otherwise \exmp{0} is returned.
\seealso{bol_fsearch, bol_bsearch_char, fsearch_char}
\seealso{CASE_SEARCH}
\done

\function{bol_bsearch_char}
\synopsis{bol_bsearch_char}
\usage{Integer bol_fsearch_char (Integer ch);}
\description
  This function searches backward for a character \var{ch} at the beginning
  of a line.  If it is found, \var{1} is returned; otherwise \exmp{0} is returned.
\seealso{bol_bsearch, bol_fsearch_char, bsearch_char}
\seealso{CASE_SEARCH}
\done

\function{fsearch_char}
\synopsis{fsearch_char}
\usage{Integer fsearch_char (Integer ch);}
\description
  This function searches forward for a character \var{ch}.  If it is
  found, \var{1} is returned; otherwise \exmp{0} is returned.
\seealso{fsearch, ffind_char, bsearch_char}
\seealso{CASE_SEARCH}
\done

\function{bsearch_char}
\synopsis{bsearch_char}
\usage{Integer fsearch_char (Integer ch);}
\description
  This function searches backward for a character \var{ch}.  If it is
  found, \var{1} is returned; otherwise \exmp{0} is returned.
\seealso{fsearch_char, ffind_char, fsearch}
\seealso{CASE_SEARCH}
\done

\function{bfind_char}
\synopsis{bfind_char}
\usage{Integer fsearch_char (Integer ch);}
\description
  This function searches backward on the current line for a character
  \var{ch}.  If it is found, \var{1} is returned; otherwise \exmp{0} is returned.
\seealso{fsearch_char, ffind_char, fsearch}
\seealso{CASE_SEARCH}
\done

\function{ffind_char}
\synopsis{ffind_char}
\usage{Integer fsearch_char (Integer ch);}
\description
  This function searches forwardward on the current line for a character
  \var{ch}.  If it is found, \var{1} is returned; otherwise \exmp{0} is returned.
\seealso{fsearch_char, bfind_char, fsearch}
\seealso{CASE_SEARCH}
\done

\function{replace}
\synopsis{replace}
\usage{Void replace(String old, String new);}
\description
  This function may be used to replace all occurances of the string
  \var{old} with the string, \var{new}, from current editing point to the end
  of the buffer. The editing point is returned to the initial location.
  That is, this function does not move the editing point.
\seealso{replace_chars, fsearch, re_fsearch, bsearch, ffind, del}
\seealso{REPLACE_PRESERVE_CASE}
\done

\function{replace_chars}
\synopsis{replace_chars}
\usage{Void replace_chars (Integer n, String new);}
\description
  This function may be used to replace the next \var{n} characters at the
  editing position by the string \var{new}.  After the replacement, the editing
  point will be moved to the end of the inserted string.  The length of
  the replacement string \var{new} is returned.
\seealso{fsearch, re_fsearch, bsearch, ffind, del}
\seealso{REPLACE_PRESERVE_CASE}
\done

\function{regexp_nth_match}
\synopsis{regexp_nth_match}
\usage{String regexp_nth_match (Integer n);}
\description
  This function returns the nth sub-expression matched by the last regular
  expression search.  If the parameter \var{n} is zero, the entire match is
  returned.
  Note: The value returned by this function is meaningful only if the
  editing point has not been moved since the match.
\seealso{re_fsearch, re_bsearch}
\done

\function{replace_match}
\synopsis{replace_match}
\usage{Integer replace_match(String s, Integer how);}
\description
  This function replaces text previously matched with \var{re_fsearch} or
  \var{re_bsearch} at the current editing point with string \var{s}.  If \var{how} is
  zero, \var{s} is a specially formatted string of the form described below.
  If \var{how} is non-zero, \var{s} is regarded as a simple string and is used
  literally.  If the replacement fails, this function returns zero
  otherwise, it returns non-zero.
\done

\function{re_fsearch}
\synopsis{re_fsearch}
\usage{Integer re_fsearch(String pattern);}
\description
  Search forward for regular expression \var{pattern}.  This function returns
  the 1 + length of the string  matched.  If no match is found, it returns
  0.
\seealso{fsearch, bol_fsearch, re_bsearch}
\done

\function{re_bsearch}
\synopsis{re_bsearch}
\usage{Integer re_bsearch(String pattern);}
\description
  Search backward for regular expression \var{pattern}.  This function returns
  the 1 + length of the string  matched.  If no match is found, it returns
  0.
\seealso{bsearch, bol_bsearch, re_fsearch}
\done

\function{is_visible_mark}
\synopsis{is_visible_mark}
\usage{is_visible_mark ();}
\description
  This function may be used to test whether or not the mark is a visible
  mark.  A visible mar is one which causes the region defined by it to
  be highlighted.
  It returns \var{1} is the mark is visible, or \exmp{0} if the mark
  is not visible or does not exist.
\seealso{markp, push_mark}
\done

\function{push_narrow}
\synopsis{push_narrow}
\usage{Void push_narrow ();}
\description
  This function saves the current narrow context.  This is useful when
  one wants to restore this context after widening the buffer.
\seealso{pop_narrow, narrow, widen, widen_buffer}
\done

\function{pop_narrow}
\synopsis{pop_narrow}
\usage{Void pop_narrow ();}
\description
  The purpose of this function is to restore the last narrow
  context that was saved via \var{push_narrow}.
\seealso{push_narrow, widen, widen_buffer}
\done

\function{set_blocal_var}
\synopsis{set_blocal_var}
\usage{Void set_blocal_var (val, String v);}
\description
  This function sets the value of the buffer local variable with name \var{v}
  to value \var{val}.  The buffer local variable specified by \var{v} must have
  been previously created by the \var{create_blocal_var} function.  \var{val} must
  have the type that was declared when \var{create_blocal_var} was called.
\seealso{get_blocal_var, create_blocal_var}
\done

\function{get_blocal_var}
\synopsis{get_blocal_var}
\usage{get_blocal_var (String name);}
\description
  This function returns the value of the buffer local variable specified
  by \var{name}.
\seealso{set_blocal_var, create_blocal_var}
\done

\function{create_blocal_var}
\synopsis{create_blocal_var}
\usage{Void create_blocal_var (String name);}
\description
  This function is used to create a buffer local variable named
  \var{name}. A buffer local variable is a variable whose value is
  local to the current buffer.
\seealso{get_blocal_var, set_blocal_var}
\done

\function{count_narrows}
\synopsis{count_narrows}
\usage{Integer count_narrows ();}
\description
  This function returns the narrow depth of the current buffer.
\seealso{narrow, widen, widen_buffer, push_narrow}
\done

\function{widen_buffer}
\synopsis{widen_buffer}
\usage{Void widen_buffer ();}
\description
  This function widens the whole buffer.  If one intends to restore the
  narrow context after calling this function, the narrow context should be
  saved via \var{push_narrow}.
\seealso{narrow, widen, push_narrow, pop_narrow}
\done

\function{get_jed_library_path}
\synopsis{get_jed_library_path}
\usage{String get_jed_library_path ();}
\description
  This function returns the current search path for jed library files.
  The path may be set using the function \var{set_jed_library_path}.
\seealso{set_jed_library_path}
\done

\function{set_jed_library_path}
\synopsis{set_jed_library_path}
\usage{Void set_jed_library_path (String p);}
\description
  This function may be used to set the search path for library files.
  Its parameter \var{p} may be a comma separated list of directories to
  search.  When the editor is first started, the path is initialized
  from the \var{JED_ROOT}, or \var{JED_LIBRARY} environment variables.
\seealso{get_jed_library_path}
\done

\function{set_line_readonly}
\synopsis{set_line_readonly}
\usage{Void set_line_readonly (Integer flag);}
\description
  This function may be used to turn on or off the read-only state of the
  current line.  If the integer parameter \var{flag} is non-zero, the line
  will be made read-only.  If the paramter is zero, the read-only state
  will be turned off.
\seealso{getbuf_info}
\done

\function{set_line_hidden}
\synopsis{set_line_hidden}
\usage{Void set_line_hidden (Integer flag);}
\description
  If the parameter \var{flag} is non-zero, the current line will be given
  the hidden attribute.  This means that it will not be displayed.  If the
  parameter is zero, the hidden attribute will be turned off.
\seealso{set_region_hidden, is_line_hidden}
\done

\function{set_region_hidden}
\synopsis{set_region_hidden}
\usage{Void set_region_hidden (Integer flag);}
\description
  This function may be used to hide the lines in a region.  If \var{flag} is
  non-zero, all lines in the region will be hidden.  If it is zero, the
  lines in the region will be made visible.
\seealso{set_line_hidden, is_line_hidden, skip_hidden_lines_forward}
\done

\function{is_line_hidden}
\synopsis{is_line_hidden}
\usage{Integer is_line_hidden ();}
\description
  This function returns a non-zero value if the current line is hidden.  It
  will return zero if the current line is visible.
\seealso{set_line_hidden}
\done

\function{skip_hidden_lines_backward}
\synopsis{skip_hidden_lines_backward}
\usage{Void skip_hidden_lines_backward (Integer type);}
\description
  This function may be used to move backward across either hidden or non-hidden
  lines depending upon whether the parameter \var{type} is non-zero or zero.
  If \var{type} is non-zero, the Point is moved backward across hidden lines
  until a visible line is reached.  If \var{type} is zero, visible lines will
  be skipped instead.  If the top of the buffer is reached before the
  appropriate line is reached, the Point will be left there.

  Note: The functions \var{up} and \var{down} are insensitive to whether or not
  a line is hidden.
\seealso{skip_hidden_lines_forward, is_line_hidden}
\done

\function{skip_hidden_lines_forward}
\synopsis{skip_hidden_lines_forward}
\usage{Void skip_hidden_lines_forward (Integer type);}
\description
  This function may be used to move forward across either hidden or non-hidden
  lines depending upon whether the parameter \var{type} is non-zero or zero.
  If \var{type} is non-zero, the Point is moved forward across hidden lines
  until a visible line is reached.  If \var{type} is zero, visible lines will
  be skipped instead.  If the end of the buffer is reached before the
  appropriate line is reached, the Point will be left there.

  Note: The functions \var{up} and \var{down} are insensitive to whether or not
  a line is hidden.
\seealso{skip_hidden_lines_backward, is_line_hidden}
\done

\function{check_buffers}
\synopsis{Check to see if any buffers have been changed on the disk}
\usage{check_buffers ();}
\description
  The \var{check_buffers} function checks to see whether or not any of
  the disk files that are associated with the editor's buffers have been
  modified since the assocation was made.  The buffer flags are
  updated accordingly.
\seealso{file_time_compare, file_changed_on_disk}
\end{\documentstyle}
