;;; x-symbol-hooks.el --- pre-loaded stuff for package x-symbol

;; Copyright (C) 1996-1999 Free Software Foundation, Inc.
;;
;; Author: Christoph Wedler <wedler@fmi.uni-passau.de>
;; Maintainer: (Please use `M-x x-symbol-package-bug' to contact the maintainer)
;; Version: $Id: x-symbol-hooks.el,v 3.3 1999/01/18 14:15:38 wedler Exp wedler $
;; Keywords: WYSIWYG, LaTeX, HTML, wp, math, internationalization
;; X-URL: http://www.fmi.uni-passau.de/~wedler/x-symbol/

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; If you want to use package x-symbol, please visit the URL (use
;; \\[x-symbol-package-web]) and read the info (use \\[x-symbol-package-info]).

;; This file provides `autoload's for package x-symbol, adds functions to hooks
;; for the automatic conversion and defines variables which control the
;; conversion.

;;; Code:

(provide 'x-symbol-hooks)
(require 'font-lock)



;;;;##########################################################################
;;;;  Variables
;;;;##########################################################################


(defvar x-symbol-data-directory (locate-data-directory "x-symbol")
  "Directory of data files that come with package X-Symbol.")

(defvar x-symbol-font-directory
  (expand-file-name "pcf/" x-symbol-data-directory)
  "Directory for additional X-Symbol fonts.
If non-nil, used by `x-symbol-initialize'.")


;;;===========================================================================
;;;  Custom groups
;;;===========================================================================

(defgroup x-symbol nil
  "Semi WYSIWYG for LaTeX, HTML, etc using additional fonts."
  :group 'wp
  :link '(info-link "(x-symbol)")
  :link '(url-link "http://www.fmi.uni-passau.de/~wedler/x-symbol/")
  :prefix "x-symbol-")

(defgroup x-symbol-mode nil
  "Controlling whether and how to turn on X-Symbol mode."
  :group 'x-symbol
  :prefix "x-symbol-")

(defgroup x-symbol-input-init nil
  "Initialization of input methods supported by X-Symbol."
  :group 'x-symbol
  :prefix "x-symbol-")

(defgroup x-symbol-input-control nil
  "Control if input methods supported by X-Symbol."
  :group 'x-symbol
  :prefix "x-symbol-")

(defgroup x-symbol-info-general nil
  "General customization of X-Symbol info in echo area."
  :group 'x-symbol
  :prefix "x-symbol-")

(defgroup x-symbol-info-strings nil
  "Customization of X-Symbol info strings in echo area."
  :group 'x-symbol
  :prefix "x-symbol-")

(defgroup x-symbol-miscellaneous nil
  "Miscellaneous customization for X-Symbol."
  :group 'x-symbol
  :prefix "x-symbol-")

(defgroup x-symbol-image-general nil
  "General customization of images in X-Symbol buffers."
  :group 'x-symbol
  :prefix "x-symbol-")			; not "x-symbol-image-" !

(defgroup x-symbol-image-language nil
  "Language dependent customization of images in X-Symbol buffers."
  :group 'x-symbol
  :prefix "x-symbol-")			; not "x-symbol-image-" !

(defgroup x-symbol-tex nil
  "X-Symbol token language \"TeX macro\"."
  :group 'x-symbol
  :prefix "x-symbol-tex-")

(defgroup x-symbol-sgml nil
  "X-Symbol token language \"SGML entity\"."
  :group 'x-symbol
  :prefix "x-symbol-sgml-")

(defgroup x-symbol-utex nil
  "X-Symbol token language \"Unique TeX macro\"."
  :group 'x-symbol
  :prefix "x-symbol-utex-")


;;;===========================================================================
;;;  Custom widgets
;;;===========================================================================

;; Shouldn't this be a generally useful widget type?
(define-widget 'x-symbol-key 'sexp
  "A key or mouse stroke."
  :tag "Key/Mouse stroke")

(define-widget 'x-symbol-modes-or-regexp 'choice
  "Regexp or list of major modes."
  :value nil
  :args '((const :tag "Always" t)
	  (regexp :tag "When matched by")
	  (repeat :tag "In major modes" :menu-tag "In major modes"
		  (function :value text-mode))))

(define-widget 'x-symbol-auto-mode 'group
  "Auto-mode setup."
  :args '((x-symbol-modes-or-regexp :tag "Turn on")
	  (sexp :tag "Language (eval'd)")
	  (option
	   (group :inline t :extra-offset -4
		  (sexp :tag "Coding (eval'd)")
		  (option
		   (group :inline t :extra-offset -4
			  (sexp :tag "Save 8bit (eval'd)")
			  (option
			   (group :inline t :extra-offset -4
				  (sexp :tag "Super/subscripts (eval'd)")
				  (option
				   (sexp :tag "Show images (eval'd)"))))))))))

;;;===========================================================================
;;;  Functions to set user options
;;;===========================================================================

(defun x-symbol-define-user-options (var options &optional after-set set-fn)
  "Define options and setting behavior for user option VAR.
OPTIONS has the form (FALLBACK . ALIST) where ALIST has elements of the
form (OPTION . MENU-TEXT).  If ALIST is non-nil, one OPTION should be
equal to FALLBACK, its MENU-TEXT is used for any values not being keys
in ALIST.  OPTIONS can also be a function which should return the form
mention above.

If ALIST is nil, `x-symbol-submenu-filter', which is used by
`x-symbol-menu', uses a toggle button with FALLBACK as the non-nil value
and \\[set-variable] offers completion with matches nil and FALLBACK.
Otherwise, the menu uses radio buttons for all OPTIONs, where MENU-TEXT
is the name of the menu item, and \\[set-variable] offers completion
over all OPTIONs.

`x-symbol-set-variable', which is invoked by the menu callbacks, uses
SET-FN instead `set' to set the value of VAR if SET-FN is non-nil.
Otherwise, all functions in AFTER-SET are invoked after `set'ting VAR."
  (put var 'x-symbol-options options)
  (put var 'variable-interactive
       (list 'x-symbol-variable-interactive (list 'quote var)))
  (while after-set
    (pushnew (pop after-set) (get var 'x-symbol-after-set-hook) :test 'equal))
  (if set-fn (put var 'x-symbol-set-function set-fn)))

;;;###autoload
(defun x-symbol-auto-mode-suffixes (&optional suffixes)
  "Return REGEXPs of three-value elements in `auto-mode-alist'.
These REGEXPs are added to SUFFIXES."
  (setq suffixes (reverse suffixes))
  (let ((alist auto-mode-alist))
    (while alist
      (and (consp (cdar alist))
	   (null (member (caar alist) suffixes))
	   (push (caar alist) suffixes))
      (setq alist (cdr alist)))
    (nreverse suffixes)))


;;;===========================================================================
;;;  Initialization
;;;===========================================================================

(defcustom x-symbol-initialize t
  "Whether to do an extended initialization of package X-Symbol.
If t, do full initialization.  Otherwise, the value should be a list
with element.  To enable, include

 * `language' to register all supported token languages,
 * `global' to turn on X-Symbol's global mode, i.e., as files are
   loaded, execute `turn-on-x-symbol-conditionally',
 * `keys' to set up the usual X-Symbol key bindings in `global-map',
 * `font-path' to add `x-symbol-font-directory' to the font-path,
 * `comint' to make X-Symbol work with comint,
 * `fast-lock' to make X-Symbol work with fast-lock,
 * `auctex' to make X-Symbol optimally work with AucTeX 9.8a+,
 * `reftex' to make X-Symbol optimally work with RefTeX 3.26+,
 * `bib-cite' to make X-Symbol not overwriting bib-cite's highlighting.

You do not have to install the packages whose initialization is
enabled."
  :group 'x-symbol-mode
  :type '(choice (const :tag "All" t)
		 (set :value (languages global keys font-path comint
					fast-lock auctex reftex bib-cite)
		      (const :tag "Token languages" languages)
		      (const :tag "Global mode" global)
		      (const :tag "Key bindings" keys)
		      (const :tag "Font path" font-path)
		      (const :tag "Package comint" comint)
		      (const :tag "Package fast-lock" fast-lock)
		      (const :tag "Package AucTeX" auctex)
		      (const :tag "Package RefTeX" reftex)
		      (const :tag "Package bib-cite" bib-cite))))

(defvar x-symbol-orig-compint-input-sender 'comint-simple-send
  "Original function which sends a string to the comint process.")


;;;===========================================================================
;;;  General Configuration
;;;===========================================================================

(defvar x-symbol-after-init-input-hook nil
  "Hook run after the initialization of all input methods.
See `x-symbol-init-input'.  If you want define key bindings starting
with \\[x-symbol-map], do it here.  It is a bad idea to use this for
additional self insert commands, use \"character descriptions\" in
`x-symbol-user-table' instead.")

(add-hook 'x-symbol-after-init-input-hook 'x-symbol-init-default-keys)

(defcustom x-symbol-compose-key '(control ?\=)
  "Key used to access command `x-symbol-map'.
By default, pressing this key twice invokes the GRID: \\[x-symbol-grid].
This is a list, no vector!"
  :group 'x-symbol-input-init
  :type '(x-symbol-key :tag "Prefix key"))


(defcustom x-symbol-auto-key-autoload t
  "*If non-nil, pressing `x-symbol-compose-key' initialize x-symbol.
The binding of `x-symbol-compose-key' is redefined after initialization.
With value nil, you must provide a prefix argument to initialize package
X-Symbol."
  :group 'x-symbol-input-init
  :type 'boolean)

(defvar x-symbol-auto-conversion-method 'auto-slow
  ;;(if (featurep 'crypt) 'slow 'fast)
  "Non-nil means, set up hooks for auto conversion.
Fast methods are used if this variable has value `fast'.  Otherwise, slower
methods are used and \\[vc-register] or \\[vc-next-action] will fail to decode
the buffer contents.

You should set this variable to value `slowest' if, for example, the
symbol for \\alpha looks like \\233a after \\[save-buffer] (this happens
on some systems).  Value `fast' should not be used, if some other
package, e.g., crypt, adds a function to `write-file-hooks' which does
not inspect the remaining functions in this hook.

Default value `auto-slow' is set to `fast' after the initialization of
XEmacs if package crypt has not been loaded by then.")

(defvar x-symbol-default-coding 'iso-8859-1
  "Coding used for 8bit characters in buffers.
Also used for a 8bit file codings where `x-symbol-coding' has value nil.
Supported values are `iso-8859-1', `iso-8859-2', `iso-8859-3' and
`iso-8859-9', it should correspond to the normal charset registry of
your `default' face.

WARNING: Under XEmacs/Mule, package x-symbol is only tested with value
`iso-8859-1'!  It is assumed that you have not changed any of the
various Mule codings-system variables, i.e., it assumes iso-8859-1.  Use
\\[x-symbol-package-bug] to give the author some advice on Mule.")

(defcustom x-symbol-safe-exec-threshold 8192
  "*Use executables for conversion if region/buffer is this large.
This value is used for decoding and encoding in buffers where markers
are not important, e.g., in temporary buffers, or buffers just created,
see argument INIT of function `x-symbol-mode'.  Value nil means value of
`x-symbol-exec-threshold'."
  :group 'x-symbol-miscellaneous
  :type '(choice (const :tag "Never" nil) (integer :tag "If larger than")))


;;;===========================================================================
;;;  Known Token Languages
;;;===========================================================================

(defvar x-symbol-language-alist nil
  "Alist of currently registered token languages.
Elements look like (LANGUAGE . NAME) where LANGUAGE is the symbol
representing and NAME is the name normally presented to the user,
see `x-symbol-language-text'.

You should not set this variable directly, use
`x-symbol-register-language' instead!")

(defcustom x-symbol-charsym-name "x-symbol charsym"
  "Name of the pseudo token language x-symbol charsym.
This pseudo language corresponds to `x-symbol-language' having value nil
and is used for input methods, not for decoding and encoding.  See
`x-symbol-language-text'."
  :group 'x-symbol-miscellaneous
  :type 'string)

(defcustom x-symbol-tex-name "TeX macro"
  "Name of token language `tex'.  See `x-symbol-register-language'."
  :group 'x-symbol-tex
  :type 'string)

(defcustom x-symbol-tex-modes
  '(tex-mode latex-mode plain-tex-mode bibtex-mode noweb-mode)
  "Major modes using language `tex'.  See `x-symbol-register-language'."
  :group 'x-symbol-tex
  :group 'x-symbol-mode
  :type '(repeat function))

(defcustom x-symbol-sgml-name "SGML entity"
  "Name of token language `sgml'.  See `x-symbol-register-language'."
  :group 'x-symbol-sgml
  :type 'string)

(defcustom x-symbol-sgml-modes '(sgml-mode html-mode)
  "Major modes using language `sgml'.  See `x-symbol-register-language'."
  :group 'x-symbol-sgml
  :group 'x-symbol-mode
  :type '(repeat function))

(defcustom x-symbol-utex-name "Unique TeX macro"
  "Name of token language `utex'.  See `x-symbol-register-language'."
  :group 'x-symbol-utex
  :type 'string)

(defcustom x-symbol-utex-modes nil
  "Major modes using language `utex'.  See `x-symbol-register-language'."
  :group 'x-symbol-utex
  :group 'x-symbol-mode
  :type '(repeat function))


;;;===========================================================================
;;;  Buffer-locals
;;;===========================================================================

(defvar x-symbol-mode nil
  "Non-nil if X-Symbol minor mode is enabled.")

(make-variable-buffer-local 'x-symbol-mode)
(x-symbol-define-user-options 'x-symbol-mode '(t)
  nil (lambda (dummy arg) (x-symbol-mode (if arg 1 0))))

(defvar x-symbol-language nil
  "*Token language used in current buffer.
A valid value is required to turn on `x-symbol-mode' which also sets
this variable to a reasonable value if the variable is not yet
buffer-local.  The value influences the conversion, i.e., decoding and
encoding of X-Symbol characters, input methods TOKEN and READ-TOKEN,
fontification of super- and subscripts, image command recognition, the
info in the echo area, etc.")

(make-variable-buffer-local 'x-symbol-language)
(put 'x-symbol-language 'permanent-local t)
(x-symbol-define-user-options 'x-symbol-language
    (lambda () (list* nil '(nil . "None") x-symbol-language-alist))
  '(x-symbol-update-modeline))

(defvar x-symbol-8bits nil
  "*If non-nil, do not encode 8bit characters.
Variable `x-symbol-coding' determines which characters are assumed to be
8bit characters.  Note that tokens representing 8bit characters are
always decoded.  Function `x-symbol-mode' sets this variable to a
reasonable value if the variable is not yet buffer-local.")

(make-variable-buffer-local 'x-symbol-8bits)
(put 'x-symbol-8bits 'permanent-local t)
(x-symbol-define-user-options 'x-symbol-8bits '(t)
  '(x-symbol-update-modeline))

(defvar x-symbol-coding nil
  "*Coding of 8bit characters in a file.
Supported values are `iso-8859-1', `iso-8859-2', `iso-8859-3' and
`iso-8859-9', value nil means the value of `x-symbol-default-coding'.
Determines which characters are considered to be 8bit characters for
file operations.  Function `x-symbol-mode' sets this variable to a
reasonable value if the variable is not yet buffer-local.

During decoding, e.g., when visiting a file, the value is always
important for the interpretation of 8bit characters, an invalid value is
considered to be equivalent to value nil.  During encoding, e.g., when
saving a buffer, 8bit characters are not encoded to tokens if the value
is valid and `x-symbol-8bits' is non-nil.")

(make-variable-buffer-local 'x-symbol-coding)
(put 'x-symbol-coding 'permanent-local t)
(x-symbol-define-user-options 'x-symbol-coding
    (lambda () (cons x-symbol-default-coding x-symbol-coding-name-alist))
  '(x-symbol-update-modeline))

(defvar x-symbol-subscripts nil
  "*If non-nil, use special fonts to display super- and subscripts.
This feature must be supported by the token language dependent font-lock
keywords.  Function `x-symbol-mode' sets this variable to a reasonable
value if the variable is not yet buffer-local.  Some parts of the text
might be invisible, see also variable `x-symbol-reveal-invisible'.")

(make-variable-buffer-local 'x-symbol-subscripts)
(x-symbol-define-user-options 'x-symbol-subscripts '(t)
  '(x-symbol-update-modeline x-symbol-fontify))

(defvar x-symbol-image nil
  "*If non-nil, show little glyphs after image insertion commands.
This feature must be supported by the token language dependent image
keywords, see `x-symbol-image-parse-buffer'.  Function `x-symbol-mode'
sets this variable to a reasonable value if the variable is not yet
buffer-local.")

(make-variable-buffer-local 'x-symbol-image)
(x-symbol-define-user-options 'x-symbol-image '(t)
  '(x-symbol-update-modeline) 'x-symbol-set-image)


;;;===========================================================================
;;;  Minor mode control
;;;===========================================================================

(defcustom x-symbol-buffer-mode-alist nil
  "*Alist to setup x-symbol values for specific buffers.
Elements look like
   (BUFFER-NAME MODE-ON LANGUAGE CODING 8BIT SUBSCRIPTS IMAGE)
If the name of the buffer in which command `x-symbol-mode' is invoked is
equal to BUFFER, the rest of the element is used to setup some
buffer-local x-symbol specific variables, see `x-symbol-auto-mode-alist'
for details.  MODE-ON is used directly as the value."
  :group 'x-symbol-mode
  :type '(repeat (group (string :tag "Use for buffer")
			(x-symbol-auto-mode :inline t))))

(defcustom x-symbol-auto-mode-suffixes (x-symbol-auto-mode-suffixes)
  "*Regexps matching file suffixes not to be considered.
All suffixes from a file name matching these regexps are deleted before
the file name is used for `x-symbol-auto-mode-alist'.  The default value
includes the REGEXP in all three-valued elements of `auto-mode-alist',
at definition time, of course."
  :group 'x-symbol-mode
  :type '(repeat regexp))

(defcustom x-symbol-auto-mode-alist
  '(((tex-mode latex-mode plain-tex-mode) "\\.tex\\'" 'tex
     (x-symbol-auto-coding-alist x-symbol-tex-auto-coding-alist 10000)
     x-symbol-coding
     t t)
    ((bibtex-mode) t 'tex)
    ((sgml-mode html-mode) (html-mode) 'sgml
     (x-symbol-auto-coding-alist x-symbol-sgml-auto-coding-alist 10000)
     x-symbol-coding
     t t))
  "*Alist to setup X-Symbol values for buffers visiting files.
Elements look like
  (MATCH MODE-ON LANGUAGE CODING 8BIT SUBSCRIPTS IMAGE)
If MATCH matches a buffer in which command `x-symbol-mode' is invoked,
the rest of the element is used to setup some buffer-local x-symbol
specific variables.  If no element matches, set `x-symbol-language' to
the symbol property `x-symbol-language' of the major mode symbol if the
variable is not already buffer-local.

If `x-symbol-mode' is not already buffer-local, MODE-ON determines
whether to turn the mode on if `x-symbol-mode' is called with a cons as
prefix argument.  LANGUAGE, CODING, 8BIT, SUBSCRIPTS and IMAGE are used
to set `x-symbol-language', `x-symbol-coding', `x-symbol-8bits',
`x-symbol-subscripts' and `x-symbol-image' in that order if these values
are not already buffer-local.

MATCH and MODE-ON are either a list of major modes which must include
the mode of the current buffer or a regexp matching the file name
ignoring some suffixes, see `x-symbol-auto-mode-suffixes', or a value
used directly.  LANGUAGE, CODING, 8BIT, SUBSCRIPTS and IMAGE are
`eval'ed."
  :group 'x-symbol-mode
  :type '(repeat (group (x-symbol-modes-or-regexp :tag "Use")
			(x-symbol-auto-mode :inline t))))


;;;===========================================================================
;;;  Images
;;;===========================================================================

(defun x-symbol-image-set-colormap (var value)
  "Set VAR's value to NEWVAL.
Custom set function of `x-symbol-image-colormap-allocation' and
`x-symbol-image-convert-colormap'."
  (if var (set var value))
  (if (boundp 'x-symbol-image-convert-colormap)
      (put 'x-symbol-image-convert-colormap 'x-symbol-image-instance
	   (and (boundp 'x-symbol-image-colormap-allocation)
		x-symbol-image-colormap-allocation
		x-symbol-image-convert-colormap
		(make-image-instance (vector x-symbol-image-colormap-allocation
					     :file
					     x-symbol-image-convert-colormap)
				     nil nil t)))))

(defcustom x-symbol-image-colormap-allocation 'xpm
  "If non-nil, prevent colors in colormap to be de-allocated.
The non-nil value should be an image format.  See
`x-symbol-image-convert-colormap'."
  :group 'x-symbol-image-general
  :initialize 'custom-initialize-default
  :set 'x-symbol-image-set-colormap
  :type '(choice (const :tag "Colors can be de-allocated" nil)
		 (const :tag "Colormap is xpm file" xpm)
		 (symbol :tag "Other image format")))

(defcustom x-symbol-image-convert-colormap
  (and x-symbol-data-directory
       (expand-file-name "colormap138.xpm" x-symbol-data-directory))
  "File name of colormap files.
Used by `x-symbol-image-start-convert-colormap' for image cache file
names not matched by `x-symbol-image-convert-mono-regexp'.  See also
`x-symbol-image-colormap-allocation'."
  :group 'x-symbol-image-general
  :initialize 'custom-initialize-default
  :set 'x-symbol-image-set-colormap
  :type '(choice (const :tag "No map" nil) file))



;;;;##########################################################################
;;;;  Code
;;;;##########################################################################


(cond ((not (string-match "XEmacs" emacs-version))
       (error "Package X-Symbol can only be used with XEmacs-20.3+"))
      ((not (and (fboundp 'emacs-version>=) (emacs-version>= 20 3)))
       ;; Yes, it probably works with XEmacs-20.2, too, but I do not
       ;; want to care about its bug in `string-match':
       (error "Package X-Symbol can only be used with XEmacs-20.3+"))
      ((emacs-version>= 21)
       (load "x-symbol-xmas21"))
      (t
       (load "x-symbol-xmas20")))

(defalias 'x-symbol-cset-registry 'caaar)
(defalias 'x-symbol-cset-coding 'cdaar)
(defalias 'x-symbol-cset-leading 'cadar)
(defalias 'x-symbol-cset-score 'caddar)
(defalias 'x-symbol-cset-left 'cadr)
(defalias 'x-symbol-cset-right 'cddr)

(defvar x-symbol-input-initialized nil
  "Internal.  If non-nil, the input methods are initialized.")


;;;===========================================================================
;;;  Key autoload
;;;===========================================================================

;;;###autoload
(defun x-symbol-key-autoload (&optional arg)
  "Initialize package x-symbol and use the keys for this command again.
Package x-symbol and the functions in `x-symbol-load-hook' should
re-bind all key-sequence which invoke this command.  You should provide
a prefix argument ARG to this command is `x-symbol-auto-key-autoload' is
nil."
  (interactive "P")
  (when x-symbol-input-initialized
    (error "%s should be rebound in `x-symbol-init-input-hook'"
	   (key-description (this-command-keys))))
  (unless (or arg x-symbol-auto-key-autoload)
    (error "Use %s with prefix argument to initialize the input methods"
	   (key-description (this-command-keys))))
  (let ((this (append (this-command-keys) nil)))
    ;; for some reason this loop is necessary...
    (while (and this (null (eq (key-binding (vector (car this))) this-command)))
      (setq this (cdr this)))
    (setq prefix-arg arg)
    (setq unread-command-events this))
  (x-symbol-init-input))

;;;###autoload
(define-function 'x-symbol-map-autoload 'x-symbol-key-autoload)


;;;===========================================================================
;;;  Minor mode, fontification
;;;===========================================================================

(defun x-symbol-auto-mode-alist ()
  "Compute values to setup X-Symbol variables."
  (let ((name (file-name-sans-versions buffer-file-name))
	(case-fold-search (eq system-type 'vax-vms))
	(suffixes x-symbol-auto-mode-suffixes)
	(alist x-symbol-auto-mode-alist)
	matcher result)
    (while suffixes
      (if (string-match (pop suffixes) name)
	  (setq name (substring name 0 (match-beginning 0)))))
    (while alist
      (if (if (consp (setq matcher (caar alist)))
	      (memq major-mode matcher)
	    (if (stringp matcher) (string-match matcher name) matcher))
	  (setq result (car alist)
		alist nil)
	(setq alist (cdr alist))))
    (if result
	(cons (if (consp (setq matcher (cadr result)))
		  (memq major-mode matcher)
		(if (stringp matcher) (string-match matcher name) matcher))
	      (cddr result))
      (and (setq result (get major-mode 'x-symbol-language))
	   (list nil (list 'quote result))))))

(defun x-symbol-auto-set-variable (symbol form)
  "Set SYMBOL's value to evaluated FORM if SYMBOL is not buffer-local."
  (or (local-variable-p symbol (current-buffer))
      (set symbol (eval form))))

;;;###autoload
(defun x-symbol-mode (&optional arg init)
  "Toggle X-Symbol mode.
If ARG is a cons, e.g., when \\[x-symbol-mode] is preceded by one or
more \\[universal-argument]'s with no digits, turn on X-Symbol mode
conditionally, see MODE-ON in `x-symbol-auto-mode-alist'.  Otherwise,
turn X-Symbol mode on if ARG is positive, else turn it off.  If some
X-Symbol specific local variables are not buffer-local, set them to
reasonable values according to `x-symbol-buffer-mode-alist' and
`x-symbol-auto-mode-alist'.

Turning X-Symbol mode on also decodes tokens if the mode was turned off
before, see \\[x-symbol-decode], it is assumed that markers in the
buffer are not important when the optional argument INIT is non-nil.
Turning X-Symbol mode off also encodes x-symbol characters if the mode
was turned on before, see \\[x-symbol-encode].  If argument INIT is
non-nil, the old mode status is assumed to be off."
  (interactive "P")
  (let ((old-mode x-symbol-mode)
	(values (or (cdr (assoc (buffer-name) x-symbol-buffer-mode-alist))
		    (and buffer-file-name
			 (x-symbol-auto-mode-alist)))))
    (when values
      (x-symbol-auto-set-variable 'x-symbol-language (cadr values))
      (x-symbol-auto-set-variable 'x-symbol-coding (caddr values))
      (x-symbol-auto-set-variable 'x-symbol-8bits (cadddr values))
      (x-symbol-auto-set-variable 'x-symbol-subscripts (nth 4 values))
      (x-symbol-auto-set-variable 'x-symbol-image (nth 5 values)))
    (or (null x-symbol-language)	; e.g., not buffer-local
	(and (symbolp x-symbol-language)
	     (get x-symbol-language 'x-symbol-feature))
	(setq x-symbol-language nil))
    (setq x-symbol-mode
	  (and x-symbol-language
	       (if arg
		   (if (consp arg)
		       (if (local-variable-p 'x-symbol-mode (current-buffer))
			   (and x-symbol-mode t)
			 (car values))
		     (> (prefix-numeric-value arg) 0))
		 (not x-symbol-mode))))
    (if init
	(if x-symbol-mode
	    (x-symbol-mode-internal t x-symbol-safe-exec-threshold))
      (x-symbol-mode-internal (and x-symbol-language
				   (eq (null old-mode) x-symbol-mode))))))

;;;###autoload
(defun turn-on-x-symbol-conditionally ()
  "Turn on x-symbol mode conditionally, see `x-symbol-mode'.
Call `x-symbol-mode' with a cons for ARG and a non-nil INIT.  Used in
`hack-local-variables-hook'."
  (x-symbol-mode '(1) t))

;;;###autoload
(defun x-symbol-fontify (&optional beg end)
  "Re-fontify region between BEG and END."
  (interactive)
  (cond ((not font-lock-mode) (turn-on-font-lock))
	((and (boundp 'lazy-shot-mode) lazy-shot-mode)
	 ;; copied from lazy-shot:
	 (setq font-lock-fontified
	       (and lazy-shot-minimum-size
		    (>= (buffer-size) lazy-shot-minimum-size)))
	 (if (fboundp 'lazy-shot-after-change-function)
	     ;; No, I'm not using `compiled-function-arglist' instead...
	     (lazy-shot-install-extents (point-min) (point-max)	; XEmacs-20.4
					font-lock-fontified)
	   (lazy-shot-install-extents font-lock-fontified))) ; XEmacs-20.3
	((and (boundp 'lazy-lock-mode) lazy-lock-mode)
	 ;; copied from lazy-lock:
	 (let ((modified (buffer-modified-p)) (inhibit-read-only t)
	       (buffer-undo-list t)
	       ;;deactivate-mark
	       buffer-file-name buffer-file-truename)
	   (remove-text-properties (or beg 1) (or end (1+ (buffer-size)))
				   '(fontified nil))
	   (or modified (set-buffer-modified-p nil))))
	(t
	 (font-lock-fontify-buffer))))


;;;===========================================================================
;;;  comint support
;;;===========================================================================

(defun x-symbol-comint-output-filter (dummy)
  "Decode output of comint's process.
Used as value in `comint-output-filter-functions'."
  (and x-symbol-mode x-symbol-language
       (save-excursion
	 (x-symbol-decode-region (if (interactive-p)
				     comint-last-input-end
				   comint-last-output-start)
				 (process-mark (get-buffer-process
						(current-buffer)))))))

(defun x-symbol-comint-send (proc string)
  "Encode STRING and send it to process PROC.
Used as value of `comint-input-sender', uses
`x-symbol-orig-compint-input-sender'."
  (and x-symbol-mode x-symbol-language
       (setq string
	     (save-excursion
	       (let ((language x-symbol-language)
		     (coding x-symbol-coding)
		     (selective selective-display))
		 (set-buffer (get-buffer-create " x-symbol comint"))
		 (erase-buffer)
		 (insert string)
		 (setq x-symbol-language language)
		 (x-symbol-encode-all nil coding)
		 (setq selective-display selective))
	       (prog1 (buffer-substring)
		 (kill-buffer (current-buffer))))))
  (funcall x-symbol-orig-compint-input-sender proc string))


;;;===========================================================================
;;;  Hooks for automatic conversion
;;;===========================================================================

(defun x-symbol-after-insert-file (len)
  ;; checkdoc-params: (len)
  "Decode tokens, e.g., after \\[vc-register] or \\[vc-next-action].
Added to `after-insert-file-functions' if
`x-symbol-auto-conversion-method' has value 'fast'."
  ;; Arg!  There is no way to know the start position of the region.  If
  ;; `insert-file-contents' is called with argument REPLACE being non-nil, it
  ;; is not always point.  Thus, we use `point-min', except when called from
  ;; `M-x insert-file'.
  (and x-symbol-mode x-symbol-language
       (let ((insert-file-p (or (eq this-command 'insert-file)
				(and (memq this-command
					   '(minibuffer-complete-and-exit
					     exit-minibuffer))
				     (eq (car-safe (car command-history))
					 'insert-file)))))
	 (when (or (null insert-file-p)
		   (<= (+ (point) len) (point-max)))
	   (save-restriction
	     (if insert-file-p (narrow-to-region (point) (+ (point) len)))
	     (let ((origpos (point))
		   (modified (buffer-modified-p)) ; t if `recover-file'!
		   ;;(buffer-undo-list t) ; do not record changes
		   ;; we cannot set buffer-undo-list to t even if the previous
		   ;; value is nil because M-x insert-file as the first command
		   ;; after reading a file would set the old insert-region
		   ;; boundaries into the undo-list
		   (buffer-read-only nil) ; always allow conversion
		   (inhibit-read-only t)
		   (after-change-functions nil)) ; no fontification
	       (when (x-symbol-decode-all x-symbol-safe-exec-threshold)
		 (goto-char origpos)
		 (or modified (set-buffer-modified-p nil))
		 ;; `len' is utterly wrong when not called in `insert-file',
		 ;; but there is no chance to get it right if we don't know the
		 ;; start position, see above.
		 (setq len (- (point-max) (point-min)))))))))
  len)

(defun x-symbol-write-region-annotate-function (start end)
  ;; checkdoc-params: (start end)
  "Encode x-symbol characters using another buffer.
Added to `write-region-annotate-functions' if
`x-symbol-auto-conversion-method' has value 'fast'."
  (and x-symbol-mode x-symbol-language
       (let ((selective selective-display))
	 (when (x-symbol-encode-all x-symbol-safe-exec-threshold nil
				    (get-buffer-create " x-symbol conversion"))
	   (setq selective-display selective))))
  nil)

(defun x-symbol-write-file-hook ()
  "Encode x-symbol characters in current buffer.
Added to `write-file-hooks' if `x-symbol-auto-conversion-method' has a
value other than nil or 'fast'.  Refontifies buffer if
`x-symbol-auto-conversion-method' has value `slowest'."
  (and x-symbol-mode x-symbol-language
      (let ((buffer-read-only nil)
	    (after-change-functions nil)) ; no fontification!
	(widen)
	;; Called inside `save-recursion' and `save-restriction', thus no way
	;; to restore point, mark, and the region boundaries when executables
	;; have been used.  Therefore, don't use them by default.
	(call-with-transparent-undo
	 (lambda ()
	   (x-symbol-encode-all)	; not the safe version!
	   (continue-save-buffer)))
	(and (eq x-symbol-auto-conversion-method 'slowest)
	     font-lock-mode
	     (x-symbol-fontify))
	(set-buffer-modified-p nil)
	'x-symbol-write-file-hook)))	; do not write again


;;;===========================================================================
;;;  Init
;;;===========================================================================

(defun x-symbol-init-language-accesses (language alist)
  "Initialize accesses for token language LANGUAGE according to ALIST.
The symbol property `x-symbol-feature' of LANGUAGE must be set before.
See also `x-symbol-language-access-alist'."
  ;;If optional NO-TEST is nil, accesses which do not point to a bound
  ;;variable are not set.
  (let ((feature (get language 'x-symbol-feature))
	symbol)
    (unless feature
      (error "Illegal X-Symbol token language `%s'" language))
    (dolist (item alist)
      (and (null (get language (car item)))
	   (stringp
	    (setq symbol (if (consp (cdr item))
			     (if (featurep 'mule) (cadr item) (cddr item))
			   (cdr item))))
	   (setq symbol (intern (format "%s-%s" feature symbol)))
	   (or (boundp symbol)
	       (progn (warn "X-Symbol language %s forgot to define `%s'"
			    language symbol)
		      nil))
	   (put language (car item) symbol)))))

;;;###autoload
(defun x-symbol-register-language (language feature modes)
  "Register token language LANGUAGE.
FEATURE is a feature which `provide's LANGUAGE.  MODES are major modes
which typically use LANGUAGE.  Using LANGUAGE's accesses will initialize
LANGUAGE, see `x-symbol-language-value'."
  (unless (get language 'x-symbol-feature)
    (put language 'x-symbol-feature feature))
  (x-symbol-init-language-accesses language '((x-symbol-name . "name")))
  (unless (assq language x-symbol-language-alist)
    (setq x-symbol-language-alist
	  (nconc x-symbol-language-alist
		 (list (cons language
			     (symbol-value (get language 'x-symbol-name)))))))
  (dolist (mode modes)
    (put mode 'x-symbol-language language)
    (put mode 'x-symbol-font-lock-language language)))

;;;###autoload
(defun x-symbol-initialize (&optional arg)
  "Initialize package X-Symbol.
See variable `x-symbol-initialize' and function `x-symbol-after-init'.
Also allocate colormap, see `x-symbol-image-colormap-allocation'.
Unless optional argument ARG is non-nil, do not initialize package
X-Symbol twice."
  (interactive "P")
  (unless (and (get 'x-symbol 'x-symbol-initialized) (null arg))
    (put 'x-symbol 'x-symbol-initialized t)
    ;; Token languages -------------------------------------------------------
    (when (or (eq x-symbol-initialize t)
	      (memq 'languages x-symbol-initialize))
      (x-symbol-register-language 'tex 'x-symbol-tex x-symbol-tex-modes)
      (x-symbol-register-language 'sgml 'x-symbol-sgml x-symbol-sgml-modes)
      (x-symbol-register-language 'utex 'x-symbol-utex x-symbol-utex-modes))
    ;; Global mode -----------------------------------------------------------
    (when (or (eq x-symbol-initialize t)
	      (memq 'global x-symbol-initialize))
      (add-hook 'hack-local-variables-hook 'turn-on-x-symbol-conditionally))
    ;; Key bindings ----------------------------------------------------------
    (when (or (eq x-symbol-initialize t)
	      (memq 'keys x-symbol-initialize))
      (global-set-key (vector x-symbol-compose-key) 'x-symbol-map-autoload)
      (global-set-key [(control ?\,)] 'x-symbol-modify-key)
      (global-set-key [(control ?\.)] 'x-symbol-rotate-key))
    ;; Font path -------------------------------------------------------------
    (and (or (eq x-symbol-initialize t)
	     (memq 'font-path x-symbol-initialize))
	 x-symbol-font-directory
	 (file-accessible-directory-p x-symbol-font-directory)
	 ;; by Jim Radford <radford@robby.caltech.edu>:
	 (default-x-device)
	 (if (fboundp 'x-set-font-path)
	     (let ((font-path (x-get-font-path)))
	       (unless (member x-symbol-font-directory font-path)
		 (x-set-font-path (nconc font-path
					 (list x-symbol-font-directory)))))
	   ;; This should be commented out until I can figure out how to
	   ;; get the display name into the -display arg for xset.
	   (with-temp-buffer
	     (call-process "xset" nil t nil "q")
	     (goto-char (point-min))
	     (unless (search-forward x-symbol-font-directory nil t)
	       (call-process "xset" nil nil nil "fp+"
			     x-symbol-font-directory)))))
    ;; Package fast-lock -----------------------------------------------------
    (when (or (eq x-symbol-initialize t)
	      (memq 'fast-lock x-symbol-initialize))
      (setq fast-lock-save-faces nil))
    ;; Package AucTeX ----------------------------------------------------------
    (when (or (eq x-symbol-initialize t)
	      (memq 'auctex x-symbol-initialize))
      (add-hook 'TeX-region-hook 'x-symbol-tex-init-auctex-region) ; v9.8a+
      (setq LaTeX-math-insert-function 'x-symbol-tex-math-insert)) ; v9.8a+
    ;; Package RefTeX --------------------------------------------------------
    (when (or (eq x-symbol-initialize t)
	      (memq 'reftex x-symbol-initialize))
      (unless (and (boundp 'reftex-translate-to-ascii-function)
		   (fboundp reftex-translate-to-ascii-function)
		   (not (eq reftex-translate-to-ascii-function
			    'reftex-latin1-to-ascii)))
	(setq reftex-translate-to-ascii-function 'x-symbol-translate-to-ascii))
      (add-hook 'reftex-pre-refontification-functions
		'x-symbol-inherit-from-buffer)
      (unless (featurep 'mule)
	(add-hook 'reftex-display-copied-context-hook
		  'x-symbol-nomule-fontify-cstrings)))
    ;; Miscellaneous ---------------------------------------------------------
    (x-symbol-image-set-colormap nil nil)
    (if init-file-loaded
	(x-symbol-after-init)
      (add-hook 'after-init-hook 'x-symbol-after-init))))

(defun x-symbol-after-init ()
  "Late initialization for package X-Symbol.
See function `x-symbol-initialize' and variables `x-symbol-initialize'
and `x-symbol-auto-conversion-method'.  Also add elements to
`x-symbol-auto-mode-suffixes' if necessary."
  (when x-symbol-auto-conversion-method
    (and (eq x-symbol-auto-conversion-method 'auto-slow)
	 (null (featurep 'crypt))
	 (setq x-symbol-auto-conversion-method 'fast))
    (if (eq x-symbol-auto-conversion-method 'fast)
	(progn
	  (add-hook 'after-insert-file-functions 'x-symbol-after-insert-file t)
	  (add-hook 'write-region-annotate-functions
		    'x-symbol-write-region-annotate-function))
      (add-hook 'write-file-hooks 'x-symbol-write-file-hook)))
  ;; misc user additions to `auto-mode-alist':
  (setq x-symbol-auto-mode-suffixes (x-symbol-auto-mode-suffixes
				     x-symbol-auto-mode-suffixes))
  ;; Package comint ----------------------------------------------------------
  (when (or (eq x-symbol-initialize t)
	    (memq 'comint x-symbol-initialize))
    (add-hook 'comint-output-filter-functions 'x-symbol-comint-output-filter)
    (and (boundp 'comint-input-sender)
	 (not (eq comint-input-sender 'x-symbol-comint-send))
	 (setq x-symbol-orig-compint-input-sender comint-input-sender))
    (setq comint-input-sender 'x-symbol-comint-send))
  ;; Package bib-cite: X-Symbol decoding would overwrite cite highlighting with
  ;; normal installation of bib-cite -----------------------------------------
  (when (and (or (eq x-symbol-initialize t)
		 (memq 'bib-cite x-symbol-initialize))
	     (or (and (boundp 'LaTeX-mode-hook)
		      (memq 'turn-on-bib-cite LaTeX-mode-hook))
		 (and (boundp 'latex-mode-hook)
		      (memq 'turn-on-bib-cite latex-mode-hook))))
    (remove-hook 'LaTeX-mode-hook 'turn-on-bib-cite)
    (remove-hook 'latex-mode-hook 'turn-on-bib-cite)
    (add-hook 'find-file-hooks
	      (lambda ()
		(if (eq major-mode 'latex-mode) (turn-on-bib-cite))))))

;;; Local Variables:
;;; eval: (put 'x-symbol-define-user-options 'lisp-indent-function 2)
;;; End:
;;; Local IspellPersDict: .ispell_xsymb
;;; x-symbol-hooks.el ends here
