// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_ReturnStatement.cc,v 1.2.2.1 1999/11/06 15:16:20 dmartin Exp $
// 
//---------------------------------------------------------------------------

#include "IIRScram_ReturnStatement.hh"
#include "IIR_Identifier.hh"
#include "error_func.hh"
#include "set.hh"
#include "resolution_func.hh"
#include "IIR_SubprogramDeclaration.hh"
#include "IIR_Label.hh"
#include "IIR_TypeDefinition.hh"

IIRScram_ReturnStatement::~IIRScram_ReturnStatement(){
}


void 
IIRScram_ReturnStatement::_publish_vhdl(ostream &_vhdl_out) {
  if (get_label() != NULL) {
    get_label()->_publish_vhdl(_vhdl_out);
    _vhdl_out << ": ";
  }
    
  _vhdl_out << "return ";
    
  if (get_return_expression() != NULL) {
    ASSERT(get_return_expression()->_is_resolved());
    get_return_expression()->_publish_vhdl(_vhdl_out);
  }
}


void 
IIRScram_ReturnStatement::_publish_cc() {
  IIR_TypeDefinition *ret_type;

  if(_allocate_return_value == TRUE) {
    _publish_cc_resolution_fn_return();
  } else {
    if (get_label() != NULL) {
      get_label()->_publish_cc();
      _cc_out << ": ";
    } // if

    _cc_out << "return ";
    if (get_return_expression() != NULL) {
      ASSERT(get_return_expression()->_is_resolved());
      ret_type = get_enclosing_subprogram()->_get_return_type();
      if (ret_type->_is_array_type() || ret_type->_is_record_type()) {
	_cc_out << "((";
	ret_type->_publish_cc();
	_cc_out << " &)";
	get_return_expression()->_publish_cc();
	_cc_out << ")";
      } else {
	_cc_out << "(";
	ret_type->_publish_cc_type_name();
	_cc_out << "(ObjectBase::VARIABLE, (const ";
	ret_type->_publish_cc_kernel_type();
	_cc_out <<" &)";
	get_return_expression()->_publish_cc();
	_cc_out << "))";
      }
    } else {
      ASSERT(get_enclosing_subprogram()->_is_iir_procedure_declaration() == TRUE);
      _cc_out << "NORMAL_RETURN";
    }
	_cc_out << ";" << endl;
  } // else
}

void 
IIRScram_ReturnStatement::_publish_cc_resolution_fn_return(){
  // Open a temporary scope.
  _cc_out << "{" << endl
	  << "VHDLType *savantTemp = new ";
  if (get_enclosing_subprogram()->_get_return_type()->_is_scalar_type() == TRUE) {
    get_enclosing_subprogram()->_get_return_type()->_publish_cc_kernel_type();
  }
  else {
    get_enclosing_subprogram()->_get_return_type()->_publish_cc_type_name();
  }
  _cc_out << "(ObjectBase::VARIABLE, ";

  if(get_return_expression()->_get_subtype()->_is_scalar_type() == TRUE) {
    _cc_out << "(const ";
    get_return_expression()->_get_subtype()->_publish_cc_kernel_type();
    _cc_out << " &) ";
  }
  
  get_return_expression()->_publish_cc();
  _cc_out << ");" << endl
	  << "return savantTemp;" << endl
	  << "}" << endl;
}

void 
IIRScram_ReturnStatement::_type_check(){
  // This needs to actually be type-checked within the context of the
  // return type.
  // See next method...
}

void 
IIRScram_ReturnStatement::_type_check( set<IIR_TypeDefinition> *context_set ){
  ASSERT( get_enclosing_subprogram() != NULL );

  if( get_enclosing_subprogram()->get_kind() == IIR_FUNCTION_DECLARATION  ){
    // Then this return statement appeared in a function declaration.
    // 1) The expression must not be NULL.
    // 2) If the context_set is NULL, something is broken.
    ASSERT( context_set != NULL );

    if( get_return_expression() != NULL ){
      // Then we need to type_check based on the context_set passed in...
      set_return_expression( get_return_expression()->_semantic_transform( context_set ));
      get_return_expression()->_type_check( context_set );

      set<IIR_TypeDefinition> *expression_types = get_return_expression()->_get_rval_set();
      if( expression_types == NULL ){
	report_undefined_symbol( get_return_expression() );
      }

      reconcile_sets( context_set, expression_types );
      
      switch( expression_types->num_elements() ){
      case 0:{
	ostrstream err;
	err << "Expression |" << *get_return_expression() << "| must have a return type "
	    << "compatible with that of its function, |" 
	    <<  *get_enclosing_subprogram()->get_declarator() << "|." <<  ends;
	report_error( this, err );
	break;
      }
      case 1:{
	IIR_TypeDefinition *my_type = expression_types->get_element() ;
	set_return_expression( get_return_expression()->_rval_to_decl( my_type ));
	break;
      }
      default:{
	report_ambiguous_error( get_return_expression(), expression_types );
	break;
      }
      }

      delete expression_types;
    }
    else{
      ostrstream err;
      err << "|" << *get_enclosing_subprogram()->get_declarator() << "| " 
	  << "was declared as a function, and should return a value." 
	  << ends;
      report_error( this, err, WARNING );
    }
  }
  else{
    // Then this return statement appeared in a procedure declaration.
    // 1) The expression must be NULL.
    ASSERT( get_enclosing_subprogram()->get_kind() == IIR_PROCEDURE_DECLARATION );

    if( get_return_expression() != NULL ){
      ostrstream err;
      err << "|" << get_enclosing_subprogram()->get_declarator() << "| " 
	  << "was declared as a procedure, and may not return a value." 
	  << ends;
      report_error( this, err );
    }
  }
}

IIR_Boolean 
IIRScram_ReturnStatement::_is_resolved(){
  ASSERT( get_enclosing_subprogram() != NULL );

  if( get_return_expression() != NULL ){
    return get_return_expression()->_is_resolved();
  }
  else {
    return TRUE;
  }
}
