// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_WaitStatement.cc,v 1.3 1999/09/22 00:38:55 tmcbraye Exp $
// 
//---------------------------------------------------------------------------
#include "IIRScram_WaitStatement.hh"
#include "IIR_WaitStatement.hh"
#include "IIR_DesignatorExplicit.hh"
#include "IIR_Declaration.hh"
#include "IIR_Identifier.hh"
#include "error_func.hh"
#include "set.hh"
#include "resolution_func.hh"
#include <strstream.h>
#include "symbol_table.hh"
#include "IIR_Label.hh"
#include "IIR_DesignatorList.hh"
#include "IIR_Designator.hh"
#include "StandardPackage.hh"
#include "IIR_EnumerationTypeDefinition.hh"
#include "IIR_PhysicalSubtypeDefinition.hh"

extern symbol_table *cgen_sym_tab_ptr;

IIRScram_WaitStatement::~IIRScram_WaitStatement() {}

void 
IIRScram_WaitStatement::_publish_vhdl(ostream &_vhdl_out) {

  if (get_label() != NULL) {
    get_label()->_publish_vhdl(_vhdl_out);
    _vhdl_out << ": ";
  }
    
  _vhdl_out << "wait";
    
  if (sensitivity_list.num_elements() != 0) {
    _vhdl_out << " on ";
    sensitivity_list._publish_vhdl(_vhdl_out);
  }
    
  if (get_condition_clause() != NULL) {
    _vhdl_out << " until ";
    get_condition_clause()->_publish_vhdl(_vhdl_out);
  }
    
  if (get_timeout_clause() != NULL) {
    _vhdl_out << " for ";
    get_timeout_clause()->_publish_vhdl(_vhdl_out);
  }
}

void 
IIRScram_WaitStatement::_publish_cc() {
  if(_is_currently_publishing_subprogram() == TRUE) {
    _publish_cc_procedure_wait();
    return;
  }
  _cc_out << "executeWait(" << wait_id;
  if(get_timeout_clause() != NULL) {
    _cc_out << ", ";
    get_timeout_clause()->_publish_cc();
    get_timeout_clause()->_add_decl_into_cgen_symbol_table();
  }
  _cc_out << ");\nreturn;\n";

  _publish_cc_wait_label();
  _cc_out << ":\n"
	  << "if (!";
  if(_is_currently_publishing_subprogram() == TRUE) {
    _cc_out << "processPtr->";
  }
  _cc_out << "resumeWait(" << wait_id;
  if(get_condition_clause() != NULL) {
    _cc_out << ", ";
    get_condition_clause()->_publish_cc();
    get_condition_clause()->_add_decl_into_cgen_symbol_table();
  }
  _cc_out << ")) return;\n";
  sensitivity_list._add_decl_into_cgen_symbol_table();
}


void 
IIRScram_WaitStatement::_publish_cc_procedure_wait() {
  IIR_Declaration *decl;
  const int _num_decls = 4;
  IIRScram_Declaration::declaration_type _proc_decl;
  IIRScram_Declaration::declaration_type _proc_decl_type[_num_decls] =
  {IIRScram_Declaration::VARIABLE, 
   IIRScram_Declaration::CONSTANT, 
   IIRScram_Declaration::INTERFACE_VARIABLE,
   IIRScram_Declaration::INTERFACE_CONSTANT};
  _cc_out << "processPtr->setProcedureWait(&(procedureWait[" << wait_id 
	  << "]));\n"
	  << "{\n"
	  << "StackElement *newElem = new StackElement;\n"
	  << "newElem->waitLabel = " << wait_id << ";\n"
	  << "int numSignals = 0;\n";
  register int i;

  // Save necessary declarations.
  for(i = 0; i < _num_decls; i++) {
    _proc_decl = _proc_decl_type[i];
    decl = cgen_sym_tab_ptr->in_scope_by_type[_proc_decl].first();
    while (decl != NULL) {
      _cc_out << "if(!";
      decl->_publish_cc();
      _cc_out << "._is_signal()) numSignals++;\n";
      decl = cgen_sym_tab_ptr->in_scope_by_type[_proc_decl].successor(decl);
    } // while
  } // for

  _cc_out << "newElem->args = (VHDLType**) new "
	  << "char[numSignals* sizeof(VHDLType*)];\n"
	  << "numSignals = 0;\n";

  for(i = 0; i < _num_decls; i++) {
    _proc_decl = _proc_decl_type[i];
    decl = cgen_sym_tab_ptr->in_scope_by_type[_proc_decl].first();
    while (decl != NULL) {
      _cc_out << "if(!";
      decl->_publish_cc();
      _cc_out << "._is_signal()) {\n"
	      << "newElem->args[numSignals] = new ";
      decl->_get_subtype()->_publish_cc_type_name();
      _cc_out << "(";

      if (decl->_get_subtype()->_is_access_type() == FALSE) {
	decl->_publish_cc();
	_cc_out << ".object->getKind()";
      }
      
      _cc_out << ");\n";
      _cc_out << "*newElem->args[numSignals++] = ";
      decl->_publish_cc();
      _cc_out << ";\n"
	      << "newElem->numArgs++;\n"
	      << "}\n";
      decl = cgen_sym_tab_ptr->in_scope_by_type[_proc_decl].successor(decl);
    } // while
  } // for

  _cc_out << "callStack->push(newElem);\n"
	  << "}\n";
  
  _cc_out << "processPtr->executeWait(WAITING_IN_PROC";
  if(get_timeout_clause() != NULL) {
    _cc_out << ", ";
    get_timeout_clause()->_publish_cc();
  }
  _cc_out << ");\n"
	  << "return EXECUTE_WAIT_RETURN;\n";
  _publish_cc_wait_label();
  _cc_out << ":\n"
	  << "if(!processPtr->resumeWait(WAITING_IN_PROC";
  if(get_condition_clause() != NULL) {
    _cc_out << ", ";
    get_condition_clause()->_publish_cc();
  }
  _cc_out << ")) "
	  << "return RESUME_WAIT_RETURN;\n";

  // Restore the required declarations.
  _cc_out << "{\n"
	  << "int numSignals = 0;\n";
  for(i = 0; i < _num_decls; i++) {
    _proc_decl = _proc_decl_type[i];
    decl = cgen_sym_tab_ptr->in_scope_by_type[_proc_decl].first();
    while (decl != NULL) {
      _cc_out << "if(!";
      decl->_publish_cc();
      _cc_out << "._is_signal()) {\n";
      decl->_publish_cc();
      _cc_out << " = (const ";
      decl->_get_subtype()->_publish_cc_type_name();
      _cc_out << " &) *(callStack->getCurrentTop()->args[numSignals]);\n"
	      << "delete (callStack->getCurrentTop()->args[numSignals++]);\n"
	      << "}\n";
      decl = cgen_sym_tab_ptr->in_scope_by_type[_proc_decl].successor(decl);
    } // while
  } // for
  _cc_out << "callStack->popAboveCurrent();\n"
	  << "}\n";
}

void 
IIRScram_WaitStatement::_publish_cc_wait_label() {
  if (get_label() != NULL) {
    get_label()->_publish_cc();
  }
  else {
    _cc_out << "wait";
    _cc_out << this;
  }
}


void 
IIRScram_WaitStatement::_publish_cc_wait_data(IIR_Char* procname) {
  // insert all sigs in the senslist into this process's data structures
  if (sensitivity_list.num_elements() != 0) {
    IIR_Declaration *sig_decl;
    IIR_Designator *sig_designator = sensitivity_list.first();
    while (sig_designator != NULL) {
      sig_decl = (IIR_Declaration*)((IIR_DesignatorExplicit*)sig_designator)->get_name();
      ASSERT(sig_decl->_is_signal() == TRUE );
      if(sig_decl->_is_iir_declaration() == FALSE) {
	sig_decl =(IIR_Declaration*)((IIR_DesignatorExplicit*)sig_decl)->get_name();
      }

      if (!cgen_sym_tab_ptr->in_scope(sig_decl)) {
	cgen_sym_tab_ptr->add_declaration(sig_decl);
      }
      sig_designator = sensitivity_list.successor(sig_designator);
    }
  }
}

void
IIRScram_WaitStatement::_publish_cc_proc_wait_data() {
  IIR_Char* old_prefix_string = _get_publish_prefix_string();
  // The wait condition and wait timeout clauses are published as
  // functions.  These functions are similar to the user-defined
  // functions, and hence we trick the code-generator by saying that we
  // are publishing a subprogram, whereas we are actually not.
  PublishedUnit _saved_publishing_unit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(PROCEDURE);
  _set_publish_prefix_string("state->");
    
  // Redefine savantnow here.
  _cc_out << "#ifdef savantnow\n"
	  << "#undef savantnow\n"
	  << "#define savantnow (PhysicalType(ObjectBase::VARIABLE,"
	  << "UniversalLongLongInteger(processPtr->getTimeNow().time),"
	  << " SavanttimeType_info))\n"
	  << "#endif\n";

  if (get_condition_clause() != NULL) {
    _cc_out << "bool\n"
	    << "waitCond" << wait_id << "_" << this
	    << "(VHDLKernelBase *processPtr) {\n"
	    << "  " << this << "_state* state;\n"
	    << "  state = (" << this << "_state*) processPtr->getState();\n"
	    << "  return (SAVANT_BOOLEAN_TRUE  == ";
    get_condition_clause()->_publish_cc(); 
    _cc_out << ");\n"
	    << "}\n\n";
  }

  if (get_timeout_clause() != NULL) {
    _cc_out << "PhysicalType\n"
	    << "waitTimeout" << wait_id << "_" << this
	    << "(VHDLKernelBase *processPtr) {\n"
	    << "  " << this << "_state* state;\n"
	    << "  state = (" << this << "_state*) processPtr->getState();\n"
	    << "  return (";
    get_timeout_clause()->_publish_cc();
    _cc_out << ");\n"
	    << "}\n\n";
  }

  // Restore the definition of savantnow.
  _cc_out << "#ifdef savantnow\n"
	  << "#undef savantnow\n"
	  << "#define savantnow (PhysicalType(ObjectBase::VARIABLE,"
	  << "UniversalLongLongInteger(getTimeNow().time),"
	  << " SavanttimeType_info))\n"
	  << "#endif\n";
  _set_publish_prefix_string(old_prefix_string);
  _set_currently_publishing_unit(_saved_publishing_unit);
}  

void 
IIRScram_WaitStatement::_publish_cc_wait_decl() {
  if (get_condition_clause() != NULL) {
    _cc_out << "extern bool waitCond" << wait_id << "_" << this
	    << "(VHDLKernelBase*);\n";
  }
  if (get_timeout_clause() != NULL) {
    _cc_out << "extern PhysicalType waitTimeout" << wait_id << "_" << this
	    << "(VHDLKernelBase*);\n";
  }
}


void
IIRScram_WaitStatement::_build_wait_list(dl_list<IIRScram_WaitStatement>* list) {
  list->append(this);
  wait_id = list->num_elements() - 1;
}


void
IIRScram_WaitStatement::_type_check() {
  IIR *signal_decl;
  IIR *current_name;

  IIR_Designator *current_designator = sensitivity_list.first();
  while (current_designator != NULL) {
    ASSERT(current_designator->get_kind() == IIR_DESIGNATOR_EXPLICIT);

    current_name = ((IIR_DesignatorExplicit*)current_designator)->get_name();

    // If a signal in the sensitivity list can't be resolved, it will
    // complain about it in this function call...
    signal_decl = _resolve_signal_name(current_name);
    if (signal_decl != NULL) {
      // This might leak a name...
      ((IIR_DesignatorExplicit *)current_designator)->set_name(signal_decl);
    }
    current_designator = sensitivity_list.successor(current_designator);
  }
  
  if( get_condition_clause() != NULL ){
    IIR_TypeDefinition *boolean_type = StandardPackage::boolean_type;
    set_condition_clause( get_condition_clause()->_semantic_transform( boolean_type ) );
    get_condition_clause()->_type_check( boolean_type );
    set_condition_clause( get_condition_clause()->_rval_to_decl( boolean_type ) );
  }

  if( get_timeout_clause() != NULL ){
    IIR_TypeDefinition *time_type = StandardPackage::time_type;
    set_timeout_clause( get_timeout_clause()->_semantic_transform( time_type ) );
    get_timeout_clause()->_type_check( time_type );
    set_timeout_clause( get_timeout_clause()->_rval_to_decl( time_type ) );
  }
  
}


IIR *
IIRScram_WaitStatement::_resolve_signal_name(IIR *sig_name) {
  IIR *retval;

  set<IIR_TypeDefinition> *signal_rvals = sig_name->_get_rval_set( &IIR::_is_signal );

  if( signal_rvals == NULL ){
    report_undefined_symbol( sig_name );
    return NULL;
  }

  switch (signal_rvals->num_elements()) {
  case 0: {
    ostrstream err;
    err << "Signal " << *sig_name << " is undeclared in this scope" << ends;
    report_error(this, err);
    break;
  }
  case 1: {
    IIR_TypeDefinition *sig_type = signal_rvals->get_element();

    IIR *sig_name_transformed = sig_name->_semantic_transform( sig_type );
    sig_name_transformed->_type_check( sig_type );
    retval = sig_name_transformed->_rval_to_decl( sig_type );

    ASSERT( retval->_is_resolved() == TRUE );
    break;
  }
  default:
    ostrstream err;
    report_ambiguous_error( sig_name, signal_rvals );
    break;    
  }

  delete signal_rvals;
  return retval;
}

void
IIRScram_WaitStatement::_get_list_of_input_signals(set<IIR_Declaration>* list) {
  if(get_condition_clause() != NULL) {
    get_condition_clause()->_get_list_of_input_signals(list);
  }
  //get_timeout_clause()->_get_list_of_input_signals(list);
  sensitivity_list._get_list_of_input_signals(list);
}

IIR*
IIRScram_WaitStatement::_clone() {
  IIR_WaitStatement *stmt;
  IIR *condition = NULL, *timeout = NULL;
  IIR_Designator *oldname, *newname;

  stmt = new IIR_WaitStatement();
  IIRScram_SequentialStatement::_clone(stmt);

  condition = get_condition_clause();
  if (condition != NULL) {
    condition = condition->_clone();
  }
  stmt->set_condition_clause(condition);

  timeout = get_timeout_clause();
  if (timeout != NULL) {
    timeout = timeout->_clone();
  }
  stmt->set_timeout_clause(timeout);

  oldname = sensitivity_list.first();
  while (oldname != NULL) {
    newname = (IIR_Designator*)oldname->_clone();
    stmt->sensitivity_list.append(newname);
    oldname = sensitivity_list.successor(oldname);
  }

  return stmt;
}

