/*C*
 *
 * Hatman - The Game of Kings
 * Copyright (C) 1997 James Pharaoh & Timothy Fisken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *C*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "Console.h"
#include "mode.h"
#include "Mouse.h"
#include "Rect.h"
#include "../util/debug.h"
#include <assert.h>

#ifdef SVGALIB
#include <vga.h>
#include <vgamouse.h>
#endif

#ifdef GGI
#include <ggi/events.h>
#include <ggi/gii.h>
#endif

//--------------------------------------------------------------------------------------------------------------------------------

#ifdef SVGALIB

template Collection<MouseEvent>;

const int mbLeft   = 0x00000004;
const int mbRight  = 0x00000001;

void mouseHandler(int b, int dx, int dy, int dz, int drx, int dry, int drz)
{
 assert(mouse);
 dx /= 30; dy /= 30; // svgalib seems to multiply these up, so we'll divide them down
 if(dx || dy)
  {
   (mouse->pPosition += Point(dx, dy)).clipTo(screenRect);
   if(mouse->lastClick &&
      (abs(mouse->lastClickPos.x - mouse->pPosition.x) >= 4 || abs(mouse->lastClickPos.y - mouse->pPosition.y) >= 4))
    mouse->lastClick = false;
   mouse->eventQ->add(new MouseMoveEvent(mouse->pPosition));
  }
 if(b != mouse->pButton)
  {
   if((mouse->pButton & mbLeft) == 0 && (b & mbLeft) == mbLeft)
    {
     if(mouse->lastClick
	&& abs(mouse->lastClickPos.x - mouse->pPosition.x) < 4 && abs(mouse->lastClickPos.y - mouse->pPosition.y) < 4)
      {
       mouse->lastClick = false;
       mouse->eventQ->add(new MouseDbClickEvent(mouse->pPosition));
      }
     else
      {
       mouse->lastClick = true;
       mouse->lastClickPos = mouse->pPosition;
       mouse->eventQ->add(new MouseLBDownEvent(mouse->pPosition));
      }
    }
   if((mouse->pButton & mbLeft) == mbLeft && (b & mbLeft) == 0)
    mouse->eventQ->add(new MouseLBUpEvent(mouse->pPosition));
   if((mouse->pButton & mbRight) == 0 && (b & mbRight) == mbRight)
    mouse->eventQ->add(new MouseRBDownEvent(mouse->pPosition));
   if((mouse->pButton & mbRight) == mbRight && (b & mbRight) == 0)
    mouse->eventQ->add(new MouseRBUpEvent(mouse->pPosition));
   mouse->pButton = b;
  }
}

#endif

//--------------------------------------------------------------------------------------------------------------------------------

Mouse::Mouse()
{
#ifdef SVGALIB
 mouse_init("/dev/mouse", vga_getmousetype(), MOUSE_DEFAULTSAMPLERATE);
 mouse_seteventhandler(mouseHandler);
 eventQ = new Collection<MouseEvent>;
 pButton = 0;
#endif

 pPosition = origin;
 sprite = NULL;

 lastClick = false;

 VPRINTF("<mouse> initialised\n");
}

Mouse::~Mouse()
{
#ifdef SVGALIB
 mouse_close();
 delete eventQ;
#endif

 VPRINTF("<mouse> closed\n");
}

//--------------------------------------------------------------------------------------------------------------------------------

MouseEvent* Mouse::getEvent()
{

#ifdef GGI
 timeval tv = {0, 0};
 if(ggiEventPoll(consoleVisual, emPointer, &tv))
  {
   gii_event e;
   ggiEventRead(consoleVisual, &e, emPointer);

   switch(e.any.type)
    {
    case evPtrAbsolute:
     if(lastClick && (abs(lastClickPos.x - pPosition.x) >= 4 || abs(lastClickPos.y - pPosition.y) >= 4))
      lastClick = false;
     return new MouseMoveEvent(pPosition = Point(e.pmove.x, e.pmove.y));

    case evPtrRelative:
     if(lastClick && (abs(lastClickPos.x - pPosition.x) >= 4 || abs(lastClickPos.y - pPosition.y) >= 4))
      lastClick = false;
     return new MouseMoveEvent(pPosition = (pPosition + Point(e.pmove.x, e.pmove.y)).clipTo(screenRect));

    case evPtrButtonPress:
     switch(e.pbutton.button)
      {
      case GII_PBUTTON_LEFT:
       {
	if(lastClick && abs(lastClickPos.x - pPosition.x) < 4 && abs(lastClickPos.y - pPosition.y) < 4)
	 {
	  lastClick = false;
	  return new MouseDbClickEvent(pPosition);
	 }
	else
	 {
	  lastClick = true;
	  lastClickPos = mouse->pPosition;
	  return new MouseLBDownEvent(mouse->pPosition);
	 }
       }
      case GII_PBUTTON_RIGHT:
       return new MouseRBDownEvent(pPosition);
      }
     break;
    case evPtrButtonRelease:
     switch(e.pbutton.button)
      {
      case GII_PBUTTON_LEFT:
       return new MouseLBUpEvent(pPosition);
      case GII_PBUTTON_RIGHT:
       return new MouseRBUpEvent(pPosition);
      }
     break;

    default:
     nonFatal("mouse got unknown ggi event %d\n", e.any.type);
    }
  }
 return NULL;
#endif

#ifdef SVGALIB
 foreach(*eventQ, i)
  if(eventQ->at(i)) return eventQ->remove(i);
 eventQ->pack();
 mouse_update();
 return NULL;
#endif

}

//--------------------------------------------------------------------------------------------------------------------------------
