/*
 * File: gtk_connect.c
 * Author: Brent Hendricks
 * Project: NetSpades
 * Date: 1/23/99
 *
 * This fils contains functions for connecting with the server
 *
 * Copyright (C) 1998 Brent Hendricks.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#include <config.h>

#include <gtk/gtk.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <gtk_connect.h>
#include <gtk_io.h>
#include <gtk_play.h>
#include <gtk_taunt.h>
#include <gtk_dlg_error.h>
#include <client.h>
#include <display.h>
#include <socketfunc.h>

/* Needed to launch offline game engine */
#include <engine.h>
gameInfo_t gameInfo;
int log = 0;

/* Delay before clearing table */
#define CLEAR_DELAY 3000

/* Global state of game variable */
extern gameState_t gameState;
extern playArea_t* playArea;
extern option_t options;
extern GtkWidget* tauntBox;

/* Various local handles */
static guint tauntHandle = 0;
static guint spadesHandle = 0;


/**
 * Begin connection to server
 */
void Connect( GtkWidget *pBar ) {

  GtkWidget *menuItem;
  guint status;

  if( gameState.gameSegment == ST_REGISTER ) { 
    status = Register();
    
    switch( status ) {
    case ERR_SOCK_OPEN:
    case ERR_HOST: 
    case ERR_SOCK_CONN:
      DisplayError( errorMsg[ status + ERR_OFFSET ] );
      ConnectDialogReset(pBar);
      break;
      
    default:
      /* Enable Game/End menu item now that we're connected*/
      menuItem = gtk_item_factory_get_widget( playArea->menu, "<main>/Game/End" );
      gtk_widget_set_sensitive( menuItem, TRUE);

      /* Update progress bar */
      gtk_progress_bar_update( GTK_PROGRESS_BAR(pBar), 0.4 );
      gtk_progress_set_format_string( GTK_PROGRESS(pBar), "Waiting for ID...");

      /* Next game state */
      UpdateGame();

      /* Start receiving events when there is data from server */
      spadesHandle = gdk_input_add_full( gameState.spadesSock, GDK_INPUT_READ, 
					ReadServerSocket, (gpointer)pBar,
					NULL);
      break; 

    } /* switch( status ) */
  } /* if( gameState.gameSegment == ST_REGISTER ) */
}


/*
 * Disconnect from server, close sockets, etc.
 */
void Disconnect( GtkWidget* widget, gpointer data ) {

  GtkWidget* menuItem;
  int tmp;
  
  /* Remove only those inputs which are enabled */
  switch( gameState.gameSegment ) {
  case ST_GET_HAND:
  case ST_GET_BIDDER:
  case ST_GET_BIDS:
  case ST_GET_LEAD:  
  case ST_GET_TRICK:  
  case ST_GET_TALLY: 
  case ST_GET_SCORE: 
  case ST_END_GAME: 
    if( tauntHandle ) { 
      gdk_input_remove( tauntHandle );
      tauntHandle = 0;
    }
  case ST_GET_ID: 
  case ST_GET_GAME: 
  case ST_GET_NAMES:
  case ST_REG_TAUNT:
    if( spadesHandle ) {
      gdk_input_remove( spadesHandle );
      spadesHandle = 0;
    }
  default:
    break;
  }

  /* Close out network connection */
  NetClose();
  
  /* Kill engine() process if one exists */
  if( gameInfo.gamePid ) {
    tmp = gameInfo.gamePid;
    gameInfo.gamePid = 0;
    kill( tmp, SIGUSR1 );
  }
   
  /* Dis/enable appropriate menu items*/
  menuItem = gtk_item_factory_get_widget( playArea->menu, "<main>/Game/New" );
  gtk_widget_set_sensitive( menuItem, TRUE);
  menuItem = gtk_item_factory_get_widget( playArea->menu, "<main>/Game/End" );
  gtk_widget_set_sensitive( menuItem, FALSE);

  DisplayStatusLine("Disconnected");

  
}


/**
 * Begin offline game (counterpart to Connect() )
 */
void PlayOffline( GtkWidget* pBar ) {

  GtkWidget *menuItem;
  int fd[2];
  
  /* Create socket pair */
  socketpair( PF_UNIX, SOCK_STREAM, 0, fd );
  gameState.spadesSock = fd[0];

  gameInfo.clientPids[0] = getpid();

  /* Engine process */
  if ( ( gameInfo.gamePid = fork() ) == 0) {
    gameInfo.gamePid = getpid();

    /* Fill gameInfo struct */
    gameInfo.gameNum = 0;
    gameInfo.players[0] = g_strdup( gameState.userName );
    gameInfo.players[1] = "Yakko";
    gameInfo.players[2] = "Wakko";
    gameInfo.players[3] = "Dot";
    gameInfo.playerSock[0] = fd[1];
    gameInfo.playerSock[1] = SOCK_COMP;
    gameInfo.playerSock[2] = SOCK_COMP;
    gameInfo.playerSock[3] = SOCK_COMP;  
    gameInfo.opt.bitOpt = options.bitOpt;
    gameInfo.opt.endGame = options.endGame;
    gameInfo.opt.minBid = options.minBid;
  
    engine();
    _exit(0);
  }
  else { /* We're still the player process */
    
    /* Enable Game/End menu item now that game is started*/
    menuItem = gtk_item_factory_get_widget( playArea->menu, "<main>/Game/End" );
    gtk_widget_set_sensitive( menuItem, TRUE);
    
    gameState.gameSegment = ST_GET_GAME;
    spadesHandle = gdk_input_add_full( gameState.spadesSock, GDK_INPUT_READ, 
				      ReadServerSocket, pBar, NULL);
  }  
}


/**
 * Read data from the Spades server 
 */
void ReadServerSocket( gpointer data, gint source, GdkInputCondition cond ) {

  Card playedCard;
  char *message;
  gint status;

  switch( gameState.gameSegment ) {

  case ST_GET_ID:
    if( GetPlayerId() == NET_FAIL ) {
      ConnectDialogReset( data );
    }
    else if( gameState.playerId < 0 ) {
      ConnectDialogReset( data );
      DisplayError( "Sorry, server full.  Try again later" );
    }
    else {
      if( gameState.playerId != 0 ) {
	DisplayWarning("Another player has already registered options for this game.\nYour options will be ignored" );
      }
      gtk_progress_bar_update( GTK_PROGRESS_BAR(data), 0.6 );
      gtk_progress_set_format_string( GTK_PROGRESS(data), "Waiting for other players to register...");
      UpdateGame();
    }
    break;

  case ST_GET_GAME:
    if( GetGame() == NET_FAIL ) {
      ConnectDialogReset( data );
    }
    else {
      gtk_progress_bar_update( GTK_PROGRESS_BAR(data), 0.8 );
      gtk_progress_set_format_string( GTK_PROGRESS(data), "Getting Player Names...");
      UpdateGame();
    }
    break;
	    
  case ST_GET_NAMES:
    if(  GetPlayers() == NET_FAIL ) {
      ConnectDialogReset( data );
    }
    else {
      gtk_progress_bar_update( GTK_PROGRESS_BAR(data), 1.0 );
      gtk_progress_set_format_string( GTK_PROGRESS(data), "Registering with TauntServer(tm)");
      UpdateGame();
      if( RegisterTaunt() >= 0 ) {
	gtk_widget_set_sensitive( tauntBox, TRUE );
	tauntHandle = gdk_input_add( gameState.tauntSock, GDK_INPUT_READ, 
				    ReadTauntSocket, NULL);
      }
      else {
	DisplayWarning("Could not connect to TauntServer.\nTaunting is disabled");
      }

      /* We're done registering now, so close the dialog box*/
      SafeCloseDialog();

      /* Next game State*/
      UpdateGame();
      
      /* Game is starting now so initialize and clear the screen*/
      GameInit();
      DisplayTable();
      DisplayTallys();
      DisplayScores();
    }
    break;

  case ST_GET_HAND:
    DisplayStatusLine("Getting Hand");
    if( GetHand() == NET_OK ) {
      DisplayHand();
      DisplayTable();
      DisplayScores();
      UpdateGame();
    }
    break;

  case ST_GET_BIDDER:
    if( GetLead() == NET_OK ) {
      DisplayStatusLine( "%s bids first.", gameState.players[gameState.lead]);
      UpdateGame();
    }
    break;
    
  case ST_GET_BIDS:
    if( gameState.curPlayer == gameState.playerId ) {
      DisplayError("Network error");
    }
    
    if( GetBid() == NET_OK ) {
      if ( gameState.bids[ gameState.curPlayer] == BID_KNEEL )
	  DisplayStatusLine( "%s bid nil", gameState.players[ gameState.curPlayer] );
      else 
	  DisplayStatusLine( "%s bid %d", gameState.players[ gameState.curPlayer],
			    gameState.bids[ gameState.curPlayer] );
      
      DisplayTallys();
      UpdateGame();
      if ( gameState.gameSegment == ST_GET_LEAD ) {
	DisplayScores();
      }
    }
    break;
    
  case ST_GET_LEAD:
    if( GetLead() == NET_OK ) {
      DisplayStatusLine( "%s leads", gameState.players[ gameState.lead]);
      UpdateGame();
    }
    break;
    
  case ST_GET_TRICK:
    if( gameState.curPlayer == gameState.playerId ) {
      DisplayError("Network error");
    }
    
    playedCard = GetPlayedCard();
    DisplayPlayedCard( playedCard, gameState.curPlayer, gameState.playerId );
    
    UpdateGame();
    break;
    
  case ST_GET_TALLY:
    if( GetTallys() == NET_OK ) {
      DisplayTallys();
      /* Set a timeout for when to clear screen */
      gtk_timeout_add( CLEAR_DELAY, TableClearTimer, NULL );
      /* Don't accept server input until we've cleared the table*/
      gdk_input_remove( spadesHandle );
    }
    break;
    
  case ST_GET_SCORE:
    if( GetScores() == NET_OK ) {
      /* Clear tallys and bids before next hand/game */
      gameState.bids[0] = BID_BLANK;
      gameState.bids[1] = BID_BLANK;
      gameState.bids[2] = BID_BLANK;
      gameState.bids[3] = BID_BLANK; 
      gameState.tallys[0] = 0;
      gameState.tallys[1] = 0; 
      gameState.tallys[2] = 0;
      gameState.tallys[3] = 0;
      DisplayScores();
      DisplayTallys();
      if( gameState.gameOver & GAME_OVER ) {
	DisplayStatusLine( "Game Over" );
	DisplayRecord();
      }
      UpdateGame();
    }
    break;
    
  case ST_END_GAME:
    if( GetNewGame() == NET_OK ) {
      UpdateGame();
      switch( gameState.sessionOver ) {
	
      case 1: /* Session is over, so disconnect */
	Disconnect( NULL, NULL);
	break;
      case 0: /* Let's play another round, eh? */
	GameInit();
	DisplayTable();
	DisplayTallys();
	DisplayScores();
	break;
      }
    }
    break;
  } /* switch( gameState.gameSegment )*/
}


/* 
 * Read data from the TauntServer(tm)
 */
void ReadTauntSocket( gpointer data, gint source, GdkInputCondition cond ) {
  
  int l;
  char* message;
  
  if( CheckReadInt( gameState.tauntSock, &l ) == NET_OK 
     && CheckReadString( gameState.tauntSock, &message ) == NET_OK ) {

    DisplayTaunt( message, l );
    free(message);
    
  }
}


/* 
 * Timer for clearing table after a trick
 */
static gint TableClearTimer( gpointer data ) {

  DisplayTable();
  UpdateGame();
  /* Don't accept server input until we've cleared the table*/
  spadesHandle = gdk_input_add( gameState.spadesSock, GDK_INPUT_READ, 
			  ReadServerSocket, NULL );
  
  return FALSE;
  
}

