/*
                      Object Parameter Macros

	Functions:

	void OPMReclaim()
        void OPMDeleteAll()
        int OPMIsGarbage(int opm_num)
        int OPMCreate(int type)
        int OPMCreateExplicit(int opm_num, int type)
        int OPMCreateWeapons(int opm_num, int total_weapons)
        void OPMRecycle(int opm_num)
 	int OPMGetTop()
 	int OPMGetByName(char *name, int type)

	int OPMModelObjectPtr(
	        xsw_object_struct *obj_ptr,
	        xsw_object_struct *opm_ptr
	)
 	int OPMModelObject(int object_num, int opm_num)

	---

 */

#include "swserv.h"


void OPMReclaim()
{
        int i, h;


        /* Get topmost opm. */
        h = OPMGetTop();
	if((h < 0) || (h >= total_opms))
            return;


        /* Do we need to reclaim memory? */
        if((total_opms - h - 1) <= OBJECT_ALLOCATE_AHEAD)
            return;


        /* Free all allocation from total_opms - 1 to h + 1. */
        for(i = total_opms - 1; i > h; i--)
        {
#ifdef DEBUG_MEM_FREE
if(opm[opm_num] != NULL)
    printf("OPM %i: Free'ed (reclaim).\n", opm_num);
#endif
	    DBDeleteObject(opm[i]);
            opm[i] = NULL;
        }


        /* Adjust total. */
        total_opms = h + 1;


        /* Reallocate OPM pointers. */
	if(total_opms > 0)
	{
            opm = DBAllocObjectPointers(
	        opm, total_opms
	    );
            if(opm == NULL)
            {
	        total_opms = 0;
                return;
            }
	}
	else
	{
#ifdef DEBUG_MEM_FREE
if(opm != NULL)
    printf("OPM pointers: Free'ed.\n");
#endif
	    free(opm);
	    opm = NULL;

	    total_opms = 0;
	}


        return;
}


/*
 *	Procedure to delete all OPMs.
 */
void OPMDeleteAll()
{
	int i;


#ifdef DEBUG_MEM_FREE
printf("OPMs: Deleting %i...\n", total_opms);
#endif

	for(i = 0; i < total_opms; i++)
	{
	    if(opm[i] == NULL)
		continue;

#ifdef DEBUG_MEM_FREE
if(opm[i] != NULL)
    printf("OPM %i: Free'ed\n", i);
#endif
	    DBDeleteObject(opm[i]);
	    opm[i] = NULL;
        }

#ifdef DEBUG_MEM_FREE
if(opm != NULL)
    printf("OPM pointers: Free'ed\n");
#endif
	free(opm);
	opm = NULL;

	total_opms = 0;


        return;
}



/*
 *	Checks if opm_num is allocated and non-garbage.
 */
int OPMIsGarbage(int opm_num)
{
        if((opm_num < 0) ||
	   (opm_num >= total_opms) ||
	   (opm == NULL)
	)
            return(-1);
        else if(opm[opm_num] == NULL)
            return(-1);
        else if(opm[opm_num]->type <= XSW_OBJ_TYPE_GARBAGE)
            return(1);
	else
            return(0);
}


/*
 *	Procedure to create an OPM of type type.
 */
int OPMCreate(int type)
{
	int opm_num, opm_rtn;
	char name[XSW_OBJ_NAME_MAX];


        /* Type may not be garbage. */
        if(type <= XSW_OBJ_TYPE_GARBAGE)
        {
            fprintf(stderr,
                "OPMCreate: Error: Useless request to create garbage.\n"
            );
            return(-1);
        }


        /* Look for available OPM already allocated. */
        opm_rtn = -1;
        for(opm_num = 0; opm_num < total_opms; opm_num++)
        {
	    if(opm[opm_num] == NULL)
		continue;

            /* Look for garbage OPMs. */
            if(opm[opm_num]->type == XSW_OBJ_TYPE_GARBAGE)
            {
                opm_rtn = opm_num;
                break;
            }
        }

        /* Did we find an OPM already allocated? */
        if( (opm_rtn > -1) && (opm_rtn < total_opms) )
        {
	    /* Reset OPM (free all allocated substructures) */
            UNVResetObject(opm[opm_num]);

	    sprintf(name, "OPM #%ld", opm_rtn);
	    name[XSW_OBJ_NAME_MAX - 1] = '\0';
	    strncpy(opm[opm_rtn]->name, name, XSW_OBJ_NAME_MAX);
	    opm[opm_rtn]->name[XSW_OBJ_NAME_MAX - 1] = '\0';

            opm[opm_rtn]->type = type;

            return(opm_rtn);
        }


	/* ************************************************************* */

        /* Not enough memory, can we allocate more? */
        if( (total_opms + OBJECT_ALLOCATE_AHEAD) > MAX_OPMS)
        {
            fprintf(stderr, "OPM maximum %i reached.\n", MAX_OPMS);
            return(-1);
        }


        /* Allocate more memory. */

        /* Allocate pointer array. */
        opm = DBAllocObjectPointers(
            opm,
            total_opms + OBJECT_ALLOCATE_AHEAD
	);
        if(opm == NULL)
        {
	    total_opms = 0;
            return(-1);
        }   


        /* Allocate each object. */
        for(opm_num = total_opms;
            opm_num < (total_opms + OBJECT_ALLOCATE_AHEAD);
            opm_num++
        )
        {
	    /* Allocate a new object. */
            opm[opm_num] = DBAllocObject();
            if(opm[opm_num] == NULL)
            {
                fprintf(stderr,
                    "OPM #%ld: Memory allocation error.\n",
                    opm_num
                );
                return(-1);
            }
        }


        /* New object will be the first object allocated. */
        opm_rtn = total_opms;


        /* Adjust global variable total_opms. */
	opm_num = total_opms;
        total_opms += OBJECT_ALLOCATE_AHEAD;

	/* Reset values on newly allocated objects. */
	while(opm_num < total_opms)
	{
	    UNVResetObject(opm[opm_num]);
	    opm_num++;
	}


        /* Set default values on the new object. */
        sprintf(name, "OPM #%ld", opm_rtn);
        name[XSW_OBJ_NAME_MAX - 1] = '\0';

        strncpy(opm[opm_rtn]->name, name, XSW_OBJ_NAME_MAX);            
	opm[opm_rtn]->name[XSW_OBJ_NAME_MAX - 1] = '\0';

        opm[opm_rtn]->type = type;


        /* Return the number of the newly created OPM. */
        return(opm_rtn);
}



/*
 *	Create an OPM explicitly.
 */
int OPMCreateExplicit(int opm_num, int type)
{
	int opm_count, opm_rtn;
	char name[XSW_OBJ_NAME_MAX];


        /* Type may not be garbage. */
        if(type <= XSW_OBJ_TYPE_GARBAGE)
        {
            fprintf(stderr,
         "OPMCreateExplicit(): Error: Useless request to create garbage.\n"
            );
            return(-1);
        }

	/* Make sure opm_num is valid. */
	else if(opm_num < 0)
        {
            fprintf(stderr,
           "OPMCreateExplicit(): Error: Request to create opm_num #%ld.\n",
		opm_num
            );
            return(-1);
	}
        
         
        /* *********************************************************** */
                
        /* If object_num already exists, recycle it. */
        if(!OPMIsGarbage(opm_num))
            OPMRecycle(opm_num);


        /* Allocate memory if opm_num is greater than total_opms. */
        if(opm_num >= total_opms)
        {
            /* Can we allocate more? */
            if((opm_num + OBJECT_ALLOCATE_AHEAD) >= MAX_OPMS)
            {
                fprintf(stderr, "OPM maximum %i reached.\n", MAX_OPMS);
                return(-1);
            }


            /* Allocate pointer array. */
            opm = DBAllocObjectPointers(
                opm,
		opm_num + OBJECT_ALLOCATE_AHEAD + 1
	    );
            if(opm == NULL)
            {
		total_opms = 0;
                return(-1);
            }


            /* Allocate each new OPM. */
            for(opm_count = total_opms;
                opm_count <= (opm_num + OBJECT_ALLOCATE_AHEAD);
                opm_count++
            )
            {
		/* Allocate memory for a new OPM. */
                opm[opm_count] = DBAllocObject();
                if(opm[opm_count] == NULL)
                {
		    total_opms = opm_count;
                    return(-1);
                }
            }
                
            /* Adjust global variable total_opms. */
            total_opms = opm_num + OBJECT_ALLOCATE_AHEAD + 1;
        }
             

	/* *************************************************** */                

        opm_rtn = opm_num;

        /* Set default values on the new OPM. */
        UNVResetObject(opm[opm_rtn]);

	sprintf(name, "OPM #%ld", opm_rtn);
	name[XSW_OBJ_NAME_MAX - 1] = '\0';

	strncpy(opm[opm_rtn]->name, name, XSW_OBJ_NAME_MAX);
	opm[opm_rtn]->name[XSW_OBJ_NAME_MAX - 1] = '\0';

        opm[opm_rtn]->type = type;
        

        /* Return 0 on no error, explicit OPM create does not
         * want OPM number.
         */
        return(0);
}


/*
 *	Allocate or deallocate weapons on opm.
 */
int OPMCreateWeapons(int opm_num, int total_weapons)
{
	int old_total_weapons;
	int weapon_num;


        /* Check if object_num is valid. */
        if(OPMIsGarbage(opm_num))
            return(-1);


        /* Sanitize total_weapons. */
        if(total_weapons > MAX_WEAPONS)
            total_weapons = MAX_WEAPONS;
	if(total_weapons < 0)
	    total_weapons = 0;


	/* Deallocate all weapons? */
	if(total_weapons == 0)
	{
	    for(weapon_num = 0;
		weapon_num < opm[opm_num]->total_weapons;
		weapon_num++
	    )
	    {
#ifdef DEBUG_MEM_FREE
if(opm[opm_num]->weapons[weapon_num] != NULL)
    printf("OPM %i weapon %i: Free'ed\n", opm_num, weapon_num);
#endif
		free(opm[opm_num]->weapons[weapon_num]);
		opm[opm_num]->weapons[weapon_num] = NULL;
	    }
#ifdef DEBUG_MEM_FREE
if(opm[opm_num]->weapons != NULL)
    printf("OPM %i weapon pointers: Free'ed\n", opm_num);
#endif
	    free(opm[opm_num]->weapons);
	    opm[opm_num]->weapons = NULL;

	    opm[opm_num]->total_weapons = 0;
	    opm[opm_num]->selected_weapon = -1;

            return(0);
	}


	/* ************************************************************ */

	/* Record previous total weapons on OPM. */
	old_total_weapons = opm[opm_num]->total_weapons;
	/* Sanitize old_total_weapons. */
	if(old_total_weapons > MAX_WEAPONS)
	    old_total_weapons = MAX_WEAPONS;
	else if(old_total_weapons < 0)
            old_total_weapons = 0;


	/* Nothing to do? */
	if(old_total_weapons == total_weapons)
	{
	    return(0);
	}
	/* Deallocate weapons? */
	else if(old_total_weapons > total_weapons)
	{
	    for(weapon_num = old_total_weapons - 1;
		weapon_num >= total_weapons;
		weapon_num--
	    )
	    {
#ifdef DEBUG_MEM_FREE
if(opm[opm_num]->weapons[weapon_num] != NULL)
    printf("OPM %i weapon %i: Free'ed\n", opm_num, weapon_num); 
#endif
		free(opm[opm_num]->weapons[weapon_num]);
		opm[opm_num]->weapons[weapon_num] = NULL;
	    }

	    opm[opm_num]->weapons =
		(xsw_weapons_struct **)realloc(
		    opm[opm_num]->weapons,
		    total_weapons * sizeof(xsw_weapons_struct *)
		);
	    if(opm[opm_num]->weapons == NULL)
	    {
		opm[opm_num]->total_weapons = 0;
		return(-1);
	    }

            /* Adjust the object's total_weapons to reflect changes. */
            opm[opm_num]->total_weapons = total_weapons;
	}
	/* Allocate more weapons. */
	else
	{
            opm[opm_num]->weapons =
                (xsw_weapons_struct **)realloc(
                    opm[opm_num]->weapons,
                    total_weapons * sizeof(xsw_weapons_struct *)
                );
            if(opm[opm_num]->weapons == NULL)
            {
                opm[opm_num]->total_weapons = 0;
		return(-1);
            }

            for(weapon_num = old_total_weapons;
                weapon_num < total_weapons;
                weapon_num++
            )
            {
		opm[opm_num]->weapons[weapon_num] =
		    (xsw_weapons_struct *)calloc(1,
			sizeof(xsw_weapons_struct)
		    );
		if(opm[opm_num]->weapons[weapon_num] == NULL)
		{
                    opm[opm_num]->total_weapons = weapon_num;
                    return(-1);
		}
            }

            /* Adjust the object's total_weapons to reflect changes. */
            opm[opm_num]->total_weapons = total_weapons;
	}


	return(0);
}


/*
 *	Recycles OPM.
 */
void OPMRecycle(int opm_num)
{
	/* Is opm_num valid? */
	if(OPMIsGarbage(opm_num)) return;


	/* Deallocates all substructures and set default values. */
	UNVResetObject(opm[opm_num]);

        /* Setting OPM type to garbage marks it as recycled. */
        opm[opm_num]->type = XSW_OBJ_TYPE_GARBAGE;


	return;
}


/*
 *	Get highest allocated and non-garbage OPM.
 */
int OPMGetTop()
{
        int current_opm, last_valid_opm;


	for(current_opm = 0, last_valid_opm = -1;
	    current_opm < total_opms;
	    current_opm++
	)
	{
	    if(opm[current_opm] == NULL) continue;
	    if(opm[current_opm]->type <= XSW_OBJ_TYPE_GARBAGE) continue;

	    last_valid_opm = current_opm;
	}


	return(last_valid_opm);
}


/*
 *	Get an OPM number by name, returns -1 on error.
 *
 *	type cannot be XSW_OBJ_TYPE_GARBAGE.
 *
 *	If type is -1, then all OPM types will be matched except
 *	XSW_OBJ_TYPE_GARBAGE.
 *
 *	Returns -1 on error, -2 on ambiguous.
 */
int OPMGetByName(char *name, int type)
{
	int opm_count, opm_found;
	int found_count;
	xsw_object_struct **opm_ptr;


	if(name == NULL)
	    return(-1);


	/* The specified type cannot be garbage. */
	if(type == XSW_OBJ_TYPE_GARBAGE)
	{
            fprintf(stderr,
 "OPMGetByName(): Useless request to match type garbage.\n"
            );
            return(-1);
	}


	/* Begin searching for an OPM. */
	opm_found = -1;
	found_count = 0;

	/* Match regardless of type? */
	if(type < 0)
	{
	    for(opm_count = 0, opm_ptr = opm;
	        opm_count < total_opms;
	        opm_count++, opm_ptr++
	    )
	    {
		if(*opm_ptr == NULL)
		    continue;

	        if((*opm_ptr)->type == XSW_OBJ_TYPE_GARBAGE)
		    continue;

		/* Check if name is the same. */
	        if(strstr((*opm_ptr)->name, name) != NULL)
	        {
		    opm_found = opm_count;
		    found_count++;
		}
	    }
	}
	/* Match by type. */
	else
	{
            for(opm_count = 0, opm_ptr = opm;
                opm_count < total_opms;
                opm_count++, opm_ptr++
            )
            {
                if(*opm_ptr == NULL)
                    continue;

                if((*opm_ptr)->type == XSW_OBJ_TYPE_GARBAGE)
                    continue;

		/* Types must match. */
		if((*opm_ptr)->type != type) continue;

		/* Check if name is the same. */
		if(strstr((*opm_ptr)->name, name) != NULL)
		{
                    opm_found = opm_count;
                    found_count++;
		}
	    }
	}

	/* Return -2 if more than one match was found. */
	if(found_count > 1)
	    return(-2);



	return(opm_found);
}


int OPMModelObjectPtr(
	xsw_object_struct *obj_ptr,
	xsw_object_struct *opm_ptr
)
{
	int i;


	if((obj_ptr == NULL) ||
           (opm_ptr == NULL)
	)
	    return(-1);


	/*   Free any dynamically allocated substructures on object
	 *   if they already allocated.
	 */
	UNVResetObject(obj_ptr);



	/*   Copy the OPM values to the object, and then allocate
	 *   and COPY any dynamically allocated substructres.
	 */

	/* Copy OPM values to object. */
	memcpy(
	    obj_ptr,	/* Destination. */
	    opm_ptr,	/* Source. */
	    sizeof(xsw_object_struct)
	);


	/* Note: Substructures on opm must be coppied to object. */

        /* Weapons. */  
        if((opm_ptr->weapons != NULL) && (opm_ptr->total_weapons > 0))
        {
            obj_ptr->weapons = (xsw_weapons_struct **)calloc(
		1,
                obj_ptr->total_weapons * sizeof(xsw_weapons_struct *)
            );
	    if(obj_ptr->weapons == NULL)
	    {
		obj_ptr->total_weapons = 0;
	    }

            for(i = 0; i < obj_ptr->total_weapons; i++)
            {
                obj_ptr->weapons[i] = (xsw_weapons_struct *)calloc(
                    1,
		    sizeof(xsw_weapons_struct)
		);
                if(obj_ptr->weapons[i] == NULL)
		{
		    obj_ptr->total_weapons = i;
		    break;
		}

                memcpy(
                    obj_ptr->weapons[i],    /* Destination. */
                    opm_ptr->weapons[i],    /* Source. */
                    sizeof(xsw_weapons_struct)
                );
            }
        }
	else
	{
	    obj_ptr->weapons = NULL;
	    obj_ptr->total_weapons = 0;
	}


        /* Tractored objects. */
        if((opm_ptr->tractored_object != NULL) &&
           (opm_ptr->total_tractored_objects > 0)
        )
        {    
            obj_ptr->tractored_object = (long *)calloc(
		1,
                opm_ptr->total_tractored_objects * sizeof(long)
            );
	    if(obj_ptr->tractored_object != NULL)
                memcpy( 
                    obj_ptr->tractored_object,	/* Destination. */
                    opm_ptr->tractored_object,	/* Source. */
                    opm_ptr->total_tractored_objects * sizeof(long)
                );
        }
        else
        {
            obj_ptr->tractored_object = NULL;
            obj_ptr->total_tractored_objects = 0;
        }


	/* Scores. */
	if(opm_ptr->score != NULL)
	{
	    obj_ptr->score = (xsw_score_struct *)calloc(
		1,
		sizeof(xsw_score_struct)
	    );
	    if(obj_ptr->score != NULL)
	    {
                memcpy(
                    obj_ptr->score,	/* Destination. */
                    opm_ptr->score,	/* Source. */
		    sizeof(xsw_score_struct)
	        );
	    }
	}
	else
	{
	    obj_ptr->score = NULL;
	}


        /* Economy. */
        if(opm_ptr->eco != NULL)
        {
            obj_ptr->eco = (xsw_ecodata_struct *)calloc(
                1,
                sizeof(xsw_ecodata_struct)
            );
            if(obj_ptr->eco != NULL)
            {
		/* Copy eco data structure. */
                memcpy(
                    obj_ptr->eco,	/* Dest. */
                    opm_ptr->eco,	/* Src. */
                    sizeof(xsw_ecodata_struct)
                );

		/* Allocate product pointers. */
		obj_ptr->eco->product = (xsw_ecoproduct_struct **)calloc(
		    1,
		    opm_ptr->eco->total_products * sizeof(xsw_ecoproduct_struct *)
                );
		if(obj_ptr->eco->product == NULL)
		{
		    obj_ptr->eco->total_products = 0;
		}
		else
		{
		    obj_ptr->eco->total_products = opm_ptr->eco->total_products;

		    /* Copy each product. */
		    for(i = 0; i < opm_ptr->eco->total_products; i++)
		    {
		        if(opm_ptr->eco->product[i] == NULL)
			    continue;

			obj_ptr->eco->product[i] = (xsw_ecoproduct_struct *)calloc(
                            1,
			    sizeof(xsw_ecoproduct_struct)
			);
			if(obj_ptr->eco->product == NULL)
			    continue;

			memcpy(
			    obj_ptr->eco->product[i],	/* Dest. */
			    opm_ptr->eco->product[i],	/* Src. */
			    sizeof(xsw_ecoproduct_struct)
                        );
		    }
		}
            }
        }
        else
        {
            obj_ptr->eco = NULL;
        }


	return(0);
}


int OPMModelObject(int object_num, int opm_num)
{
        xsw_object_struct *obj_ptr;
        xsw_object_struct *opm_ptr;


        if(DBIsObjectGarbage(object_num)) 
            return(-1);
	else
	    obj_ptr = xsw_object[object_num];

	if(OPMIsGarbage(opm_num))
            return(-1);
        else
            opm_ptr = opm[opm_num];


	return(OPMModelObjectPtr(obj_ptr, opm_ptr));
}
