/*
 * map.c: Loads map data from disk for XTux.
 * Copyright 1999 David Lawrence (philaw@camtech.net.au)
 * Last modified July 26.
 */

#include "header.h"
#include "entity.h"
#include "win.h"
#include "main.h"
#include "tile.h"
#include "map.h"

extern entity *player;

/* load_map(), load map->file_name, fills map structure. */
int load_map(struct map_t *map)
{

  FILE *mapf;
  unsigned char *ptr, tile;
  char layer[8], buf[256];
  int i,l,x,y;

  if( chdir("maps") < 0 )
    die("Can't find subdirectory 'maps'\n", 1);

#if VERBOSE
  fprintf(stderr, "Loading map \"%s\"\n", map->file_name);
#endif

  if( access(map->file_name, R_OK) < 0 ) {
    fprintf(stderr, "Error trying to access \"%s\"\n", map->file_name);
    die("Cannot read from map file!\n", 1);
  }

  if( !(mapf = fopen(map->file_name, "r")) ) {
    fprintf(stderr, "Error loading map \"%s\"!\n", map->file_name);
    fclose(mapf);
    return 0;
  }
#if VERBOSE
  else printf("Loaded map %s\n", map->file_name);
#endif

  /* Read info from top of map file */
  for(l=0 ; l<MAP_INFO_LINES ; l++) {

    /* Read a line, It doesn't like more than 80 chars! */
    if( !fgets(buf, 80, mapf) ) {
      fclose(mapf);
      fprintf(stderr, "Error reading line %d\n", l);
      return 0;
    }
    
    /* Ignore lines starting with '#' */
    if( *buf == '#' ) {
      l--; /* This line doesn't count */
      continue;
    }

    /* Handle line */
    switch( l ) {
    case 0:
      buf[ strlen( buf ) - 1 ] = '\0'; /* chop */ 
      strncpy( map->name, buf, 32 );
      *(map->name + 31) = '\0'; /* Zero the string. */
      break;

    case 1:
      if( sscanf(buf, "%d%d", &map->width, &map->height) != 2) {
	fprintf(stderr, "Error reading width & height!\n");
	fclose(mapf);
	return 0;
      }
      break;

    case 2:
      buf[ strlen(buf) - 1 ] = '\0';
      if( !strcmp( buf, "real") )
	map->tileset = REAL_TILESET;
      else if( !strcmp( buf, "tech" ) )
	map->tileset = TECH_TILESET;
      /* Check for other tilesets here */
#if VERBOSE
      printf("Using tileset: %d (%s)\n", map->tileset, buf);
#endif
      break;

    default:
      fprintf(stderr, "load_map:(Handle line) We shouldn't be here!\n");
    }
  }


  /* Read BASE & OBJECT LAYERS */
  for(i=0 ; i<3 ; i++) {
    if( (ptr = (unsigned char *)malloc(map->width*map->height * sizeof(char)))
       == NULL )
      die("Malloc() failed!!\n", 1);

    fgets(buf, 8, mapf);
    buf[strlen(buf) - 1] = '\0';

    switch( i ) {
    case 0:
      strcpy( layer, "BASE");
      break;
    case 1:
      strcpy( layer, "OBJECT");
      break;
    case 2:
      strcpy( layer, "ENTITY");
      break;
    default:
      die("Shouldn't be here! (load_map: Read Map data)\n", 1);
    }

    if( strcmp( buf, layer)) {
       printf("Error reading map layer: %d\nRead %s\n", i , buf);
       fclose(mapf);
       exit(1);
    }
#if VERBOSE
    else printf("Reading map layer: %s\n", buf);
#endif

    for( y=0 ; y<map->height; y++) {
      if( !fgets(buf, map->width+2 , mapf) ) {
	fprintf(stderr, "Error reading map data\n");
	fclose(mapf);
	return 0;
      }
      
      for( x=0 ; x<map->width; x++) {
	switch( i ) {
	case 0:
	  tile = base_tile( buf[x], map->tileset);
	  break;
	case 1:
	  tile = object_tile( buf[x], map->tileset);
	  break;
	case 2:
	  new_entity( buf[x], x, y );
	  break;
	default:
	  fclose(mapf);
	  fprintf(stderr, "i is %d", i);
	  die(" (a bad thing)\n", 1);
	}

	if( i < 2 )
	  *( ptr + y*map->width + x ) = tile;
      
      }

    } /* End of layer. */

    if( y != map->height )
      die("Error reading map\n", 1);

    if( i == 0 )
      map->base = ptr;
    else if( i == 1 )
      map->object = ptr;

  }

  ptr = NULL;
  chdir(".."); /* Go back to tux root directory */

  return 1; /* Ok. */

} /* load_map */


unsigned char base_tile( unsigned char c, int tileset) {


  /* Common between tilesets (GLOBALS) */
  switch( c ) {
  case ' ': return T_BLANK;
  case '*': return T_WHITE;
  }

  if( tileset == REAL_TILESET ) {

    switch( c ) {

      /* Walkable */
    case '.': return RB_GRASS;
    case ',': return RB_GRASS1;
    case 'C': return RB_CONC;
    case '#': return RB_PLANK;
    case ':': return RB_CARPET;
    case 'M': return RB_MARBLE;
    case 'm': return RB_MARBLE1;
    case 'S': return RB_SLATE;
    case 'i': return RB_ICE;
    case 'r': return RB_ROCKICE;
    case 's': return RB_SNOW;
    case 'f': return RB_SNOWF;

      /* Walls */
    case 'X': return RB_BRICK;
    case 'x': return RB_BRICK1;
    case '/': return RB_IWALLNW;
    case '\\': return RB_IWALLNE;
    case '(': return RB_IWALLSW;
    case ')': return RB_IWALLSE;
    case '|': return RB_IWALLV;
    case '-': return RB_IWALLH;
    case 'T': return RB_IWALLTN;
    case 'U': return RB_IWALLTS;
    case 'E': return RB_IWALLTE;
    case 'W': return RB_IWALLTW;
    case 'K' : return RB_KBENCH;

      /* Special */
    case '%': return RB_H2O;
    case '@': return RB_H2O1;

    default:
      fprintf(stderr, "Unknown tile \"%c\" ASCII %d\n", c, c);
      return 0; /* T_BLANK is drawn where unknown tiles are read */
    }

  } else if( tileset == TECH_TILESET ) {

    switch( c ) {

    case '.': return TB_LIGHTGREY;
    case '#': return TB_GREEN;
    case '_': return TB_GREENBASE;
    case '(': return TB_SD_CORNER;
    case 'B': return TB_SD_TOPICBSD;
    case 'C': return TB_SD_TOPICCRYPT;
    case 'G': return TB_SD_TOPICGAMES;
    case 'N': return TB_SD_TOPICGNU;
    case 'L': return TB_SD_TOPICLINUX;
    case 'M': return TB_SD_TOPICMS;
    case 'I': return TB_SD_TOPICNET;
    case 'Q': return TB_SD_TOPICQUAKE;
    case 'S': return TB_SD_TOPICSCIENCE;
    case 'P': return TB_SD_TOPICSPAM;

    default:
      fprintf(stderr, "Unknown tile \"%c\" ASCII %d\n", c, c);
      return 0; /* T_BLANK is drawn where unknown tiles are read */
    }

  }
    
  fprintf(stderr, "Fell off the end of base tile!\n");
  return 0; /* Shouldn't be here! */

} /* base_tile. */
  


unsigned char object_tile( unsigned char c, int tileset) {

  /* Common between tilesets (GLOBALS) */
  switch( c ) {
  case ' ':
  case '-':
    return O_NULL; /* these are used for spacing. */
  }

  /* Messages, independant of tileset */
  switch( c ) {
  case '0': return O_MSG0;
  case '1': return O_MSG1;
  case '2': return O_MSG2;
  case '3': return O_MSG3;
  case '4': return O_MSG4;
  case '5': return O_MSG5;
  case '6': return O_MSG6;
  case '7': return O_MSG7;
  case '8': return O_MSG8;
  case '9': return O_MSG9;
  }

  if( tileset == REAL_TILESET ) {
    
    switch( c ) {

    case '?': return RO_PIC;
    case '!': return RO_PIC1;
    case '@': return RO_PIC2;
    case 'Z': return RO_RED;
    case 'z': return RO_RUM;
    case 'V': return RO_TV;
    case 'R': return RO_FRIDGE;
    case 'r': return RO_FRIDGE1;
    case 'T': return RO_TABLE;
    case 'P': return RO_PC;
    case 'G': return RO_GPATHV;
    case 'g': return RO_GPATHH;
    case 'h': return RO_CHAIR;
    case 'M': return RO_MSN;
    case 'B': return RO_BED;
    case 'b': return RO_BED1;
    case 'D': return RO_DRAWERS;
    case 'd': return RO_DESK;
    case 'L': return RO_TOILET;
    case 't': return RO_TOWELS;
    case 'W': return RO_BATH;
    case 'w': return RO_BATH1;
    case 'S': return RO_SINK;
    case 's': return RO_SINK1;
    case 'K': return RO_KEND;
    case '^': return RO_TOASTER;
    case 'C': return RO_COUCH;
    case 'c': return RO_COUCH1;
    case 'F': return RO_TREE;
    case 'f': return RO_TREE1;
    case '+': return RO_WINDOW;
    case '#': return RO_BMARK;
    case 'Y': return RO_NOFEED;

    default:
      fprintf(stderr, "Unknown object \"%c\" ASCII %d\n", c, c);
      return O_NULL;
    }

  } else if( tileset == TECH_TILESET ) {

    switch( c ) {
    case 'T': return TO_GY_TXT_LN;
    case 't': return TO_GY_TXT;
    case 'G': return TO_GRN_TXT;
    case '=': return TO_GB_TOP;
    case '_': return TO_GB_BOTT;
    case '/': return TO_SD_TITLE;
    case '.': return TO_SD_TITLE1;

    default:
      fprintf(stderr, "Unknown tile \"%c\" ASCII %d\n", c, c);
      return O_NULL;
    }

  }

  fprintf(stderr, "Fell off the end of object tile!\n");
  return 0; /* We shouldn't be here! */

} /* object_tile */


int new_entity(char c, int x, int y)
{

  char ent_type[16];
  entity *ent;


  ent = NULL;

  switch(c) {
  case ' ':
  case '-':
    return 0; /* Nothing. */
  case 'S': /* Player's Spawn */
    strcpy(ent_type, "Player Spawn");
    ent = player;
    break;
  case 'D':
    strcpy(ent_type, "BSD");
    ent = create_entity(BSD);
    break;
  case 'G':
    strcpy(ent_type, "Gnu");
    ent = create_entity(GNU);
    break;
  case 'V':
    strcpy(ent_type, "VI");
    ent = create_entity(VI);
    break;
  case 'P': /* Peon */
    strcpy(ent_type, "Peon");
    ent = create_entity(MSCP);
    break;
  case 'C': /* Clippy */
    strcpy(ent_type, "Clippy");
    ent = create_entity(CLIPPY);
    break;
  case 'B': /* Bug */
    strcpy(ent_type, "Bug");
    ent = create_entity(BUG);
    break;
  case 'T': /* Troll */
    strcpy(ent_type, "Troll");
    ent = create_entity(TROLL);
    break;
  case 'F': /* Flamer */
    strcpy(ent_type, "Flamer");
    ent = create_entity(FLAMER);
    break;
  case 'L': /* coboL */
    strcpy(ent_type, "COBOL");
    ent = create_entity(COBOL);
    break;
  case 'R':
    strcpy(ent_type, "Bunny");
    ent = create_entity(BUNNY);
    break;
    /* ITEMS */
  case 'n':
    strcpy(ent_type, "Nuke Crate");
    ent = create_entity(NUKE_CRATE);
    break;
  case 'c':
    strcpy(ent_type, "Can");
    ent = create_entity(CAN);
    break;
  case 'd':
    strcpy(ent_type, "No Doze");
    ent = create_entity(NODOZE);
    break;
  case 'k': /* disK */
    strcpy(ent_type, "Disk");
    ent = create_entity(DISK);
    break;
  case 'p':
    strcpy(ent_type, "Spoon");
    ent = create_entity(SPOON);
    break;
  default:
    fprintf(stderr, "Unknown entity located @(%d,%d)! Read \"%c\"\n", x, y, c);
    break;
  }

  if(ent == NULL) {
    fprintf(stderr, "Entity@[%d,%d] is null!! read\"%c\")!\n", x,y,c);
    return 0;
  }

#if VERBOSE
  printf("%s@[%d,%d]\n", ent_type, x, y);
#endif

  /* Fill in rest of entity */
  ent->x = x * TILE_W;
  ent->y = y * TILE_H;
  ent->weapon = 1;

  return ent->type;

}

/* Gets the message from the .dat file */
void get_message(char *mapfile, int msg_num, char *msg_str)
{

  FILE *mapf;
  int num;
  char line[MAXMSGLEN];

#if VERBOSE
  fprintf(stderr, "Loading message from map \"%s\"\n", mapfile);
#endif

  /* Change mapfile from a .map to a .dat */
  strcpy( (mapfile + strlen(mapfile) - 3), "dat");

  if( chdir("maps") < 0 )
    die("Can't find subdirectory 'maps'\n", 1);  

  if( access(mapfile, R_OK) < 0 ) {
    fprintf(stderr, "Error trying to access \"%s\"\n", mapfile);
    die("Cannot read from map data file!\n", 1);
  }

  if( !(mapf = fopen(mapfile, "r")) ) {
    fprintf(stderr, "Error loading map \"%s\"!\n", mapfile);
    fclose(mapf);
    return;
  }
#if VERBOSE
  else fprintf(stderr, "Loaded map data file %s\n", mapfile);
#endif

  while( fgets(line, MAXMSGLEN, mapf) != NULL ) {
    
    if( sscanf(line, "%d", &num) == 1 && num == msg_num ) {
      /* Grab the string */
      if( fgets(msg_str, MAXMSGLEN, mapf) == NULL )
	die("Message string not found!\n", 1);
#if VERBOSE
      else
	fprintf(stderr, "Found message %d!, it's %s\n", num, msg_str);
#endif
      break;
    }
  }

  fclose(mapf);

  if( chdir("..") < 0 )
    die("Can't go up a directory!\n", 1);

}




