// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  DArray.C

  Expandable Direct Array. Well, like a list.

  Stephane Rehel
  November 11 1996
*/

#include <string.h>

#include "DArray.h"

/////////////////////////////////////////////////////////////////////////////

NAME::NAME( int _blockSize /* = 8 */ )
{
  if( _blockSize <= 0 )
    _blockSize= 0;

  // get the biggest bse so that 2^bse be lower or equal to _blockSize
  // ex: 7 -> 4
  //     8 -> 8
  //     9 -> 8
  int bse= 0;
  while( _blockSize >= (1<<bse) )
    ++bse;
  --bse;

  blockSizeExp= bse;
  blockSize= 1 << blockSizeExp;
  blockSizeAnd= blockSize - 1;

  nBlocks= blockSize;
  blocks= new ELEMENT* [ nBlocks ];
  for( int i= 0; i < nBlocks; ++i )
    blocks[i]= 0;

  nElements= 0;
  maxElements= blockSize * nBlocks;
}

/////////////////////////////////////////////////////////////////////////////

NAME::~NAME()
{
  clear();
  delete blocks;
  blocks= 0;
  nBlocks= 0;
  maxElements= 0;
}

/////////////////////////////////////////////////////////////////////////////

void NAME::clear()
{
  if( nElements == 0 )
    return; // speed up

  for( register int i= 0; i < nBlocks; ++i )
    {
    ELEMENT* one_block= blocks[i];
    if( one_block != 0 )
      {
      // call destructor for each element

      delete [] one_block;
      blocks[i]= 0;
      }
    }

  nElements= 0;
}

/////////////////////////////////////////////////////////////////////////////

void NAME::expand()
{
  if( nElements < maxElements )
    {
    int block_index= nElements >> blockSizeExp;
    if( blocks[block_index] == 0 )
      {
      // Add a new block
      blocks[block_index]= new ELEMENT [ blockSize ];
      }
    }
   else
    {
    // Expand nBlocks: +25%
    int new_nBlocks= nBlocks + nBlocks / 4;
    ELEMENT** new_blocks= new ELEMENT* [ new_nBlocks ];
    if( blocks != 0 )
      memcpy( (void*) new_blocks, (void*) blocks, nBlocks * sizeof(void*) );
    memset( (void*) (new_blocks+nBlocks), 0,
            sizeof(void*) * (new_nBlocks-nBlocks) );

    delete blocks;
    blocks= new_blocks;
    nBlocks= new_nBlocks;

    maxElements= nBlocks * blockSize;
    }
}

/////////////////////////////////////////////////////////////////////////////

ELEMENT& NAME::append()
{
  if( nElements >= maxElements )
    expand();

  register int block_index= nElements >> blockSizeExp;
  if( blocks[block_index] == 0 )
    expand();

  ELEMENT& e= blocks[ block_index ][ nElements & blockSizeAnd ];
  ++nElements;

  return e;
}

/////////////////////////////////////////////////////////////////////////////

void NAME::remove( int i )
{
  if( i < 1 || i > nElements )
    return;

  if( i == nElements )
    {
    // speed up for last element
    --nElements;
    return;
    }

  --i;
  int block_index= i >> blockSizeExp;
  int index= i && blockSizeAnd;
  if( index < blockSize-1 )
    memmove( (void*) &blocks[block_index][index],
             (void*) &blocks[block_index][index+1],
             (blockSize-1-index) * sizeof(ELEMENT) );

  for(;;)
    {
    ++block_index;
    if( block_index >= nBlocks )
      break;
    if( blocks[block_index] == 0 )
      break;
    memcpy( (void*) &blocks[block_index-1][blockSize-1],
            (void*) &blocks[block_index][0],
            sizeof(ELEMENT) );
    memmove( (void*) &blocks[block_index][0],
             (void*) &blocks[block_index][1],
             (blockSize-1) * sizeof(ELEMENT) );
    }

  --nElements;
}

/////////////////////////////////////////////////////////////////////////////

void NAME::alloc( int size )
{
  int delta= size - nElements;

  while( delta > 0 )
    {
    append();
    --delta;
    }
}

/////////////////////////////////////////////////////////////////////////////

int NAME::getSizeOf() const
{
  int s= sizeof(*this) + nBlocks * sizeof(blocks[0]);

  int n= 0;
  for( int i= 0; i < nBlocks; ++i )
    if( blocks[i] != 0 )
      ++n;
  s+= n * (blockSize * sizeof(blocks[0][0]));

  return s;
}

/////////////////////////////////////////////////////////////////////////////
