/***************************************************************************
                          ractdraw.cpp  -  description                              
                             -------------------                                         
    begin                : Wed Apr 28 1999                                           
    copyright            : (C) 1999 by Andreas Mustun                         
    email                : mustun@ribbonsoft.com                                     
 ***************************************************************************/


#include <qapplication.h>
#include <qbitmap.h>
#include <qcursor.h>

#include "rappwin.h"
#include "ractdraw.h"
#include "rbehaviordef.h"
#include "rfontdialog.h"
#include "rgraphic.h"
#include "rhatchdialog.h"
#include "rinputpanel.h"
#include "rconfig.h"
#include "rlog.h"
#include "rmath.h"
#include "rsnap.h"
#include "rstatuspanel.h"

#include "xpm/arcdir1.xpm"
#include "xpm/arcdir2.xpm"


/*! \class RActDraw
    \brief Action handling for drawing functions
    \author Andreas Mustun
*/



/*! Constructor
*/
RActDraw::RActDraw()
{

}



/*! Destructor
*/
RActDraw::~RActDraw()
{
}



/*! Serve an event to the corresponding function
    \param _action The Action (e.g. ACT_LINES_NORMAL)
    \param _myREvent The event (e.g. REVENT_BEGIN)
*/
void
RActDraw::serveEvent(int _action, int _myREvent )
{
  if(_action>=ACT_DRAW_FIRST &&
     _action<=ACT_DRAW_LAST     ) {

    switch(_action) {

      case ACT_POINTS_SINGLE:
        pointsSingle(_myREvent);
        break;

      case ACT_LINES_NORMAL:
        linesNormal(_myREvent);
        break;

      case ACT_LINES_HOR:
        linesHor(_myREvent);
        break;

      case ACT_LINES_VER:
        linesVer(_myREvent);
        break;

      case ACT_LINES_ANGLE:
        linesAngle(_myREvent);
        break;

      case ACT_LINES_RECT:
        linesRect(_myREvent);
        break;

      case ACT_LINES_PARA:
        linesPara(_myREvent);
        break;

      case ACT_LINES_BISECTOR:
        linesBisector(_myREvent);
        break;

      case ACT_LINES_TAN1:
        linesTan1(_myREvent);
        break;

      case ACT_LINES_TAN2:
        linesTan2(_myREvent);
        break;

      case ACT_LINES_ORTHO:
        linesOrtho(_myREvent);
        break;

      case ACT_LINES_ORTHO2:
        linesOrtho(_myREvent, true);
        break;

      case ACT_ARCS_CRAA:
        arcsCRAA(_myREvent);
        break;

      case ACT_ARCS_PPP:
        arcsPPP(_myREvent);
        break;

      case ACT_ARCS_PARA:
        arcsPara(_myREvent);
        break;

      case ACT_CIRCLES_CR:
        circlesCR(_myREvent);
        break;

      case ACT_TEXTS_NORMAL:
        textsNormal(_myREvent);
        break;

      case ACT_DIMS_PARA:
        dimsPara(_myREvent);
        break;

      case ACT_DIMS_HOR:
        dimsHor(_myREvent);
        break;

      case ACT_DIMS_VER:
        dimsVer(_myREvent);
        break;

      case ACT_DIMS_ANGLE:
        dimsAngle(_myREvent);
        break;

      case ACT_DIMS_DIAM:
        dimsDiam(_myREvent);
        break;

      case ACT_DIMS_RAD:
        dimsRad(_myREvent);
        break;

      case ACT_DIMS_ARROW:
        dimsArrow(_myREvent);
        break;
        
      case ACT_HATCH:
        hatching(_myREvent);
        break;

      default:
        break;
    }
  }
}



/*! Let the user create single points
    <p>States:
    <ul>
      <li>1: Set point
    </ul>
*/
void
RActDraw::pointsSingle(int _myREvent)
{
  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(269), RMES(270), RMES(21));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          int el=currentDoc()->addPoint(px, py);
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();
        }
        snapper()->snapPoint();
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        statusPanel()->clear();
        currentDoc()->setCursor(arrowCursor);
      }
      break;

    default:
      break;
  }
}



/*! Lets the user create normal lines (two points)
    <p>States:
    <ul>
      <li>1: Set 1st point
      <li>2: Set next point
    </ul>
*/
void
RActDraw::linesNormal(int _myREvent)
{
  static float x1=DEF_AREAMAX;  // Coordinates of first (last) point
  static float y1=DEF_AREAMAX;  //

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(271), RMES(270), RMES(22));
      break;

    case REVENT_MOUSEMOVE:

      // Move before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting next point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_LINE, x1, y1, px, py)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set 1st point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&x1, &y1, true)) {
          currentDoc()->moveRelZero(x1, y1);
          statusPanel()->setLeftButtonStatus(RMES(275));
          currentDoc()->setCurrentState(2);
        }
      }

      // Set next point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();
          int el=currentDoc()->addLine(x1, y1, px, py);
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();
          x1=px;
          y1=py;
        }
        snapper()->snapPoint();
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting of first point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(271));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user create horizontal lines
    <p>States:
    <ul>
      <li>1: Set point
    </ul>
*/
void
RActDraw::linesHor(int _myREvent)
{
  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(458), RMES(270), RMES(455));
      break;

    case REVENT_MOUSEMOVE:

      // Move for setting the point:
      //
      if(currentDoc()->getCurrentState()==1) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_LINE,
                                             currentDoc()->screenToRealX(0), py,
                                             currentDoc()->screenToRealX(currentDoc()->width()), py)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set point:
      //
      if(currentDoc()->getCurrentState()==1) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();
          int el=currentDoc()->addLine(currentDoc()->screenToRealX(0), py,
                                       currentDoc()->screenToRealX(currentDoc()->width()), py);
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();
        }
        snapper()->snapPoint();
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      break;

    default:
      break;
  }
}



/*! Lets the user create vertical lines
    <p>States:
    <ul>
      <li>1: Set point
    </ul>
*/
void
RActDraw::linesVer(int _myREvent)
{
  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(458), RMES(270), RMES(456));
      break;

    case REVENT_MOUSEMOVE:

      // Move for setting the point:
      //
      if(currentDoc()->getCurrentState()==1) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_LINE,
                                             px, currentDoc()->screenToRealY(0),
                                             px, currentDoc()->screenToRealY(currentDoc()->height()))) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set point:
      //
      if(currentDoc()->getCurrentState()==1) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();
          int el=currentDoc()->addLine(px, currentDoc()->screenToRealY(0),
                                       px, currentDoc()->screenToRealY(currentDoc()->height()));
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();
        }
        snapper()->snapPoint();
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      break;

    default:
      break;
  }
}




/*! Lets the user create lines with given angles
    <p>States:
    <ul>
      <li>1: Set point
    </ul>
*/
void
RActDraw::linesAngle(int _myREvent)
{
  static float angle=45.0;

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(458), RMES(270), RMES(457));
      inputPanel()->reset();
      inputPanel()->addLineEdit(RMES(459), false);
      inputPanel()->show();
      inputPanel()->setValue(0, angle);
      break;

    case REVENT_MOUSEMOVE:

      // Move for setting the point:
      //
      if(currentDoc()->getCurrentState()==1) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          angle=inputPanel()->getValue(0);
          float halfLength=currentDoc()->screenToRealX(currentDoc()->width());
          if(currentDoc()->setPreviewElement(T_LINE,
                                             px - cos(angle/ARAD) * halfLength,
                                             py - sin(angle/ARAD) * halfLength,
                                             px + cos(angle/ARAD) * halfLength,
                                             py + sin(angle/ARAD) * halfLength)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set point:
      //
      if(currentDoc()->getCurrentState()==1) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();
          angle=inputPanel()->getValue(0);
          float halfLength=currentDoc()->screenToRealX(currentDoc()->width());
          int el=currentDoc()->addLine(px - cos(angle/ARAD) * halfLength,
                                       py - sin(angle/ARAD) * halfLength,
                                       px + cos(angle/ARAD) * halfLength,
                                       py + sin(angle/ARAD) * halfLength);
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();
        }
        snapper()->snapPoint();
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        inputPanel()->reset();
        inputPanel()->hide();
      }
      break;

    default:
      break;
  }
}




/*! Lets the user create rectangles
    <p>States:
    <ul>
      <li>1: Set 1st edge
      <li>2: Set 2nd edge
    </ul>
*/
void
RActDraw::linesRect(int _myREvent)
{
  static float x1=DEF_AREAMAX;  // Coordinates of first point
  static float y1=DEF_AREAMAX;  //

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(272), RMES(270), RMES(23));
      break;

    case REVENT_MOUSEMOVE:

      // Move before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting next point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_RECT, x1, y1, px, py)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set 1st point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&x1, &y1, true)) {
          currentDoc()->moveRelZero(x1, y1);
          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(RMES(276));
        }
      }

      // Set 2nd point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();
          int el=currentDoc()->addLine(x1, y1, px, y1);
          currentDoc()->drawElement(el);
          el=currentDoc()->addLine(px, y1, px, py);
          currentDoc()->drawElement(el);
          el=currentDoc()->addLine(px, py, x1, py);
          currentDoc()->drawElement(el);
          el=currentDoc()->addLine(x1, py, x1, y1);
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();
          x1=DEF_AREAMAX;
          y1=DEF_AREAMAX;
          currentDoc()->setCurrentState(1);
          statusPanel()->setLeftButtonStatus(RMES(272));
        }
        snapper()->snapPoint();
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting of first point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(272));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user create parallels
    <p>States:
    <ul>
      <li>1: Set parallel
    </ul>
*/
void
RActDraw::linesPara(int _myREvent)
{
  static RElement prevEl;
  static float paraDistance=10.0;

  switch(_myREvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(RMES(273), RMES(270), RMES(24));
      inputPanel()->reset();
      inputPanel()->addLineEdit(RMES(300), false);
      inputPanel()->show();
      inputPanel()->setValue(0, paraDistance);
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);

        // Preview parallel:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          // Get distance from input panel:
          //
          paraDistance = inputPanel()->getValue(0);

          prevEl.reset();
          snapper()->getSnappedElement()->getParallel(&prevEl,
                                                      currentDoc()->getRealMousePosX(),
                                                      currentDoc()->getRealMousePosY(),
                                                      paraDistance);

          if(currentDoc()->setPreviewElement(&prevEl)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        // Draw parallel:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          // Get distance from input panel:
          //
          paraDistance = inputPanel()->getValue(0);

          prevEl.reset();
          snapper()->getSnappedElement()->getParallel(&prevEl,
                                                      currentDoc()->getRealMousePosX(),
                                                      currentDoc()->getRealMousePosY(),
                                                      paraDistance);
          if(prevEl.getFlag(E_VISIBLE)) {
            snapper()->stop();
            int el=currentDoc()->addElement(&prevEl);
            currentDoc()->drawElement(el);
            currentDoc()->terminateAction();

            currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);

            linesPara(REVENT_MOUSEMOVE);
          }
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        inputPanel()->reset();
        inputPanel()->hide();
      }
      break;

    default:
      break;
  }
}



/*! Lets the user create bisectors
    <p>States:
    <ul>
      <li>1: Set 1st element
      <li>2: Set 2nd element
    </ul>
*/
void
RActDraw::linesBisector(int _myREvent)
{
  static RElement* leg1=0;
  static RElement  prevEl;
  static float biLength=50.0;

  switch(_myREvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      currentDoc()->setHighlightFilter(T_LINE);
      statusPanel()->setActionStatus(RMES(274), RMES(270), RMES(25));
      inputPanel()->reset();
      inputPanel()->addLineEdit(RMES(301), false);
      inputPanel()->show();
      inputPanel()->setValue(0, biLength);
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_LINE);
      }

      else if(currentDoc()->getCurrentState()==2) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_LINE, leg1);

        // Preview bisector:
        //
        if(snapper() &&
           snapper()->getSnappedElement() &&
           leg1) {

          // Get distance from input panel:
          //
          biLength = inputPanel()->getValue(0);

          prevEl.reset();
          snapper()->getSnappedElement()->getBisector(&prevEl,
                                                      leg1,
                                                      currentDoc()->getRealMousePosX(),
                                                      currentDoc()->getRealMousePosY(),
                                                      biLength);

          if(currentDoc()->setPreviewElement(&prevEl)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {

        // Set first leg:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {
          leg1 = snapper()->getSnappedElement();
          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(RMES(277));
        }
      }
      else if(currentDoc()->getCurrentState()==2) {
        // Draw bisector:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          // Get distance from input panel:
          //
          biLength = inputPanel()->getValue(0);

          prevEl.reset();
          snapper()->getSnappedElement()->getBisector(&prevEl,
                                                      leg1,
                                                      currentDoc()->getRealMousePosX(),
                                                      currentDoc()->getRealMousePosY(),
                                                      biLength);
          if(prevEl.getFlag(E_VISIBLE)) {
            snapper()->stop();
            int el=currentDoc()->addElement(&prevEl);
            currentDoc()->drawElement(el);
            currentDoc()->terminateAction();

            currentDoc()->setCurrentState(1);
            linesBisector(REVENT_MOUSEMOVE);
            currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
            statusPanel()->setLeftButtonStatus(RMES(274));
          }
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        inputPanel()->reset();
        inputPanel()->hide();
      }
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
        statusPanel()->setLeftButtonStatus(RMES(274));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user create tangents (point-circle)
    <p>States:
    <ul>
      <li>1: Set point
      <li>2: Set circle
    </ul>
*/
void
RActDraw::linesTan1(int _myREvent)
{
  static RElement  prevEl;    // previewed tangent
  static float px;            // Coordinate of point
  static float py;            //

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(278), RMES(270), RMES(26));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      else if(currentDoc()->getCurrentState()==2) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_CIRCULAR);

        // Preview tangent:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          prevEl.reset();
          snapper()->getSnappedElement()->getTangent1(&prevEl,
                                                      px, py,
                                                      currentDoc()->getRealMousePosX(),
                                                      currentDoc()->getRealMousePosY());

          if(currentDoc()->setPreviewElement(&prevEl)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set the point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()) {
          snapper()->snapPoint(&px, &py, true);
          currentDoc()->moveRelZero(px, py);
        }
        currentDoc()->setCurrentState(2);
        statusPanel()->setLeftButtonStatus(RMES(279));
      }
      else if(currentDoc()->getCurrentState()==2) {
        // Draw tangent:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {
          prevEl.reset();
          snapper()->getSnappedElement()->getTangent1(&prevEl,
                                                      px, py,
                                                      currentDoc()->getRealMousePosX(),
                                                      currentDoc()->getRealMousePosY());
          if(prevEl.getFlag(E_VISIBLE)) {
            snapper()->stop();
            int el=currentDoc()->addElement(&prevEl);
            currentDoc()->drawElement(el);
            currentDoc()->terminateAction();

            currentDoc()->setCurrentState(1);
            linesTan1(REVENT_MOUSEMOVE);
            currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
            statusPanel()->setLeftButtonStatus(RMES(278));
          }
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
        statusPanel()->setLeftButtonStatus(RMES(278));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user create tangents (circle-circle)
    <p>States:
    <ul>
      <li>1: Set 1st circle
      <li>2: Set 2nd circle
    </ul>
*/
void
RActDraw::linesTan2(int _myREvent)
{
  static RElement  prevEl;    // previewed tangent
  static RElement  circle1;   // Coordinate of point

  switch(_myREvent) {

    case REVENT_BEGIN:
      //RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(280), RMES(270), RMES(27));
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_CIRCULAR);
        snapper()->snapElement();
      }

      else if(currentDoc()->getCurrentState()==2) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_CIRCULAR);

        // Preview tangent:
        //
        if(snapper() &&
           snapper()->getSnappedElement()) {

          prevEl.reset();

          snapper()->getSnappedElement()->getTangent2(&prevEl,
                                                      &circle1,
                                                      currentDoc()->getRealMousePosX(),
                                                      currentDoc()->getRealMousePosY());

          if(currentDoc()->setPreviewElement(&prevEl)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set the 1st element:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()) {
          snapper()->snapElement(true, T_CIRCULAR);
          circle1.getMeasuresFrom(snapper()->getSnappedElement());
        }
        currentDoc()->setCurrentState(2);
        statusPanel()->setLeftButtonStatus(RMES(281));
      }
      else if(currentDoc()->getCurrentState()==2) {
        // Draw tangent 2:
        //
        if(snapper() &&
           snapper()->snapElement(true, T_CIRCULAR, &circle1)) {
          prevEl.reset();
          snapper()->getSnappedElement()->getTangent2(&prevEl,
                                                      &circle1,
                                                      currentDoc()->getRealMousePosX(),
                                                      currentDoc()->getRealMousePosY());
          if(prevEl.getFlag(E_VISIBLE)) {
            snapper()->stop();
            int el=currentDoc()->addElement(&prevEl);
            currentDoc()->drawElement(el);
            currentDoc()->terminateAction();

            currentDoc()->setCurrentState(1);
            linesTan2(REVENT_MOUSEMOVE);
            currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
            statusPanel()->setLeftButtonStatus(RMES(280));
          }
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
        statusPanel()->setLeftButtonStatus(RMES(280));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user create orthogonal lines
    <p>States:
    <ul>
      <li>1: Set base element
      <li>2: Set pos
    </ul>
*/
void
RActDraw::linesOrtho(int _myREvent, bool _useAngle)
{
  static RElement baseEl;
  static RElement prevEl;
  static float angle=90.0;

  if(_useAngle) angle=90.0;

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      statusPanel()->setActionStatus(RMES(479), RMES(270), RMES(476));
      if(_useAngle) {
        inputPanel()->reset();
        inputPanel()->addLineEdit(RMES(314), false);
        inputPanel()->show();
        inputPanel()->setValue(0, angle);
      }
      break;

    case REVENT_MOUSEMOVE:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        snapper()->snapElement();
      }

      else if(currentDoc()->getCurrentState()==2) {
        // Preview orthogonal line:
        //
        float px, py;
        if(snapper() && snapper()->snapPoint(&px, &py)) {

          if(_useAngle) {
            // Get angle from input panel:
            //
            angle = inputPanel()->getValue(0);
          }

          prevEl.reset();
          baseEl.getOrtho(&prevEl,
                          px, py,
                          angle,
                          currentDoc()->width()/currentDoc()->getZoom());

          if(currentDoc()->setPreviewElement(&prevEl)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()) {
          snapper()->snapElement(true);
          baseEl.getMeasuresFrom(snapper()->getSnappedElement());
        }
        currentDoc()->setCurrentState(2);
        statusPanel()->setLeftButtonStatus(RMES(458));
      }

      else if(currentDoc()->getCurrentState()==2) {
        // Draw orthogonal line:
        //
        float px, py;
        if(snapper() && snapper()->snapPoint(&px, &py)) {

          prevEl.reset();

          if(_useAngle) {
            // Get angle from input panel:
            //
            angle = inputPanel()->getValue(0);
          }

          snapper()->getSnappedElement()->getOrtho(&prevEl,
                                                   px, py,
                                                   angle,
                                                   currentDoc()->width()/currentDoc()->getZoom());
          if(prevEl.getFlag(E_VISIBLE)) {
            snapper()->stop();
            int el=currentDoc()->addElement(&prevEl);
            currentDoc()->drawElement(el);
            currentDoc()->terminateAction();

            currentDoc()->setCurrentState(1);
            linesOrtho(REVENT_MOUSEMOVE, _useAngle);
            currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
            statusPanel()->setLeftButtonStatus(RMES(479));
          }
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        if(_useAngle) {
          inputPanel()->reset();
          inputPanel()->hide();
        }
      }
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
        statusPanel()->setLeftButtonStatus(RMES(479));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user create arcs (center, radius, angles)
    <p>States:
    <ul>
      <li>1: Set center
      <li>2: Set radius
      <li>3: Set angle 1
      <li>4: Set angle 2
    </ul>
*/
void
RActDraw::arcsCRAA(int _myREvent)
{
  static float cx=DEF_AREAMAX;  // Coordinates of center
  static float cy=DEF_AREAMAX;  //
  static float cr=DEF_AREAMAX;  // Radius
  static float a1=DEF_AREAMAX;  // Angle 1
  static float a2=DEF_AREAMAX;  // Angle 2
  static bool  direction=false; // Arc direction (0=pos/1=neg)

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(282), RMES(270), RMES(28));
      inputPanel()->reset();
      inputPanel()->newGroup(RMES(303));
      inputPanel()->addRadioButton(QPixmap(arcdir1_xpm), RMES(305), 0, 2);
      inputPanel()->addRadioButton(QPixmap(arcdir2_xpm), RMES(304), 1, 2);
      inputPanel()->endGroup();
      inputPanel()->show();
      inputPanel()->setCheckedRadioButton(direction);
      break;

    case REVENT_MOUSEMOVE:

      // Move before center is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting radius:
      //
      else if(currentDoc()->getCurrentState()==2) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_CIRCLE, cx, cy, mtGetDistance(cx,cy, px,py))) {
            currentDoc()->drawPreviewElement();
          }
        }
      }

      // Move for setting angle 1:
      //
      else if(currentDoc()->getCurrentState()==3) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          direction= (bool)inputPanel()->isRadioButtonChecked(1);
          float angleOffset;
          if(!direction) angleOffset=45.0;
          else           angleOffset=-45.0;
          if(currentDoc()->setPreviewElement(T_ARC,
                                             cx, cy,
                                             cr,
                                             mtGetAngle(cx,cy, px,py),
                                             mtCorrAngle(mtGetAngle(cx,cy, px,py)+angleOffset),
                                             0.0, 0.0,
                                             direction ? E_REVERSED : 0) ) {
            currentDoc()->drawPreviewElement();
          }
        }
      }

      // Move for setting angle 2:
      //
      else if(currentDoc()->getCurrentState()==4) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          direction= (bool)inputPanel()->isRadioButtonChecked(1);
          if(currentDoc()->setPreviewElement(T_ARC,
                                             cx, cy,
                                             cr,
                                             a1,
                                             mtGetAngle(cx,cy, px,py),
                                             0.0, 0.0,
                                             direction ? E_REVERSED : 0) ) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set center:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx, &cy, true)) {
          currentDoc()->moveRelZero(cx, cy);
          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(RMES(283));
        }
      }

      // Set radius:
      //
      else if(currentDoc()->getCurrentState()==2) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();
          cr = mtGetDistance(cx, cy, px, py);
          currentDoc()->setCurrentState(3);
          statusPanel()->setLeftButtonStatus(RMES(284));
        }
        snapper()->snapPoint();
      }

      // Set angle 1:
      //
      else if(currentDoc()->getCurrentState()==3) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();
          a1=mtGetAngle(cx,cy, px,py);
          currentDoc()->setCurrentState(4);
          statusPanel()->setLeftButtonStatus(RMES(285));
        }
        snapper()->snapPoint();
      }

      // Set angle 2 / construct the circle arc itself:
      //
      else if(currentDoc()->getCurrentState()==4) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();
          a2=mtGetAngle(cx,cy, px,py);
          direction= (bool)inputPanel()->isRadioButtonChecked(1);
          int el=currentDoc()->addArc(cx, cy, cr, a1, a2, direction);
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();

          currentDoc()->setCurrentState(1);
          statusPanel()->setLeftButtonStatus(RMES(282));
        }
        snapper()->snapPoint();
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before center is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
        inputPanel()->reset();
        inputPanel()->hide();
      }

      // Go back to setting of center:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        statusPanel()->setLeftButtonStatus(RMES(282));
        snapper()->snapPoint();
      }

      // Go back to setting of radius:
      //
      else if(currentDoc()->getCurrentState()==3) {
        snapper()->stop();
        currentDoc()->setCurrentState(2);
        statusPanel()->setLeftButtonStatus(RMES(283));
        snapper()->snapPoint();
      }

      // Go back to setting of angle1:
      //
      else if(currentDoc()->getCurrentState()==4) {
        snapper()->stop();
        currentDoc()->setCurrentState(3);
        statusPanel()->setLeftButtonStatus(RMES(284));
        snapper()->snapPoint();
      }
      break;

    default:
      break;
  }

}



/*! Lets the user create arcs (3 points)
    <p>States:
    <ul>
      <li>1: Set point 1
      <li>2: Set point 2
      <li>3: Set point 3
    </ul>
*/
void
RActDraw::arcsPPP(int _myREvent)
{
  static float x1=DEF_AREAMAX;  // Coordinates of first point
  static float y1=DEF_AREAMAX;  //
  static float x2=DEF_AREAMAX;  // Coordinates of second point
  static float y2=DEF_AREAMAX;  //

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(286), RMES(270), RMES(29));
      break;

    case REVENT_MOUSEMOVE:

      // Move before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting second point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->snapPoint();
      }

      // Move for setting third point:
      //
      else if(currentDoc()->getCurrentState()==3) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          RElement dumEl;
          dumEl.createArcPPP(x1, y1, x2, y2, px, py, false);
          //dumEl.createLine(x1, y1, x2, y2);
          if(currentDoc()->setPreviewElement(&dumEl)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set 1st point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&x1, &y1, true)) {
          currentDoc()->moveRelZero(x1, y1);
          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(RMES(287));
        }
      }

      // Set 2nd point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper()->snapPoint(&x2, &y2, true)) {
          currentDoc()->moveRelZero(x2, y2);
          currentDoc()->setCurrentState(3);
          statusPanel()->setLeftButtonStatus(RMES(288));
        }
      }

      // Set 3rd point:
      //
      else if(currentDoc()->getCurrentState()==3) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();
          int el=currentDoc()->addArcPPP(x1, y1,
                                  x2, y2,
                                  px, py,
                                  false);
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();
        }
        currentDoc()->setCurrentState(1);
        statusPanel()->setLeftButtonStatus(RMES(286));
        snapper()->snapPoint();
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting of first point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(286));
      }

      // Go back to setting of second point:
      //
      else if(currentDoc()->getCurrentState()==3) {
        snapper()->stop();
        currentDoc()->setCurrentState(2);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(287));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user create parallel arcs
    <p>States:
    <ul>
      <li>1: Set parallel
    </ul>
*/
void
RActDraw::arcsPara(int _myREvent)
{
  linesPara(_myREvent);
}



/*! Lets the user create circles (center, radius)
    <p>States:
    <ul>
      <li>1: Set center
      <li>2: Set radius
    </ul>
*/
void
RActDraw::circlesCR(int _myREvent)
{
  static float cx=DEF_AREAMAX;  // Coordinates of center
  static float cy=DEF_AREAMAX;  //
  static float cr=DEF_AREAMAX;  // Radius

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(282), RMES(270), RMES(302));
      break;

    case REVENT_MOUSEMOVE:

      // Move before center is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting radius:
      //
      else if(currentDoc()->getCurrentState()==2) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_CIRCLE, cx, cy, mtGetDistance(cx,cy, px,py))) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set center:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&cx, &cy, true)) {
          currentDoc()->moveRelZero(cx, cy);
          currentDoc()->setCurrentState(2);
          statusPanel()->setLeftButtonStatus(RMES(283));
        }
      }

      // Set radius:
      //
      else if(currentDoc()->getCurrentState()==2) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();
          cr = mtGetDistance(cx, cy, px, py);
          int el=currentDoc()->addCircle(cx, cy, cr, 0.0, 360.0, false);
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();

          currentDoc()->setCurrentState(1);
          statusPanel()->setLeftButtonStatus(RMES(284));
        }
        snapper()->snapPoint();
      }
      break;

    case REVENT_RBUTTONUP:

      // Break before center is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting of center:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        statusPanel()->setLeftButtonStatus(RMES(282));
        snapper()->snapPoint();
      }
      break;

    default:
      break;
  }

}



/*! Lets the user create texts
    <p>States:
    <ul>
      <li>1: Set text
    </ul>
*/
void
RActDraw::textsNormal(int _myREvent)
{
  static QString text="";   // The text
  static float prevLeft;    // Border of preview
  static float prevTop;
  static float prevRight;
  static float prevBottom;

  switch(_myREvent) {

    case REVENT_BEGIN:
      if(true) {
        // Font Options Dialog:
        //
        RFontDialog fontdlg(currentDoc(), qApp->mainWidget());
        if(fontdlg.exec()) {
          text=fontdlg.getText();

          // Create an element with this text (at 0/0):
          //
          RElement prevEl(currentDoc());
          prevEl.createText(0.0, 0.0,
                            currentDoc()->getFontNumber(),
                            (unsigned char*)text.latin1(),
                            currentDoc()->getFontFlags()->getFlags(),
                            currentDoc()->getFontHeight(),
                            currentDoc()->getFontAngle(),
                            currentDoc()->getFontRadius(),
                            currentDoc()->getFontLetterSpacing(),
                            currentDoc()->getFontWordSpacing(),
                            currentDoc()->getFontLineDistance());
          prevEl.getBorders(&prevLeft, &prevBottom, &prevRight, &prevTop);

          RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
          currentDoc()->setCurrentState(1);
          currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
          statusPanel()->setActionStatus(RMES(323), RMES(270), RMES(324));
          currentDoc()->update();
        }
        else {
          currentDoc()->setBehavior(BVR_NO);
          currentDoc()->resetCurrentState();
          currentDoc()->resetCurrentAction();
        }
      }

      break;

    case REVENT_MOUSEMOVE:
      // Move before position is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_RECT,
                                             px+prevLeft, py+prevTop,
                                             px+prevRight, py+prevBottom)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:
      // Set position:
      //
      if(currentDoc()->getCurrentState()==1) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();

          int el=
          currentDoc()->addText(px, py,
                                currentDoc()->getFontNumber(),
                                (unsigned char*)text.latin1(),
                                currentDoc()->getFontFlags()->getFlags(),
                                currentDoc()->getFontHeight(),
                                currentDoc()->getFontAngle(),
                                currentDoc()->getFontRadius(),
                                currentDoc()->getFontLetterSpacing(),
                                currentDoc()->getFontWordSpacing(),
                                currentDoc()->getFontLineDistance());
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();
          currentDoc()->setCurrentState(1);
        }
      }
      break;

    case REVENT_RBUTTONUP:
      // Break before position is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      break;

    default:
      break;
  }

}



/*! Lets the user create aligning dimensions
    <p>States:
    <ul>
      <li>1: Set 1st point
      <li>2: Set 2nd point
      <li>3: Set location
    </ul>
*/
void
RActDraw::dimsPara(int _myREvent)
{
  static float x1=DEF_AREAMAX;  // Coordinates of first point
  static float y1=DEF_AREAMAX;  //
  static float x2=DEF_AREAMAX;  // Coordinates of second point
  static float y2=DEF_AREAMAX;  //

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(430), RMES(270), RMES(423));
      break;

    case REVENT_MOUSEMOVE:

      // Move before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting 2nd point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper()->snapPoint(&x2, &y2)) {
          if(currentDoc()->setPreviewElement(T_DIMENSION, x1, y1, x2, y2, 0.0, 0.0, 0.0, E_STRAIGHT)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }

      // Move for setting distance:
      //
      else if(currentDoc()->getCurrentState()==3) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          RElement tmpEl;
          tmpEl.createLine(x1, y1, x2, y2);
          float dist=tmpEl.getDistanceToPoint(px, py, true);
          if(tmpEl.getRelAngleToPoint(px, py, true)<0.0) dist*=-1;
          if(currentDoc()->setPreviewElement(T_DIMENSION, x1, y1, x2, y2, 0.0, 0.0, dist, E_STRAIGHT)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }

      break;

    case REVENT_LBUTTONUP:

      // Set 1st point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&x1, &y1, true)) {
          currentDoc()->moveRelZero(x1, y1);
          statusPanel()->setLeftButtonStatus(RMES(431));
          currentDoc()->setCurrentState(2);
        }
      }

      // Set 2nd point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper()->snapPoint(&x2, &y2, true)) {
          currentDoc()->moveRelZero(x2, y2);
          statusPanel()->setLeftButtonStatus(RMES(432));
          currentDoc()->setCurrentState(3);
        }
      }

      // Set distance:
      //
      else if(currentDoc()->getCurrentState()==3) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();
          RElement tmpEl;
          tmpEl.createLine(x1, y1, x2, y2);
          float dist=tmpEl.getDistanceToPoint(px, py, true);
          if(tmpEl.getRelAngleToPoint(px, py, true)<0.0) dist*=-1;
          int el=currentDoc()->addDimension(x1, y1, x2, y2, 0.0, 0.0, dist, E_STRAIGHT);
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();

          statusPanel()->setLeftButtonStatus(RMES(430));
          currentDoc()->setCurrentState(1);
        }
        snapper()->snapPoint();
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting of 1st point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(430));
      }

      // Go back to setting of 2nd point:
      //
      else if(currentDoc()->getCurrentState()==3) {
        snapper()->stop();
        currentDoc()->setCurrentState(2);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(431));
      }

      break;

    default:
      break;
  }
}



/*! Lets the user create horizontal dimensions
    <p>States:
    <ul>
      <li>1: Set 1st pos
      <li>2: Set 2nd pos
      <li>3: Set location
    </ul>
*/
void
RActDraw::dimsHor(int _myREvent)
{
  static float x1=DEF_AREAMAX;  // Coordinates of first point
  static float y1=DEF_AREAMAX;  //
  static float x2=DEF_AREAMAX;  // Coordinates of second point
  static float y2=DEF_AREAMAX;  //

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(430), RMES(270), RMES(423));
      break;

    case REVENT_MOUSEMOVE:

      // Move before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting 2nd point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper()->snapPoint(&x2, &y2)) {
          if(currentDoc()->setPreviewElement(T_DIMENSION, x1, y1, x2, y2, 0.0, 0.0, y2, E_STRAIGHT|E_HORIZONTAL)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }

      // Move for setting distance:
      //
      else if(currentDoc()->getCurrentState()==3) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_DIMENSION, x1, y1, x2, y2, 0.0, 0.0, py, E_STRAIGHT|E_HORIZONTAL)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }

      break;

    case REVENT_LBUTTONUP:

      // Set 1st point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&x1, &y1, true)) {
          currentDoc()->moveRelZero(x1, y1);
          statusPanel()->setLeftButtonStatus(RMES(431));
          currentDoc()->setCurrentState(2);
        }
      }

      // Set 2nd point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper()->snapPoint(&x2, &y2, true)) {
          currentDoc()->moveRelZero(x2, y2);
          statusPanel()->setLeftButtonStatus(RMES(432));
          currentDoc()->setCurrentState(3);
        }
      }

      // Set distance:
      //
      else if(currentDoc()->getCurrentState()==3) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();
          int el=currentDoc()->addDimension(x1, y1, x2, y2, 0.0, 0.0, py, E_STRAIGHT|E_HORIZONTAL);
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();

          statusPanel()->setLeftButtonStatus(RMES(430));
          currentDoc()->setCurrentState(1);
        }
        snapper()->snapPoint();
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting of 1st point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(430));
      }

      // Go back to setting of 2nd point:
      //
      else if(currentDoc()->getCurrentState()==3) {
        snapper()->stop();
        currentDoc()->setCurrentState(2);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(431));
      }

      break;

    default:
      break;
  }
}



/*! Lets the user create vertical dimensions
    <p>States:
    <ul>
      <li>1: Set 1st pos
      <li>2: Set 2nd pos
      <li>3: Set location
    </ul>
*/
void
RActDraw::dimsVer(int _myREvent)
{
  static float x1=DEF_AREAMAX;  // Coordinates of first point
  static float y1=DEF_AREAMAX;  //
  static float x2=DEF_AREAMAX;  // Coordinates of second point
  static float y2=DEF_AREAMAX;  //

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(430), RMES(270), RMES(423));
      break;

    case REVENT_MOUSEMOVE:

      // Move before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting 2nd point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper()->snapPoint(&x2, &y2)) {
          if(currentDoc()->setPreviewElement(T_DIMENSION, x1, y1, x2, y2, 0.0, 0.0, x2, E_STRAIGHT|E_VERTICAL)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }

      // Move for setting distance:
      //
      else if(currentDoc()->getCurrentState()==3) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_DIMENSION, x1, y1, x2, y2, 0.0, 0.0, px, E_STRAIGHT|E_VERTICAL)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }

      break;

    case REVENT_LBUTTONUP:

      // Set 1st point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&x1, &y1, true)) {
          currentDoc()->moveRelZero(x1, y1);
          statusPanel()->setLeftButtonStatus(RMES(431));
          currentDoc()->setCurrentState(2);
        }
      }

      // Set 2nd point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper()->snapPoint(&x2, &y2, true)) {
          currentDoc()->moveRelZero(x2, y2);
          statusPanel()->setLeftButtonStatus(RMES(432));
          currentDoc()->setCurrentState(3);
        }
      }

      // Set distance:
      //
      else if(currentDoc()->getCurrentState()==3) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();
          int el=currentDoc()->addDimension(x1, y1, x2, y2, 0.0, 0.0, px, E_STRAIGHT|E_VERTICAL);
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();

          statusPanel()->setLeftButtonStatus(RMES(430));
          currentDoc()->setCurrentState(1);
        }
        snapper()->snapPoint();
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting of 1st point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(430));
      }

      // Go back to setting of 2nd point:
      //
      else if(currentDoc()->getCurrentState()==3) {
        snapper()->stop();
        currentDoc()->setCurrentState(2);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(431));
      }

      break;

    default:
      break;
  }
}



/*! Lets the user create dimensions for angles
    <p>States:
    <ul>
      <li>1: Set 1st leg
      <li>2: Set 2nd leg
      <li>3: Set radius
      <li>4: Shorten leg 1
      <li>5: Shorten leg 2
    </ul>
*/
void
RActDraw::dimsAngle(int _myREvent)
{
  static RElement* leg1=0;  // First leg
  static RElement* leg2=0;  // Second leg
  static RElement  prevEl;  // Preview
  static float cx=0.0;      // Center of dimension
  static float cy=0.0;
  static float rad1=0.0;    // Inner Radiuses
  static float rad2=0.0;
  static float cr=50.0;     // Radius of dimension line
  static float a1=0.0;      // Angles
  static float a2=0.0;
  static float tpx1=0.0;    // Touch points of legs
  static float tpy1=0.0;
  static float tpx2=0.0;
  static float tpy2=0.0;

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
      currentDoc()->setHighlightFilter(T_LINE);
      statusPanel()->setActionStatus(RMES(274), RMES(270), RMES(437));
      break;

    case REVENT_MOUSEMOVE:

      // Set 1st leg:
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_LINE);
      }

      else if(currentDoc()->getCurrentState()==2) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_LINE, leg1);

        // Set 2nd leg / preview dimension:
        if(snapper() &&
           snapper()->getSnappedElement() &&
           leg1) {

          leg2 = snapper()->getSnappedElement();

          if(leg2 && leg2!=leg1) {
            bool inter=false;
            leg1->getIntersection(leg2, &inter, &cx, &cy, 0,0,0,0,false);

            if(inter) {
              a1 = mtGetAngle(cx, cy, tpx1, tpy1);
              leg2->getTouchPoint(&tpx2, &tpy2,
                                  currentDoc()->getRealMousePosX(),
                                  currentDoc()->getRealMousePosY());
              a2 = mtGetAngle(cx, cy, tpx2, tpy2);
              cr = 50.0;

              prevEl.createDimension(cx, cy, a1, a2,
                                     rad1, rad2,
                                     cr,
                                     E_ROUNDOUT);

              if(currentDoc()->setPreviewElement(&prevEl)) {
                currentDoc()->drawPreviewElement();
              }
            }
          }
        }
      }

      // Set radius:
      else if(currentDoc()->getCurrentState()==3) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          cr = mtGetDistance(cx, cy, px, py);

          prevEl.createDimension(cx, cy, a1, a2,
                                 rad1, rad2,
                                 cr,
                                 E_ROUNDOUT);

          if(currentDoc()->setPreviewElement(&prevEl)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }

      // Shorten leg 1:
      else if(currentDoc()->getCurrentState()==4) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          rad1 = mtGetDistance(cx, cy, px, py);

          prevEl.createDimension(cx, cy, a1, a2,
                                 rad1, rad2,
                                 cr,
                                 E_ROUNDOUT);

          if(currentDoc()->setPreviewElement(&prevEl)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }

      // Shorten leg 2:
      else if(currentDoc()->getCurrentState()==5) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          rad2 = mtGetDistance(cx, cy, px, py);

          prevEl.createDimension(cx, cy, a1, a2,
                                 rad1, rad2,
                                 cr,
                                 E_ROUNDOUT);

          if(currentDoc()->setPreviewElement(&prevEl)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:
      // Set first leg:
      if(currentDoc()->getCurrentState()==1) {
        if(snapper() &&
           snapper()->getSnappedElement()) {
          leg1 = snapper()->getSnappedElement();
          if(leg1) {
            leg1->getTouchPoint(&tpx1, &tpy1,
                                currentDoc()->getRealMousePosX(),
                                currentDoc()->getRealMousePosY());
            currentDoc()->setCurrentState(2);
            statusPanel()->setLeftButtonStatus(RMES(277));
            currentDoc()->setHighlightFilter(T_LINE, leg1);
          }
        }
      }

      // Set 2nd leg:
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper() &&
           snapper()->getSnappedElement()) {

          leg2 = snapper()->getSnappedElement();

          if(leg2) {
            leg2->getTouchPoint(&tpx2, &tpy2,
                                currentDoc()->getRealMousePosX(),
                                currentDoc()->getRealMousePosY());

            currentDoc()->setCurrentState(3);
            statusPanel()->setLeftButtonStatus(RMES(283));
          }
        }
      }

      // Set radius:
      else if(currentDoc()->getCurrentState()==3) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          cr = mtGetDistance(cx, cy, px, py);
          currentDoc()->setCurrentState(4);
          statusPanel()->setLeftButtonStatus(RMES(446));
        }
      }

      // Shorten 1st leg:
      else if(currentDoc()->getCurrentState()==4) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          rad1 = mtGetDistance(cx, cy, px, py);
          currentDoc()->setCurrentState(5);
          statusPanel()->setLeftButtonStatus(RMES(447));
        }
      }

      // Shorten 2nd leg:
      else if(currentDoc()->getCurrentState()==5) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          rad2 = mtGetDistance(cx, cy, px, py);

          bool inter=false;
          leg1->getIntersection(leg2, &inter, &cx, &cy, 0,0,0,0,false);

          if(inter) {
            prevEl.createDimension(cx, cy, a1, a2,
                                   rad2, rad2,
                                   cr,
                                   E_ROUNDOUT);

            if(prevEl.getFlag(E_VISIBLE)) {
              snapper()->stop();
              int el=currentDoc()->addDimension(cx, cy, a1, a2, rad1, rad2, cr, E_ROUNDOUT);
              currentDoc()->drawElement(el);
              currentDoc()->terminateAction();

              currentDoc()->setCurrentState(1);
              dimsAngle(REVENT_MOUSEMOVE);
              currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
              statusPanel()->setLeftButtonStatus(RMES(274));
            }
          }
        }
      }
      break;

    case REVENT_RBUTTONUP:
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
        statusPanel()->setLeftButtonStatus(RMES(274));
      }
      else if(currentDoc()->getCurrentState()==3) {
        snapper()->stop();
        currentDoc()->setCurrentState(2);
        currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
        statusPanel()->setLeftButtonStatus(RMES(277));
      }
      else if(currentDoc()->getCurrentState()==4) {
        snapper()->stop();
        currentDoc()->setCurrentState(3);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(283));
      }
      else if(currentDoc()->getCurrentState()==5) {
        snapper()->stop();
        currentDoc()->setCurrentState(4);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(446));
      }

      break;

    default:
      break;
  }
}



/*! Lets the user create dimensions for diameters
    <p>States:
    <ul>
      <li>1: Set circle
      <li>2: Set angle
      <li>3: Set location
    </ul>
*/
void
RActDraw::dimsDiam(int _myREvent)
{
  static RElement* el=0;        // Circular element
  static float px=DEF_AREAMAX;  // Coordinates of dimension position
  static float py=DEF_AREAMAX;  //
  static float x1=0.0;          // Endpoints of the extension lines
  static float y1=0.0;
  static float x2=0.0;
  static float y2=0.0;
  static float dist=0.0;        // Pos of dimension line
  static float ang=0.0;         // Angle

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(448), RMES(270), RMES(438));
      break;

    case REVENT_MOUSEMOVE:

      // Move before circular element is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_CIRCULAR);

        if(snapper()) {
          snapper()->getSnappedElement();
        }
      }

      // Move for setting angle:
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(el && snapper()->snapPoint(&px, &py)) {
          ang = mtGetAngle(el->getCx(), el->getCy(), px, py);
          x1 = el->getCx() + cos(ang/ARAD) * el->getCr();
          y1 = el->getCy() + sin(ang/ARAD) * el->getCr();
          x2 = el->getCx() - cos(ang/ARAD) * el->getCr();
          y2 = el->getCy() - sin(ang/ARAD) * el->getCr();
          if(currentDoc()->setPreviewElement(T_DIMENSION, x1, y1, x2, y2, 0.0, 0.0, 0.0, E_STRAIGHT|E_DIAMETER)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }

      // Move for setting distance:
      //
      else if(currentDoc()->getCurrentState()==3) {
        if(el && snapper()->snapPoint(&px, &py)) {
          RElement tmpEl;
          tmpEl.createLine(x1, y1, x2, y2);
          dist = tmpEl.getDistanceToPoint(px, py, true);
          if(tmpEl.getRelAngleToPoint(px, py, true)<0.0) dist*=-1;
          if(currentDoc()->setPreviewElement(T_DIMENSION, x1, y1, x2, y2, 0.0, 0.0, dist, E_STRAIGHT|E_DIAMETER)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set circular element:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper() &&
           snapper()->getSnappedElement()) {
          el = snapper()->getSnappedElement();
          statusPanel()->setLeftButtonStatus(RMES(432));
          currentDoc()->setCurrentState(2);
        }
      }

      // Set angle:
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper()->snapPoint(&px, &py, true)) {
          statusPanel()->setLeftButtonStatus(RMES(449));
          currentDoc()->setCurrentState(3);
        }
      }

      // Set position:
      //
      else if(currentDoc()->getCurrentState()==3) {
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();
          RElement tmpEl;
          tmpEl.createLine(x1, y1, x2, y2);
          dist = tmpEl.getDistanceToPoint(px, py, true);
          if(tmpEl.getRelAngleToPoint(px, py, true)<0.0) dist*=-1;
          int elNr=currentDoc()->addDimension(x1, y1, x2, y2, 0.0, 0.0, dist, E_STRAIGHT|E_DIAMETER);
          currentDoc()->drawElement(elNr);
          currentDoc()->terminateAction();

          statusPanel()->setLeftButtonStatus(RMES(448));
          currentDoc()->setCurrentState(1);
        }
      }
      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting element:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
        statusPanel()->setLeftButtonStatus(RMES(448));
      }

      // Go back to setting angle:
      //
      else if(currentDoc()->getCurrentState()==3) {
        snapper()->stop();
        currentDoc()->setCurrentState(2);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(432));
      }

      break;

    default:
      break;
  }
}



/*! Lets the user create dimensions for radiuses
    <p>States:
    <ul>
      <li>1: Set circular element
      <li>2: Set location
    </ul>
*/
void
RActDraw::dimsRad(int _myREvent)
{
  static RElement* el=0;        // Circular element
  static float px=DEF_AREAMAX;  // Coordinates of dimension position
  static float py=DEF_AREAMAX;  //

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(448), RMES(270), RMES(439));
      break;

    case REVENT_MOUSEMOVE:

      // Move before circular element is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        currentDoc()->setBehavior(BVR_SNAP_NEXT_ELEMENT);
        currentDoc()->setHighlightFilter(T_CIRCULAR);

        if(snapper()) {
          snapper()->getSnappedElement();
        }
      }

      // Move for setting angle and pos:
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(el && snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_DIMENSION,
                                             el->getCx(), el->getCy(),
                                             px, py, 0.0, 0.0, el->getCr(),
                                             E_STRAIGHT|E_RADIUS)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set circular element:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper() &&
           snapper()->getSnappedElement()) {
          el = snapper()->getSnappedElement();
          statusPanel()->setLeftButtonStatus(RMES(432));
          currentDoc()->setCurrentState(2);
        }
      }

      // Set angle / position:
      //
      else if(currentDoc()->getCurrentState()==2) {
        if(snapper()->snapPoint(&px, &py, true)) {
          snapper()->stop();
          int elNr=currentDoc()->addDimension(el->getCx(), el->getCy(),
                                              px, py,
                                              0.0, 0.0, el->getCr(),
                                              E_STRAIGHT|E_RADIUS);
          currentDoc()->drawElement(elNr);
          currentDoc()->terminateAction();

          statusPanel()->setLeftButtonStatus(RMES(448));
          currentDoc()->setCurrentState(1);
        }
      }
      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting element:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        currentDoc()->doBehaviorSnapNextElement(REVENT_MOUSEMOVE);
        statusPanel()->setLeftButtonStatus(RMES(448));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user create arrows
    <p>States:
    <ul>
      <li>1: Set arrow point
      <li>2: Set end of arrow
    </ul>
*/
void
RActDraw::dimsArrow(int _myREvent)
{
  static float x1=DEF_AREAMAX;  // Coordinates of first (last) point
  static float y1=DEF_AREAMAX;  //

  switch(_myREvent) {

    case REVENT_BEGIN:
      RAppWin::getRAppWin()->changeMenu(TOOLBAR_SNAP);
      currentDoc()->setCurrentState(1);
      currentDoc()->setCursor(*RAppWin::getRAppWin()->curCross);
      statusPanel()->setActionStatus(RMES(460), RMES(270), RMES(461));
      break;

    case REVENT_MOUSEMOVE:

      // Move before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->snapPoint();
      }

      // Move for setting 2nd point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        float px, py;
        if(snapper()->snapPoint(&px, &py)) {
          if(currentDoc()->setPreviewElement(T_DIMENSION,
                                             x1, y1, px, py,
                                             0.0, 0.0, 0.0,
                                             E_STRAIGHT|E_ARROW)) {
            currentDoc()->drawPreviewElement();
          }
        }
      }
      break;

    case REVENT_LBUTTONUP:

      // Set 1st point:
      //
      if(currentDoc()->getCurrentState()==1) {
        if(snapper()->snapPoint(&x1, &y1, true)) {
          currentDoc()->moveRelZero(x1, y1);
          statusPanel()->setLeftButtonStatus(RMES(462));
          currentDoc()->setCurrentState(2);
        }
      }

      // Set 2nd point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        float px, py;
        if(snapper()->snapPoint(&px, &py, true)) {
          currentDoc()->moveRelZero(px, py);
          snapper()->stop();
          int el=currentDoc()->addDimension(x1, y1,
                                            px, py,
                                            0.0, 0.0, 0.0,
                                            E_STRAIGHT|E_ARROW);
          currentDoc()->drawElement(el);
          currentDoc()->terminateAction();
        }
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(460));
        currentDoc()->setCurrentState(1);
      }

      break;

    case REVENT_RBUTTONUP:

      // Break before 1st point is set:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }

      // Go back to setting of first point:
      //
      else if(currentDoc()->getCurrentState()==2) {
        snapper()->stop();
        currentDoc()->setCurrentState(1);
        snapper()->snapPoint();
        statusPanel()->setLeftButtonStatus(RMES(462));
      }
      break;

    default:
      break;
  }
}



/*! Lets the user create hatches
*/
void
RActDraw::hatching(int _rEvent)
{

  switch(_rEvent) {

    case REVENT_BEGIN:
      currentDoc()->setCurrentState(1);
      if(true) {

        snapper()->stop();

        currentDoc()->setPreviewUpdate(false);

        // Show hatching Dialog:
        //
        QString pattern;
        float factor;

        RHatchDialog* dlg =
          new RHatchDialog(RMES(451),
                           RMES(452),
                           qApp->mainWidget());
        if(dlg->exec()) {
          pattern = dlg->getPattern();
          factor = dlg->getFactor();

          int elNr=currentDoc()->addHatching(factor, pattern);
          currentDoc()->drawElement(elNr);
          currentDoc()->terminateAction();

          currentDoc()->resetCurrentState();
          currentDoc()->resetCurrentAction();
        }
        delete dlg;
      }
      break;

    case REVENT_RBUTTONUP:

      // Break:
      //
      if(currentDoc()->getCurrentState()==1) {
        snapper()->stop();
        currentDoc()->setBehavior(BVR_NO);
        currentDoc()->resetCurrentState();
        currentDoc()->resetCurrentAction();
        currentDoc()->setCursor(arrowCursor);
        statusPanel()->clear();
      }
      break;

    default:
      break;
  }

}



// EOF






















