/***************************************************************************
                          rfiledxf.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/


/****************************************************************************
** rfiledxf.cpp 1998/08/28 A. Mustun RibbonSoft
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#include "rfiledxf.h"

#include <stdlib.h>

#include <qstring.h>
#include <qtextstream.h>
#include <qdatetime.h>

#include "rappwin.h"
#include "rfilebase.h"
#include "rfonts.h"
#include "rgraphic.h"
#include "rconfig.h"
#include "rlog.h"
#include "rmath.h"
#include "rprgdef.h"
#include "rstatuspanel.h"

// Constructor:
//
RFileDxf::RFileDxf(const char* _name, RGraphic* _graphic)
  :RFileBase(_name)
{
  graphic=_graphic;
}



// Destructor:
//
RFileDxf::~RFileDxf()
{
  
}



// load a dxf file (read in buffer / count / read from buufer):
//
bool
RFileDxf::load(bool add)
{
  if(readFileInBuffer()) {
    seperateBuf();
    return readFromBuffer(add);
  }
  else {
    return false;
  }
}



// read a dxf file from buffer:
//
bool
RFileDxf::readFromBuffer(bool add)
{
  bool      ret;                    // returned value
  QString   dxfLine;                // A line in the dxf file
  QString   dxfCode;                // A Code in the dxf file as string
  int       code=-1;                // Dxf-code as number
  float     vx1=0.0, vy1=0.0,       // Start point
            vx2=0.0, vy2=0.0,       // End point
            vcx=0.0, vcy=0.0,       // Centre
            vcr=0.0,                // Radius
            va1=0.0, va2=0.0,       // Start / End Angle
            vab=0.0,                // Bulge
            vpx=0.0, vpy=0.0;       // First Polyline point
  float     ax=0.0, ay=0.0;         // Current coordinate
  bool      plClose=false;          // Polyline closed-flag
  QString   lastLayer;              // Last used layer name (test adding only
                                    //   if the new layer!=lastLayer)
  int       currentLayerNum=0;      // Current layer number

  if(!add) graphic->clearLayers();

  graphic->addLayer(DEF_DEFAULTLAYER);

  resetBufP();

  statusPanel()->iniProgress(fSize, RMES(292));

  if(fBuf) {
  
    do {
      dxfLine=getBufLine();
      
      statusPanel()->setProgress(fBufP, 1000);

      if(dxfLine && 
         dxfLine[0]>='A' && dxfLine[0]<='V') {
         	
        if(dxfLine=="EOF") {
        	// End of file reached
        	//
        }
      
        // ------
        // Layer:
        // ------
        else if(dxfLine=="LAYER") {
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  2:  // Layer name
                    currentLayerNum = graphic->addLayer(dxfLine.data());
                    lastLayer=dxfLine.data();
                    break;
                  case 70:  // Visibility
                    /*
                    if(dxfLine.toInt()&5) {
                      if(currentLayerNum>=0 && currentLayerNum<DEF_MAXLAYERS) {
                        graphic->layer[currentLayerNum].DelFlag(Y_VISIBLE);
                      }
                    }
                    */
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
        }

        // ------
        // Point:
        // ------
        else if(dxfLine=="POINT") {
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case 10:  // X1
                    vx1 = dxfLine.toFloat();
                    break;
                  case 20:  // Y1
                    vy1 = dxfLine.toFloat();
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          graphic->addPoint(vx1, vy1, currentLayerNum, add);
        }
        
        // -----
        // Line:
        // -----
        else if(dxfLine=="LINE") {
          do {
            dxfCode=getBufLine();

            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              
              dxfLine=getBufLine();

              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case 10:  // X1
                    vx1 = dxfLine.toFloat();
                    break;
                  case 20:  // Y1
                    vy1 = dxfLine.toFloat();
                    break;
                  case 11:  // X2
                    vx2 = dxfLine.toFloat();
                    break;
                  case 21:  // Y2
                    vy2 = dxfLine.toFloat();
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          
          if(!mtCompFloat(vx1, vx2) || !mtCompFloat(vy1, vy2)) {
            graphic->addLine(vx1, vy1, vx2, vy2, currentLayerNum, add);
          }
        }
        
        // ----
        // Arc:
        // ----
        else if(dxfLine=="ARC") {
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case 10:  // Centre X
                    vcx = dxfLine.toFloat();
                    break;
                  case 20:  // Centre Y
                    vcy = dxfLine.toFloat();
                    break;
                  case 40:  // Radius
                    vcr = dxfLine.toFloat();
                    break;
                  case 50:  // Start Angle
                    va1 = mtCorrAngle(dxfLine.toFloat());
                    break;
                  case 51:  // End Angle
                    va2 = mtCorrAngle(dxfLine.toFloat());
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          if(vcr>0.0 && !mtCompFloat(va1, va2)) {
            graphic->addArc(vcx, vcy, vcr, va1, va2, false, currentLayerNum, add);
          }
        }
        
        // -------
        // Circle:
        // -------
        else if(dxfLine=="CIRCLE") {
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case 10:  // Centre X
                    vcx = dxfLine.toFloat();
                    break;
                  case 20:  // Centre Y
                    vcy = dxfLine.toFloat();
                    break;
                  case 40:  // Radius
                    vcr = dxfLine.toFloat();
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          if(vcr>0.0) {
            graphic->addCircle(vcx, vcy, vcr, 0.0, 360.0, false, currentLayerNum, add);
          }
        }

        // --------
        // Ellipse:
        // --------
        else if(dxfLine=="ELLIPSE") {
          float vapx=0.0, vapy=0.0; // Apex (endpoint of center axis)
          float vratio=1.0;         // Ratio of minor axis to major axis
          bool  flipY=false;        // Flip y-axis (if extrusion direction is -1)
          do {
            dxfCode=getBufLine();
            
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case 10:  // Centre X
                    vcx = dxfLine.toFloat();
                    break;
                  case 20:  // Centre Y
                    vcy = dxfLine.toFloat();
                    break;
                  case 11:  // Endpoint of major axis X
                    vapx = dxfLine.toFloat();
                    break;
                  case 21:  // Endpoint of major axis Y
                    vapy = dxfLine.toFloat();
                    break;
                  case 40:  // Ratio of minor axis to major axis (eg.: 0.75)
                    vratio = dxfLine.toFloat();
                    break;
                  case 41:  // Start angle in rad (0.0 for a full ellipse)
                    va1 = dxfLine.toFloat();
                    break;
                  case 42:  // End angle in rad (2PI=6.283 for a full ellipse)
                    va2 = dxfLine.toFloat();
                    break;
                  case 230: // extrusion direction
                    flipY = ((int)dxfLine.toFloat()==-1);
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          // Create the ellipse as lines:
          //
          float ac;            // angle counter
          float refAng;        // ref angle (major axis)
          float tx, ty;        // tmp pos on ellipse
          float otx, oty;      // old tmp pos on ellipse
          float aStep = RCONFIG->getSettingDouble("DXF:DxfEllipseSegmentAngle");  // angle step

          if(aStep<0.01) aStep=0.01;

          vapx+=vcx;
          vapy+=vcy;
          vcr=mtGetDistance(vcx, vcy, vapx, vapy);
          refAng=mtGetAngle(vcx, vcy, vapx, vapy);

          otx=vcx + cos(va1) * vcr;
          if(!flipY) oty=vcy + sin(va1) * vcr * vratio;
          else       oty=vcy - sin(va1) * vcr * vratio;
          mtRotatePoint(vcx, vcy, &otx, &oty, refAng);
          
          for(ac=va1*ARAD+aStep; ac<=va2*ARAD; ac+=aStep) {
            tx=vcx + cos(ac/ARAD) * vcr;
            if(!flipY) ty=vcy + sin(ac/ARAD) * vcr * vratio;
            else       ty=vcy - sin(ac/ARAD) * vcr * vratio;

            if(!mtCompFloat(refAng, 0.0)) {
              mtRotatePoint(vcx, vcy, &tx, &ty, refAng);
            }
            
            if(!mtCompFloat(otx, tx) || !mtCompFloat(oty, ty)) {
              graphic->addLine(otx, oty, tx, ty, currentLayerNum, add);
            }

            otx=tx;
            oty=ty;
          }

          if(!mtCompFloat(ac, va2*ARAD)) {
            tx=vcx + cos(va2) * vcr;
            if(!flipY) ty=vcy + sin(va2) * vcr * vratio;
            else       ty=vcy - sin(va2) * vcr * vratio;
            mtRotatePoint(vcx, vcy, &tx, &ty, refAng);

            if(!mtCompFloat(otx, tx) || !mtCompFloat(oty, ty)) {
              graphic->addLine(otx, oty, tx, ty, currentLayerNum, add);
            }
          }
        }

        // -------
        // Spline:
        // -------
        else if(dxfLine=="SPLINE") {
          //bool  flipY=false;      // Flip y-axis (if extrusion direction is -1)
          int   ctrlPts=1;        // number of control points
          bool  closed=false;

          // Get number of control points and flags:
          //
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case 70:
                    closed=(bool)(1&dxfLine.toInt());
                    break;
                  case 73:  // number of control points
                    ctrlPts=dxfLine.toInt();
                    code=0;  // break while found what we need...
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          if(closed) ++ctrlPts;

          // allocate memory for ctrl-points:
          //
          float* px;      // tmp used coordinates of ctrl pts
          float* py;      // 
          int    ac=0;    // array counter

          px=new float[ctrlPts];
          py=new float[ctrlPts];
          
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case 10:  // X-value
                    vx1 = dxfLine.toFloat();
                    break;
                  case 20:  // Y-value
                    vy1 = dxfLine.toFloat();

                    px[ac]=vx1;
                    py[ac]=vy1;

                    // closed spline: start and end the same:
                    //
                    if(ac==0 && closed) {
                      px[ctrlPts-1]=vx1;
                      py[ctrlPts-1]=vy1;
                    }

                    // dont read too much points:
                    //
                    if(ac<ctrlPts-1) ++ac;

                    break;
                  case 40:  // Knot value (? kind of an angle)
                    break;
                  case 73:  // number of control points
                    ctrlPts=dxfLine.toInt();
                    break;
                  case 230: // extrusion direction
                    //flipY = ((int)dxfLine.toFloat()==-1);
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          // Create the spline as lines:
          //
          int   i;           // counter for control points
          float u;           // counter for float value (loop)
          float nc1, nc2,    // what ever...
                nc3, nc4;
          float tx, ty;      // tmp pos on spline
          float otx, oty;    // old tmp pos on spline
          int   ao1, ao2,    // array offsets
                ao3, ao4;
          int   segm;        // segments

          segm=RCONFIG->getSettingInt("DXF:DxfSplineSegments");
          if(segm<=0) segm=1;

          otx=px[0];
          oty=py[0];

          for(i=0; i<=ctrlPts-1; ++i) {
            for(u=0.0; u<=1.0; u+=1.0/segm) {
              nc1=-(u*u*u/6.0)+u*u/2.0-u/2.0+1.0/6.0;
              nc2=u*u*u/2.0-u*u+2.0/3.0;
              nc3=(-u*u*u+u*u+u)/2.0+1.0/6.0;
              nc4=u*u*u/6.0;

              ao1=-1;     // normal case
              ao2= 0;
              ao3= 1;
              ao4= 2;

              if     (i==0        ) { ao1=0;        }   // 1st part of curve
              else if(i==ctrlPts-2) { ao4=1;        }   // 2nd last part of curve
              else if(i==ctrlPts-1) { ao3=0; ao4=0; }   // last part of curve

              tx = nc1*px[i+ao1] + nc2*px[i+ao2] +
                   nc3*px[i+ao3] + nc4*px[i+ao4]   ;
              ty = nc1*py[i+ao1] + nc2*py[i+ao2] +
                   nc3*py[i+ao3] + nc4*py[i+ao4]   ;

              if(!mtCompFloat(otx, tx, 0.0015) || !mtCompFloat(oty, ty, 0.0015)) {
                graphic->addLine(otx, oty, tx, ty, currentLayerNum, add);
                otx=tx;
                oty=ty;
              }
            }
          }

          if(!mtCompFloat(otx, px[ctrlPts-1], 0.0001) || !mtCompFloat(oty, py[ctrlPts-1], 0.0001)) {
            graphic->addLine(otx, oty, px[ctrlPts-1], py[ctrlPts-1], currentLayerNum, add);
          }

          delete[] px;
          delete[] py;
        }

        // ---------------------
        // LWPOLYLINE
        // ---------------------
        else if(dxfLine=="LWPOLYLINE") {
          plClose=false;
          float plX=DEF_AREAMAX,              // current position
                plY=DEF_AREAMAX;              //
          float oplX=0.0, oplY=0.0;           // last position
          float fstX=0.0, fstY=0.0;           // first position
          bool  firstPoint=true;              // first point in list?
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  0:
                    break;
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case  9:
                    break;
                  case 10:
                    oplX=plX;
                    plX=dxfLine.toFloat();

                    if(firstPoint) {
                      fstX=plX;
                    }
                    break;
                  case 20:
                    oplY=plY;
                    plY=dxfLine.toFloat();

                    if(firstPoint) {
                      fstY=plY;
                      firstPoint=false;
                    }
                    else {
                      if(!mtCompFloat(oplX, plX) || !mtCompFloat(oplY, plY)) {
                        graphic->addLine(oplX, oplY, plX, plY, currentLayerNum, add);
                      }
                    }
                    break;
                  case 70:  // Flag "Polyline closed"
                    plClose = (bool)(1&dxfLine.toInt());
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          // Close Polyline:
          //
          if(plClose) {
            if(!mtCompFloat(plX, fstX) || !mtCompFloat(plY, fstY)) {
              graphic->addLine(plX, plY, fstX, fstY, currentLayerNum, add);
            }
          }
        }

        // ------
        // Hatch:
        // ------
        /*
        if(dxfLine=="HATCH") {
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case 10:  // X1
                    vx1 = dxfLine.toFloat();
                    break;
                  case 20:  // Y1
                    vy1 = dxfLine.toFloat();
                    //graphic->Vec[vc].CreatePoint(vy1, vx1, currentLayerNum);
                    //if(vc<vElements-1) ++vc;
                    break;
                  case 11:  // X2
                    vx2 = dxfLine.toFloat();
                    break;
                  case 21:  // Y2
                    vy2 = dxfLine.toFloat();
                    //graphic->Vec[vc].CreatePoint(vy2, vx2, currentLayerNum);
                    //if(vc<vElements-1) ++vc;
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          / *
          if(!mt.CompFloat(vx1, vx2) || !mt.CompFloat(vy1, vy2)) {
            graphic->Vec[vc].CreateLine(vx1, vy1, vx2, vy2, currentLayerNum);
            if(vc<vElements-1) ++vc;
          }
          if(++updProgress==1000) {
            np->getStateWin()->UpdateProgressBar((int)(pcFact*vc)+25);
            updProgress=0;
          }
          * /
        }
        */

        // ---------------------
        // First Polyline-Point:
        // ---------------------
        else if(dxfLine=="POLYLINE") {
          plClose=false;
          // Get Flag "Polyline closed":
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  0:
                    break;
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case  9:
                    break;
                  case 70:  // Flag "Polyline closed"
                    plClose = (bool)(1&dxfLine.toInt());
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          gotoBufLine("VERTEX");
          vab=0;
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  0:
                    break;
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case  9:
                    break;
                  case 10:   // Next Coordinate X
                    vpx = ax = dxfLine.toFloat();
                    break;
                  case 20:   // Next Coordinate Y
                    vpy = ay = dxfLine.toFloat();
                    break;
                  case 42:   // Next Bulge
                    vab = dxfLine.toFloat();
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
        }

        // --------------------
        // Next Polyline-Point:
        // --------------------
        else if(dxfLine=="VERTEX") {
          //graphic->Vec[vc].Reset();
          float nvab=0.0;     // Next Bulge
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  0:
                    break;
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case  9:
                    break;
                  case 10:    // Next Coordinate X
                    vx2 = dxfLine.toFloat();
                    break;
                  case 20:    // Next Coordinate Y
                    vy2 = dxfLine.toFloat();
                    break;
                  case 42:    // Next Bulge
                    nvab = dxfLine.toFloat();
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          if(!mtCompFloat(ax, vx2) || !mtCompFloat(ay, vy2) || !mtCompFloat(vab, 0.0)) {
            graphic->addPolylineElement(ax, ay,
                                        vx2, vy2,
                                        vab,
                                        currentLayerNum,
                                        add);
          }
          ax = vx2;
          ay = vy2;
          vab=nvab;
        }

        // ----------------
        // Close Polylinie:
        // ----------------
        else if(dxfLine=="SEQEND" && plClose) {
          if(!mtCompFloat(ax, vpx) || !mtCompFloat(ay, vpy) || !mtCompFloat(vab, 0.0)) {
            graphic->addPolylineElement(ax, ay,
                                        vpx, vpy,
                                        vab,
                                        currentLayerNum,
                                        add);
          }
        }

        // -----
        // Text:
        // -----
        else if(dxfLine=="TEXT") {

          char  vtext[32768];     // the text
          //QString vtext;          // the text
          char  vtextStyle[256];  // text style (normal_ro, cursive_ri, normal_st, ...)
          float vheight=10.0,     // text height
                vtextAng=0.0,     // text angle
                vradius=0.0,      // text radius
                vletterspace=2.0, // Text letter space
                vwordspace=6.0;   // Text wordspace
          QCString vfont;         // font "normal", "cursive", ...
          int   valign=0;         // alignment (0=left, 1=center, 2=right)
          uint  vfl=0;            // special flags
          bool  codeSeven=false;  // Have we found a code seven?

          vtextStyle[0] = '\0';
          vfont="normal";

          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              if(code!=1 && code!=7) dxfLine=getBufLine();
              if(dxfLine || code==1 || code==7) {

                switch(code) {

                  case  1:  // Text itself
                    //vtext = getBufLine();
                    qstrncpy(vtext, getBufLine(), 32759);
                    //RLOG("\nInt text: ");
                    //RLOG((int)vtext.at(0));
                    break;

                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;

                  case  7:  // Text style (normal_ro#50.0, cursive_ri#20.0, normal_st)
                    qstrncpy(vtextStyle, getBufLine(), 249);

                    // get font typ:
                    //
                    if(true) {
                      char dummy[256];
                      sscanf(vtextStyle, "%[^_#\n]", dummy);
                      vfont=dummy;
                    }
                    /*if(strstr(vtextStyle, "cursive")) vfont="cursive";
                    else                              vfont="normal";*/

                    // get text style:
                    //
                    if(strstr(vtextStyle, "_ro"))      vfl=vfl|E_ROUNDOUT;
                    else if(strstr(vtextStyle, "_ri")) vfl=vfl|E_ROUNDIN;
                    else                               vfl=vfl|E_STRAIGHT;

                    if(strstr(vtextStyle, "_fix"))     vfl=vfl|E_FIXEDWIDTH;

                    // get radius, letterspace, wordspace:
                    //
                    if(true) {
                      char *ptr;  // pointer to value
                      ptr = strchr(vtextStyle, '#');
                      if(ptr) {
                        // Parse radius
                        if(vfl&E_ROUNDOUT || vfl&E_ROUNDIN) {
                          ++ptr;
                          if(ptr[0]) {
                            sscanf(ptr, "%f", &vradius);
                          }
                          ptr = strchr(ptr, '#');
                        }
                        if(ptr) {
                          // Parse letter space:
                          ++ptr;
                          if(ptr[0]) {
                            sscanf(ptr, "%f", &vletterspace);
                          }
                          // Parse word space:
                          ptr = strchr(ptr, '#');
                          if(ptr) {
                            ++ptr;
                            if(ptr[0]) {
                              sscanf(ptr, "%f", &vwordspace);
                            }
                          }
                        }
                      }
                    }
                    codeSeven=true;
                    break;

                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;

                  case 10:  // X1
                    vx1 = dxfLine.toFloat();
                    break;
                  case 20:  // Y1
                    vy1 = dxfLine.toFloat();
                    break;
                  case 40:  // height
                    vheight = dxfLine.toFloat();
                    if(!codeSeven) {
                      vletterspace = vheight*0.2;
                      vwordspace = vheight*0.6;
                    }
                    break;
                  case 50:  // angle
                    vtextAng = dxfLine.toFloat();
                    break;
                  case 72:  // alignment
                    valign = dxfLine.toInt();
                    if(valign==1)      vfl=vfl|E_CENTER;
                    else if(valign==2) vfl=vfl|E_RIGHT;
                    else               vfl=vfl|E_LEFT;
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;
                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          graphic->addText(vx1, vy1,
                           RFonts::getRFonts()->getFontNumber(vfont.data()),
                           (unsigned char*)vtext,
                           vfl,
                           vheight,
                           vtextAng,
                           vradius,
                           vletterspace,
                           vwordspace,
                           vheight*1.4,
                           currentLayerNum,
                           add);
          //graphic->addSingleElements(graphic->elementCurrent());
        }

        // ----------
        // Dimension:
        // ----------
        else if(dxfLine=="DIMENSION") {
          int typ=1;
          float v10=0.0, v20=0.0,
                v13=0.0, v23=0.0,
                v14=0.0, v24=0.0,
                v15=0.0, v25=0.0,
                v16=0.0, v26=0.0,
                v40=0.0, v50=0.0;
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case 10:  // line position x
                    v10 = dxfLine.toFloat();
                    break;
                  case 20:  // line position y
                    v20 = dxfLine.toFloat();
                    break;
                  case 13:  // X1
                    v13 = dxfLine.toFloat();
                    break;
                  case 23:  // Y1
                    v23 = dxfLine.toFloat();
                    break;
                  case 14:  // X2
                    v14 = dxfLine.toFloat();
                    break;
                  case 24:  // Y2
                    v24 = dxfLine.toFloat();
                    break;
                  case 15:  // X2
                    v15 = dxfLine.toFloat();
                    break;
                  case 25:  // Y2
                    v25 = dxfLine.toFloat();
                    break;
                  case 16:  // X2
                    v16 = dxfLine.toFloat();
                    break;
                  case 26:  // Y2
                    v26 = dxfLine.toFloat();
                    break;
                  case 40:
                    v40 = dxfLine.toFloat();
                    break;
                  case 50:
                    v50 = dxfLine.toFloat();
                    break;
                  case 70:  // Typ
                    typ = dxfLine.toInt();
                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;

                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);
          float dist;
          RElement tmpEl;

          // Remove Bit values:
          if(typ>=128) typ-=128;   // Location of Text
          if(typ>= 64) typ-= 64;   // Ordinate

          switch(typ) {
            // Horiz. / vert.:
            case 0:
              if(mtCompFloat(v50, 0.0)) {
                graphic->addDimension(v13, v23, v14, v24,
                                      0.0, 0.0,
                                      v20,
                                      E_STRAIGHT|E_HORIZONTAL,
                                      currentLayerNum,
                                      add);
              }
              else {
                graphic->addDimension(v13, v23, v14, v24,
                                      0.0, 0.0,
                                      v10,
                                      E_STRAIGHT|E_VERTICAL,
                                      currentLayerNum,
                                      add);
              }
              break;

            // Aligned:
            case 1:
            default:
              tmpEl.createLine(v13, v23, v14, v24);
              dist = tmpEl.getDistanceToPoint(v10, v20, true);
              if(tmpEl.getRelAngleToPoint(v10, v20, true)<0.0) dist*=-1;
              graphic->addDimension(v13, v23, v14, v24,
                                    0.0, 0.0,
                                    dist,
                                    E_STRAIGHT,
                                    currentLayerNum,
                                    add);
              break;

            // Angle:
            case 2:
              if(true) {
                RElement tmpEl1, tmpEl2;
                tmpEl1.createLine(v13, v23, v14, v24);
                tmpEl2.createLine(v10, v20, v15, v25);
                tmpEl1.getIntersection(&tmpEl2,
                                       0, &vcx, &vcy, 0,0,0,0, false);
                vcr = mtGetDistance(vcx, vcy, v16, v26);

                if(mtGetDistance(vcx,vcy, v13,v23)<vcr) {
                  va1 = tmpEl1.getDirection1();
                }
                else {
                  va1 = tmpEl1.getDirection2();
                }

                if(mtGetDistance(vcx,vcy, v10,v20)<vcr) {
                  va2 = tmpEl2.getDirection1();
                }
                else {
                  va2 = tmpEl2.getDirection2();
                }
                graphic->addDimension(vcx, vcy, va1, va2,
                                      mtGetDistance(vcx, vcy, v13, v23),
                                      mtGetDistance(vcx, vcy, v10, v20),
                                      vcr,
                                      E_ROUNDOUT,
                                      currentLayerNum,
                                      add);
              }
              break;

            // Radius:
            case 4:
              graphic->addDimension(v10, v20, v15, v25,
                                    0.0, 0.0,
                                    v40,
                                    E_STRAIGHT|E_RADIUS,
                                    currentLayerNum,
                                    add);
              break;

            // Arrow:
            case 7:
              graphic->addDimension(v13, v23, v14, v24,
                                    0.0, 0.0, 0.0,
                                    E_STRAIGHT|E_ARROW,
                                    currentLayerNum,
                                    add);
              break;
          }
        }

        // ---------
        // Hatching:
        // ---------
        else if(dxfLine=="HATCH") {
          QString patternName="45";
          float patternScale=1.0;
          int numPaths=1;
          int numEdges=1;
          int nextObjectTyp=T_LINE;
          float v10=0.0, v20=0.0,
                v11=0.0, v21=0.0,
                v40=0.0, v50=0.0,
                v51=0.0;
          do {
            dxfCode=getBufLine();
            if(dxfCode) code=dxfCode.toInt();
            if(dxfCode && code!=0) {
              dxfLine=getBufLine();
              if(dxfLine) {
                switch(code) {
                  case  2:
                    patternName = dxfLine;
                    break;
                  case  6:  // style
                    graphic->setStyle(dxfLine);
                    break;
                  case  8:  // Layer
                    if(dxfLine!=lastLayer) {
                      currentLayerNum = graphic->addLayer(dxfLine.data());
                      lastLayer=dxfLine.data();
                    }
                    break;
                  case 10:  // Start point/center of boundary line/arc
                    v10=dxfLine.toFloat();
                    break;
                  case 20:  // Start point/center of boundary line/arc
                    v20=dxfLine.toFloat();
                    break;
                  case 11:  // End point of boundary line
                    v11=dxfLine.toFloat();
                    break;
                  case 21:  // End point of boundary line
                    v21=dxfLine.toFloat();
                    if(nextObjectTyp==T_LINE) {
                      int elnu=graphic->addLine(v10, v20, v11, v21, currentLayerNum, add);
                      graphic->elementAt(elnu)->setFlag(E_TAGGED);
                    }
                    break;
                  case 40:  // Radius of boundary entity
                    v40=dxfLine.toFloat();
                    break;
                  case 50:  // Start angle
                    v50=dxfLine.toFloat();
                    break;
                  case 51:  // End angle
                    v51=dxfLine.toFloat();
                    break;
                  case 73:  // Counterclockwise?
                    if(nextObjectTyp==T_ARC) {
                      int elnu=graphic->addArc(v10, v20, v40, v50, v51, (bool)dxfLine.toInt(), currentLayerNum, add);
                      graphic->elementAt(elnu)->setFlag(E_TAGGED);
                      //newEl = new RElement( graphic );
                      //newEl->createArc(v10, v20, v40, v50, v51, (bool)dxfLine.toInt());
                      //boundaryList.append(newEl);
                    }
                    break;
                  case 41:  // Scale
                    patternScale=dxfLine.toFloat();
                    break;
                  case 52:  // Angle

                    break;
                  case 70:  // Solid (=1) or pattern (=0)

                    break;
                  case 39:  // Thickness
                    graphic->setWidth(dxfLine.toInt());
                    break;
                  case 62:  // Color
                    graphic->setColor(dxfLine.toInt());
                    break;
                  case 91:  // Number of boundary paths (loops)
                    numPaths=dxfLine.toInt();
                    break;
                  case 92:  // Typ of boundary

                    break;
                  case 93:  // Number of edges in this boundary
                    numEdges=dxfLine.toInt();
                    break;
                  case 72:  // Edge typ
                    switch(dxfLine.toInt()) {
                      case 1: nextObjectTyp=T_LINE; break;
                      case 2: nextObjectTyp=T_ARC;  break;
                      default: break;
                    }
                    break;

                  default:
                    break;
                }
              }
            }
          }while(dxfCode && code!=0);

          graphic->addHatching(patternScale,
                               patternName,
                               currentLayerNum,
                               add);

          graphic->editDelete(false);

          /*RElement* el;
          for(el=graphic->elementFirst(); el!=0; el=graphic->elementNext()) {
            graphic->
          }*/

          /*for(el=boundaryList.first(); el!=0; el=boundaryList.next()) {

          }*/

        }

      }
    }while(dxfLine && dxfLine!="EOF");
    
    graphic->terminateAction();

    ret=true;
  }
  else {
    ret=false;
  }

  statusPanel()->delProgress();

  return ret;
}



// save a dxf file:
//
bool
RFileDxf::save()
{
  bool      ret;       // returned value
  int       i;         // counter
  RLayer* lay;         // Pointer to a layer in the graphic
  RElement *el;        // pointer which walks through elements

  statusPanel()->iniProgress(graphic->count(), RMES(293));

  fPointer = fopen(name(), "wt");
  if(fPointer!=NULL) {

    // Write DXF-header:
    
    // HEADER:
    //
    fprintf(fPointer,
            "  0\nSECTION\n  2\nHEADER\n");
            
    // LIMITS:
    //
    float minX, minY, maxX, maxY; // Borders of graphic
    graphic->getBorders(minX, maxY, maxX, minY);
    
    fprintf(fPointer,
            "  9\n$LIMMIN\n 10\n%f\n 20\n%f\n",
            minX, minY);
    fprintf(fPointer,
            "  9\n$LIMMAX\n 10\n%f\n 20\n%f\n",
            maxX, maxY);
    fprintf(fPointer,
            "  9\n$EXTMIN\n 10\n%f\n 20\n%f\n",
            minX, minY);
    fprintf(fPointer,
            "  9\n$EXTMAX\n 10\n%f\n 20\n%f\n",
            maxX, maxY);
    fprintf(fPointer, "  0\nENDSEC\n  0\nSECTION\n  2\nTABLES\n  0\n");
    fprintf(fPointer, "TABLE\n  2\nLTYPE\n 70\n     1\n  0\nLTYPE\n  2\n");
    fprintf(fPointer, "CONTINUOUS\n 70\n    64\n  3\nAusgez\n 72\n    65\n 73\n");
    fprintf(fPointer, "  0\n 40\n0.000000\n  0\nENDTAB\n  0\nTABLE\n");
    
    // LAYER:
    //
    fprintf(fPointer, "  2\nLAYER\n 70\n     1\n");
    for(i=0; i<graphic->countLayers(); ++i) {
      lay = graphic->getLayer(i);
      if(lay->getFlag(Y_USED)) {
        fprintf(fPointer, "  0\nLAYER\n  2\n");
        fprintf(fPointer, "%s\n", lay->getName());
        fprintf(fPointer, " 70\n    64\n 62\n     0\n  6\nCONTINUOUS\n");
      }
    }
    
    // HEADER END:
    //
    fprintf(fPointer, "  0\nENDTAB\n  0\nENDSEC\n  0\nSECTION\n  2\n");
    fprintf(fPointer, "BLOCKS\n  0\nENDSEC\n  0\nSECTION\n  2\nENTITIES\n");

    // ELEMENTS:
    //
    for(el=graphic->elementFirst(); el!=0; el=graphic->elementNext()) {
      //if(el->getFlag(E_VISIBLE)) {   // Save elements on invisible layers too!!
      if(el->getFlag(E_ACTIVE) && !el->getFlag(E_UNDO)) {

         // ONLY NOT TAGGED ELEMENTS!!!!!
         //
         //&& !el->getFlag(E_TAGGED)) {
         //
         // /////////////////////////////

        statusPanel()->setProgress(graphic->elementAt(), 200);

        switch(el->getElementTyp()) {
        
          // Point:
          //
          case T_POINT:
            fprintf(fPointer, "  0\nPOINT\n");
            fprintf(fPointer, "  8\n%s\n", graphic->getLayerName(el->getLayer()));
            fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
            fprintf(fPointer, " 10\n%f\n",
                              el->getX1());
            fprintf(fPointer, " 20\n%f\n",
                              el->getY1());

            fprintf(fPointer, " 39\n%d\n",
                              el->getWidth());
            fprintf(fPointer, " 62\n%d\n",
                              graphic->colorToNumber(el->getColor()));
            break;
            
          // Line:
          //
          case T_LINE:
            fprintf(fPointer, "  0\nLINE\n");
            fprintf(fPointer, "  8\n%s\n", graphic->getLayerName(el->getLayer()));
            fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
            fprintf(fPointer, " 10\n%f\n",
                              el->getX1());
            fprintf(fPointer, " 20\n%f\n",
                              el->getY1());
            fprintf(fPointer, " 11\n%f\n",
                              el->getX2());
            fprintf(fPointer, " 21\n%f\n",
                              el->getY2());
            fprintf(fPointer, " 39\n%d\n",
                              el->getWidth());
            fprintf(fPointer, " 62\n%d\n",
                              graphic->colorToNumber(el->getColor()));
            break;
            
          // Arc:
          //
          case T_ARC:
            fprintf(fPointer, "  0\nARC\n");
            fprintf(fPointer, "  8\n%s\n", graphic->getLayerName(el->getLayer()));
            fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
            fprintf(fPointer, " 10\n%f\n",
                              el->getCx());
            fprintf(fPointer, " 20\n%f\n",
                              el->getCy());
            fprintf(fPointer, " 40\n%f\n",
                              el->getCr());
            if(!el->getFlag(E_REVERSED)) {
              fprintf(fPointer, " 50\n%f\n",
                                el->getA1());
              fprintf(fPointer, " 51\n%f\n",
                                el->getA2());
            } 
            else {
              fprintf(fPointer, " 50\n%f\n",
                                el->getA2());
              fprintf(fPointer, " 51\n%f\n",
                                el->getA1());
            }
            fprintf(fPointer, " 39\n%d\n",
                              el->getWidth());
            fprintf(fPointer, " 62\n%d\n",
                              graphic->colorToNumber(el->getColor()));
            break;
            
          // Circle:
          //
          case T_CIRCLE:
            fprintf(fPointer, "  0\nCIRCLE\n");
            fprintf(fPointer, "  8\n%s\n", graphic->getLayerName(el->getLayer()));
            fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
            fprintf(fPointer, " 10\n%f\n",
                              el->getCx());
            fprintf(fPointer, " 20\n%f\n",
                              el->getCy());
            fprintf(fPointer, " 40\n%f\n",
                              el->getCr());
            fprintf(fPointer, " 39\n%d\n",
                              el->getWidth());
            fprintf(fPointer, " 62\n%d\n",
                              graphic->colorToNumber(el->getColor()));
            break;
            
          // Text:
          //
          case T_TEXT:
            if(true) {
              int   align=0;          // Align: (0=left, 1=center, 2=right)
              char  style[255];       // Style: (normal_ro#, cursive_ri#)
              int   tlc;              // Textline-Counter
              char  textLine[32768];  // Style: (normal_ro#, cursive_ri#)
              float radius;           // radius counter
              float xp, yp;           // position
              
              if(el->getFlag(E_CENTER)) align=1;
              if(el->getFlag(E_RIGHT )) align=2;
              
              style[0]='\0';
            
              // Style: (normal, cursive, ...)
              //
              strcat(style, RFonts::getRFonts()->getFontName((int)el->getFont()));
  
              if(el->getFlag(E_ROUNDOUT)) strcat(style, "_ro");
              if(el->getFlag(E_ROUNDIN )) strcat(style, "_ri");
              if(el->getFlag(E_FIXEDWIDTH)) strcat(style, "_fix");
  
              //if(el->getFlag(E_ROUNDIN|E_ROUNDOUT)) strcat(style, "#");

              xp = el->getX1();
              yp = el->getY1();
              
              radius = el->getCr();
              for(tlc=0; tlc<el->getNumberOfTextLines(); ++tlc) {
                 el->getTextLine(tlc, textLine, 32760);
                 if(strlen(textLine)>0) {
            
                  fprintf(fPointer, "  0\nTEXT\n");
                  fprintf(fPointer, "  8\n%s\n", graphic->getLayerName(el->getLayer()));
                  fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
                  fprintf(fPointer, " 10\n%f\n",
                                    xp);
                  fprintf(fPointer, " 20\n%f\n",
                                    yp);
                  fprintf(fPointer, " 40\n%f\n",
                                    el->getY2());
                  fprintf(fPointer, " 50\n%f\n",
                                    el->getA1());
                  fprintf(fPointer, " 72\n%d\n",
                                    align);
                  fprintf(fPointer, "  7\n%s",
                                    style);
                  
                  if(el->getFlag(E_ROUNDIN|E_ROUNDOUT)) {
                    fprintf(fPointer, "#%.6f", radius);
                  }
                  fprintf(fPointer, "#%.6f#%.6f",
                                    el->getX2(), el->getCx()); // Letterdist. / Worddist
                  fprintf(fPointer, "\n");
  
                  fprintf(fPointer, "  1\n%s\n",
                                    textLine);
                  fprintf(fPointer, " 39\n%d\n",
                                    el->getWidth());
                  fprintf(fPointer, " 62\n%d\n",
                                    graphic->colorToNumber(el->getColor()));
                }
  
                // Go to pos of next line:
                //
                if(el->getFlag(E_ROUNDIN)) {
                  radius-=el->getCy();
                }
                else {
                  if(el->getFlag(E_ROUNDOUT)) {
                    radius+=el->getCy();
                  }
                  else {
                    xp+=sin(el->getA1()/ARAD) * el->getCy();
                    yp-=cos(el->getA1()/ARAD) * el->getCy();
                  }
                }
              }
            }
            break;

          // Dimension:
          //
          case T_DIMENSION:

            fprintf(fPointer, "  0\nDIMENSION\n");
            fprintf(fPointer, "  8\n%s\n", graphic->getLayerName(el->getLayer()));
            fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
            fprintf(fPointer, "  1\n \n");

            if(el->getFlag(E_STRAIGHT)) {
              if(el->getFlag(E_HORIZONTAL)) {
                fprintf(fPointer, " 10\n%f\n",    // dim line pos x
                                  el->getX1());
                fprintf(fPointer, " 20\n%f\n",    // dim line pos y
                                  el->getDimDist());
              }
              else if(el->getFlag(E_VERTICAL)) {
                fprintf(fPointer, " 10\n%f\n",    // dim line pos x
                                  el->getDimDist());
                fprintf(fPointer, " 20\n%f\n",    // dim line pos y
                                  el->getY1());
              }
              else if(el->getFlag(E_RADIUS)) {
                fprintf(fPointer, " 10\n%f\n",    // dim center x
                                  el->getCx());
                fprintf(fPointer, " 20\n%f\n",    // dim center y
                                  el->getCy());
                fprintf(fPointer, " 15\n%f\n",    // dim center x
                                  el->getX1());
                fprintf(fPointer, " 25\n%f\n",    // dim center y
                                  el->getY1());
                fprintf(fPointer, " 40\n%f\n",    // dim radius
                                  el->getCr());
              }
              else {
                float ang = mtGetAngle(el->getX1(), el->getY1(), el->getX2(), el->getY2());
                fprintf(fPointer, " 10\n%f\n",    // dim line pos x
                                  el->getX1() + cos((ang+90.0)/ARAD) * el->getDimDist());
                fprintf(fPointer, " 20\n%f\n",    // dim line pos y
                                  el->getY1() + sin((ang+90.0)/ARAD) * el->getDimDist());
              }

              if(!el->getFlag(E_RADIUS)) {
                fprintf(fPointer, " 13\n%f\n",
                                  el->getX1());
                fprintf(fPointer, " 23\n%f\n",
                                  el->getY1());
                fprintf(fPointer, " 14\n%f\n",
                                  el->getX2());
                fprintf(fPointer, " 24\n%f\n",
                                  el->getY2());
              }

              if(el->getFlag(E_HORIZONTAL)) {
                fprintf(fPointer, " 50\n%f\n", 0.0);
                fprintf(fPointer, " 70\n%d\n", 0);
              }
              else if(el->getFlag(E_VERTICAL)) {
                fprintf(fPointer, " 50\n%f\n", 90.0);
                fprintf(fPointer, " 70\n%d\n", 0);
              }
              else if(el->getFlag(E_RADIUS)) {
                fprintf(fPointer, " 70\n%d\n", 4);
              }
              else if(el->getFlag(E_ARROW)) {
                fprintf(fPointer, " 70\n%d\n", 7);   // Not DXF compatible!!!
              }
              else {
                fprintf(fPointer, " 70\n%d\n", 1);
              }
            }
            else if(el->getFlag(E_ROUNDOUT)) {
              // 1st extension line
              fprintf(fPointer, " 13\n%f\n",
                                el->getCx() + cos(el->getA1()/ARAD) * el->getX1());
              fprintf(fPointer, " 23\n%f\n",
                                el->getCy() + sin(el->getA1()/ARAD) * el->getX1());
              fprintf(fPointer, " 14\n%f\n",
                                el->getCx() + cos(el->getA1()/ARAD) * el->getDimDist());
              fprintf(fPointer, " 24\n%f\n",
                                el->getCy() + sin(el->getA1()/ARAD) * el->getDimDist());

              // 2nd extension line
              fprintf(fPointer, " 10\n%f\n",
                                el->getCx() + cos(el->getA2()/ARAD) * el->getX2());
              fprintf(fPointer, " 20\n%f\n",
                                el->getCy() + sin(el->getA2()/ARAD) * el->getX2());
              fprintf(fPointer, " 15\n%f\n",
                                el->getCx() + cos(el->getA2()/ARAD) * el->getDimDist());
              fprintf(fPointer, " 25\n%f\n",
                                el->getCy() + sin(el->getA2()/ARAD) * el->getDimDist());

              // Def line:
              fprintf(fPointer, " 16\n%f\n",
                                el->getCx() + cos(el->getA1()/ARAD) * el->getDimDist());
              fprintf(fPointer, " 26\n%f\n",
                                el->getCy() + sin(el->getA1()/ARAD) * el->getDimDist());

              fprintf(fPointer, " 70\n%d\n",
                                2);
            }
            fprintf(fPointer, " 39\n%d\n",
                              el->getWidth());
            fprintf(fPointer, " 62\n%d\n",
                              graphic->colorToNumber(el->getColor()));
            break;

          // Hatching:
          //
          case T_HATCHING:
            fprintf(fPointer, "  0\nHATCH\n");
            fprintf(fPointer, "  8\n%s\n", graphic->getLayerName(el->getLayer()));
            fprintf(fPointer, "  6\n%s\n",
                              graphic->styleToName(el->getStyle()).latin1());
            fprintf(fPointer, "  1\n \n");
            fprintf(fPointer, "  2\n%s\n",
                              el->getText());
            fprintf(fPointer, " 70\n%d\n",
                              0);
            fprintf(fPointer, " 91\n%d\n",
                              1);

            // Save boundary:
            if(true) {
              for(int i=0; i<el->getSubElementNum(); ++i) {
                if(!el->getSubElement(i)->getFlag(E_VISIBLE)) {
                  switch(el->getSubElement(i)->getElementTyp()) {
                    case T_LINE:
                      fprintf(fPointer, " 72\n%d\n",
                                        1);
                      fprintf(fPointer, " 10\n%f\n",
                                        el->getSubElement(i)->getX1());
                      fprintf(fPointer, " 20\n%f\n",
                                        el->getSubElement(i)->getY1());
                      fprintf(fPointer, " 11\n%f\n",
                                        el->getSubElement(i)->getX2());
                      fprintf(fPointer, " 21\n%f\n",
                                        el->getSubElement(i)->getY2());
                      break;
  
                    case T_ARC:
                    case T_CIRCLE:
                      fprintf(fPointer, " 72\n%d\n",
                                        2);
                      fprintf(fPointer, " 10\n%f\n",
                                        el->getSubElement(i)->getCx());
                      fprintf(fPointer, " 20\n%f\n",
                                        el->getSubElement(i)->getCy());
                      fprintf(fPointer, " 40\n%f\n",
                                        el->getSubElement(i)->getCr());
                      fprintf(fPointer, " 50\n%f\n",
                                        el->getSubElement(i)->getA1());
                      fprintf(fPointer, " 51\n%f\n",
                                        el->getSubElement(i)->getA2());
                      fprintf(fPointer, " 73\n%d\n",
                                       el->getSubElement(i)->getFlag(E_REVERSED));
                      break;
                    default:
                      break;
                  }
                }
              }
            }

            fprintf(fPointer, " 75\n%d\n",
                              0);
            fprintf(fPointer, " 76\n%d\n",
                              1);
            fprintf(fPointer, " 52\n%f\n",
                              0.0);
            fprintf(fPointer, " 91\n%d\n",
                              1);
            fprintf(fPointer, " 41\n%f\n",
                              el->getCr());
            fprintf(fPointer, " 77\n%d\n",
                              0);
            fprintf(fPointer, " 39\n%d\n",
                              el->getWidth());
            fprintf(fPointer, " 62\n%d\n",
                              graphic->colorToNumber(el->getColor()));
            break;
            
          default:
            break;
        }
      }
    }

    // DXF-End:
    //
    fprintf(fPointer, "  0\nENDSEC\n  0\nEOF\n");
  
    fclose(fPointer);

    statusPanel()->delProgress();

    ret=true;
  }
  else {
    ret=false;
  }

  return ret;
  
}


// EOF



















