/***************************************************************************
                          rgraphic.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/


/****************************************************************************
** rgraphic.cpp 1998/08/24 A. Mustun RibbonSoft
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/


#include "rgraphic.h"

#include <stdlib.h>
#include <math.h>

#include <qapplication.h>
#include <qfileinfo.h>
#include <qpixmap.h>
#include <qmessagebox.h>


#include "ractdraw.h"
#include "ractedit.h"
#include "racttag.h"
#include "ractzoom.h"
#include "rappwin.h"
#include "rcadwidget.h"
#include "relement.h"
#include "rbehaviordef.h"
#include "rfilecxf.h"
#include "rfiledxf.h"
#include "rfilegbx.h"
#include "rfileplt.h"
#include "rfontdialog.h"
#include "rfonts.h"
#include "rconfig.h"
#include "rlayoutdef.h"
#include "rlog.h"
#include "rmacros.h"
#include "rmath.h"
#include "rpainter.h"
#include "rsnap.h"
#include "rxpmgraphic.h"

#ifdef DEF_CAM_EXPERT
#include "rfilenc.h"
#include "ractcam.h"
#endif

// Constructor:
//
RGraphic::RGraphic(QWidget* _parent, 
                   const char* _name, 
                   WFlags _f)
:QFrame(_parent, _name, _f)
{
  int i;

  setBackgroundColor(QColor(255, 255, 255));
  setFrameStyle(WinPanel|Sunken);
  setLineWidth(2);
  setMouseTracking(true);
  element.setAutoDelete(true);
  layer.setAutoDelete(true);
  fileName="";
  menuId=32768;
  zoom=10.0;
  relZeroX=0.0;
  relZeroY=0.0;
  offsetX=50;
  offsetY=240;
  printerFactorX=1.0;
  printerFactorY=1.0;

  // Key coords (used by REVENT_COORDSET):
  keyX = keyY = 0.0;
  enableKeyCoords=false;
  
  // Set original flags:
  //
  setFlags(G_GRID|G_GRIDWIDTH|G_ZERO);

  // Preview / highlighting:
  //
  preview.setAutoDelete(true);
  borderElement.setGraphic(this);
  previewElement.setGraphic(this);
  prevVisible=false;
  markVisible=false;
  markEnabled=false;
  prevElementVisible=false;
  prevElementsVisible=false;
  highlightElementVisible=false;
  snapFilterTyp=T_ALL;
  snapFilterException=0;

  // Layers:
  //
  autoUpdateLayerList=true;
  selectedLayer=0;

  // Info:
  //
  autoUpdateInfo=true;
  autoUpdateGraphic=true;

  // Behavior / Action Handlers:
  //
  eventsEnabled=true;
  busy=false;
  previewUpdate=true;
  active=false;
  updateBroken=false;
  lastAction=-1;
  lastState=-1;
  lastCursor=0;
  currentAction=-1;
  currentState=-1;
  
  undoP   = 0;
  redoP   = 0;
  nextP   = 1;
  for(i=0; i<256; ++i) {
    undoReg[i] = '0';
  }
    
  behavior=BVR_NO;
  lastBehavior=BVR_NO;

  actZoom = new RActZoom(this);
  actDraw = new RActDraw;
  actEdit = new RActEdit;
  actTag  = new RActTag;
#ifdef DEF_CAM_EXPERT
  actCam  = new RActCam;
#endif
  snap    = new RSnap;

  // Simulation:
  //
  drawing=false;
  printing=false;
  delay=0;
  delayPosition=DEF_SPEEDRES;
  smooth=false;
  noMoreMemory=false;
  setFlag(G_GRID);

  currentColor=Qt::black;
  currentWidth=0;
  currentStyle=Qt::SolidLine;

  currentGridWidth=10.0;

  // CAM-data:
  //
  initialized=false;
  machineGenerator = RCONFIG->getSetting("CAM:MachineGenerator");
  extension="";
  mode='a';
  optimization=true;
  contourCenter=false;
  sorting='n';
  direction='n';
  pointing=false;

  contourStartInHeader=false;
  layerStartInHeader=false;
  numberStart="10";
  numberStep="10";
  maxArcAngle="0.0";
  maxArcLength="0.0";
  tolerance="0.001";
  digits="3";
  factor="1.0";
  pointingAngle=60.0;
  pointingDepth=-2.0;
  pointingLimit=5.0;
  pointingMaxLength=100.0;

  // CAD-Data:
  //
  fontNumber=RFonts::getRFonts()->getFontNumber("normal");
  if(fontNumber<0) fontNumber=0;
  fontProportional=true;
  fontHeight=10.0;
  fontLetterSpacing=2.0;
  fontWordSpacing=6.0;
  fontLineDistance=14.0;
  fontAngle=0.0;
  fontRadius=0.0;
  fontFlags.setFlags(E_STRAIGHT|E_LEFT);

  contNum=0;

  clearParameters();
  clearLayerParameters();
}



// Destructor:
//
RGraphic::~RGraphic()
{
  deleteElements();
}



//
//
// Copy functions:
//
//


// Copy all (objects, layers) from graphic _source:
//
bool    
RGraphic::copyAllFrom(RGraphic* _source)
{
  if(copyPropertiesFrom(_source)     &&
     copyVisibleObjectsFrom(_source) &&
     copyLayerListFrom(_source)         ) {
    graphicHasChanged();
    return true;
  }
  else {
    return false;
  }
}



// Copy tagged (objects, layers) from graphic _source:
//
bool
RGraphic::copyTaggedFrom(RGraphic* _source, float _ox, float _oy)
{
  if(copyPropertiesFrom(_source)                     &&
     copyVisibleObjectsFrom(_source, true, _ox, _oy) &&
     copyLayerListFrom(_source)                         ) {

    graphicHasChanged();
    return true;
  }
  else {
    return false;
  }
}



// Copy all properties from an other graphic:
//   exceptions: preview, previewElement, prevVisible,
//   autoUpdateLayerList, Actions, States, Behaviors,
//   prevElementVisible, markVisible, 
//
bool
RGraphic::copyPropertiesFrom(RGraphic* _source)
{
  if(_source) {
    fileName = _source->getFileName();
    menuId   = 32768;
    zoom     = _source->getZoom();
    offsetX  = _source->getOffsetX();
    offsetY  = _source->getOffsetY();
  
    initialized          = _source->isInitialized();
    machineGenerator     = _source->getMachineGenerator().data();
    extension            = _source->getExtension().data();
    mode                 = _source->getMode();
    optimization         = _source->getOptimization();
    contourCenter        = _source->getContourCenter();
    sorting              = _source->getSorting();
    direction            = _source->getDirection();
    pointing             = _source->getPointing();
    contourStartInHeader = _source->getContourStartInHeader();
    layerStartInHeader   = _source->getLayerStartInHeader();
    numberStart.sprintf("%d", _source->getNumberStart());
    numberStep.sprintf("%d", _source->getNumberStep());
    maxArcAngle.sprintf("%f", _source->getMaxArcAngle());
    maxArcLength.sprintf("%f", _source->getMaxArcLength());
    tolerance.sprintf("%f", _source->getTolerance());
    digits.sprintf("%d", _source->getDigits());
    factor.sprintf("%f", _source->getFactor());
    pointingAngle        = _source->getPointingAngle();
    pointingDepth        = _source->getPointingDepth();
    pointingLimit        = _source->getPointingLimit();
    pointingMaxLength    = _source->getPointingMaxLength();
    contNum              = _source->getContNum();
    
    int i, k;
    
    parameterString      = _source->getParameterString().data();
    for(i=0; i<DEF_MAXPARAMETERS; ++i) {
      parameter[i] = _source->getParameter(i).data();
    }
    
    layerParameterString = _source->getLayerParameterString().data();
    for(k=0; k<DEF_MAXLAYERS; ++k) {
      for(i=0; i<DEF_MAXPARAMETERS; ++i) {
        layerParameter[k][i] = _source->getLayerParameter(k, i).data();
      }
    }
    return true;
  }
  else {
    return false;
  }
}
  
  
  
// Copy all visible objects from an other graphic:
//
// return : true: successful
//        : false: no graphic or no visible elements
//
bool    
RGraphic::copyVisibleObjectsFrom(RGraphic* _source, bool _tagged, float _ox, float _oy)
{
  RElement* el;
  RElement* newEl;
  bool offset=true;

  if(mtCompFloat(_ox, 0.0, 0.000001) && mtCompFloat(_oy, 0.0, 0.000001)) {
    offset=false;
  }

  if(_source) {
    for(el=_source->elementFirst(); el!=0; el=_source->elementNext()) {
      if(el->getFlag(E_VISIBLE) && (!_tagged || el->getFlag(E_TAGGED))) {
        newEl = new RElement(this);
	      if(newEl) {
          newEl->getMeasuresFrom(el);
          if(offset) newEl->move(_ox, _oy);
          element.append(newEl);
        }
        else {
          badAllocation();
        }
      }
    }
    graphicHasChanged();
    return true;
  }
  else {
    return false;
  }
}



// Copy all elements from an existing list
//
bool    
RGraphic::copyVisibleObjectsFrom(QList<RElement> &_list)
{
  RElement *el;          // pointer which walks through elements

  if(!element.isEmpty()) element.clear();
  
  if(!_list.isEmpty()) {
    for(el=_list.first(); el!=0; el=_list.next()) {
      if(el->getFlag(E_VISIBLE)) {
        element.append(el);
      }
    }
    graphicHasChanged();
    return true;
  }
  else {
    return false;
  }
}



// Copy layer list from graphic _source:
//
bool    
RGraphic::copyLayerListFrom(RGraphic* _source)
{
  bool ret=false;
  RLayer* lay;
  RLayer* newLay;

  if(_source) {
    for(lay=_source->layerFirst(); lay!=0; lay=_source->layerNext()) {
      if(lay->getFlag(Y_USED)) {
        newLay = new RLayer;
        newLay->setName(lay->getName());
        newLay->setNumber(lay->getNumber());
        newLay->setFlags(lay->getFlags());
        layer.append(newLay);
        ret=true;
      }
    }
  }

  return ret;
}



//
//
// Reset functions:
//
//



// reset converted flag from all objects:
//
void 
RGraphic::resetConvertedFlags()
{
  RElement *el;          // pointer which walks through elements
  RLayer *lay;           // pointer which walks through layers
  
  for(el=element.first(); el!=0; el=element.next()) {
    el->delFlag(E_CONV);
  }
  
  for(lay=layer.first(); lay!=0; lay=layer.next()) {
    lay->delFlag(Y_CONV);
  }
}



// reset contour flags from all objects:
//   (E_CONTBEG, E_CONTEND)
//
void 
RGraphic::resetContourFlags()
{
  RElement *el;          // pointer which walks through elements

  for(el=element.first(); el!=0; el=element.next()) {
    el->delFlag(E_CONTBEG);
    el->delFlag(E_CONTEND);
  }
}



// Show event:
//   (SLOT)
//
void
RGraphic::show()
{
  QFrame::show();
}


// Get Extension:
//   extension may be a parameter ($e)
//   ->translate it if so...
//
QString
RGraphic::getExtension()
{ 
  if(extension.at(0)=='$') {
    int ind=parameterString.find(extension.at(1));
    if(ind>=0 && ind<DEF_MAXPARAMETERS) {
      extension=parameter[ind].data();
    }
  }
  
  return extension; 
}



// get number start (may be a parameter):
//
int 
RGraphic::getNumberStart() const 
{ 
  if(!numberStart.isEmpty()) {
    if(numberStart.at(0)=='$') {
      int ind=parameterString.find(numberStart.at(1));
      if(ind>=0 && ind<DEF_MAXPARAMETERS) {
        if(!parameter[ind].isEmpty()) {
          return parameter[ind].toInt();
        }
      }
    }
    else {
      return numberStart.toInt();
    }
  }
  
  return 10; 
}



// get number step (may be a parameter):
//
int
RGraphic::getNumberStep() const 
{ 
  if(!numberStep.isEmpty()) {
    if(numberStep.at(0)=='$') {
      int ind=parameterString.find(numberStep.at(1));
      if(ind>=0 && ind<DEF_MAXPARAMETERS) {
        if(!parameter[ind].isEmpty()) {
          return parameter[ind].toInt();
        }
      }
    }
    else {
      return numberStep.toInt();
    }
  }
  
  return 10; 
}



// get max angle for arcs (may be a parameter):
//
float 
RGraphic::getMaxArcAngle() const 
{ 
  if(!maxArcAngle.isEmpty()) {
    if(maxArcAngle.at(0)=='$') {
      int ind=parameterString.find(maxArcAngle.at(1));
      if(ind>=0 && ind<DEF_MAXPARAMETERS) {
        if(!parameter[ind].isEmpty()) {
          return parameter[ind].toFloat();
        }
      }
    }
    else {
      return maxArcAngle.toFloat();
    }
  }
  
  return 0.0; 
}



// get max length for arcs (may be a parameter):
//
float 
RGraphic::getMaxArcLength() const 
{ 
  if(!maxArcLength.isEmpty()) {
    if(maxArcLength.at(0)=='$') {
      int ind=parameterString.find(maxArcLength.at(1));
      if(ind>=0 && ind<DEF_MAXPARAMETERS) {
        if(!parameter[ind].isEmpty()) {
          return parameter[ind].toFloat();
        }
      }
    }
    else {
      return maxArcLength.toFloat();
    }
  }
  
  return 0.0; 
}



// get tolerance (may be a parameter):
//
float 
RGraphic::getTolerance() const 
{ 
  if(!tolerance.isEmpty()) {
    if(tolerance.at(0)=='$') {
      int ind=parameterString.find(tolerance.at(1));
      if(ind>=0 && ind<DEF_MAXPARAMETERS) {
        if(!parameter[ind].isEmpty()) {
          return parameter[ind].toFloat();
        }
      }
    }
    else {
      return tolerance.toFloat();
    }
  }
  
  return 0.001; 
}



// get digits (may be a parameter):
//
int 
RGraphic::getDigits() const 
{ 
  if(!digits.isEmpty()) {
    if(digits.at(0)=='$') {
      int ind=parameterString.find(digits.at(1));
      if(ind>=0 && ind<DEF_MAXPARAMETERS) {
        if(!parameter[ind].isEmpty()) {
          return parameter[ind].toInt(); 
        }
      }
    }
    else {
      return digits.toInt();
    }
  }
  
  return 3; 
}


// get factor (may be a parameter):
//
float 
RGraphic::getFactor() const 
{
  if(!factor.isEmpty()) {
    if(factor.at(0)=='$') {
      int ind=parameterString.find(factor.at(1));
      if(ind>=0 && ind<DEF_MAXPARAMETERS) {
        if(!parameter[ind].isEmpty()) {
          return parameter[ind].toFloat();
        }
      }
    }
    else {
      return factor.toFloat();
    }
  }
  
  return 1.0; 
}



//
//
// User Actions:
//
//


// Load a file into graphic:
//
bool
RGraphic::loadFile(const char* _fileName, bool add)
{
  QFileInfo fileInfo(_fileName);
  bool ret=false;
  autoUpdateLayerList=false;
  autoUpdateInfo=false;

  QColor oriColor=currentColor;
	int oriWidth=currentWidth;
	Qt::PenStyle oriStyle=currentStyle;
  
  if(fileInfo.extension().upper()=="DXF") {
    RFileDxf fileDxf(_fileName, this);
    if(fileDxf.load(add)) {
      setFileName(_fileName);
      ret=true;
    }
  }

  else if(fileInfo.extension().upper()=="CXF") {
    RFileCxf fileCxf(_fileName, this);
    if(fileCxf.load()) {
      setFileName(_fileName);
      ret=true;
    }
  }

  else if(fileInfo.extension().upper()=="GBX") {
    RFileGbx fileGbx(_fileName, this);
    if(fileGbx.load()) {
      setFileName(_fileName);
      ret=true;
    }
  }

#ifdef DEF_CAM_EXPERT
  else if(fileInfo.extension().upper()=="NC"  ||
          fileInfo.extension().upper()=="CNC" ||
          fileInfo.extension().upper()=="D"      ) {
    RFileNc fileNc(_fileName, this);
    if(fileNc.load(add)) {
      setFileName(_fileName);
      ret=true;
    }
  }
#endif

  else if(fileInfo.extension().upper()=="PLT" ||
          fileInfo.extension().upper()=="HP"     ) {
    RFilePlt filePlt(_fileName, this);
    if(filePlt.load(add)) {
      setFileName(_fileName);
      ret=true;
    }
  }

  setColor(oriColor);
  setWidth(oriWidth);
  setStyle(oriStyle);

  autoUpdateLayerList=true;
  autoUpdateInfo=true;
  emit updateLayers();
  graphicHasChanged();
  delFlag( G_CHANGED );  // Reset changed flag
  return ret;
}



// Save a file from graphic:
//
bool
RGraphic::saveFile(const char* _fileName)
{
  QFileInfo fileInfo(_fileName);
  bool ret=false;
  
  if(fileInfo.extension().upper()=="DXF") {
    RFileDxf fileDxf(_fileName, this);
    if(fileDxf.save()) {
      setFileName(_fileName);
      ret=true;
    }
  }

  else if(fileInfo.extension().upper()=="CXF") {
    RFileCxf fileCxf(_fileName, this);
    if(fileCxf.save()) {
      setFileName(_fileName);
      ret=true;
    }
  }

  return ret;
}



// Get graphics snapper:
//
RSnap* 
RGraphic::snapper()
{
  return snap;
}


/**
 * Return clipboard pointer
 */
RGraphic*
RGraphic::clipboard()
{
  return RAppWin::getRAppWin()->getClipboard();
}


// Delete all elements
//
void 
RGraphic::deleteElements()
{
  element.clear();
  graphicHasChanged();
}


// Set current element:
//
int
RGraphic::setCurrentElement(const RElement* _curEl)
{
  return element.findRef(_curEl);
}



// Exchange two elements in the list (slow):
//
void
RGraphic::exchangeElements(RElement* _el1, RElement* _el2)
{
  RElement tmpEl(this);
  
  if(_el1 && _el2) {
    tmpEl.getMeasuresFrom(_el1);
    _el1->getMeasuresFrom(_el2);
    _el2->getMeasuresFrom(&tmpEl);
    graphicHasChanged();
  }
}



//
//
// Add Elements:
//
//


// Add an element from an existing one:
//
int
RGraphic::addElement(RElement* _el,
                     bool _setCreationNum)
{
  if(_el && _el->getFlag(E_ACTIVE)) {
    RElement* el=new RElement(this);
    if(el) {
      el->getMeasuresFrom(_el);
      if(_setCreationNum) el->setCreationNumber(nextP);
      if(currentColor!=Qt::black) el->setColor(currentColor);
      if(currentWidth!=0) el->setWidth(currentWidth);
      if(currentStyle!=Qt::SolidLine) el->setStyle(currentStyle);
      element.append(el);
      int ret=element.at();
      graphicHasChanged();
      return ret;
    }
    else {
      badAllocation();
    }
  }
  return -1;
}



// Add a point to the graphic (in existing memory)
//
int
RGraphic::addPoint(float _x1, float _y1, 
                   byte _layer,
                   bool _setCreationNum)
{
  if(_layer==0) _layer=checkLayer(_layer);
  RElement* el=new RElement(this);
  if(el) {
    el->createPoint(_x1, _y1, _layer);
    if(_setCreationNum) el->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el->setColor(currentColor);
    if(currentWidth!=0) el->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el->setStyle(currentStyle);
    element.append(el);
    int ret=element.at();
    graphicHasChanged();
    return ret;
  }
  else {
    badAllocation();
    return -1;
  }
}



// Add a line to the graphic (in existing memory)
//
int
RGraphic::addLine(float _x1, float _y1, 
                  float _x2, float _y2,
                  byte _layer,
                  bool _setCreationNum)
{
  if(_layer==0) _layer=checkLayer(_layer);
  RElement* el=new RElement(this);
  if(el) {
    el->createLine(_x1, _y1, _x2, _y2, _layer);
    if(_setCreationNum) el->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el->setColor(currentColor);
    if(currentWidth!=0) el->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el->setStyle(currentStyle);
    element.append(el);
    int ret=element.at();
    graphicHasChanged();
    return ret;
  }
  else {
    badAllocation();
    return -1;
  }
}



// Add an arc to the graphic (in existing memory)
//
int
RGraphic::addArc(float _cx, float _cy, 
                 float _cr, 
                 float _a1, float _a2, 
                 bool _reversed, 
                 byte _layer,
                 bool _setCreationNum)
{
  if(_layer==0) _layer=checkLayer(_layer);
  RElement* el=new RElement(this);
  if(el) {
    el->createArc(_cx, _cy, _cr, _a1, _a2, _reversed, _layer);
    if(_setCreationNum) el->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el->setColor(currentColor);
    if(currentWidth!=0) el->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el->setStyle(currentStyle);
    element.append(el);
    int ret=element.at();
    graphicHasChanged();
    return ret;
  }
  else {
    badAllocation();
    return -1;
  }
}



// Add an arc to the graphic (in existing memory)
//   Given by 3 points on arc
//
int
RGraphic::addArcPPP(float _x1, float _y1,
                    float _x2, float _y2,
                    float _x3, float _y3,
                    bool _reversed,
                    byte _layer,
                    bool _setCreationNum)
{
  if(_layer==0) _layer=checkLayer(_layer);
  RElement* el=new RElement(this);
  if(el) {
    el->createArcPPP(_x1, _y1, _x2, _y2, _x3, _y3, _reversed, _layer);
    if(_setCreationNum) el->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el->setColor(currentColor);
    if(currentWidth!=0) el->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el->setStyle(currentStyle);
    element.append(el);
    int ret=element.at();
    graphicHasChanged();
    return ret;
  }
  else {
    badAllocation();
    return -1;
  }
}



// Add a circle to the graphic (in existing memory)
//
int
RGraphic::addCircle(float _cx, float _cy, 
                    float _cr,
                    float _a1, float _a2,
                    bool _reversed, 
                    byte _layer,
                    bool _setCreationNum)
{
  if(_layer==0) _layer=checkLayer(_layer);
  RElement* el=new RElement(this);
  if(el) {
    el->createCircle(_cx, _cy, _cr, _a1, _a2, _reversed, _layer);
    if(_setCreationNum) el->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el->setColor(currentColor);
    if(currentWidth!=0) el->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el->setStyle(currentStyle);
    element.append(el);
    int ret=element.at();
    graphicHasChanged();
    return ret;
  }
  else {
    badAllocation();
    return -1;
  }
}



// Add a polyline element (arc or line) to the graphic 
//   (in existing memory)
//
int
RGraphic::addPolylineElement(float _x1, float _y1,
                             float _x2, float _y2,
                             float _ab, 
                             byte _layer,
                             bool _setCreationNum)
{
  if(_layer==0) _layer=checkLayer(_layer);
  RElement* el=new RElement(this);
  if(el) {
    el->createPolylineElement(_x1, _y1, _x2, _y2, _ab, _layer);
    if(_setCreationNum) el->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el->setColor(currentColor);
    if(currentWidth!=0) el->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el->setStyle(currentStyle);
    element.append(el);
    int ret=element.at();
    graphicHasChanged();
    return ret;
  }
  else {
    badAllocation();
    return -1;
  }
}



// Add a rectangle to the graphic (in existing memory)
//
//   return: first added element
// 
int
RGraphic::addRect(float _x1, float _y1, 
                  float _x2, float _y2,
                  byte _layer,
                  bool _setCreationNum)
{
  if(_layer==0) _layer=checkLayer(_layer);
  RElement* el1=new RElement(this);
  RElement* el2=new RElement(this);
  RElement* el3=new RElement(this);
  RElement* el4=new RElement(this);
  
  if(el1 && el2 && el3 && el4) {
  
    el1->createLine(_x1, _y1, _x2, _y1, _layer);
    if(_setCreationNum) el1->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el1->setColor(currentColor);
    if(currentWidth!=0) el1->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el1->setStyle(currentStyle);
    el2->createLine(_x2, _y1, _x2, _y2, _layer);
    if(_setCreationNum) el2->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el2->setColor(currentColor);
    if(currentWidth!=0) el2->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el2->setStyle(currentStyle);
    el3->createLine(_x2, _y2, _x1, _y2, _layer);
    if(_setCreationNum) el3->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el3->setColor(currentColor);
    if(currentWidth!=0) el3->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el3->setStyle(currentStyle);
    el4->createLine(_x1, _y2, _x1, _y1, _layer);
    if(_setCreationNum) el4->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el4->setColor(currentColor);
    if(currentWidth!=0) el4->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el4->setStyle(currentStyle);
  
    element.append(el1);
    int ret=element.at();
    element.append(el2);
    element.append(el3);
    element.append(el4);
  
    graphicHasChanged();
    return ret;
  }
  else {
    badAllocation();
    return -1;
  }
}



// Add a text to the graphic:
//
int 
RGraphic::addText(float _x, float _y,
                  int   _font,
                  const unsigned char* _text,
                  unsigned int _flags,
                  float _textheight,
                  float _angle,
                  float _radius,
                  float _letterspace,
                  float _wordspace,
                  float _linedistance,
                  byte _layer,
                  bool _setCreationNum)
{
  if(_layer==0) _layer=checkLayer(_layer);
  RElement* el=new RElement(this);
  if(el) {
    el->createText(_x, _y,
                   _font,
                   _text,
                   _flags,
                   _textheight,
                   _angle,
                   _radius,
                   _letterspace,
                   _wordspace,
                   _linedistance,
                   _layer);
    if(_setCreationNum) el->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el->setColor(currentColor);
    if(currentWidth!=0) el->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el->setStyle(currentStyle);
    element.append(el);
    int ret=element.at();
    graphicHasChanged();
    return ret;
  }
  else {
    badAllocation();
    return -1;
  }
}


/**
 * Add a dimension to the given line to the graphic:
 *   Possible flags: E_HORIZONTAL / E_VERTICAL
 *                   E_STRAIGHT / E_ROUNDOUT
 */
int
RGraphic::addDimension(float _v1, float _v2, float _v3, float _v4,
                       float _v5, float _v6,
                       float _pos,
                       uint _flags,
                       byte _layer,
                       bool _setCreationNum)
{
  if(_layer==0) _layer=checkLayer(_layer);
  RElement* el=new RElement(this);
  if(el) {
    el->createDimension(_v1, _v2, _v3, _v4,
                        _v5, _v6,
                        _pos,
                        _flags,
                        _layer);
    if(_setCreationNum) el->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el->setColor(currentColor);
    if(currentWidth!=0) el->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el->setStyle(currentStyle);
    element.append(el);
    int ret=element.at();
    graphicHasChanged();
    return ret;
  }
  else {
    badAllocation();
    return -1;
  }
}



/**
 * Add a hatching to the graphic:
 */
int
RGraphic::addHatching(float _factor, QString _pattern,
                      byte _layer,
                      bool _setCreationNum)
{
  if(_layer==0) _layer=checkLayer(_layer);
  RElement* el=new RElement(this);
  if(el) {
    // Create hatch based on the tagged elements in the graphic:
    el->createHatching(_factor, _pattern, _layer);
    if(_setCreationNum) el->setCreationNumber(nextP);
    if(currentColor!=Qt::black) el->setColor(currentColor);
    if(currentWidth!=0) el->setWidth(currentWidth);
    if(currentStyle!=Qt::SolidLine) el->setStyle(currentStyle);

    element.append(el);
    int ret=element.at();
    graphicHasChanged();
    return ret;
  }
  else {
    badAllocation();
    return -1;
  }
}



// Add a rapidmove to the graphic (in existing memory)
//
int
RGraphic::addRapid(float _x1, float _y1, 
                   byte _layer,
                   bool /*_setCreationNum*/)
{
  if(_layer==0) _layer=checkLayer(_layer);
  RElement* el=new RElement(this);
  if(el) {
    el->createRapid(_x1, _y1, _layer);
    element.append(el);
    int ret=element.at();
    graphicHasChanged();
    return ret;
  }
  else {
    badAllocation();
    return -1;
  }
}







// Do something with the single elements of text, rectangle, ...
//
//  _add:    Add elements to this graphic
//  _draw:   Draw elements
//  _border: Get borders
//  _left:   ptr to left
//  _top:    ptr to top
//  _right:  ptr to right
//  _bottom: ptr to bottom
//  _prev:   Preview element
//  _curr:   Use current settings for drawing
//
void
RGraphic::singleElements(RElement* _el,
                         bool _add,
                         bool _draw,
                         RPainter* _paint,
                         bool _border,
                         float* _left,
                         float* _top,
                         float* _right,
                         float* _bottom,
                         bool _setCreationNum,
                         bool _terminateAction)
{
  if(!_el) return;

  // Valid object:
  //
  if(_border) {
    if(_left)   *_left = DEF_AREAMAX;
    if(_top)    *_top = DEF_AREAMIN;
    if(_right)  *_right = DEF_AREAMIN;
    if(_bottom) *_bottom = DEF_AREAMAX;
  }

  // Text:
  //
  if(_el->getElementTyp()==T_TEXT &&
     _el->getText() &&
     strlen((char*)_el->getText())>0          ) {

    float curX,                 // Pos. virtual Cur. X
          curY;                 //                   Y
    float curA,                 // Pos. virtual Cur. Angle
          curAM,                //                   Angle in middle of letter
          curR;                 //                   Radius
    float lineWidth;            // width of line
    float letterWidth;          // width of letter
    float lWidthA;              // Letter width (angle)
    float sWidthA;              // Letter space width (angle)
    bool  isFirst=true;         // This letter / char is the first of a new line (align)
    bool  lastCharWasALetter=false; // Last char was a letter (with a letter Space)
    int   lCounter=1;           // counter for line number (starting with 1)
    int   i;                    // counter for letters
    uint  textMode;             // text mode
    uint  textAlign;            // text alignment
    RFonts* fnts=RFonts::getRFonts();  // Pointer to fonts

    textMode = _el->getTextMode();
    textAlign = _el->getTextAlign();

    switch(textMode) {

      // Straight text:
      // --------------
      case E_STRAIGHT:
        curX=_el->getX1();
        curY=_el->getY1();

        // Create the text:
        //
        for(i=0; i<(int)strlen((char*)_el->getText()); ++i) {
          switch(_el->getText()[i]) {
            case 10:
              // Line feed:
              //
              curY = _el->getY1()
                     + sin((_el->getA1()-90.0)/ARAD) * lCounter*_el->getCy();
              curX = _el->getX1()
                     + cos((_el->getA1()-90.0)/ARAD) * lCounter*_el->getCy();

              ++lCounter;
              isFirst=true;
              lastCharWasALetter=false;
              break;

            case 13:
              // do nothing
              //
              break;

            default:
              // Alignment:
              //
              if(isFirst) {
                lineWidth=fnts->getTextLineWidth(_el->getFont(),
                                                 _el->getText(),
                                                 i,
                                                 _el->getTextHeight(),
                                                 _el->getLetterSpacing(),
                                                 _el->getWordSpacing(),
                                                 _el->getFixedWidth());
                switch(textAlign) {
                  // Center:
                  //
                  case E_CENTER:
                    curX -= cos(_el->getA1()/ARAD) * lineWidth/2.0;
                    curY -= sin(_el->getA1()/ARAD) * lineWidth/2.0;
                    break;
                  // Right:
                  //
                  case E_RIGHT:
                    curX -= cos(_el->getA1()/ARAD) * lineWidth;
                    curY -= sin(_el->getA1()/ARAD) * lineWidth;
                    break;
                  default:
                    break;
                }
              }

              // Add elements for this letter:
              //
              // Space:
              //
              if(_el->getText()[i]==' ') {
                letterWidth = _el->getWordSpacing();
                if(lastCharWasALetter) letterWidth -= _el->getLetterSpacing();

                if(_el->getFlag(E_FIXEDWIDTH)) {
                  letterWidth = fnts->getLetterWidth(_el->getFont(),
                                                     'W',
                                                     _el->getTextHeight())
                                + _el->getLetterSpacing();
                }

                curX += cos(_el->getA1()/ARAD)*letterWidth;
                curY += sin(_el->getA1()/ARAD)*letterWidth;
                lastCharWasALetter=false;
              }
              else {
                // Create letter:
                //
                int elNum = fnts->getLetterElementNumber(_el->getFont(),
                                                         _el->getText()[i]);

                int k;
                RElement tmpEl(this);
                for(k=0; k<elNum; ++k) {
                  fnts->getLetterElement(_el->getFont(),
                                         _el->getText()[i],
                                         _el->getTextHeight(),
                                         curX, curY,
                                         _el->getA1(),
                                         //_el->getTextAlign(),
                                         E_LEFT,
                                         k,
                                         &tmpEl);
                  tmpEl.getAttribFrom(_el);

                  // Add element to this graphic:
                  //
                  if(_add) {
                    RElement* newEl = new RElement(this);
                    if(newEl) {
                      newEl->getMeasuresFrom(&tmpEl);
                      newEl->setLayer(_el->getLayer());
                      if(_setCreationNum) newEl->setCreationNumber(nextP);
                      element.append(newEl);
                    }
                    else {
                      badAllocation();
                    }
                  }

                  // Draw element:
                  //
                  else if(_draw && _paint) {
                    if(tmpEl.isOnScreen() || printing) {
                      tmpEl.fastDraw(_paint);
                    }
                  }

                  // Get borders:
                  //
                  else if(_border) {
                    if(_left   && tmpEl.left()  < *_left  ) *_left   = tmpEl.left();
                    if(_top    && tmpEl.top()   > *_top   ) *_top    = tmpEl.top();
                    if(_right  && tmpEl.right() > *_right ) *_right  = tmpEl.right();
                    if(_bottom && tmpEl.bottom()< *_bottom) *_bottom = tmpEl.bottom();
                  }

                }

                // Letter space:
                //
                if(elNum>0) {
                  unsigned char letter=_el->getText()[i];
                  if(_el->getFlag(E_FIXEDWIDTH)) letter='W';   // Fixed width for all lettters
                  letterWidth = fnts->getLetterWidth(_el->getFont(),
                                                     letter,
                                                     _el->getTextHeight())
                                + _el->getLetterSpacing();

                  curX += cos(_el->getA1()/ARAD) * letterWidth;
                  curY += sin(_el->getA1()/ARAD) * letterWidth;
                  lastCharWasALetter=true;
                }
              }

              isFirst=false;
              break;
          }
        }
        break;


      // Round in text:
      // --------------
      case E_ROUNDIN:
        curA=_el->getA1();
        curR=_el->getCr();

        // Create the Text:
        //
        for(i=0; i<(int)strlen((char*)_el->getText()); ++i) {
          switch(_el->getText()[i]) {
            case 10:
              // Line feed:
              //
              curA=_el->getA1();
              curR-=_el->getLineDistance();

              ++lCounter;
              isFirst=true;
              lastCharWasALetter=false;
              break;

            case 13:
              // do nothing
              //
              break;

            default:
              // Alignment:
              //
              if(isFirst) {
                lineWidth=fnts->getTextLineWidth(_el->getFont(),
                                                 _el->getText(),
                                                 i,
                                                 _el->getTextHeight(),
                                                 _el->getLetterSpacing(),
                                                 _el->getWordSpacing(),
                                                 _el->getFixedWidth());
                switch(textAlign) {
                  // Center:
                  //
                  case E_CENTER:
                    curA += (lineWidth/2.0)*ARAD/curR;
                    break;

                  // Right:
                  //
                  case E_RIGHT:
                    curA += lineWidth*ARAD/curR;
                    break;
                  default:
                    break;
                }
              }

              // Add elements for this letter:
              //
              // Space:
              //
              if(_el->getText()[i]==' ') {
                letterWidth = _el->getWordSpacing();
                if(lastCharWasALetter) letterWidth -= _el->getLetterSpacing();

                if(_el->getFlag(E_FIXEDWIDTH)) {
                  letterWidth = fnts->getLetterWidth(_el->getFont(),
                                                     'W',
                                                     _el->getTextHeight());
                }

                curA -=letterWidth*ARAD/curR;
                lastCharWasALetter=false;
              }
              else {
                // Create letter:
                //
                unsigned char letter=_el->getText()[i];
                if(_el->getFlag(E_FIXEDWIDTH)) letter='W';   // Fixed width for all lettters
                lWidthA =fnts->getLetterWidth(_el->getFont(),
                                              letter,
                                              _el->getTextHeight())*ARAD/curR;
                sWidthA =_el->getLetterSpacing()*ARAD/curR;
                curAM =(curA-lWidthA/2.0);

                curX =_el->getX1()+ cos(curAM/ARAD) * curR;
                curY =_el->getY1()+ sin(curAM/ARAD) * curR;

                int elNum = fnts->getLetterElementNumber(_el->getFont(),
                                                         _el->getText()[i]);
                int k;
                RElement tmpEl(this);
                for(k=0; k<elNum; ++k) {

                  fnts->getLetterElement(_el->getFont(),
                                         _el->getText()[i],
                                         _el->getTextHeight(),
                                         curX, curY,
                                         mtCorrAngle(curAM-90.0),
                                         E_CENTER,
                                         k,
                                         &tmpEl);
                  tmpEl.getAttribFrom(_el);

                  // Add element to this graphic:
                  //
                  if(_add) {
                    RElement* newEl = new RElement(this);
                    newEl->getMeasuresFrom(&tmpEl);
                    newEl->setLayer(_el->getLayer());
                    if(_setCreationNum) newEl->setCreationNumber(nextP);

                    element.append(newEl);
                  }

                  // Draw element:
                  //
                  else if(_draw && _paint) {
                    if(tmpEl.isOnScreen() || printing) {
                      tmpEl.fastDraw(_paint);
                    }
                  }

                  // Get borders:
                  //
                  else if(_border) {
                    if(_left   && tmpEl.left()  < *_left  ) *_left   = tmpEl.left();
                    if(_top    && tmpEl.top()   > *_top   ) *_top    = tmpEl.top();
                    if(_right  && tmpEl.right() > *_right ) *_right  = tmpEl.right();
                    if(_bottom && tmpEl.bottom()< *_bottom) *_bottom = tmpEl.bottom();
                  }

                }

                curA -=lWidthA+sWidthA;
                lastCharWasALetter=true;

              }
              isFirst=false;
              break;
          }
        }
        break;


      // Round out text:
      // ---------------
      case E_ROUNDOUT:
        curA=_el->getA1();
        curR=_el->getCr();

        // Create the Text:
        //
        for(i=0; i<(int)strlen((char*)_el->getText()); ++i) {
          switch(_el->getText()[i]) {
            case 10:
              // Line feed:
              //
              curA=_el->getA1();
              curR+=_el->getLineDistance();

              ++lCounter;
              isFirst=true;
              lastCharWasALetter=false;
              break;

            case 13:
              // do nothing
              //
              break;

            default:
              // Alignment:
              //
              if(isFirst) {
                lineWidth=fnts->getTextLineWidth(_el->getFont(),
                                                 _el->getText(),
                                                 i,
                                                 _el->getTextHeight(),
                                                 _el->getLetterSpacing(),
                                                 _el->getWordSpacing(),
                                                 _el->getFixedWidth());
                switch(textAlign) {
                  // Center:
                  //
                  case E_CENTER:
                    curA -= (lineWidth/2.0)*ARAD/curR;
                    break;

                  // Right:
                  //
                  case E_RIGHT:
                    curA -= lineWidth*ARAD/curR;
                    break;
                  default:
                    break;
                }
              }


              // Add elements for this letter:
              //
              // Space:
              //
              if(_el->getText()[i]==' ') {
                letterWidth = _el->getWordSpacing();
                if(lastCharWasALetter) letterWidth -= _el->getLetterSpacing();

                if(_el->getFlag(E_FIXEDWIDTH)) {
                  letterWidth = fnts->getLetterWidth(_el->getFont(),
                                                     'W',
                                                     _el->getTextHeight());
                }

                curA +=letterWidth*ARAD/curR;
                lastCharWasALetter=false;
              }
              else {
                // Create letter:
                //
                unsigned char letter=_el->getText()[i];
                if(_el->getFlag(E_FIXEDWIDTH)) letter='W';   // Fixed width for all lettters
                lWidthA =fnts->getLetterWidth(_el->getFont(),
                                              letter,
                                              _el->getTextHeight())*ARAD/curR;
                sWidthA =_el->getLetterSpacing()*ARAD/curR;
                curAM =(curA+lWidthA/2.0);

                curX =_el->getX1()+ cos(curAM/ARAD) * curR;
                curY =_el->getY1()+ sin(curAM/ARAD) * curR;

                int elNum = fnts->getLetterElementNumber(_el->getFont(),
                                                         _el->getText()[i]);
                int k;
                RElement tmpEl(this);
                for(k=0; k<elNum; ++k) {

                  fnts->getLetterElement(_el->getFont(),
                                         _el->getText()[i],
                                         _el->getTextHeight(),
                                         curX, curY,
                                         mtCorrAngle(curAM+90.0),
                                         E_CENTER,
                                         k,
                                         &tmpEl);
                  tmpEl.getAttribFrom(_el);

                  // Add element to this graphic:
                  //
                  if(_add) {
                    RElement* newEl = new RElement(this);
                    newEl->getMeasuresFrom(&tmpEl);
                    newEl->setLayer(_el->getLayer());
                    if(_setCreationNum) newEl->setCreationNumber(nextP);

                    element.append(newEl);
                  }

                  // Draw element:
                  //
                  else if(_draw && _paint) {
                    if(tmpEl.isOnScreen() || printing) {
                      tmpEl.fastDraw(_paint);
                    }
                  }

                  // Get borders:
                  //
                  else if(_border) {
                    if(_left   && tmpEl.left()  < *_left  ) *_left   = tmpEl.left();
                    if(_top    && tmpEl.top()   > *_top   ) *_top    = tmpEl.top();
                    if(_right  && tmpEl.right() > *_right ) *_right  = tmpEl.right();
                    if(_bottom && tmpEl.bottom()< *_bottom) *_bottom = tmpEl.bottom();
                  }

                }

                curA += lWidthA+sWidthA;
                lastCharWasALetter=true;
              }
              isFirst=false;
              break;
          }
        }
        break;

      default:
        break;
    }

    if(_add && _terminateAction) {
      terminateAction();
    }

  }


  // Dimension:
  //
  else if( _el->getElementTyp()==T_DIMENSION ) {

    uint dimMode = _el->getTextMode();
    RElement tmpEl[8];
    int c;
    //float aPixel=1.0/zoom;
    float aPixel=1.0;
    float dist=0.0;
    bool textInside=false;   // Text inside circle (radius dimension)
    unsigned char text[128];
    text[0]='\0';

    // ??
    //if(printing) aPixel*=RCONFIG->getSettingDouble( "CAD:PrintFactor" );

    for(c=0; c<8; ++c) {
      tmpEl[c].setGraphic(this);
      tmpEl[c].getAttribFrom(_el);
    }

    switch(dimMode) {

      // Straight dimension:
      //
      case E_STRAIGHT:
        // Horizontal:
        //
        if(_el->getFlag(E_HORIZONTAL)) {
          int neg;           // Negative?

          // Border line 1:
          if(_el->getY1()>_el->getDimDist()) neg=-1;
          else                               neg=1;
          tmpEl[0].createLine(_el->getX1(), _el->getY1(),
                              _el->getX1(),
                              _el->getDimDist()+aPixel*RCONFIG->cadDimensionOverLength*neg);
          // Border line 2:
          if(_el->getY2()>_el->getDimDist()) neg=-1;
          else                               neg=1;
          tmpEl[1].createLine(_el->getX2(), _el->getY2(),
                              _el->getX2(),
                              _el->getDimDist()+aPixel*RCONFIG->cadDimensionOverLength*neg);
          // Main line:
          tmpEl[2].createLine(_el->getX1(), _el->getDimDist(),
                              _el->getX2(), _el->getDimDist());

          dist = mtGetDistance(_el->getX1(), _el->getDimDist(), _el->getX2(), _el->getDimDist());
        }

        // Vertical:
        //
        else if(_el->getFlag(E_VERTICAL)) {
          int neg;           // Negative?

          // Border line 1:
          if(_el->getX1()>_el->getDimDist()) neg=-1;
          else                               neg=1;
          tmpEl[0].createLine(_el->getX1(), _el->getY1(),
                              _el->getDimDist()+aPixel*RCONFIG->cadDimensionOverLength*neg, _el->getY1());
          // Border line 2:
          if(_el->getX2()>_el->getDimDist()) neg=-1;
          else                               neg=1;
          tmpEl[1].createLine(_el->getX2(), _el->getY2(),
                              _el->getDimDist()+aPixel*RCONFIG->cadDimensionOverLength*neg, _el->getY2());
          // Main line:
          tmpEl[2].createLine(_el->getDimDist(), _el->getY1(),
                              _el->getDimDist(), _el->getY2());

          dist = mtGetDistance(_el->getDimDist(), _el->getY1(), _el->getDimDist(), _el->getY2());
        }

        // Radius:
        //
        else if(_el->getFlag(E_RADIUS)) {
          // Line:
          float angle = mtGetAngle(_el->getCx(), _el->getCy(), _el->getX1(), _el->getY1());
          tmpEl[2].createLine(_el->getCx() - cos(angle/ARAD) * aPixel*RCONFIG->cadDimensionOverLength,
                              _el->getCy() - sin(angle/ARAD) * aPixel*RCONFIG->cadDimensionOverLength,
                              _el->getX1(),
                              _el->getY1());

          dist = mtGetDistance(_el->getCx(), _el->getCy(), _el->getX1(), _el->getY1())-_el->getCr();
          if(dist<0.0) {
            dist = _el->getCr();
            textInside=true;
          }
        }

        // Arrow:
        //
        else if(_el->getFlag(E_ARROW)) {
          tmpEl[2].createLine(_el->getX1(), _el->getY1(),
                              _el->getX2(), _el->getY2());

        }

        // Parallel:
        //
        else {
          float angle = mtGetAngle(_el->getX1(), _el->getY1(), _el->getX2(), _el->getY2());
          int ff = _el->getDimDist()<0.0 ? -1 : 1;
          // Border line 1:
          tmpEl[0].createLine(_el->getX1(), _el->getY1(),
                              _el->getX1() + cos((angle+90.0)/ARAD) * (_el->getDimDist()+ff*aPixel*RCONFIG->cadDimensionOverLength),
                              _el->getY1() + sin((angle+90.0)/ARAD) * (_el->getDimDist()+ff*aPixel*RCONFIG->cadDimensionOverLength));
          // Border line 2:
          tmpEl[1].createLine(_el->getX2(), _el->getY2(),
                              _el->getX2() + cos((angle+90.0)/ARAD) * (_el->getDimDist()+ff*aPixel*RCONFIG->cadDimensionOverLength),
                              _el->getY2() + sin((angle+90.0)/ARAD) * (_el->getDimDist()+ff*aPixel*RCONFIG->cadDimensionOverLength));
          // Main line:
          tmpEl[2].createLine(_el->getX1(), _el->getY1(),
                              _el->getX2(), _el->getY2());
          tmpEl[2].move(cos((angle+90.0)/ARAD) * _el->getDimDist(), sin((angle+90.0)/ARAD) * _el->getDimDist());

          dist = mtGetDistance(_el->getX1(), _el->getY1(), _el->getX2(), _el->getY2());
        }
        break;

      case E_ROUNDOUT:

        // Angular:
        //
        if(true) {
          int neg;           // Negative?

          // Border line 1:
          if(_el->getX1()>_el->getDimDist()) neg=-1;
          else                               neg=1;
          tmpEl[0].createLine(_el->getCx() + cos(_el->getA1()/ARAD) * _el->getX1(),
                              _el->getCy() + sin(_el->getA1()/ARAD) * _el->getX1(),
                              _el->getCx() + cos(_el->getA1()/ARAD) * (_el->getDimDist()+aPixel*RCONFIG->cadDimensionOverLength*neg),
                              _el->getCy() + sin(_el->getA1()/ARAD) * (_el->getDimDist()+aPixel*RCONFIG->cadDimensionOverLength*neg));
          // Border line 2:
          if(_el->getX2()>_el->getDimDist()) neg=-1;
          else                               neg=1;
          tmpEl[1].createLine(_el->getCx() + cos(_el->getA2()/ARAD) * _el->getX2(),
                              _el->getCy() + sin(_el->getA2()/ARAD) * _el->getX2(),
                              _el->getCx() + cos(_el->getA2()/ARAD) * (_el->getDimDist()+aPixel*RCONFIG->cadDimensionOverLength*neg),
                              _el->getCy() + sin(_el->getA2()/ARAD) * (_el->getDimDist()+aPixel*RCONFIG->cadDimensionOverLength*neg));
          // Main arc:
          tmpEl[2].createArc(_el->getCx(), _el->getCy(),
                             _el->getCr(),
                             _el->getA1(), _el->getA2(),
                             _el->getFlag(E_REVERSED));
          dist = tmpEl[2].getCr();
          if(tmpEl[2].getLength()/4<tmpEl[2].getCr()) dist = tmpEl[2].getLength();
        }

        break;

      default:
        break;
    }


    // Arrow 1:
    float dir1 = tmpEl[2].getDirection1();
    float dir2 = tmpEl[2].getDirection2();
    float ax, ay;     // Arrow pos

    if(_el->getFlag(E_RADIUS)) {
      ax = _el->getCx() + cos(dir1/ARAD) * _el->getCr();
      ay = _el->getCy() + sin(dir1/ARAD) * _el->getCr();
    }
    else {
      ax = tmpEl[2].getX2();           // Arrow at the end (most dims)
      ay = tmpEl[2].getY2();
    }

    tmpEl[3].createLine(ax, ay,
                        ax + cos((dir2+15.0)/ARAD) * aPixel*RCONFIG->cadDimensionArrowLength,
                        ay + sin((dir2+15.0)/ARAD) * aPixel*RCONFIG->cadDimensionArrowLength );
    tmpEl[4].createLine(ax, ay,
                        ax + cos((dir2-15.0)/ARAD) * aPixel*RCONFIG->cadDimensionArrowLength,
                        ay + sin((dir2-15.0)/ARAD) * aPixel*RCONFIG->cadDimensionArrowLength );

    if(!_el->getFlag(E_RADIUS) && !_el->getFlag(E_ARROW)) {
      // Arrow 2:
      tmpEl[5].createLine(tmpEl[2].getX1(), tmpEl[2].getY1(),
                          tmpEl[2].getX1() + cos((dir1+15.0)/ARAD) * aPixel*RCONFIG->cadDimensionArrowLength,
                          tmpEl[2].getY1() + sin((dir1+15.0)/ARAD) * aPixel*RCONFIG->cadDimensionArrowLength );
      tmpEl[6].createLine(tmpEl[2].getX1(), tmpEl[2].getY1(),
                          tmpEl[2].getX1() + cos((dir1-15.0)/ARAD) * aPixel*RCONFIG->cadDimensionArrowLength,
                          tmpEl[2].getY1() + sin((dir1-15.0)/ARAD) * aPixel*RCONFIG->cadDimensionArrowLength );
    }

    // Trim line 2 out to the circular line if necessary:
    if(_el->getFlag(E_RADIUS)) {
      if(mtGetDistance(_el->getCx(), _el->getCy(), _el->getX1(), _el->getY1())<_el->getCr()) {
        float angle = mtGetAngle(_el->getCx(), _el->getCy(), _el->getX1(), _el->getY1());
        tmpEl[2].setX2(_el->getCx() + cos(angle/ARAD)*_el->getCr());
        tmpEl[2].setY2(_el->getCy() + sin(angle/ARAD)*_el->getCr());
      }
    }

    // Text:
    if(!_el->getFlag(E_ARROW)) {
      _el->getDimText((char*)text, 128);
      /*
      float height=aPixel*12;
      / *if(printing)* / height=cadDimensionTextHeight;
      if(cadDimensionTextAutoScale) while(height>dist/4 && height>2) height*=0.8;
      */
      
      float height=RCONFIG->cadDimensionTextHeight;

      float textAngle;
      uint  lFlags;
      float textX, textY;
      float textRad;

      if(dimMode==E_STRAIGHT) {
        textAngle = tmpEl[2].getMiddleDirection();
        if(textAngle>=90.01 && textAngle<=270.01) textAngle = mtCorrAngle(textAngle+180.0);
        lFlags=E_STRAIGHT;
        if(!_el->getFlag(E_RADIUS) || (dist<(_el->getCr()+0.01) && textInside)) {
          textX = tmpEl[2].getMiddleX();
          textY = tmpEl[2].getMiddleY();
        }
        else {
          float angle = mtGetAngle(_el->getCx(), _el->getCy(), _el->getX1(), _el->getY1());
          textX = _el->getCx() + cos(angle/ARAD) * (_el->getCr()+dist/2);
          textY = _el->getCy() + sin(angle/ARAD) * (_el->getCr()+dist/2);
        }
        textX = textX+cos((textAngle+90.0)/ARAD) * aPixel*RCONFIG->cadDimensionTextDistance;
        textY = textY+sin((textAngle+90.0)/ARAD) * aPixel*RCONFIG->cadDimensionTextDistance;
        textRad=0.0;
      }
      else {
        textAngle = mtGetAngle(tmpEl[2].getCx(), tmpEl[2].getCy(),
                               tmpEl[2].getMiddleX(), tmpEl[2].getMiddleY());
        if(textAngle<=180.01 && textAngle>=0.01) {
          lFlags=E_ROUNDIN;
          textRad = tmpEl[2].getCr()+aPixel*RCONFIG->cadDimensionTextDistance;
        }
        else {
          lFlags=E_ROUNDOUT;
          textRad = tmpEl[2].getCr()-aPixel*RCONFIG->cadDimensionTextDistance;
        }
        textX = tmpEl[2].getCx();
        textY = tmpEl[2].getCy();
      }

      tmpEl[7].createText(textX, textY,
                          RFonts::getRFonts()->getFontNumber( "normal" ),
                          text, lFlags|E_CENTER, height, textAngle, textRad,
                          lFlags==E_ROUNDOUT ? height*0.3 : height*0.2,
                          height*0.6, height*1.4 );
    }

    // Add elements:
    //
    if(_add) {
      for(c=0; c<8; ++c) {
        if(tmpEl[c].getFlag(E_VISIBLE)) {
          RElement* newEl = new RElement(this);
          if(newEl) {
            newEl->getMeasuresFrom(&tmpEl[c]);
            newEl->setLayer(_el->getLayer());
            if(_setCreationNum) newEl->setCreationNumber(nextP);
            element.append(newEl);
          }
          else {
            badAllocation();
          }
        }
      }
    }

    // Draw elements:
    //
    else if(_draw && _paint) {
      for(c=0; c<8; ++c) {
        if(tmpEl[c].getFlag(E_VISIBLE)) {
          if(tmpEl[c].isOnScreen() || printing) {
            tmpEl[c].fastDraw(_paint);
          }
        }
      }
    }

    // Get borders:
    //
    else if(_border) {
      for(c=0; c<8; ++c) {
        if(tmpEl[c].getFlag(E_VISIBLE)) {
          if(_left   && tmpEl[c].left()  < *_left  ) *_left   = tmpEl[c].left();
          if(_top    && tmpEl[c].top()   > *_top   ) *_top    = tmpEl[c].top();
          if(_right  && tmpEl[c].right() > *_right ) *_right  = tmpEl[c].right();
          if(_bottom && tmpEl[c].bottom()< *_bottom) *_bottom = tmpEl[c].bottom();
        }
      }
    }

    if(_add && _terminateAction) {
      terminateAction();
    }
  }



  // Hatching:
  //
  else if(_el->getElementTyp()==T_HATCHING ) {

    // Add elements:
    //
    for(int i=0; i<_el->getSubElementNum(); ++i) {
      if(_el->getSubElement(i)) {
        if(_add) {
          RElement* newEl = new RElement(this);
          if(newEl) {
            newEl->getMeasuresFrom(_el->getSubElement(i));
            newEl->setLayer(_el->getLayer());
            if(_setCreationNum) newEl->setCreationNumber(nextP);
            element.append(newEl);
          }
          else {
            badAllocation();
          }
        }
      }
    }

    if(_add && _terminateAction) {
      terminateAction();
    }
  }
}





// 
//
// Bad Allocation:
//
//
void
RGraphic::badAllocation()
{
  QMessageBox::warning(parentWidget(), DEF_APPNAME, "Too few memory!\nSave your work and exit the program please!");
  noMoreMemory=true;
}



//
//
// Edit functions:
//
//

// Edit: Move
//
void
RGraphic::editMove(int _copies, 
                   float _ox, float _oy,
                   float _x, float _y, 
                   float _angle,
                   bool _all,
                   bool _upd)
{
  bool move=false;       // only move (no copy)?
  int copies=_copies;    // number of copies
  int c;                 // copy-counter
  RElement* el;          // element ptr (walks through elements of preview)
  RElement* tmpEl;       // Temp element for adding new elements
  RPainter paint;        // painter
  
  if(copies==0) {
    move=true;
    copies=1;
  }
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
  
  // Move: delete original elements (screen):
  //
  if(_upd && move) {
    paint.setPen(white);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && (el->getFlag(E_TAGGED) || _all)) {
        el->draw(&paint, 0.0, 0.0, false, true);
        
        el->setDestructionNumber(nextP);
        el->delFlag(E_VISIBLE);
        el->delFlag(E_TAGGED);
        el->setFlag(E_UNDO);
      }
    }
  }
  
  // Copy:
  //
  if(_upd) {
    paint.setPen(black);
  }
  
  for(c=1; c<=copies; ++c) {
    for(el=preview.first(); el!=0; el=preview.next()) {
      if(el->getFlag(E_VISIBLE) && (el->getFlag(E_TAGGED) || _all)) {
        tmpEl = new RElement(this);
        tmpEl->setCreationNumber(nextP);
        tmpEl->getMeasuresFrom(el);
        if(!mtCompFloat(_angle, 0.0)) tmpEl->rotate(_angle, 0.0, 0.0);
        tmpEl->move(_ox+_x*c, _oy+_y*c);
        if(!_all) tmpEl->delFlag(E_TAGGED);
        element.append(tmpEl);
        
        if(_upd) tmpEl->draw(&paint);
      }
    }
  }

  // Move: delete original elements (memory):
  //
  /*if(move) {
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && (el->getFlag(E_TAGGED) || _all)) {
        el->setDestructionNumber(nextP);
        el->delFlag(E_VISIBLE);
        el->delFlag(E_TAGGED);
        el->setFlag(E_UNDO);
      }
    }
  }*/
  
  if(_upd) {
    paint.end();
  }

  graphicHasChanged();
  
  terminateAction();
}




// Edit: Rotate
//
void
RGraphic::editRotate(int _copies,
                     float _ang, 
                     float _ox, float _oy,
                     bool _upd)
{
  bool move=false;       // only move (no copy)?
  int copies=_copies;    // number of copies
  int c;                 // copy-counter
  RElement* el;          // element ptr (walks through elements of preview)
  RElement* tmpEl;       // Temp element for adding new elements
  RPainter paint;        // painter
  
  if(copies==0) {
    move=true;
    copies=1;
  }
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
  
  // Move: delete original elements (screen):
  //
  if(_upd && move) {
    paint.setPen(white);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->draw(&paint, 0.0, 0.0, false, true);
      }
    }
  }
  
  // Copy:
  //
  if(_upd) {
    paint.setPen(black);
  }
  
  for(c=1; c<=copies; ++c) {
    for(el=preview.first(); el!=0; el=preview.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        tmpEl = new RElement(this);
        tmpEl->setCreationNumber(nextP);
        tmpEl->getMeasuresFrom(el);
        tmpEl->rotate(_ang*c, _ox, _oy);
        tmpEl->delFlag(E_TAGGED);
        element.append(tmpEl);
        
        if(_upd) tmpEl->draw(&paint);
      }
    }
  }

  // Move: delete original elements (memory):
  //
  if(move) {
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->setDestructionNumber(nextP);
        el->delFlag(E_VISIBLE);
        el->delFlag(E_TAGGED);
        el->setFlag(E_UNDO);
      }
    }
  }
  
  if(_upd) {
    paint.end();
  }

  graphicHasChanged();
  
  terminateAction();
}



// Edit: Scale
//
void
RGraphic::editScale(int _copies,
                    float _factor, 
                    float _ox, float _oy,
                    bool _upd)
{
  bool move=false;       // only move (no copy)?
  int copies=_copies;    // number of copies
  int c;                 // copy-counter
  RElement* el;          // element ptr (walks through elements of preview)
  RElement* tmpEl;       // Temp element for adding new elements
  RPainter paint;        // painter
  
  if(copies==0) {
    move=true;
    copies=1;
  }
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
  
  // Move: delete original elements (screen):
  //
  if(_upd && move) {
    paint.setPen(white);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->draw(&paint, 0.0, 0.0, false, true);
      }
    }
  }
  
  // Copy:
  //
  if(_upd) {
    paint.setPen(black);
  }
  
  for(c=1; c<=copies; ++c) {
    for(el=preview.first(); el!=0; el=preview.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        tmpEl = new RElement(this);
        tmpEl->setCreationNumber(nextP);
        tmpEl->getMeasuresFrom(el);
        tmpEl->scale(_factor, _ox, _oy);
        tmpEl->delFlag(E_TAGGED);
        element.append(tmpEl);
        
        if(_upd) tmpEl->draw(&paint);
      }
    }
    _factor*=_factor;
  }

  // Move: delete original elements (memory):
  //
  if(move) {
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->setDestructionNumber(nextP);
        el->delFlag(E_VISIBLE);
        el->delFlag(E_TAGGED);
        el->setFlag(E_UNDO);
      }
    }
  }
  
  if(_upd) {
    paint.end();
  }

  graphicHasChanged();
  
  terminateAction();
}



// Edit: Mirror
//
void
RGraphic::editMirror(int _copies,
                     float _x1, float _y1,
                     float _x2, float _y2,
                     bool _upd)
{
  bool move=false;       // only move (no copy)?
  int copies=_copies;    // number of copies
  int c;                 // copy-counter
  RElement* el;          // element ptr (walks through elements of preview)
  RElement* tmpEl;       // Temp element for adding new elements
  RPainter paint;        // painter
  
  if(copies==0) {
    move=true;
    copies=1;
  }
  
  if(copies>1) copies=1;
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
  
  // Move: delete original elements (screen):
  //
  if(_upd && move) {
    paint.setPen(white);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->draw(&paint, 0.0, 0.0, false, true);
      }
    }
  }
  
  // Copy:
  //
  if(_upd) {
    paint.setPen(black);
  }
  
  for(c=1; c<=copies; ++c) {
    for(el=preview.first(); el!=0; el=preview.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        tmpEl = new RElement(this);
        tmpEl->setCreationNumber(nextP);
        tmpEl->getMeasuresFrom(el);
        tmpEl->mirror(_x1, _y1, _x2, _y2);
        tmpEl->delFlag(E_TAGGED);
        element.append(tmpEl);
        
        if(_upd) tmpEl->draw(&paint);
      }
    }
  }

  // Move: delete original elements (memory):
  //
  if(move) {
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->setDestructionNumber(nextP);
        el->delFlag(E_VISIBLE);
        el->delFlag(E_TAGGED);
        el->setFlag(E_UNDO);
      }
    }
  }
  
  if(_upd) {
    paint.end();
  }

  graphicHasChanged();
  
  terminateAction();
}





// Edit: Move and Rotate
//
void
RGraphic::editMoveRotate(int _copies, 
                         float _angle,
                         float _x, float _y,
                         float _ox1, float _oy1,
                         float _ox2, float _oy2,
                         bool _upd)
{
  bool move=false;       // only move (no copy)?
  int copies=_copies;    // number of copies
  int c;                 // copy-counter
  RElement* el;          // element ptr (walks through elements of preview)
  RElement* tmpEl;       // Temp element for adding new elements
  RPainter paint;        // painter
  
  if(copies==0) {
    move=true;
    copies=1;
  }
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
  
  // Move: delete original elements (screen):
  //
  if(_upd && move) {
    paint.setPen(white);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->draw(&paint, 0.0, 0.0, false, true);
      }
    }
  }
  
  // Copy:
  //
  if(_upd) {
    paint.setPen(black);
  }
  
  for(c=1; c<=copies; ++c) {

    _ox2+=_x;
    _oy2+=_y;
    for(el=preview.first(); el!=0; el=preview.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        tmpEl = new RElement(this);
        tmpEl->setCreationNumber(nextP);
        tmpEl->getMeasuresFrom(el);
        tmpEl->move(_ox1+_x*c, _oy1+_y*c);
        tmpEl->rotate(_angle*c, _ox2, _oy2);
        tmpEl->delFlag(E_TAGGED);
        element.append(tmpEl);
        
        if(_upd) tmpEl->draw(&paint);
      }
    }
  }

  // Move: delete original elements (memory):
  //
  if(move) {
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->setDestructionNumber(nextP);
        el->delFlag(E_VISIBLE);
        el->delFlag(E_TAGGED);
        el->setFlag(E_UNDO);
      }
    }
  }
  
  if(_upd) {
    paint.end();
  }

  graphicHasChanged();
  
  terminateAction();
}



// Edit: Rotate and Counter-Rotate
//
void
RGraphic::editRotateRotate(int _copies, 
                           float _angle1,
                           float _angle2,
                           float _ox1, float _oy1,
                           float _ox2, float _oy2,
                           bool _upd)
{

  bool move=false;       // only move (no copy)?
  int copies=_copies;    // number of copies
  int c;                 // copy-counter
  RElement* el;          // element ptr (walks through elements of preview)
  RElement* tmpEl;       // Temp element for adding new elements
  RPainter paint;        // painter
  float rad, ang;        // tmp radius and angle
  float ox3, oy3;        // tmp reference point which rotates
  
  if(copies==0) {
    move=true;
    copies=1;
  }
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
  
  // Move: delete original elements (screen):
  //
  if(_upd && move) {
    paint.setPen(white);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->draw(&paint, 0.0, 0.0, false, true);
      }
    }
  }
  
  // Copy:
  //
  if(_upd) {
    paint.setPen(black);
  }
  
  rad = mtGetDistance(_ox1, _oy1, _ox2, _oy2);
  ang = mtGetAngle(_ox1, _oy1, _ox2, _oy2); 
  
  for(c=1; c<=copies; ++c) {
    ox3 = _ox1 + cos((ang + _angle1*c)/ARAD) * rad;
    oy3 = _oy1 + sin((ang + _angle1*c)/ARAD) * rad;
    for(el=preview.first(); el!=0; el=preview.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        tmpEl = new RElement(this);
        tmpEl->setCreationNumber(nextP);
        tmpEl->getMeasuresFrom(el);
        tmpEl->move(_ox2, _oy2);
        tmpEl->rotate(_angle1*c, _ox1, _oy1);
        tmpEl->rotate(_angle2*c, ox3, oy3);
        tmpEl->delFlag(E_TAGGED);
        element.append(tmpEl);
        
        if(_upd) tmpEl->draw(&paint);
      }
    }
  }

  // Move: delete original elements (memory):
  //
  if(move) {
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->setDestructionNumber(nextP);
        el->delFlag(E_VISIBLE);
        el->delFlag(E_TAGGED);
        el->setFlag(E_UNDO);
      }
    }
  }
  
  if(_upd) {
    paint.end();
  }

  graphicHasChanged();
  
  terminateAction();
}




// Edit trim:
//
// _tEl      : trim element
// _ep       : trim endpoint (=true) or trim startpoint (=false)
// _ix1/_iy1 : first  intersection point (trim to this point)
// _ix2/_iy2 : second intersection point (circles also trim to this point)
// _mang     : mouse angle (define part of a circle to trim)
// _upd      : update graphic view?
//
void 
RGraphic::editTrim(RElement* _tEl,
                   bool _ep,
                   float _ix1, float _iy1,
                   float _ix2, float _iy2,
                   float _mang, 
                   bool _upd ) 
{
  RPainter paint;        // painter
  RElement* tmpEl;
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }

  // delete original element (screen):
  //
  if(_upd) {
    paint.setPen(white);
    _tEl->draw(&paint, 0.0, 0.0, false, true);
  }

  // Trim:
  //

  // Element trimmed to zero?
  //
  if(!_tEl->isTrimmedToZero(_ep, _ix1, _iy1)) {
    tmpEl = new RElement(this);
    tmpEl->setCreationNumber(nextP);
    tmpEl->getMeasuresFrom(_tEl);
    
    tmpEl->trim(_ep, _ix1, _iy1, _ix2, _iy2, _mang);
    
    element.append(tmpEl);
    
    if(_upd) {
      paint.setPen(black);
      tmpEl->draw(&paint);
    }
  }

  // Move: delete original elements (memory):
  //
  _tEl->setDestructionNumber(nextP);
  _tEl->delFlag(E_VISIBLE);
  _tEl->delFlag(E_TAGGED);
  _tEl->setFlag(E_UNDO);
  
  terminateAction();
}




// Cut an element at a certain point
//   _cEl   : Cut element
//   _px/_py: Cutting point
//   _upd   : Update graphic view?
//
void 
RGraphic::editCut(RElement* _cEl,
                  float _px, float _py,
                  bool _upd)
{
  RPainter paint;        // painter
  RElement* tmpEl;
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }

  // delete original element (screen):
  //
  if(_upd) {
    paint.setPen(white);
    _cEl->draw(&paint, 0.0, 0.0, false, true);
  }

  // Cut:
  //

  // Elements which fall in two parts:
  //
  if(_cEl->getElementTyp()!=T_CIRCLE) {

    // 1st part:
    //
    tmpEl = new RElement(this);
    tmpEl->setCreationNumber(nextP);
    tmpEl->getMeasuresFrom(_cEl);

    tmpEl->trim(true, _px, _py, DEF_AREAMAX, DEF_AREAMAX, 0.0);

    element.append(tmpEl);

    if(_upd) {
      paint.setPen(black);
      tmpEl->draw(&paint);
    }

    // 2nd part:
    //
    tmpEl = new RElement(this);
    tmpEl->setCreationNumber(nextP);
    tmpEl->getMeasuresFrom(_cEl);

    tmpEl->trim(false, _px, _py, 0.0, 0.0, 0.0);

    element.append(tmpEl);

    if(_upd) {
      paint.setPen(black);
      tmpEl->draw(&paint);
    }
  }

  // Circles:
  //
  else {
    // New arc:
    //
    tmpEl = new RElement(this);
    tmpEl->setCreationNumber(nextP);
    tmpEl->getMeasuresFrom(_cEl);
    tmpEl->setElementTyp(T_ARC);
    tmpEl->setX1(_px);
    tmpEl->setY1(_py);
    tmpEl->setX2(_px);
    tmpEl->setY2(_py);
    tmpEl->recalculateAngles();

    element.append(tmpEl);

    if(_upd) {
      paint.setPen(black);
      tmpEl->draw(&paint);
    }
  }
  

  // Move: delete original elements (memory):
  //
  _cEl->setDestructionNumber(nextP);
  _cEl->delFlag(E_VISIBLE);
  _cEl->delFlag(E_TAGGED);
  _cEl->setFlag(E_UNDO);
  
  terminateAction();
}



// Edit: Stretch
//
void
RGraphic::editStretch(float _x, float _y, 
                      bool _upd)
{
  RElement* el;          // element ptr (walks through elements of preview)
  RElement* tmpEl;       // Temp element for adding new elements
  RPainter paint;        // painter
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
  
  // Move: delete original elements (screen):
  //
  if(_upd) {
    paint.setPen(white);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && 
           (el->getFlag(E_TAGGED) || 
            el->getFlag(E_TAGSTP) ||
            el->getFlag(E_TAGEND)    )) {
        el->draw(&paint, 0.0, 0.0, false, true);
      }
    }
  }
  
  // Copy:
  //
  if(_upd) {
    paint.setPen(black);
  }
  
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE)) {
    
      // common preparations:
      //
      if(el->getFlag(E_TAGGED) || 
         el->getFlag(E_TAGSTP) || 
         el->getFlag(E_TAGEND)    ) {
         
        int bakEl;  // Backup element pointer
         
        tmpEl = new RElement(this);
        tmpEl->setCreationNumber(nextP);
        tmpEl->getMeasuresFrom(el);
        tmpEl->delFlag(E_TAGGED|E_TAGSTP|E_TAGEND);
        
        bakEl = element.at();
        element.append(tmpEl);
        element.at(bakEl);
    
        if(el->getFlag(E_TAGGED)) {
          tmpEl->move(_x, _y);
        }
        
        else if(el->getFlag(E_TAGSTP)) {
          tmpEl->moveStartPoint(_x, _y);
        }
        
        else if(el->getFlag(E_TAGEND)) {
          tmpEl->moveEndPoint(_x, _y);
        }
      
        // Common endings:
        //
        if(_upd) {
          paint.setPen(black);
          tmpEl->draw(&paint);
        }
      }
      
    }
  }

  // Delete original elements (memory):
  //
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE) && 
         (el->getFlag(E_TAGGED) || 
          el->getFlag(E_TAGSTP) || 
          el->getFlag(E_TAGEND)    )) {
          
      el->setDestructionNumber(nextP);
      el->delFlag(E_VISIBLE);
      el->delFlag(E_TAGGED|E_TAGSTP|E_TAGEND);
      el->setFlag(E_UNDO);
    }
  }
  
  if(_upd) {
    paint.end();
  }

  graphicHasChanged();
  
  terminateAction();
}



// Edit Round
//
void 
RGraphic::editRound(RElement* _el1,
                    RElement* _el2,
                    float _r,
                    bool _trim,
                    bool _trimEp1,
                    bool _trimEp2,
                    float _ix, float _iy,
                    float _mang1, float _mang2,
                    bool _upd)
{
  RPainter paint;        // painter
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setPen(black);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
  
  float tpx1, tpy1, tpx2, tpy2;      // tangent-points of rounding
  bool i1, i2;                       // Are there intersections at all
  RElement* tmpEl = new RElement(this);  // Rounding
  
  tmpEl->createCircle(_ix, _iy, _r, 0.0, 360.0, false);
  tmpEl->setCreationNumber(nextP);
  if(currentColor!=Qt::black) tmpEl->setColor(currentColor);
  if(currentWidth!=0) tmpEl->setWidth(currentWidth);
  if(currentStyle!=Qt::SolidLine) tmpEl->setStyle(currentStyle);
  element.append(tmpEl);
  
  _el1->getIntersection(tmpEl, 
                        &i1, &tpx1, &tpy1,
                        0, 0, 0,
                        0, false);
                        
  _el2->getIntersection(tmpEl, 
                        &i2, &tpx2, &tpy2,
                        0, 0, 0,
                        0, false);
  
  if(_trim) {
  
    // Trim: delete original elements (screen):
    //
    if(_upd) {
      paint.setPen(white);
      
      if(_el1->getFlag(E_VISIBLE)) {
        _el1->draw(&paint, 0.0, 0.0, false, true);
      }
      
      if(_el2->getFlag(E_VISIBLE)) {
        _el2->draw(&paint, 0.0, 0.0, false, true);
      }
      
      paint.setPen(black);
    }
    
    // ### Solaris ### (true was tmpEl)
  
    if(i1 && !(_el1->isTrimmedToZero(true, tpx1, tpy1))) {
      RElement* trimEl = new RElement(this);
      trimEl->getMeasuresFrom(_el1);
      if(_el1->getElementTyp()!=T_CIRCLE) {
        trimEl->trim(_trimEp1, tpx1, tpy1, 0.0, 0.0, _mang1);
      }
      element.append(trimEl);
      trimEl->setCreationNumber(nextP);
      if(_upd) trimEl->draw(&paint);
    }
    
    // ### Solaris ### (true was tmpEl)
    
    if(i2 && !_el2->isTrimmedToZero(true, tpx2, tpy2)) {
      RElement* trimEl = new RElement(this);
      trimEl->getMeasuresFrom(_el2);
      if(_el2->getElementTyp()!=T_CIRCLE) {
        trimEl->trim(_trimEp2, tpx2, tpy2, 0.0, 0.0, _mang2);
      }
      element.append(trimEl);
      trimEl->setCreationNumber(nextP);
      if(_upd) trimEl->draw(&paint);
    }
    
    // Trim: delete original elements (memory):
    //

    if(_el1->getFlag(E_VISIBLE)) {
      _el1->setDestructionNumber(nextP);
      _el1->delFlag(E_VISIBLE);
      _el1->setFlag(E_UNDO);
    }
    
    if(_el2->getFlag(E_VISIBLE)) {
      _el2->setDestructionNumber(nextP);
      _el2->delFlag(E_VISIBLE);
      _el2->setFlag(E_UNDO);
    }
  }
  
  // Trim the Rounding:
  //
  tmpEl->setElementTyp(T_ARC);
  tmpEl->setX1(tpx1);
  tmpEl->setY1(tpy1);
  tmpEl->setX2(tpx2);
  tmpEl->setY2(tpy2);
  tmpEl->recalculateAngles();

  if(mtGetAngleDiff(tmpEl->getA1(), tmpEl->getA2())>180.0) {
    tmpEl->turnFlag(E_REVERSED);
  }
    
  if(_upd) tmpEl->draw(&paint);
  
  graphicHasChanged();
  
  terminateAction();
}



// Beveling:
//   _el1 bevel element 1
//   _el2 bevel element 2
//   _width1    bevel width 1
//   _width2    bevel width 2
//   _trim      Trim elements to rounding?
//   _trimEp1   trim endpoint (=true) or trim startpoint (=false) of 1st element
//   _trimEp2   trim endpoint (=true) or trim startpoint (=false) of 2nd element
//   _ipx/_ipy  intersection point (trim to this point)
//   _upd update graphic view?
//
void 
RGraphic::editBevel(RElement* _el1,
                    RElement* _el2,
                    float _width1,
                    float _width2,
                    bool _trim,
                    bool _trimEp1,
                    bool _trimEp2,
                    float _ix, float _iy,
                    bool _upd)
{

  RPainter paint;        // painter
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setPen(black);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }

  RElement tel1, tel2;                    // Temp elements for calculating the bevel
  RElement* bevelEl = new RElement(this); // Beveling
  RElement* tmpEl;                        // Temp pointer to add elements

  // delete original element (screen):
  //
  if(_trim) {
  
    // Trim: delete original elements (screen):
    //
    if(_upd) {
      paint.setPen(white);
      
      if(_el1->getFlag(E_VISIBLE)) {
        _el1->draw(&paint, 0.0, 0.0, false, true);
      }
      
      if(_el2->getFlag(E_VISIBLE)) {
        _el2->draw(&paint, 0.0, 0.0, false, true);
      }
      
      paint.setPen(black);
    }
  }
  

  // trim temp elements to intersection:
  //
  tel1.getMeasuresFrom(_el1);
  tel2.getMeasuresFrom(_el2);

  if(!_el1->isTrimmedToZero(_trimEp1, _ix, _iy)) {
    if(tel1.getElementTyp()!=T_CIRCLE) {
      tel1.trim(_trimEp1, _ix, _iy, 0.0, 0.0, 0.0);
    }
  }
  
  if(!_el2->isTrimmedToZero(_trimEp2, _ix, _iy)) {
    if(tel2.getElementTyp()!=T_CIRCLE) {
      tel2.trim(_trimEp2, _ix, _iy, 0.0, 0.0, 0.0);
    }
  }


  // The bevel:
  //
  float tpx1, tpy1, tpx2, tpy2;      // end-points of line
  tel1.getDistPoint(&tpx1, &tpy1, _width1, !_trimEp1);
  tel2.getDistPoint(&tpx2, &tpy2, _width2, !_trimEp2);
  bevelEl->createLine(tpx1, tpy1, tpx2, tpy2);
  if(currentColor!=Qt::black) bevelEl->setColor(currentColor);
  if(currentWidth!=0) bevelEl->setWidth(currentWidth);
  if(currentStyle!=Qt::SolidLine) bevelEl->setStyle(currentStyle);
  bevelEl->setCreationNumber(nextP);
  element.append(bevelEl);
    
  // trim bevel elements to bevelline:
  //
  if(_trim) {
    tel1.trim(_trimEp1, tpx1, tpy1, 0.0, 0.0, 0.0);
    tel2.trim(_trimEp2, tpx2, tpy2, 0.0, 0.0, 0.0);
  }

  // draw all new:
  //
  if(_trim) {
    tmpEl = new RElement(this);
    tmpEl->getMeasuresFrom(&tel1);
    tmpEl->setCreationNumber(nextP);
    element.append(tmpEl);
    if(_upd) tmpEl->draw(&paint);
    
    tmpEl = new RElement(this);
    tmpEl->getMeasuresFrom(&tel2);
    tmpEl->setCreationNumber(nextP);
    element.append(tmpEl);
    if(_upd) tmpEl->draw(&paint);
  }
  
  if(_upd) bevelEl->draw(&paint);
  
  // Trim: delete original elements (memory):
  //
  if(_trim) {
    if(_el1->getFlag(E_VISIBLE)) {
      _el1->setDestructionNumber(nextP);
      _el1->delFlag(E_VISIBLE);
      _el1->setFlag(E_UNDO);
    }
    
    if(_el2->getFlag(E_VISIBLE)) {
      _el2->setDestructionNumber(nextP);
      _el2->delFlag(E_VISIBLE);
      _el2->setFlag(E_UNDO);
    }
  }
  
  graphicHasChanged();
  
  terminateAction();

}



// Edit: Convert text into elements:
//   _upd update graphic view?
//
void
RGraphic::editTextElements(bool _upd)
{
  RElement* el;          // element ptr (walks through elements)
  RElement* lastEl;      // Last element in original list 
  
  RPainter paint;        // painter
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
  
  // Delete original elements (screen):
  //
  if(_upd) {
    paint.setPen(white);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        if(el->getElementTyp()==T_TEXT ||
           el->getElementTyp()==T_DIMENSION ||
           el->getElementTyp()==T_HATCHING) {
          el->draw(&paint, 0.0, 0.0, false, true);
        }
      }
    }
  }
  
  lastEl=element.last();
  
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
      if(el->getElementTyp()==T_TEXT ||
         el->getElementTyp()==T_DIMENSION ||
         el->getElementTyp()==T_HATCHING) {
        singleElements(el, true,
                       false,
                       0,
                       false,
                       0, 0, 0, 0,
                       true,
                       false);
        setCurrentElement(el);
      }
    }
  }
  
  if(element.findRef(lastEl)!=-1) {
    paint.setPen(black);
    for(el=lastEl; el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE)) {
        el->draw(&paint, 0.0, 0.0, true);
      }
    }
  }
  
  // Delete original elements (memory):
  //
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
      if(el->getElementTyp()==T_TEXT ||
         el->getElementTyp()==T_DIMENSION ||
         el->getElementTyp()==T_HATCHING) {
        el->setDestructionNumber(nextP);
        el->delFlag(E_VISIBLE);
        el->delFlag(E_TAGGED);
        el->setFlag(E_UNDO);
      }
    }
  }
  
  if(_upd) {
    paint.end();
  }

  graphicHasChanged();
  
  terminateAction();

}



// Edit: Edit text contents / attributes:
//   _el  original text element
//   _upd update graphic view?
//
void
RGraphic::editText(RElement* _el,
                   bool _upd)
{

  if(_el->getFlag(E_VISIBLE)) {
    if(_el->getElementTyp()==T_TEXT) {
      RPainter paint;        // painter
      
      if(_upd) {
        paint.begin(this);
        paint.setRasterOp(CopyROP);
        paint.setClipRect(2, 2, width()-4, height()-4);
      }
      
      // Set current font metrics from text element:
      //
      setFontNumber(_el->getFont());
      setFontProportional(_el->getFlag(E_PROPORTIONAL));
      setFontFixedWidth(_el->getFlag(E_FIXEDWIDTH));
      setFontHeight(_el->getTextHeight());
      setFontLetterSpacing(_el->getLetterSpacing());
      setFontWordSpacing(_el->getWordSpacing());
      setFontLineDistance(_el->getLineDistance());
      setFontAngle(_el->getA1());
      setFontRadius(_el->getCr());
      setFontFlags(_el->getFontFlags());
      
      // Show text dialog:
      //
      RFontDialog fontdlg(this, qApp->mainWidget());
      fontdlg.setText((char*)_el->getText());
      if(fontdlg.exec()) {
        QString text=fontdlg.getText().data();
      
        // Delete original element (screen):
        //
        if(_upd) {
          paint.setPen(white);
          _el->draw(&paint, 0.0, 0.0, false, true);
        }
      
        addText(_el->getX1(), _el->getY1(),
                getFontNumber(),
                (unsigned char*)text.latin1(),
                getFontFlags()->getFlags(),
                getFontHeight(),
                getFontAngle(),
                getFontRadius(),
                getFontLetterSpacing(),
                getFontWordSpacing(),
                getFontLineDistance(),
                0,
                true);
        drawElement();

        // Delete original element (memory):
        //
        _el->setDestructionNumber(nextP);
        _el->delFlag(E_VISIBLE);
        _el->delFlag(E_TAGGED);
        _el->setFlag(E_UNDO);
        
      }
      
      if(_upd) {
        paint.end();
      }
    
      graphicHasChanged();
      
      terminateAction();
    }
  }
}



// Edit: Delete
//
void
RGraphic::editDelete(bool _upd)
{
  RElement* el;          // element ptr (walks through elements of preview)
  RPainter paint;        // painter
  
  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
  
  // delete elements (screen):
  //
  if(_upd) {
    paint.setPen(white);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->draw(&paint, 0.0, 0.0, false, true);
      }
    }
  }
  
  // Move: delete original elements (memory):
  //
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
      el->setDestructionNumber(nextP);
      el->delFlag(E_VISIBLE);
      el->delFlag(E_TAGGED);
      el->setFlag(E_UNDO);
    }
  }
  
  if(_upd) {
    paint.end();
  }

  graphicHasChanged();
  
  terminateAction();
}



// Edit: Layer
//
void
RGraphic::editLayer(const char* _layerName, bool _upd)
{
  if(_layerName) {
    int newLayer = addLayer(_layerName);
    
    RLOG("\nnewLayer:");
    RLOG(newLayer);
    
    RElement* el;          // element ptr (walks through elements of preview)
    RElement* tmpEl;       // Temp element for adding new elements
    RPainter paint;        // painter
  
    if(_upd) {
      paint.begin(this);
      paint.setRasterOp(CopyROP);
      paint.setClipRect(2, 2, width()-4, height()-4);
    }
    
    // Delete original elements (screen):
    //
    if(_upd) {
      paint.setPen(white);
      for(el=element.first(); el!=0; el=element.next()) {
        if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
          el->draw(&paint, 0.0, 0.0, false, true);
        }
      }
    }
    
    if(_upd) {
      paint.setPen(black);
    }
    
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        tmpEl = new RElement(this);
        tmpEl->setCreationNumber(nextP);
        tmpEl->getMeasuresFrom(el);
        tmpEl->setLayer(newLayer);
        tmpEl->delFlag(E_TAGGED);
        if(!isLayerVisible(newLayer)) {
          tmpEl->delFlag(E_VISIBLE);
        }
        element.append(tmpEl);
        
        if(_upd) tmpEl->draw(&paint);
        
        setCurrentElement(el);
      }
    }
  
    // Delete original elements (memory):
    //
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->setDestructionNumber(nextP);
        el->delFlag(E_VISIBLE);
        el->delFlag(E_TAGGED);
        el->setFlag(E_UNDO);
      }
    }
    
    if(_upd) {
      paint.end();
    }
  
    graphicHasChanged();
    
    terminateAction();

    emit updateLayers();
  }

}



// Edit: Attributes
//
void
RGraphic::editAttrib(int _color, int _width, int _style, bool _upd)
{
  RElement* el;          // element ptr (walks through elements of preview)
  RElement* tmpEl;       // Temp element for adding new elements
  RPainter paint;        // painter

  if(_upd) {
    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }

  // Delete original elements (screen):
  //
  if(_upd) {
    paint.setPen(white);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
        el->draw(&paint, 0.0, 0.0, false, true);
      }
    }
  }

  if(_upd) {
    paint.setPen(black);
  }

  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
      tmpEl = new RElement(this);
      tmpEl->setCreationNumber(nextP);
      tmpEl->getMeasuresFrom(el);
      if(_color!=-1) tmpEl->setColor(numberToColor(_color));
      if(_width!=-1) tmpEl->setWidth(_width);
      if(_style!=-1) tmpEl->setStyle(numberToStyle(_style));
      tmpEl->delFlag(E_TAGGED);
      element.append(tmpEl);

      if(_upd) tmpEl->draw(&paint);

      setCurrentElement(el);
    }
  }

  // Delete original elements (memory):
  //
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
      el->setDestructionNumber(nextP);
      el->delFlag(E_VISIBLE);
      el->delFlag(E_TAGGED);
      el->setFlag(E_UNDO);
    }
  }

  if(_upd) {
    paint.end();
  }

  graphicHasChanged();

  terminateAction();

}




// 
//
// Tag functions:
//
//



// Tag nothing:
//
void 
RGraphic::tagNothing(bool _upd)
{
  RElement* el;           // element ptr (walks through elements)
  RPainter paint;        // painter
  
  if(_upd) {
    paint.begin(this);
    paint.setPen(black);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }

  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
      el->untag();
      if(_upd) el->draw(&paint);
    }
  }
  
  if(_upd) {
    paint.end();
  }
  
}



// Tag all:
//
void 
RGraphic::tagAll(bool _upd)
{
  RElement* el;          // element ptr (walks through elements)
  RPainter paint;        // painter
  
  if(_upd) {
    paint.begin(this);
    paint.setPen(red);

    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }

  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE) && !el->getFlag(E_TAGGED)) {
      el->tag();
      if(_upd) el->draw(&paint);
    }
  }
  
  if(_upd) {
    paint.end();
  }

}



// Tag closed contours:
//
//   _el: an element of the contour
//   _tag: true: tag / false: untag
//
void 
RGraphic::tagContour(RElement* _el, bool _tag, bool _upd)
{
  if(_el && _el->getGraphic()==this) {
    RElement* el;          // element ptr (walks through elements)
    RPainter paint;        // painter
    
    bool  tagThis;         // tag/untag actual element
    float cbx, cby,        // contour begin
          cex, cey;        //         end
    bool  done;            // contour marking finished
  
    if(_upd) {
      paint.begin(this);
      if(_tag) paint.setPen(red);
      else     paint.setPen(black);
      paint.setRasterOp(CopyROP);
      paint.setClipRect(2, 2, width()-4, height()-4);
    }
    
    
    cbx = _el->getX1();
    cby = _el->getY1();
    cex = _el->getX2();
    cey = _el->getY2();

    if(_tag) _el->tag();
    else     _el->untag();
    _el->draw(&paint);

    do {
      done=true;
      for(el=element.first(); el!=0; el=element.next()) {
        if(el->getFlag(E_VISIBLE)) {
          tagThis=false;

          if(el->getFlag(E_TAGGED)!=_tag) {

            if(mtCompFloat(cbx, el->getX1(), DEF_MMTOL) &&
               mtCompFloat(cby, el->getY1(), DEF_MMTOL)    ) {

              tagThis=true;
              cbx = el->getX2();
              cby = el->getY2();
            }
            else if(mtCompFloat(cbx, el->getX2(), DEF_MMTOL) &&
                    mtCompFloat(cby, el->getY2(), DEF_MMTOL)    ) {

              tagThis=true;
              cbx = el->getX1();
              cby = el->getY1();
            }
            else if(mtCompFloat(cex, el->getX1(), DEF_MMTOL) &&
                    mtCompFloat(cey, el->getY1(), DEF_MMTOL)    ) {

              tagThis=true;
              cex = el->getX2();
              cey = el->getY2();
            }
            else if(mtCompFloat(cex, el->getX2(), DEF_MMTOL) &&
                    mtCompFloat(cey, el->getY2(), DEF_MMTOL)    ) {

              tagThis=true;
              cex = el->getX1();
              cey = el->getY1();
            }

            if(tagThis) {
              if(_tag) el->tag();
              else     el->untag();
              el->draw(&paint);
              done=false;
            }
          }
        }
      }
    } while(!done);

    if(_upd) {
      paint.end();
    }
  }
}



// Tag range:
//
//  _minX, ...: the range
//
void 
RGraphic::tagRange(float _x1, float _y1, 
                   float _x2, float _y2, 
                   bool _tag, bool _upd,
                   bool _tagEndpoints)
{
  
  RElement* el;          // element ptr (walks through elements)
  RPainter paint;        // painter
    
  if(_upd) {
    paint.begin(this);
    if(_tag) paint.setPen(red);
    else     paint.setPen(black);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
    
    
  float dum;                 // Dummy for switching of rect-coords

  // switch (x1/y1) / (x2/y2):
  //
  if(_x1>_x2) { dum=_x1; _x1=_x2; _x2=dum; }
  if(_y1>_y2) { dum=_y1; _y1=_y2; _y2=dum; }

  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE)) {
    
      // Tag elements inside the range:
      //
      if(el->getFlag(E_TAGGED)!=_tag) {
        if( el->left()  >= _x1 && el->bottom() >= _y1 &&
            el->right() <= _x2 && el->top()    <= _y2    ) {
          el->turnFlag(E_TAGGED);
          if(_upd) el->draw(&paint);
          
          RLOG("\nElement tagged");
        }
      }
      
      // Tag startpoints inside the range:
      //
      if(_tagEndpoints               && 
         el->getFlag(E_TAGGED)!=_tag && 
         el->getFlag(E_TAGSTP)!=_tag    ) {
        if( el->getX1() >= _x1 && el->getY1() >= _y1 &&
            el->getX1() <= _x2 && el->getY1() <= _y2    ) {
          el->turnFlag(E_TAGSTP);
          RLOG("\nElement start point tagged");
        }
      }
      
      // Tag endpoints inside the range:
      //
      if(_tagEndpoints               && 
         el->getFlag(E_TAGGED)!=_tag && 
         el->getFlag(E_TAGEND)!=_tag    ) {
        if( el->getX2() >= _x1 && el->getY2() >= _y1 &&
            el->getX2() <= _x2 && el->getY2() <= _y2    ) {
          el->turnFlag(E_TAGEND);
          RLOG("\nElement end point tagged");
        }
      }
      
    }
  }
  
  if(_upd) {
    paint.end();
  }
}




// Tag elements intersected by a line:
//
void 
RGraphic::tagInters(float _x1, float _y1, 
                    float _x2, float _y2, 
                    bool _tag, bool _upd)
{
  RElement* el;          // element ptr (walks through elements)
  RPainter paint;        // painter
    
  if(_upd) {
    paint.begin(this);
    if(_tag) paint.setPen(red);
    else     paint.setPen(black);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
  
  bool is1, is2;
  float sx1, sy1, sx2, sy2;
  RElement dumEl(this);
  dumEl.createLine(_x1, _y1, _x2, _y2);
  
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE)) {
      if(el->getFlag(E_TAGGED)!=_tag) {
        el->getIntersection(&dumEl, 
                            &is1, &sx1, &sy1, 
                            &is2, &sx2, &sy2, 
                            0, true);
        if(is1 || is2) {
          el->turnFlag(E_TAGGED);
          if(_upd) el->draw(&paint);
        }
      }
    }
  }
  
  if(_upd) {
    paint.end();
  }
}



// Invert selection:
//
void 
RGraphic::tagInvert(bool _upd)
{
  RElement* el;          // element ptr (walks through elements)

  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_TAGGED)) {
      el->untag();
    }
    else {
      el->tag();
    }
  }
  
  if(_upd) {
    repaint(3, 3, width()-6, height()-6);
  }

}
  
  
  
// Tag a layer:
//
void 
RGraphic::tagLayer(byte _layerNum, bool _tag, bool _upd)
{
  RElement* el;          // element ptr (walks through elements)
  RPainter paint;        // painter
    
  if(_upd) {
    paint.begin(this);
    if(_tag) paint.setPen(red);
    else     paint.setPen(black);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }
  
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE)) {
      if(el->getFlag(E_TAGGED)!=_tag) {
        if(el->getLayer()==_layerNum) {
          el->turnFlag(E_TAGGED);
          if(_upd) el->draw(&paint);
        }
      }
    }
  }
  
  if(_upd) {
    paint.end();
  }
}


// Tag double elements:
//
void 
RGraphic::tagDoubles(float _tolerance, bool _upd)
{
  RElement* el1;           // element ptr (walks through elements)
  RElement* el2;           // element ptr (walks through elements)
  
  RPainter paint;        // painter
  
  if(_upd) {
    paint.begin(this);
    paint.setPen(red);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);
  }

  for(el1=element.first(); el1!=0; el1=element.next()) {
    if(el1->getFlag(E_VISIBLE)) {
      element.next();
      for(el2=element.current(); el2!=0; el2=element.next()) {
        if(el1!=el2) {
          if(el1->compareWith(el2, _tolerance, true)) {
            el2->tag();
            if(_upd) el2->draw(&paint);
          }
        }
      }
      setCurrentElement(el1);
    }
  }
  
  if(_upd) {
    paint.end();
  }
}



//
//
// Undo / Redo functions:
//
//


// terminate a action (construction, ...)
//
void 
RGraphic::terminateAction()
{
  undoP = nextP;
  increaseNextP();
  updateUndoRegister();
}


// Increase the position of the next action
//
void 
RGraphic::increaseNextP()
{
  ++nextP;
  if(nextP>=DEF_MAXUNDO) {
    shiftUndoRegister();
  }
}


// Update undo-register:
//
void 
RGraphic::updateUndoRegister()
{
  if(undoP!=0) undoReg[undoP] = 'u';
  if(redoP!=0) undoReg[redoP] = 'r';
}


// Shift undo-register:
//
void 
RGraphic::shiftUndoRegister()
{
  int i;
  RElement *el;            // pointer which walks through elements

  // Delete undo/redo memory of elements on Register 1-DEF_UNDOSHIFT:
  //
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getCreationNumber()<=DEF_UNDOSHIFT) {
      el->setCreationNumber(0);
    }
    if(el->getDestructionNumber()<=DEF_UNDOSHIFT) {
      el->setDestructionNumber(0);
      //if(!el->GetFlag(V_VISIBLE)) el->Reset();
    }
  }

  // Shift elements DEF_UNDOSHIFT fields to left:
  //
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getCreationNumber()!=0) {
      el->setCreationNumber(el->getCreationNumber()-DEF_UNDOSHIFT);
    }
    if(el->getDestructionNumber()!=0) {

      el->setDestructionNumber(el->getDestructionNumber()-DEF_UNDOSHIFT);
    }
  }

  // Shift undo register DEF_UNDOSHIFT fields to left:
  //
  for(i=1; i<DEF_MAXUNDO; ++i) {
    if(i+DEF_UNDOSHIFT<=DEF_MAXUNDO) undoReg[i] = undoReg[i+DEF_UNDOSHIFT];
    else                             undoReg[i] = '0';
  }

  // Shift pointers DEF_UNDOSHIFT fields to left:
  //
  if(undoP>DEF_UNDOSHIFT) undoP-=DEF_UNDOSHIFT;
  else                    undoP=0;

  if(redoP>DEF_UNDOSHIFT) redoP-=DEF_UNDOSHIFT;
  else                    redoP=0;

  if(nextP>DEF_UNDOSHIFT) nextP-=DEF_UNDOSHIFT;
  else                    nextP=1;
}



/**
 * Edit copy (copy selected elements to internal clipboard):
 */
void
RGraphic::editClipCopy(float _x, float _y)
{
  clipboard()->deleteElements();
  clipboard()->copyTaggedFrom(this, _x, _y);
}



/**
 * Edit cut (cut selected elements to internal clipboard):
 */
void
RGraphic::editClipCut(float _x, float _y)
{
  clipboard()->deleteElements();
  clipboard()->copyTaggedFrom(this, _x, _y);
}


/**
 * Edit paste (paste copied elements from internal clipboard):
 */
void
RGraphic::editClipPaste(float _x, float _y)
{
  copyVisibleObjectsFrom(clipboard(), false, _x, _y);
}


// Undo:
//
void 
RGraphic::undo()
{
  RElement *el;            // pointer which walks through elements
  
  if(undoP!=0) {
  
    autoUpdateInfo=false;
    
    RPainter paint;        // painter

    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);

    // delete constructed elements:
    //
    paint.setPen(white);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getCreationNumber()==undoP) {
        el->setFlag(E_VISIBLE|E_ACTIVE);
        el->delFlag(E_TAGGED);
        el->draw(&paint);
        el->delFlag(E_VISIBLE|E_TAGGED);
        el->setFlag(E_UNDO);
      }
    }

    // draw deleted elements:
    //
    paint.setPen(black);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getDestructionNumber()==undoP) {
        el->setFlag(E_VISIBLE);
        el->delFlag(E_UNDO);
        el->draw(&paint);

        // Move to an existing layer:
        //
        if(getLayerName(el->getLayer())==NULL) {
          el->setLayer(addLayer(DEF_DEFAULTLAYER));
        }
      }
    }
    
    paint.end();

    undoReg[undoP] = 'r';

    redoP=undoP;
    undoP=DEF_MAXUNDO;
    while(undoReg[undoP]!='u' && undoP>0) --undoP;

    if(undoP==DEF_MAXUNDO) shiftUndoRegister();
    
    autoUpdateInfo=true;
    graphicHasChanged();

    //updateLayerFlags(true);
  }
}


// Redo:
//
void 
RGraphic::redo()
{
  RElement *el;            // pointer which walks through elements

  if(redoP!=0) {
  
    autoUpdateInfo=false;
    
    RPainter paint;        // painter

    paint.begin(this);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);

    // redelete deleted elements:
    //
    paint.setPen(white);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getDestructionNumber()==redoP) {
        el->setFlag(E_VISIBLE);
        el->draw(&paint);
        el->delFlag(E_VISIBLE);
        el->setFlag(E_UNDO);
      }
    }

    // redraw constructed elements:
    //
    paint.setPen(black);
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getCreationNumber()==redoP) {
        el->setFlag(E_VISIBLE);
        el->delFlag(E_UNDO);
        el->draw(&paint);
        
        // Move to an existing layer:
        //
        if(getLayerName(el->getLayer())==NULL) {
          el->setLayer(addLayer(DEF_DEFAULTLAYER));
        }
      }
    }
    
    paint.end();

    undoReg[redoP] = 'u';

    ++redoP;
    undoP=DEF_MAXUNDO;
    while(undoReg[undoP]!='u' && undoP>0) --undoP;

    if(undoP==DEF_MAXUNDO) shiftUndoRegister();

    while(undoReg[redoP]!='r' && redoP<=DEF_MAXUNDO) ++redoP;
    if(redoP>DEF_MAXUNDO) {
      redoP=undoP;
      while(undoReg[redoP]!='r' && redoP>0) --redoP;
    }

    //updateLayerFlags(true);
    autoUpdateInfo=true;
    graphicHasChanged();
  }
}



//
//
// Count functions:
//
//


// Get number of (visible/invisible) elements
//
// return: number
//
int
RGraphic::count()
{
  RElement *el;            // pointer which walks through elements
  int ret=0;               // return value
  
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_ACTIVE) && el->getElementTyp()!=T_RAPID) {
      ++ret;
    }
  }
  
  return ret;
  
  //return element.count();
}



// Get number of (tagged) elements
//
// return: number
//
int
RGraphic::countTagged()
{
  RElement *el;            // pointer which walks through elements
  int ret=0;               // return value

  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_ACTIVE) && el->getFlag(E_TAGGED)) {
      ++ret;
    }
  }

  return ret;
}



// get number of (visible/invisible) elements 
//   on layer '_lay':
//   (by layer name)
//
int
RGraphic::countElementsOnLayer(const char* _layerName)
{
  int layerNumber=getLayerNum(_layerName);
  return countElementsOnLayer(layerNumber);
}



// get number of (visible/invisible) elements 
//   on layer '_layerNumber':
//   (by layer number)
//
// return: number
//         -1 if layer doesn't exist (is 0 or out of range)
//
int
RGraphic::countElementsOnLayer(int _layerNumber)
{
  if(_layerNumber<=0) return -1;
  
  RElement *el;          // pointer which walks through elements
  int ret=0;

  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_ACTIVE) &&
       el->getLayer()==_layerNumber) {
      ++ret;
    }
  }
  return ret;
}


//
//
// Layer handling:
//
//


// Add a layer:
//
// ret: Number of added layer
//      0: No layer added!
//
int
RGraphic::addLayer(const char* _name)
{
  RLOG("\nAdd Layer: ");
  RLOG(_name);

  // No valid Layername:
  //
  if(!_name || strlen(_name)==0) return 0;

  RLayer *lay;          // pointer which walks through layers

  int num=0;            // Returned number of added layer
  bool exist=false;     // Layer already exist

  // Test if layer doesn't already exist:
  //
  for(lay=layer.first(); lay!=0; lay=layer.next()) {
    if(lay->getFlag(Y_USED)) {
      if(!strcmp(_name, lay->getName())) {
        exist=true;
        num=lay->getNumber();
        break;
      }
    }
  }

  if(!exist) {

    // Find out first free number (not index):
    //
    bool done;
    num=1;
    do {
      done=true;
      for(lay=layer.first(); lay!=0; lay=layer.next()) {
        if(lay->getNumber()==num) {
          done=false;
          break;
        }
      }
      if(!done) ++num;
    }while(!done);

    if(done) {
      // Add the layer in the first free index with
      //   the determined number:
      //
      RLayer* newLayer = new RLayer;
      newLayer->setFlag(Y_USED|Y_VISIBLE);
      newLayer->setName(_name);
      newLayer->setNumber(num);

      layer.append(newLayer);
          
      RLOG("\nAdded Layer: ");
      RLOG(_name);
      RLOG(" ");
      RLOG(num);

    }

    if(autoUpdateLayerList) emit updateLayers();
  }

  return num;
}



// Select a layer (signal comes from layerlist):
//   (SLOT)

//
void  
RGraphic::selectLayer(const char* _name)
{
  selectedLayer=getLayerNum(_name);
}



// Check a layer number:
//
// _num: number to check
// 
// return: new number if number wasn't valid
//
int  
RGraphic::checkLayer(int _num)
{
  // Number is 0: Set to selected layer
  //
  if(_num==0) {
    _num=selectedLayer;

    // Number is still 0: Set to default layer:
    //
    if(_num==0) {
      _num = addLayer(DEF_DEFAULTLAYER);
    }
  }

  return _num;
}



// Count the layers:
//
int
RGraphic::countLayers()
{
  RLayer *lay;          // pointer which walks through layers
  int ret=0;

  for(lay=layer.first(); lay!=0; lay=layer.next()) {
    if(lay->getFlag(Y_USED)) {
      ++ret;
    }
  }
  return ret;
}



// Remove all empty layers:
//
// ret: true:  empty layer(s) found and removed
//      false: no empty layer found
//
bool 
RGraphic::removeEmptyLayers()
{
  RLayer *lay;      // pointer which walks through elements
  bool ret=false;   // return value
  bool found;       // Still an empty layer found
  int  remLayer;    // layer (number) to remove

  autoUpdateLayerList=false;
  
  do {
    found=false;
    for(lay=layer.first(); lay!=0; lay=layer.next()) {
      remLayer=lay->getNumber();
      if(countElementsOnLayer(remLayer)==0) {
        removeLayer(remLayer, false);
        found=true;
        ret=true;
      }
    }
  }while(found);

  autoUpdateLayerList=true;
  emit updateLayers();

  return ret;
}



// Remove a layer (by its name):
//
// ret: true:  successful
//      false: no such layer found
//
bool
RGraphic::removeLayer(const char* _name, bool _alsoObjects)
{
  return removeLayer(getLayerNum(_name), _alsoObjects);
}



// Remove a layer (by its number):
//
// ret: true:  successful
//      false: no such layer found
//
bool
RGraphic::removeLayer(int _num, bool _alsoObjects)
{
  bool   ret=false;  // returned value
  RElement *el;      // pointer which walks through elements
  RLayer *lay;       // pointer which walks through layers

  // Remove the layer _num:
  //
  for(lay=layer.first(); lay!=0; lay=layer.next()) {
    if(lay->getNumber()==_num) {
      layer.remove(layer.at());
      ret=true;

      RLOG("\nRemoved Layer: ");
      RLOG(_num);
    }
  }

  // Delete elements on deleted layer:
  //
  if(ret && _alsoObjects) {
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_ACTIVE) && el->getLayer()==_num) {
         el->delFlag(E_ACTIVE|E_VISIBLE);
        
        // ###
        // TO DO:
        //   add undo/redo info
        // ###
        
      }
    }
  }
  
  if(autoUpdateLayerList) emit updateLayers();
  return ret;
}



// Rename a layer (by its name):
//
// ret: true:  successful
//      false: no such layer found
//
bool
RGraphic::renameLayer(const char* _oldName, const char* _newName)
{
  return renameLayer(getLayerNum(_oldName), _newName);
}




// Rename a layer (by its number):
//
// ret: true:  successful
//      false: no such layer found
//
bool
RGraphic::renameLayer(int _num, const char* _newName)
{
  bool   ret=false;
  RLayer *lay;          // pointer which walks through layers

  // Search and rename layer:
  //
  for(lay=layer.first(); lay!=0; lay=layer.next()) {
    if(lay->getNumber()==_num) {
      lay->setName(_newName);
      ret=true;
      break;
    }
  }
  
  if(autoUpdateLayerList) emit updateLayers();
  
  return ret;
}



// Clear layers:
//
void
RGraphic::clearLayers()
{
  layer.clear();
  if(autoUpdateLayerList) emit updateLayers();
}



// Sort layers alphabetic:
//
void 
RGraphic::sortLayers()
{
  RLayer *lay;          // pointer which walks through layers
  RLayer *lay2;         // exchange with this?
  RLayer tmpLay;        // exchanger
  bool   done;          // no more items to sort
  
  do {
    done=true;
    for(lay=layer.first(); lay!=0; lay=layer.next()) {
      if(lay->getFlag(Y_USED)) {
        lay2=layer.next();
        if(lay2 && lay2->getFlag(Y_USED)) {
          if(strcmp(lay->getName(), lay2->getName())>0) {
            tmpLay.setName(lay->getName());
            tmpLay.setNumber(lay->getNumber());
            lay->setName(lay2->getName());
            lay->setNumber(lay2->getNumber());
            lay2->setName(tmpLay.getName());
            lay2->setNumber(tmpLay.getNumber());
            done=false;
          }
        }
        layer.prev();
      }
    }
  }while(!done);

  debugLayers();

}



// Get the number of a layer given by its name:
//
// ret: Number of layer
//      -1: layer doesn't exist!
//
int
RGraphic::getLayerNum(const char* _name)
{
  RLayer *lay;          // pointer which walks through layers
  int num=0;

  // Compare layernames:
  //
  for(lay=layer.first(); lay!=0; lay=layer.next()) {
    if(lay->getFlag(Y_USED)) {
      if(!strcmp(_name, lay->getName())) {
        num=lay->getNumber();
        break;
      }
    }
  }

  return num;
}


// Get the name of a layer given by its number:
//
const char*
RGraphic::getLayerName(int _num)
{
  RLayer *lay;          // pointer which walks through layers

  for(lay=layer.first(); lay!=0; lay=layer.next()) {
    if(lay->getFlag(Y_USED)) {
      if(lay->getNumber()==_num) {
        return lay->getName();
      }
    }
  }
  return 0;
}



// Get index from layer with number _num:
//
// return : index of layer _num
//          -1: no such layer
//
int 
RGraphic::getLayerIndex(int _num)
{
  RLayer *lay;          // pointer which walks through layers

  for(lay=layer.first(); lay!=0; lay=layer.next()) {
    if(lay->getFlag(Y_USED)) {
      if(lay->getNumber()==_num) {
        return layer.at();
      }
    }
  }
  return -1;
}



// Get a layer (pointer):
//
RLayer* 
RGraphic::getLayer(byte _index)
{
  return layer.at(_index);
}



// Is the layer with number _num visible:
//
// return: true : yes
//         false: no 
//
bool
RGraphic::isLayerVisible(int _num)
{
  RLayer *lay;          // pointer which walks through layers

  for(lay=layer.first(); lay!=0; lay=layer.next()) {
    if(lay->getFlag(Y_USED)) {
      if(lay->getNumber()==_num) {
        if(lay->getFlag(Y_VISIBLE)) {
          return true;
        }
        else {
          return false;
        }
      }
    }
  }
  return false;
}



// Switch Layer from on to off and otherwise:
//   (SLOT)
//
void 
RGraphic::switchLayer(const char* _name)
{
  RLayer *lay;          // pointer which walks through layers
  int index=-1;         // Layer name
  int number;           // Layer number

  // Search layername:
  //
  for(lay=layer.first(); lay!=0; lay=layer.next()) {
    if(lay->getFlag(Y_USED)) {
      if(!strcmp(_name, lay->getName())) {
        index=layer.at();
        break;
      }
    }
  }
  
  // Switch on / off layer:
  //
  if(index>=0) {
    layer.at(index)->turnFlag(Y_VISIBLE);
    number=layer.at(index)->getNumber();

    // Show/hide elements:
    //
    RElement *el;          // pointer which walks through elements
  
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getLayer()==number && el->getFlag(E_ACTIVE) && !el->getFlag(E_UNDO)) {
        el->turnFlag(E_VISIBLE);
      }
    }
  }
}



//
// 
//   Parameter handling:
//
//



// Clear parameter list:
//
void
RGraphic::clearParameters()
{
  int i;
  for(i=0; i<(int)parameterString.length() && i<DEF_MAXPARAMETERS; ++i) {
    parameter[i]="";
  }
  parameterString="";
}



// Clear layer parameter list:
//
void
RGraphic::clearLayerParameters()
{
  int i, k;
  for(k=0; k<(int)layer.count() && k<DEF_MAXLAYERS; ++k) {
    for(i=0; i<(int)layerParameterString.length() && i<DEF_MAXPARAMETERS; ++i) {
      layerParameter[k][i]="";
    }
  }
  layerParameterString="";
}




// Add a parameter:
//
// _code: code of parameter $f -> 'f'
// _value: value of parameter
//
void
RGraphic::addParameter(char _code, const char* _value)
{
  RLOG("\nAdd parameter: ");
  RLOG(_code);
  RLOG(" ");
  RLOG(_value);

  parameterString+=_code;
  parameter[parameterString.length()-1] = _value;

  RLOG("\nNew parameter string: ");
  RLOG(parameterString);
}



// Add a layer parameter:
//
// _lay: layer index
// _code: code of parameter $f -> 'f'
// _value: value of parameter
//
void
RGraphic::addLayerParameter(int _lay, 
                            char _code, 
                            const char* _value)

{
  if(_lay>=0 && _lay<DEF_MAXLAYERS) {
    if(!layerParameterString.contains(_code)) {
      layerParameterString+=_code;
    }

    if(layerParameterString.find(_code)<DEF_MAXPARAMETERS) {
      layerParameter[_lay][layerParameterString.find(_code)] = _value;
    }
  }
}



// 
//
// Zoom handling:
//
//



// Get Borders of graphic:
//
void  
RGraphic::getBorders(float& _left, float& _top, 
                     float& _right, float& _bottom)
{
  RElement *el;                 // pointer which walks through elements
  bool elFound=false;           // Is there a visible element?
  float minX, minY, maxX, maxY; // Borders of graphic

  minX=minY=maxX=maxY=0.0;

  // Get the outher limits of the graphic:
  //
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE)) {
      elFound=true;
      if(el->left()  <minX) minX = el->left();
      if(el->bottom()<minY) minY = el->bottom();
      if(el->right() >maxX) maxX = el->right();
      if(el->top()   >maxY) maxY = el->top();
    }
  }

  // Prevent wrong zoom (no elements or one 0 or 90 line):
  //
  if(!elFound) {
    minX    = -20.0;              // min X-Coordinate
    maxX    = 300.0;              // max X-Coordinate
    minY    = -20.0;              // min Y-Coordinate
    maxY    = 200.0;              // max Y-Coordinate
  }
  else {
    if(minX>=maxX-DEF_MMTOL) {
      minX-=10.0;
      maxX+=10.0;
    }
    if(minY>=maxY-DEF_MMTOL) {
      minY-=10.0;
      maxY+=10.0;
    }
  }

  _left=minX;
  _top=maxY;
  _right=maxX;
  _bottom=minY;
}



// Set auto zoom for graphic:
//
void
RGraphic::zoomAuto(QRect* _rect)
{
  float minX, minY, maxX, maxY; // Borders of graphic
  getBorders(minX, maxY, maxX, minY);

  // Adjust for relative zero:
  //
  if(!getFlag(G_PRINTPREV)) {
    if(relZeroX<minX) minX=relZeroX;
    if(relZeroY<minY) minY=relZeroY;
    if(relZeroX>maxX) maxX=relZeroX;
    if(relZeroY>maxY) maxY=relZeroY;
  }

  // Set new zoom:
  //
  setZoom(minX, maxY, maxX, minY, DEF_ZOOMBORDER, _rect);
}



// Zoom tagged elements:
//
void  
RGraphic::zoomTagged(QRect* _rect)
{
  RElement *el;                 // pointer which walks through elements
  bool elFound=false;           // Is there a visible tagged element?
  float minX, minY, maxX, maxY; // Borders of tagged elements

  minX=minY=DEF_AREAMAX;
  maxX=maxY=DEF_AREAMIN;

  // Get the outher limits of the tagged elements:
  //
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE) && el->getFlag(E_TAGGED)) {
      elFound=true;
      if(el->left()  <minX) minX = el->left();
      if(el->bottom()<minY) minY = el->bottom();
      if(el->right() >maxX) maxX = el->right();
      if(el->top()   >maxY) maxY = el->top();
    }
  }

  // Prevent wrong zoom (no elements or one 0' or 90' line):
  //
  if(!elFound) {
    getBorders(minX, maxY, maxX, minY);
  }
  else {
    if(minX>=maxX-DEF_MMTOL) {
      minX-=10.0;
      maxX+=10.0;
    }
    if(minY>=maxY-DEF_MMTOL) {
      minY-=10.0;
      maxY+=10.0;
    }
  }

  
  // Set new zoom:
  //
  setZoom(minX, maxY, maxX, minY, DEF_ZOOMBORDER, _rect);
}




// Set a new zoom so that:
//   the screen rectangle _left, _top, _right, _bottom
//   is visible
//
void
RGraphic::setZoom(int _left,  int _top, 
                  int _right, int _bottom, 
                  int _zoomBorder,
                  QRect* _rect)
{
  setZoom(screenToRealX(_left),
          screenToRealY(_top),
          screenToRealX(_right),
          screenToRealY(_bottom),
          _zoomBorder,
          _rect);
}




// Set a new zoom so that:
//   the graphic rectangle _left, _top, _right, _bottom
//   is visible
//
void
RGraphic::setZoom(float _left,  float _top, 
                  float _right, float _bottom, 
                  int   _zoomBorder,
                  QRect* _rect)
{
  float zoomX=480.0,    // Zoom for X-Axis
        zoomY=640.0;    // Zoom for Y-Axis
                        //   (Set smaller one)
  float dum;            // Dummy for switching values

  // No specific rectangle:
  //   Take the whole widget
  //
  if(!_rect) {
    _rect = new QRect(rect().left(), rect().top(), 
                      rect().width(), rect().height());
  }

  // Switch left/right and top/bottom is necessary:
  //
  if(_left>_right) { dum=_left; _left=_right; _right=dum; }
  if(_bottom>_top) { dum=_top; _top=_bottom; _bottom=dum; }

  // Get zoom in X and zoom in Y:
  //
  if(_right-_left!=0.0) zoomX = _rect->width()  / (_right-_left);
  if(_top-_bottom!=0.0) zoomY = _rect->height() / (_top-_bottom);

  // Take smaller zoom:
  //
  if(zoomX<zoomY) {
    if(_rect->width()!=0) {
      zoom=((float)(_rect->width()-2*_zoomBorder)) /
             (float)_rect->width()*zoomX;
    }
  }
  else {
    if(_rect->height()!=0) {
      zoom=((float)(_rect->height()-2*_zoomBorder)) /
             (float)_rect->height()*zoomY;
    }
  }

  if(zoom<0.0) zoom*=-1.0;
  
  // Borders in pixel after zoom
  //
  int pixLeft  =(int)(_left  *zoom);    
  int pixTop   =(int)(_top   *zoom);
  int pixRight =(int)(_right *zoom);
  int pixBottom=(int)(_bottom*zoom);

  // Set new offet for zero point:
  //
  offsetX = -                pixLeft + (_rect->width() -pixRight +pixLeft)/2;
  offsetY =  _rect->height()+pixTop  - (_rect->height()-pixBottom+pixTop )/2;
	
	currentZoomRect.setRect(_rect->x(), _rect->y(), _rect->width(), _rect->height());

  //emit zoomChanged();
}



// Decrease zoom by 1.5:
//
void
RGraphic::zoomDecrease(float _fact)
{
  int nWidth  = (int)((float)width() *_fact);
  int nHeight = (int)((float)height()*_fact);

  int bw = (width() -nWidth )/2;
  int bh = (height()-nHeight)/2;

  setZoom( screenToRealX(bw),
           screenToRealY(bh),
           screenToRealX(width()-bw),
           screenToRealY(height()-bh));
}



// Increase zoom by 1.5:
//
void
RGraphic::zoomIncrease(float _fact)
{
  if(zoom<200.0) {
    int nWidth  = (int)((float)width() /_fact);
    int nHeight = (int)((float)height()/_fact);

    int bw = (width() -nWidth )/2;
    int bh = (height()-nHeight)/2;

    setZoom( screenToRealX(bw),
             screenToRealY(bh),
             screenToRealX(width()-bw),
             screenToRealY(height()-bh));
  }
}



// Set same zoom like in graphic _gr:
//   used for synchronizing zoom
//
void 
RGraphic::setZoomFromGraphic(RGraphic* _gr)
{
  if(_gr) {
    zoom=_gr->getZoom();
    offsetX=_gr->getOffsetX()-(_gr->width()-width())/2;
    offsetY=_gr->getOffsetY();
    //repaint();
  }
}



//
//
// Attributes:
//
//

// Set current color:
//
void
RGraphic::setColor(QColor _col)
{
  currentColor=_col;
}


// Set current color:
//
void
RGraphic::setColor(int _col)
{
  currentColor=numberToColor(_col);
}


// Set current width:
//
void
RGraphic::setWidth(int _width)
{
  currentWidth=_width;
}


// Set current style:
//
void
RGraphic::setStyle(Qt::PenStyle _style)
{
  currentStyle=_style;
}


// Translate a color into a number in the 256 color table
//
int
RGraphic::colorToNumber(QColor _col)
{
  if(_col==Qt::black      ) return 0;
  if(_col==Qt::darkBlue   ) return 1;
  if(_col==Qt::darkGreen  ) return 2;
  if(_col==Qt::darkCyan   ) return 3;
  if(_col==Qt::darkRed    ) return 4;
  if(_col==Qt::darkMagenta) return 5;
  if(_col==Qt::darkYellow ) return 6;
  if(_col==Qt::lightGray  ) return 7;
  if(_col==Qt::darkGray   ) return 8;
  if(_col==Qt::blue       ) return 9;
  if(_col==Qt::green      ) return 10;
  if(_col==Qt::cyan       ) return 11;
  if(_col==Qt::red        ) return 12;
  if(_col==Qt::magenta    ) return 13;
  if(_col==Qt::yellow     ) return 14;
  if(_col==Qt::white      ) return 15;

  return 0;
}


// Translate a number from the 256 colors table into a color object
QColor
RGraphic::numberToColor(int _num)
{
  switch(_num) {
    default:
    case 0:  return Qt::black      ; break;
    case 1:  return Qt::darkBlue   ; break;
    case 2:  return Qt::darkGreen  ; break;
    case 3:  return Qt::darkCyan   ; break;
    case 4:  return Qt::darkRed    ; break;
    case 5:  return Qt::darkMagenta; break;
    case 6:  return Qt::darkYellow ; break;
    case 7:  return Qt::lightGray  ; break;
    case 8:  return Qt::darkGray   ; break;
    case 9:  return Qt::blue       ; break;
    case 10: return Qt::green      ; break;
    case 11: return Qt::cyan       ; break;
    case 12: return Qt::red        ; break;
    case 13: return Qt::magenta    ; break;
    case 14: return Qt::yellow     ; break;
    //case 15: return Qt::white      ; break;
    case 15: return Qt::black      ; break;
  }
}


// Gets the style from a style name
//
Qt::PenStyle
RGraphic::nameToStyle(QString _name)
{
  if(_name.lower()=="continuous") return Qt::SolidLine;
  if(_name.lower()=="dash")       return Qt::DashLine;
  if(_name.lower()=="dot")        return Qt::DotLine;
  if(_name.lower()=="dashdot")    return Qt::DashDotLine;
  if(_name.lower()=="dashdotdot") return Qt::DashDotDotLine;

  return Qt::SolidLine;
}


// Gets the style from a style number
//
Qt::PenStyle
RGraphic::numberToStyle(int _num)
{
  switch(_num) {
    default:
    case 0: return Qt::SolidLine;      break;
    case 1: return Qt::DashLine;       break;
    case 2: return Qt::DotLine;        break;
    case 3: return Qt::DashDotLine;    break;
    case 4: return Qt::DashDotDotLine; break;
  }

  return Qt::SolidLine;
}



QString
RGraphic::styleToName(Qt::PenStyle _style)
{
  QString ret;

  switch(_style) {
    default:
    case Qt::SolidLine: ret="CONTINUOUS"; break;
    case Qt::DashLine: ret="DASH"; break;
    case Qt::DotLine: ret="DOT"; break;
    case Qt::DashDotLine: ret="DASHDOT"; break;
    case Qt::DashDotDotLine: ret="DASHDOTDOT"; break;
  }

  return ret;
}



// 
//
// Paint Actions:
//
//


// Paint event:
//
void 
RGraphic::paintEvent(QPaintEvent *_ev)
{
  RLOG("\npaintEvent");

  QFrame::paintEvent(_ev);
  
  /*QRect updateRect;
  
  if(_ev) updateRect=_ev->rect();
  else    updateRect=rect();*/
  
  //if(!eventsEnabled) return;
  
  if(drawing) return;
  else        drawing=true;
  
  RPainter paint;        // painter

  paint.begin(this);
  paint.setPen(black);
  paint.setRasterOp(CopyROP);
  paint.setClipRect(2, 2, width()-4, height()-4);

  draw(&paint);

  paint.end();

  drawing=false;

  // Draw active border:
  //
  if(active) drawActiveBorder(true);
  
  prevVisible=false;
  markVisible=false;
  prevElementVisible=false;
  prevElementsVisible=false;
  
  RLOG("\nSimulate mouse move in paintEvent:");
  if(previewUpdate) distributeEvent(REVENT_MOUSEMOVE);
}



// Resize Event:
//
void    
RGraphic::resizeEvent(QResizeEvent* /*_ev*/)
{
  //QWidget::resizeEvent(_ev);
  
  RLOG("\nresizeEvent");
  
  // Check if size has really changed:
  //
  /*if(size()!=_ev->oldSize()) {
    emit zoomChanged();
  }*/
}



// Mouse enters the graphic:
//
void 
RGraphic::enterEvent(QEvent *)
{
  distributeEvent(REVENT_MOUSEENTER);
}




// Mouse leaves the graphic:
//
void 
RGraphic::leaveEvent(QEvent *)
{
  //if(snap) snap->stop();
  distributeEvent(REVENT_MOUSELEAVE);
}



// Set simulation delay:
//
void  
RGraphic::setDelay(int _val)
{
  delayPosition=_val;
  delay=int((sqrt((float)DEF_SPEEDRES)-sqrt((float)_val))*(float)RCONFIG->getSettingDouble("Simulation:DelayFactor"));
}



// Simulate smooth: 
//   ->see also: draw();
//               simulate();
//
void 
RGraphic::simulateSmooth()
{
  if(drawing) return;
  
  smooth=true;
  simulate();
  smooth=false;
}



// Simulate:
//   ->see also: draw();
//               simulateSmooth();
//
void 
RGraphic::simulate()
{
  if(drawing) return;
  else        drawing=true;

  RAppWin::getRAppWin()->enableAll(false);

  QFrame::repaint();
  
  RPainter paint;        // painter

  paint.begin(this);
  paint.setPen(black);
  paint.setRasterOp(CopyROP);
  paint.setClipRect(2, 2, width()-4, height()-4);

  draw(&paint, true);

  paint.end();

  drawing=false;

  // Draw active border:
  //
  if(active) drawActiveBorder(true);

  prevVisible=false;
  markVisible=false;
  prevElementVisible=false;
  prevElementsVisible=false;
  distributeEvent(REVENT_MOUSEMOVE);

  RAppWin::getRAppWin()->enableAll(true);
}



// Wait (Simulation delay):
//
void
RGraphic::simulationDelay()
{
  qApp->processEvents(1);

  int t1;
  for(t1=0; t1<(smooth ? (delay/16) : delay); ++t1) {
    // Just wait... (and process events!)
    //
    qApp->processEvents(1);
  }

}



// Get the cutting way (excluding rapid move) in this graphic (for info-widget)
//
float 
RGraphic::getCuttingWay()
{
  RElement *el;            // pointer which walks through elements
  float ret=0.0;           // return value
  
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE) && el->getElementTyp()!=T_RAPID) {
      ret+=el->getLength();
    }
  }
  
  return ret;
}



// Get the total way (including rapid move) in this graphic (for info-widget)
//
float 
RGraphic::getTotalWay()
{
  RElement *el;            // pointer which walks through elements
  RElement rapidEl(this);  // Element for rapid move
  float ret=0.0;           // return value
  float lastX=0.0,         // Last position
        lastY=0.0;         //
  
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE)) {
    
      if(!mtCompFloat(el->getX1(), lastX) || 
         !mtCompFloat(el->getY1(), lastY)    ) {
        rapidEl.createRapid(lastX,
                            lastY,
                            el->getX1(),
                            el->getY1());
        ret+=rapidEl.getLength();
      }
                                
      ret+=el->getLength();
      
      // Save last pos for rapid move:
      //
      if(el->getElementTyp()==T_TEXT) {
        lastX=el->getX1();
        lastY=el->getY1();
      }
      else {
        lastX=el->getX2();
        lastY=el->getY2();
      }
    }
  }
  
  return ret;
}




// Draw graphic to device _paint:
//
// _sim: simulation mode?
// _print: printing mode (no clipping)
// _specialFlags: Special flags for this single drawing
//                G_GRID, G_GRIDWIDTH, G_ZERO, G_INFO, G_TAGEDONLY, G_TAGGEDCOLOR
//                or G_DEFAULT if we don't want to use any special flags
//
void 
RGraphic::draw(RPainter* _paint, bool _sim, bool _print, int _specialFlags)
{
	if(width()<=0 || height()<=0 || !isVisible()) {
		return;
	}
	
	QColor oriColor=currentColor;
	int oriWidth=currentWidth;
	Qt::PenStyle oriStyle=currentStyle;
	
	if(_print) printing=true;

  RElement *el;            // pointer which walks through elements
  RElement rapidEl(this);  // Element for rapid move

  if(_paint) {

    // Draw Zero points:
    //
    if(_specialFlags&G_ZERO || (_specialFlags==G_DEFAULT && getFlag(G_ZERO))) {
      drawAbsoluteZero(_paint);
      drawRelativeZero(_paint);
    }

    // Draw Grid:
    //
    if(_specialFlags&G_GRID || (_specialFlags==G_DEFAULT && getFlag(G_GRID))) {
      drawGrid(_paint);
    }

    // Draw Abs graphic limits:
    //
    if(true) {
      if(screenToRealX(width())>DEF_AREAMAX) {
        _paint->fillRect(realToScreenX(DEF_AREAMAX), 0,
                         width(), height(), QBrush(gray));
      }
      if(screenToRealX(0)<DEF_AREAMIN) {
        _paint->fillRect(realToScreenX(DEF_AREAMIN), 0,
                         -width(), height(), QBrush(gray));
      }
      if(screenToRealY(0)>DEF_AREAMAX) {
        _paint->fillRect(0, realToScreenY(DEF_AREAMAX),
                         width(), -height(), QBrush(gray));
      }
      if(screenToRealY(height())<DEF_AREAMIN) {
        _paint->fillRect(0, realToScreenY(DEF_AREAMIN),
                         width(), height(), QBrush(gray));
      }
    }

    // Set flags for highlighted elements in broken contours:
    //
    if(getFlag(G_BROKEN) && updateBroken) {
      highlightBrokenContours();
    }

    float lastX=0.0, 
          lastY=0.0; 

    // Draw Graphic:
    //
    for(el=element.first(); el!=0; el=element.next()) {
      if(el->getFlag(E_VISIBLE)) {
      
        // Draw Rapid move:
        //
        if(getFlag(G_RAPID)) {
          if(!mtCompFloat(el->getX1(), lastX) || 
             !mtCompFloat(el->getY1(), lastY)    ) {
            rapidEl.createRapid(lastX,
                                lastY,
                                el->getX1(),
                                el->getY1());
            if(rapidEl.isOnScreen()) {
              if(_sim) {
                if(!smooth) {
                  rapidEl.draw(_paint);
                  simulationDelay();
                }
                else {
                  rapidEl.drawSmooth(_paint);
                }
              }
              else {
                rapidEl.draw(_paint);
              }
            }
          }
        }


        // Draw Element:
        //
        if(el->isOnScreen() || printing) {

          // Show details (start point):
          //
          if(getFlag(G_DETAILS)) {
            drawEndPoint(_paint, 
                         el->getX1(), el->getY1());
          }

          if(_sim) {
            if(!smooth) {
              el->draw(_paint);
              simulationDelay();
            }
            else {
              el->drawSmooth(_paint);
            }
          }
          else {
            el->draw(_paint);
          }
        
          // Show Details (direction / endpoint):
          //
          if(getFlag(G_DETAILS) &&
             el->getElementTyp()!=T_POINT &&
             el->getElementTyp()!=T_TEXT     ) {

            if(!mtCompFloat(el->getX1(), lastX, 0.005) || 
               !mtCompFloat(el->getY1(), lastY, 0.005) ||
               el->getElementTyp()==T_CIRCLE              ) {
                  
              drawDirection(_paint,
                            el->getMiddleX(), 
                            el->getMiddleY(),
                            el->getMiddleDirection());
            }

            drawEndPoint(_paint, 
                         el->getX2(), el->getY2());
          }
        }
        
        // Save last pos for rapid move:
        //
        if(el->getElementTyp()==T_TEXT) {
          lastX=el->getX1();
          lastY=el->getY1();
        }
        else {
          lastX=el->getX2();
          lastY=el->getY2();
        }
        
      }
    }

  }

  setColor(oriColor);
  setWidth(oriWidth);
  setStyle(oriStyle);
	
	printing=false;
}



// Draw a single element:
// _relZero: Delete/restore relative zero?
//
void
RGraphic::drawElement(int _ind, bool _relZero)
{
  RPainter paint;        // painter

  paint.begin(this);

  if(_relZero) drawRelativeZero(&paint);

  paint.setPen(black);
  paint.setRasterOp(CopyROP);
  paint.setClipRect(2, 2, width()-4, height()-4);

  if(_ind>0 && _ind<(int)element.count() && element.at(_ind)) {
    element.at(_ind)->draw(&paint);
  }
  else {
    if(element.current()) {
      element.current()->draw(&paint);
    }
  }

  if(_relZero) drawRelativeZero(&paint);

  paint.end();

}




// Draw a single element:
//
void
RGraphic::drawElement(RElement* _el)
{
  if(_el && _el->getGraphic()==this) {
  
    RPainter paint;        // painter

    paint.begin(this);
    paint.setPen(black);
    paint.setRasterOp(CopyROP);
    paint.setClipRect(2, 2, width()-4, height()-4);

    _el->draw(&paint);
  
    paint.end();
  }

}



// Draw the absolute zero:
//
void 
RGraphic::drawAbsoluteZero(RPainter* _paint)
{
  if(_paint) {
  	
  	// Screen: Draw a pixmap:
  	//
  	if(!printing) {
      QPixmap zeroPixmap(zero_xpm);
      _paint->drawPixmap(realToScreenX(0.0)-zeroPixmap.width()/2,
                         realToScreenY(0.0)-zeroPixmap.height()/2, 
                         zeroPixmap);
  	}
  	
  	// Printer: draw lines and circles:
  	//
  	else {
  		float zr = RCONFIG->getSettingDouble("Print:PrintZeroSize")/2;
  		
      _paint->setPen(red);
  		_paint->setBrush(NoBrush);
  	
  		_paint->drawLine(realToScreenX(0.0)-(int)(zr*printerFactorX), 
  			               realToScreenY(0.0),
  			               realToScreenX(0.0)+(int)(zr*printerFactorX), 
  			               realToScreenY(0.0));
  		
  		_paint->drawLine(realToScreenX(0.0), 
  			               realToScreenY(0.0)-(int)(zr*printerFactorY),
                       realToScreenX(0.0), 
                       realToScreenY(0.0)+(int)(zr*printerFactorY));
  		
  		_paint->drawEllipse(realToScreenX(0.0)-(int)(zr/2*printerFactorX), 
  			                  realToScreenY(0.0)-(int)(zr/2*printerFactorY),
  			                  2*(int)(zr/2*printerFactorX), 
  			                  2*(int)(zr/2*printerFactorY));
  		
  		_paint->setBrush(red);
  		_paint->setPen(NoPen);
  		
  		_paint->drawPie(realToScreenX(0.0)-(int)(zr/2*printerFactorX), 
  			              realToScreenY(0.0)-(int)(zr/2*printerFactorY),
                      2*(int)(zr/2*printerFactorX), 
                      2*(int)(zr/2*printerFactorY),
                      2880, 1440);
  		
  	  _paint->setBrush(NoBrush);
      _paint->setPen(black);
  	}
  }
}



/**
 * Deletes the relative zero point.
 */
void
RGraphic::delRelativeZero(RPainter* _paint)
{
  if(_paint) {
  	
  	// Draw a white pixmap:
  	//
  	if(!printing) {
      QPixmap zeroPixmap(zerorelwhite_xpm);
      _paint->drawPixmap(realToScreenX(relZeroX)-zeroPixmap.width()/2,
                         realToScreenY(relZeroY)-zeroPixmap.height()/2,
                         zeroPixmap);
  	}
  }
}



/**
 * Draws the relative zero poiint.
 */
void
RGraphic::drawRelativeZero(RPainter* _paint)
{
  if(_paint &&
     (!mtCompFloat(relZeroX, 0.0) ||
      !mtCompFloat(relZeroY, 0.0)    )  ) {

    _paint->setRasterOp(XorROP);
  	
  	// Screen: Draw a pixmap:
  	//
  	if(!printing) {
#ifndef DEF_WINDOWS
      QPixmap zeroPixmap(zerorel_xpm);
      _paint->drawPixmap(realToScreenX(relZeroX)-zeroPixmap.width()/2,
                         realToScreenY(relZeroY)-zeroPixmap.height()/2,
                         zeroPixmap);
#else
      float zr = 8;
  		
      _paint->setPen(cyan);
  		_paint->setBrush(NoBrush);
  	
  		_paint->drawLine(realToScreenX(relZeroX)-(int)(zr/2),
  			               realToScreenY(relZeroY),
  			               realToScreenX(relZeroX)+(int)(zr/2),
  			               realToScreenY(relZeroY));
  		
  		_paint->drawLine(realToScreenX(relZeroX),
  			               realToScreenY(relZeroY)-(int)(zr/2),
                       realToScreenX(relZeroX),
                       realToScreenY(relZeroY)+(int)(zr/2));
  		
  		_paint->drawEllipse(realToScreenX(relZeroX)-(int)(zr/2),
  			                  realToScreenY(relZeroY)-(int)(zr/2),
  			                  2*(int)(zr/2),
  			                  2*(int)(zr/2));
  		
  		_paint->setBrush(cyan);
  		_paint->setPen(NoPen);
  		
  		_paint->drawPie(realToScreenX(relZeroX)-(int)(zr/2),
  			              realToScreenY(relZeroY)-(int)(zr/2),
                      2*(int)(zr/2),
                      2*(int)(zr/2),
                      2880, 1440);
  		
  	  _paint->setBrush(NoBrush);
      _paint->setPen(black);
#endif
  	}
  	
  	// Printer: draw lines and circles:
  	//
  	else {
  		float zr = RCONFIG->getSettingDouble("Print:PrintZeroSize")/3;
  		
      _paint->setPen(red);
  		_paint->setBrush(NoBrush);
  	
  		_paint->drawLine(realToScreenX(relZeroX)-(int)(zr/2*printerFactorX),
  			               realToScreenY(relZeroY),
  			               realToScreenX(relZeroX)+(int)(zr/2*printerFactorX),
  			               realToScreenY(relZeroY));
  		
  		_paint->drawLine(realToScreenX(relZeroX),
  			               realToScreenY(relZeroY)-(int)(zr/2*printerFactorY),
                       realToScreenX(relZeroX),
                       realToScreenY(relZeroY)+(int)(zr/2*printerFactorY));
  		
  		_paint->drawEllipse(realToScreenX(relZeroX)-(int)(zr/2*printerFactorX),
  			                  realToScreenY(relZeroY)-(int)(zr/2*printerFactorY),
  			                  2*(int)(zr/2*printerFactorX),
  			                  2*(int)(zr/2*printerFactorY));
  		
  		_paint->setBrush(red);
  		_paint->setPen(NoPen);
  		
  		_paint->drawPie(realToScreenX(relZeroX)-(int)(zr/2*printerFactorX),
  			              realToScreenY(relZeroY)-(int)(zr/2*printerFactorY),
                      2*(int)(zr/2*printerFactorX),
                      2*(int)(zr/2*printerFactorY),
                      2880, 1440);
  		
  	  _paint->setBrush(NoBrush);
      _paint->setPen(black);
  	}
  	
  	_paint->setRasterOp(CopyROP);
  }
}



// Draw the grid:
//
void 
RGraphic::drawGrid(RPainter* _paint)
{
  if(_paint) {
    float startCX,           // start/stop drawing
          stopCX,            // the grid here
          startCY,
          stopCY;
    float gpX;               // Grid point X
    float gpY;               // Grid point Y
    float mg;                // Multi of gridwidth
    float wleft,             // window-border in
          wtop,              // real-coordinates
          wright,
          wbottom;
                              // Grid width from INI-File
    float gridWidth=RCONFIG->getSettingDouble("CAD:GridWidth");
    float c;                  // counter
    int   bigStep=10;         // big step for calculating the raster border
  	
    if(printing) {
    	wleft   = screenToRealX(0);
      wtop    = screenToRealY(0);
      wright  = screenToRealX(currentZoomRect.width());
      wbottom = screenToRealY(currentZoomRect.height());
    }
  	else {
      wleft   = screenToRealX(0);
      wtop    = screenToRealY(0);
      wright  = screenToRealX(width());
      wbottom = screenToRealY(height());
  	}
  
    if(wleft  <DEF_AREAMIN) wleft  =DEF_AREAMIN;
    if(wright >DEF_AREAMAX) wright =DEF_AREAMAX;
    if(wbottom<DEF_AREAMIN) wbottom=DEF_AREAMIN;
    if(wtop   >DEF_AREAMAX) wtop   =DEF_AREAMAX;
  
    mg = gridWidth;
  	
    // Adjust to minimum grid space for printing:
  	//
    if(printing) {
    	while(mg*zoom<6.0*printerFactorX && mg<10000.0) { mg*=10.0; }
    }
  	
  	// Adjust to minimum grid space for screen:
    //
  	else {
  		while(mg*zoom<6.0 && mg<10000.0) { mg*=10.0; }
  	}
  	
  	currentGridWidth = mg;
  
    // Get the range where to draw the grid
    // (in factor 100 of grid pieces)
    //
    if(wleft>=0.0) {
      for(c=0.0; c<=DEF_AREAMAX; c+=mg*bigStep) {
        if(c>wleft) break;
      }
      startCX = c-mg*bigStep;
    }
    else {
      for(c=0.0; c>=DEF_AREAMIN; c-=mg*bigStep) {
        if(c<wleft) break;
      }
      startCX = c;
    }
  
    if(wbottom>=0.0) {
      for(c=0.0; c<=DEF_AREAMAX; c+=mg*bigStep) {
        if(c>wbottom) break;
      }
      startCY = c-mg*bigStep;
    }
    else {
      for(c=0.0; c>=DEF_AREAMIN; c-=mg*bigStep) {
        if(c<wbottom) break;
      }
      startCY = c;
    }
  
    for(c=wleft; c<=DEF_AREAMAX; c+=mg*bigStep) {
      if(c>wright || c<wleft) break;
    }
    stopCX = c;
  
    for(c=wbottom; c<=DEF_AREAMAX; c+=mg*bigStep) {
      if(c>wtop || c<wbottom) break;
    }
    stopCY = c;

    // Meta grid (factor 10):
    //
    if( !printing && RCONFIG->getSettingInt("CAD:MetaGrid")==1 ) {
      _paint->setPen( QPen(lightGray, 0, DotLine) );
      _paint->setRasterOp(CopyROP);
      for(gpX=startCX; gpX<=stopCX; gpX+=mg*10) {
        _paint->drawLine(realToScreenX(gpX), 0,
                         realToScreenX(gpX), height());
      }
      for(gpY=startCY; gpY<=stopCY; gpY+=mg*10) {
        _paint->drawLine(0, realToScreenY(gpY),
                         width(), realToScreenY(gpY));
      }
      _paint->setPen(gray);
    }
  	
  	// Grid for printing under windows:
  	//
    if(printing && RUNNING_WINDOWS) {
      for(gpX=startCX; gpX<=stopCX; gpX+=mg) {
        for(gpY=startCY; gpY<=stopCY; gpY+=mg) {
          _paint->drawRect(realToScreenX(gpX)-1, 
                           realToScreenY(gpY)-1, 3, 3);
      	}
      }
    }
  	
  	// Grid on screen and for printing under linux:
  	//
  	else {
  		for(gpX=startCX; gpX<=stopCX; gpX+=mg) {
        for(gpY=startCY; gpY<=stopCY; gpY+=mg) {
          _paint->drawPoint(realToScreenX(gpX), 
                            realToScreenY(gpY));
        }
      }
  	}
  
    // Output grid wide at bottom right of draw window:
    //
  	if(!printing && getFlag(G_GRIDWIDTH)) {
  		char strGridWide[16];
      sprintf(strGridWide, "%.2f", mg);
      _paint->setFont(QFont("helvetica", RCONFIG->getSettingInt("Application:FontSize1")));
  		
  		if(printing) {
        _paint->drawText(0, 0, currentZoomRect.width()-4, currentZoomRect.height()-4,
                         AlignRight|AlignBottom, 
                         strGridWide);
  		}
  		else {
        _paint->drawText(0, 0, width()-4, height()-4,
                         AlignRight|AlignBottom, 
                         strGridWide);
  		}
  	}
  }
}



// Draw border for activation:
//
void  
RGraphic::drawActiveBorder(bool _draw)
{
  RPainter paint;        // painter
  paint.begin(this);
  paint.setRasterOp(CopyROP);

  if(_draw) paint.setPen(black);
  else      paint.setPen(white);

  paint.drawRect(2, 2, width()-4, height()-4);

  RLOG("\nwidth: ");
  RLOG(width());

  paint.end();
}



// Draw direction:
//
void 
RGraphic::drawDirection(RPainter* _paint,
                        float _dx, float _dy,
                        float _dang)
{
  if(_paint) {
    int  px1, py1, px2, py2, px3, py3;
    int  dsize=RCONFIG->getSettingInt("Simulation:ArrowSize");
    //if(printing) dsize*=4;

    px1 = realToScreenX(_dx+cos((_dang-130.0)/ARAD) * dsize/zoom);
    py1 = realToScreenY(_dy+sin((_dang-130.0)/ARAD) * dsize/zoom);

    px2 = realToScreenX(_dx+cos((_dang+130.0)/ARAD) * dsize/zoom);
    py2 = realToScreenY(_dy+sin((_dang+130.0)/ARAD) * dsize/zoom);

    px3 = realToScreenX(_dx+cos(_dang/ARAD) * dsize/zoom);
    py3 = realToScreenY(_dy+sin(_dang/ARAD) * dsize/zoom);
    
    _paint->setPen(magenta);
    _paint->moveTo(px1, py1);
    _paint->lineTo(px2, py2);
    _paint->lineTo(px3, py3);
    _paint->lineTo(px1, py1);
    _paint->setPen(black);
  }
}



// Draw end point:
//
void 
RGraphic::drawEndPoint(RPainter* _paint,
                       float _ex, float _ey)
{
  if(_paint) {
    int iex=realToScreenX(_ex);
    int iey=realToScreenY(_ey);
    int isize=RCONFIG->getSettingInt("Simulation:EndpointSize");
    
    _paint->setPen(magenta);
    _paint->drawRect(iex-isize, iey-isize, 
                     isize*2, isize*2);
    _paint->setPen(black);
  }
}



// Draw preview mark (circle) from real coordinates:
//
//   _drawNew: true: draw new preview
//             false: delete only old prev
//
void
RGraphic::drawMark(float _x, float _y, bool _drawNew)
{
  drawMark(realToScreenX(_x),
           realToScreenY(_y),
           _drawNew);
}



// Draw preview mark (circle) from screen coordinates:
//
//   _drawNew: true: draw new preview
//             false: delete only old prev
//   _xor: draw anyway in xor mode at last pos. 
//         (for removing temporarly during other drawing actions)
//
void
RGraphic::drawMark(int _x, int _y, bool _drawNew, bool _xor)
{
  static int lastX=DEF_AREAMAX;       // Last used X-Pos for marking
  static int lastY=DEF_AREAMAX;       // Last used Y-Pos for marking
  
  int markSize = RCONFIG->getSettingInt("CAD:MarkSize");
  RPainter paint;        // painter

  if(paint.begin(this)) {
    paint.setPen(cyan);
    paint.setRasterOp(XorROP);
    paint.setClipRect(2, 2, width()-4, height()-4);

    // Delete old preview:
    //
    if((markVisible || _xor) && 
       lastX<DEF_AREAMAX && 
       lastY<DEF_AREAMAX    ) {
      paint.drawEllipse(lastX-markSize, lastY-markSize,
                        markSize*2, markSize*2);
      if(!_xor) markVisible=false;
      else      markVisible=!markVisible;
    }

    // Draw new preview:
    //
    if(_drawNew && markEnabled) {
      paint.drawEllipse(_x-markSize, _y-markSize,
                        markSize*2, markSize*2);
      lastX=_x;
      lastY=_y;
      markVisible=true;
    }
  }
  
  paint.end();
}



//
//
// Preview Element handling:
//
//



// Reset Preview Element:
//
void 
RGraphic::resetPreviewElement()
{
  previewElement.reset();
}


// Reset all preview elements from List "preview":
//
void  
RGraphic::resetPreview()
{
  preview.clear();
}



// Set Preview Element:
//
//   returns true if the element has changed
//
bool
RGraphic::setPreviewElement(int _typ,
                            float _v1, float _v2,
                            float _v3, float _v4,
                            float _v5, float _v6,
                            float _v7, uint _flags)
{
  bool ret;
  RElement oldElement;          // Save old previewelement's data
  oldElement.getMeasuresFrom(&previewElement);

  switch(_typ) {
    case T_LINE:
      previewElement.createLine(_v1, _v2, _v3, _v4);
      break;

    case T_RECT:
      previewElement.createRect(_v1, _v2, _v3, _v4);
      break;

    case T_ARC:
      previewElement.createArc(_v1, _v2, _v3, _v4, _v5, _flags==E_REVERSED);
      break;

    case T_CIRCLE:
      previewElement.createCircle(_v1, _v2, _v3, 0.0, 360.0, false);
      break;

    case T_DIMENSION:
      previewElement.createDimension(_v1, _v2, _v3, _v4, _v5, _v6, _v7, _flags);
      break;

    default:
      break;
  }

  // Return true if element has changed:
  //
  ret = !previewElement.compareWith(&oldElement, DEF_MMTOL, true);

  return ret;
}



// set preview element from an existing element:
//
bool  
RGraphic::setPreviewElement(RElement* _el)
{
  if(!previewElement.compareWith(_el)) {
    RLOG("\nPreview Element not equal");
    previewElement.getMeasuresFrom(_el);
    return true;
  }
  else {
    return false;
  }
}


// Draw preview element:
//
//   _drawNew: true: draw new preview
//             false: delete only old prev
//
void
RGraphic::drawPreviewElement(bool _drawNew, bool _xor)
{
  static RElement lastElement(this);

  if(keyCoordsEnabled()) return;

  lastElement.setGraphic(this);


  RPainter paint;        // painter

  paint.begin(this);
  paint.setPen(cyan);
  paint.setRasterOp(XorROP);
  paint.setClipRect(2, 2, width()-4, height()-4);

  // Delete old preview:
  //
  if((prevElementVisible || _xor) && 
     lastElement.getFlag(E_VISIBLE)

     //&&
     //(!_drawNew || !lastElement.compareWith(&previewElement))
     
     ) {
     
    lastElement.draw(&paint);
    if(!_xor) prevElementVisible=false;
    else      prevElementVisible=!prevElementVisible;
  }

  // Draw new preview:
  //
  if(_drawNew

     //&&
     //(!lastElement.compareWith(&previewElement))

     ) {
     
    previewElement.draw(&paint);
    lastElement.getMeasuresFrom(&previewElement);
    prevElementVisible=true;
  }
  
  paint.end();
}



// Fill preview with tagged elements and their border:
//
// _refx/_refy: reference point
//              (prev is relative)
// _all       : Fill with all (not only tagged)
//
void 
RGraphic::fillPreviewWithTagged(float _refx, float _refy, bool _all)
{
  RElement* el;           // element ptr (walks through elements)
  RElement* tmpEl;        // temp element for new elements of preview
  float bx1=DEF_AREAMAX,  // border coordinates
        by1=DEF_AREAMAX,
        bx2=DEF_AREAMIN,
        by2=DEF_AREAMIN;
        
  preview.clear();
  
  // Fill preview list with the elements
  //
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE) && (el->getFlag(E_TAGGED) || _all)) {
    
      // take into preview:
      //
      tmpEl = new RElement(this);
      tmpEl->getMeasuresFrom(el);
      tmpEl->move(-_refx, -_refy);

      // Check border:
      //
      if(tmpEl->left()  <bx1) bx1 = tmpEl->left();
      if(tmpEl->bottom()<by1) by1 = tmpEl->bottom();
      if(tmpEl->right() >bx2) bx2 = tmpEl->right();
      if(tmpEl->top()   >by2) by2 = tmpEl->top();
      
      preview.append(tmpEl);
    }
  }

  // create border:
  //
  borderElement.createRect(bx1, by1, bx2, by2);
}



// Fill preview with clipboard elements and their border:
//
void
RGraphic::fillPreviewWithClipboard()
{
  RElement* el;           // element ptr (walks through elements)
  RElement* tmpEl;        // temp element for new elements of preview
  float bx1=DEF_AREAMAX,  // border coordinates
        by1=DEF_AREAMAX,
        bx2=DEF_AREAMIN,
        by2=DEF_AREAMIN;

  preview.clear();

  // Fill preview list with the elements
  //
  for(el=clipboard()->elementFirst(); el!=0; el=clipboard()->elementNext()) {
    if(el->getFlag(E_VISIBLE)) {

      // take into preview:
      //
      tmpEl = new RElement(this);
      tmpEl->getMeasuresFrom(el);
      tmpEl->delFlag(E_TAGGED);

      // Check border:
      //
      if(tmpEl->left()  <bx1) bx1 = tmpEl->left();
      if(tmpEl->bottom()<by1) by1 = tmpEl->bottom();
      if(tmpEl->right() >bx2) bx2 = tmpEl->right();
      if(tmpEl->top()   >by2) by2 = tmpEl->top();

      preview.append(tmpEl);
    }
  }

  // create border:
  //
  borderElement.createRect(bx1, by1, bx2, by2);
}



// Has the preview moved since the last call?
//
bool
RGraphic::previewMoved(float _x, float _y, float _angle)
{
  static float lx=DEF_AREAMAX;
  static float ly=DEF_AREAMAX;
  static float lAngle=DEF_AREAMAX;
  
  if(mtCompFloat(_x, lx) &&
     mtCompFloat(_y, ly) &&
     mtCompFloat(_angle, lAngle)) {
    return false;
  }
  
  lx=_x;
  ly=_y;
  lAngle=_angle;
  return true;

}



// Draw the preview:
//   _drawNew: true: draw new preview
//             false: delete only old prev
//
void 
RGraphic::drawPreview(float _x, float _y, float _angle, bool _drawNew, bool _xor)
{
  RElement *el;            // pointer which walks through elements
  static float lx=DEF_AREAMAX;
  static float ly=DEF_AREAMAX;
  static float langle=0.0;
  int counter;             // Counter (we don't want to draw more elements 
                           //   than given in ini-file)
  int prevEl = RCONFIG->getSettingInt("CAD:PreviewElements");
                           // Max. number of preview elements
  bool rotate=false;       // Is the preview rotated?
  bool rotate2=false;      // Was the preview rotated?
  
  RPainter paint;          // painter

  paint.begin(this);
  paint.setPen(cyan);
  paint.setRasterOp(XorROP);
  paint.setClipRect(2, 2, width()-4, height()-4);

  RElement* rotatedEl=0;    // Rotated element needed for angle!=0.0
  // Rotate preview?
  if(!mtCompFloat(_angle, 0.0, 0.000001)) rotate=true;
  if(!mtCompFloat(langle, 0.0, 0.000001)) rotate2=true;
  if(rotate || rotate2) rotatedEl = new RElement(this);

  // Delete old preview:
  //
  if((prevElementsVisible || _xor) && 
     !mtCompFloat(lx, DEF_AREAMAX) &&
     !mtCompFloat(ly, DEF_AREAMAX)    ) {

    counter=0;
    for(el=preview.first(); el!=0 && counter<prevEl; el=preview.next()) {
      if(el->getFlag(E_VISIBLE) && el->getElementTyp()!=T_HATCHING) {
        //if(el->isOnScreen()) {
        if(!rotate2) {
          el->draw(&paint, lx, ly, true);
        }
        else {
          if(rotatedEl) {
            rotatedEl->getMeasuresFrom(el);
            rotatedEl->rotate(langle, 0.0, 0.0, false);
            rotatedEl->draw(&paint, lx, ly, true);
          }
        }
        ++counter;
        //}
      }
    }

    if(!rotate2) {
      borderElement.draw(&paint, lx, ly, true);
    }
    
    if(!_xor) prevElementsVisible=false;
    else      prevElementsVisible=!prevElementsVisible;
  }

  // Draw new preview:
  //
  if(_drawNew) {
  
    counter=0;
    for(el=preview.first(); el!=0 && counter<prevEl; el=preview.next()) {
      if(el->getFlag(E_VISIBLE) && el->getElementTyp()!=T_HATCHING) {
        //if(el->isOnScreen()) {

        if(!rotate) {
          el->draw(&paint, _x, _y, true);
        }
        else {
          if(rotatedEl) {
            rotatedEl->getMeasuresFrom(el);
            rotatedEl->rotate(_angle, 0.0, 0.0, false);
            rotatedEl->draw(&paint, _x, _y, true);
          }
        }
        ++counter;
      }
    }

    if(!rotate) {
      borderElement.draw(&paint, _x, _y, true);
    }
    
    lx=_x;
    ly=_y;
    langle=_angle;
    prevElementsVisible=true;
  }

  if(rotatedEl) {
    delete rotatedEl;
  }
  
  paint.end();

}




// Reset Highlight Element:
//
void 
RGraphic::resetHighlightElement()
{
  highlightElement=0;
}



// Set Highlight Element:
//
void 
RGraphic::setHighlightElement(RElement* _el)
{
  highlightElement=_el;
}



// Draw highlight element:
//
//   _drawNew: true: draw new highlight
//             false: delete only old highlight
//
void
RGraphic::drawHighlightElement(bool _drawNew)
{
  static RElement* lastElement=0;

  RPainter paint;        // painter

  // Delete old highlight:
  //   (only if we want to delete it or it has changed)
  //
  if(highlightElementVisible         && 
     lastElement                     &&
     lastElement->getFlag(E_VISIBLE) &&
     (!_drawNew || lastElement!=highlightElement)) {

    bool doXorMark=false;
    if(markVisible) doXorMark=true;
    bool doXorPrev=false;
    if(prevVisible) doXorPrev=true;
    
    if(doXorMark) xorMark();
    if(doXorPrev) xorPreviewElement();
    paint.begin(this);
    paint.setClipRect(2, 2, width()-4, height()-4);
    lastElement->draw(&paint);
    paint.end();
    if(doXorMark) xorMark();
    if(doXorPrev) xorPreviewElement();
    highlightElementVisible=false;
  }

  // Draw new highlight:
  //   (only if it has changed)
  //
  if(_drawNew                      && 
     lastElement!=highlightElement    ) {
    if(highlightElement) {
      paint.begin(this);
      paint.setClipRect(2, 2, width()-4, height()-4);
      highlightElement->draw(&paint);
      paint.end();
      highlightElementVisible=true;
    }
  }
  
  lastElement=highlightElement;
}



// Set new snap filter:
//
//   _typ: T_ALL, T_STRAIGHT, T_LINE, ...
//   _exception: don't snap to this element (or 0)
//
void  
RGraphic::setHighlightFilter(elementEnum _typ, RElement* _exception)
{
  snapFilterTyp = _typ;
  snapFilterException = _exception;
}



// Set highlight-flag for elements in open 
//   (broken) contours:
//
// Highlight Elements in broken contours:
//
void
RGraphic::highlightBrokenContours()
{
  float     fstContX, fstContY;     // first position of a contour
  float     scdContX, scdContY;     // second position of a contour (cut from center)
  float     currentX, currentY;     // current pos
  RElement* fstContEl;              // first element of this contour
  RElement* prevEl=0;               // previous element (set broken 2 flag)
  RElement* el;                     // Pointer which walks through elements

  float     tol=RCONFIG->getSettingDouble("Simulation:BrokenContourTolerance");

  scdContX=fstContX=0.0;
  scdContY=fstContY=0.0;
  currentX=currentY=DEF_AREAMAX;
  fstContEl=0;
  
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE)) {
      
      switch(el->getElementTyp()) {
        
        // Rapid move without an element:
        //   
      case T_RAPID:
        scdContX=fstContX=el->getX1();
        scdContY=fstContY=el->getY1();
        currentX=el->getX1();
        currentY=el->getY1();
        fstContEl=el;
        break;
        
      case T_LINE:
      case T_ARC:
        
        // Start of a new contour:
        //
        if(!mtCompFloat(el->getX1(), currentX, tol) || 
           !mtCompFloat(el->getY1(), currentY, tol)    ) {
          fstContX=el->getX1();
          fstContY=el->getY1();
          scdContX=el->getX2();
          scdContY=el->getY2();
          fstContEl=el;
          if(prevEl) prevEl->setFlag(E_BROKEN2);
          el->setFlag(E_BROKEN1);
        }
        
        el->setFlag(E_BROKEN);
        
        // End of a closed contour:
        //
        if((mtCompFloat(el->getX2(), fstContX, tol) && 
            mtCompFloat(el->getY2(), fstContY, tol)    ) ||
           (mtCompFloat(el->getX2(), scdContX, tol) && 
            mtCompFloat(el->getY2(), scdContY, tol) &&
            fstContEl!=el                                )    ) {
          
          RElement* el2;
          
          bool stop=false;
          setCurrentElement(fstContEl);
          for(el2=element.current(); el2!=0 && !stop; el2=element.next()) {
            el2->delFlag(E_BROKEN|E_BROKEN1|E_BROKEN2);
            if(el2==el) stop=true;
          }
          fstContX=el->getX2();
          fstContY=el->getY2();
          fstContEl=el;
          
          setCurrentElement(el);
        }
        currentX=el->getX2();
        currentY=el->getY2();

        prevEl=el;
        break;
        
      default:
        break;
      }
    }
  }
  
  if(prevEl && prevEl->getFlag(E_BROKEN)) {
    prevEl->setFlag(E_BROKEN2);
  }
  
  updateBroken=false;
}



// Reset highlight-flag for elements in open 
//   (broken) contours:
//
void
RGraphic::resetHighlighting()
{
  RElement* el;                     // Pointer which walks through elements
  
  for(el=element.first(); el!=0; el=element.next()) {
    if(el->getFlag(E_VISIBLE)) {
      el->delFlag(E_BROKEN);
    }
  }
}



// 
// 
// Coordinate system:
//
//


// Real to screen translate:
//
int 
RGraphic::realToScreenX(float _realX) 
{ 
  return((int)(_realX*zoom+(float)offsetX)); 
}


int 
RGraphic::realToScreenY(float _realY) 
{ 
  return((int)(-_realY*zoom+(float)offsetY));
}


int
RGraphic::realToScreenDist(float _realDist)
{
  return (int)(_realDist*zoom);
}



// Real to screen translate (more exactly):
//
float
RGraphic::realToExactScreenX(float _realX) 
{ 
  return(_realX*zoom+(float)offsetX); 
}


float 
RGraphic::realToExactScreenY(float _realY) 
{ 
  return(-_realY*zoom+(float)offsetY);
}


float
RGraphic::realToExactScreenDist(float _realDist)
{
  return _realDist*zoom;
}



// Screen to real translation:
//
float
RGraphic::screenToRealX(int _screenX)
{
  if(zoom!=0) {
    return( (float)(_screenX-offsetX)/zoom);
  }
  else {
    return 0;
  }
}

float
RGraphic::screenToRealY(int _screenY)
{
  if(zoom!=0) {
    return(-(float)(_screenY-offsetY)/zoom);
  }
  else {
    return 0.0;
  }
}



// Transform a real distance into a screen distance:
//
float 
RGraphic::screenToRealDist(int _screenDist)
{
  return((float)(_screenDist)/zoom);
}



// Is point (_x/_y) on screen?
//
bool  
RGraphic::isPointOnScreen(int _x, int _y)
{
  if(_x>0 && _x<width() &&
     _y>0 && _y<height()   ) {
    return true;
  }
  else {
    return false;
  }
}



/**
 * Moves the relative zero point to a new
 *   location (deleting/drawing inclusive).
 */
void
RGraphic::moveRelZero(float _x, float _y)
{
  RPainter paint;        // painter

  paint.begin(this);
  //paint.setRasterOp(XorROP);
  paint.setClipRect(2, 2, width()-4, height()-4);

  drawRelativeZero(&paint);

  relZeroX=_x;
  relZeroY=_y;

  drawRelativeZero(&paint);

  paint.end();
}



//
//
//    Event handling:
//
//

// Get the REvent for events (_evm, _evk):
//
// return REVENT_LBUTTONDOWN, ...
//        0: no event
//
int
RGraphic::getREvent(QMouseEvent* _evm, 
                    QKeyEvent*   _evk )
{
  // Mouse event:
  //
  if(_evm) {
    switch(_evm->type()) {

      case QEvent::MouseMove:
        return REVENT_MOUSEMOVE;
        break;
    
      case QEvent::MouseButtonPress:
        switch(_evm->button()) {
          case LeftButton:
            return REVENT_LBUTTONDOWN;
            break;
          case RightButton:
            return REVENT_RBUTTONDOWN;
            break;
          default:
            break;
        }
        break;

      case QEvent::MouseButtonRelease:
        switch(_evm->button()) {
          case LeftButton:
            return REVENT_LBUTTONUP;
            break;
          case RightButton:
            return REVENT_RBUTTONUP;
            break;
          default:
            break;
        }
        break;

      default:
        break;
    }
  }

  // Key event:
  //
  if(_evk) {
    switch(_evk->type()) {

      case QEvent::KeyPress:
        return REVENT_KEYDOWN;
        break;

      case QEvent::KeyRelease:
        return REVENT_KEYUP;
        break;

      default:
        break;
    }
  }

  return 0;
}



// Mouse press Event:
//
void 
RGraphic::mousePressEvent(QMouseEvent* _ev)
{
  if(_ev->button()==LeftButton) {
    RAppWin::getRAppWin()->activateDocument(this);
  }
  distributeEvent(_ev, 0);
}



// Mouse release Event:
//
void 
RGraphic::mouseReleaseEvent(QMouseEvent* _ev)
{
  distributeEvent(_ev, 0);
}



// Mouse move Event:
//
void 
RGraphic::mouseMoveEvent(QMouseEvent* _ev)
{
  if(!eventsEnabled) return;
  
  mouseX = _ev->x();
  mouseY = _ev->y();
  distributeEvent(_ev, 0);
}



// Key press Event:
//
void 
RGraphic::keyPressEvent(QKeyEvent* _ev)
{
  distributeEvent(0, _ev);
}



// Key release Event:
//
void 
RGraphic::keyReleaseEvent(QKeyEvent* _ev)
{
  distributeEvent(0, _ev);
}



// Distribute a mouse / key event to action handlers 
//   (zoom, draw, ...)
//
// Mouse- OR Key-Event is NULL (both parameters are
//   token for easier distribution
//
void
RGraphic::distributeEvent(QMouseEvent* _evm,
                          QKeyEvent* _evk)
{
  distributeEvent(getREvent(_evm, _evk));
}



// Distribute an event to action handlers 
//   (zoom, draw, ...)
//
void
RGraphic::distributeEvent(int _rEvent)
{
  if(busy || !eventsEnabled) return;


  // A mouse leave event:
  //
  if(_rEvent==REVENT_MOUSELEAVE) {
    if(snap) snap->stop();
  }

  // Do graphics own behaviour:
  //
  doBehavior(_rEvent);

  // A Zoom Action:
  //
  if(currentAction>=ACT_ZOOM_FIRST &&
     currentAction<=ACT_ZOOM_LAST     ) {

    if(actZoom) {
      actZoom->serveEvent(currentAction, _rEvent);
    }
  }
  
  // A Draw Action:
  //
  else if(currentAction>=ACT_DRAW_FIRST &&
          currentAction<=ACT_DRAW_LAST     ) {

    if(actDraw) {
      actDraw->serveEvent(currentAction, _rEvent);
    }
  }

  // A Tag Action:
  //
  else if(currentAction>=ACT_TAG_FIRST &&
          currentAction<=ACT_TAG_LAST     ) {

    if(actTag) {
      actTag->serveEvent(currentAction, _rEvent);
    }
  }
  
  // An Edit Action:
  //
  else if(currentAction>=ACT_EDIT_FIRST &&
          currentAction<=ACT_EDIT_LAST     ) {

    if(actEdit) {
      actEdit->serveEvent(currentAction, _rEvent);
    }
  }

  // A CAM Action:
  //
#ifdef DEF_CAM_EXPERT
  else if(currentAction>=ACT_CAM_FIRST &&
          currentAction<=ACT_CAM_LAST     ) {

    if(actCam) {
      actCam->serveEvent(currentAction, _rEvent);
    }
  }
#endif

  // No action / go back to last menu:
  //
  if(currentAction==-1 && _rEvent==REVENT_RBUTTONUP) {
    //RAppWin::getRAppWin()->changeMenu(TOOLBAR_MAIN);
    
    RAppWin::getRAppWin()->backMenu();
  }
}



// Save current Action settings:
//
void
RGraphic::saveCurrentActionSettings()
{
  if(lastCursor) delete lastCursor;
  lastCursor = new QCursor(cursor());
  lastAction = currentAction;
  lastState = currentState;
  lastBehavior = behavior;
}



// Restore last Action settings:
//

void
RGraphic::restoreLastActionSettings()
{
  setCursor(*lastCursor);
  currentAction = lastAction;
  currentState = lastState;
  behavior = lastBehavior;
}



// Set a behavior for the graphic widget:
//
void 
RGraphic::setBehavior(int _bvr)
{
  doBehavior(REVENT_END);
  behavior=_bvr;
  doBehavior(REVENT_BEGIN);
}



// React to an event based on graphics behavior:
//
// _rEvent: event
//
void
RGraphic::doBehavior(int _rEvent)
{
  switch(behavior) {
    case BVR_FREEPOINT:
      doBehaviorFreePoint(_rEvent);
      break;

    case BVR_FREEPOINT_PREV_RECT:
      doBehaviorFreePointPrevRect(_rEvent);
      break;

    case BVR_FREEPOINT_PREV_RECT_CENTER:
      doBehaviorFreePointPrevRectCenter(_rEvent);
      break;

    case BVR_SNAP_NEXT_ELEMENT:
      doBehaviorSnapNextElement(_rEvent);
      break;

    default:
      break;
  }
}



// Do behavior for free point:
//
void
RGraphic::doBehaviorFreePoint(int _rEvent)
{
  RLOG("\ndo behavior: Free Point");
  
  switch(_rEvent) {
    case REVENT_BEGIN:
      break;

    case REVENT_END:
      behavior=BVR_NO;
      break;

    default:
      break;
  }
}



// Do behavior for free point with 
//   rectangle preview:
//
void
RGraphic::doBehaviorFreePointPrevRect(int _rEvent)
{
  RLOG("\ndo behavior: Free Point Prev Rect");
  
  static int px1,         // 1st position of rectangle
             py1,         //
             px2,         // 2nd
             py2;         //
  
  RPainter paint;        // painter
        
  paint.begin(this);
  paint.setPen(cyan);
  paint.setRasterOp(XorROP);
  paint.setClipRect(2, 2, width()-4, height()-4);
         
  switch(_rEvent) {
    case REVENT_BEGIN:
      px1=mouseX;
      py1=mouseY;
      prevVisible=false;
      break;

    case REVENT_MOUSEMOVE:
      if(currentState) {
        RLOG("\ndo behavior: Free Point Prev Rect: Mouse move");

        if(prevVisible) {
          paint.drawRect(px1, py1, 
                         px2-px1, py2-py1);
        }
        paint.drawRect(px1, py1, 
                       mouseX-px1, mouseY-py1);
        prevVisible=true;
        px2=mouseX;
        py2=mouseY;
      }
      break;
    
    case REVENT_MOUSELEAVE:
      if(prevVisible) {
        paint.drawRect(px1, py1, 
                       px2-px1, py2-py1);
        prevVisible=false;
      }
      break;
    
    case REVENT_MOUSEENTER:
      if(!prevVisible) {
        paint.drawRect(px1, py1, 
                       px2-px1, py2-py1);
       prevVisible=true;
      }
      break;

    case REVENT_END:
      prevVisible=false;
      behavior=BVR_NO;
      break;

    default:
      break;
  }
  paint.end();
}



// Do behavior for free point with 
//   preview element preview:
//
void
RGraphic::doBehaviorFreePointPrevRectCenter(int _rEvent)
{
  RLOG("\ndo behavior: Free Point Prev Rect Center");
  
  static int px,          // Position of rectangle (center)
             py,          //
             sx,          // Size of rectangle
             sy;          // 
         
  switch(_rEvent) {
    case REVENT_BEGIN:
      px=mouseX;
      py=mouseY;
      sx=width();
      sy=height();
      prevVisible=false;
      break;

    case REVENT_MOUSEMOVE:
      if(currentState) {
        RLOG("\ndo behavior: Free Point Prev Rect Center: Mouse move");

        RPainter paint;        // painter
        
        paint.begin(this);
        paint.setPen(cyan);
        paint.setRasterOp(XorROP);
        paint.setClipRect(2, 2, width()-4, height()-4);

        if(prevVisible) {
          paint.drawRect(px-sx/2, py-sy/2, 
                         sx, sy);
        }
        paint.drawRect(mouseX-sx/2, mouseY-sy/2,
                       sx, sy);
        prevVisible=true;
        px=mouseX;
        py=mouseY;
        paint.end();
      }
      break;

    case REVENT_END:
      if(prevVisible) {
        RPainter paint;        // painter
        paint.begin(this);
        paint.setPen(cyan);
        paint.setRasterOp(XorROP);
        paint.setClipRect(2, 2, width()-4, height()-4);
        paint.drawRect(px-sx/2, py-sy/2, 
                       sx, sy);
        paint.end();
      }
      prevVisible=false;
      behavior=BVR_NO;
      break;

    default:
      break;
  }
}


// Do behavior for snapping the next element
//   (preview element)
//
void
RGraphic::doBehaviorSnapNextElement(int _rEvent)
{
  RLOG("\ndo behavior: Snap Next Element");
  RLOG("\nsnapFilterTyp: ");
  RLOG(snapFilterTyp);

  static RElement oldEl;      // Last snapped element
  
  switch(_rEvent) {
    case REVENT_BEGIN:
      setHighlightFilter(T_ALL, 0);
      break;

    case REVENT_MOUSEMOVE:
      if(currentState) {

        snap->snapElement(false, snapFilterTyp, snapFilterException);
        if(!oldEl.compareWith(snap->getSnappedElement())) {
      
          bool doXorMark=false;

          if(markVisible && markEnabled) {
            doXorMark=true;
          }

          if(prevElementVisible) {
            delPreviewElement();
          }

          if(doXorMark) xorMark();

          //snap->snapElement(true, snapFilterTyp, snapFilterException);
          setHighlightElement(snap->getSnappedElement());
          drawHighlightElement();

          if(doXorMark) xorMark();

          if(!prevElementVisible) {
            drawPreviewElement();
          }

          oldEl.getMeasuresFrom(snap->getSnappedElement());
        }
      }
      behavior=BVR_NO;
      break;

    case REVENT_END:
      behavior=BVR_NO;
      break;

    default:
      break;
  }
}



// Call this function if Graphic has changed:
//
void  
RGraphic::graphicHasChanged()
{
  updateBroken=true;
  setFlag(G_CHANGED);
  if(autoUpdateInfo) emit graphicChanged();
}



// Output to debug file (layer list):
//
void
RGraphic::debugLayers()
{
  RLayer *lay;          // pointer which walks through layers

  RLOG("\nDebug Layers:");

  for(lay=layer.first(); lay!=0; lay=layer.next()) {
    if(lay->getFlag(Y_USED)) {
      RLOG("\n* ");
    }
    else {
      RLOG("\n  ");
    }
    RLOG(lay->getName());
    
  }
}



// Output to debug file (element list):
//
void
RGraphic::debugElements()
{
  RElement *el;          // pointer which walks through elements
  for(el=element.first(); el!=0; el=element.next()) {
    el->outputDebug();
  }
}


// Output to debug file (parameter / layer parameter):
//
void
RGraphic::debugParameters()
{
  int i, k;

  RLOG("\nparameterString: ");
  RLOG(parameterString);
  for(i=0; i<DEF_MAXPARAMETERS && i<(int)parameterString.length(); ++i) {
    RLOG("\nparameter[");
    RLOG(i);
    RLOG("]: ");
    RLOG(parameter[i].data());
  }

  RLOG("\nlayerParameterString: ");
  RLOG(layerParameterString);
  for(k=0; k<DEF_MAXLAYERS && k<countLayers(); ++k) {
    for(i=0; i<DEF_MAXPARAMETERS && i<(int)layerParameterString.length(); ++i) {
      RLOG("\nlayerParameter[");
      RLOG(k);
      RLOG("][");
      RLOG(i);
      RLOG("]: ");
      RLOG(layerParameter[k][i].data());
    }
  }
}


// EOF














































































