/*
 * Program to control ICOM radios
 *
 * Main program
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include "icom.h"

#ifndef MSDOS			/* include for Unix */
#include <sys/inttypes.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <termios.h>
#endif /* MSDOS */
#ifdef AUDIO			/* include for Sun audio */
#include <sys/audioio.h>
#endif /* AUDIO */

/*
 * Module definitions
 */
#define ARGMAX 20		/* maximum number of command args */

/*
 * External functions
 */
extern FILE *fopen();
extern char *strtok(), *strcpy();
extern char *optarg;
extern int optind, opterr;

/*
 * Local function prototypes
 */
static int scanarg(struct icom *, char *, struct cmdtable *, int);
static int getline(char *);
static int args(struct icom *, char *);
static int sw_keypad(void);
static int sw_keybd(void);
static void reset(struct icom *, struct cmdtable *);
static void update(struct icom *);
char *modetoa(int, struct cmdtable *);
double fabs(double);

/*
 * Local variables
 */
int flags;			/* program flags */
static char *argv[ARGMAX];	/* command line args */
static int argn;		/* number of command line args */
static FILE *fp_cmd;		/* command input file */
static FILE *fp_in;		/* data input file */
static FILE *fp_out;		/* data output file */
double logtab[] = {		/* tuning rate table */
	1., 2., 5.,		/* 0.000000 */
	10., 25., 50.,		/* 0.00000 */
	1e2, 2.5e2, 5e2,	/* 0.0000 */
	1e3, 2.5e3, 5e3,	/* 0.000 */
	1e4, 2.5e4, 5e4,	/* 0.00 */
	1e5, 2.5e5, 5e5,	/* 0.0 */
	1e6, 2.5e6, 5e6		/* 0. */
};
int sigtab[] = {		/* significant digits table */
	6, 6, 6,		/* 0.000000 */
	5, 6, 5,		/* 0.00000 */
	4, 5, 4,		/* 0.0000 */
	3, 4, 3,		/* 0.000 */
	2, 3, 2,		/* 0.00 */
	1, 2, 1,		/* 0.0 */
	0, 0, 0			/* 0. */
};

#ifndef MSDOS
static int fd;			/* terminal control file descriptor */
static struct termios terma, termb; /* Unix terminal interface */
extern int errno;
#endif /* MSDOS */
#ifdef AUDIO
static struct audio_device device; /* audio device ident */
static struct audio_info info;	/* audio device info */
static int ctl_fd;		/* audio control file descriptor */
#endif /* AUDIO */

/*
 * Main program
 */
main(argc, argcv)
	int argc;		/* number of command-line arguments */
	char **argcv;		/* vector of command-line argument
				   pointers */

{
	char s1[VALMAX], s2[VALMAX]; /* parameter strings */
	char chr, *ptr;		/* char temps */
	int i, temp, sw;	/* int temps */
	double freq, step, dtemp; /* double temps */
	int par1, par2	;	/* integer parameters */
	struct icom *radio;	/* radio structure pointer */
	struct icom *tradio;	/* temporary radio structure pointer */

	/*
	 * Initialize
	 */
	flags = 0;
	radio = NULL;
	init();
	flags = P_VERB;

#ifndef MSDOS
	/*
	 * Process command-line arguments
	 */
	while ((temp = getopt(argc, argcv, "r:ac:m:o:g:dkf:")) != -1) {
		switch (temp) {

		/*
		 * -a - open audio device
		 */
		case 'a':
#ifdef AUDIO
			if ((ctl_fd = open("/dev/audioctl", O_RDWR)) < 0)
				printf("*** audio not available\n");
			continue;
#else /* AUDIO */
			printf("*** audio not configured\n");
			continue;
#endif /* AUDIO */

		/*
		 * -f <file> - open command file
		 */
		case 'f':
			if ((fp_cmd = fopen(optarg, "r")) == NULL) {
				printf("*** file not found\n");
				flags |= P_EXIT;
			} else
				flags &= ~P_VERB;
			continue;

		/*
		 * -k - specify keypad mode
		 */
		case 'k':
			flags |= P_PAD;
			continue;

		/*
		 * -r <radio> - select radio
		 */
		case 'r':
			temp = argsx(optarg, ident);
			if (temp == C_ERROR)
				exit(1);
			radio = select_radio(temp);
			if (radio == NULL) {
				printf("*** radio not found\n",
				    temp);
				exit(1);
			}
			continue;
		}

		/*
		 * The remaining options are valid only after a valid
		 * radio has been selected. If any are present, the
		 * program exits after executing the command line
		 * options.
		 */
		if (radio == NULL) {
			printf("*** radio not specified\n");
			exit(1);
		}
		flags |= P_EXIT;
		switch (temp) {

		/*
		 * -c <chan> - set channel
		 */
		case 'c':
			if (sscanf(optarg, "%d", &temp) != 1)
				printf("*** bad channel format\n");
			else
				readchan(radio, temp);
			continue;

		/*
		 * -d - display current frequency and mode
		 */
		case 'd':
			flags |= P_DISP;
			continue;

		/*
		 * -g <frequency> - set frequency (MHz)
		 */
		case 'g':
			if (sscanf(optarg, "%lf", &freq) != 1)
				printf("*** bad frequency format\n");
			else {
				if (freq > 1000)
					freq /= 1000;
				loadfreq(radio, freq + radio->offset);
			}
			continue;

		/*
		 * -m <mode> - set mode
		 */
		case 'm':
			temp = argsx(optarg, radio->modetab);
			if (temp != C_ERROR)
				loadmode(radio, temp);
			continue;

		/*
		 * -o <offset> - set offset (kHz)
		 */
		case 'o':
			if (sscanf(optarg, "%lf", &radio->offset) != 1)
				printf("*** bad offset format\n");
			else
				radio->offset /= 1000;
			continue;
		}
	}

	/*
	 * If a radio was found, initialize it. If its settings were
	 * changed and a command file is not open, assume this is run
	 * from a script and nothing more needs to be done.
	 */
	if (radio != NULL)
		reset(radio, probe);
	if ((flags & P_EXIT) && (flags & P_VERB))
		exit(1);
	if (flags & P_PAD) {
		if (sw_keypad())
			flags &= ~P_PAD;
	}
#endif /* MSDOS */

	/*
	 * Main loop
	 */
	while (1) {
		flags &= ~(P_DISP | P_KEYP | P_ESC);
		if (flags & P_PAD) {

			/*
			 * Keypad mode. Collect digits, '+', '-', '.',
			 * until next different character, which
			 * determines command, or \n, which is mapped to
			 * 'f'. Help '?' displays list of command names
			 * and descriptions.
			 */
			printf(">");
			ptr = s1;
			for (i = 0; i < 6; i++)
				*ptr++ = ' ';
			while (1) {
				chr = (char)getchar();
				if (chr == '\033') {
					flags |= P_ESC;
					*ptr = '\0';
					ptr = s1;
					continue;
				}
				if (flags & P_ESC) {
					*ptr++ = chr;
					if (isalpha((int)chr) || chr ==
					    '\n')
						break;
					continue;
				}
				if (isdigit((int)chr) || chr == '.'
				    || chr == '+' || chr == '-') {
					*ptr++ = chr;
 					putchar(chr);
					flags |= P_KEYP;
					continue;
				} else if (chr != '\n') {
					flags |= P_KEYP;
					putchar(chr);
				}
				if (flags & P_KEYP)
					putchar('\n');
				*ptr = '\0';
				*s1 = chr;
				break;
			}
			if ((argn = getline(s1)) == 0)
				sw = C_FREQ;
			else
				sw = argsx(argv[0], key);
		} else {

			/*
			 * Keyboard mode. Get next command, convert to
			 * lower case and parse tokens. Select command
			 * based on first token. Ignore '#' and rest of
			 * line. This is for command scripts.
			 */
			if (fp_cmd != NULL) {
				if (fgets(s1, VALMAX, fp_cmd) == NULL)
					exit(0);
				printf("%s", s1);
			} else {
				printf("icom>");
				if (gets(s1) == NULL)
					exit(0);
			}
			if (*s1 == '#')
				continue;
			if ((argn = getline(s1)) == 0)
				sw = C_FREQ;
			else
				sw = argsx(argv[0], cmd);
		}
		switch (sw) {

		/*
		 * radio [ <name> ]
		 *
		 * Select the <name> radio for further commands and
		 * display its description and band limits. If name is
		 * missing and the radio has not been previously
		 * defined, the buss is probed for all known radios,
		 * which takes some time. If previouslh defined, its
		 * description and band limits of the are displayed.
		 */
		case C_RADIO:
			if (argn < 2) {
				if (radio != NULL) {
					printf("radio %s\n",
					    getcap("radio",
					    radio->cap));
					continue;
				}
				for (i = 0; name[i].name[0] != '\0';
				    i++) {
					tradio =
					    select_radio(name[i].ident);
					if (tradio != NULL) {
						radio = tradio;
						reset(radio, probe);
						printf("radio %s\n",
						    getcap("radio",
						    radio->cap));
					}
				}
				continue;
			}
			temp = argsx(argv[1], ident);
			if (temp == C_ERROR)
				continue;
			radio = select_radio(temp);
			if (radio == NULL) {
				printf("*** radio not found\n");
				continue;
			}
			reset(radio, probe);
			break;
 
		/*
		 * quit
		 *
		 * Quit the dance
		 */
		case C_QUIT:
			exit(0);
 
		/*
		 * verbose off | on
		 *
		 * Set verbose mode
		 */
		case C_VERB:
			if (argn < 2) {
				printf("*** missing argument\n");
				continue;
			}
			temp = argsx(argv[1], verbx);
			if (temp != C_ERROR)
				flags = (flags & ~P_VERB) | temp;
			continue;

		/*
		 * trace [ all | none | bus | pkt ]
		 *
		 * Set debug flags
		 */
		case C_DEBUG:
			if (argn < 2) {
				printf("*** missing argument\n");
				continue;
			}
			temp = argsx(argv[1], dbx);
			if (temp != C_ERROR)
				flags = (flags & ~(P_TRACE | P_ERMSG)) |
				    temp;
			continue;

		/*
		 * pad
		 *
		 * Switch to keypad mode.
		 */
		case C_KEYPAD:
			if (!sw_keypad())
				flags |= P_PAD;
			continue;

		/*
		 * # (keypad mode)
		 *
		 * Erase input
		 */
		case C_ERROR:
		case C_ERASE:
			continue;

		/*
		 * q (keypad mode)
		 *
		 * Switch to keyboard mode.
		 */
		case C_KEYBD:
			if (!sw_keybd())
				flags &= ~P_PAD;
			continue;
#ifdef AUDIO
		/*
		 * gain [ <gain> ]
		 *
		 * Adjust record gain in 16 levels.
		 */
		case C_GAIN:
			if (ctl_fd <= 0)
				printf("*** audio not available\n");
			else if (argn < 2)
				printf("audio gain %d\n",
				    info.record.gain);
			else if (sscanf(argv[1], "%d", &temp) != 1)
				printf("*** bad gain format\n");
			else {
				if (temp > 0)
					temp = (temp << 4) - 1;
				if (temp > AUDIO_MAX_GAIN)
					temp = AUDIO_MAX_GAIN;
				else if (temp < AUDIO_MIN_GAIN)
					temp = AUDIO_MIN_GAIN;
				ioctl(ctl_fd, (int)AUDIO_GETINFO,
				    &info);
				info.record.gain = temp;
				ioctl(ctl_fd, (int)AUDIO_SETINFO,
				    &info);
			}
			continue;

		/*
		 * mute
		 *
		 * Mute output (toggle)
		 */
		case C_MUTE:
			if (ctl_fd <= 0) {
				printf("*** audio not available\n");
				continue;
			}
			ioctl(ctl_fd, (int)AUDIO_GETINFO, &info);
			if (info.output_muted) {
				info.output_muted = 0;
			} else {
				info.output_muted = 1;
				printf("audio muted\n");
			}
			ioctl(ctl_fd, (int)AUDIO_SETINFO, &info);
			continue;

		/*
		 * port <port>
		 *
		 * Select input port (1 = mike, 2 = line in)
		 */
		case C_PORT:
			if (ctl_fd <= 0)
				printf("*** audio not available\n");
			else if (argn < 2)
				printf("audio port %d\n",
				    info.record.port);
			else if (sscanf(argv[1], "%d", &temp) != 1)
				printf("*** invalid port format\n");
			else if (temp < 1 || temp > 2)
				printf("*** invalid port number\n");
			else {
				ioctl(ctl_fd, (int)AUDIO_GETINFO,
				    &info);
				info.record.port = temp;
				ioctl(ctl_fd, (int)AUDIO_SETINFO,
				    &info);
			}
			continue;
#endif /* AUDIO */
		}

		/*
		 * The remaining commands are valid only after a radio
		 * has been selected.
		 */
		if (radio == NULL) {
			printf("*** radio not specified\n");
			continue;
		}
		switch (sw) {

		/*
		 * cap [ <args> ]
		 */
		case C_PROBE:
			if (argn < 2) {
				(void)getcap("?", radio->cap);
				break;
			}
			printf("%s %s\n", argv[1], getcap(argv[1],
			    radio->cap));
			break;

		/*
		 * restore file [ <first> ] [ <last> ]
		 *
		 * Restore memory channels from file to the range of
		 * memory channels numbered first through last in
		 * sequence. If last is missing, the range is from first
		 * to max. If first and last are missing, the range is
		 * from 1 to max. The sequence is scanned always from
		 * lowest to highest, no matter which order is given.
		 */
		case C_RESTORE:
			if (argn < 2) {
				printf(
				"usage: restore file [first] [last]\n");
				break;
			}
			if ((fp_in = fopen(argv[1], "r")) == NULL) {
				printf("*** file not found\n");
				break;
			}
			par1 = 1;
			par2 = radio->maxch;
			if (argn > 2)
				par1 = args(radio, argv[2]);
			if (argn > 3)
				par2 = args(radio, argv[3]);
			if (par2 < par1) {
				temp = par1;
				par1 = par2;
				par2 = temp;
			}
			for (i = par1; i <= par2;) {
				if (fgets(s1, VALMAX, fp_in) == NULL)
					break;
				printf("chan %3i %s", i, s1);
				if ((argn = getline(s1)) == 0)
					continue;
				if (setchan(radio, i))
					break;
				if (sscanf(argv[0], "%lf", &freq) != 1)
				    {
					printf("*** bad frequency format %s\n",
					    argv[0]);
					break;
				}
				if (freq > 1000)
					freq /= 1000;
				if (loadfreq(radio, freq))
					break;
				if (argn > 1) {
					temp = argsx(argv[1],
					    radio->modetab);
					if (temp == C_ERROR)
						break;
					if (loadmode(radio, temp))
						break;
				}
				if (argn > 2 && radio->flags & F_OFFSET)
				    {
					if (sscanf(argv[2], "%lf",
					    &step) != 1) {
						printf(
					"*** bad offsetformat %s\n",
						    argv[0]);
						break;
					}
					if (loadoffset(radio, step))
						break;
				}
				if (setcmd(radio, V_WRITE, FI)) {
					printf(
					"*** invalid channel write\n");
					break;
				}
				i++;
			}
			close(fp_in);
			break;

		/*
		 * save file [ <first> ] [ <last> ]
		 *
		 * Save memory channels in file from the range of memory
		 * channels numbered first through last in sequence. If
		 * last is missing, the range is from first to max. If
		 * first and last are missing, the range is from 1 to
		 * max. The sequence is scanned always from lowest to
		 * highest, no matter which order is given.
		 */
		case C_SAVE:
			if (argn < 2) {
				printf(
				"usage: save file [first] [last]\n");
				break;
			}
			if ((fp_out = fopen(argv[1], "w")) == NULL) {
				printf(
				"*** output file error\n");
				break;
			}
			par1 = 1;
			par2 = radio->maxch;
			if (argn > 2)
				par1 = args(radio, argv[2]);
			if (argn > 3)
				par2 = args(radio, argv[3]);
			if (par2 < par1) {
				temp = par1;
				par1 = par2;
				par2 = temp;
			}
			for (i = par1; i <= par2; i++) {
				if (readchan(radio, i))
					break;
				if (radio->mode == 0xff)
					continue;
				if (radio->flags & F_OFFSET)
 					fprintf(fp_out,
					    "%.*lf %s %+.0lf\n",
					    sigtab[radio->rate],
					    radio->vfo,
					    modetoa(radio->mode,
					    radio->modetab),
					    radio->duplex);
				else
					fprintf(fp_out, "%.*lf %s\n",
					    sigtab[radio->rate],
					    radio->vfo,
					    modetoa(radio->mode,
					    radio->modetab));
			}
			fclose(fp_out);
			break;

		/*
		 * write [ <chan> ]
		 *
		 * Write the frequency, mode and (optional) duplex
		 * offset memory channel chan. If chan is missing, use
		 * the current channel.
		 */
		case C_WRITE:
			if (argn > 1) {
				if (setchan(radio, args(radio,
				    argv[1])))
					break;
			}
			if (setcmd(radio, V_WRITE, FI))
				printf("*** invalid write channel\n");
			break;

		/*
		 * clear [ <chan> ]
		 *
		 * Clear the memory channel. If chan is missing, use the
		 * current channel.
		 */
		case C_CLEAR:
			if (argn > 1) {
				if (setchan(radio, args(radio,
				    argv[1])))
					break;
			}
			if ((radio->flags & F_VFO)) {
				if (setcmd(radio, V_SMEM, FI))
				printf("*** invalid clear channel\n");
				else if (setcmd(radio, V_CLEAR, FI))
				printf("*** V_CLEAR protocol error\n");
				else if (setcmd(radio, V_SVFO, FI))
				printf("*** V_SVFO protocol error\n");
				else
					radio->mode = 0xff;
			} else {
				if (setcmd(radio, V_CLEAR, FI))
				printf("*** invalid clear channel\n");
				else
					radio->mode = 0xff;
			}
			break;

		/*
		 * chan [ <channel> ]
		 *
		 * Display memory channel.
		 */
		case C_CHAN:
			if (argn < 2) {
				flags |= P_DISP | P_DSPCH;
				break;
			} else if (strcasecmp(argv[1], "+") == 0)
				temp = radio->mchan + 1;
			else if (strcasecmp(argv[1], "-") == 0)
				temp = radio->mchan - 1;
			else if (sscanf(argv[1], "%d", &temp) != 1) {
				printf("*** bad channel format\n");
				break;
			}
			if (readchan(radio, temp))
				break;
			if (flags & P_VERB)
				flags |= P_DISP | P_DSPCH;
			break;

		/*
		 * bank [ <channel> ]
		 *
		 * Set memory bank (R8500)
		 */
		case C_BANK:
			if (argn > 1) {
				temp = argsx(argv[1], bank);
				if (temp == C_ERROR)
					break;
			}
			setbank(radio, temp);
			break;

		/*
		 * freq [ <frequency> ] [ <mode> ]
		 *
		 * Set current operating frequency (kHz or MHz) and
		 * optional mode. The default is to display the current
		 * frequency and mode. Note that the mode has to be set
		 * first, since some radios shift only the BFO and don't
		 * compensate the local oscillator. Watch out for
		 * ham/general-coverage switch; some radios won't let
		 * you set the frequency outside the ham bands if the
		 * switch is in the ham position. This holds also when
		 * reading a memory channel, which can be stored either
		 * way. 
		 */
		case C_FREQ:
			if (argn < 2) {
				flags |= P_DISP;
				break;
			}
			if (sscanf(argv[1], "%lf", &freq) != 1) {
				printf("*** bad frequency format\n");
				break;
			}
			if (freq == 0 || *argv[1] == '+' || *argv[1] ==
			    '-')
				freq = radio->vfo + freq / 1000;
			else if (freq > 1000)
				freq /= 1000;
			if (loadfreq(radio, freq + radio->offset))
				break;
			if (argn > 2) {
				temp = argsx(argv[2], radio->modetab);
				if (temp != C_ERROR)
					break;
				if (loadmode(radio, temp))
					break;
			}
			if (flags & P_VERB)
				flags |= P_DISP;
			break;

		/*
		 * (command not found)
		 *
		 * If the command consists of a single + or - character,
		 * change the channel up or down one. If it has valid
		 * floating point format, set the frequency as given. If
		 * so and there is an additional argument, set the mode
		 * as given.
		 */
		case C_FREQX:
			if (strcasecmp(argv[0], "+") == 0) {
				if (readchan(radio, radio->mchan + 1))
					break;
				if (flags & P_VERB)
					flags |= P_DISP | P_DSPCH;
				break;
			} else if (strcasecmp(argv[0], "-") == 0) {
				if (readchan(radio, radio->mchan - 1))
					break;
				if (flags & P_VERB)
					flags |= P_DISP | P_DSPCH;
				break;
			}
			if (sscanf(argv[0], "%lf", &freq) != 1) {
				printf("*** unknown command %s\n",
				    argv[0]);
				break;
			}
			if (freq > 1000)
				freq /= 1000;
			if (freq == 0 || *argv[0] == '+' || *argv[0] ==
			    '-')
				freq = radio->vfo + freq / 1000;
			else if (freq > 1000)
				freq /= 1000;
			if (loadfreq(radio, freq + radio->offset))
				break;
			if (argn > 1) {
				temp = argsx(argv[1], radio->modetab);
				if (temp != C_ERROR)
					break;
				if (loadmode(radio, temp))
					break;
			}
			if (flags & P_VERB)
				flags |= P_DISP;
			break;

		/*
		 * offset [ <offset> ]
		 *
		 * Set VFO frequency offset.
		 */
		case C_OFFSET:
			if (argn < 2) {
				printf("VFO offset %.0lf Hz\n",
				    radio->offset);
				break;
			}
			if (sscanf(argv[1], "%lf", &freq) != 1) {
				printf("*** bad offset format\n");
				break;
			}
			radio->offset = freq;
			break;

		/*
		 * comp [ <offset> ]
		 *
		 * Compensate VFO frequency offset.
		 */
		case C_COMP:
			if (argn < 2) {
				printf("VFO compensation %.3lf PPM\n",
				    radio->vfo_comp * 1e6);
				break;
			}
			if (sscanf(argv[1], "%lf", &freq) != 1) {
				printf("*** bad compensation format\n");
				break;
			}
			radio->vfo_comp = freq / 1e6;
			break;

		/*
		 * band [ <low> ] [ <high> ]
		 *
		 * Set band limits.
		 */
		case C_BAND:
			if (argn < 2) {
				printf("band %s\n", getcap("band",
				    radio->cap));
				break;
			}
			if (argn < 3) {
				printf("*** two arguments required\n");
				break;
			}
			if (sscanf(argv[1], "%lf", &freq) != 1) {
				printf("*** bad format arg 1\n");
				break;
			}
			if (sscanf(argv[2], "%lf", &step) != 1) {
				printf("*** bad format arg 2\n");
				break;
			}
			if (freq > step) {
				dtemp = freq;
				freq = step;
				step = dtemp;
			}
			if (freq < radio->lband)
				freq = radio->lband;
			radio->lstep = freq;
			if (step > radio->uband)
				step = radio->uband;
			radio->ustep = step;
			break;

		/*
		 * mode [ <mode> ] [ offset ]
		 *
		 * Set current operating mode and BFO offset.
		 */
		case C_MODE:
			if (argn < 2) {
				freq = radio->bfo[radio->mode & 0x7];
				printf("mode %s BFO offset %.0lf Hz\n",
				    modetoa(radio->mode,
				    radio->modetab), freq);
				break;
			}
			temp = argsx(argv[1], radio->modetab);
			if (temp == C_ERROR)
				break;
			if (loadmode(radio, temp))
				break;
			if (argn > 2) {
				if (sscanf(argv[2], "%lf", &freq) != 1)
				    {
					printf(
					    "*** bad offset format\n");
					break;
				}
				radio->bfo[radio->mode & 0x7] = freq;
			}
			break;

		/*
		 * LSB | USB | CW | AM | RTTY | FM
		 *
		 * Set the operating mode.
		 */
		case C_MODEG:
			temp = argsx(argv[0], radio->modetab);
			if (temp == C_ERROR)
				break;
			if (loadmode(radio, temp))
				break;
			if (flags & P_VERB)
				flags |= P_DISP;
			break;

		/*
		 * The following commands require the transmit split
		 * feature in some HF transceivers.
		 *
		 * xsplit [ <split> ]
		 *
		 * Set the transmit frequency relative to the receive
		 * frequency and turn the split on. If no arguments,
		 * just turn ths split on and off.
		 */
		case C_XSPLT:
			if (argn < 2 && radio->sub != 0) {
				if (radio->flags & F_SPLIT) {
					if (setcmd(radio, V_SPLIT,
					    S_OFF))
						printf(
					"*** radio %s can't do that\n",
						    radio->name);
					else
						radio->flags &=
						    ~F_SPLIT;
				} else {
					if (setcmd(radio, V_SPLIT,
					    S_ON))
						printf(
					"*** radio %s can't do that\n",
						    radio->name);
					else
						radio->flags |= F_SPLIT;
				}
				flags |= P_DISP;
				break;
			}
			if (setcmd(radio, V_SPLIT, S_ON)) {
				printf("*** radio %s can't do that\n",
				    radio->name);
				break;
			}
			if (argn < 2) {
				step = 0;
			} else {
				if (sscanf(argv[1], "%lf", &step) != 1)
				    {
					printf("*** bad format\n");
					break;
				}
			}
			if (step == 0 || *argv[1] == '+' || *argv[1] ==
			    '-')
				step = radio->vfo + step / 1000;
			else if (step > 1000)
				step /= 1000;
			freq = radio->vfo;
			if (setcmd(radio, V_SVFO, S_EQUAL)) {
				printf("*** V_SVFO protocol error\n");
				break;
			}
			if (loadfreq(radio, step))
				break;
			if (setcmd(radio, V_SVFO, S_XCHNG)) {
				printf("*** V_XCHNG protocol error\n");
				break;
			}
			radio->vfo = freq;
			radio->sub = step;
			radio->flags |= F_SPLIT;
			if (flags & P_VERB)
				flags |= P_DISP;
			break;

		/*
		 * rsplit [ <split> ]
		 *
		 * Set the receive receive frequency relative to the
		 * transmit frequency. The split must be on.
		 */
		case C_RSPLT:
			if (!(radio->flags & F_SPLIT)) {
				printf(
				    "*** transmit split must be on\n");
				break;
			}
			if (argn < 2) {
				step = 0;
			} else {
				if (sscanf(argv[1], "%lf", &step) != 1)
				    {
					printf("*** bad format\n");
					break;
				}
			}
			if (step == 0 || *argv[1] == '+' || *argv[1] ==
			    '-')
				step = radio->sub + step / 1000;
			else if (step > 1000)
				step /= 1000;
			if (loadfreq(radio, step))
				break;
			if (flags & P_VERB)
				flags |= P_DISP;
			break;

		/*
		 * reverse (toggle)
		 *
		 * Swap the transmit and receive frequencies. The split
		 * must be on.
		 */
		case C_REVER:
			if (!(radio->flags & F_SPLIT)) {
				printf(
				    "*** transmit split must be on\n");
				break;
			}
			if (setcmd(radio, V_SVFO, S_XCHNG)) {
				printf("*** radio %s can't do that\n",
				    radio->name);
				break;
			}
			freq = radio->vfo;
			radio->vfo = radio->sub;
			radio->sub = freq;
			if (flags & P_VERB)
				flags |= P_DISP;
			break;

		/*
		 * The following commands require the transmit duplex
		 * offset feature in most VHF/UHV transceivers.
		 *
		 * duplex [ <duplex> ]
		 *
		 * Set transmit offset for FM duplex in kHz.
		 */
		case C_DUPLEX:
			if (!(radio->flags & F_OFFSET)) {
				printf("*** radio %s can't do that\n",
				    radio->name);
				break;
			}
			if (argn < 2) {
				printf("duplex %+.0lf kHz\n",
				    radio->duplex);
				break;
			}
			radio->flags &= ~F_SMPLX;
			if (sscanf(argv[1], "%lf", &step) != 1) {
				printf("*** bad format\n");
				break;
			}
			if (loadoffset(radio, step))
				break;
			if (flags & P_VERB)
				flags |= P_DISP;
			break;

		/*
		 * simplex (toggle)
		 *
		 * Receive on transmit frequency.
		 */
		case C_SMPLX:
			if (!(radio->flags & F_OFFSET)) {
				printf("*** radio %s can't do that\n",
				    radio->name);
				break;
			}
			if (radio->flags & F_SMPLX) {
				step = (radio->vfo - radio->oldplex) *
				    1000;
				if (loadoffset(radio, step))
					break;
				if (loadfreq(radio, radio->oldplex))
					break;
				radio->flags &= ~F_SMPLX;
			} else {
				freq = radio->vfo + radio->duplex /
				    1000;
				radio->oldplex = radio->vfo;
				if (loadoffset(radio, 0))
					break;
				if (loadfreq(radio, freq))
					break;
				radio->flags |= F_SMPLX;
			}
			if (flags & P_VERB)
				flags |= P_DISP;
			break;

		/*
		 * The following commands should work in all receivers
		 * and transceivers.
		 *
		 * Tuning rate commands. These adjust the tuning steps
		 * in 1-2.5-5 sequence from the minimum step usable with
		 * the radio to 5 MHz per step(!). Each time the tuning
		 * step is changed, the vfo frequency is aligned to the
		 * current least significant digit padded by zeros.
		 *
		 * rate [ <rate> ]
		 *
		 * Set tuning rate. The values of <rate> from 0 through
		 * 20 select the rate values in a 1-2.5-5-10 sequence.
		 */
		case C_RATE:
			if (argn < 2) {
				printf("rate %d step %.0lf Hz\n",
				    radio->rate, radio->step);
				break;
			} else if (sscanf(argv[1], "%d", &temp) != 1) {
				printf("*** bad rate format\n");
				break;
			}
			if (temp > 20)
				temp = 20;
			else if (temp < radio->minstep)
				temp = radio->minstep;
			radio->rate = temp;
			radio->step = logtab[radio->rate];
			break;

		/*
		 * rate up (keypad)
		 *
		 * Set tuning rate up one notch.
		 */
		case C_RUP:
			if (radio->rate < 20)
				radio->rate++;
			radio->step = logtab[radio->rate];
			step = modf(radio->vfo / (radio->step * 1e-6),
			    &freq);
			freq *= radio->step * 1e-6;
			if (!loadfreq(radio, freq))
				printf("%.0lf Hz\n", radio->step);
			break;

		/*
		 * rate down (keypad)
		 *
		 * Set tuning rate down one notch.
		 */
		case C_RDOWN:
			if (radio->rate > radio->minstep)
				radio->rate--;
			radio->step = logtab[radio->rate];
			step = modf(radio->vfo / (radio->step * 1e-6),
			    &freq);
			freq *= radio->step * 1e-6;
			if (!loadfreq(radio, freq))
				printf("%.0lf Hz\n", radio->step);
			break;

		/*
		 * Tuning step commands. The step command sets the
		 * tuning step directly to an arbitrary value. The up
		 * and down commands shift the frequency up or down by
		 * the value of the tuning step.
		 *
		 * step [ <step> ]
		 *
		 * Set tuning step directly in Hz. This is useful when
		 * scanning odd channel spacings, such as aviation and
		 * marine radio channels. Note that the tuning rate is
		 * set to minimum here, since otherwise the rounding
		 * process would violate the principle of least
		 * astonishment.
		 */
		case C_STEP:
			if (argn < 2) {
				printf("rate %d step %.0lf Hz\n",
				    radio->rate, radio->step);
				break;
			} else {
				if (sscanf(argv[1], "%lf", &step) != 1)
				    {
					printf("*** bad step format\n");
					break;
				}
			}
			radio->rate = radio->minstep;
			if (step < logtab[radio->minstep])
				step = logtab[radio->minstep];
			radio->step = step;
			break;

		/*
		 * up (keypad)
		 *
		 * Tune up one step.
		 */
		case C_UP:
			freq = radio->vfo + radio->step / 1e6;
			if (freq > radio->ustep + radio->step / 2e6)
				freq = radio->lstep;
			if (!loadfreq(radio, freq) && flags & P_VERB)
				flags |= P_DISP;
			break;

		/*
		 * down (keypad)
		 *
		 * Tune down one step.
		 */
		case C_DOWN:
			freq = radio->vfo - radio->step / 1e6;
			if (freq < radio->lstep - radio->step / 2e6)
				freq = radio->ustep;
			if (!loadfreq(radio, freq) && flags & P_VERB)
				flags |= P_DISP;
			break;

		/*
		 * Following are specialty commands implemented only in
		 * a few radios.
		 *
		 * vfo [ <command> ]
		 *
		 * Set vfo (V_VFO) subcommands.
		 */
		case C_VFO:
			if (argn < 2)
				printf("vfo %s\n",
				    getcap("vfo", radio->cap));
			else if ((temp = scanarg(radio, argv[1], vfo,
			    V_SVFO)) != C_ERROR) {
				setcap("vfo", radio->cap, argv[1]);
				switch (temp) {

				case S_XCHNG:
					freq = radio->sub;
					radio->sub = radio->vfo;
					radio->vfo = freq;
					break;

				case S_EQUAL:
					radio->sub = radio->vfo;
					break;
				}
			}
			break;

		/*
		 * atten [ 0 | 10 | 20 | 30 ]
		 *
		 * Select attenuator (V_ATTEN) subcommands.
		 */
		case C_ATTEN:
			if (argn < 2)
				printf("atten %s dB\n",
				    getcap("atten", radio->cap));
			else if (scanarg(radio, argv[1], atten,
			    V_ATTEN) != C_ERROR)
				setcap("atten", radio->cap, argv[1]);
			break;

		/*
		 * tune [ <step> ]
		 *
		 * Set dial tuning step.
		 */
		case C_TUNE:
			if (argn < 2) {
				printf("tune %s kHz\n",
				    getcap("tune", radio->cap));
			} else if (sscanf(argv[1], "%lf", &step) != 1) {
				printf(
				   "*** bad dial tuning step format\n");
			} else if (!loadtune(radio, step))
				setcap("tune", radio->cap, argv[1]);
			break;

		/*
		 * ant [ 1 | 2 ]
		 *
		 * Select antenna (V_SANT) subcommands.
		 */
		case C_ANT:
			if (argn < 2)
				printf("ant %s\n",
				    getcap("ant", radio->cap));
			else if (scanarg(radio, argv[1], ant, V_SANT)
			    != C_ERROR)
				setcap("ant", radio->cap, argv[1]);
			break;

		/*
		 * scan [ <command> ]
		 *
		 * Scan control (V_SCAN) subcommands.
		 */
		case C_SCAN:
			if (argn < 2) {
				printf("scan %s\n",
				    getcap("scan", radio->cap));
			} else if (scanarg(radio, argv[1], scan, V_SCAN)
			    != C_ERROR) {
				setcap("scan", radio->cap, argv[1]);
				if (strcasecmp(argv[1], "stop") == 0) {
					readfreq(radio);
					flags |= P_DISP;
				}
			}
			break;

		/*
		 * say [ <command> ]
		 *
		 * Announce control (V_ANNC) subcommands.
		 */
		case C_ANNC:
			if (argn < 2)
				printf("say %s\n", getcap("say",
				    radio->cap));
			else if (scanarg(radio, argv[1], annc, V_ANNC)
			    != C_ERROR)
				setcap("say", radio->cap, argv[1]);
			break;

		/*
		 * meter [ <command> ]
		 *
		 * Read squelch/signal strength (S_SQSG) subcommands
		 */
		case C_SIGNAL:
			if (argn < 2)
				printf("meter %s\n", getcap("meter",
				   radio->cap));
			else if (scanarg(radio, argv[1], meter, V_ANNC)
			    != C_ERROR)
				setcap("meter", radio->cap, argv[1]);
			break;

		/*
		 * misc [ <command> ]
		 *
		 * Miscellaneous control (S_CTRL) subcommands
		 */
		case C_MISC:
			if (argn < 2)
				printf("misc %s\n", getcap("misc",
				    radio->cap));
			else if (scanarg(radio, argv[1], misc, V_CTRL)
			    != C_ERROR)
				setcap("misc", radio->cap, argv[1]);
			break;

		/*
		 * key <string>
		 *
		 * Transmit ASCII string as CW.
		 */
		case C_KEY:
			if (argn < 2)
				printf("*** missing argument\n");
			else if (argn < 3) {
				sendcw(radio, getcap(argv[1],
				    radio->cap));
			} else
				setcap(argv[1], radio->cap, argv[2]);
			break;
		}
		update(radio);
	}
}

/*
 * reset(radio, table) - initialize radio
 */
static void
reset(radio, table)
	struct icom *radio;	/* radio structure */
	struct cmdtable *table;	/* capability table */
{
	int i;
	char s1[VALMAX];

	/*
	 * Initialize radio capabilities
	 */
	sprintf(s1,
	    "%s (%02x), %.2f-%.2f MHz, %d channels, step %.0lf Hz",
	    getcap(radio->name, ident), radio->ident, radio->lband,
	    radio->uband, radio->maxch, radio->step);
	setcap("radio", radio->cap, s1);
	update(radio);
	for (i = 0; table[i].name[0] != '\0'; i++) {
		if (!setcmd(radio, table[i].ident & 0xff,
			(table[i].ident >> 8) & 0xff))
			setcap(table[i].name, radio->cap,
			    table[i].descr);
	}
}

/*
 * update(radio) - update capabilities
 */
static void
update(radio)
	struct icom *radio;	/* radio structure */
{
	char s1[VALMAX];
	int temp;

	/*
	 * Update chan, freq, mode, duplex, split capabilities.
	 */
	temp = sigtab[radio->rate];
	sprintf(s1, "%d, %.*lf MHz %s", radio->mchan, temp, radio->vfo,
	    modetoa(radio->mode, radio->modetab));
	setcap("chan", radio->cap, s1);
	sprintf(s1,"%.*lf-%.*lf MHz, step %.0lf Hz", temp, radio->lstep,
	    temp, radio->ustep, radio->step);
	setcap("band", radio->cap, s1);
	if (radio->flags & F_SPLIT) {
 		sprintf(s1, "%.*lf MHz", temp, radio->sub);
		setcap("split", radio->cap, s1);
	}
	if (radio->flags & F_OFFSET) {
		sprintf(s1, "%+.0lf kHz", radio->duplex);
		setcap("duplex", radio->cap, s1);
	}
	if (!(flags & P_DISP))
		return;
	if (flags & P_DSPCH)
		printf("chan %d ", radio->mchan);
	printf("%.*lf %s", temp, radio->vfo, modetoa(radio->mode,
	    radio->modetab));
	if (radio->flags & F_SPLIT)
		printf(" %.*lf", temp, radio->sub);
	if (radio->duplex != 0)
		printf(" %+.0lf", radio->duplex);
	if (radio->flags & F_SMPLX)
		printf(" S");
	printf("\n");
	flags &= ~(P_DISP | P_DSPCH); 
}

/*
 * modetoa(ident, table) - returns capability name
 */
char *
modetoa(ident, table)
	int ident;		/* capability key */
	struct cmdtable *table; /* capability table */
{
	int i;

	for (i = 0; table[i].name[0] != '\0'; i++) {
		if (table[i].ident == ident)
			return(table[i].name);
	}
	return("");
}

/*
 * argsx(name, table) - returns capability key
 */
int
argsx(name, table)
	char *name;		/* capability name */
	struct cmdtable *table; /* capability table */
{
	int i, temp;

	if (*name == '?') {
		for (i = 0; table[i].name[0] != '\0'; i++)
			printf("%10s %s\n", table[i].name,
			    table[i].descr);
		return(C_ERROR);
	}
	for (i = 0; table[i].name[0] != '\0'; i++) {
		if (strcasecmp(name, table[i].name) == 0)
			break;
	}
	if (table[i].ident == C_ERROR)
		printf("*** %s\n", table[i].descr);
	return(table[i].ident);
}

/*
 * getcap(name, table) - return pointer to capability string
 */
char *
getcap(name, table)
	char *name;		/* capability name */
	struct cmdtable *table; /* capability table */
{
	int i;

	if (*name == '?') {
		for (i = 0; table[i].name[0] != '\0'; i++)
			printf("%10s %s\n", table[i].name,
			    table[i].descr);
		return("\0");
	}
	for (i = 0; table[i].name[0] != '\0'; i++) {
		if (strcasecmp(name, table[i].name) == 0)
			return(table[i].descr);
	}
	return("*** unknown capability");
}

/*
 * setcap(name, table, string) - insert capability string
 */
void
setcap(name, table, string)
	char *name;		/* capability name */
	struct cmdtable *table; /* capability table */
	char *string;		/* capability string */
{
	int i;

	for (i = 0; table[i].name[0] != '\0'; i++) {
		if (strcasecmp(name, table[i].name) == 0) {
			strcpy(table[i].descr, string);
			return;
		}
	}
	strcpy(table[i].name, name);
	strcpy(table[i].descr, string);
	table[i + 1].ident = C_ERROR;
	strcpy(table[i + 1].descr, "*** unknown capability name");
}


/*
 * scanarg(radio, arg, table, sub) - execute capability
 */
static int
scanarg(radio, name, table, cmd)
	struct icom *radio;	/* radio structure */
	char *name;		/* subcommand */
	struct cmdtable *table; /* subcommand table */
	int cmd;		/* command */
{
	int temp;		/* temp */

	temp = argsx(name, table);
	if (temp != C_ERROR) {
		if (setcmd(radio, cmd, temp)) {
			printf("*** radio %s can't do that\n",
			    radio->name);
			temp = C_ERROR;
		}
	}
	return(temp);
}

/*
 * getline(str) - process input line and extract tokens
 *
 * Blank lines and comments beginning with '#' are ignored and the
 * string converted to lower case. The resulting tokens are saved in the
 * *argv[] array. The number of tokens found is returned to the caller.
 */
static int
getline(str)			/* returns number of tokens */
	char *str;		/* pointer to input string */
{
	char *ptr;
	char xbreak[] = " ,\t\n\0";
	char sbreak[] = "\"\n\0";

	int i, j, temp;

	ptr = strchr(str, '#');
	if (ptr != NULL)
		*ptr = '\0';
	ptr = str;
	for (i = 0; i < ARGMAX;) {
		temp = strspn(ptr, xbreak);
		ptr += temp;
		if (*ptr == '\0')
			break;
		if (*ptr == '"') {
			argv[i++] = ++ptr;
			temp = strcspn(ptr, sbreak);
		} else {
			argv[i++] = ptr;
			temp = strcspn(ptr, xbreak);
		}
		ptr += temp;
		if (*ptr == '\0')
			break;
		*ptr++ = '\0';
	}
	return(i);
}

/*
 * args(radio, sptr) - decode channel argument
 *
 * $	last channel
 * dig	specified channel
 * oth	current channel
 *
 * argument is clamped at low and high limits of valid range.
 */
static int
args(radio, sptr)		/* returns 0 (ok), 1 (error) */
	struct icom *radio;	/* radio structure */
	char *sptr;		/* ascii argument pointer */
{
	int temp;

	if (*sptr == '$')
		temp = radio->maxch;
	else if (isdigit(*sptr))
		temp = atoi(sptr);
	else
		temp = radio->mchan;
	if (temp < 1)
		temp = 1;
	if (temp > radio->maxch)
		temp = radio->maxch;
	return(temp);
}

/*
 * sw_keypad() - switch to keypad mode
 */
static int
sw_keypad()			/* returns 0 (ok), 1 (error) */
{
	fd = open("/dev/tty", O_RDONLY);
	if (fd < 0) {
		printf("*** open error %d\n", errno);
		return(1);
	}
	if (tcgetattr(fd, &terma) < 0) {
		printf("*** tcgetattr error %d\n", errno);
		return(1);
	}
	tcgetattr(fd, &termb);
	termb.c_lflag &= ~(ICANON | ECHO);
	termb.c_cc[VMIN] = 1;
	termb.c_cc[VTIME] = 0;
	if (tcsetattr(fd, TCSADRAIN, &termb) < 0) {
		printf("*** tcsetattr error %d\n", errno);
		return(1);
	}
	return(0);
}

/*
 * sw_keybd() - switch to keyboard mode
 */
static int
sw_keybd()			/* returns 0 (ok), 1 (error) */
{
	if (tcsetattr(fd, TCSADRAIN, &terma) < 0) {
		printf("*** tcsetattr error %d\n", errno);
		return(1);
	}
	return(0);
}

/* end program */
