/*
 * UnixCW - Unix CW (Morse code) training program
 * Copyright (C) 1998  Simon Baldwin (simonb@sco.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * cwcp - curses-based morse practice program, with menuing interface
 *
 */

/* trap for when no valid platform selected */
#if !defined(LINUX) && !defined(SCO) && !defined(UNIXWARE2) && !defined(UNIXWARE7)
#include <-DLINUX,-DSCO,_or_-DUNIXWARE2/7_must_be_defined>
#endif

/* include files */
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/param.h>
#include <signal.h>
#include <sys/wait.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <curses.h>
#include <ctype.h>

/* conditional include files */
#if defined(LINUX)
#	include <strings.h>
#	include <getopt.h>
#endif
#if defined(SCO)
#	include <string.h>
#	include <limits.h>
#	define	MAXPATHLEN	_POSIX_PATH_MAX
#endif
#if defined(UNIXWARE2) || defined(UNIXWARE7)
#	include <string.h>
#	include <limits.h>
#endif

/* include definitions of CW control parameters */
#include "cw.h"

/* definitions */
#define	TITLE		"UNIX Morse tutor v1.1, (c) 1997 G0FRD"
#define	VERSION		"cwcp version 1.1"
#define COPYRIGHT	"Copyright (C) 1998  Simon Baldwin\n\
This program comes with ABSOLUTELY NO WARRANTY; for details\n\
please see the file 'COPYING' supplied with the source code.\n\
This is free software, and you are welcome to redistribute it\n\
under certain conditions; again, see 'COPYING' for details.\n\
This program is released under the GNU General Public License."

/* data for generating characters to send */
#define	LETTERS_CHARSET	"A|B|C|D|E|F|G|H|I|J|K|L|M|N|O|P|Q|R|S|T|U|V|W|X|Y|Z"
#define	NUMBERS_CHARSET	"0|1|2|3|4|5|6|7|8|9"
#define ALPHANUM_CHARSET "A|B|C|D|E|F|G|H|I|J|K|L|M|N|O|P|Q|R|S|T|U|V|W|X|Y|Z|\
			0|1|2|3|4|5|6|7|8|9"
#define ALLCHAR_CHARSET	"A|B|C|D|E|F|G|H|I|J|K|L|M|N|O|P|Q|R|S|T|U|V|W|X|Y|Z|\
			0|1|2|3|4|5|6|7|8|9|=|-|,|?|.|;|)|/"
#define	EISH5_CHARSET	"E|I|S|H|5"
#define	TMO0_CHARSET	"T|M|O|0"
#define	AUV4_CHARSET	"A|U|V|4"
#define	NDB6_CHARSET	"N|D|B|6"
#define	KXffRP_CHARSET	"K|X|=|-|R|P"
#define	FLYQC_CHARSET	"F|L|Y|Q|C"
#define	WJ1GZ_CHARSET	"W|J|1|G|Z"
#define	x23789_CHARSET	"2|3|7|8|9"
#define	ffffff_CHARSET	",|?|.|;|)|/"
#define WORDS_CHARSET   "\
	ADA |ADO |AGE |AGO |AID |AIM |AIR |ALE |AMP |APT |ARE |ART |\
	ASH |ASK |ASR |ATE |BAY |BEN |BOB |BOY |BUG |BUT |BUY |CAL |\
	COG |COP |COS |COX |CUP |CUT |DAM |DAN |DEE |DES |DID |DIE |\
	DIM |DIP |DON |DOT |DRY |DUE |ELF |EST |ETA |ETC |EYE |FAN |\
	FAR |FEB |FED |FEE |FEW |FIG |FIT |FOX |FUN |FUR |GAP |GAS |\
	GIN |GNU |GOO |GOT |GUY |HAD |HAM |HAS |HAT |HER |HIS |HOG |\
	HOP |HOT |HOW |ICE |ILL |ION |ITS |JAN |JAY |JIM |JOB |JOE |\
	KEN |KID |LAB |LAP |LAX |LAY |LED |LEE |LET |LEX |LIB |LIE |\
	LIN |LOT |LOW |MAC |MAD |MAP |MAR |MAY |MET |MIT |MIX |MOD |\
	MRS |NAP |NED |NON |NOR |NOV |NOW |OAK |ODD |OUR |OWN |PAD |\
	PAL |PAN |PAT |PAY |PEN |PEP |PER |PIT |POT |PRO |PUT |RAN |\
	RED |REF |REP |RID |RIM |RIO |RIP |ROB |ROD |RON |ROW |SAG |\
	SAN |SAP |SAT |SAW |SAY |SEC |SEE |SHE |SIN |SIP |SIX |SKI |\
	SKY |SOS |SPA |SPY |TAG |TAR |TED |TEE |TEN |TIC |TIE |TIN |\
	TIP |TOM |TOP |TOR |TRY |TWO |USA |VAN |VIA |WAN |WAS |WEB |\
	WED |WET |WHY |YET |YOU |ZIP |\
	ABLE |ACRE |ACTS |ADDS |AGED |AIDE |ALAN |ALLY |ALMA |ALSO |AQUA |\
	ARCH |AREA |ARTS |ASIA |ASKS |AUTO |AWAY |BACH |BACK |BALL |BAND |\
	BANK |BARE |BASH |BATH |BAYS |BEAR |BEEN |BEEP |BELL |BENT |BEST |\
	BETA |BILL |BIND |BLOW |BLUE |BODY |BOLD |BOMB |BOOK |BORN |BOTH |\
	BUGS |BULK |BULL |BUMP |CAFE |CALL |CAME |CAMP |CANS |CANT |CAPS |\
	CARE |CART |CASH |CAST |CELL |CENT |CHAP |CHAT |CHIN |CHIP |CITY |\
	CLIP |COLD |COMB |COME |COOL |COPE |COPS |COPY |CORE |CORP |COST |\
	CRAY |CUBE |CUTE |CUTS |DARK |DASH |DAVE |DAYS |DEAD |DEAL |DEAR |\
	DECK |DEEP |DELL |DEMO |DENY |DESK |DEWY |DHAL |DIED |DIES |DISC |\
	DOES |DOLL |DONE |DOTS |DOUG |DOWN |DRAG |DRAM |DRAW |DROP |DUAL |\
	DUCT |DUFF |DUMB |DUNG |DUST |EACH |EASE |EAST |EASY |EATS |EDGE |\
	EDIT |EMIT |ENDS |ERGO |ERIC |EVEN |EVER |FACE |FACT |FAIR |FAKE |\
	FALL |FANS |FAST |FEAR |FEEL |FEES |FEET |FELL |FELT |FIAT |FILL |\
	FILM |FINE |FIRM |FITS |FIVE |FLAG |FLAT |FLAW |FLEX |FLIP |FLUX |\
	FOIL |FONT |FOOL |FOOT |FORD |FORE |FORM |FOUR |FUSE |GAIL |GAIN |\
	GAME |GAPS |GATE |GAVE |GENE |GINA |GIVE |GLUE |GOAL |GODS |GOES |\
	GOLD |GOOD |GOOF |GRAB |GRAY |GREW |GRID |GROW |GULF |HACK |HALF |\
	HALL |HALT |HAND |HANG |HARD |HARM |HASH |HAVE |HEAP |HEAR |HEAT |\
	HEED |HELD |HIDE |HIGH |HILL |HITS |HOGS |HOLD |HOLE |HOLT |HOOD |\
	HOOK |HOPE |HUGE |HUNG |HUNT |HURT |ICES |ICON |IDEA |IDLE |INCH |\
	INTO |IRAN |ITEM |JACK |JANE |JAVA |JEAN |JEFF |JOBS |JOHN |JOIN |\
	JULY |JUMP |JUNE |JUST |KEPT |KICK |KILL |KIND |KITS |KIWI |KNOW |\
	LABS |LACK |LAID |LAMB |LAST |LAWS |LAZY |LEAD |LEAF |LEAK |LEAP |\
	LEAR |LEDS |LEES |LEFT |LESS |LETS |LEVY |LIBS |LIES |LIEU |LIFE |\
	LIFT |LIKE |LINK |LINT |LISP |LIVE |LOGO |LOGS |LONE |LOOK |LOSE |\
	LOSS |LOST |LOTS |LOUD |LOVE |LYNX |MADE |MANA |MANY |MAPS |MARS |\
	MASS |MATE |MAZE |MEAN |MEET |MEMO |MEND |MESS |MICE |MIKE |MIND |\
	MINE |MING |MINI |MISS |MODS |MOST |MOVE |MUCH |MYTH |NEAR |NEWS |\
	NICE |NIMH |NOEL |NUTS |OAKS |OBEY |OKAY |OMIT |ONES |ONTO |OUCH |\
	OURS |OUTS |OVER |OWNS |PACK |PAID |PAIR |PALS |PARA |PARK |PART |\
	PAST |PAUL |PEAK |PEEK |PETE |PHIL |PICK |PIKE |PING |PINK |PINS |\
	PIPE |PLAN |PLAY |PLUG |PLUM |POLL |POOR |POPS |POSE |POST |PRAY |\
	PREP |PROF |PROP |PULL |PURE |PUSH |PUTS |QUAD |RACE |RACK |RAIN |\
	RARE |RAYS |REAL |REAR |REDO |REEL |RELY |REST |RICE |RICH |RICK |\
	RING |RIPE |RISK |ROAD |ROBS |ROLE |ROLL |ROOM |RULE |RUNS |RYAN |\
	SAFE |SAGE |SAID |SAKE |SANE |SAYS |SEAL |SEEK |SEEM |SEEN |SEES |\
	SELF |SELL |SEND |SETS |SHIP |SHOT |SHOW |SHUT |SIDE |SIGN |SING |\
	SITS |SLED |SLEW |SLIP |SLOT |SLOW |SNAP |SNIP |SNOW |SOCK |SOFT |\
	SOLD |SONS |SONY |SOON |SOUP |SPAN |SPIN |SPOT |SPRY |SPUN |STAY |\
	STOP |STUB |SUCH |SUDS |SUED |SUIT |SUMS |SURE |TACT |TAGS |TAIL |\
	TAKE |TALE |TASK |TELL |TEND |THAN |THEM |THEY |THIN |THUS |TICK |\
	TIDY |TIED |TIER |TIFF |TILL |TIPS |TOLD |TOLL |TONE |TONY |TOOK |\
	TOOL |TORY |TRAP |TREE |TRIM |TRIO |TRIP |TRUE |TUBA |TUBS |TUNE |\
	TURN |TWIN |TWOS |UGLY |UNDO |UPON |USES |VAST |VERY |VICE |VIDE |\
	VIEW |VISA |VIVA |VOID |VOLT |WADE |WAKE |WALK |WALL |WANT |WARD |\
	WARE |WARM |WARN |WAYS |WEAK |WEAR |WEEK |WELL |WEND |WENT |WERE |\
	WEST |WHAT |WHEN |WHOM |WHOS |WILD |WILL |WINK |WIPE |WIRE |WISE |\
	WISH |WITH |WONT |WOOD |WORD |WORM |WORN |WRAP |WREN |YANK |YEAH |\
	YEAR |YELL |ZERO |ZION |ZOOM |\
	ABIDE |ABORT |ABOUT |ACTED |ADAPT |ADDED |ADOPT |AGENT |AGING |AGREE |\
	AHEAD |AIMED |ALARM |ALERT |ALIGN |ALIKE |ALLEN |ALLOC |ALLOW |ALLOY |\
	ALONE |ALONG |ALPHA |ALTER |ALTON |ALTOS |AMEND |AMONG |AMPLE |ANGLE |\
	ANNEX |ANVIL |APART |APPLE |APPLY |APRIL |APTLY |AREAS |ARISE |ARMED |\
	ASIDE |ASKED |ATING |AUDIO |AUSER |AVOID |AWARD |AWARE |BACKS |BADLY |\
	BANDS |BANKS |BASED |BASES |BASIC |BASIS |BATCH |BEARS |BEATS |BEEPS |\
	BEGAN |BEGIN |BEGUN |BEING |BELLS |BIBLE |BILLS |BINDS |BISON |BLACK |\
	BLINK |BLOWN |BLOWS |BLURS |BOLTS |BOUND |BOXES |BRACE |BRAND |BRAVO |\
	BRIEF |BRING |BROAD |BROKE |BROWN |BRUCE |BRYAN |BUDDY |BUFFS |BUGGY |\
	BUILD |BUILT |BUMPS |BURST |BUSES |CALLY |CARDS |CARET |CARRY |CARTE |\
	CATCH |CATER |CEASE |CELLS |CENTS |CHAIN |CHAIR |CHANG |CHARM |CHART |\
	CHASE |CHEAP |CHECK |CHESS |CHIEF |CHILD |CHILE |CHINA |CHING |CHIPS |\
	CHIRP |CHOCK |CHOSE |CHRIS |CHUNK |CITED |CLAIM |CLASH |CLASS |CLEAN |\
	CLICK |CLIFF |CLIPS |CLOCK |CLONE |CLUES |COAST |CODES |COMES |COMET |\
	COMMA |COSTS |COULD |COURT |COVER |CRASH |CROSS |CRUEL |CRYPT |CURLY |\
	CURRY |CURVE |CYCLE |CZECH |DAILY |DAISY |DATED |DATES |DAVID |DEALS |\
	DEATH |DEBIT |DEBRA |DEBUG |DECAY |DEFER |DELLS |DEMON |DEPTH |DIALS |\
	DIGIT |DIODE |DIRTY |DITTO |DITTY |DIVER |DOING |DOORS |DOUBT |DOZEN |\
	DRAFT |DRAWN |DRAWS |DRESS |DRIFT |DROOL |DROPS |DUMMY |DUMPS |DUTCH |\
	DYING |EAGLE |EARLY |EARTH |ECHOS |EDITS |EGRET |EIGHT |EJECT |ELECT |\
	ELITE |ELMER |EMBED |EMITS |EMPTY |ENDED |ENVOY |EPOCH |EQUAL |ERASE |\
	EVENT |EVERY |EXACT |EXITS |FACED |FACES |FACET |FACTS |FAILS |FAIRY |\
	FALLS |FALSE |FATAL |FAULT |FEEDS |FEELS |FETCH |FEWER |FIFTH |FILED |\
	FILET |FILLS |FINAL |FINDS |FIRES |FIXED |FIXES |FLAKY |FLAWS |FLOAT |\
	FLOCK |FLUSH |FOCAL |FOCUS |FONTS |FORCE |FORGO |FORKS |FORMS |FORTH |\
	FORUM |FRAME |FREED |FREES |FRESH |FRONT |FROWN |FRUIT |FUDGE |FULLY |\
	FUSER |GAMES |GATED |GATES |GENOA |GENRE |GHOST |GIVEN |GIVES |GOING |\
	GRADE |GRANT |GRAPH |GREAT |GREEK |GREEN |GROWN |GROWS |GUARD |GUESS |\
	GUEST |HALTS |HANDS |HANDY |HAPPY |HARRY |HEADS |HEARD |HEAVY |HELPS |\
	HENCE |HENRY |HERTZ |HIDES |HINTS |HIRES |HOLDS |HOLES |HONEY |HOOKS |\
	HOPED |HOSTS |HOURS |HOUSE |HUMAN |ICONS |IDEAL |IDEAS |IDLER |IMPLY |\
	INCUR |INDIA |INNER |INTEL |IRATE |IRISH |ISSUE |ITEMS |JACKS |JAMES |\
	JAPAN |JELLY |JERKY |JERRY |JOINS |JOKES |JONES |JULEP |JUMBO |JUMPY |\
	KAREN |KEEPS |KEITH |KEVIN |KEYED |KICKS |KILLS |KILTS |KINDS |KNOWN |\
	KNOWS |KOREA |LACKS |LARGE |LARRY |LASER |LASTS |LATCH |LATIN |LAUGH |\
	LAYER |LEADS |LEAKS |LEARN |LEASE |LEAST |LEAVE |LEGAL |LEVER |LIFTS |\
	LIGHT |LIMIT |LINDA |LINED |LINKS |LINTS |LINUS |LISTS |LIVES |LOADS |\
	LOCKS |LOCUS |LOGIC |LOGON |LONGS |LOOPS |LOOSE |LOSES |LOUIS |LOWER |\
	LYNCH |MACRO |MAGIC |MAILS |MAINS |MAKER |MAKES |MALLS |MARCH |MARKS |\
	MARRY |MASKS |MASON |MAYBE |MEANS |MEANT |MEDAL |MEETS |MERGE |MEYER |\
	MICRO |MIDGE |MIGHT |MIMIC |MINED |MINIS |MINUS |MIXED |MODEM |MODES |\
	MONEY |MONTH |MOTIF |MOTOR |MOUTH |MOVED |MOVES |MULTI |MUSIC |NAMED |\
	NEEDS |NESTS |NEWER |NEWLY |NICER |NIGHT |NINTH |NOISE |NOISY |NORTH |\
	NOTED |NULLS |OCCUR |OCEAN |OCTAL |OCTET |OFFER |OFTEN |OLDER |ONSET |\
	OPENS |OPTIC |ORDER |OUGHT |OUTER |OWNED |PACKS |PAGER |PAINT |PAIRS |\
	PANEL |PAPER |PARSE |PARTY |PASTE |PATCH |PATHS |PEAKS |PETER |PHASE |\
	PIPED |PIPES |PITCH |PLACE |PLAIN |PLANS |PLATE |PLAYS |PLUGS |POKES |\
	POLLS |PORTS |POSES |POSTS |POUND |POWER |PRICE |PRIME |PRINT |PRIOR |\
	PROBE |PRONE |PROVE |PROXY |PULLS |PULSE |PURGE |QUERY |QUEST |QUICK |\
	QUIET |QUITE |QUITS |QUOTE |RACES |RADAR |RADIO |RADIX |RAISE |RAJAH |\
	RAPID |RARER |RATED |RATES |RATIO |RAVEN |REACH |REACT |READS |READY |\
	REFER |RELAX |REMAP |REPLY |RERUN |RESET |RETRY |REUSE |RIDGE |RIGHT |\
	RINGS |RISER |RISKS |RISKY |RIVER |ROGER |ROLES |ROMAN |ROOTS |ROUND |\
	ROUTE |RUINS |RULES |SABRE |SAFER |SALES |SAMOA |SANDY |SAVED |SAVES |\
	SCALE |SCANS |SCENE |SCOPE |SCOTT |SEEKS |SEEMS |SEIZE |SELLS |SENDS |\
	SERVE |SERVO |SETUP |SEVEN |SHACK |SHAKE |SHALL |SHAPE |SHARE |SHARP |\
	SHEDS |SHEER |SHEET |SHELF |SHIPS |SHORT |SHOWS |SHUTS |SIDED |SIDES |\
	SIGMA |SIGNS |SINCE |SITES |SIXTH |SIXTY |SIZED |SIZES |SKIPS |SLACK |\
	SLASH |SLICE |SLINK |SLOTS |SLOWS |SMALL |SMART |SMITH |SMOKE |SNOOP |\
	SOLID |SOLVE |SORRY |SORTS |SOUND |SOUTH |SPAIN |SPANS |SPARE |SPAWN |\
	SPEAK |SPECS |SPELL |SPEND |SPENT |SPINS |SPLAT |SPLIT |SPOIL |SPOTS |\
	SPREE |STAFF |STAGE |STAIR |STALE |STAMP |STAND |STAYS |STERN |STEVE |\
	STICK |STILL |STOCK |STOPS |STORE |STORY |STRAP |STRAY |STRIP |STUCK |\
	STUDY |STUFF |STYLE |SUING |SUITE |SUPER |SUSAN |SWAPS |SWIFT |SWISS |\
	TAKEN |TAKES |TALES |TALKS |TAPES |TAXES |TEACH |TEAMS |TELLS |TENTH |\
	TENTS |TERMS |TERRY |TERSE |TESTS |TEXAS |TEXTS |THANK |THEIR |THERE |\
	THESE |THING |THINK |THIRD |THORN |THOSE |THREE |THROW |THUMB |TICKS |\
	TIERS |TIGHT |TIMED |TIMER |TIMES |TITAN |TODAY |TOKEN |TOMER |TONES |\
	TOOTH |TOPIC |TOUCH |TOWER |TRACE |TRADE |TRAIL |TRAPS |TRASH |TREAT |\
	TREES |TREND |TRICK |TRIED |TRIES |TRULY |TRUST |TULIP |TUNED |TUNER |\
	TUNES |TUPLE |TURNS |TUTOR |TWICE |TWIST |TYLER |TYPED |ULTRA |UNDER |\
	UNION |UNITS |UPPER |USUAL |VAGUE |VALID |VERSA |VICES |VIDEO |VIEWS |\
	VINES |VIPER |VIRUS |VISIT |VITAL |VOICE |VOLTS |WAITS |WANTS |WARNS |\
	WASTE |WATCH |WATER |WEEKS |WEIRD |WELSH |WHEEL |WHERE |WHICH |WHILE |\
	WHITE |WHOLE |WHOSE |WIDER |WIDTH |WINGE |WIPED |WIRED |WIRES |WORDS |\
	WORKS |WORLD |WORMS |WORRY |WORSE |WORST |WORTH |WOULD |WRAPS |WRECK |\
	WRITE |WRONG |WROTE |XEROX |YATES |YEARS |YIELD |YOURS |YUKON |ZEROS |\
	ZONES "
#define	CWWORDS_CHARSET	"\
	( |+ |73 |88 |AA |AB |ABT |ADR |AGN |AM |ANT |AS |B4 |BCI |\
	BCL |BK |BN |BUG |C |CFM |CL |CLD |CLG |CQ |CW |DLD |DLVD |\
	DR |DX |ES |FB |FM |GA |GM |GN |GND |GUD |HI |HR |HV |HW |\
	K |LID |MA |MILS |MSG |N |NCS |ND |NIL |NM |NR |NW |OB |OC |\
	OM |OP |OPR |OT |PBL |PSE |PWR |PX |QRG |QRH |QRI |QRJ |QRK |\
	QRL |QRM |QRN |QRO |QRP |QRQ |QRS |QRT |QRU |QRV |QRW |QRX |\
	QRY |QRZ |QSA |QSB |QSD |QSG |QSK |QSL |QSM |QSN |QSO |QSP |\
	QST |QSU |QSW |QSX |QSY |QSZ |QTA |QTB |QTC |QTH |QTR |R |\
	RCD |RCVR |REF |RFI |RIG |RTTY |RX |SASE |SED |SIG |SKED |\
	SRI |SSB |SVC |T |TFC |TKS |TMW |TNX |TT |TU |TVI |TX |TXT |\
	UR |URS |VFO |VY |WA |WB |WD |WDS |WKD |WKG |WL |WUD |WX |\
	XCVR |XMTR |XTAL |XYL |YL |\
	BURO |HPE |CU |SN |TEST |TEMP |MNI |RST |599 |5NN |559 |CPY |\
	DE |OK |RPT |ANI |BCNU |BD |BLV |CC |CK |CNT |CO |CONDX |CPSE |\
	CRD |CUD |CUAGN |CUL |ELBUG |ENUF |FER |FONE |FREQ |GB |GD |\
	GE |HVY |II |INPT |LSN |PA |PP |RPRT |RPT |SA |STN |SUM |\
	SWL |TRX |WID "
#define	PARIS_CHARSET	"PARIS "

/* keyboard send items */
#define	OK_KB_CHARS	\
		" ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789\"'$()+,-./:;=?_%!@^*&#"
#define	TRANS_BUFLEN	32			/* buf for [..] translate */

/* general */
#define	EOS			'\0'		/* end of string */
#define	ASC_SPACE		' '		/* ASCII space char */
#define	ASC_CR			'\012'		/* ASCII CR char */
#define	ASC_ESC			'\033'		/* ASCII Esc char */
#define	ASC_CTRLL		'\014'		/* ASCII ^L char */
#define	ASC_CTRLC		'\003'		/* ASCII ^C char */
#define	ASC_PLUS		'+'		/* ASCII + char */
#define	ASC_MINUS		'-'		/* ASCII - char */
#define	ASC_STAR		'*'		/* ASCII * char */
#define	ASC_SLASH		'/'		/* ASCII / char */
#define	ASC_TAB			'\t'		/* ASCII tab char */
#define	ASC_FNS			'/'		/* ASCII filename sep char */
#define	ASC_ELES		'|'		/* ASCII charset ele sep char */
#define	GROUPSIZE		5		/* size of char groups */
#define	MBUF_LEN		80		/* general msg buffer */
#define	MAXARGS			128		/* args to exec */
#define	UI_POLL			500		/* num uS to poll curses
						   when waiting for CW echo */
#define	HANG_ABOUT		1		/* num secs to wait for CW
						   child proc to settle */

/* alternative F-keys for folks without (some) F-keys */
#define	CTRL_OFFSET	0100			/* ctrl keys are 'X' - 0100 */
#define	PSEUDO_KEYF1	('Q'-CTRL_OFFSET)	/* alternative FKEY1 */
#define	PSEUDO_KEYF2	('W'-CTRL_OFFSET)	/* alternative FKEY2 */
#define	PSEUDO_KEYF3	('E'-CTRL_OFFSET)	/* alternative FKEY3 */
#define	PSEUDO_KEYF4	('R'-CTRL_OFFSET)	/* alternative FKEY4 */
#define	PSEUDO_KEYF5	('T'-CTRL_OFFSET)	/* alternative FKEY5 */
#define	PSEUDO_KEYF6	('Y'-CTRL_OFFSET)	/* alternative FKEY6 */
#define	PSEUDO_KEYF7	('U'-CTRL_OFFSET)	/* alternative FKEY7 */
#define	PSEUDO_KEYF8	('I'-CTRL_OFFSET)	/* alternative FKEY8 */
#define	PSEUDO_KEYF9	('A'-CTRL_OFFSET)	/* alternative FKEY9 */
#define	PSEUDO_KEYF10	('S'-CTRL_OFFSET)	/* alternative FKEY10 */
#define	PSEUDO_KEYF11	('D'-CTRL_OFFSET)	/* alternative FKEY11 */
#define	PSEUDO_KEYF12	('F'-CTRL_OFFSET)	/* alternative FKEY12 */

/* limits to the CW parameters */
#define	MIN_WPM			4
#define	MAX_WPM			60
#define	STEP_WPM		1
#define	MIN_HZ			10
#define	MAX_HZ			4000
#define	STEP_HZ			10
#define	MIN_GAP			0
#define	MAX_GAP			99
#define	STEP_GAP		1
#define	MIN_TIME		1
#define	MAX_TIME		999
#define	STEP_TIME		1

/* some colour definitions */
static short	colour_array[] = {
		COLOR_BLACK, COLOR_RED, COLOR_GREEN, COLOR_YELLOW,
		COLOR_BLUE, COLOR_MAGENTA, COLOR_CYAN, COLOR_WHITE };
#define	NUM_COLOURS		(sizeof( colour_array ) / sizeof( short ))
#define	BOX_COLOURS		1		/* normal colour pair */
#define	DISP_COLOURS		2		/* blue colour pair */
#if defined(UNIXWARE2)
#	define	REV_DISP_COLOURS	3	/* reverse blue colour pair */
#endif
#define DISP_FGIND		7		/* white foreground */
#define DISP_BGIND		4		/* blue background */
#define BOX_FGIND		7		/* white foreground */
#define BOX_BGIND		0		/* black background */

/* operating modes and their strings */
static char	*modes[] = {
	" Letter groups    ", " Number groups    ", " Alphanum groups  ",
	" All char groups  ", " Words            ", " CW words         ",
	" Keyboard         ", " PARIS calibrate  ", " EISH5 groups     ",
	" TMO0 groups      ", " AUV4 groups      ", " NDB6 groups      ",
	" KX=-RP groups    ", " FLYQC groups     ", " WJ1GZ groups     ",
	" 23789 groups     ", " ,?.;)/ groups    ",
	" Exit             " };
#define	NUM_MODES		(sizeof( modes ) / sizeof( char* ))
#define	MODE_LETTERS		0
#define	MODE_NUMBERS		1
#define	MODE_ALPHANUM		2
#define	MODE_ALLCHAR		3
#define	MODE_WORDS		4
#define	MODE_CWWORDS		5
#define	MODE_KEYBOARD		6
#define	MODE_PARIS		7
#define	MODE_ESIH5		8
#define	MODE_TMO0		9
#define	MODE_AUV4		10
#define	MODE_NDB6		11
#define	MODE_KXffRP		12
#define	MODE_FLYQC		13
#define	MODE_WJ1GZ		14
#define	MODE_23789		15
#define	MODE_ffffff		16
#define	MODE_QUIT		17

#define	MAXARGS			128		/* args to exec */
#define	MAXOPTSTR		1024		/* longest _OPTIONS env var */
#define	ARGS_WHITESPACE		" \t"		/* argument separators */
#define	ENV_OPTIONS		"CWCP_OPTIONS"	/* env string holding options */

/* initial values for some variables */
#define	INIT_WPM		12
#define	INIT_HZ			800
#define	INIT_GAP		0
#define	INIT_TIME		15
#define	INIT_ADJ		0
#define	INIT_CWCOMMAND		"cw -d /dev/tty"

/* lots of work just to send '73' on exit! */
#define	x73_WPM			30
#define	x73_HZ			1000
#define	x73_GAP			0
#define	x73_ADJ			0
#define	x73_MSGS		0
#define	x73_ECHO		1
#define	x73_CMDS		1
#define	x73_COMBO		1
#define	x73_STRING		"73 "

/* warbling tone stuff */
#define	WARBLE_WPM		60		/* use 60WPM morse */
#define	WARBLE_TONE1		500		/* 500/1000 Hz */
#define	WARBLE_TONE2		1000
#define	WARBLE_CHAR		'E'		/* nice short character */

/* global variables defining state of program */
static int	cw_wpm		= INIT_WPM;	/* current WPM */
static int	cw_hz		= INIT_HZ;	/* current tone in Hz */
static int	cw_gap		= INIT_GAP;	/* current extra sending gap */
static int	cw_adj		= INIT_ADJ;	/* current timing adjustment */
static int	cw_msgs		= 0;		/* CW msgs setting */
static int	cw_echo		= 1;		/* CW echo setting */
static int	cw_cmds		= 1;		/* CW commands setting */
static int	cw_combo	= 1;		/* CW combo-grouping setting */

static int	currmode	= 0;		/* current operating mode */
#define	STATE_IDLE		0		/* idling state */
#define	STATE_ACTIVE		1		/* active state */
#define	STATE_STOPPING		2		/* soon to be idle state */
static int	sending_state	= STATE_IDLE;	/* shows state of sounding */
static bool	cw_resync	= TRUE;		/* forces reload of CW sender
						   operating parameters - init
						   TRUE ensures reload on first
						   character sent to cw */

static char	*cwcommand	= INIT_CWCOMMAND;/* options intially for cw */
static int	ptime		= INIT_TIME;	/* practice timer */
static bool	docolours	= TRUE;		/* initially attempt colours */
static bool	docheerio	= TRUE;		/* send 73 on exit */
static bool	in_windows	= FALSE;	/* set if curses active */

static int	infd, outfd;			/* file descrs for cw data */
static int	pid = 0;			/* child PID */

/* colour values as arrays into colour_array */
static int	disp_fgind	= DISP_FGIND;	/* white foreground */
static int	disp_bgind	= DISP_BGIND;	/* blue background */
static int	box_fgind	= BOX_FGIND;	/* white foreground */
static int	box_bgind	= BOX_BGIND;	/* black background */

/* curses windows */
static WINDOW	*scr, *base_bkgd;
static WINDOW	*titlebox,  *modebox,  *charbox,  *wpmbox,  *hzbox,  *gapbox,
		*timebox;
static WINDOW	*titledisp, *modedisp, *chardisp, *wpmdisp, *hzdisp, *gapdisp,
		*timedisp;

/* forward declarations */
static void send_echoed( char, bool, bool );
static void ui_init();
static void ui_destroy();


/*
 * print_help - give help information
 */
static void print_help( char *pname ) {
	printf(
#if defined(LINUX)
	"Usage: %s [options...]\n\n\
	-x, --cwcommand=CWCMD	specify cw sounder command\n\
			 	(default '%s')\n\
	-t, --hz,--tone=HZ	set initial tone to HZ (default %d)\n\
				valid HZ values are between %d and %d\n\
	-w, --wpm=WPM		set initial words per minute (default %d)\n\
				valid WPM values are between %d and %d\n\
	-g, --gap=GAP		set extra gap between letters (default %d)\n\
				valid GAP values are between %d and %d\n\
	-p, --time=TIME		set initial practice time (default %d mins)\n\
				valid TIME values are between %d and %d\n\
	-a, --adj=ADJ		adjust speed by +/-ADJ %% (default %d)\n\
				valid ADJ values are between %d and %d\n\
	-c, --colours=CSET	set initial display colours where available\n\
				(default %d,%d,%d,%d)\n\
	-m, --mono		specify no colours (default colours)\n\
	-q, --no73		specify no cheery '73' on exit (default on)\n\
	-h, --help		print this message\n\
	-v, --version		output version information and exit\n\n",
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
	"Usage: %s [options...]\n\n\
	-x CWCMD	specify cw sounder command\n\
		 	(default '%s')\n\
	-t HZ		set initial tone to HZ (default %d)\n\
			valid HZ values are between %d and %d\n\
	-w WPM		set initial words per minute (default %d)\n\
			valid WPM values are between %d and %d\n\
	-g GAP		set extra gap between letters (default %d)\n\
			valid GAP values are between %d and %d\n\
	-p TIME		set initial practice time (default %d mins)\n\
			valid TIME values are between %d and %d\n\
	-a ADJ		adjust speed by +/-ADJ %% (default %d)\n\
			valid ADJ values are between %d and %d\n\
	-c CSET		set initial display colours where available\n\
			(default %d,%d,%d,%d)\n\
	-m		specify no colours (default colours)\n\
	-q		specify no cheery '73' on exit (default on)\n\
	-h		print this message\n\
	-v		output version information and exit\n\n",
#endif
	pname, INIT_CWCOMMAND,
	INIT_HZ, MIN_HZ, MAX_HZ,
	INIT_WPM, MIN_WPM, MAX_WPM,
	INIT_GAP, MIN_GAP, MAX_GAP,
	INIT_TIME, MIN_TIME, MAX_TIME,
	INIT_ADJ, CW_MIN_ADJ, CW_MAX_ADJ,
	DISP_FGIND, DISP_BGIND, BOX_FGIND, BOX_BGIND );
}


/*
 * parse_cmdline - parse command line options for initial values
 */
static void parse_cmdline( int argc, char **argv ) {
	int	c;				/* option character */
	char	*argv0;				/* program name */
	int	argind;				/* loop index */
	char	env_options[MAXOPTSTR];		/* env options string */
	char	*sptr;				/* string pointer */
	char	*local_argv[MAXARGS];		/* local argv array */
	int	local_argc = 0;			/* local argc */
#if defined(LINUX)
	int	option_index;			/* option index */
	static struct option long_options[] = {	/* options table */
		{ "cwcommand",	1, 0, 0 },
		{ "tone",	1, 0, 0 },
		{ "hz",		1, 0, 0 },
		{ "wpm",	1, 0, 0 },
		{ "gap",	1, 0, 0 },
		{ "time",	1, 0, 0 },
		{ "adj",	1, 0, 0 },
		{ "colours",	1, 0, 0 },
		{ "mono",	0, 0, 0 },
		{ "no73",	0, 0, 0 },
		{ "help",	0, 0, 0 },
		{ "version",	0, 0, 0 },
		{ 0, 0, 0, 0 }};
#endif

	/* set argv0 to be the basename part of the program name */
	argv0 = argv[0] + strlen( argv[0] );
	while ( *argv0 != ASC_FNS && argv0 > argv[0] )
		argv0--;
	if ( *argv0 == ASC_FNS ) argv0++;

	/* build a new view of argc and argv by first prepending
	   the strings from ..._OPTIONS, if defined, then putting the
	   command line args on (so they take precedence) */
	local_argv[ local_argc++ ] = argv[0];
	if ( getenv( ENV_OPTIONS ) != NULL ) {
		strcpy( env_options, getenv( ENV_OPTIONS ));
		sptr = env_options;
		while ( local_argc < MAXARGS - 1 ) {
			for ( ; strchr( ARGS_WHITESPACE, *sptr ) != (char *)NULL
					&& *sptr != EOS;
					sptr++ );
			if ( *sptr == EOS )
				break;
			else {
				local_argv[ local_argc++ ] = sptr;
				for ( ; strchr( ARGS_WHITESPACE, *sptr )
						== (char *)NULL && *sptr != EOS;
						sptr++ );
				if ( strchr( ARGS_WHITESPACE, *sptr )
						!= (char *)NULL && *sptr != EOS ) {
					*sptr = EOS; sptr++;
				}
			}
		}
	}
	for ( argind = 1; argind < argc; argind++ ) {
		local_argv[ local_argc++ ] = argv[ argind ];
	}

	/* process every option */
#if defined(LINUX)
	while ( (c=getopt_long( local_argc, local_argv, "x:t:w:g:p:a:c:mqhv",
			long_options, &option_index )) != -1 ) {
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
	while ( (c=getopt( local_argc, local_argv, "x:t:w:g:p:a:c:mqhv" ))
			!= -1 ) {
#endif

		/* check for -x or --cwcommand argument */
		if ( c == 'x'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"cwcommand" )) {
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
					) {
#endif
			cwcommand = optarg;
		}

		/* check for -t or --tone/--hz argument */
		else if ( c == 't'
#if defined(LINUX)
			|| c == 0 && ( !strcmp( long_options[option_index].name,
					"tone" ) ||
				    !strcmp( long_options[option_index].name,
					"hz" ))) {
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
					) {
#endif
			if ( sscanf( optarg, "%d", &cw_hz ) != 1 ||
					cw_hz < MIN_HZ || cw_hz > MAX_HZ ) {
				fprintf( stderr, "%s: invalid tone value\n",
						argv0 );
				exit( 1 );
			}
		}

		/* check for -w or --wpm argument */
		else if ( c == 'w'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"wpm" )) {
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
					) {
#endif
			if ( sscanf( optarg, "%d", &cw_wpm ) != 1 ||
					cw_wpm < MIN_WPM || cw_wpm > MAX_WPM ) {
				fprintf( stderr, "%s: invalid wpm value\n",
						argv0 );
				exit( 1 );
			}
		}

		/* check for -g or --gap argument */
		else if ( c == 'g'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"gap" )) {
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
					) {
#endif
			if ( sscanf( optarg, "%d", &cw_gap ) != 1 ||
					cw_gap < MIN_GAP || cw_gap > MAX_GAP ) {
				fprintf( stderr, "%s: invalid gap value\n",
						argv0 );
				exit( 1 );
			}
		}

		/* check for -p or --time argument */
		else if ( c == 'p'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"time" )) {
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
					) {
#endif
			if ( sscanf( optarg, "%d", &ptime ) != 1 ||
					ptime < MIN_TIME || ptime > MAX_TIME ) {
				fprintf( stderr, "%s: invalid time value\n",
						argv0 );
				exit( 1 );
			}
		}

		/* check for -a or --adj argument */
		else if ( c == 'a'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"adj" )) {
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
					) {
#endif
			if ( sscanf( optarg, "%d", &cw_adj ) != 1 ||
					cw_adj < CW_MIN_ADJ ||
					cw_adj > CW_MAX_ADJ ) {
				fprintf( stderr, "%s: invalid adj value\n",
						argv0 );
				exit( 1 );
			}
		}

		/* check for -c or --colours argument */
		else if ( c == 'c'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"colours" )) {
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
					) {
#endif
			if ( sscanf( optarg, "%d,%d,%d,%d",
				&disp_fgind, &disp_bgind,
				&box_fgind,  &box_bgind ) != 4 ||
				disp_fgind < 0 || disp_fgind >= NUM_COLOURS ||
				disp_bgind < 0 || disp_bgind >= NUM_COLOURS ||
				box_fgind < 0 || box_fgind >= NUM_COLOURS ||
				box_bgind < 0 || box_bgind >= NUM_COLOURS ) {
				fprintf( stderr, "%s: invalid colours value\n",
						argv0 );
				exit( 1 );
			}
		}

		/* check for -m or --mono argument */
		else if ( c == 'm'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"mono" )) {
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
					) {
#endif
			docolours = FALSE;
		}

		/* check for -q or --no73 argument */
		else if ( c == 'q'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"no73" )) {
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
					) {
#endif
			docheerio = FALSE;
		}

		/* check for -h or --help argument */
		else if ( c == 'h'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"help" )) {
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
					) {
#endif
			print_help( argv0 );
			exit( 0 );
		}

		/* check for -v or --version argument */
		else if ( c == 'v'
#if defined(LINUX)
			|| c == 0 && !strcmp( long_options[option_index].name,
				"version" )) {
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
					) {
#endif
			printf( "%s, ", VERSION );
			printf( "%s\n", COPYRIGHT );
			exit( 0 );
		}

		/* check for illegal option */
		else if ( c == '?' ) {
			fprintf( stderr,
#if defined(LINUX)
				"Try '%s --help' for more information.\n",
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
				"Try '%s -h' for more information.\n",
#endif
				argv0 );
			exit( 1 );
		}

		/* nothing else left to do */
		else {
			fprintf( stderr, "%s: getopts returned %c\n",
					argv0, c );
			exit( 1 );
		}

	}
	if ( optind != local_argc ) {
		fprintf( stderr,
#if defined(LINUX)
			"Try '%s --help' for more information.\n",
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
			"Try '%s -h' for more information.\n",
#endif
			argv0 );
		exit( 1 );
	}

}


/*
 * prog_exit - shut down any child processes, close any open
 *             files and streams, and exit
 */
static void prog_exit( int status ) {
	char	*sptr;			/* pointer to 73 string */
	int	cld_status;		/* status from cw sender */

	/* send 73 if normal exit */
	if ( status == 0 ) {

		/* may not actually want to send 73 */
		if ( docheerio ) {

			/* reset the CW sender to a known state */
			cw_wpm =	x73_WPM;
			cw_hz =		x73_HZ;
			cw_gap =	x73_GAP;
			cw_adj =	x73_ADJ;
			cw_msgs =	x73_MSGS;
			cw_echo =	x73_ECHO;
			cw_cmds =	x73_CMDS;
			cw_combo =	x73_COMBO;

			/* send 73 - boy, this is hard work! */
			for ( sptr = x73_STRING; *sptr != EOS; sptr++ ) {
				send_echoed( *sptr, FALSE, FALSE );
			}
			sleep( HANG_ABOUT );
		}
	}

	/* close fifos and exit */
	close( infd ); close( outfd );

	/* suppress death-of-child handling, then kill the cw sender */
	signal( SIGCLD, SIG_IGN );
	kill( pid, SIGINT ); /* child may already be dead if ^C used */
	waitpid( pid, &cld_status, 0 );

	/* complete curses if relevant, check sender status, and exit */
	ui_destroy();
	if ( WIFEXITED( cld_status ) && WEXITSTATUS( cld_status ) != 0 ) {
		printf( "Cw sender exited with status %d\n",
				WEXITSTATUS( cld_status ));
	}
	else if ( WIFSIGNALED( cld_status ) ) {
		printf( "Cw sender signaled with signal %d\n",
				WTERMSIG( cld_status ));
	}
	exit( status );
}


/*
 * ui_getch - get a character from the keyboard, with optional wait
 */
static int ui_getch( bool getch_delay ) {

	/* set up curses input mode to suit the user interface */
	keypad( scr, TRUE );
	noecho(); cbreak(); curs_set( 0 ); raw();

	/* if delay specified, then wait until a command is received */
	if ( getch_delay )
		nodelay( scr, FALSE );
	else
		nodelay( scr, TRUE );

	/* return what's in the input queue */
	return( getch() );
}


/*
 * ui_interpret - assess a user command, and action it if valid
 */
static void ui_interpret( int c ) {
	char	str[MBUF_LEN];			/* string buffer - general */
	int	prev_mode;			/* previous mode on mode chg */

	/* interpret the command passed in */
	switch ( c ) {

	case ASC_PLUS:
		disp_bgind++;
		if ( disp_bgind >= NUM_COLOURS ) disp_bgind = 0;
		goto colour_update;

	case ASC_MINUS:
		disp_fgind++;
		if ( disp_fgind >= NUM_COLOURS ) disp_fgind = 0;
		goto colour_update;

	case ASC_STAR:
		box_bgind++;
		if ( box_bgind >= NUM_COLOURS ) box_bgind = 0;
		goto colour_update;

	case ASC_SLASH:
		box_fgind++;
		if ( box_fgind >= NUM_COLOURS ) box_fgind = 0;
		goto colour_update;

colour_update:
		if ( docolours && has_colors() ) {
			init_pair( BOX_COLOURS,
					colour_array[box_fgind],
					colour_array[box_bgind] );
			init_pair( DISP_COLOURS,
					colour_array[disp_fgind],
					colour_array[disp_bgind] );
#if defined(UNIXWARE2)
			init_pair( REV_DISP_COLOURS,
					colour_array[box_bgind],
					colour_array[box_fgind] );
#endif
			wrefresh( curscr );
		}
		break;

	case ASC_CTRLL:
		wrefresh( curscr );
		cw_resync = TRUE;
		break;


	case KEY_F(1):
	case PSEUDO_KEYF1:
	case KEY_LEFT:
		if ( cw_wpm <= MIN_WPM ) break;
		cw_wpm -= STEP_WPM;
		goto wpm_upd;

	case KEY_F(2):
	case PSEUDO_KEYF2:
	case KEY_RIGHT:
		if ( cw_wpm >= MAX_WPM ) break;
		cw_wpm += STEP_WPM;
		goto wpm_upd;

wpm_upd:
		sprintf( str, "%2d WPM", cw_wpm );
		mvwaddstr( wpmdisp, 0, 6, str );
		wrefresh( wpmdisp );
		break;


	case KEY_F(3):
	case PSEUDO_KEYF3:
	case KEY_END:
		if ( cw_hz <= MIN_HZ ) break;
		cw_hz -= STEP_HZ;
		goto hz_upd;

	case KEY_F(4):
	case PSEUDO_KEYF4:
	case KEY_HOME:
		if ( cw_hz >= MAX_HZ ) break;
		cw_hz += STEP_HZ;
		goto hz_upd;

hz_upd:
		sprintf( str, "%4d Hz", cw_hz );
		mvwaddstr( hzdisp, 0, 4, str );
		wrefresh( hzdisp );
		break;


	case KEY_F(5):
	case PSEUDO_KEYF5:
	case KEY_NPAGE:
		if ( cw_gap <= MIN_GAP ) break;
		cw_gap -= STEP_GAP;
		goto gap_upd;

	case KEY_F(6):
	case PSEUDO_KEYF6:
	case KEY_PPAGE:
		if ( cw_gap >= MAX_GAP ) break;
		cw_gap += STEP_GAP;
		goto gap_upd;

gap_upd:
		sprintf( str, "%2d dot(s)", cw_gap );
		mvwaddstr( gapdisp, 0, 4, str );
		wrefresh( gapdisp );
		break;


	case KEY_F(7):
	case PSEUDO_KEYF7:
		if ( ptime <= MIN_TIME ) break;
		ptime -= STEP_TIME;
		goto time_upd;

	case KEY_F(8):
	case PSEUDO_KEYF8:
		if ( ptime >= MAX_TIME ) break;
		ptime += STEP_TIME;
		goto time_upd;

time_upd:
		sprintf( str, "%3d min(s)", ptime );
		mvwaddstr( timedisp, 0, 4, str );
		wrefresh( timedisp );
		break;


	case KEY_F(11):
	case PSEUDO_KEYF11:
	case KEY_UP:
		if ( sending_state == STATE_ACTIVE )
			sending_state = STATE_STOPPING;
		prev_mode = currmode;
		do {
			currmode--;
			if ( currmode < 0 ) currmode = NUM_MODES - 1;
		} while ( modes[currmode][0] == EOS );
		goto mode_upd;

	case KEY_F(10):
	case PSEUDO_KEYF10:
	case KEY_DOWN:
		if ( sending_state == STATE_ACTIVE )
			sending_state = STATE_STOPPING;
		prev_mode = currmode;
		do {
			currmode++;
			if ( currmode >= NUM_MODES ) currmode = 0;
		} while ( modes[currmode][0] == EOS );
		goto mode_upd;

mode_upd:
#if defined(UNIXWARE2)
		wattron( modedisp, COLOR_PAIR( DISP_COLOURS ));
#else
		wattroff( modedisp, A_REVERSE );
#endif
		mvwaddstr( modedisp, prev_mode, 0, modes[prev_mode] );
#if defined(UNIXWARE2)
		wattron( modedisp, COLOR_PAIR( REV_DISP_COLOURS ));
#else
		wattron( modedisp, A_REVERSE );
#endif
		mvwaddstr( modedisp, currmode, 0, modes[currmode] );
		wrefresh( modedisp );
		break;


	case KEY_F(9):
	case PSEUDO_KEYF9:
		if ( sending_state == STATE_ACTIVE )
			sending_state = STATE_STOPPING;
		else if ( sending_state == STATE_IDLE )
			sending_state = STATE_ACTIVE;
		break;

	case ASC_CR:
		if ( sending_state == STATE_IDLE )
			sending_state = STATE_ACTIVE;
		break;

	case ASC_ESC:
		if ( sending_state == STATE_ACTIVE )
			sending_state = STATE_STOPPING;
		break;

	case KEY_F(12):
	case PSEUDO_KEYF12:
	case ASC_CTRLC:
		/* done */
		prog_exit( 0 );
		break;

	}
}


/*
 * ui_init - initialize the user interface, boxes and windows
 */
static void ui_init() {
	char	str[MBUF_LEN];			/* string buffer - general */
	int	index;				/* modes string index */

	/* start curses initialization */
	scr = initscr(); in_windows = TRUE;
	wrefresh( scr );
	if ( docolours && has_colors() ) {
		start_color();
		init_pair( BOX_COLOURS,
				colour_array[box_fgind],
				colour_array[box_bgind] );
		init_pair( DISP_COLOURS,
				colour_array[disp_fgind],
				colour_array[disp_bgind] );
#if defined(UNIXWARE2)
		init_pair( REV_DISP_COLOURS,
				colour_array[box_bgind],
				colour_array[box_fgind] );
#endif
		base_bkgd = newwin( 24, 80, 0, 0 );
#if defined(LINUX)
		wbkgdset( base_bkgd, COLOR_PAIR( BOX_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( base_bkgd, COLOR_PAIR( BOX_COLOURS ) | ' ' );
#endif
		werase( base_bkgd );
		wrefresh( base_bkgd );
	}

	/* create and box in the main windows */
	titlebox = newwin( 3, 60, 0, 20 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( titlebox, COLOR_PAIR( BOX_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( titlebox, COLOR_PAIR( BOX_COLOURS ) | ' ' );
#endif
		werase( titlebox );
		wattron( titlebox, COLOR_PAIR( BOX_COLOURS ));
	}
	else
		wattron( titlebox, A_REVERSE );
	box( titlebox, (unsigned char)0, (unsigned char)0 );
	wrefresh( titlebox );

	titledisp = newwin( 1, 58, 1, 21 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( titledisp, COLOR_PAIR( DISP_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( titledisp, COLOR_PAIR( DISP_COLOURS ) | ' ' );
#endif
		wattron( titledisp, COLOR_PAIR( DISP_COLOURS ));
		werase( titledisp );
	}
	mvwaddstr( titledisp, 0, ( 58 - strlen( TITLE )) / 2, TITLE );
	wrefresh( titledisp );

	modebox = newwin( NUM_MODES + 2, 20, 0, 0 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( modebox, COLOR_PAIR( BOX_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( modebox, COLOR_PAIR( BOX_COLOURS ) | ' ' );
#endif
		werase( modebox );
		wattron( modebox, COLOR_PAIR( BOX_COLOURS ));
	}
	else
		wattron( modebox, A_REVERSE );
	box( modebox, (unsigned char)0, (unsigned char)0 );
	mvwaddstr( modebox, 0, 1, "Mode(F10v,F11^)" );
	wrefresh( modebox );

	modedisp = newwin( NUM_MODES, 18, 1, 1 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( modedisp, COLOR_PAIR( DISP_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( modedisp, COLOR_PAIR( DISP_COLOURS ) | ' ' );
#endif
		werase( modedisp );
		wattron( modedisp, COLOR_PAIR( DISP_COLOURS ));
	}
	for ( index = 0; index < NUM_MODES; index++ ) {
		if ( index == currmode )
#if defined(UNIXWARE2)
			wattron( modedisp, COLOR_PAIR( REV_DISP_COLOURS ));
#else
			wattron( modedisp, A_REVERSE );
#endif
		else
#if defined(UNIXWARE2)
			wattron( modedisp, COLOR_PAIR( DISP_COLOURS ));
#else
			wattroff( modedisp, A_REVERSE );
#endif
		mvwaddstr( modedisp, index, 0, modes[index] );
	}
	wrefresh( modedisp );

	charbox = newwin( 18, 60, 3, 20 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( charbox, COLOR_PAIR( BOX_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( charbox, COLOR_PAIR( BOX_COLOURS ) | ' ' );
#endif
		werase( charbox );
		wattron( charbox, COLOR_PAIR( BOX_COLOURS ));
	}
	else
		wattron( charbox, A_REVERSE );
	box( charbox, (unsigned char)0, (unsigned char)0 );
	mvwaddstr( charbox, 0, 1, "Start(F9)" );
	wrefresh( charbox );

	chardisp = newwin( 16, 58, 4, 21 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( chardisp, COLOR_PAIR( DISP_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( chardisp, COLOR_PAIR( DISP_COLOURS ) | ' ' );
#endif
		werase( chardisp );
		wattron( chardisp, COLOR_PAIR( DISP_COLOURS ));
	}
	idlok( chardisp, TRUE );
	immedok( chardisp, TRUE );
	scrollok( chardisp, TRUE );
	wrefresh( chardisp );

	/* create the control feedback boxes */
	wpmbox = newwin( 3, 20, 21, 0 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( wpmbox, COLOR_PAIR( BOX_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( wpmbox, COLOR_PAIR( BOX_COLOURS ) | ' ' );
#endif
		werase( wpmbox );
		wattron( wpmbox, COLOR_PAIR( BOX_COLOURS ));
	}
	else
		wattron( wpmbox, A_REVERSE );
	box( wpmbox, (unsigned char)0, (unsigned char)0 );
	mvwaddstr( wpmbox, 0, 1, "Speed(F1-,F2+)" );
	wrefresh( wpmbox );

	wpmdisp = newwin( 1, 18, 22, 1 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( wpmdisp, COLOR_PAIR( DISP_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( wpmdisp, COLOR_PAIR( DISP_COLOURS ) | ' ' );
#endif
		werase( wpmdisp );
		wattron( wpmdisp, COLOR_PAIR( DISP_COLOURS ));
	}
	sprintf( str, "%2d WPM", cw_wpm );
	mvwaddstr( wpmdisp, 0, 6, str );
	wrefresh( wpmdisp );

	hzbox = newwin( 3, 20, 21, 20 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( hzbox, COLOR_PAIR( BOX_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( hzbox, COLOR_PAIR( BOX_COLOURS ) | ' ' );
#endif
		werase( hzbox );
		wattron( hzbox, COLOR_PAIR( BOX_COLOURS ));
	}
	else
		wattron( hzbox, A_REVERSE );
	box( hzbox, (unsigned char)0, (unsigned char)0 );
	mvwaddstr( hzbox, 0, 1, "Tone(F3-,F4+)" );
	wrefresh( hzbox );

	hzdisp = newwin( 1, 18, 22, 21 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( hzdisp, COLOR_PAIR( DISP_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( hzdisp, COLOR_PAIR( DISP_COLOURS ) | ' ' );
#endif
		werase( hzdisp );
		wattron( hzdisp, COLOR_PAIR( DISP_COLOURS ));
	}
	sprintf( str, "%4d Hz", cw_hz );
	mvwaddstr( hzdisp, 0, 4, str );
	wrefresh( hzdisp );

	gapbox = newwin( 3, 20, 21, 40 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( gapbox, COLOR_PAIR( BOX_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( gapbox, COLOR_PAIR( BOX_COLOURS ) | ' ' );
#endif
		werase( gapbox );
		wattron( gapbox, COLOR_PAIR( BOX_COLOURS ));
	}
	else
		wattron( gapbox, A_REVERSE );
	box( gapbox, (unsigned char)0, (unsigned char)0 );
	mvwaddstr( gapbox, 0, 1, "Spacing(F5-,F6+)" );
	wrefresh( gapbox );

	gapdisp = newwin( 1, 18, 22, 41 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( gapdisp, COLOR_PAIR( DISP_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( gapdisp, COLOR_PAIR( DISP_COLOURS ) | ' ' );
#endif
		werase( gapdisp );
		wattron( gapdisp, COLOR_PAIR( DISP_COLOURS ));
	}
	sprintf( str, "%2d dot(s)", cw_gap );
	mvwaddstr( gapdisp, 0, 4, str );
	wrefresh( gapdisp );

	timebox = newwin( 3, 20, 21, 60 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( timebox, COLOR_PAIR( BOX_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( timebox, COLOR_PAIR( BOX_COLOURS ) | ' ' );
#endif
		werase( timebox );
		wattron( timebox, COLOR_PAIR( BOX_COLOURS ));
	}
	else
		wattron( timebox, A_REVERSE );
	box( timebox, (unsigned char)0, (unsigned char)0 );
	mvwaddstr( timebox, 0, 1, "Time(F7-,F8+)" );
	wrefresh( timebox );

	timedisp = newwin( 1, 18, 22, 61 );
	if ( docolours && has_colors() ) {
#if defined(LINUX)
		wbkgdset( timedisp, COLOR_PAIR( DISP_COLOURS ) );
#endif
#if defined(SCO) || defined(UNIXWARE2) || defined(UNIXWARE7)
		wbkgdset( timedisp, COLOR_PAIR( DISP_COLOURS ) | ' ' );
#endif
		werase( timedisp );
		wattron( timedisp, COLOR_PAIR( DISP_COLOURS ));
	}
	sprintf( str, "%3d min(s)", ptime );
	mvwaddstr( timedisp, 0, 4, str );
	wrefresh( timedisp );
}


/*
 * ui_destroy - dismantle the user interface, boxes and windows
 */
static void ui_destroy() {

	/* dismember the interface if it is active */
	if ( in_windows ) {

		/* clear the screen for neatness */
		werase( scr );
		wrefresh( scr );

		/* end curses processing - anything else goes as standard */
		endwin(); in_windows = FALSE;
	}
}


/*
 * send_echoed - send a character, and when the ender echoes it back,
 *               display it in the output window.  Also, check for
 *               any changes of speed, tone etc while sending the char
 *               if check_ui is specified
 */
static void send_echoed( char c, bool check_ui, bool echo_ui ) {
	static int	s_cw_wpm = -1;		/* look for changed wpm */
	static int	s_cw_hz = -1;		/* look for changed hz */
	static int	s_cw_gap = -1;		/* look for changed gap */
	static int	s_cw_adj = -1000;	/* look for changed adj */
	static int	s_cw_msgs = -1;		/* look for changed msgs */
	static int	s_cw_cmds = -1;		/* look for changed cmds */
	static int	s_cw_echo = -1;		/* look for changed echo */
	static int	s_cw_combo = -1;	/* look for changed combo */
	char	str[MBUF_LEN];			/* string buffer - general */
	char	w;				/* char echoed from sender */
	fd_set	rfds;				/* select fd set */
	struct timeval	tv;			/* select timer value */
	int		echo_count = 0;		/* count of echo chars seen */

	/* see if any parameter has changed since the last call, and if
	   it has, of if a complete resync is requested, send the command
	   to update the CW sender process */
	if ( cw_resync || cw_msgs != s_cw_msgs ) {
		sprintf( str, "%c%c%d%c", CW_CMD_ESCAPE, CW_CMDV_MSGS,
			cw_msgs, CW_CMD_END );
		write( infd, str, strlen( str ));
		s_cw_msgs = cw_msgs;
	}
	if ( cw_resync || cw_cmds != s_cw_cmds ) {
		sprintf( str, "%c%c%d%c", CW_CMD_ESCAPE, CW_CMDV_CMDS,
			cw_cmds, CW_CMD_END );
		write( infd, str, strlen( str ));
		s_cw_cmds = cw_cmds;
	}
	if ( cw_resync || cw_combo != s_cw_combo ) {
		sprintf( str, "%c%c%d%c", CW_CMD_ESCAPE, CW_CMDV_COMBO,
			cw_combo, CW_CMD_END );
		write( infd, str, strlen( str ));
		s_cw_combo = cw_combo;
	}
	if ( cw_resync || cw_echo != s_cw_echo ) {
		sprintf( str, "%c%c%d%c", CW_CMD_ESCAPE, CW_CMDV_ECHO,
			cw_echo, CW_CMD_END );
		write( infd, str, strlen( str ));
		s_cw_echo = cw_echo;
	}
	if ( cw_resync || cw_wpm != s_cw_wpm ) {
		sprintf( str, "%c%c%d%c", CW_CMD_ESCAPE, CW_CMDV_WPM,
			cw_wpm, CW_CMD_END );
		write( infd, str, strlen( str ));
		s_cw_wpm = cw_wpm;
	}
	if ( cw_resync || cw_hz != s_cw_hz ) {
		sprintf( str, "%c%c%d%c", CW_CMD_ESCAPE, CW_CMDV_TONE,
			cw_hz, CW_CMD_END );
		write( infd, str, strlen( str ));
		s_cw_hz = cw_hz;
	}
	if ( cw_resync || cw_gap != s_cw_gap ) {
		sprintf( str, "%c%c%d%c", CW_CMD_ESCAPE, CW_CMDV_GAP,
			cw_gap, CW_CMD_END );
		write( infd, str, strlen( str ));
		s_cw_gap = cw_gap;
	}
	if ( cw_resync || cw_adj != s_cw_adj ) {
		sprintf( str, "%c%c%d%c", CW_CMD_ESCAPE, CW_CMDV_ADJ,
			cw_adj, CW_CMD_END );
		write( infd, str, strlen( str ));
		s_cw_adj = cw_adj;
	}
	cw_resync = FALSE;

	/* send the character, and wait for its echo, processing any
	   user interface things that arrive in the interim, so... */

	/* write the character */
	if ( write( infd, &c, 1 ) != 1 ) {
		fprintf( stderr, "Error writing to CW sender\n" );
		perror( "" ); exit( 1 );
	}

	/* OK, this bit's tricky.  Firstly, we want to read characters
	   back from the CW sender until there's no more to read.
	   Secondly, we want to keep checking to user commands in the
	   meantime.  Select() would be nice, but it seems to mess
	   up curses.  So... */

	/* loop until complete echo received from CW sender */
	echo_count = 0;
	while ( TRUE ) {

		/* set up a select call to timeout after a few mS */
		FD_ZERO( &rfds );
		FD_SET( outfd, &rfds );
		tv.tv_sec = 0;
		tv.tv_usec = UI_POLL;

		/* see if the echo has arrived yet */
		if ( ! select( outfd + 1, &rfds, NULL, NULL, &tv )) {

			/* no echo yet - look for user interface
			   actions, but don't wait */
			if ( check_ui )
				ui_interpret( ui_getch( FALSE ) );

			/* if this timeout happens after some echo has
			   been seen, consider the pipe drained */
			if ( echo_count > 0 )
				break;
		}
		else {

			/* something available from the sender */
			if( read( outfd, &w, 1 ) != 1 ) {
				fprintf( stderr, "Error reading CW sender\n" );
				perror( "" ); exit( 1 );
			}
			if ( echo_ui )
				waddch( chardisp, w );

			/* increment the count of echoed chars */
			echo_count++;
		}
	}

	/* update display with the echo read back */
	if ( echo_ui )
		wrefresh( chardisp );

}


/*
 * warble - make a nice tone to indicate the start or end of something
 */
static void warble() {
	int	s_cw_wpm, s_cw_hz;		/* preserved cw values */

	/* save the current sender parameters */
	s_cw_wpm = cw_wpm; s_cw_hz = cw_hz;

	/* create a warbly sound (ab)using the cw sender program */
	cw_wpm = WARBLE_WPM;
	send_echoed( CW_COMBO_START, FALSE, FALSE );
	cw_hz = WARBLE_TONE1; send_echoed( WARBLE_CHAR, FALSE, FALSE );
	cw_hz = WARBLE_TONE2; send_echoed( WARBLE_CHAR, FALSE, FALSE );
	cw_hz = WARBLE_TONE1; send_echoed( WARBLE_CHAR, FALSE, FALSE );
	cw_hz = WARBLE_TONE2; send_echoed( WARBLE_CHAR, FALSE, FALSE );
	send_echoed( CW_COMBO_END, FALSE, FALSE );

	/* restore the things we messed with */
	cw_wpm = s_cw_wpm; cw_hz = s_cw_hz;
}


/*
 * generate_chars - generate and send random chars from the specified set
 */
static void generate_chars( char *charset, bool do_groups ) {
	int	charset_size;			/* elements in charset strng */
	char	*sptr;				/* string pointer */
	int	rand_elenum;			/* random element number */
	int	sep_count;			/* count of separators read */
	int	ele_count = 0;			/* elements sent this call */
	time_t	check_time;			/* timer to check against */
	char	str[MBUF_LEN];			/* string buffer - general */

	/* put up an indicator that we are sending */
	mvwaddstr( charbox, 0, 1, "Sending(F9 or Esc to exit)" );
	wnoutrefresh( charbox );

	/* clear the display window */
	werase( chardisp ); wmove( chardisp, 0, 0 );
	wnoutrefresh( chardisp );
	doupdate();

	/* initialize the random number generation */
	srand( time( NULL ) );

	/* find the number of elements in the charset string; the | character
	   is the separator */
	for ( sptr = charset, charset_size = 0; *sptr != EOS; sptr++ ) {
		if ( *sptr == ASC_ELES ) charset_size++;
	}
	charset_size++;			/* always one more than num of |s */

	/* cancel the despatcher when no longer sending or time expired */
	check_time = time( NULL );
	while ( TRUE ) {

		/* decide which item to send */
		rand_elenum = rand() % charset_size;

		/* read sptr up to that element in the charset string */
		for ( sptr = charset, sep_count = 0;
				( sep_count < rand_elenum ) && *sptr != EOS;
				sptr++ ) {
			if ( *sptr == ASC_ELES ) sep_count++;
		}

		/* now send characters in that element, ignoring tabs
		   as these are for formatting only */
		for ( ; *sptr != ASC_ELES && *sptr != EOS; sptr++ ) {
			if ( *sptr != ASC_TAB )
				send_echoed( *sptr, TRUE, TRUE );

			/* update practice timer, or terminate if time's up */
			if ( time( NULL ) - check_time >= 60 ) {

				/* if ptime down to 1 the last min elapsed */
				if ( ptime <= 1 )
					goto loop_end;

				/* decrement the practice timer one minute */
				ptime--;
				sprintf( str, "%3d min(s)", ptime );
				mvwaddstr( timedisp, 0, 4, str );
				wrefresh( timedisp );

				/* reset our check timer value */
				check_time = time( NULL );
			}

			/* check for despatcher termination condn */
			if ( sending_state != STATE_ACTIVE )
				goto loop_end;
		}

		/* increment the count of charset elements sent */
		ele_count++;

		/* if grouping, send separator after groupsize elements */
		if ( do_groups && ( ele_count % GROUPSIZE == 0) ) {
			send_echoed( ASC_SPACE, TRUE, TRUE );
		}

		/* check again for despatcher termination condn */
		if ( sending_state != STATE_ACTIVE )
			goto loop_end;
	}
loop_end:

	/* rebuild the box we messed up with the sending indicator */
	box( charbox, (unsigned char)0, (unsigned char)0 );
	mvwaddstr( charbox, 0, 1, "Start(F9)" );
	wnoutrefresh( charbox );
	touchwin( chardisp ); wnoutrefresh( chardisp );
	doupdate();
}


/*
 * generate_kb - read characters from the keyboard and send them
 */
static void generate_kb() {
	int	c;				/* input character */
	char	*sptr;				/* string pointer */
	char	buf[TRANS_BUFLEN];		/* translation buffer */

	/* put up an indicator that we are taking kb input */
	mvwaddstr( charbox, 0, 1, "Keyboard input(F9 or Esc to exit)" );
	wnoutrefresh( charbox );

	/* clear the display window */
	werase( chardisp ); wmove( chardisp, 0, 0 );
	wnoutrefresh( chardisp );
	doupdate();

	/* cancel the despatcher when no longer sending */
	while ( TRUE ) {

		/* get a character from the keyboard, but delay
		   converting to uppercase until it's been checked
		   for sendability and ui commands (especially F8 stop) */
		keypad( scr, TRUE );
		noecho(); cbreak(); curs_set( 0 ); raw();
		nodelay( scr, FALSE );
		c = getch();

		/* check that character against the list of OK ones;
		   if char not found then assume that it's a user
		   command and pass it to the ui processing (it may
		   of course not be) */
		if ( c > UCHAR_MAX ||
				strchr( OK_KB_CHARS, toupper( c ) ) == NULL ) {

			/* c isn't a straight ASCII character, and it isn't
			   in the list of things we could send, so have
			   a stab at interpreting it as a command */
			ui_interpret( c );

		}

		else {
			/* char found - safe to convert to uppercase,
			   then translate or send verbatim */
			c = toupper( c );
			switch ( c ) {
			case '%':	strcpy( buf, "[VA]" ); break;
			case '!':	strcpy( buf, "[BK]" ); break;
			case '@':	strcpy( buf, "[AR]" ); break;
			case '^':	strcpy( buf, "[AA]" ); break;
			case '*':	strcpy( buf, "[AS]" ); break;
			case '&':	strcpy( buf, "ES" ); break;
			case '#':	strcpy( buf, "[KN]" ); break;
			default:	buf[0] = c; buf[1] = EOS; break;
			}

			/* send buffer, and don't check ui when echoing */
			for ( sptr = buf; *sptr != EOS; sptr++ ) {
				send_echoed( *sptr, FALSE, TRUE );

				/* note - don't check termination here, since
				   this might stop in mid [..] group */
			}
		}

		/* check for despatcher termination conditions */
		if ( sending_state != STATE_ACTIVE )
			goto loop_end;
	}
loop_end:

	/* rebuild the box we messed up with the kb input indicator */
	box( charbox, (unsigned char)0, (unsigned char)0 );
	mvwaddstr( charbox, 0, 1, "Start(F9)" );
	wnoutrefresh( charbox );
	touchwin( chardisp ); wnoutrefresh( chardisp );
	doupdate();
}


/*
 * catcher - signal handler for signals, to clear up on kill
 */
static void catcher( int sig ) {

	/* attempt to wrestle the screen back from curses */
	ui_destroy();

	/* show the signal caught and exit */
	fprintf( stderr, "\nCaught signal %d, exiting...\n", sig );
	prog_exit( 1 );
}


/*
 * ui_despatcher - start various flavours of character generators
 */
static void ui_despatcher() {

	/* despatch character generators forever */
	while( TRUE ) {

		/* wait for a command */
		ui_interpret( ui_getch( TRUE ) );

		/* if sending requested, despatch the relevant generator */
		if ( sending_state == STATE_ACTIVE ) {
			switch ( currmode ) {
			case MODE_LETTERS:
				generate_chars( LETTERS_CHARSET, TRUE );
				break;
			case MODE_NUMBERS:
				generate_chars( NUMBERS_CHARSET, TRUE );
				break;
			case MODE_ALPHANUM:
				generate_chars( ALPHANUM_CHARSET, TRUE );
				break;
			case MODE_ALLCHAR:
				generate_chars( ALLCHAR_CHARSET, TRUE );
				break;
			case MODE_WORDS:
				generate_chars( WORDS_CHARSET, FALSE );
				break;
			case MODE_CWWORDS:
				generate_chars( CWWORDS_CHARSET, FALSE );
				break;
			case MODE_KEYBOARD:
				generate_kb();
				break;
			case MODE_PARIS:
				generate_chars( PARIS_CHARSET, FALSE );
				break;
			case MODE_ESIH5:
				generate_chars( EISH5_CHARSET, TRUE );
				break;
			case MODE_TMO0:
				generate_chars( TMO0_CHARSET, TRUE );
				break;
			case MODE_AUV4:
				generate_chars( AUV4_CHARSET, TRUE );
				break;
			case MODE_NDB6:
				generate_chars( NDB6_CHARSET, TRUE );
				break;
			case MODE_KXffRP:
				generate_chars( KXffRP_CHARSET, TRUE );
				break;
			case MODE_FLYQC:
				generate_chars( FLYQC_CHARSET, TRUE );
				break;
			case MODE_WJ1GZ:
				generate_chars( WJ1GZ_CHARSET, TRUE );
				break;
			case MODE_23789:
				generate_chars( x23789_CHARSET, TRUE );
				break;
			case MODE_ffffff:
				generate_chars( ffffff_CHARSET, TRUE );
				break;
			case MODE_QUIT:
				prog_exit( 0 );
				break;
			}
			sending_state = STATE_IDLE;
			warble();
		}
	}
}


/*
 * main - fork the child cw sender and send it characters
 */
int main( int argc, char **argv ) {
	char	*argv0;			/* program name */
	char	*argp;			/* argument string pointer */
	int	acount;			/* argument string counter */
	char	*exec_argv[MAXARGS];	/* args to execvp */
	char	cp_cwcommand[MAXPATHLEN];/* working copy of cwcommand */
	int	infiledes[2];		/* CW input file pipe */
	int	outfiledes[2];		/* CW output file pipe */

	/* parse the command line arguments */
	parse_cmdline( argc, argv );

	/* set argv0 to be the basename part of the program name */
	argv0 = argv[0] + strlen( argv[0] );
	while ( *argv0 != ASC_FNS && argv0 > argv[0] )
		argv0--;
	if ( *argv0 == ASC_FNS ) argv0++;

	/* create pipes for the cw stdin and stdout streams */
	if ( pipe( infiledes ) != 0 ) {
		fprintf( stderr, "%s: can't create input pipe\n", argv0 );
		perror( "" ); exit( 1 );
	}
	if ( pipe( outfiledes ) != 0 ) {
		fprintf( stderr, "%s: can't create output pipe\n", argv0 );
		perror( "" ); exit( 1 );
	}

	/* build the array of arguments from cwcommand string */
	acount = 0;
	strcpy( cp_cwcommand, cwcommand );
	argp = cp_cwcommand;
	while ( acount < MAXARGS - 1 ) {
		for ( ; *argp == ASC_SPACE; argp++ );
		if ( *argp == EOS )
			break;
		else {
			exec_argv[ acount++ ] = argp;
			for ( ; *argp != ASC_SPACE && *argp != EOS; argp++ );
			if ( *argp == ASC_SPACE ) {
				*argp = EOS; argp++;
			}
		}
	}
	exec_argv[ acount++ ] = NULL;

	/* fork in order to exec cw sounder binary */
	if ( (pid = fork()) == -1 ) {
		fprintf( stderr, "%s: fork failed", argv0 );
		perror( "" ); exit( 1 );
	}

	/* parent processing - handle the user interface and despatch
	   the required character generators */
	else if ( pid != 0 ) {

		/* install handler for signals */
		signal( SIGHUP, catcher );
		signal( SIGINT, catcher );
		signal( SIGQUIT, catcher );
		signal( SIGCLD, catcher );
		signal( SIGPIPE, catcher );
		signal( SIGTERM, catcher );

		/* wait for a short period in case the child exec fails,
		   in which case we get a sigcld */
		sleep( HANG_ABOUT );

		/* sort out the pipe file handles we use */
		infd = infiledes[ 1 ];		/* write cmds to CW */
		outfd = outfiledes[ 0 ];	/* read echo from CW */

		/* initialize the curses user interface */
		ui_init();

		/* despatch character generators forever - this routine
		   never returns */
		ui_despatcher();
	}

	/* child processing - reset the notion of stdin and stdout/stderr
	   to the fifos, and exec the sounder */
	else {

		/* complete detach into a new process group, to isolate
		   us from the parent's signal handling */
		if ( setpgrp() == -1 ) {
			perror( "setpgrp" ); exit( 1 );
		}

		/* redirect the stdin and out, leave err */
		close( 0 ); close( 1 );
		dup( infiledes[ 0 ] ); dup( outfiledes[ 1 ] );

		/* exec the sounder */
		execvp( exec_argv[0], exec_argv );
		fprintf( stderr, "%s: can't exec sender program\n", argv0 );
		perror( exec_argv[0] ); exit( 1 );
	}

	return( 0 ); /* for LINT */
}
