/*
 * afhist.c
 *
 *  afhist supports the history function of the afos monitor.  It logs each
 *       afos product type seen along with the quantity of each.
 *
 *	$Id: afhist.c,v 1.8 1996/08/31 05:02:45 ron Exp $
 *
 */
#include "afos.h"
#include <unistd.h>
#include <stdio.h>

AF_HIST *history_list ;
static char args[8][80] ;
static void add_chan_to_list ( AF_CHANNEL_LINK **list, AF_CHANNEL *chan );

afhist_init()
{
	history_list = NULL ;
}

string_compare ( char * s1, int size1, char *s2, int size2 )
{
	int len, temp ;

	len = ( ( size1 <= size2 ) ? size1 : size2 ) ;
	temp = strncmp (s1, s2, len ) ;

	if ( temp == 0 )
	{
		if ( size1 < size2 )
			return -1 ;
		else if ( size1 == size2 )
			return 0 ;
		else
			return 1 ;
	}
	else
		return temp ;
}

/*
	find_prev finds the node that should be previous to the specified string

	returns a pointer to the node that should be previous or is identical to the
	node specified by string 's', or NULL if node specified 
	by string 's' should be the start of the list.
*/
AF_HIST *
find_prev(char *s, int size)
{
	AF_HIST *p, *last_p;

	if ( !history_list )
		return NULL ;

	last_p = NULL ;
	for ( p = history_list ; 
		  ( p != NULL && 
			   ( string_compare ( p->name, strlen(p->name), s, size ) <= 0 ) ) ;
		  p = p->next )
	{
		last_p = p ;
	}

	return last_p ;
}

/*
	check_hist looks for an existing node matching the string 's' of size 'size'

	Returns a pointer to the matching node, or NULL if no match exists
*/
AF_HIST *
check_hist ( char *s, int size )
{
	AF_HIST *p;

	p = find_prev(s,size);

	/* If NULL, there is nothing to match */
	if ( !p )
	{
		return NULL ;
	}

	/* If the same length AND the same string, return pointer to this node */
	if ( string_compare ( s, size, p->name, strlen(p->name) )  == 0 )
	{
		return p ;
	}

	/* Else, return NULL since no match exists */
	else
	{
		return NULL ;
	}
}

/*
	add_hist adds the node pointed to by 'p' to the history list
*/
void
add_hist(AF_HIST *p)
{
	AF_HIST *q;

	if (history_list == NULL)
	{
		history_list = p ;
		p->next = NULL ;
		return ;
	}
	else
	{
		q = find_prev(p->name, strlen(p->name));

		/* if find_prev returns NULL, the new node should become the start
		   of the list */
		if (!q)
		{
			p->next = history_list ;
			history_list = p ;
			return ;
		}


		/* Check for duplicate node */
		if ( ( strlen (p->name) == strlen(q->name) )  && 
			 (strcmp (p->name, q->name) == 0 ) )
		{
			write (2, 
			"\r\nhistory error: trying to add a duplicate node\r\n",49);
			/* MEMORY LEAK!  If this error occurs, node p is lost */
			return  ;
		}

		/* Add this node to the list after previous node 'q' */
		p->next = q->next ;
		q->next = p ;
		return ;
	}
}
	
/*
	adhistlog logs the occurance of an AFOS product with the name 'prod'.

	adhistlog returns a pointer to the node that describes the product.

*/
AF_HIST *
afhistlog( char *prod, int size )
{
	AF_HIST *p ;
	int i ;

	p = check_hist (prod, size) ;
	if ( p )
	{
		p->count++;
		return p ;
	}
	else
	{
		p = (AF_HIST *)malloc(sizeof(*p));
		if (!p)
		{
			write(2,"unable to allocate memory space for history\r\n",45);
			exit (-1);
		}
		strncpy( p->name, prod, AFHIST_PRODUCT_SIZE ) ;

		if ( size > AFHIST_PRODUCT_SIZE )
			p->name[AFHIST_PRODUCT_SIZE] = 0 ;
		else
			p->name[size] = 0;

		p->count = 1 ;
		p->chanlist = NULL ;
	}

	add_hist(p);
	return p ;
}

/*
	afhist_display is a primitive debug tool to display AFOS product summary
	information
*/
void
afhist_display( void )
{
	AF_HIST *p ;

	fprintf(stderr,"\r\nAFOS product summary and tally");
	fprintf(stderr,"\r\n------------------------------\r\n");
	for ( p = history_list ; p ; p = p->next )
		fprintf (stderr,"%-20s:%4d\r\n", p->name, p->count );
	fprintf(stderr,"\r\n\r\n");
}

void 
process_filter_buffer( unsigned char *s, int size )
{
	AF_HIST *p;
	AF_CHANNEL_LINK *l ;

	/* 
		call afhistlog to log the occurance and find the node that
	    describes this product
	*/
	p = afhistlog ( &s[5], 9 ) ;

	/*
		run down the channel list for this product and send this
		product to those channels
	*/
	/* REL: how much of the product is sent ?  With ZCZC and NNNN ? */
	for ( l = p->chanlist ; l ; l = l->next )
		l->channel->chanwrite ( l->channel, s, size ) ;
}

static void
afhist_add_config_entry ( char *product, int size, AF_CHANNEL_LINK *link )
{
	AF_HIST *p ;
	AF_CHANNEL_LINK *l ;
	int i ;

	p = check_hist (product, size) ;
	if ( p )
	{
		/* history entry for this product already exists */
		if (!p->chanlist )
		{
			/* EMPTY Channel list, just put new list on it */
			p->chanlist = link ;
		}
		else
		{
			/* Find the end of the existing list */
			for (l = p->chanlist ; l->next ; l = l->next )
				{ }
			/* Add the new link to the end of the existing list */
			l->next = link ;
		}
		fprintf ( stderr, 
		"AFOSD: Filter file has multiple entries for PIL '%*s'\n",size,product) ;
		return ;
	}
	else
	{
		p = (AF_HIST *)malloc(sizeof(*p));
		if (!p)
		{
			write(2,"unable to allocate memory space for history\r\n",45);
			exit (-1);
		}
		strncpy( p->name, product, AFHIST_PRODUCT_SIZE ) ;

		if ( size > AFHIST_PRODUCT_SIZE )
			p->name[AFHIST_PRODUCT_SIZE] = 0 ;
		else
			p->name[size] = 0;

		p->count = 0 ;
		p->chanlist = link ;
	}
	add_hist(p);
	return ;
}


int
afconfigfilter(char *filename)
{
	FILE *filterfp ;
	int count ;
	unsigned char line[128] ;
	AF_CHANNEL *chan ;
	AF_CHANNEL_LINK *link ;
	int i ;

	/* Open the filter configuration file */
	filterfp = fopen ( filename, "r" ) ;
	if ( filterfp == NULL )
	{
		fprintf(stderr, "Unable to open port configuration file '%s'\n",
			filename );
		exit ( -1 ) ;
	}

	while ( fgets (line, 256, filterfp) )
	{
		count = sscanf ( line, "%[^ 	\n]%*[ 	,]"
							   	"%[^ 	\n]%*[ 	,]"
							   	"%[^ 	\n]%*[ 	,]"
							   	"%[^ 	\n]%*[ 	,]"
							   	"%[^ 	\n]%*[ 	,]"
							   	"%[^ 	\n]%*[ 	,]"
							   	"%[^ 	\n]%*[ 	,]"
							   	"%[^ 	\n]%*[ 	,]",
			args[0], args[1], args[2], args[3], 
			args[4], args[5], args[6], args[7]) ;

		if (count > 1)
		{
			if ( args[0][0] == '#' )
				continue ;
			
			link = NULL ;
			for ( i = 1 ; i < count ; i++ )
			{
				chan = find_channel_by_name ( args[i] ) ;
				if (chan)
					add_chan_to_list ( &link, chan ) ;
				else
					fprintf(stderr, 
		"AFOSD Warning: Channel '%s', mentioned in filter file, is undefined\n",
					args[i] );
			}
			/*
				add the product history entry
			*/
			afhist_add_config_entry ( args[0], strlen(args[0]), link ) ;
		}
	}
}

AF_CHANNEL *
find_channel_by_name ( char *name )
{
	AF_CHANNEL *p ;
	/*
		Find the channel object the matches this name
	*/
	for ( p = channel_list ; p ; p = p->next )
	{
		if ( strcmp ( p->name, name ) == 0 )
			break ;
	}

	if (!p)
		return NULL ;
	else 
		return p ;
}

static void
add_chan_to_list ( AF_CHANNEL_LINK **list, AF_CHANNEL *chan )
{
	AF_CHANNEL_LINK *p ;

	p = (AF_CHANNEL_LINK *)malloc(sizeof(AF_CHANNEL_LINK)) ;
	if (!p)
	{
		fprintf(stderr, 
		"AFOSD: unable to allocate memory for filter description\n");
		exit(-1);
	}

	p->channel = chan ;
	p->next = *list ;
	*list = p;
}
