/*
 * afproc.c
 *
 * This module supports process channels
 *
 * $Id: afproc.c,v 1.10 1998/07/29 00:09:03 ron Exp $
 *
 *
 */
#include <signal.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "afos.h"

int proc_init ( AF_CHANNEL *chan, void (*read_callback)() ) ;
int proc_write ( AF_CHANNEL *chan, unsigned char *dat, int size ) ;
int proc_getchar ( AF_CHANNEL *chan ) ;
int argify (char *s, char **argv) ;
static struct sigaction child_sigstuff ;

typedef struct _PROC_STRUCT_ {
	char process_name[AFCHAN_NAME_SIZE];
	char process_options[512];
	char *argv[20] ;
	} PROC_STRUCT ;

/*
	parse_proc parse out the 'port' line in the config file
*/
int
parse_process( char *line )
{
	static char args[8][80] ;
	AF_CHANNEL *chan ;
	PROC_STRUCT *mystuff ;
	int count ;

	count = sscanf ( line,  "%[^ 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^\n]",
		args[0], args[1], args[2], args[3] )  ;

	if ( count < 3 )
	{
		fprintf ( stderr, "\r\nAFOSD: Unable to parse the following line:\n%s\n"
			"Insufficient parameters for process definition\n"
			"Usage:\n"
			"process <CHANNEL NAME> <APPLICATION NAME> [OPTIONS]\n", line);
		exit (-1);
	}

	if (!verify_unique_portname( args[1] ) )
	{
		fprintf(stderr,"\r\nERROR in port configuration file on LINE:\n%s\n"
					   "Port '%s' already defined\n",
						line, args[1] ) ;
		exit (-1);
	}

	chan = (AF_CHANNEL *)malloc( sizeof(*chan) ) ;
	if (!chan)
	{
		fprintf(stderr, "Unable to allocate new channel definition space\n");
		exit (-1);
	}
	
	/* Put new channel on the channel list */
	chan->next = channel_list ;
	channel_list = chan ;
	strncpy ( chan->name, args[1], AFCHAN_NAME_SIZE ) ;

	/* Allocate our private data structure and fill it out with our
	   configuration info */
	mystuff = (PROC_STRUCT *)malloc( sizeof(*mystuff) ) ;
	if (!mystuff)
	{
		fprintf(stderr, "Unable to allocate new channel definition space\n");
		exit (-1);
	}
	strncpy ( mystuff->process_name, args[2], AFCHAN_NAME_SIZE ) ;
	if ( count > 3 )
	{
		strncpy ( mystuff->process_options, args[3], 512 ) ;
		mystuff->argv[0] = mystuff->process_name ;
		argify (mystuff->process_options, mystuff->argv);
	}
	chan->private = mystuff ;

	/* Fill out the rest of the channel info */
	chan->fd = -1 ;
	chan->init = proc_init ;
	chan->chanwrite = proc_write ;
	chan->changetchar = proc_getchar ;
#if 1
	printf( "config line: '%s'\n"
			"channel name: '%s'\n"
			"process name: '%s'\n"
			"process options:'%s'\n",
			line, chan->name, mystuff->process_name, mystuff->process_options);
#endif
}



static void 
child_signal_handler ( int sig )
{
	/* 
		How do I figure out which pipe is causing me grief ?
	*/
	fprintf(stderr,"\nbroken pipe signal handler.\n");
	waitpid ( 0, NULL, WNOHANG ) ;
}

/*
	proc_init initializes a serial port
*/
int
proc_init ( AF_CHANNEL *chan, void (*read_callback)() )
{
	/* Check to see if previously initialized */
	if ( chan->fd != -1 )
	{
		/* Already initialized, just return */
		return 0 ;
	}

	/*
		anything?  Verify application exists and is executable ?
	*/
	
	/* set up signal handler for broken pipes */
	child_sigstuff.sa_handler = child_signal_handler ;
	__sigemptyset(&child_sigstuff.sa_mask);
	child_sigstuff.sa_flags = SA_RESTART ;
	sigaction ( SIGCHLD, &child_sigstuff, NULL ) ;
}

int
argify (char *s, char **argv)
{
	char *p = s ;
	int argc = 1 ;

	p = s ;
	while ( *p ) {
		argv[argc] = p ;
		while ( *p && (*p!='	') && (*p!=' ') ) {
			if (*p == '"') {
				if ( *(p-1) ) {
					/* 
						This is NOT the first character in the arg,
						make the quote go away by squishing it with
						a space
					*/
					*p = ' ' ;
					p++ ;
				}
				else {
					/* 
						If this is the first character in a new argument
						(indicated by the previous character being null),
						remove the quote by setting the pointer for this
						arg to one past the quote
					*/
					p++ ;
					argv[argc] = p ;
				}

				while ( *p && (*p != '"') )
					p++ ;
				if (!*p) {
					fprintf(stderr,"afproc: unmatched quote\n");
					argv[argc] = NULL ;
					return argc;
				}
				*p = 0 ;
			}
			p++ ;
		}
		if (*p)
			*p++ = 0 ;
		argc++ ;
		while (  *p && ( (*p=='	') || (*p==' ') )  )
			p++ ;
	}
	argv[argc] = NULL ;
	return argc ;
}


/*
	proc_write writes a message out the serial port
*/
int
proc_write ( AF_CHANNEL *chan, unsigned char *dat, int size )
{
	int pid ;
	int stat ;
	int i ;
	int pipes[2] ;
	PROC_STRUCT *mystuff = chan->private ;
	
	chan->write_count++ ;
	chan->bytes_written += size ;

	pipe (pipes) ;
	if (pid=fork())
	{
		write( pipes[1], dat, size ) ;
		close (pipes[1]);
	}
	else
	{
		dup2 ( pipes[0], 0 ) ;
		close (pipes[1]);
#if 0
		fprintf(stderr,"afproc: calling '%s' with arguments: ",
			mystuff->process_name ) ;
		for ( i = 0 ; mystuff->argv[i] ; i++ )
			fprintf(stderr,"'%s' ",mystuff->argv[i]);
		fprintf(stderr,"<end>\n");
#endif
		execvp( mystuff->process_name, mystuff->argv ) ;
		exit(0);
	}
	return size ;
}

/*
	proc_getchar reads one character from the process ?.  This call is
	non-blocking and may return a negative value if an error occurs
*/
int
proc_getchar ( AF_CHANNEL *chan )
{
	int i, result;

	i = 0 ;
	result = read ( chan->fd, &i, 1 ) ;

	if ( result <= 0 )
		return -1 ;

	chan->bytes_read++ ;
	return 0xff & i ;
}
