/*
 * aftcp.c
 *
 * This module supports network channels via tcp
 *
 * $Id: aftcp.c,v 1.12 1998/07/29 00:09:03 ron Exp $
 *
 */
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <time.h>
#include <fcntl.h>
#include <signal.h>
#include "afos.h"

int aftcp_chan_init ( AF_CHANNEL *chan, void (*read_callback)() ) ;
int tcp_write ( AF_CHANNEL *chan, unsigned char *dat, int size ) ;
int tcp_getchar ( AF_CHANNEL *chan ) ;
void aftcp_listen_select_handler ( AF_CHANNEL *chan ) ;
void aftcp_read_select_handler( void *private, int event ) ;
int aftcp_connect_select_handler (AF_CHANNEL *chan, int event ) ;
void aftcp_connect_to_client (AF_CHANNEL *chan) ;
void aftcp_timeout_handler ( ) ;

typedef struct _AFTCP_TIME_ELEMENT_ {
	struct _AFTCP_TIME_ELEMENT_ *next;
	AF_CHANNEL *chan ;
	int (* func)(AF_CHANNEL *chan, int event );
	} AFTCP_TIME_ELEMENT ;

typedef struct _TCP_HOST_STRUCT_ {
	struct _TCP_HOST_STRUCT_ *next ;
	char host_name[AFCHAN_NAME_SIZE] ;
	struct in_addr *sin_addr[10] ;
	} TCP_HOST_STRUCT ;

typedef struct _TCP_SESSION_STRUCT_ {
	struct _TCP_SESSION_STRUCT_ *next ;
	AF_CHANNEL *chan ;
	char *host_name ;
	int remote_tcp_port ;
	int fd ;
	} TCP_SESSION_STRUCT ;

typedef struct _TCP_STRUCT_ {
	char host_name[AFCHAN_NAME_SIZE] ;  /* antiquated ?? */
	TCP_HOST_STRUCT *host_list ;
	TCP_SESSION_STRUCT *session_list ;
	int tcp_port_number ;
	int listen_fd ;
	int connect_fd ;
	time_t connect_time ;
	time_t disconnect_time ;
	time_t last_connect_attempt ;
	void (*read_callback)();
	} TCP_STRUCT ;

static AFTCP_TIME_ELEMENT *timer_list ;
static struct sigaction brokenpipe_sigstuff ;

static void 
brokenpipe_handler ( int sig )
{
	/* 
		How do I figure out which pipe is causing me grief ?
	*/
	fprintf(stderr,"\nbroken pipe signal handler.\n");
}

aftcp_init()
{
	timer_list = NULL;
	af_timeout_set_handler ( aftcp_timeout_handler ) ;

	/* set up signal handler for broken pipes */
	brokenpipe_sigstuff.sa_handler = brokenpipe_handler ;
	/* sa_mask has been changed to be a structure */
	/* Ignore /usr/include/asm/signal.h */
	/* Look at /usr/include/signal.h, /usr/include/sigset.h, */
	/* and /usr/include/sigaction.h.   SURPRISE!  */
	__sigemptyset(&brokenpipe_sigstuff.sa_mask);
	brokenpipe_sigstuff.sa_flags = SA_RESTART ;
	sigaction ( SIGPIPE, &brokenpipe_sigstuff, NULL ) ;

}

/*
	parse_tcp parse out the 'tcp' line in the config file
*/
int
parse_tcp( char *line )
{
	static char args[8][80] ;
    struct hostent *hp;
	AF_CHANNEL *chan ;
	TCP_STRUCT *mystuff ;
	TCP_HOST_STRUCT *host_entry ;
	int i ;
	int count ;

	count = sscanf ( line,  "%[^ 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]"
							"%[^, 	\n]%*[ 	,]",
		args[0], args[1], args[2], args[3] ) ;

	if ( count < 3 )
	{
		fprintf ( stderr, "\r\nAFOSD: Unable to parse the following line:\n%s\n"
			"Insufficient parameters for tcp definition\n"
			"Usage:\n"
	"tcp <CHANNEL NAME> <REMOTE HOST> [SOCKET NUMBER]\n",
			line);
		exit (-1);
	}

	if (!verify_unique_portname( args[1] ) )
	{
		fprintf(stderr,"\r\nERROR in port configuration file on LINE:\n%s\n"
					   "Port '%s' already defined\n",
						line, args[1] ) ;
		exit (-1);
	}

	chan = (AF_CHANNEL *)malloc( sizeof(*chan) ) ;
	if (!chan)
	{
		fprintf(stderr, "Unable to allocate new channel definition space\n");
		exit (-1);
	}
	
	/* Put new channel on the channel list */
	chan->next = channel_list ;
	channel_list = chan ;
	strncpy ( chan->name, args[1], AFCHAN_NAME_SIZE ) ;

	/* Allocate our private data structure and fill it out with our
	   configuration info */
	mystuff = (TCP_STRUCT *)malloc( sizeof(*mystuff) ) ;
	if (!mystuff)
	{
		fprintf(stderr, "Unable to allocate new channel definition space\n");
		exit (-1);
	}

	/* 
		Create hosts list 
	*/
	host_entry = (TCP_HOST_STRUCT *)malloc( sizeof(TCP_HOST_STRUCT) ) ;
	if (!host_entry)
	{
		fprintf(stderr, "Unable to allocate new channel definition host space\n");
		exit (-1);
	}
	strncpy ( mystuff->host_name, args[2], AFCHAN_NAME_SIZE ) ;
	strncpy ( host_entry->host_name, args[2], AFCHAN_NAME_SIZE ) ;
	if ((hp = gethostbyname(args[2])) != NULL)  {
		for ( i = 0 ; ( hp->h_addr_list[i])  && (i < 10 ) ; i++ ) {
			host_entry->sin_addr[i] = 
					(struct in_addr *)malloc(sizeof(struct in_addr)) ;
			memcpy ((char *)host_entry->sin_addr[i], 
					hp->h_addr_list[i],
					sizeof(struct in_addr)) ;
		}
		if ( i < 10 )
			host_entry->sin_addr[i] = NULL ;
		fprintf(stderr,"gethostbyname successful\n");
	}
	else {
		host_entry->sin_addr[0] = 
				(struct in_addr *)malloc(sizeof(struct in_addr)) ;
		host_entry->sin_addr[1] = NULL ;
#if 0
		if (! inet_aton(args[2], host_entry->sin_addr[0] ) ) {
			fprintf(stderr,"Unable to resolve hostname '%s'\n",
				host_entry->host_name );
			fprintf(stderr,"inet_aton UN-successful\n");
		}
#endif
	}
	fprintf(stderr, "Parsing tcp statement: host seems to be: ");
	for ( i = 0 ; ( ( i < 10 ) && (host_entry->sin_addr[i] ) ) ; i++ )
		fprintf(stderr, "(%s) ", inet_ntoa(* host_entry->sin_addr[i] ) ) ;
	fprintf(stderr, "\n");
		
	host_entry->next = NULL ;
	mystuff->host_list = host_entry ;

	mystuff->tcp_port_number = 16255 ;
	mystuff->read_callback = NULL ;

	if ( count > 3 )
		mystuff->tcp_port_number = atoi ( args[3] ) ;

	chan->private = mystuff ;

	/* Fill out the rest of the channel info */
	chan->fd = -1 ;
	chan->init = aftcp_chan_init ;
	chan->chanwrite = tcp_write ;
	chan->changetchar = tcp_getchar ;
#if 1
	printf( "config line: '%s'\n"
			"channel name: '%s'\n"
			"host name: '%s'\n"
			"socket number: '%d'\n",
			line, chan->name, mystuff->host_name, mystuff->tcp_port_number ) ;
#endif
}


/*
	tcp_write writes a message out the network channel
*/
int
tcp_write ( AF_CHANNEL *chan, unsigned char *dat, int size )
{
	TCP_STRUCT *mystuff = chan->private ;

	/* Should be modified to write to all sessions ?? */
	if ( chan->fd < 0 )
		return ;
	chan->write_count++ ;
	chan->bytes_written += size ;
	if ( write ( chan->fd, dat, size ) <= 0 )
	{
		/* 
			Handle write errors.
			This usually means that the connection was lost, so...
			* close this socket
			* mark channel as being unable to write
			* record the time the connection was lost
			* try to re-establish connection by calling connect_to_client.
		*/
		close ( chan->fd ) ;
		chan->fd = -2 ;
		time ( &mystuff->disconnect_time ) ;
		aftcp_connect_to_client( chan ) ;
	}
}

/*
	tcp_getchar reads one character from the network channel.  This call is
	non-blocking and may return a negative value if an error occurs
*/
int
tcp_getchar ( AF_CHANNEL *chan )
{
	int i, result;

	i = 0 ;
	result = read ( chan->fd, &i, 1 ) ;

	if ( result <= 0 )
		return -1 ;

	chan->bytes_read++ ;
	return 0xff & i ;
}


aftcp_remove_session ( TCP_STRUCT *mystuff, TCP_SESSION_STRUCT *s )
{
	TCP_SESSION_STRUCT *p, *pp ;

	pp = NULL ;
	for ( p = mystuff->session_list ; p ; p = p->next ) {
		if ( p == s ) {
			if (pp) {
				pp->next = p->next ;
				p->next = NULL ;
				return ;
			}
			else {
				mystuff->session_list = mystuff->session_list->next ;
				return ;
			}
		}
		pp = p ;
	}
	fprintf(stderr, "aftcp_remove_session: session for host '%s' on port %d\n"
		"not found on list\n", s->host_name, s->remote_tcp_port ) ;
}

aftcp_add_session ( TCP_STRUCT *mystuff, TCP_SESSION_STRUCT *s )
{
	if (!mystuff)
		fprintf(stderr,"aftcp_add_session: passed a NULL TCP_STRUCT pointer\n");
	if (!s)
		fprintf(stderr,"aftcp_add_session: passed a NULL SESSION pointer\n");

	s->next = mystuff->session_list ;
	mystuff->session_list = s ;
}

void
aftcp_timer_DEschedule(AF_CHANNEL *chan)
{
	AFTCP_TIME_ELEMENT *p, *pp, *q ;

	pp = NULL ;
	p = timer_list ;
	while ( p )
	{
		q = p->next ;
		if ( p->chan == chan ) {
			/* 
				handler returned TRUE, it no longer needs to be called,
			    remove it from the timer list
			*/
			if (!pp)
				timer_list = p->next ;
			else 
				pp->next = p->next ;

			free (p);
			return ;
		}
		else
		{
			pp = p ;  /* Do NOT increment the previous pointer if p was
							removed from the list.
						 When p is removed from list, the next p will have
							the same pp as the removed p. */
		}
		p = q ;
	}
}

void
aftcp_timer_schedule(int (*handler)(AF_CHANNEL *chan, int event ),
					 AF_CHANNEL *chan)
{
	AFTCP_TIME_ELEMENT *p, *pp ;

	/* Sanity check */
	for ( p = timer_list ; p ; p = p->next )
		if (p->chan == chan) {
			fprintf(stderr, "TCP TIMER: channel multiply scheduled");
			return ;
		}

	p = (AFTCP_TIME_ELEMENT *)malloc (sizeof(*p));
	if (!p)
	{
		fprintf(stderr, "tcp code unable to allocate space for timer\n");
		return ;
	}
	p->next = timer_list ;
	timer_list = p ;
	p->func = handler ;
	p->chan = chan ;
}

void
aftcp_timeout_handler ( )
{
	/* 
		This code gets called every 5 seconds or so by the select loop
		If the handler returns non-zero (true), it satisfied its timer,
		so it can be removed from the list.
	*/
	AFTCP_TIME_ELEMENT *p, *pp, *q ;

	pp = NULL ;
	p = timer_list ;
	while ( p )
	{
		fprintf(stderr, 
		   "tcp timer is calling handler for channel '%s'\n", p->chan->name );
		q = p->next ;
		if ( p->func(p->chan, AFEVENT_TIMEOUT) ) {
			/* 
				handler returned TRUE, it no longer needs to be called,
			    remove it from the timer list
			*/
			if (!pp)
				timer_list = p->next ;
			else 
				pp->next = p->next ;

			free (p);
		}
		else {
			pp = p ;  /* Do NOT increment the previous pointer if p was
							removed from the list.
						 When p is removed from list, the next p will have
							the same pp as the removed p. */
		}
		p = q ;
	}
}

int
aftcp_connect_select_handler (AF_CHANNEL *chan,  int event )
{
	TCP_STRUCT *mystuff = chan->private ;
	unsigned char  dummy ;
	int ret ;

	fprintf(stderr, "aftcp_connect_select_handler got event #%d\n",event);
	/* 
		Select loop calls us when write is ready on channel.
		This indicates the the connection is complete.
	*/
	switch ( event )
	{
	case AFEVENT_WRITE: 
		fprintf(stderr,"connect_select_handler sees a write event\n");
		if ( mystuff->connect_fd < 0 ) {
			/* 
				The connection is established, so this is a write event,
				not a connect-related event.

				Assume that its a write error, and close the channel1
			*/
			if (chan->fd > 0 ) {
				close (chan->fd) ;
				chan->fd = -1 ;
			}
		}

		/* 
			Write a null to the channel to see what the result of the
			connect was
		*/
		dummy = 0 ;
		ret = write ( mystuff->connect_fd, &dummy, 1 ) ;
		if (ret < 0 )
		{
			/* connect was unsuccessful */
			close ( mystuff->connect_fd ) ;
			afloop_write_select_list_remove ( mystuff->connect_fd ) ;
			time ( &mystuff->last_connect_attempt ) ;
			aftcp_timer_schedule(aftcp_connect_select_handler,chan); 
			fprintf(stderr,"connect_select_handler sees a bad write, starting timer\n");
		}
		else
		{
			/*
				- mark connect time
				- remove fd from write_select_list
				- mark channel state such that writes REALLY write
			*/
			fprintf(stderr,"conn_select_handler sees a good write, stopping timer, marking channel as open\n");
			time ( &mystuff->connect_time ) ;
			afloop_write_select_list_remove ( mystuff->connect_fd ) ;
			chan->fd = mystuff->connect_fd ;
			mystuff->connect_fd = -1 ;
			aftcp_timer_DEschedule(chan); 
#if 1
			fprintf(stderr,"aftcp: connection to client established\n");
#endif
		}
	break;
	case AFEVENT_TIMEOUT:
		/* Make sure that this channel is disconnected */
		if ( chan->fd != -1 )
			return 1 ;

		if ( time(NULL) > 
				( mystuff->last_connect_attempt + AFTCP_CONNECT_RETRY_TIME ) )
		{
			fprintf(stderr, "conn_handler sees a timer event, expired, calling connect_to_client\n");
			aftcp_connect_to_client(chan);
			return 1 ;
		}
		else
			fprintf(stderr, "conn_handler sees a timer event, not expired\n");
			return 0 ;
	break;
	}

}

void
aftcp_connect_to_client (AF_CHANNEL *chan)
{
    int s;
    struct sockaddr_in addr;
    struct hostent *hp;
	int ret ;
	TCP_STRUCT *mystuff = chan->private ;

#if 1
    fprintf(stderr,"Calling %s on port %d.\n", 
		mystuff->host_name, mystuff->tcp_port_number );
#endif

	/* Mark channel as disconnected */
	chan->fd = -1 ;

    addr.sin_family = AF_INET;
    addr.sin_port = htons(mystuff->tcp_port_number);
    if ((addr.sin_addr.s_addr = inet_addr(mystuff->host_name)) == -1) 
	{
		if ((hp = gethostbyname(mystuff->host_name)) == NULL) 
		{
			fprintf(stderr,"Who is %s?\n", mystuff->host_name);
			exit(0);
		} 
		else 
		{
			/* Shouldn't this do a htons or something ? --REL */
			addr.sin_addr.s_addr = *(long *) hp->h_addr;
		}
	}

    if ((s=socket(AF_INET, SOCK_STREAM, 0)) < 0) 
	{
		printf("I can't create a socket\n");
		perror("Socket creation");
		exit(0);
    }

	/* try an fcntl to make this socket ASYNC_IO */
	ret = fcntl(s, F_SETFL, O_NONBLOCK ) ;
	if ( ret < 0 )
	{
		fprintf(stderr,"fcntl returned %d. Connect may BLOCK!\n",ret);
		perror("fcntl");
	}

    if (connect(s, (struct sockaddr *)&addr, sizeof(addr)) < 0) 
	{
		perror("connect error");
		if (errno != EINPROGRESS) {
			/* connect was unsuccessful */
			close ( s ) ;
			time ( &mystuff->last_connect_attempt ) ;
			aftcp_timer_schedule(aftcp_connect_select_handler,chan); 
			mystuff->connect_fd = -1 ;
			return ;
		}
    }
	afloop_write_select_list_add ( s, aftcp_connect_select_handler, chan ) ;
	mystuff->connect_fd = s ;
}


void
aftcp_listen_for_server( AF_CHANNEL *chan ) 
{
    struct sockaddr_in addr;
	TCP_STRUCT *mystuff = chan->private ;

    fprintf(stderr,"aftcp: Listening for connection on port %d.\n",
					mystuff->tcp_port_number);
    if ((mystuff->listen_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		printf("I can't create a socket\n");
		perror("socket creation");
		exit(2);
    }
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = htons(mystuff->tcp_port_number);

    if (bind(mystuff->listen_fd, (struct sockaddr *)&addr, sizeof(addr)) < 0) 
	{
		fprintf(stderr, "I can't bind to port %d!\n", mystuff->tcp_port_number);
		/* Should there be a time-out/retry here on the bind ? */
		return ;
    }

	/* Set up listen function's select entry */
    listen(mystuff->listen_fd,1);
	afloop_select_list_add ( mystuff->listen_fd, aftcp_listen_select_handler, chan ) ;
}


void
aftcp_listen_select_handler ( AF_CHANNEL *chan )
{
    struct sockaddr_in naddr;
    int len;
	int sock ; /* Is this temp needed ? */
	TCP_STRUCT *mystuff = chan->private ;
	TCP_SESSION_STRUCT *session ;
	TCP_HOST_STRUCT *q ;
    struct hostent *hp, *revhp;
	struct in_addr temp_addr ;
	char **p ;
	int i, j ;

	/*
		accept the session
	*/
    len=sizeof(naddr);
    sock=accept(mystuff->listen_fd, (struct sockaddr *) &naddr, &len);
    if (sock==-1) 
	{
		perror("aftcp accept error");
		return  ;
    }

#if 1
    fprintf(stderr,"aftcp: Got connection.\n");
#endif
	session = (TCP_SESSION_STRUCT *)malloc( sizeof(TCP_SESSION_STRUCT) ) ;
	if (!session)
	{
		fprintf(stderr, "Unable to allocate space for new tcp session description\n");
		close (sock) ;
		return ;
	}
    /*
     *  Determine the address of the remote host, check against
     *  the allowed hosts list (probably just one host in the list)
     */

#if 0
	/* compare this address with any numeric hosts */
	for ( q = mystuff->host_list ; q ; q = q->next ) {
		if (inet_aton(q->host_name, &temp_addr ) ) {
			/* This is a valid numeric address */
			if ( memcmp ( (char *)&temp_addr, 
						  (char *)&naddr.sin_addr.s_addr, 
						  sizeof (temp_addr) ) == 0 )     {
				goto skip_dns_lookup ;
			}
		}
	}
#endif

#if 0
	/* 
		is calling getpeername necessary ?  Doesn't accept fill out
		naddr with the same data ?
	*/
    len = sizeof(struct sockaddr_in);
    if (getpeername(sock, (struct sockaddr *) &addr, &len) < 0) 
	{
		perror("unable to get peername");
		serverName = "nowhere";
    } 
	else 
#endif
	{
		hp = gethostbyaddr((char *)&naddr.sin_addr.s_addr, sizeof(long),AF_INET);
		if (hp != NULL) 
		{
			session->host_name = (char *) malloc(strlen(hp->h_name)+1);
			strcpy(session->host_name, hp->h_name);
			revhp = gethostbyname(session->host_name) ;
			if (revhp != NULL) {
				/*
					verify that the address in use is one of the addresses
					for this hostname. (Do a forward lookup)
				*/
				p = revhp->h_addr_list ;
				for (  i = 0 ; p[i] ; i++ )	 {
					fprintf(stderr, "comparing addresses (%s)",
					inet_ntoa( * ( (struct in_addr*)&naddr.sin_addr.s_addr))) ;
					fprintf(stderr, " and (%s)\n",
						inet_ntoa ( *((struct in_addr *)p[i]) ) ) ;

/*
				p[i], (char *)&naddr.sin_addr.s_addr ) ;
*/
					if ( memcmp (  p[i],
								   (char *)&naddr.sin_addr.s_addr,
								   revhp->h_length  ) == 0 ) {
						/* Found a matching entry, this host is legit. */
						break ;
					}
				}
				/* 
					See how we escaped the preceding loop...
						if p[i] is NULL, there was no matching entry
						if p[i] is non-NULL, there WAS a matching entry

					A matching entry indicates a genuine host identity.
				*/
				if ( p[i] == NULL ) {
					/*
						Not convinced of the remote host's authenticity,
						either DNS failed us or the remote host's DNS 
						entries forward and reverse records don't agree.
					*/
#if 1
					fprintf(stderr,
	"Dumping connection. Remote host (%s) address resolved to hostname '%s'.\n"
	"     while hostname '%s' resolved to addresses : ",
						inet_ntoa( * ( (struct in_addr*)&naddr.sin_addr.s_addr) ),
						session->host_name, session->host_name );
					for (j = 0 ; p[j] ; j++ )
						fprintf(stderr, "(%s) ", 
								inet_ntoa ( *((struct in_addr *)p[i]) ) ) ;
					fprintf(stderr,"\n");
#endif
				write(sock, 
					"502 You have no permission to talk.  Goodbye.\n",46);
				close (sock);
				free(session);
				return ;
				}
#if 1
					fprintf(stderr,
						"Connection established with remote host '%s'.\n"
						"    Its address (%s) resolved to hostname '%s'.\n"
						"    while hostname '%s' resolved to addresses : ",
						session->host_name,
						inet_ntoa(*((struct in_addr*)&naddr.sin_addr.s_addr) ),
						session->host_name, session->host_name );
					for (j = 0 ; p[j] ; j++ )
						fprintf(stderr, "(%s) ", 
								inet_ntoa ( *((struct in_addr *)p[j]) ) ) ;
					fprintf(stderr,"\n");
#endif
			}
			else {
				/* gethostbyname returned NULL */
#if 1
					fprintf(stderr,
	"Dumping connection. Remote host (%s) address resolved to hostname '%s'.\n"
	"     while hostname '%s' could NOT be resolved.\n",
						inet_ntoa( * ( (struct in_addr*)&naddr.sin_addr.s_addr) ),
						session->host_name, session->host_name );
#endif
				write(sock, 
					"502 You have no permission to talk.  Goodbye.\n",46);
				close (sock);
				free(session);
				return ;
			}

		} 
		else 
		{
			/* 
				unable to resolve host name...
				If there was a numeric address match, it would have been
				detected already, so dump this session
			*/
			fprintf ( stderr, "gethostbyaddr failed\n");
			write(sock, 
				"502 You have no permission to talk.  Goodbye.\n",46);
			close (sock);
			free(session);
			return ;

			#if 0
			session->host_name = 
					(char *) malloc(strlen(inet_ntoa(naddr.sin_addr))+1);
			strcpy(session->host_name, inet_ntoa(naddr.sin_addr));
			#endif
		}
    }

	for ( q = mystuff->host_list ; q ; q = q->next ) {
		fprintf(stderr,"q loop ");
		for ( i = 0 ; ( i < 10) && (q->sin_addr[i]) ; i++ ) {
			fprintf(stderr,"i loop ");
			if ( memcmp ((char *)q->sin_addr[i], 
						 (char *)&naddr.sin_addr, 
						 sizeof (*q->sin_addr) ) == 0 ) {
				fprintf(stderr,"Break!\n");
				goto double_break ;
				}
			}
		}
double_break:

	if ( !q || !q->sin_addr[i] ) {
		fprintf ( stderr, "Connection dumped from unknown host (%s)\n"
			"Host not found in the configuration for this channel\n",
			inet_ntoa(naddr.sin_addr) ) ;
		write(sock, 
			"502 You have no permission to talk.  Goodbye.\n",46);
		close (sock);
		free(session);
		return ;
	}
		
skip_dns_lookup:
	session->fd = sock ;
	session->remote_tcp_port = naddr.sin_port ;
	session->chan = chan ;
#if 1
    fprintf(stderr,
	"aftcp: established session with client %s (%s) on client port %d\n", 
	session->host_name, inet_ntoa(naddr.sin_addr), session->remote_tcp_port );
#endif
	/*
		Let's just maintain all input sessions, closing them when they
		give us a read error
	*/
	aftcp_add_session( chan->private, session )  ;
	afloop_select_list_add ( session->fd, aftcp_read_select_handler, session ) ;
}

void
aftcp_read_select_handler( void *private, int event )
{
	int ret ;
	unsigned char buffer ; 
	TCP_SESSION_STRUCT *session = private ;

#if 0
	fprintf(stderr, "tcp_read: reading on fd=%d\n",session->fd);
#endif
	if ( (ret = read ( session->fd, &buffer, 1 )) <= 0 )
	{
		if 	( ( errno == EAGAIN ) || ( errno == EINTR )  )
			return ;
#if 0
		fprintf(stderr,"\nread from socket returned %d. errno=%d h_errno=%d\n",
			ret, errno, h_errno ) ;
		perror("network read");
#endif
		afloop_select_list_remove ( session->fd ) ;
		close (session->fd);
		aftcp_remove_session(session->chan->private, session);
		free(session->host_name);
		free(session);
	}
	else
		((TCP_STRUCT *)session->chan->private)->read_callback(
											session->chan,buffer) ;
}

/*
	tcp_init initializes a tcp channel
*/
int
aftcp_chan_init ( AF_CHANNEL *chan, void (*read_callback)() )
{
	TCP_STRUCT *mystuff = chan->private ;
	/* Check to see if previously initialized */
	if ( chan->fd != -1 )
	{
		/* Already initialized, just return */
		return 0 ;
	}

	/* 
		open tcp connection
	*/

	if (read_callback)
	{
		/* 
			If this is an intialization as a read channel, call the
			client initialization.
		*/
		/* do the socket->bind->connect thing */
		mystuff->read_callback = read_callback ;
		aftcp_listen_for_server (chan) ;
	}
	else
	{
		/* set up the tcp port listener */
		aftcp_connect_to_client (chan);
	}
}
