/********************************************************************************
*                                                                               *
*                          T r e e L i s t   O b j e c t                        *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXOldTreeList.cpp,v 1.2 1999/09/21 18:02:58 jeroen Exp $                 *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXLabel.h"
#include "FXButton.h"
#include "FXComposite.h"
#include "FXScrollbar.h"
#include "FXScrollWindow.h"
#include "FXOldTreeList.h"



/*
  To do:
  - Draw stuff differently when disabled.
  - Optimize element insert/delete: don't always have to redraw everything.
  - Tooltip should show text of current item, if any.
  - Tooltip should pop up exactly on top of current item.
  - Should allow for more flexible icon sizes.
  - You always get a SEL_COMMAND, even if no item.
  - Clicking on + does not make it current?
  - Text layout not good if bigger texts.
  - Need box on toplevel items, sometimes...
  - Need indent variable.
  - Should compute size of icons etc.
  - I guess select==open, but open is also open...
  - Need API to expand all parent of certain item so that it may be visible
  - Need translate right-clicks into message with item figured out...
  - Return key simulates double click.
  - In autoselect mode, all items are expanded.
  - Double-click on + box is wrong.
  - When doing makeItemVisible(), it may have to expand the parents!!
  - Sortfunc's will be hard to serialize, and hard to write w/o secretly #including
    the FXOldTreeItem header!

*/


#define ICON_SPACING        4         // Spacing between parent and child in x direction
#define TEXT_SPACING        4         // Spacing between icon and text
#define LINE_SPACING        1         // Spacing between lines
#define SIDE_SPACING        4         // Spacing between side and item

#define DEFAULT_ICONWIDTH   16
#define DEFAULT_ICONHEIGHT  16
#define DEFAULT_INDENT      20        // Indent between parent and child


#define DRAG_NONE   0                 // No action going on
#define DRAG_DND    1                 // Dragging icons
#define DRAG_TRYDND 2                 // Tentative dragging

#define OLDTREELIST_MASK   (OLDTREELIST_MULTIPLESELECT|OLDTREELIST_AUTOSELECT|OLDTREELIST_SHOWS_LINES|OLDTREELIST_SHOWS_BOXES|OLDTREELIST_WANTSELECTION)


/*******************************************************************************/

// Map
FXDEFMAP(FXOldTreeList) FXOldTreeListMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXOldTreeList::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXOldTreeList::onMotion),
  FXMAPFUNC(SEL_TIMEOUT,FXWindow::ID_AUTOSCROLL,FXOldTreeList::onAutoScroll),
  FXMAPFUNC(SEL_TIMEOUT,FXOldTreeList::ID_TIPTIMER,FXOldTreeList::onTipTimer),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXOldTreeList::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXOldTreeList::onLeftBtnRelease),
  FXMAPFUNC(SEL_UNGRABBED,0,FXOldTreeList::onUngrabbed),
  FXMAPFUNC(SEL_KEYPRESS,0,FXOldTreeList::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXOldTreeList::onKeyRelease),
  FXMAPFUNC(SEL_ENTER,0,FXOldTreeList::onEnter),
  FXMAPFUNC(SEL_LEAVE,0,FXOldTreeList::onLeave),
  FXMAPFUNC(SEL_ACTIVATE,0,FXOldTreeList::onActivate),
  FXMAPFUNC(SEL_DEACTIVATE,0,FXOldTreeList::onDeactivate),
  FXMAPFUNC(SEL_FOCUSIN,0,FXOldTreeList::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXOldTreeList::onFocusOut),
  FXMAPFUNC(SEL_OPENED,0,FXOldTreeList::onItemOpened),
  FXMAPFUNC(SEL_CLOSED,0,FXOldTreeList::onItemClosed),
  FXMAPFUNC(SEL_EXPANDED,0,FXOldTreeList::onItemExpanded),
  FXMAPFUNC(SEL_COLLAPSED,0,FXOldTreeList::onItemCollapsed),
  FXMAPFUNC(SEL_CHANGED,0,FXOldTreeList::onChanged),
  FXMAPFUNC(SEL_CLICKED,0,FXOldTreeList::onClicked),
  FXMAPFUNC(SEL_DOUBLECLICKED,0,FXOldTreeList::onDoubleClicked),
  FXMAPFUNC(SEL_TRIPLECLICKED,0,FXOldTreeList::onTripleClicked),
  FXMAPFUNC(SEL_SELECTED,0,FXOldTreeList::onSelected),
  FXMAPFUNC(SEL_DESELECTED,0,FXOldTreeList::onDeselected),
  FXMAPFUNC(SEL_SELECTION_LOST,0,FXOldTreeList::onSelectionLost),
  FXMAPFUNC(SEL_SELECTION_GAINED,0,FXOldTreeList::onSelectionGained),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_TIP,FXOldTreeList::onQueryTip),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_HELP,FXOldTreeList::onQueryHelp),
  };


// Object implementation
FXIMPLEMENT(FXOldTreeList,FXScrollArea,FXOldTreeListMap,ARRAYNUMBER(FXOldTreeListMap))

  
/*******************************************************************************/

  
// Tree List
FXOldTreeList::FXOldTreeList(){
  flags|=FLAG_ENABLED;
  firstitem=NULL;
  lastitem=NULL;
  anchoritem=NULL;
  currentitem=NULL;
  font=(FXFont*)-1;
  sortfunc=NULL;
  textColor=0;
  selbackColor=0;
  seltextColor=0;
  lineColor=0;
  iconWidth=DEFAULT_ICONWIDTH;
  iconHeight=DEFAULT_ICONHEIGHT;
  itemWidth=0;
  itemHeight=0;
  totalWidth=0;
  totalHeight=0;
  indent=DEFAULT_INDENT;
  timer=NULL;
  }


// Tree List
FXOldTreeList::FXOldTreeList(FXComposite *p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
  FXScrollArea(p,opts,x,y,w,h){
  flags|=FLAG_ENABLED;
  target=tgt;
  message=sel;
  firstitem=NULL;
  lastitem=NULL;
  anchoritem=NULL;
  currentitem=NULL;
  font=getApp()->normalFont;
  sortfunc=NULL;
  textColor=getApp()->foreColor;
  selbackColor=getApp()->selbackColor;
  seltextColor=getApp()->selforeColor;
  lineColor=getApp()->shadowColor;
  iconWidth=DEFAULT_ICONWIDTH;
  iconHeight=DEFAULT_ICONHEIGHT;
  itemWidth=0;
  itemHeight=0;
  totalWidth=0;
  totalHeight=0;
  indent=DEFAULT_INDENT;
  timer=NULL;
  }


// Create window
void FXOldTreeList::create(){
  register FXOldTreeItem* n=firstitem;
  FXScrollArea::create();
  while(n){
    if(n->openIcon){n->openIcon->create();}
    if(n->closedIcon){n->closedIcon->create();}
    if(n->first){n=n->first;continue;}
    while(!n->next && n->parent){n=n->parent;}
    n=n->next;
    }
  if(!deleteType){deleteType=getApp()->registerDragType(deleteTypeName);}
  if(!textType){textType=getApp()->registerDragType(textTypeName);}
  font->create();
  dropEnable();
  }


// Detach window
void FXOldTreeList::detach(){
  register FXOldTreeItem* n=firstitem;
  FXScrollArea::detach();
  while(n){
    if(n->openIcon){n->openIcon->detach();}
    if(n->closedIcon){n->closedIcon->detach();}
    if(n->first){n=n->first;continue;}
    while(!n->next && n->parent){n=n->parent;}
    n=n->next;
    }
  font->detach();
  deleteType=0;
  textType=0;
  }


// Get number of toplevel items
FXint FXOldTreeList::getNumItems() const {
  register FXOldTreeItem *item=firstitem;
  register FXint n=0;
  while(item){
    item=item->next;
    n++;
    }
  return n;
  }


// Get number of child items
FXint FXOldTreeList::getNumChildItems(FXOldTreeItem* par) const {
  if(!par){ fxerror("%s::getNumChildItems: item is NULL.\n",getClassName()); } 
  register FXOldTreeItem *item=par->first;
  register FXint n=0;
  while(item){
    item=item->next;
    n++;
    }
  return n;
  }


// Get number of selected items
FXint FXOldTreeList::getNumSelectedItems() const {
  register FXOldTreeItem *item=firstitem;
  register FXint n=0;
  while(item){
    if(item->state&OLDTREEITEM_SELECTED) n++;
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  return n;
  }


// Get list of selected ite,s
FXOldTreeItem** FXOldTreeList::getSelectedItems() const {
  FXOldTreeItem *item=firstitem;
  FXOldTreeItem **itemlist;
  FXint i=0;
  FXMALLOC(&itemlist,FXOldTreeItem*,getNumSelectedItems()+1);
  while(item){
    if(item->state&OLDTREEITEM_SELECTED) itemlist[i++]=item;
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  itemlist[i]=NULL;
  return itemlist;
  }
 

// Propagate size change
void FXOldTreeList::recalc(){ 
  FXScrollArea::recalc();
  flags|=FLAG_RECALC;
  }


// Can have focus
FXbool FXOldTreeList::canFocus() const { return TRUE; }


// Create item
FXOldTreeItem* FXOldTreeList::createItem(){ 
  return new FXOldTreeItem; 
  }


// Destroy item
void FXOldTreeList::deleteItem(FXOldTreeItem* item){ 
  delete item; 
  }


// Get item X
FXint FXOldTreeList::getItemX(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::getItemX: item is NULL.\n",getClassName()); } 
  return item->x; 
  }


// Get item Y
FXint FXOldTreeList::getItemY(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::getItemY: item is NULL.\n",getClassName()); } 
  return item->y; 
  }


// Get item width
FXint FXOldTreeList::getItemWidth(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::getItemWidth: item is NULL.\n",getClassName()); } 
  return iconWidth+font->getTextWidth(item->label.text(),item->label.length())+TEXT_SPACING+SIDE_SPACING;
  }


// Get Item height
FXint FXOldTreeList::getItemHeight(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::getItemHeight: item is NULL.\n",getClassName()); }
  return FXMAX(iconHeight,(font->getFontHeight()+1))+LINE_SPACING; 
  }


// True if item is selected
FXbool FXOldTreeList::isItemSelected(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::isItemSelected: item is NULL.\n",getClassName()); } 
  return (item->state&OLDTREEITEM_SELECTED)!=0; 
  }


// True if item is current
FXbool FXOldTreeList::isItemCurrent(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::isItemCurrent: item is NULL.\n",getClassName()); } 
  return (item->state&OLDTREEITEM_CURRENT)!=0; 
  }


// True if item (partially) visible
FXbool FXOldTreeList::isItemVisible(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::isItemVisible: item is NULL.\n",getClassName()); } 
  return 0<(pos_y+item->y+itemHeight) && (pos_y+item->y)<viewport_h; 
  }


// Check if item is expanded
FXbool FXOldTreeList::isItemExpanded(const FXOldTreeItem* item) const {
  if(!item){ fxerror("%s::isItemExpanded: item is NULL.\n",getClassName()); } 
  return (options&OLDTREELIST_AUTOSELECT) || (item->state&OLDTREEITEM_EXPANDED); 
  }
  

// Is item a leaf item
FXbool FXOldTreeList::isItemLeaf(const FXOldTreeItem* item) const {
  if(!item){ fxerror("%s::isItemLeaf: item is NULL.\n",getClassName()); } 
  return item->first!=NULL; 
  }


// Get next item
FXOldTreeItem* FXOldTreeList::getNextItem(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::getNextItem: item is NULL.\n",getClassName()); } 
  return item->next; 
  }


// Get previous item
FXOldTreeItem* FXOldTreeList::getPrevItem(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::getPrevItem: item is NULL.\n",getClassName()); } 
  return item->prev; 
  }


// Get item (logically) above item
FXOldTreeItem* FXOldTreeList::getItemAbove(FXOldTreeItem* item) const {
  if(!item) return NULL;
  if(!item->prev) return item->parent;
  item=item->prev;
  while(item->last) item=item->last;
  return item;
  }


// Get item (logically) below
FXOldTreeItem* FXOldTreeList::getItemBelow(FXOldTreeItem* item) const {
  if(!item) return NULL;
  if(item->first) return item->first;
  while(!item->next && item->parent) item=item->parent;
  return item->next;
  }


// Get item's first child
FXOldTreeItem* FXOldTreeList::getFirstChildItem(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::getFirstChildItem: item is NULL.\n",getClassName()); } 
  return item->first; 
  }


// Get item's last child
FXOldTreeItem* FXOldTreeList::getLastChildItem(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::getLastChildItem: item is NULL.\n",getClassName()); } 
  return item->last; 
  }


// Get item's parent
FXOldTreeItem* FXOldTreeList::getParentItem(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::getParentItem: item is NULL.\n",getClassName()); } 
  return item->parent; 
  }


// Did we hit the item box
FXbool FXOldTreeList::hitItemBox(const FXOldTreeItem* item,FXint x,FXint y) const {
  if(!item){ fxerror("%s::hitItemBox: item is NULL.\n",getClassName()); }
  if((options&OLDTREELIST_SHOWS_BOXES) && ((item->state&OLDTREEITEM_HASITEMS) || item->first)){
    x=x-pos_x;
    y=y-pos_y;
    return item->y<=y && y<item->y+itemHeight && item->x-indent+iconWidth/2+(SIDE_SPACING/2)-4<=x && x<=item->x-indent+iconWidth/2+(SIDE_SPACING/2)+4;
    }
  return FALSE;
  }


// Did we hit the item icon
FXbool FXOldTreeList::hitItemIcon(const FXOldTreeItem* item,FXint x,FXint y) const {
  if(!item){ fxerror("%s::hitItemIcon: item is NULL.\n",getClassName()); }
  x=x-pos_x;
  y=y-pos_y;
  return item->y<=y && y<item->y+itemHeight && item->x+(SIDE_SPACING/2)<=x && x<item->x+(SIDE_SPACING/2)+iconWidth;
  }


// Did we hit the item
FXbool FXOldTreeList::hitItem(const FXOldTreeItem* item,FXint x,FXint y) const {
  if(!item){ fxerror("%s::hitItem: item is NULL.\n",getClassName()); } 
  x=x-pos_x;
  y=y-pos_y;
  return item->y<=y && y<item->y+itemHeight && item->x+(SIDE_SPACING/2)<=x;
  }


// Repaint
void FXOldTreeList::updateItem(FXOldTreeItem* item){
  if(xid){update(0,pos_y+item->y,content_w,itemHeight);}
  }

  
// Sort items
void FXOldTreeList::sort(FXOldTreeItem*& f1,FXOldTreeItem*& t1,FXOldTreeItem*& f2,FXOldTreeItem*& t2,int n){
  FXOldTreeItem *ff1,*tt1,*ff2,*tt2,*q;
  FXint m;
  if(f2==NULL){ 
    f1=NULL; 
    t1=NULL; 
    return; 
    }
  if(n>1){
    m=n/2;
    n=n-m;
    sort(ff1,tt1,f2,t2,n);  // 1 or more
    sort(ff2,tt2,f2,t2,m);  // 0 or more
    FXASSERT(ff1);
    if(ff2 && sortfunc(ff1,ff2)){
      f1=ff2;
      ff2->prev=NULL;
      ff2=ff2->next;
      }
    else{
      f1=ff1;
      ff1->prev=NULL;
      ff1=ff1->next;
      }
    t1=f1;
    t1->next=NULL;
    while(ff1 || ff2){
      if(ff1==NULL){ t1->next=ff2; ff2->prev=t1; t1=tt2; break; }
      if(ff2==NULL){ t1->next=ff1; ff1->prev=t1; t1=tt1; break; }
      if(sortfunc(ff1,ff2)){
        t1->next=ff2;
        ff2->prev=t1;
        t1=ff2;
        ff2=ff2->next;
        }
      else{
        t1->next=ff1;
        ff1->prev=t1;
        t1=ff1;
        ff1=ff1->next;
        }
      t1->next=NULL;
      }
    return;
    }
  FXASSERT(f2);
  f1=f2;
  t1=f2;
  f2=f2->next;
  while(f2){
    f2->prev=NULL;
    if(sortfunc(f2,t1)){
      t1->next=f2;
      f2->prev=t1;
      t1=f2;
      f2=f2->next;
      continue;
      }
    if(sortfunc(f1,f2)){
      q=f2;
      f2=f2->next;
      q->next=f1;
      f1->prev=q;
      f1=q;
      continue;
      }
    break;
    }
  FXASSERT(f1);
  FXASSERT(t1);
  f1->prev=NULL;
  t1->next=NULL;
  }


// Sort the items based on the sort function
void FXOldTreeList::sortItems(){
  if(sortfunc){
    FXOldTreeItem* f=firstitem;
    FXOldTreeItem* l=lastitem;
    sort(firstitem,lastitem,f,l,getNumItems());
    recalc();
    }
  }


// Sort child items
void FXOldTreeList::sortChildItems(FXOldTreeItem* item){
  if(sortfunc){
    FXOldTreeItem* f=item->first;
    FXOldTreeItem* l=item->last;
    sort(item->first,item->last,f,l,getNumChildItems(item));
    if(item->state&OLDTREEITEM_EXPANDED) recalc();     // No need to recalc if it ain't visible!
    }
  }


// All turned on
FXbool FXOldTreeList::selectItemRange(FXOldTreeItem* beg,FXOldTreeItem* end){
  register FXbool selected=FALSE;
  register FXOldTreeItem *item=beg;
  while(item){
    selected|=selectItem(item);
    if(item==end) break;
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  return selected;
  }


// All turned off
FXbool FXOldTreeList::deselectItemRange(FXOldTreeItem* beg,FXOldTreeItem* end){
  register FXbool deselected=FALSE;
  register FXOldTreeItem *item=beg;
  while(item){
    deselected|=deselectItem(item);
    if(item==end) break;
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  return deselected;
  }



// Toggle items
FXbool FXOldTreeList::toggleItemRange(FXOldTreeItem* beg,FXOldTreeItem* end){
  register FXbool toggled=FALSE;
  register FXOldTreeItem *item=beg;
  while(item){
    toggled|=toggleItem(item);
    if(item==end) break;
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  return toggled;
  }


// Restore items to previous state
FXbool FXOldTreeList::restoreItemRange(FXOldTreeItem* beg,FXOldTreeItem* end){
  register FXbool restored=FALSE;
  register FXOldTreeItem *item=beg;
  while(item){
    restored|=restoreItem(item);
    if(item==end) break;
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  return restored;
  }


// Select one item
FXbool FXOldTreeList::selectItem(FXOldTreeItem* item){
  if(item){
    if(!(item->state&OLDTREEITEM_SELECTED)){
      item->state|=OLDTREEITEM_SELECTED;
      updateItem(item);
      return TRUE;
      }
    }
  return FALSE;
  }


// Deselect one item
FXbool FXOldTreeList::deselectItem(FXOldTreeItem* item){
  if(item){
    if(item->state&OLDTREEITEM_SELECTED){
      item->state&=~OLDTREEITEM_SELECTED;
      updateItem(item);
      return TRUE;
      }
    }
  return FALSE;
  }


// toggle one item
FXbool FXOldTreeList::toggleItem(FXOldTreeItem* item){
  if(item){
    item->state^=OLDTREEITEM_SELECTED;
    updateItem(item);
    return TRUE;
    }
  return FALSE;
  }


// Restore items to previous state
FXbool FXOldTreeList::restoreItem(FXOldTreeItem* item){
  if(item){
    if((item->state&OLDTREEITEM_MARK) && !(item->state&OLDTREEITEM_SELECTED)){
      item->state|=OLDTREEITEM_SELECTED;
      updateItem(item);
      return TRUE;
      }
    if(!(item->state&OLDTREEITEM_MARK) && (item->state&OLDTREEITEM_SELECTED)){
      item->state&=~OLDTREEITEM_SELECTED;
      updateItem(item);
      return TRUE;
      }
    }
  return FALSE;
  }


// Mark items
void FXOldTreeList::markItems(){
  register FXOldTreeItem *item=firstitem;
  while(item){
    (item->state&OLDTREEITEM_SELECTED)?item->state|=OLDTREEITEM_MARK:item->state&=~OLDTREEITEM_MARK;
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  }


// Recompute interior 
void FXOldTreeList::recompute(){
  register FXOldTreeItem* item;
  register FXint x,y,t;
  totalWidth=0;
  totalHeight=0;
  itemWidth=0;
  itemHeight=FXMAX(iconHeight,font->getFontHeight()+1)+LINE_SPACING;
  if(itemHeight&1) itemHeight++;    // Got to be even for the dotted lines!!
  x=y=0;
  if(options&OLDTREELIST_ROOT_BOXES) x=indent;
  item=firstitem;
  while(item){
    item->x=x;
    item->y=y;
    y+=itemHeight;
    if(item->label.text()){
      t=x+iconWidth+TEXT_SPACING+SIDE_SPACING+font->getTextWidth(item->label.text(),item->label.length());
      if(t>itemWidth) itemWidth=t;
      }
    if(item->first && ((options&OLDTREELIST_AUTOSELECT)||(item->state&OLDTREEITEM_EXPANDED))){
      x+=indent;
      item=item->first;
      continue;
      }
    while(!item->next && item->parent){
      x-=indent;
      item=item->parent;
      }
    item=item->next;
    }
  totalWidth=itemWidth;
  totalHeight=y;
  flags&=~FLAG_RECALC;
  }

    
// Determine content width of tree list
FXint FXOldTreeList::getContentWidth(){
  if(flags&FLAG_RECALC) recompute();
  return totalWidth;
  }


// Determine content height of tree list
FXint FXOldTreeList::getContentHeight(){
  if(flags&FLAG_RECALC) recompute();
  return totalHeight;
  }


// Recalculate layout
void FXOldTreeList::layout(){

  // Calculate contents
  FXScrollArea::layout();
  
  // Set line size based on font
  vertical->setLine(itemHeight);
  horizontal->setLine(1);
  
  // Force repaint
  update();
  
  // No more dirty
  flags&=~FLAG_DIRTY;
  }


// Start motion timer while in this window
long FXOldTreeList::onEnter(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onEnter(sender,sel,ptr);
  if(!timer){timer=getApp()->addTimeout(getApp()->menuPause,this,ID_TIPTIMER);}
  return 1;
  }


// Stop motion timer when leaving window
long FXOldTreeList::onLeave(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onLeave(sender,sel,ptr);
  if(timer){getApp()->removeTimeout(timer);timer=NULL;}
  return 1;
  }


// Gained focus
long FXOldTreeList::onFocusIn(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusIn(sender,sel,ptr);
  if(currentitem) updateItem(currentitem);
  return 1;
  }


// Lost focus
long FXOldTreeList::onFocusOut(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusOut(sender,sel,ptr);
  if(currentitem) updateItem(currentitem);
  return 1;
  }


// We have the selection
long FXOldTreeList::onSelectionGained(FXObject* sender,FXSelector sel,void* ptr){
  if(FXScrollArea::onSelectionGained(sender,sel,ptr)) return 1;
  //update();
  return 1;
  }


// We lost the selection
long FXOldTreeList::onSelectionLost(FXObject* sender,FXSelector sel,void* ptr){
  if(FXScrollArea::onSelectionLost(sender,sel,ptr)) return 1;
  deselectItemRange(firstitem,NULL);
  return 1;
  }


// Draw dashed focus rectangle
void FXOldTreeList::drawFocusRectangle(FXDCWindow& dc,FXint x,FXint y,FXint w,FXint h){
  static const char onoff[]={1,2};
  dc.setForeground(textColor);
  dc.setDashes(0,onoff,2);
  dc.setLineStyle(LINE_ONOFF_DASH);
  dc.drawLine(x,y,x+w-1,y);
  dc.drawLine(x,y,x,y+h-1);
  dc.drawLine(x,y+h-1,x+w-1,y+h-1);
  dc.drawLine(x+w-1,y,x+w-1,y+h-1);
  dc.setLineStyle(LINE_SOLID);
  }


// Draw item list
long FXOldTreeList::onPaint(FXObject*,FXSelector,void* ptr){
  static const char onoff[]={1,1};
  FXEvent* event=(FXEvent*)ptr;
  FXOldTreeItem* n=firstitem;
  FXint eylo,eyhi,tw,th,yh,xh,x,y,xp,hh;
  FXint iconoffx,iconoffy,textoffx,textoffy;
  FXIcon *icon;
  FXDCWindow dc(this,event);
  eylo=event->rect.y-itemHeight;
  eyhi=event->rect.y+event->rect.h;
  th=font->getFontHeight();
  dc.setForeground(backColor);
  dc.fillRectangle(event->rect.x,event->rect.y,event->rect.w,event->rect.h);
  dc.setTextFont(font);
  hh=(itemHeight/2)&~1;   // Got to be even for the dotted lines!
  while(n){
    x=n->x+pos_x;
    y=n->y+pos_y;
    if(eylo<=y && y<eyhi){
      
      // Show as open or closed
      if(n->state&OLDTREEITEM_OPEN)
        icon=n->openIcon;
      else
        icon=n->closedIcon;
      
      // Show the icon
      if(icon){
        iconoffx=x+(SIDE_SPACING/2);
        iconoffy=y+(LINE_SPACING/2)+(itemHeight-icon->getHeight())/2;
        dc.drawIcon(icon,iconoffx,iconoffy);
        }
      
      // Show the text
      if(n->label.text()){
        textoffx=x+(SIDE_SPACING/2)+TEXT_SPACING+iconWidth;
        textoffy=y+(LINE_SPACING/2)+(itemHeight-th)/2;
        tw=font->getTextWidth(n->label.text(),n->label.length());
        if(n->state&OLDTREEITEM_SELECTED){
          if(hasFocus() && (n->state&OLDTREEITEM_CURRENT)){
            dc.setForeground(selbackColor);
            dc.fillRectangle(textoffx+1,textoffy+1,tw,th-1);
            drawFocusRectangle(dc,textoffx,textoffy,tw+2,th+1);
            }
          else{
            dc.setForeground(selbackColor);
            dc.fillRectangle(textoffx,textoffy,tw+2,th+1);
            }
          dc.setForeground(seltextColor);
          dc.drawText(textoffx+1,textoffy+font->getFontAscent(),n->label.text(),n->label.length());
          }
        else{
          if(n->state&OLDTREEITEM_CURRENT){
            drawFocusRectangle(dc,textoffx,textoffy,tw+2,th+1);
            }
          dc.setForeground(textColor);
          dc.drawText(textoffx+1,textoffy+font->getFontAscent(),n->label.text(),n->label.length());
          }
        }
      
      // Show other paraphernalia such as dotten lines and expand-boxes
      if((options&(OLDTREELIST_SHOWS_LINES|OLDTREELIST_SHOWS_BOXES)) && (n->parent || (options&OLDTREELIST_ROOT_BOXES))){
        yh=y+hh;
        xh=x-indent+(SIDE_SPACING/2)+iconWidth/2;
        dc.setForeground(lineColor);
        dc.setDashes(0,onoff,2);
        if(options&OLDTREELIST_SHOWS_LINES){                   // Connect items with lines
          FXOldTreeItem* p=n->parent;
          xp=xh;
          dc.setLineStyle(LINE_ONOFF_DASH);
          while(p){
            xp-=indent;
            if(p->next) dc.drawLine(xp,y,xp,y+itemHeight);
            p=p->parent;
            }
          if((options&OLDTREELIST_SHOWS_BOXES) && ((n->state&OLDTREEITEM_HASITEMS) || n->first)){
            if(n->prev || n->parent) dc.drawLine(xh,y,xh,yh-4);
            if(n->next) dc.drawLine(xh,yh+4,xh,y+itemHeight);
            }
          else{
            if(n->prev || n->parent) dc.drawLine(xh,y,xh,yh);
            if(n->next) dc.drawLine(xh,yh,xh,y+itemHeight);
            dc.drawLine(xh,yh,x+(SIDE_SPACING/2)-2,yh);
            }
          dc.setLineStyle(LINE_SOLID);
          }
        
        // Boxes before items for expand/collapse of item
        if((options&OLDTREELIST_SHOWS_BOXES) && ((n->state&OLDTREEITEM_HASITEMS) || n->first)){
          dc.setLineStyle(LINE_ONOFF_DASH);
          dc.drawLine(xh+4,yh,x+(SIDE_SPACING/2)-2,yh);
          dc.setLineStyle(LINE_SOLID);
          dc.drawLine(xh-4,yh-4,xh+4,yh-4);
          dc.drawLine(xh-4,yh+4,xh+4,yh+4);
          dc.drawLine(xh+4,yh-4,xh+4,yh+4);
          dc.drawLine(xh-4,yh-4,xh-4,yh+4);
          dc.setForeground(textColor);
          dc.drawLine(xh-2,yh,xh+2,yh);
          if(!(options&OLDTREELIST_AUTOSELECT) && !(n->state&OLDTREEITEM_EXPANDED)){
            dc.drawLine(xh,yh-2,xh,yh+2);
            }
          }
        }
      }
    
    // Move on to the next item
    if(n->first && ((options&OLDTREELIST_AUTOSELECT)||(n->state&OLDTREEITEM_EXPANDED))){n=n->first;continue;}
    while(!n->next && n->parent){n=n->parent;}
    n=n->next;
    }
  return 1;
  }


// We were asked about tip text
long FXOldTreeList::onQueryTip(FXObject* sender,FXSelector,void*){
  FXint x,y; FXuint state;
  if((flags&FLAG_TIP) && !(options&OLDTREELIST_AUTOSELECT)){   // No tip when autoselect!
    getCursorPosition(x,y,state);
    FXTRACE((250,"%s::onQueryTip %08x (%d,%d)\n",getClassName(),this,x,y));
    FXOldTreeItem *item=getItemAt(x,y);
    if(item){
      FXString string=item->label;
      sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&string);
      return 1;
      }
    }
  return 0;
  }


// We were asked about status text
long FXOldTreeList::onQueryHelp(FXObject* sender,FXSelector,void*){
  if(help.text() && (flags&FLAG_HELP)){
    FXTRACE((200,"%s::onQueryHelp %08x\n",getClassName(),this));
    sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&help);
    return 1;
    }
  return 0;
  }


// Extend selection
FXbool FXOldTreeList::extendSelection(FXOldTreeItem* item){
  FXbool changes=FALSE;
  if(item && anchoritem){
      
    // Browse select mode
    if((options&OLDTREELIST_BROWSESELECT) && !(options&OLDTREELIST_SINGLESELECT)){
      changes|=deselectItemRange(firstitem,getItemAbove(item));
      changes|=deselectItemRange(getItemBelow(item),NULL);
      changes|=selectItem(item);
      }

    // Extended number selected
    else if(!(options&(OLDTREELIST_SINGLESELECT|OLDTREELIST_BROWSESELECT))){
      if(item==anchoritem){
        changes|=restoreItemRange(firstitem,getItemAbove(anchoritem));
        changes|=restoreItemRange(getItemBelow(anchoritem),NULL);
        }
      else if(item->y < anchoritem->y){
        changes|=restoreItemRange(firstitem,getItemAbove(item));
        changes|=restoreItemRange(getItemBelow(anchoritem),NULL);
        if(anchoritem->state&OLDTREEITEM_SELECTED){
          changes|=selectItemRange(item,getItemAbove(anchoritem));
          }
        else{
          changes|=deselectItemRange(item,getItemAbove(anchoritem));
          }
        }
      else{
        changes|=restoreItemRange(firstitem,getItemAbove(anchoritem));
        changes|=restoreItemRange(getItemBelow(item),NULL);
        if(anchoritem->state&OLDTREEITEM_SELECTED){
          changes|=selectItemRange(getItemBelow(anchoritem),item);
          }
        else{
          changes|=deselectItemRange(getItemBelow(anchoritem),item);
          }
        }
      }
    }
  return changes;
  }
  
  
// Get item at position x,y (NULL if none)
FXOldTreeItem* FXOldTreeList::getItemAt(FXint x,FXint y) const {
  FXOldTreeItem* n=firstitem;
  FXint yy=0;
  y=y-pos_y;
  x=x-pos_x;
  while(n){
    if(y<yy) return NULL;
    yy+=itemHeight;
    if(y<yy){
      if((x<n->x-indent) && !(options&OLDTREELIST_AUTOSELECT)) return NULL;
      return n;
      }
    if(n->first && ((options&OLDTREELIST_AUTOSELECT)||(n->state&OLDTREEITEM_EXPANDED))){n=n->first;continue;}
    while(!n->next && n->parent){n=n->parent;}
    n=n->next;
    }
  return NULL;
  }


// Make item fully visible
void FXOldTreeList::makeItemVisible(FXOldTreeItem* item){
  FXint x,y,w,h;
  if(!xid) return;
  if(item){
    
    // Expand parents of this node
    if(!(options&OLDTREELIST_AUTOSELECT)){
      FXOldTreeItem *par=item->parent;
      FXbool expanded=FALSE;
      while(par){
        if(!(par->state&OLDTREEITEM_EXPANDED)){
          par->state|=OLDTREEITEM_EXPANDED;
          expanded=TRUE;
          }
        par=par->parent;
        }
      
      // If any nodes have expanded that weren't previously, recompute list size
      if(expanded){
        recalc();
        layout();
        }
      }
      
    // Now we adjust the scrolled position to fit everything
    x=pos_x;
    y=pos_y;

    w=iconWidth+TEXT_SPACING+SIDE_SPACING+font->getTextWidth(item->label.text(),item->label.length());
    h=itemHeight;

    if(viewport_w<=x+item->x+w) x=viewport_w-item->x-w;
    if(x+item->x<=0) x=-item->x; 

    if(viewport_h<=y+item->y+h) y=viewport_h-item->y-h;
    if(y+item->y<=0) y=-item->y; 

    setPosition(x,y);
    }
  }


// Add item as first one under parent p
FXOldTreeItem* FXOldTreeList::addItemFirst(FXOldTreeItem* p,const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr){
  FXOldTreeItem *item=createItem();
  if(p){
    item->prev=NULL;
    item->next=p->first;
    if(item->next) item->next->prev=item; else p->last=item;
    p->first=item;
    }
  else{
    item->prev=NULL;
    item->next=firstitem;
    if(item->next) item->next->prev=item; else lastitem=item;
    firstitem=item;
    }
  item->parent=p;
  item->first=NULL;
  item->last=NULL;
  item->label=text;
  item->openIcon=oi;
  item->closedIcon=ci;
  item->state=0;
  item->data=ptr;
  item->x=0;
  item->y=0;
  recalc();
  return item;
  }


// Add item as last one under parent p
FXOldTreeItem* FXOldTreeList::addItemLast(FXOldTreeItem* p,const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr){
  FXOldTreeItem *item=createItem();
  if(p){
    item->prev=p->last;
    item->next=NULL;
    if(item->prev) item->prev->next=item; else p->first=item;
    p->last=item;
    }
  else{
    item->prev=lastitem;
    item->next=NULL;
    if(item->prev) item->prev->next=item; else firstitem=item;
    lastitem=item;
    }
  item->parent=p;
  item->first=NULL;
  item->last=NULL;
  item->label=text;
  item->openIcon=oi;
  item->closedIcon=ci;
  item->state=0;
  item->data=ptr;
  item->x=0;
  item->y=0;
  recalc();
  return item;
  }


// Link item after other
FXOldTreeItem* FXOldTreeList::addItemAfter(FXOldTreeItem* other,const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr){
  if(other==NULL){ fxerror("%s::addItemAfter: other item is NULL.\n",getClassName()); }
  FXOldTreeItem *item=createItem();
  item->prev=other;
  item->next=other->next;
  other->next=item;
  if(item->next) item->next->prev=item; else if(other->parent) other->parent->last=item; else lastitem=item; 
  item->parent=other->parent;
  item->first=NULL;
  item->last=NULL;
  item->label=text;
  item->openIcon=oi;
  item->closedIcon=ci;
  item->state=0;
  item->data=ptr;
  item->x=0;
  item->y=0;
  recalc();
  return item;
  }


// Link item before other 
FXOldTreeItem* FXOldTreeList::addItemBefore(FXOldTreeItem* other,const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr){
  if(other==NULL){ fxerror("%s::addItemBefore: other item is NULL.\n",getClassName()); }
  FXOldTreeItem *item=createItem();
  item->next=other;
  item->prev=other->prev;
  other->prev=item;
  if(item->prev) item->prev->next=item; else if(other->parent) other->parent->first=item; else firstitem=item; 
  item->parent=other->parent;
  item->first=NULL;
  item->last=NULL;
  item->label=text;
  item->openIcon=oi;
  item->closedIcon=ci;
  item->state=0;
  item->data=ptr;
  item->x=0;
  item->y=0;
  recalc();
  return item;
  }


// Remove node from list
void FXOldTreeList::removeItem(FXOldTreeItem* item){
  if(item){
    if(item->prev) item->prev->next=item->next; else if(item->parent) item->parent->first=item->next; else firstitem=item->next;
    if(item->next) item->next->prev=item->prev; else if(item->parent) item->parent->last=item->prev; else lastitem=item->prev;
    if(currentitem==item) currentitem=NULL;
    if(anchoritem==item) anchoritem=NULL;
    removeItems(item->first,item->last);
    deleteItem(item);
    recalc();
    }
  }


// Remove all siblings from [fm,to]
void FXOldTreeList::removeItems(FXOldTreeItem* fm,FXOldTreeItem* to){
  if(fm && to){
    FXOldTreeItem *item;
    if(fm->prev) fm->prev->next=to->next; else if(fm->parent) fm->parent->first=to->next; else firstitem=to->next;
    if(to->next) to->next->prev=fm->prev; else if(to->parent) to->parent->last=fm->prev; else lastitem=fm->prev;
    do{
      item=fm;
      if(currentitem==item) currentitem=NULL;
      if(anchoritem==item) anchoritem=NULL;
      removeItems(item->first,item->last);
      fm=fm->next;
      deleteItem(item);
      }
    while(item!=to);
    recalc();
    }
  }


// Remove all items
void FXOldTreeList::removeAllItems(){
  removeItems(firstitem,lastitem);
  }


// Set current item
void FXOldTreeList::setCurrentItem(FXOldTreeItem* item){
  if(item!=currentitem){ 
    if(item){
      item->state|=OLDTREEITEM_CURRENT;
      updateItem(item);
      }
    if(currentitem){
      currentitem->state&=~OLDTREEITEM_CURRENT;
      updateItem(currentitem);
      }
    currentitem=item;
    }
  }


// Set anchor item
void FXOldTreeList::setAnchorItem(FXOldTreeItem* item){
  anchoritem=item;
  }


// Check item is open
FXbool FXOldTreeList::isItemOpen(const FXOldTreeItem* item) const { 
  if(!item){ fxerror("%s::isItemOpen: item is NULL.\n",getClassName()); } 
  return (item->state&OLDTREEITEM_OPEN)!=0; 
  }


// Open item
FXbool FXOldTreeList::openItem(FXOldTreeItem* item){
  if(item==NULL){ fxerror("%s::openItem: item is NULL.\n",getClassName()); }
  if(!(item->state&OLDTREEITEM_OPEN)){
    item->state|=OLDTREEITEM_OPEN;
    updateItem(item);
    return TRUE;
    }
  return FALSE;
  }  
  
 
// Close item
FXbool FXOldTreeList::closeItem(FXOldTreeItem* item){
  if(item==NULL){ fxerror("%s::closeItem: item is NULL.\n",getClassName()); }
  if(item->state&OLDTREEITEM_OPEN){
    item->state&=~OLDTREEITEM_OPEN;
    updateItem(item);
    return TRUE;
    }
  return FALSE;
  }  


// Collapse all subtrees under item
FXbool FXOldTreeList::collapseTree(FXOldTreeItem* tree){
  if(tree==NULL){ fxerror("%s::collapseTree: tree is NULL.\n",getClassName()); }
  if(tree->state&OLDTREEITEM_EXPANDED){
    tree->state&=~OLDTREEITEM_EXPANDED;
    if(!(options&OLDTREELIST_AUTOSELECT)){     // In autoselect, already shown as expanded!
      if(tree->first){
        recalc();
        }
      else{
        updateItem(tree);
        }
      }
    return TRUE;
    }
  return FALSE;
  }

 
// Expand subtree under item
FXbool FXOldTreeList::expandTree(FXOldTreeItem* tree){
  if(tree==NULL){ fxerror("%s::expandTree: tree is NULL.\n",getClassName()); }
  if(!(tree->state&OLDTREEITEM_EXPANDED)){
    tree->state|=OLDTREEITEM_EXPANDED;
    if(!(options&OLDTREELIST_AUTOSELECT)){     // In autoselect, already shown as expanded!
      if(tree->first){
        recalc();
        }
      else{
        updateItem(tree);
        }
      }
    return TRUE;
    }
  return FALSE;
  }


void FXOldTreeList::setItemText(FXOldTreeItem* item,const FXString& text){
  if(item==NULL){ fxerror("%s::setItemText: item is NULL.\n",getClassName()); }
  item->label=text;
  recalc();
  }


FXString FXOldTreeList::getItemText(const FXOldTreeItem* item) const {
  if(item==NULL){ fxerror("%s::getItemText: item is NULL.\n",getClassName()); }
  return item->label;
  }


void FXOldTreeList::setItemOpenIcon(FXOldTreeItem* item,FXIcon* icon){
  if(item==NULL){ fxerror("%s::setItemOpenIcon: item is NULL.\n",getClassName()); }
  item->openIcon=icon;
  updateItem(item);
  }


FXIcon* FXOldTreeList::getItemOpenIcon(const FXOldTreeItem* item) const {
  if(item==NULL){ fxerror("%s::getItemOpenIcon: item is NULL.\n",getClassName()); }
  return item->openIcon;
  }


void FXOldTreeList::setItemClosedIcon(FXOldTreeItem* item,FXIcon* icon){
  if(item==NULL){ fxerror("%s::setItemClosedIcon: item is NULL.\n",getClassName()); }
  item->closedIcon=icon;
  updateItem(item);
  }


FXIcon* FXOldTreeList::getItemClosedIcon(const FXOldTreeItem* item) const {
  if(item==NULL){ fxerror("%s::getItemClosedIcon: item is NULL.\n",getClassName()); }
  return item->closedIcon;
  }


void FXOldTreeList::setItemData(FXOldTreeItem* item,void* ptr) const {
  if(item==NULL){ fxerror("%s::setItemData: item is NULL.\n",getClassName()); }
  item->data=ptr;
  }


void* FXOldTreeList::getItemData(const FXOldTreeItem* item) const {
  if(item==NULL){ fxerror("%s::getItemData: item is NULL.\n",getClassName()); }
  return item->data;
  }


// Find toplevel item by prefix of length len
FXOldTreeItem* FXOldTreeList::findItem(const FXString& text,FXuint len) const {
  register FXOldTreeItem* item;
  for(item=firstitem; item; item=item->next){
    if(equal(text,item->label,len)) return item;
    }
  return NULL;
  }


// Find child item by prefix of length len
FXOldTreeItem* FXOldTreeList::findChildItem(FXOldTreeItem* parentitem,const FXString& text,FXuint len) const {
  register FXOldTreeItem* item;
  if(parentitem==NULL){ fxerror("%s::findChildItem: parent is NULL.\n",getClassName()); }
  for(item=parentitem->first; item; item=item->next){
    if(equal(text,item->label,len)) return item;
    }
  return NULL;
  }


// Key Press
long FXOldTreeList::onKeyPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXOldTreeItem *item,*tmp;
  FXDragAction action;
  flags&=~FLAG_TIP;
  if(!isEnabled()) return 0;
  if(target && target->handle(this,MKUINT(message,SEL_KEYPRESS),ptr)) return 1;
  switch(event->code){
    case KEY_Up:
    case KEY_Down:
    case KEY_Right:
    case KEY_Left:
    case KEY_Home:
    case KEY_End:
      item=currentitem;
      switch(event->code){
        case KEY_Up:                      // Move up
          if(!item){ 
            item=lastitem;
            }
          else if(item->prev){
            item=item->prev; 
            while(item->first && (item->state&OLDTREEITEM_EXPANDED)) item=item->last;
            }
          else if(item->parent){ 
            item=item->parent;
            }
          break;
        case KEY_Down:                    // Move down
          if(!item){
            item=firstitem;
            }
          else if(item->first && (item->state&OLDTREEITEM_EXPANDED)){
            item=item->first;
            }
          else if(item->next){
            item=item->next;
            }
          else{
            tmp=item;
            while(item->parent && !item->parent->next) item=item->parent;
            if(item->parent && item->parent->next)
              item=item->parent->next;
            else
              item=tmp;
            }
          break;
        case KEY_Right:                   // Move right/down and open subtree
          if(!item){
            item=firstitem;
            }
          else if(!isItemExpanded(item)){
            handle(this,MKUINT(0,SEL_EXPANDED),item);
            }
          else if(item->first){
            item=item->first;
            }
          else if(item->next){
            item=item->next;
            }
          else{
            tmp=item;
            while(item->parent && !item->parent->next) item=item->parent;
            if(item->parent && item->parent->next)
              item=item->parent->next;
            else
              item=tmp;
            }
          break;
        case KEY_Left:                    // Move left/up and close subtree
          if(!item){
            item=firstitem;
            }
          else if(isItemExpanded(item)){
            handle(this,MKUINT(0,SEL_COLLAPSED),item);
            }
          else if(item->parent){
            item=item->parent;
            }
          else if(item->prev){
            item=item->prev;
            }
          break;
        case KEY_Home: 
          item=firstitem; 
          break;
        case KEY_End:  
          item=lastitem; 
          break;
        }
      if(item && item!=currentitem){
        handle(this,MKUINT(0,SEL_CHANGED),(void*)item);
        if((options&OLDTREELIST_BROWSESELECT) && !(options&OLDTREELIST_SINGLESELECT)){
          handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
          }
        flags&=~FLAG_UPDATE;
        flags|=FLAG_KEY;
        }
      return 1;
    case KEY_space:
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      flags&=~FLAG_UPDATE;
      flags|=FLAG_KEY;
      return 1;
    case KEY_KP_Enter:
    case KEY_Return:
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      flags&=~FLAG_UPDATE;
      flags|=FLAG_KEY;
      return 1;
    case KEY_Page_Up:
      setPosition(pos_x,pos_y+verticalScrollbar()->getPage());
      return 1;
    case KEY_Page_Down:
      setPosition(pos_x,pos_y-verticalScrollbar()->getPage());
      return 1;
    case KEY_Shift_L:
    case KEY_Shift_R:
      flags|=FLAG_KEY;
      if(isDragging()){
        action=DRAG_MOVE;
        if(didAccept()!=DRAG_REJECT){
          setDragCursor(getApp()->dndMoveCursor);
          }
        else{
          setDragCursor(getApp()->dontdropCursor);
          }
        }
      return 1;
    case KEY_Control_L:
    case KEY_Control_R:
      flags|=FLAG_KEY;
      if(isDragging()){
        action=DRAG_COPY;
        if(didAccept()!=DRAG_REJECT){
          setDragCursor(getApp()->dndCopyCursor);
          }
        else{
          setDragCursor(getApp()->dontdropCursor);
          }
        }
      return 1;
    case KEY_Alt_L:
    case KEY_Alt_R:
      if(isDragging()){
        action=DRAG_LINK;
        if(didAccept()!=DRAG_REJECT){
          setDragCursor(getApp()->dndLinkCursor);
          }
        else{
          setDragCursor(getApp()->dontdropCursor);
          }
        }
      return 1;
    }
  return 0;
  }


// Key Release 
long FXOldTreeList::onKeyRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXDragAction action;
  if(!isEnabled()) return 0;
  if(target && target->handle(this,MKUINT(message,SEL_KEYRELEASE),ptr)) return 1;
  switch(event->code){
    case KEY_Up:
    case KEY_Down:
    case KEY_Right:
    case KEY_Left:
    case KEY_Home:
    case KEY_End:
      if(flags&FLAG_KEY){
        if((options&OLDTREELIST_BROWSESELECT) && !(options&OLDTREELIST_SINGLESELECT)){
          handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
          handle(this,MKUINT(0,SEL_CLICKED),(void*)currentitem);
          }
        flags&=~FLAG_KEY;
        }
      return 1;
    case KEY_space:
      handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
      handle(this,MKUINT(0,SEL_CLICKED),(void*)currentitem);
      flags&=~FLAG_KEY;
      return 1;
    case KEY_KP_Enter:
    case KEY_Return:
      handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
      handle(this,MKUINT(0,SEL_DOUBLECLICKED),currentitem);
      flags&=~FLAG_KEY;
      return 1;
    case KEY_Page_Up:
    case KEY_Page_Down:
      return 1;
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Control_L:
    case KEY_Control_R:
      flags&=~FLAG_KEY;
    case KEY_Alt_L:
    case KEY_Alt_R:
      if(isDragging()){
        action=DRAG_COPY;
        if(isDropTarget()) action=DRAG_MOVE;
        if(didAccept()!=DRAG_REJECT){
          if(action==DRAG_MOVE)
            setDragCursor(getApp()->dndMoveCursor);
          else
            setDragCursor(getApp()->dndCopyCursor);
          }
        else{
          setDragCursor(getApp()->dontdropCursor);
          }
        }
      return 1;
    }
  return 0;
  }


// We timed out, i.e. the user didn't move for a while
long FXOldTreeList::onTipTimer(FXObject*,FXSelector,void*){
  FXTRACE((200,"%s::onTipTimer %08x\n",getClassName(),this));
  timer=NULL;
  flags|=FLAG_TIP;
  return 1;
  }


// Scroll timer
long FXOldTreeList::onAutoScroll(FXObject* sender,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXOldTreeItem *item;
  FXint xx,yy;
  
  // In autoselect mode, stop scrolling when mouse outside window
  if(options&OLDTREELIST_AUTOSELECT){
  
    // Scroll window
    FXScrollArea::onAutoScroll(sender,sel,ptr);

    // Validated position
    xx=event->win_x; if(xx<0) xx=0; else if(xx>=viewport_w) xx=viewport_w-1;
    yy=event->win_y; if(yy<0) yy=0; else if(yy>=viewport_h) yy=viewport_h-1;
    
    // Find item
    item=getItemAt(xx,yy);

    // Got item and different from last time
    if(item && item!=currentitem){

      // Make it the current item
      handle(this,MKUINT(0,SEL_CHANGED),(void*)item);

      // Extend the selection to it
      extendSelection(item);
      }
    return 1;
    }
  
  // Drag and drop mode
  if(flags&FLAG_DODRAG){
    
    // Scroll the content
    FXScrollArea::onAutoScroll(sender,sel,ptr);
    
    // Content scrolled, so perhaps something else under cursor
    handle(this,MKUINT(0,SEL_DRAGGED),ptr);
  
    return 1;
    }
  
  return 0;
  }


// Mouse motion
long FXOldTreeList::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXOldTreeItem *item;
  
  // In autoselect, change to item under cursor
  if(options&OLDTREELIST_AUTOSELECT){
    
    // Start auto scrolling?
    if(startAutoScroll(event->win_x,event->win_y,FALSE)) return 1;

    // Find item
    item=getItemAt(event->win_x,event->win_y);

    // Got an item different from before
    if(item && item!=currentitem){
      
      // Make it the current item
      handle(this,MKUINT(0,SEL_CHANGED),(void*)item);

      // In autoselect, pretend we pressed it
      if(options&OLDTREELIST_AUTOSELECT){
        handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
        }

      // Extend the selection to it
      else{
        extendSelection(item);
        }
      }
    return 1;
    }

  // Drag and drop mode
  if(flags&FLAG_DODRAG){
    flags&=~FLAG_TIP;

    // Start auto scrolling?
    if(startAutoScroll(event->win_x,event->win_y,TRUE)) return 1;

    handle(this,MKUINT(0,SEL_DRAGGED),ptr);
    return 1;
    }

  // Tentative drag and drop
  if(flags&FLAG_TRYDRAG){
    flags&=~FLAG_TIP;
    if(event->moved){
      flags&=~FLAG_TRYDRAG;
      if(handle(this,MKUINT(0,SEL_BEGINDRAG),ptr)){
        flags|=FLAG_DODRAG;
        }
      }
    return 1;
    }


  // Reset tip timer
  if(timer) getApp()->removeTimeout(timer);
  timer=getApp()->addTimeout(getApp()->menuPause,this,ID_TIPTIMER);

  // Movement kills tip
  if(flags&FLAG_TIP){
    flags&=~FLAG_TIP;
    return 1;
    }
  
  // No change
  return 0;
  }


// Pressed a button
long FXOldTreeList::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXOldTreeItem *item;
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    if(options&OLDTREELIST_AUTOSELECT) return 1;
    item=getItemAt(event->win_x,event->win_y);
    if(item && hitItemBox(item,event->win_x,event->win_y)){
      if(isItemExpanded(item))
        handle(this,MKUINT(0,SEL_COLLAPSED),(void*)item);
      else
        handle(this,MKUINT(0,SEL_EXPANDED),(void*)item);
      return 1;
      }
    if(event->click_count==1){
      if(item!=currentitem){
        handle(this,MKUINT(0,SEL_CHANGED),(void*)item);
        }
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      if(item){
        flags|=(FLAG_PRESSED|FLAG_TRYDRAG);
        }
      }
    flags|=FLAG_PRESSED;
    flags&=~FLAG_UPDATE;
    return 1;
    }
  return 0;
  }


// Released button
long FXOldTreeList::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXuint flg=flags;
  if(isEnabled()){
    ungrab();
    flags|=FLAG_UPDATE;
    flags&=~(FLAG_PRESSED|FLAG_TRYDRAG);
    stopAutoScroll();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    // The button press is implied in autoselect mode, but the release isn't
    if(!(flg&FLAG_PRESSED) && !(options&OLDTREELIST_AUTOSELECT)) return 1;       
    if(event->click_count==1){
      if(flags&FLAG_DODRAG){
        handle(this,MKUINT(0,SEL_ENDDRAG),ptr);
        handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
        flags&=~FLAG_DODRAG;
        }
      else{
        handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
        handle(this,MKUINT(0,SEL_CLICKED),(void*)currentitem);
        }
      }
    else if(event->click_count==2){
      handle(this,MKUINT(0,SEL_DOUBLECLICKED),(void*)currentitem);
      }
    else if(event->click_count==3){
      handle(this,MKUINT(0,SEL_TRIPLECLICKED),(void*)currentitem);
      }
    return 1;
    }
  return 0;
  }


// The widget lost the grab for some reason
long FXOldTreeList::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onUngrabbed(sender,sel,ptr);
  flags&=~(FLAG_DODRAG|FLAG_LASSO|FLAG_TRYDRAG|FLAG_PRESSED|FLAG_CHANGED);
  flags|=FLAG_UPDATE;
  stopAutoScroll();
  return 1;
  }


// Button or Key activate
long FXOldTreeList::onActivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  register FXOldTreeItem *item=firstitem;
  FXDragType types[2];
  
  // Remember previous state
  while(item){
    if(item->state&OLDTREEITEM_SELECTED)
      item->state|=OLDTREEITEM_HISTORY;
    else
      item->state&=~OLDTREEITEM_HISTORY;
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }

  // Have currentitem
  if(currentitem){
    
    // Multiple selection mode
    if((options&OLDTREELIST_SINGLESELECT)&&(options&OLDTREELIST_BROWSESELECT)){
      if(isItemSelected(currentitem)){
        deselectItem(currentitem);
        }
      else{
        selectItem(currentitem);
        }
      //markItems();
      setAnchorItem(currentitem);
      }

    // Browse select mode
    else if(options&OLDTREELIST_BROWSESELECT){
      deselectItemRange(firstitem,getItemAbove(currentitem));
      deselectItemRange(getItemBelow(currentitem),NULL);
      selectItem(currentitem);
      setAnchorItem(currentitem);
      }

    // Single selection mode
    else if(options&OLDTREELIST_SINGLESELECT){
      if(isItemSelected(currentitem)){
        deselectItemRange(firstitem,NULL);
        }
      else{
        deselectItemRange(firstitem,getItemAbove(currentitem));
        deselectItemRange(getItemBelow(currentitem),NULL);
        selectItem(currentitem);
        }
      setAnchorItem(currentitem);
      }

    // Add selection mode
    else if(event->state&SHIFTMASK){
      if(anchoritem){
        selectItem(anchoritem);
        markItems();
        extendSelection(currentitem);
        }
      else{
        setAnchorItem(currentitem);
        selectItem(currentitem);
        //markItems();
        }
      }
    
//     // Toggle selection mode
//     else if(event->state&CONTROLMASK){
//       setAnchorItem(currentitem);
//       toggleItem(currentitem);
//       markItems();
//       }
    
    // Extended selection mode
    else if(!(event->state&CONTROLMASK)){
      if(!isItemSelected(currentitem)){
        deselectItemRange(firstitem,getItemAbove(currentitem));
        deselectItemRange(getItemBelow(currentitem),NULL);
        selectItem(currentitem);
        }
      //markItems();
      setAnchorItem(currentitem);
      }
    }
  
  // No currentitem & not multiple or browse select mode
  else if(!(options&OLDTREELIST_BROWSESELECT)){
    
    // If not add or toggle mode, turn them off
    if(!(event->state&(SHIFTMASK|CONTROLMASK))){
      deselectItemRange(firstitem,NULL);
      }
    setAnchorItem(currentitem);
    }
  
  // If we want selection, see wether to acquire or release it
  if(options&OLDTREELIST_WANTSELECTION){
    if(getNumSelectedItems()){
      types[0]=stringType;
      types[1]=textType;
      acquireSelection(types,2);
      }
    else{
      releaseSelection();
      }
    }
  return 1;
  }
  

// Button or Key deactivate
long FXOldTreeList::onDeactivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  setAnchorItem(currentitem);
  
  // Toggle selection mode
  if(!event->moved && (event->state&CONTROLMASK) && currentitem){
    if(!(options&(OLDTREELIST_SINGLESELECT|OLDTREELIST_BROWSESELECT))){
      toggleItem(currentitem);
      }
    }
  return 1;
  }



// Current item changed
long FXOldTreeList::onChanged(FXObject*,FXSelector,void* ptr){
  FXOldTreeItem *olditem=currentitem;
  makeItemVisible((FXOldTreeItem*)ptr);
  setCurrentItem((FXOldTreeItem*)ptr);
  if(target && target->handle(this,MKUINT(message,SEL_CHANGED),ptr)) return 1;
  if(olditem){
    handle(this,MKUINT(0,SEL_CLOSED),olditem);
    }
  if(currentitem){
    handle(this,MKUINT(0,SEL_OPENED),currentitem);
    }
  return 1;
  }


// Selected items
long FXOldTreeList::onSelected(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_SELECTED),ptr);
  }


// Deselected items
long FXOldTreeList::onDeselected(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DESELECTED),ptr);
  }


// Clicked in list
long FXOldTreeList::onClicked(FXObject*,FXSelector,void* ptr){
  FXOldTreeItem **selectedlist,**deselectedlist,*item;
  FXint numselected,numdeselected,i,j;
  
  // Notify target of the click
  if(target){
    
    // Clicked message indicates a click anywhere in the widget
    if(target->handle(this,MKUINT(message,SEL_CLICKED),ptr)) return 1;
  
    // Command message indicates click on an item
    if(ptr && target->handle(this,MKUINT(message,SEL_COMMAND),ptr)) return 1;
    }

  // Find out number of items whose selection state changed
  item=firstitem; numselected=numdeselected=0;
  while(item){
    if((item->state&OLDTREEITEM_SELECTED) && !(item->state&OLDTREEITEM_HISTORY)) numselected++;
    if(!(item->state&OLDTREEITEM_SELECTED) && (item->state&OLDTREEITEM_HISTORY)) numdeselected++;
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }

  // Make some room
  FXMALLOC(&selectedlist,FXOldTreeItem*,numselected+1);
  FXMALLOC(&deselectedlist,FXOldTreeItem*,numdeselected+1);

  // Add items to the proper lists
  item=firstitem; i=j=0;
  while(item){
    if((item->state&OLDTREEITEM_SELECTED) && !(item->state&OLDTREEITEM_HISTORY)) selectedlist[i++]=item;
    if(!(item->state&OLDTREEITEM_SELECTED) && (item->state&OLDTREEITEM_HISTORY)) deselectedlist[j++]=item;
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }

  // Close off lists
  selectedlist[i]=NULL;
  deselectedlist[j]=NULL;

  // Tell the target about the newly selected items
  handle(this,MKUINT(0,SEL_SELECTED),selectedlist);

  // Tell the target about the newly deselected items
  handle(this,MKUINT(0,SEL_DESELECTED),deselectedlist);

  // Free the lists
  FXFREE(&selectedlist);
  FXFREE(&deselectedlist);
  return 1;
  }


// Double clicked in list; ptr may or may not point to an item
long FXOldTreeList::onDoubleClicked(FXObject*,FXSelector,void* ptr){
  
  // Double click anywhere in the widget
  if(target && target->handle(this,MKUINT(message,SEL_DOUBLECLICKED),ptr)) return 1;
  
  // Double click on an item
  if(ptr){
    
    // Expand the item 
    if(isItemExpanded((FXOldTreeItem*)ptr)){
      if(handle(this,MKUINT(0,SEL_COLLAPSED),ptr)) return 1;
      }
    else{
      if(handle(this,MKUINT(0,SEL_EXPANDED),ptr)) return 1;
      }
    }
  return 0;
  }


// Triple clicked in list; ptr may or may not point to an item
long FXOldTreeList::onTripleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_TRIPLECLICKED),ptr);
  }


// Item opened
long FXOldTreeList::onItemOpened(FXObject*,FXSelector,void* ptr){
  openItem((FXOldTreeItem*)ptr);
  return target && target->handle(this,MKUINT(message,SEL_OPENED),ptr);
  }


// Item closed
long FXOldTreeList::onItemClosed(FXObject*,FXSelector,void* ptr){
  closeItem((FXOldTreeItem*)ptr);
  return target && target->handle(this,MKUINT(message,SEL_CLOSED),ptr);
  }


// Item expanded
long FXOldTreeList::onItemExpanded(FXObject*,FXSelector,void* ptr){
  expandTree((FXOldTreeItem*)ptr);
  return target && target->handle(this,MKUINT(message,SEL_EXPANDED),ptr);
  }


// Item collapsed
long FXOldTreeList::onItemCollapsed(FXObject*,FXSelector,void* ptr){
  collapseTree((FXOldTreeItem*)ptr);
  return target && target->handle(this,MKUINT(message,SEL_COLLAPSED),ptr);
  }


// Change the font
void FXOldTreeList::setFont(FXFont* fnt){
  if(!fnt){ fxerror("%s::setFont: NULL font specified.\n",getClassName()); }
  font=fnt;
  recalc();
  }


// Change help text
void FXOldTreeList::setHelpText(const FXString& text){
  help=text;
  }


// Set text color
void FXOldTreeList::setTextColor(FXColor clr){
  if(clr!=textColor){
    textColor=clr;
    update();
    }
  }


// Set select background color
void FXOldTreeList::setSelBackColor(FXColor clr){
  if(clr!=selbackColor){
    selbackColor=clr;
    update();
    }
  }


// Set selected text color
void FXOldTreeList::setSelTextColor(FXColor clr){
  if(clr!=seltextColor){
    seltextColor=clr;
    update();
    }
  }


// Set line color
void FXOldTreeList::setLineColor(FXColor clr){
  if(clr!=lineColor){
    lineColor=clr;
    update();
    }
  }


// Set icon width
void FXOldTreeList::setIconWidth(FXint w){
  if(iconWidth!=w){
    iconWidth=w;
    recalc();
    }
  }


// Set icon height
void FXOldTreeList::setIconHeight(FXint h){
  if(iconHeight!=h){
    iconHeight=h;
    recalc();
    }
  }


// Set parent to child indent amount
void FXOldTreeList::setIndent(FXint in){
  if(indent!=in){
    indent=in;
    recalc();
    }
  }


// Change list style
void FXOldTreeList::setListStyle(FXuint style){
  FXuint opts=(options&~OLDTREELIST_MASK) | (style&OLDTREELIST_MASK);
  if(options!=opts){
    options=opts;
    recalc();
    }
  }


// Get list style
FXuint FXOldTreeList::getListStyle() const { 
  return (options&OLDTREELIST_MASK); 
  }


// Save data
void FXOldTreeList::save(FXStream& store) const {
  FXScrollArea::save(store);
  store << font;
  store << textColor;
  store << selbackColor;
  store << seltextColor;
  store << lineColor;
  store << iconWidth;
  store << iconHeight;
  store << itemWidth;
  store << itemHeight;
  store << indent;
  store << help;
  }


// Load data
void FXOldTreeList::load(FXStream& store){ 
  FXScrollArea::load(store);
  store >> font;
  store >> textColor;
  store >> selbackColor;
  store >> seltextColor;
  store >> lineColor;
  store >> iconWidth;
  store >> iconHeight;
  store >> itemWidth;
  store >> itemHeight;
  store >> indent;
  store >> help;
  }


// Cleanup
FXOldTreeList::~FXOldTreeList(){
  removeAllItems();
  if(timer) getApp()->removeTimeout(timer);
  firstitem=(FXOldTreeItem*)-1;
  lastitem=(FXOldTreeItem*)-1;
  anchoritem=(FXOldTreeItem*)-1;
  currentitem=(FXOldTreeItem*)-1;
  font=(FXFont*)-1;
  timer=(FXTimer*)-1;
  }


