/********************************************************************************
*                                                                               *
*                           S t r i n g   O b j e c t                           *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXString.cpp,v 1.2 1999/09/27 21:38:06 jeroen Exp $                      *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"


/*
  Notes:
  - It should be safe to compare NULL strings.
  - When comparing NULL is smallest possible string value.
  - Always FXRESIZE() in size() might be safer.
*/



// No string will ever get less than this size; this should reduce memory
// fragmentation during frequent string operations.  Note that this size is
// probably large enough to fit 99% of all strings in a typical GUI...
// Do not make this smaller than 16!
#define MINSIZE 16


/*******************************************************************************/

// Simple construct
FXString::FXString(){
  str=NULL;
  }


// Copy construct
FXString::FXString(const FXString& s){
  str=NULL;
  if(s.str){
    register FXint sz=s.size();
    FXMALLOC(&str,FXchar,FXMAX(sz,MINSIZE));
    memcpy(str,s.str,sz);
    }
  }


// Construct and init
FXString::FXString(const FXchar* s){
  str=NULL;
  if(s){
    register FXint sz=strlen(s)+1;
    FXMALLOC(&str,FXchar,FXMAX(sz,MINSIZE));
    memcpy(str,s,sz);
    }
  }


// Construct and init with substring
FXString::FXString(const FXchar* s,FXint n){
  str=NULL;
  if(s){
    FXMALLOC(&str,FXchar,FXMAX((n+1),MINSIZE));
    strncpy(str,s,n);
    str[n]=0;
    }
  }


// Construct and fill with constant
FXString::FXString(FXchar c,FXint n){
  FXMALLOC(&str,FXchar,FXMAX((n+1),MINSIZE));
  memset(str,c,n);
  str[n]=0;
  }


// Construct string from two parts
FXString::FXString(const FXchar *s1,const FXchar* s2){
  register FXint sz1,sz2,sz;
  str=NULL;
  if(s1 || s2){
    sz1=s1?strlen(s1):0;
    sz2=s2?strlen(s2):0;
    sz=sz1+sz2;
    FXMALLOC(&str,FXchar,FXMAX((sz+1),MINSIZE));
    memcpy(str,s1,sz1);
    memcpy(&str[sz1],s2,sz2);
    str[sz]=0;
    }
  }


// Size to capacity 
void FXString::size(FXint sz){
  if(sz==0){                        // Sized to zero length means ditch the buffer
    FXFREE(&str);
    }
  else if(!str){                    // Allocate a buffer as we didn't have one
    FXMALLOC(&str,FXchar,FXMAX(sz,MINSIZE));
    }
  else if(sz>MINSIZE){              // No string will ever be less than MINSIZE; so this is pretty efficient!
    FXRESIZE(&str,FXchar,sz);
    }
  }


// Extract partition of delimiter separated string
FXString FXString::extract(FXint part,FXchar delim) const {
  FXString res;
  if(str){
    register const FXchar *s=str;
    while(s[0] && part){part-=(s[0]==delim);s++;}
    if(s[0]){
      register FXint sz=0;
      while(s[sz] && s[sz]!=delim) sz++;
      if(sz){
        FXMALLOC(&res.str,FXchar,FXMAX((sz+1),MINSIZE));
        memcpy(res.str,s,sz);
        res.str[sz]='\0';
        }
      }
    }
  return res;
  }


// Extract partition of string, interpreting escapes
FXString FXString::extract(FXint part,FXchar delim,FXchar esc) const {
  FXString res;
  if(str){
    register const FXchar *s=str;
    while(s[0] && part){part-=(s[0]==delim);s++;}
    if(s[0]){
      register FXint sz=0;
      while(s[sz] && s[sz]!=delim) sz++;
      if(sz){
        register FXint i,j;
        FXMALLOC(&res.str,FXchar,FXMAX((sz+1),MINSIZE));
        for(i=j=0; j<sz; j++){
          if(s[j]==esc){ if(s[j+1]!=esc) continue; j++; }
          res.str[i++]=s[j];
          }
        res.str[i]='\0';
        }
      }
    }
  return res;
  }


// Assignment
FXString& FXString::operator=(const FXString& s){
  if(this!=&s){
    if(s.str){
      register FXint sz=s.size();
      size(sz);
      memcpy(str,s.str,sz);
      }
    else{
      FXFREE(&str);
      }
    }
  return *this;
  }


// Assign a string
FXString& FXString::operator=(const FXchar* s){
  if(s){
    register FXint sz=strlen(s)+1;
    size(sz);
    memcpy(str,s,sz);
    }
  else{
    FXFREE(&str);
    }
  return *this;
  }


// Size of text array
FXint FXString::size() const {
  return str ? strlen(str)+1 : 0;
  }


// Length of text only
FXint FXString::length() const { 
  return str ? strlen(str) : 0; 
  }


// Equality up to a length
FXbool equal(const FXString &s1,const FXString &s2,FXint len){
  return (s1.str==s2.str) || (s1.str && s2.str && strncmp(s1.str,s2.str,len)==0);
  }

FXbool equal(const FXString &s1,const char *s2,FXint len){
  return (s1.str==s2) || (s1.str && s2 && strncmp(s1.str,s2,len)==0);
  }

FXbool equal(const FXchar *s1,const FXString &s2,FXint len){
  return (s1==s2.str) || (s1 && s2.str && strncmp(s1,s2.str,len)==0);
  }


// Equality 
FXbool operator==(const FXString &s1,const FXString &s2){ 
  return (s1.str==s2.str) || (s1.str && s2.str && strcmp(s1.str,s2.str)==0); 
  }

FXbool operator==(const FXString &s1,const FXchar *s2){ 
  return (s1.str==s2) || (s1.str && s2 && strcmp(s1.str,s2)==0); 
  }

FXbool operator==(const FXchar *s1,const FXString &s2){ 
  return (s1==s2.str) || (s1 && s2.str && strcmp(s1,s2.str)==0); 
  }


// Inequality 
FXbool operator!=(const FXString &s1,const FXString &s2){ 
  return !((s1.str==s2.str) || (s1.str && s2.str && strcmp(s1.str,s2.str)==0));
  }

FXbool operator!=(const FXString &s1,const FXchar *s2){ 
  return !((s1.str==s2) || (s1.str && s2 && strcmp(s1.str,s2)==0));
  }

FXbool operator!=(const FXchar *s1,const FXString &s2){ 
  return !((s1==s2.str) || (s1 && s2.str && strcmp(s1,s2.str)==0));
  }


// Less than
FXbool operator<(const FXString &s1,const FXString &s2){ 
  return (!s1.str && s2.str) || (s1.str && s2.str && (strcmp(s1.str,s2.str)<0)); 
  }

FXbool operator<(const FXString &s1,const FXchar *s2){ 
  return (!s1.str && s2) || (s1.str && s2 && (strcmp(s1.str,s2)<0)); 
  }

FXbool operator<(const FXchar *s1,const FXString &s2){ 
  return (!s1 && s2.str) || (s1 && s2.str && (strcmp(s1,s2.str)<0)); 
  }


// Less than or equal
FXbool operator<=(const FXString &s1,const FXString &s2){ 
  return (!s1.str) || (s1.str && s2.str && (strcmp(s1.str,s2.str)<=0)); 
  }

FXbool operator<=(const FXString &s1,const FXchar *s2){ 
  return (!s1.str) || (s1.str && s2 && (strcmp(s1.str,s2)<=0)); 
  }

FXbool operator<=(const FXchar *s1,const FXString &s2){ 
  return (!s1) || (s1 && s2.str && (strcmp(s1,s2.str)<=0)); 
  }


// Greater than 
FXbool operator>(const FXString &s1,const FXString &s2){ 
  return (s1.str && !s2.str) || (s1.str && s2.str && (strcmp(s1.str,s2.str)>0)); 
  }

FXbool operator>(const FXString &s1,const FXchar *s2){ 
  return (s1.str && !s2) || (s1.str && s2 && (strcmp(s1.str,s2)>0)); 
  }

FXbool operator>(const FXchar *s1,const FXString &s2){ 
  return (s1 && !s2.str) || (s1 && s2.str && (strcmp(s1,s2.str)>0)); 
  }


// Greater than or equal
FXbool operator>=(const FXString &s1,const FXString &s2){ 
  return (!s2.str) || (s1.str && s2.str && (strcmp(s1.str,s2.str)>=0)); 
  }

FXbool operator>=(const FXString &s1,const FXchar *s2){ 
  return (!s2) || (s1.str && s2 && (strcmp(s1.str,s2)>=0)); 
  }

FXbool operator>=(const FXchar *s1,const FXString &s2){ 
  return (!s2.str) || (s1 && s2.str && (strcmp(s1,s2.str)>=0)); 
  }


// Append FXString
FXString& FXString::operator+=(const FXString& s){
  register FXint len,n;
  if(s.str){
    n=strlen(s.str);
    if(n){
      len=length();
      size(len+n+1);
      memcpy(&str[len],s.str,n);
      str[len+n]=0;
      }
    }
  return *this;
  }


// Append string
FXString& FXString::operator+=(const FXchar* s){
  register FXint len,n;
  if(s){
    n=strlen(s);
    if(n){
      len=length();
      size(len+n+1);
      memcpy(&str[len],s,n);
      str[len+n]=0;
      }
    }
  return *this;
  }


// Concatenate two FXStrings
FXString operator+(const FXString& s1,const FXString& s2){
  return FXString(s1.str,s2.str);
  }


// Concatenate FXString and string
FXString operator+(const FXString& s1,const FXchar* s2){
  return FXString(s1.str,s2);
  }


// Concatenate string and FXString
FXString operator+(const FXchar* s1,const FXString& s2){
  return FXString(s1,s2.str);
  }


// Fill with a constant
FXString& FXString::fill(FXchar c,FXint len){
  size(len+1);
  memset(str,c,len);
  str[len]=0;
  return *this;
  }


// Fill up to current length
FXString& FXString::fill(FXchar c){
  if(str){memset(str,c,strlen(str));}
  return *this;
  }


// Insert string at position
FXString& FXString::insert(FXint pos,const FXString& s){
  if(s.str){insert(pos,s.str,strlen(s.str));}
  return *this;
  }


// Insert string at position
FXString& FXString::insert(FXint pos,const FXchar* s){
  if(s){insert(pos,s,strlen(s));}
  return *this;
  }


// Insert string at position
FXString& FXString::insert(FXint pos,const FXchar* s,FXint n){
  register FXint len=length();
  if(len<=pos){
    size(pos+n+1);
    memset(&str[len],' ',pos-len);    // Fill with spaces
    memcpy(&str[pos],s,n);
    str[pos+n]=0;
    }
  else{
    size(len+n+1);
    memmove(&str[pos+n],&str[pos],len-pos);
    memcpy(&str[pos],s,n);
    str[len+n]=0;
    }
  return *this;
  }


// Add string to the end
FXString& FXString::append(const FXString& s){
  if(s.str){append(s.str,strlen(s.str));}
  return *this;
  }


// Add string to the end
FXString& FXString::append(const FXchar *s){
  if(s){append(s,strlen(s));}
  return *this;
  }


// Add string to the end
FXString& FXString::append(const FXchar *s,FXint n){
  register FXint len=length();
  size(len+n+1);
  memcpy(&str[len],s,n);
  str[len+n]=0;
  return *this;
  }


// Prepend string
FXString& FXString::prepend(const FXString& s){
  if(s.str){prepend(s.str,strlen(s.str));}
  return *this;
  }


// Prepend string
FXString& FXString::prepend(const FXchar *s){
  if(s){prepend(s,strlen(s));}
  return *this;
  }


// Prepend string
FXString& FXString::prepend(const FXchar *s,FXint n){
  register FXint len=length();
  size(len+n+1);
  memmove(&str[n],str,len);
  memcpy(str,s,n);
  str[len+n]=0;
  return *this;
  }
  

// Replace part of string
FXString& FXString::replace(FXint pos,FXint len,const FXString& s){
  if(s.str){replace(pos,len,s.str,strlen(s.str));}
  return *this;
  }
  
  
// Replace part of string
FXString& FXString::replace(FXint pos,FXint len,const FXchar *s){
  if(s){replace(pos,len,s,strlen(s));}
  return *this;
  }


// Replace part of string
FXString& FXString::replace(FXint pos,FXint len,const FXchar *s,FXint n){
  register FXint ll=length();
  if(ll<=pos){
    size(pos+n+1);
    memset(&str[ll],' ',pos-ll);      // Fill with spaces
    memcpy(&str[pos],s,n);
    str[pos+n]=0;
    }
  else{
    if(pos+len>ll) len=ll-pos;
    size(ll-len+n+1);
    if(n!=len) memmove(&str[pos+n],&str[pos+len],ll-pos-len);
    memcpy(&str[pos],s,n);
    str[ll-len+n]=0;
    }
  return *this;
  }
  

// Remove section from buffer
FXString& FXString::remove(FXint pos,FXint n){
  register FXint len=length();
  if(pos<len){
    if(pos+n<len){
      memmove(&str[pos],&str[pos+n],len-pos-n+1);
      }
    else{
      str[pos]=0;
      }
    }
  return *this;
  }


// Remove leading and trailing whitespace
FXString& FXString::trim(){
  if(str && str[0]){
    register FXint s=0;
    register FXint e=length();
    while(0<e && isspace(str[e-1])) e--;
    while(s<e && isspace(str[s])) s++;
    if(s<e) memmove(str,&str[s],e-s);
    str[e-s]=0;
    }
  return *this;
  }  


// Remove leading whitespace
FXString& FXString::trimBegin(){
  if(str && str[0]){
    register FXint s=0;
    register FXint x=0;
    while(isspace(str[s])) s++;
    while(str[s]) str[x++]=str[s++];
    str[x]=0;
    }
  return *this;
  }  


// Remove trailing whitespace
FXString& FXString::trimEnd(){
  if(str && str[0]){
    register FXint e=length();
    while(0<e && isspace(str[e-1])) e--;
    str[e]=0;
    }
  return *this;
  }


// Truncate string
FXString& FXString::trunc(FXint pos){
  if(str && pos<strlen(str)) str[pos]=0;
  return *this;
  }


// Clean string
FXString& FXString::clear(){
  if(str) str[0]=0;
  return *this;
  }


// Get leftmost part
FXString FXString::left(FXint n) const {
  return FXString(str,n);
  }


// Get rightmost part
FXString FXString::right(FXint n) const {
  register FXint len=length();
  if(n>len) n=len;
  return FXString(str+len-n,n);
  }


// Get some part in the middle
FXString FXString::mid(FXint pos,FXint n) const {
  register FXint len=length();
  if(pos>len) pos=len;
  if(pos+n>len) n=len-pos;
  return FXString(str+pos,n);
  }


// Return all characters before the nth occurrence of ch,
// or entire string if not found
FXString FXString::before(FXchar ch,FXint n) const {
  register FXint len=-1;
  if(str){
    do{ n-=(str[++len]==ch); }while(0<n && str[len]);
    return FXString(str,len);
    }
  return FXString(NULL);
  }


// Return all characters after the nth occurrence of ch,
// or empty string if not found
FXString FXString::after(FXchar ch,FXint n) const {
  register FXint len=0;
  if(str){
    while(0<n && str[len]){ if(str[len++]==ch) n--; }
    return FXString(&str[len]);
    }
  return FXString(NULL);
  }


// Convert to lower case
FXString& FXString::lower(){
  register FXchar *p=str;
  if(p){while(*p){*p=tolower(*p);p++;}}
  return *this;
  }


// Convert to upper case
FXString& FXString::upper(){
  register FXchar *p=str;
  if(p){while(*p){*p=toupper(*p);p++;}}
  return *this;
  }


// Find a character, searching forward; return position or -1
FXint FXString::findf(FXchar c,FXint pos) const {
  if(0<=pos){
    register FXint len=length();
    while(pos<len){ if(str[pos]==c){ return pos; } pos++; }
    }
  return -1;
  }


// Find a character, searching backward; return position or -1
FXint FXString::findb(FXchar c,FXint pos) const {
  if(0<=pos){
    register FXint len=length();
    if(pos>=len) pos=len-1;
    while(0<=pos){ if(str[pos]==c){ return pos; } pos--; }
    }
  return -1;
  }


// Find number of occurances of character in string
FXint FXString::count(FXchar c) const {
  FXint n=0;
  if(str){for(FXchar *ptr=str; *ptr; ptr++){n+=(c==*ptr);}}
  return n;
  }


// Get hash value
FXint FXString::hash() const {
  register const FXchar *s=str;
  register FXint h=0;
  register FXint g;
  while(*s) {
    h=(h<<4)+*s++;
    g=h&0xF0000000UL;
    if(g) h^=g>>24;
    h&=~g;
    }
  return h;
  }
  

// Save
FXStream& operator<<(FXStream& store,const FXString& s){
  FXint len=s.size();
  store << len;
  store.save(s.str,len);
  return store;
  }


// Load
FXStream& operator>>(FXStream& store,FXString& s){
  FXint len;
  store >> len;
  s.size(len+1);
  store.load(s.str,len);
  s.str[len]='\0';
  return store;
  }


// Format
FXString& FXString::format(const char *fmt,...){
  register FXint len;
  va_list arguments;
  va_start(arguments,fmt);
#ifdef _GNU_SOURCE
  register FXint sz=256;
  size(sz);
  while((len=vsnprintf(str,sz,fmt,arguments))<0){
    sz*=2;
    FXRESIZE(&str,FXchar,sz);
    }
#else
  size(1024);
  len=vsprintf(str,fmt,arguments);
  FXASSERT(len<1024);
#endif
  size(len+1);
  va_end(arguments);
  return *this;
  }


// Format a string a-la printf
FXString FXStringFormat(const FXchar *fmt,...){
  FXchar buf[1024];
  va_list arguments;
  va_start(arguments,fmt);
#ifdef _GNU_SOURCE
  vsnprintf(buf,sizeof(buf),fmt,arguments);
#else
  vsprintf(buf,fmt,arguments);
#endif
  va_end(arguments);
  return FXString(buf);
  }


// Conversion of integer to string
FXString FXStringVal(FXint num){
  FXchar buf[20];
#ifdef _GNU_SOURCE
  snprintf(buf,sizeof(buf),"%d",num);
#else
  sprintf(buf,"%d",num);
#endif
  return FXString(buf);
  }

// Conversion of unsigned integer to string
FXString FXStringVal(FXuint num){
  FXchar buf[20];
#ifdef _GNU_SOURCE
  snprintf(buf,sizeof(buf),"%u",num);
#else
  sprintf(buf,"%u",num);
#endif
  return FXString(buf);
  }


// Conversion of float to string
FXString FXStringVal(FXfloat num,FXint prec,FXExponent exp){
  const char *const expo[]={"%.*f","%.*G","%.*E"};
  FXchar buf[100];
#ifdef _GNU_SOURCE
  snprintf(buf,sizeof(buf),expo[exp],prec,num);
#else
  sprintf(buf,expo[exp],prec,num);
#endif
  return FXString(buf);
  }


// Conversion of double to string
FXString FXStringVal(FXdouble num,FXint prec,FXExponent exp){
  const char *const expo[]={"%.*f","%.*G","%.*E"};
  FXchar buf[100];
#ifdef _GNU_SOURCE
  snprintf(buf,sizeof(buf),expo[exp],prec,num);
#else
  sprintf(buf,expo[exp],prec,num);
#endif
  return FXString(buf);
  }


// Conversion of string to integer
FXint FXIntVal(const FXString& s){
  if(s.str){ return strtol(s.str,NULL,10); }
  return 0;
  }

// Conversion of string to unsigned integer
FXuint FXUIntVal(const FXString& s){
  if(s.str){ return strtoul(s.str,NULL,10); }
  return 0;
  }


// Conversion of string to float
FXfloat FXFloatVal(const FXString& s){
  if(s.str){ return (FXfloat)strtod(s.str,NULL); }
  return 0;
  }


// Conversion of string to double
FXdouble FXDoubleVal(const FXString& s){
  if(s.str){ return strtod(s.str,NULL); }
  return 0;
  }


// Delete
FXString::~FXString(){
  FXFREE(&str);
  str=(FXchar*)-1;
  }


