/********************************************************************************
*                                                                               *
*                            W i n d o w   O b j e c t                          *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* Major Contributions for Windows NT by Lyle Johnson                            *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXWindow.cpp,v 1.35 1999/11/17 07:31:08 jeroen Exp $                     *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXObjectList.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXVisual.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXCursor.h"
#include "FXDrawable.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXComposite.h"
#include "FXRootWindow.h"
#include "FXShell.h"
#include "FXTopWindow.h"
#include "FXMainWindow.h"

/*
 Notes:
  - Look at destroy(); cannot call overload from base class's destructor!
  - When window is disabled, it should loose focus
  - When no subclass handles SEL_SELECTION_REQUEST, send back None property here
  - Add URL jump text also.
  - Update should only happen if widget is not in some sort of transaction.
  - Not every widget needs help and tip data; move this down to buttons etc.
  - Default constructors [for serialization] should initialize dynamic member 
    variables same as regular constructor.  Dynamic variables are those that 
    are not saved during serialization.
  - If FLAG_DIRTY gets reset at the END of layout(), it will be safe to have
    show() and hide() to call recalc(), in case they get called from the
    application code.
  - Need some sort of journal facility.
  - Use INCR mechanism if it gets large.
  - Change: passModalEvents() should not rely on FXPopups only; perhaps a flag?
  - Perhaps should post CLIPBOARD type list in advance also, just like windows insists.
    We will need to keep this list some place [perhaps same place as XDND] and clean it
    after we're done [This would imply the CLIPBOARD type list would be the same as the
    XDND type list, which is probably OK].
*/

#ifndef FX_NATIVE_WIN32

// Basic events
#define BASIC_EVENT_MASK   (ExposureMask|PropertyChangeMask|EnterWindowMask|LeaveWindowMask|KeyPressMask|KeyReleaseMask)

// Additional events for shell widget events
#define SHELL_EVENT_MASK   (FocusChangeMask|StructureNotifyMask)

// Additional events for enabled widgets        
#define ENABLED_EVENT_MASK (ButtonPressMask|ButtonReleaseMask|PointerMotionMask)

// These events are grabbed for mouse grabs
#define GRAB_EVENT_MASK    (ButtonPressMask|ButtonReleaseMask|PointerMotionMask|EnterWindowMask|LeaveWindowMask)

// Do not propagate mask
#define NOT_PROPAGATE_MASK (KeyPressMask|KeyReleaseMask|ButtonPressMask|ButtonReleaseMask|PointerMotionMask|ButtonMotionMask)

#endif


// Side layout modes
#define LAYOUT_SIDE_MASK (LAYOUT_SIDE_LEFT|LAYOUT_SIDE_RIGHT|LAYOUT_SIDE_TOP|LAYOUT_SIDE_BOTTOM)


// Layout modes
#define LAYOUT_MASK (LAYOUT_SIDE_MASK|LAYOUT_RIGHT|LAYOUT_CENTER_X|LAYOUT_BOTTOM|LAYOUT_CENTER_Y|LAYOUT_FIX_X|LAYOUT_FIX_Y|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT|LAYOUT_FILL_X|LAYOUT_FILL_Y)


/*******************************************************************************/

// Map
FXDEFMAP(FXWindow) FXWindowMap[]={
  FXMAPFUNC(SEL_UPDATE,0,FXWindow::onUpdate),
  FXMAPFUNC(SEL_PAINT,0,FXWindow::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXWindow::onMotion),
  FXMAPFUNC(SEL_DRAGGED,0,FXWindow::onDragged),
  FXMAPFUNC(SEL_ENTER,0,FXWindow::onEnter),
  FXMAPFUNC(SEL_LEAVE,0,FXWindow::onLeave),
  FXMAPFUNC(SEL_DESTROY,0,FXWindow::onDestroy),
  FXMAPFUNC(SEL_FOCUSIN,0,FXWindow::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXWindow::onFocusOut),
  FXMAPFUNC(SEL_SELECTION_LOST,0,FXWindow::onSelectionLost),
  FXMAPFUNC(SEL_SELECTION_GAINED,0,FXWindow::onSelectionGained),
  FXMAPFUNC(SEL_SELECTION_REQUEST,0,FXWindow::onSelectionRequest),
  FXMAPFUNC(SEL_CLIPBOARD_LOST,0,FXWindow::onClipboardLost),
  FXMAPFUNC(SEL_CLIPBOARD_GAINED,0,FXWindow::onClipboardGained),
  FXMAPFUNC(SEL_CLIPBOARD_REQUEST,0,FXWindow::onClipboardRequest),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXWindow::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXWindow::onLeftBtnRelease),
  FXMAPFUNC(SEL_MIDDLEBUTTONPRESS,0,FXWindow::onMiddleBtnPress),
  FXMAPFUNC(SEL_MIDDLEBUTTONRELEASE,0,FXWindow::onMiddleBtnRelease),
  FXMAPFUNC(SEL_RIGHTBUTTONPRESS,0,FXWindow::onRightBtnPress),
  FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,0,FXWindow::onRightBtnRelease),
  FXMAPFUNC(SEL_UNGRABBED,0,FXWindow::onUngrabbed),
  FXMAPFUNC(SEL_KEYPRESS,0,FXWindow::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXWindow::onKeyRelease),
  FXMAPFUNC(SEL_DND_ENTER,0,FXWindow::onDNDEnter),
  FXMAPFUNC(SEL_DND_LEAVE,0,FXWindow::onDNDLeave),
  FXMAPFUNC(SEL_DND_DROP,0,FXWindow::onDNDDrop),
  FXMAPFUNC(SEL_DND_MOTION,0,FXWindow::onDNDMotion),
  FXMAPFUNC(SEL_DND_REQUEST,0,FXWindow::onDNDRequest),
  FXMAPFUNC(SEL_FOCUS_SELF,0,FXWindow::onFocusSelf),
  FXMAPFUNC(SEL_BEGINDRAG,0,FXWindow::onBeginDrag),
  FXMAPFUNC(SEL_ENDDRAG,0,FXWindow::onEndDrag),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_TOGGLESHOWN,FXWindow::onUpdToggleShown),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_DELETE,FXWindow::onUpdYes),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SHOW,FXWindow::onCmdShow),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_HIDE,FXWindow::onCmdHide),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_TOGGLESHOWN,FXWindow::onCmdToggleShown),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_RAISE,FXWindow::onCmdRaise),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_LOWER,FXWindow::onCmdLower),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_ENABLE,FXWindow::onCmdEnable),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_DISABLE,FXWindow::onCmdDisable),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_UPDATE,FXWindow::onCmdUpdate),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_DELETE,FXWindow::onCmdDelete),
  };


// Object implementation
FXIMPLEMENT(FXWindow,FXDrawable,FXWindowMap,ARRAYNUMBER(FXWindowMap))
  
  
/*******************************************************************************/

  
// Drag type names
const FXchar FXWindow::deleteTypeName[]="DELETE";
const FXchar FXWindow::textTypeName[]="text/plain";
const FXchar FXWindow::colorTypeName[]="application/x-color";


// Drag type atoms; first widget to need it should register the type
FXDragType FXWindow::deleteType=0;
FXDragType FXWindow::textType=0;
FXDragType FXWindow::colorType=0;


/*******************************************************************************/

  
// For deserialization
FXWindow::FXWindow(){
  parent=(FXWindow*)-1;
  owner=(FXWindow*)-1;
  shell=(FXWindow*)-1;
  first=(FXWindow*)-1;
  last=(FXWindow*)-1;
  next=(FXWindow*)-1;
  prev=(FXWindow*)-1;
  focus=NULL;
  defaultCursor=(FXCursor*)-1;
  dragCursor=(FXCursor*)-1;
  accelTable=(FXAccelTable*)-1;
  target=NULL;
  message=0;
  userData=NULL;
  xpos=0;
  ypos=0;
  backColor=0;
  flags=FLAG_DIRTY|FLAG_UPDATE|FLAG_RECALC;
  options=0;
  }


// Only used for the root window
FXWindow::FXWindow(FXApp* a,FXVisual *vis):FXDrawable(a,1,1){
  visual=vis;
  parent=NULL;
  owner=NULL;
  shell=this;
  first=last=NULL;
  next=prev=NULL;
  focus=NULL;
  defaultCursor=getApp()->arrowCursor;
  dragCursor=getApp()->arrowCursor;
  accelTable=NULL;
  target=NULL;
  message=0;
  userData=NULL;
  xpos=0;
  ypos=0;
  backColor=0;
  flags=FLAG_DIRTY|FLAG_SHOWN|FLAG_UPDATE|FLAG_RECALC;
  options=LAYOUT_FIX_X|LAYOUT_FIX_Y|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT;
  }


// This constructor is used for shell windows
FXWindow::FXWindow(FXApp* a,FXWindow* own,FXuint opts,FXint x,FXint y,FXint w,FXint h):FXDrawable(a,w,h){
  parent=a->root;
  shell=this;
  owner=own;
  visual=getApp()->getDefaultVisual();
  first=last=NULL;
  prev=parent->last;
  next=NULL;
  parent->last=this;
  if(prev) prev->next=this; else parent->first=this;
  focus=NULL;
  defaultCursor=getApp()->arrowCursor;
  dragCursor=getApp()->arrowCursor;
  accelTable=NULL;
  target=NULL;
  message=0;
  userData=NULL;
  xpos=x;
  ypos=y;
  backColor=getApp()->baseColor;
  flags=FLAG_DIRTY|FLAG_UPDATE|FLAG_RECALC;
  options=opts;
  }


// This constructor is used for all child windows
FXWindow::FXWindow(FXComposite* p,FXuint opts,FXint x,FXint y,FXint w,FXint h):FXDrawable(p->getApp(),w,h){
  parent=p;
  owner=parent;
  shell=parent->getShell();
  visual=parent->getVisual();
  first=last=NULL;
  prev=parent->last;
  next=NULL;
  parent->last=this;
  if(prev) prev->next=this; else parent->first=this;
  focus=NULL;
  defaultCursor=getApp()->arrowCursor;
  dragCursor=getApp()->arrowCursor;
  accelTable=NULL;
  target=NULL;
  message=0;
  userData=NULL;
  xpos=x;
  ypos=y;
  backColor=getApp()->baseColor;
  flags=FLAG_DIRTY|FLAG_UPDATE|FLAG_RECALC;
  options=opts;
  }


/*******************************************************************************/


// Save data
void FXWindow::save(FXStream& store) const {
  FXDrawable::save(store);
  store << parent;
  store << owner;
  store << shell;
  store << first;
  store << last;
  store << next;
  store << prev;
  store << focus;
  store << defaultCursor;
  store << dragCursor;
  store << accelTable;
  store << target;
  store << message;
  store << xpos;
  store << ypos;
  store << backColor;
  store << options;
  }


// Load data
void FXWindow::load(FXStream& store){ 
  FXDrawable::load(store);
  store >> parent;
  store >> owner;
  store >> shell;
  store >> first;
  store >> last;
  store >> next;
  store >> prev;
  store >> focus;
  store >> defaultCursor;
  store >> dragCursor;
  store >> accelTable;
  store >> target;
  store >> message;
  store >> xpos;
  store >> ypos;
  store >> backColor;
  store >> options;
  }


/*******************************************************************************/


// Handle repaint 
long FXWindow::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent *ev=(FXEvent*)ptr;
  FXDCWindow dc(this,ev);
  dc.setForeground(backColor);
  dc.fillRectangle(ev->rect.x,ev->rect.y,ev->rect.w,ev->rect.h);
  return 1;
  }


// Handle destroy event
long FXWindow::onDestroy(FXObject*,FXSelector,void*){
  FXTRACE((250,"%s::onDestroy %08x\n",getClassName(),this));
#ifndef FX_NATIVE_WIN32
  XDeleteContext(getApp()->display,xid,getApp()->wcontext);
#else
  SetWindowLong((HWND)xid,0,0);
#endif
  if(getApp()->grabWindow==this) getApp()->grabWindow=NULL;
  flags&=~(FLAG_SHOWN|FLAG_FOCUSED);
  xid=0;
  return 1;
  }


// Mouse motion
long FXWindow::onMotion(FXObject*,FXSelector,void*){
  return 0;
  }


// Handle incoming button events; generally, we give the target a 
// first shot at it; if it doesn't handle it, we'll bounce it around
// as an activate/deactivate event to this widget, after we
// set the focus to this widget!
long FXWindow::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    return handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
    }
  return 0;
  }

long FXWindow::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    return handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
    }
  return 0;
  }

long FXWindow::onMiddleBtnPress(FXObject*,FXSelector,void* ptr){
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_MIDDLEBUTTONPRESS),ptr)) return 1;
    return 1;
    }
  return 0;
  }

long FXWindow::onMiddleBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    if(target && target->handle(this,MKUINT(message,SEL_MIDDLEBUTTONRELEASE),ptr)) return 1;
    return 1;
    }
  return 0;
  }

long FXWindow::onRightBtnPress(FXObject*,FXSelector,void* ptr){
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_RIGHTBUTTONPRESS),ptr)) return 1;
    return 1;
    }
  return 0;
  }

long FXWindow::onRightBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    if(target && target->handle(this,MKUINT(message,SEL_RIGHTBUTTONRELEASE),ptr)) return 1;
    return handle(this,MKUINT(ID_QUERY_MENU,SEL_COMMAND),ptr);
    }
  return 0;
  }


// Keyboard press
long FXWindow::onKeyPress(FXObject* sender,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;

  // Trace message
  FXTRACE((200,"%s::onKeyPress %08x keysym=0x%04x state=%04x\n",getClassName(),this,event->code,event->state));
  
  // Try bounce to the target first
  if(!getFocus() && isEnabled() && target && target->handle(this,MKUINT(message,SEL_KEYPRESS),ptr)) return 1;
  
  // Bounce to focus widget
  if(getFocus() && getFocus()->handle(sender,sel,ptr)) return 1;
  
  // Next, check the accelerators...
  if(getAccelTable() && getAccelTable()->handle(this,sel,ptr)) return 1;
  
  // Otherwise, perform the default keyboard processing
  switch(event->code){
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Control_L:
    case KEY_Control_R:
    case KEY_Caps_Lock:
    case KEY_Shift_Lock:
    case KEY_Meta_L:
    case KEY_Meta_R:
    case KEY_Alt_L:
    case KEY_Alt_R:
    case KEY_Super_L:
    case KEY_Super_R:
    case KEY_Hyper_L:
    case KEY_Hyper_R:
    case KEY_Scroll_Lock:
    case KEY_Sys_Req:
      return 1;
    case KEY_Tab:
      if(event->state&SHIFTMASK) goto prv;
    case KEY_Next:
      return handle(this,MKUINT(0,SEL_FOCUS_NEXT),ptr);
    case KEY_Prior: 
    case KEY_ISO_Left_Tab:
prv:  return handle(this,MKUINT(0,SEL_FOCUS_PREV),ptr);
    case KEY_Up:    
      return handle(this,MKUINT(0,SEL_FOCUS_UP),ptr);
    case KEY_Down:  
      return handle(this,MKUINT(0,SEL_FOCUS_DOWN),ptr);
    case KEY_Left:  
      return handle(this,MKUINT(0,SEL_FOCUS_LEFT),ptr);
    case KEY_Right: 
      return handle(this,MKUINT(0,SEL_FOCUS_RIGHT),ptr);
    case KEY_Home:
      return handle(this,MKUINT(0,SEL_FOCUS_HOME),ptr);
    case KEY_End:
      return handle(this,MKUINT(0,SEL_FOCUS_END),ptr);
    case KEY_space:
    case KEY_KP_Space:
      return isEnabled() && handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
    }
  return 0;
  }


// Keyboard release
long FXWindow::onKeyRelease(FXObject* sender,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  
  // Trace message
  FXTRACE((200,"%s::onKeyRelease %08x keysym=0x%04x state=%04x\n",getClassName(),this,event->code,event->state));
  
  // Try bounce to the target first
  if(!getFocus() && isEnabled() && target && target->handle(this,MKUINT(message,SEL_KEYRELEASE),ptr)) return 1;
  
  // Bounce to focus widget
  if(getFocus() && getFocus()->handle(sender,sel,ptr)) return 1;
  
  // Next, check the accelerators...
  if(getAccelTable() && getAccelTable()->handle(this,sel,ptr)) return 1;
  
  // Otherwise, perform the routine keyboard processing
  switch(event->code){
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Control_L:
    case KEY_Control_R:
    case KEY_Caps_Lock:
    case KEY_Shift_Lock:
    case KEY_Meta_L:
    case KEY_Meta_R:
    case KEY_Alt_L:
    case KEY_Alt_R:
    case KEY_Super_L:
    case KEY_Super_R:
    case KEY_Hyper_L:
    case KEY_Hyper_R:
    case KEY_Scroll_Lock:
    case KEY_Sys_Req:
      return 1;
    case KEY_space:
    case KEY_KP_Space:
      return isEnabled() && handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
    }
  return 0;
  }
  

// Start a drag operation
long FXWindow::onBeginDrag(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_BEGINDRAG),ptr);
  }


// End drag operation
long FXWindow::onEndDrag(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_ENDDRAG),ptr);
  }


// Dragged stuff around
long FXWindow::onDragged(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_DRAGGED),ptr);
  }


// Entering window
long FXWindow::onEnter(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXTRACE((250,"%s::onEnter %08x\n",getClassName(),this));
  if(event->code!=CROSSINGGRAB){
    getApp()->cursorWindow=this;
    if(!(event->state&(SHIFTMASK|CONTROLMASK|LEFTBUTTONMASK|MIDDLEBUTTONMASK|RIGHTBUTTONMASK))) flags|=FLAG_TIP;
    flags|=FLAG_HELP;
    }
  if(isEnabled()){
    if(target) target->handle(this,MKUINT(message,SEL_ENTER),ptr);
    }
  return 1;
  }


// Leaving window
long FXWindow::onLeave(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXTRACE((250,"%s::onLeave %08x\n",getClassName(),this));
  if(event->code!=CROSSINGUNGRAB){
    getApp()->cursorWindow=parent;
    flags&=~(FLAG_TIP|FLAG_HELP);
    }
  if(isEnabled()){
    if(target) target->handle(this,MKUINT(message,SEL_LEAVE),ptr);
    }
  return 1;
  }


// True if window under the cursor
FXbool FXWindow::underCursor() const {
  return (getApp()->cursorWindow==this);
  }


// Gained focus
long FXWindow::onFocusIn(FXObject*,FXSelector,void* ptr){
  FXTRACE((250,"%s::onFocusIn %08x\n",getClassName(),this));
  flags|=FLAG_FOCUSED;
  if(focus) focus->handle(focus,MKUINT(0,SEL_FOCUSIN),NULL);
  if(target) target->handle(this,MKUINT(message,SEL_FOCUSIN),ptr);
  return 1;
  }


// Lost focus
long FXWindow::onFocusOut(FXObject*,FXSelector,void* ptr){
  FXTRACE((250,"%s::onFocusOut %08x\n",getClassName(),this));
  flags&=~FLAG_FOCUSED;
  if(focus) focus->handle(focus,MKUINT(0,SEL_FOCUSOUT),NULL);
  if(target) target->handle(this,MKUINT(message,SEL_FOCUSOUT),ptr);
  return 1;
  }


// Focus on widget itself
long FXWindow::onFocusSelf(FXObject*,FXSelector,void*){
  FXTRACE((250,"%s::onFocusSelf %08x\n",getClassName(),this));
  setFocus();
  return 1;
  }


// Handle drag-and-drop enter
long FXWindow::onDNDEnter(FXObject*,FXSelector,void* ptr){
  FXTRACE((100,"%s::onDNDEnter %08x\n",getClassName(),this));
  if(target && target->handle(this,MKUINT(message,SEL_DND_ENTER),ptr)) return 1;
  return 0;
  }


// Handle drag-and-drop leave
long FXWindow::onDNDLeave(FXObject*,FXSelector,void* ptr){
  FXTRACE((100,"%s::onDNDLeave %08x\n",getClassName(),this));
  if(target && target->handle(this,MKUINT(message,SEL_DND_LEAVE),ptr)) return 1;
  return 0;
  }


// Handle drag-and-drop motion
long FXWindow::onDNDMotion(FXObject*,FXSelector,void* ptr){
  FXTRACE((100,"%s::onDNDMotion %08x\n",getClassName(),this));
  if(target && target->handle(this,MKUINT(message,SEL_DND_MOTION),ptr)) return 1;
  return 0;
  }


// Handle drag-and-drop drop
long FXWindow::onDNDDrop(FXObject*,FXSelector,void* ptr){
  FXTRACE((100,"%s::onDNDDrop %08x\n",getClassName(),this));
  if(target && target->handle(this,MKUINT(message,SEL_DND_DROP),ptr)) return 1;
  return 0;
  }


// Request for DND data
long FXWindow::onDNDRequest(FXObject*,FXSelector,void* ptr){
  FXTRACE((100,"%s::onDNDRequest %08x\n",getClassName(),this));
  if(target && target->handle(this,MKUINT(message,SEL_DND_REQUEST),ptr)) return 1;
  return 0;
  }


// Show window
long FXWindow::onCmdShow(FXObject*,FXSelector,void*){ 
  if(!shown()){ show(); recalc(); }
  return 1; 
  }


// Hide window
long FXWindow::onCmdHide(FXObject*,FXSelector,void*){ 
  if(shown()){ hide(); recalc(); }
  return 1; 
  }

// Hide or show window
long FXWindow::onCmdToggleShown(FXObject*,FXSelector,void*){ 
  shown() ? hide() : show(); 
  recalc(); 
  return 1;
  }


// Update hide or show window 
long FXWindow::onUpdToggleShown(FXObject* sender,FXSelector,void*){ 
  FXuint msg=shown() ? ID_CHECK : ID_UNCHECK;
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(msg,SEL_COMMAND),NULL);
  return 1;
  }


// Raise window
long FXWindow::onCmdRaise(FXObject*,FXSelector,void*){ 
  raise(); 
  return 1; 
  }


// Lower window
long FXWindow::onCmdLower(FXObject*,FXSelector,void*){ 
  lower(); 
  return 1; 
  }


// Delete window
long FXWindow::onCmdDelete(FXObject*,FXSelector,void*){ 
  delete this; 
  return 1; 
  }


// Enable window
long FXWindow::onCmdEnable(FXObject*,FXSelector,void*){ 
  enable(); 
  return 1; 
  }


// Disable window
long FXWindow::onCmdDisable(FXObject*,FXSelector,void*){ 
  disable(); 
  return 1; 
  }


// In combination with autograying/autohiding widgets,
// this could be used to ungray or show when you don't want
// to write a whole handler routine just to do this.
long FXWindow::onUpdYes(FXObject* sender,FXSelector,void*){
  sender->handle(this,MKUINT(ID_ENABLE,SEL_COMMAND),NULL);
  sender->handle(this,MKUINT(ID_SHOW,SEL_COMMAND),NULL);
  return 1;
  }


// Update (repaint) window
long FXWindow::onCmdUpdate(FXObject*,FXSelector,void*){ 
  update(); 
  return 1; 
  }


/*******************************************************************************/


// If window can have focus
FXbool FXWindow::canFocus() const {
  return FALSE; 
  }


// Has window the focus
FXbool FXWindow::hasFocus() const {
  return (flags&FLAG_FOCUSED)!=0; 
  }


// Set focus to this widget.
// The chain of focus from shell down to a control is changed.
// Widgets now in the chain may or may not gain real focus,
// depending on whether parent window already had a real focus!
// Setting the focus to a composite will cause descendants to loose it.
void FXWindow::setFocus(){
  FXASSERT(parent);
  if(parent->focus==this){
    if(focus) focus->killFocus();
    }
  else{
    if(parent->focus) 
      parent->focus->killFocus();
    else
      parent->setFocus();
    parent->focus=this;
    if(parent->hasFocus()) handle(this,MKUINT(0,SEL_FOCUSIN),NULL);
    }
  flags|=FLAG_HELP;
  }


// Kill focus to this widget.
void FXWindow::killFocus(){
  FXASSERT(parent);
  if(parent->focus!=this) return;
  if(focus) focus->killFocus();
  if(hasFocus()) handle(this,MKUINT(0,SEL_FOCUSOUT),NULL);
  parent->focus=NULL;
  flags&=~FLAG_HELP;
  }


/*******************************************************************************/


#ifndef FX_NATIVE_WIN32

// Add this window to the list of colormap windows
void FXWindow::addColormapWindows(){
  Window windows[2],*windowsReturn,*windowList;
  int countReturn,i;
    
  // Check to see if there is already a property
  Status status=XGetWMColormapWindows(getApp()->display,getShell()->id(),&windowsReturn,&countReturn);

  // If no property, just create one
  if(!status){
    windows[0]=id();
    windows[1]=getShell()->id();
    XSetWMColormapWindows(getApp()->display,getShell()->id(),windows,2);
    }

  // There was a property, add myself to the beginning
  else{
    windowList=(Window*)malloc((sizeof(Window))*(countReturn+1));
    windowList[0]=id();
    for(i=0; i<countReturn; i++) windowList[i+1]=windowsReturn[i];
    XSetWMColormapWindows(getApp()->display,getShell()->id(),windowList,countReturn+1);
    XFree((char*)windowsReturn);
    free(windowList);
    }
  }


// Remove it from colormap windows
void FXWindow::remColormapWindows(){
  Window *windowsReturn;
  int countReturn,i;
  Status status=XGetWMColormapWindows(getApp()->display,getShell()->id(),&windowsReturn,&countReturn);
  if(status){
    for(i=0; i<countReturn; i++){
      if(windowsReturn[i]==id()){
        for(i++; i<countReturn; i++) windowsReturn[i-1]=windowsReturn[i];
        XSetWMColormapWindows(getApp()->display,getShell()->id(),windowsReturn,countReturn-1);
        break;
        }
      }
    XFree((char*)windowsReturn);
    }
  }

#endif


/*******************************************************************************/


// Create X window
void FXWindow::create(){ 
  if(!xid){
    FXTRACE((100,"%s::create %08x\n",getClassName(),this));
    
#ifndef FX_NATIVE_WIN32
    
    XSetWindowAttributes wattr;
    unsigned long mask;
    
    // Gotta have display open!
    if(!getApp()->display){ fxerror("%s::create: trying to create window before opening display.\n",getClassName()); }
  
    // Gotta have a parent already created!
    if(!parent->id()){ fxerror("%s::create: trying to create window before creating parent window.\n",getClassName()); }

    // If window has owner, owner should have been created already  
    if(owner && !owner->id()){ fxerror("%s::create: trying to create window before creating owner window.\n",getClassName()); }
  
    // Got to have a visual    
    if(!visual){ fxerror("%s::create: trying to create window without a visual.\n",getClassName()); }
  
    // Initialize visual
    visual->init();

    // Create default cursor
    if(defaultCursor) defaultCursor->create();
    
    // Create drag cursor
    if(dragCursor) dragCursor->create();
  
    // Fill in the attributes
    mask=CWBackPixmap|CWWinGravity|CWBitGravity|CWBorderPixel|CWEventMask|CWDontPropagate|CWCursor|CWOverrideRedirect|CWSaveUnder|CWColormap;
    
    // Events for normal windows
    wattr.event_mask=BASIC_EVENT_MASK;
    
    // Events for shell windows
    if(shell==this) wattr.event_mask|=SHELL_EVENT_MASK;
    
    // If enabled, turn on some more events
    if(flags&FLAG_ENABLED) wattr.event_mask|=ENABLED_EVENT_MASK;
    
    // FOX will not propagate events to ancestor windows
    wattr.do_not_propagate_mask=NOT_PROPAGATE_MASK;

    // Obtain colormap
    wattr.colormap=visual->colormap;
    
    // This is needed for OpenGL
    wattr.border_pixel=0;
    
    // No background
    wattr.background_pixmap=None;
    
    // We don't seem to be able to do ForgetGravity for win_gravity
    // Its the same as UnmapGravity, which is useless...
    wattr.bit_gravity=ForgetGravity;
    //wattr.bit_gravity=NorthWestGravity;   ///// Someday, we'll turn this on for even higher performance!
    wattr.win_gravity=NorthWestGravity;
    
    // Determine override redirect
    wattr.override_redirect=doesOverrideRedirect();
    
    // Determine save-unders
    wattr.save_under=doesSaveUnder();
  
    // Set cursor  
    wattr.cursor=defaultCursor->id();
    
    // Finally, create the window
    xid=XCreateWindow(getApp()->display,parent->id(),xpos,ypos,FXMAX(width,1),FXMAX(height,1),0,visual->depth,InputOutput,visual->visual,mask,&wattr);
    
    // Uh-oh, we failed
    if(!xid){ fxerror("%s::create: unable to create window.\n",getClassName()); }
  
    // Set resource and class name for toplevel windows. 
    // In a perfect world this would be set in FXTopWindow, but for some strange reasons
    // some window-managers (e.g. fvwm) this will be too late and they will not recognize them.
    // Patch from axel.kohlmeyer@chemie.uni-ulm.de
    if(shell==this){
      XClassHint hint;
      hint.res_name=(char*)getApp()->getAppName();
      hint.res_class=(char*)getClassName();
      XSetClassHint(getApp()->display,xid,&hint);
      }

    // Store for xid to C++ object mapping
    XSaveContext(getApp()->display,xid,getApp()->wcontext,(XPointer)this);
    
    // If window is a shell and it has an owner, make it stay on top of the owner
    if(shell==this && owner){
      XSetTransientForHint(getApp()->display,xid,owner->getShell()->id());
      }
    
    // If colormap different, set WM_COLORMAP_WINDOWS property properly
    if(visual->colormap!=DefaultColormap(getApp()->display,DefaultScreen(getApp()->display))){
      FXTRACE((150,"%s::create: %08x: adding to WM_COLORMAP_WINDOWS\n",getClassName(),this));
      addColormapWindows();
      }
    
    // Show if it was supposed to be
    if((flags&FLAG_SHOWN) && 0<width && 0<height) XMapWindow(getApp()->display,xid);
    
#else
    
    // Gotta have a parent already created!
    if(!parent->id()){ fxerror("%s::create: trying to create window before creating parent window.\n",getClassName()); }

    // If window has owner, owner should have been created already  
    if(owner && !owner->id()){ fxerror("%s::create: trying to create window before creating owner window.\n",getClassName()); }
  
    // Got to have a visual    
    if(!visual){ fxerror("%s::create: trying to create window without a visual.\n",getClassName()); }
  
    // Initialize visual
    visual->init();

    // Create default cursor
    if(defaultCursor) defaultCursor->create();
    
    // Create drag cursor
    if(dragCursor) dragCursor->create();

    // Most windows use these style bits
    DWORD dwStyle=WS_CHILD|WS_VISIBLE|WS_CLIPSIBLINGS|WS_CLIPCHILDREN;
    DWORD dwExStyle=0;

    HWND hParent=(HWND)parent->id();
    if(shell==this){
      if(isMemberOf(FXMETACLASS(FXMainWindow))){
        dwStyle=WS_OVERLAPPEDWINDOW;
        }
      else if(doesOverrideRedirect()){
        // To control window placement or control decoration, a window manager
        // often needs to redirect map or configure requests. Popup windows, however, 
        // often need to be mapped without a window manager getting in the way.
        dwStyle=WS_POPUP;
        dwExStyle=WS_EX_TOOLWINDOW;
        }
      else{
        // Other top-level shell windows (like dialogs)
        dwStyle=WS_POPUP|WS_CAPTION;
        dwExStyle=WS_EX_DLGMODALFRAME|WS_EX_TOOLWINDOW; 
        // if(hParent==GetDesktopWindow() && getApp()->getMainWindow()!=0){
        // if(getApp()->getMainWindow()->id()) hParent=(HWND)getApp()->getMainWindow()->id();
        }
      if(owner) hParent=(HWND)owner->id();
      }

    // Create this window
    xid=CreateWindowEx(dwExStyle,"FXWindow",NULL,dwStyle,xpos,ypos,FXMAX(width,1),FXMAX(height,1),hParent,NULL,FXApp::hInstance,this);

    // Uh-oh, we failed
    if(!xid){ fxerror("%s::create: unable to create window.\n",getClassName()); }

    // Show if it was supposed to be
    if(flags&FLAG_SHOWN) ShowWindow((HWND)xid,SW_SHOWNOACTIVATE);

#endif
    }
  }


// Detach window
void FXWindow::detach(){
  xid=0;
  }


// Destroy
void FXWindow::destroy(){
  if(xid){
    
    FXTRACE((100,"%s::destroy %08x\n",getClassName(),this));
    
#ifndef FX_NATIVE_WIN32

    // If colormap different, set WM_COLORMAP_WINDOWS property properly
    if(visual->colormap!=DefaultColormap(getApp()->display,DefaultScreen(getApp()->display))){
      FXTRACE((150,"%s::destroy: %08x: removing from WM_COLORMAP_WINDOWS\n",getClassName(),this));
      remColormapWindows();
      }
    
    // Delete the window
    XDestroyWindow(getApp()->display,xid);
    
    // Remove from xid to C++ object mapping
    XDeleteContext(getApp()->display,xid,getApp()->wcontext);

#else
    
    // Remove XdndAware property (if set)
    RemoveProp((HWND)xid,(LPCTSTR)MAKELONG(getApp()->xdndAware,0));

    // Zap the window
    DestroyWindow((HWND)xid);

    // Remove from xid to C++ object mapping
    SetWindowLong((HWND)xid,0,0);

#endif
    
    // No longer grabbed
    if(getApp()->grabWindow==this) getApp()->grabWindow=NULL;
    
    xid=0;
    }
  }


#ifdef FX_NATIVE_WIN32

// Get this window's device context
HDC FXWindow::GetDC() const {
  return ::GetDC((HWND)xid);
  }

// Release it
int FXWindow::ReleaseDC(HDC hdc) const { 
  return ::ReleaseDC((HWND)xid,hdc); 
  }

#endif


/*******************************************************************************/


// Test if active
FXbool FXWindow::isActive() const {
  return (flags&FLAG_ACTIVE)!=0;
  }


// Get default width
FXint FXWindow::getDefaultWidth(){ 
  return 1; 
  }


// Get default height
FXint FXWindow::getDefaultHeight(){ 
  return 1; 
  }


// Set X position
void FXWindow::setX(FXint x){ 
  xpos=x; 
  recalc();
  }


// Set Y position
void FXWindow::setY(FXint y){ 
  ypos=y; 
  recalc();
  }


// Set width
void FXWindow::setWidth(FXint w){ 
  if(w<0) w=0;
  width=w; 
  recalc();
  }


// Set height
void FXWindow::setHeight(FXint h){ 
  if(h<0) h=0;
  height=h; 
  recalc();
  }


// Get maximum child width
FXint FXWindow::maxChildWidth() const {
  register FXWindow* child;
  register FXuint hints;
  register FXint t,m;
  for(m=0,child=getFirst(); child; child=child->getNext()){
    if(child->shown()){
      hints=child->getLayoutHints();
      if(hints&LAYOUT_FIX_WIDTH) t=child->getWidth();
      else t=child->getDefaultWidth();
      if(m<t) m=t;
      }
    }
  return m;
  }


// Get maximum child height
FXint FXWindow::maxChildHeight() const {
  register FXWindow* child;
  register FXuint hints;
  register FXint t,m;
  for(m=0,child=getFirst(); child; child=child->getNext()){
    if(child->shown()){
      hints=child->getLayoutHints();
      if(hints&LAYOUT_FIX_HEIGHT) t=child->getHeight();
      else t=child->getDefaultHeight();
      if(m<t) m=t;
      }
    }
  return m;
  }


// Change layout
void FXWindow::setLayoutHints(FXuint lout){
  FXuint opts=(options&~LAYOUT_MASK) | (lout&LAYOUT_MASK);
  if(options!=opts){
    options=opts;
    recalc();
    }
  }


// Get layout hints
FXuint FXWindow::getLayoutHints() const { 
  return (options&LAYOUT_MASK); 
  }


// Is widget a composite
FXbool FXWindow::isComposite() const { 
  return 0; 
  }


// Window does override-redirect
FXbool FXWindow::doesOverrideRedirect() const { 
  return FALSE; 
  }


// Window does save-unders
FXbool FXWindow::doesSaveUnder() const { 
  return FALSE; 
  }


// Modal events to this window should be passed
FXbool FXWindow::passModalEvents() const {
  // We're assuming [for now] only popups do save-unders!!!
  return !getApp()->invocation || getApp()->invocation->window==shell->xid || shell->doesSaveUnder();
  }


// Add hot key to closest ancestor's accelerator table
void FXWindow::addHotKey(FXHotKey code){
  FXAccelTable *accel; FXWindow *win=this;
  while((accel=win->getAccelTable())==NULL && win!=shell) win=win->parent;
  if(accel) accel->addAccel(code,this,MKUINT(ID_HOTKEY,SEL_KEYPRESS),MKUINT(ID_HOTKEY,SEL_KEYRELEASE));
  }


// Remove hot key from closest ancestor's accelerator table
void FXWindow::remHotKey(FXHotKey code){
  FXAccelTable *accel; FXWindow *win=this;
  while((accel=win->getAccelTable())==NULL && win!=shell) win=win->parent;
  if(accel) accel->removeAccel(code);
  }


// Check if child is related
FXbool FXWindow::containsChild(const FXWindow* child) const {
  while(child){ 
    if(child==this) return 1;
    child=child->parent;
    }
  return 0;
  }


// Check if window is related to parent window
FXbool FXWindow::isChildOf(const FXWindow* window) const {
  if(window!=this){
    const FXWindow* w=this;
    while(w){
      if(w==window) return TRUE;
      w=w->parent;
      }
    }
  return FALSE;
  }


// Get child at x,y
FXWindow* FXWindow::getChildAt(FXint x,FXint y) const {
  register FXWindow* child;
  for(child=first; child; child=child->next){
    if(child->shown() && child->FXWindow::contains(x,y)) return child;
    }
  return NULL;
  }


// Count number of children
FXint FXWindow::numChildren() const {
  register const FXWindow *child=first;
  register FXint num=0;
  while(child){
    child=child->next;
    num++;
    }
  return num;
  }


// Get index of child window
FXint FXWindow::indexOfChild(const FXWindow *window) const {
  register FXint index=0;
  if(!window) return -1;
  while(window->prev){
    window=window->prev;
    index++;
    }
  return index;
  }


// Get child window at index
FXWindow* FXWindow::childAtIndex(FXint index) const {
  register FXWindow* child=first;
  if(index<0) return NULL;
  while(index && child){
    child=child->next;
    index--;
    }
  return child;
  }


// Find common ancestor (if any)
FXWindow* FXWindow::commonAncestor(const FXWindow* window) const {
  FXASSERT(this!=0);
  FXASSERT(window!=0);
  register const FXWindow *p1,*p2;
  if(window->isChildOf(this)) return this->parent;
  if(this->isChildOf(window)) return window->parent;
  p1=this;
  while(p1->parent){
    p2=window;
    while(p2->parent){
      if(p2->parent==p1->parent) return p1->parent;
      p2=p2->parent;
      }
    p1=p1->parent;
    }
  return NULL;
  }


/*******************************************************************************/


// Set cursor
void FXWindow::setDefaultCursor(FXCursor* cur){
  if(defaultCursor!=cur){
    if(cur==NULL){ fxerror("%s::setDefaultCursor: NULL cursor argument.\n",getClassName()); }
    if(xid){
      if(cur->id()==0){ fxerror("%s::setDefaultCursor: Cursor has not been created yet.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
      XDefineCursor(getApp()->display,xid,cur->id());
#else
      if(!grabbed()) SetCursor((HCURSOR)cur->id());
#endif
      }
    defaultCursor=cur;
    }
  }


// Set drag cursor
void FXWindow::setDragCursor(FXCursor* cur){
  if(dragCursor!=cur){
    if(cur==NULL){ fxerror("%s::setDragCursor: NULL cursor argument.\n",getClassName()); }
    if(xid){
      if(cur->id()==0){ fxerror("%s::setDragCursor: Cursor has not been created yet.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
      if(grabbed()){ XChangeActivePointerGrab(getApp()->display,GRAB_EVENT_MASK,cur->id(),CurrentTime); }
#else
      if(grabbed()) SetCursor((HCURSOR)cur->id());
#endif
      }
    dragCursor=cur;
    }
  }
  

// Set window background
void FXWindow::setBackColor(FXColor clr){
  if(clr!=backColor){
    backColor=clr;
    update();
    }
  }


/*******************************************************************************/


// Lost the selection
long FXWindow::onSelectionLost(FXObject*,FXSelector,void* ptr){
  FXTRACE((100,"%s::onSelectionLost %08x\n",getClassName(),this));
  if(target && target->handle(this,MKUINT(message,SEL_SELECTION_LOST),ptr)) return 1;
  return 0;
  }


// Gained the selection
long FXWindow::onSelectionGained(FXObject*,FXSelector,void* ptr){
  FXTRACE((100,"%s::onSelectionGained %08x\n",getClassName(),this));
  if(target && target->handle(this,MKUINT(message,SEL_SELECTION_GAINED),ptr)) return 1;
  return 0;
  }


// Somebody wants our the selection
long FXWindow::onSelectionRequest(FXObject*,FXSelector,void* ptr){
  FXTRACE((100,"%s::onSelectionRequest %08x\n",getClassName(),this));
  if(target && target->handle(this,MKUINT(message,SEL_SELECTION_REQUEST),ptr)) return 1;
  return 0;
  }


// Has this window the selection
FXbool FXWindow::hasSelection() const {
  return (getApp()->selectionWindow==this);
  }


// Acquire the selection
FXbool FXWindow::acquireSelection(const FXDragType *types,FXuint numtypes){
  if(xid){
    if(types==NULL){ fxerror("%s::acquireSelection: should have at least one type to select.\n",getClassName()); }
    if(numtypes<1){ fxerror("%s::acquireSelection: should have at least one type to select.\n",getClassName()); }
    if(numtypes>1000){ fxerror("%s::acquireSelection: too many types to select.\n",getClassName()); }

    // Old window looses selection  
    if(getApp()->selectionWindow && getApp()->selectionWindow!=this){
      getApp()->selectionWindow->handle(getApp(),MKUINT(0,SEL_SELECTION_LOST),&getApp()->event);
      getApp()->selectionWindow=NULL;
      }

#ifndef FX_NATIVE_WIN32
    // Free old list of selected types
    FXFREE(&getApp()->xselTypeList);
    getApp()->xselNumTypes=0;
#endif
    
    // New window acquires selection
    if(!getApp()->selectionWindow){
#ifndef FX_NATIVE_WIN32
      XSetSelectionOwner(getApp()->display,XA_PRIMARY,xid,getApp()->event.time);
      if(XGetSelectionOwner(getApp()->display,XA_PRIMARY)!=xid) return FALSE;
#endif
      handle(this,MKUINT(0,SEL_SELECTION_GAINED),&getApp()->event);
      getApp()->selectionWindow=this;
      }
    
    // New list of selection types
#ifndef FX_NATIVE_WIN32
    FXMALLOC(&getApp()->xselTypeList,FXDragType,numtypes);
    memcpy(getApp()->xselTypeList,types,sizeof(FXDragType)*numtypes);
    getApp()->xselNumTypes=numtypes;
#endif
    return TRUE;
    }
  return FALSE;
  }


// Release the selection
FXbool FXWindow::releaseSelection(){
  if(xid){
    if(getApp()->selectionWindow==this){
      
      // Free list of selection types
#ifndef FX_NATIVE_WIN32
      FXFREE(&getApp()->xselTypeList);
      getApp()->xselNumTypes=0;
      XSetSelectionOwner(getApp()->display,XA_PRIMARY,None,CurrentTime);
#endif
      handle(this,MKUINT(0,SEL_SELECTION_LOST),&getApp()->event);
      getApp()->selectionWindow=NULL;
      return TRUE;
      }
    }
  return FALSE;
  }


/*******************************************************************************/


// Lost the selection
long FXWindow::onClipboardLost(FXObject*,FXSelector,void* ptr){
  FXTRACE((100,"%s::onClipboardLost %08x\n",getClassName(),this));
  if(target && target->handle(this,MKUINT(message,SEL_CLIPBOARD_LOST),ptr)) return 1;
  return 0;
  }


// Gained the selection
long FXWindow::onClipboardGained(FXObject*,FXSelector,void* ptr){
  FXTRACE((100,"%s::onClipboardGained %08x\n",getClassName(),this));
  if(target && target->handle(this,MKUINT(message,SEL_CLIPBOARD_GAINED),ptr)) return 1;
  return 0;
  }


// Somebody wants our the selection
long FXWindow::onClipboardRequest(FXObject*,FXSelector,void* ptr){
  FXTRACE((100,"%s::onClipboardRequest %08x\n",getClassName(),this));
  if(target && target->handle(this,MKUINT(message,SEL_CLIPBOARD_REQUEST),ptr)) return 1;
  return 0;
  }


// Has this window the selection
FXbool FXWindow::hasClipboard() const {
  return (getApp()->clipboardWindow==this);
  }


// Acquire the clipboard
FXbool FXWindow::acquireClipboard(const FXDragType *types,FXuint numtypes){
  if(xid){ 
    if(types==NULL){ fxerror("%s::acquireClipboard: should have at least one type to place on clipboard.\n",getClassName()); }
    if(numtypes<1){ fxerror("%s::acquireClipboard: should have at least one type to place on clipboard.\n",getClassName()); }
    if(numtypes>1000){ fxerror("%s::acquireClipboard: too many types placed on clipboard.\n",getClassName()); }
  
    // Old window looses the clipboard
    if(getApp()->clipboardWindow && getApp()->clipboardWindow!=this){
      getApp()->clipboardWindow->handle(getApp(),MKUINT(0,SEL_CLIPBOARD_LOST),&getApp()->event);
      getApp()->clipboardWindow=NULL;
      }
    
    // Free old list of clip types
#ifndef FX_NATIVE_WIN32
    FXFREE(&getApp()->xcbTypeList);
    getApp()->xcbNumTypes=0;
#endif
    
    // New window to own clipboard
    if(!getApp()->clipboardWindow){
#ifndef FX_NATIVE_WIN32
      XSetSelectionOwner(getApp()->display,getApp()->xcbSelection,xid,getApp()->event.time);
      if(XGetSelectionOwner(getApp()->display,getApp()->xcbSelection)!=xid) return FALSE;
#endif
      handle(this,MKUINT(0,SEL_CLIPBOARD_GAINED),&getApp()->event);
      getApp()->clipboardWindow=this;
      }
    
    // New list of clipboard types
#ifndef FX_NATIVE_WIN32
    FXMALLOC(&getApp()->xcbTypeList,FXDragType,numtypes);
    memcpy(getApp()->xcbTypeList,types,sizeof(FXDragType)*numtypes);
    getApp()->xcbNumTypes=numtypes;
#else
    if(OpenClipboard((HWND)xid)){
      register FXuint i;
      EmptyClipboard();
      for(i=0; i<numtypes; i++){ SetClipboardData(types[i],NULL); }
      CloseClipboard();
      }
#endif
    return TRUE;
    }
  return FALSE;
  }


// Release the clipboard
FXbool FXWindow::releaseClipboard(){
  if(xid){
    if(getApp()->clipboardWindow==this){
      
      // Free clipboard types
#ifndef FX_NATIVE_WIN32
      FXFREE(&getApp()->xcbTypeList);
      getApp()->xcbNumTypes=0;
      XSetSelectionOwner(getApp()->display,getApp()->xcbSelection,None,CurrentTime); 
#else
      if(OpenClipboard((HWND)xid)){
        EmptyClipboard();
        CloseClipboard();
        }
#endif
      handle(this,MKUINT(0,SEL_CLIPBOARD_LOST),&getApp()->event);
      getApp()->clipboardWindow=NULL;
      return TRUE;
      }
    }
  return FALSE;
  }


/*******************************************************************************/


// Get pointer location (in window coordinates)
FXint FXWindow::getCursorPosition(FXint& x,FXint& y,FXuint& buttons) const {
  if(xid!=0){
#ifndef FX_NATIVE_WIN32
    Window dum; int rx,ry;
    return XQueryPointer(getApp()->display,xid,&dum,&dum,&rx,&ry,&x,&y,&buttons);
#else
    POINT pt;
    GetCursorPos(&pt);
    ScreenToClient((HWND)xid,&pt);
    x=pt.x; y=pt.y;
    buttons=fxmodifierkeys();
    return TRUE;
#endif
    }
  return FALSE;
  }


// Set pointer location (in window coordinates) 
// Contributed by David Heath <dave@hipgraphics.com>
FXint FXWindow::setCursorPosition(FXint x,FXint y){
  if(xid!=0){
#ifndef FX_NATIVE_WIN32
    XWarpPointer(getApp()->display,None,xid,0,0,0,0,x,y);
    return TRUE;
#else
    POINT pt;
    pt.x=x; 
    pt.y=y;
    ClientToScreen((HWND)xid,&pt);
    SetCursorPos(pt.x,pt.y);
    return TRUE;
#endif
    }
  return FALSE;
  }


// Update this widget if allowed
long FXWindow::onUpdate(FXObject*,FXSelector,void*){
  
  // Do layout
  if(flags&FLAG_DIRTY) layout();
   
  // Do GUI update
  if(flags&FLAG_UPDATE){
    
    // No target, so we're done
    if(!target) return 0;
    
    // Check to see if the target object has been deleted
    if(*((void**)target) == (void*)-1L){fxerror("%s::onUpdate: %08x references a deleted target object at %08x.\n",getClassName(),this,target);}
  
    // Ask the target object to update the state of this widget
    return target->handle(this,MKUINT(message,SEL_UPDATE),NULL);
    }
  return 1;
  }


// Propagate window size change
void FXWindow::layout(){ 
  flags&=~FLAG_DIRTY;
  }


// Propagate layout changes upward and schedule a refresh
void FXWindow::recalc(){
  getApp()->refresh();// As long as layout clean is done with GUI update
  if(parent) parent->recalc(); 
  flags|=FLAG_DIRTY;
  }


// Update dirty rectangle
void FXWindow::update(FXint x,FXint y,FXint w,FXint h){
  if(xid){

    // We toss out expose rectangles outside the visible area
    if(x>=width || y>=height || x+w<=0 || y+h<=0) return; 
  
    // Intersect with the window
    if(x<0){w+=x;x=0;}
    if(y<0){h+=y;y=0;}
    if(x+w>width){w=width-x;}
    if(y+h>height){h=height-y;}

    // Append the rectangle; it is a synthetic expose event!!
    if(w>0 && h>0){
#ifndef FX_NATIVE_WIN32
      getApp()->addRepaint(xid,x,y,w,h,1);
#else
      RECT r;
      r.left=x;
      r.top=y;
      r.right=x+w;
      r.bottom=y+h;
      InvalidateRect((HWND)xid,&r,TRUE);
#endif
      }
    }
  }


// Update dirty window
void FXWindow::update(){
  update(0,0,width,height);
  }


// Move window
void FXWindow::move(FXint x,FXint y){
  if((flags&FLAG_DIRTY) || (x!=xpos) || (y!=ypos)){
    xpos=x;
    ypos=y;
    if(xid){
      // Similar as for position(), we have to generate protocol here...
      FXTRACE((100,"%s::move x=%d y=%d\n",getClassName(),x,y));
#ifndef FX_NATIVE_WIN32
      XMoveWindow(getApp()->display,xid,x,y);
#else
      SetWindowPos((HWND)xid,NULL,x,y,0,0,SWP_NOSIZE|SWP_NOZORDER);
#endif
      if(flags&FLAG_DIRTY){
        layout();
        }
      }
    }
  }
    

// Move and resize
void FXWindow::position(FXint x,FXint y,FXint w,FXint h){
  if(w<0) w=0;
  if(h<0) h=0;
  if((flags&FLAG_DIRTY) || (x!=xpos) || (y!=ypos) || (w!=width) || (h!=height)){
    xpos=x;
    ypos=y;
    if(xid){
      // Alas, we have to generate some protocol here even if the placement
      // as recorded in the widget hasn't actually changed.  This is because
      // there are ways to change the placement w/o going through position()!
#ifndef FX_NATIVE_WIN32
      if(0<w && 0<h){
        if((flags&FLAG_SHOWN) && (width<=0 || height<=0)){
          XMapWindow(getApp()->display,xid);
          }
        XMoveResizeWindow(getApp()->display,xid,x,y,w,h);
        }
      else if(0<width && 0<height){
        XUnmapWindow(getApp()->display,xid);
        }
#else
      SetWindowPos((HWND)xid,NULL,x,y,w,h,SWP_NOZORDER|SWP_NOACTIVATE);
#endif
      if((flags&FLAG_DIRTY) || (w!=width) || (h!=height)){
        width=w;
        height=h;
        layout();
        }
      else{
        width=w;
        height=h;
        }
      }
    else{
      width=w;
      height=h;
      }
    }
  }

      
// Resize
void FXWindow::resize(FXint w,FXint h){
  if(w<0) w=0;
  if(h<0) h=0;
  if((flags&FLAG_DIRTY) || (w!=width) || (h!=height)){
    if(xid){
      // Similar as for position(), we have to generate protocol here..
#ifndef FX_NATIVE_WIN32
      if(0<w && 0<h){
        if((flags&FLAG_SHOWN) && (width<=0 || height<=0)){
          XMapWindow(getApp()->display,xid);
          }
        XResizeWindow(getApp()->display,xid,w,h);
        }
      else if(0<width && 0<height){
        XUnmapWindow(getApp()->display,xid);
        }
#else
      SetWindowPos((HWND)xid,NULL,0,0,w,h,SWP_NOMOVE|SWP_NOZORDER);
#endif
      width=w;
      height=h;
      layout();
      }
    else{
      width=w;
      height=h;
      }
    }
  }
  

// Force GUI refresh of this window and all of its children
void FXWindow::forceRefresh(){
  register FXWindow *child;
  handle(this,MKUINT(0,SEL_UPDATE),NULL);
  for(child=first; child; child=child->next){
    child->forceRefresh();
    }
  }


// Map window
void FXWindow::show(){
  if(!(flags&FLAG_SHOWN)){
    flags|=FLAG_SHOWN;
    if(xid){
#ifndef FX_NATIVE_WIN32
      if(0<width && 0<height) XMapWindow(getApp()->display,xid);
#else
      ShowWindow((HWND)xid,SW_SHOWNOACTIVATE);
#endif
      }
    }
  }


// Unmap
void FXWindow::hide(){
  if(flags&FLAG_SHOWN){
    killFocus();
    flags&=~FLAG_SHOWN;
    if(xid){
#ifndef FX_NATIVE_WIN32
      XUnmapWindow(getApp()->display,xid);
#else
      ShowWindow((HWND)xid,SW_HIDE);
#endif
      }
    }
  }


// Check if logically shown
FXbool FXWindow::shown() const { 
  return (flags&FLAG_SHOWN)!=0; 
  }


// Reparent window under a new parent
void FXWindow::reparent(FXComposite* newparent){
  if(newparent==NULL){ fxerror("%s::reparent: NULL parent specified.\n",getClassName()); }
  if(parent==NULL){ fxerror("%s::reparent: cannot reparent root window.\n",getClassName()); }
  if(shell==this){ fxerror("%s::reparent: cannot reparent toplevel window.\n",getClassName()); }
  if(newparent!=parent){
    
    // Check for funny cases
    if(containsChild(newparent)){ fxerror("%s::reparent: new parent is child of window.\n",getClassName()); }
    
    // Both windows created or both non-created
    if(xid && !newparent->xid){ fxerror("%s::reparent: new parent not created yet.\n",getClassName()); }
    if(!xid && newparent->xid){ fxerror("%s::reparent: window not created yet.\n",getClassName()); }
    
    // Kill focus chain through this window
    killFocus();
    
    // Flag old parent as to be recalculated
    parent->recalc();
    
    // Unlink from old parent
    if(prev) prev->next=next; else if(parent) parent->first=next;
    if(next) next->prev=prev; else if(parent) parent->last=prev;
    
    // Link to new parent
    parent=newparent;
    prev=parent->last;
    next=NULL;
    parent->last=this;
    if(prev) prev->next=this; else parent->first=this;
    
    // New toplevel shell too
    shell=parent->getShell();
    
    // New owner is the new parent
    owner=parent;
    
    // Hook up to new window in server too
    if(xid && parent->xid){
#ifndef FX_NATIVE_WIN32
      XReparentWindow(getApp()->display,xid,parent->xid,0,0);
#else
      SetParent((HWND)xid,(HWND)parent->xid);
#endif
      }
    
    // Flag as to be recalculated
    recalc();
    }
  }


// Move window in chain before a sibling
void FXWindow::linkBefore(FXWindow* sibling){
  if(!sibling){fxerror("%s::linkBefore: NULL argument.\n",getClassName());}
  if(sibling->parent!=parent){fxerror("%s::linkBefore: windows should have same parent.\n",getClassName());}
  if(sibling==this){fxerror("%s::linkBefore: same window.\n",getClassName());}
  if(prev) prev->next=next; else if(parent) parent->first=next;
  if(next) next->prev=prev; else if(parent) parent->last=prev; 
  prev=sibling->prev;
  next=sibling;
  if(prev) prev->next=this; else parent->first=this;
  sibling->prev=this;
  recalc();
  }


// Move window in chain after a sibling
void FXWindow::linkAfter(FXWindow* sibling){
  if(!sibling){fxerror("%s::linkAfter: NULL argument.\n",getClassName());}
  if(sibling->parent!=parent){fxerror("%s::linkAfter: windows should have same parent.\n",getClassName());}
  if(sibling==this){fxerror("%s::linkAfter: same window.\n",getClassName());}
  if(prev) prev->next=next; else if(parent) parent->first=next;
  if(next) next->prev=prev; else if(parent) parent->last=prev; 
  next=sibling->next;
  prev=sibling;
  if(next) next->prev=this; else parent->last=this;
  sibling->next=this;
  recalc();
  }


// Test if logically inside
FXbool FXWindow::contains(FXint parentx,FXint parenty) const { 
  return xpos<=parentx && parentx<xpos+width && ypos<=parenty && parenty<ypos+height;
  }


// Enable the window
void FXWindow::enable(){
  if(!(flags&FLAG_ENABLED)){
    if(xid){
#ifndef FX_NATIVE_WIN32
      FXuint events=BASIC_EVENT_MASK|ENABLED_EVENT_MASK;
      if(shell==this) events|=SHELL_EVENT_MASK;
      XSelectInput(getApp()->display,xid,events);
#else
      EnableWindow((HWND)xid,TRUE);
#endif
      }
    flags|=FLAG_ENABLED;
    }
  }


// Disable the window
void FXWindow::disable(){ 
  if(flags&FLAG_ENABLED){
    if(xid){
#ifndef FX_NATIVE_WIN32
      FXuint events=BASIC_EVENT_MASK;
      if(shell==this) events|=SHELL_EVENT_MASK;
      XSelectInput(getApp()->display,xid,events);
#else
      EnableWindow((HWND)xid,FALSE);
#endif
      }
    flags&=~FLAG_ENABLED;
    }
  }


// Is window enabled
FXbool FXWindow::isEnabled() const { 
  return (flags&FLAG_ENABLED)!=0; 
  }


// Raise (but do not activate!)
void FXWindow::raise(){ 
  if(xid){ 
#ifndef FX_NATIVE_WIN32
    XRaiseWindow(getApp()->display,xid); 
#else
    SetWindowPos((HWND)xid,HWND_TOP,0,0,0,0,SWP_NOMOVE|SWP_NOSIZE|SWP_NOACTIVATE);
#endif
    }
  }
  

// Lower
void FXWindow::lower(){ 
  if(xid){ 
#ifndef FX_NATIVE_WIN32
    XLowerWindow(getApp()->display,xid); 
#else
    SetWindowPos((HWND)xid,HWND_BOTTOM,0,0,0,0,SWP_NOMOVE|SWP_NOSIZE);
#endif
    }
  }



// Locate window at root coordinate position x,y
FXID FXWindow::getWindowAt(FXint x,FXint y) const {
#ifndef FX_NATIVE_WIN32
  Window win,child=XDefaultRootWindow(getApp()->display);
  int dropx,dropy;
  while(1){
    if(!XTranslateCoordinates(getApp()->display,XDefaultRootWindow(getApp()->display),child,x,y,&dropx,&dropy,&win)){
      return XDefaultRootWindow(getApp()->display);
      }
    if(win==None) break;
    child=win;
    }
  return child;
#else
  POINT point;
  point.x=x; 
  point.y=y;
  return WindowFromPoint(point);
#endif
  }


// Get coordinates from another window (for symmetry)
void FXWindow::translateCoordinatesFrom(FXint& tox,FXint& toy,const FXWindow* fromwindow,FXint fromx,FXint fromy) const {
  if(fromwindow==NULL){ fxerror("%s::translateCoordinatesFrom: from-window is NULL.\n",getClassName()); }
  if(xid && fromwindow->xid){
#ifndef FX_NATIVE_WIN32
    Window tmp;
    XTranslateCoordinates(getApp()->display,fromwindow->xid,xid,fromx,fromy,&tox,&toy,&tmp);
#else
    POINT pt;
    pt.x=fromx; 
    pt.y=fromy;
    ClientToScreen((HWND)fromwindow->xid,&pt);
    ScreenToClient((HWND)xid,&pt);
    tox=pt.x; 
    toy=pt.y;
#endif
    }
  }


// Get coordinates to another window (for symmetry)
void FXWindow::translateCoordinatesTo(FXint& tox,FXint& toy,const FXWindow* towindow,FXint fromx,FXint fromy) const {
  if(towindow==NULL){ fxerror("%s::translateCoordinatesTo: to-window is NULL.\n",getClassName()); }
  if(xid && towindow->xid){
#ifndef FX_NATIVE_WIN32
    Window tmp;
    XTranslateCoordinates(getApp()->display,xid,towindow->xid,fromx,fromy,&tox,&toy,&tmp);
#else
    POINT pt;
    pt.x=fromx; 
    pt.y=fromy;
    ClientToScreen((HWND)xid,&pt);
    ScreenToClient((HWND)towindow->xid,&pt);
    tox=pt.x; 
    toy=pt.y;
#endif
    }
  }


/*******************************************************************************/


// Acquire grab
// Grabs also switch to the drag cursor
void FXWindow::grab(){
  if(xid){
    FXTRACE((150,"%s::grab\n",getClassName()));
    if(dragCursor->id()==0){ fxerror("%s::grab: Cursor has not been created yet.\n",getClassName()); }
    if(!(flags&FLAG_SHOWN)){ fxwarning("%s::grab: Window is not visible.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
    if(GrabSuccess!=XGrabPointer(getApp()->display,xid,FALSE,GRAB_EVENT_MASK,GrabModeAsync,GrabModeAsync,None,dragCursor->id(),CurrentTime)){ 
      fxwarning("%s::grab: grab failed!\n",getClassName());
      if(getApp()->grabWindow==this){ fxwarning("%s::grab: was already grabbed to this window\n"); }
      else if(getApp()->grabWindow!=NULL){ fxwarning("%s::grab: was already grabbed to another window\n"); }
      }
#else
    SetCapture((HWND)xid);
    if(GetCapture()!=(HWND)xid){ 
      fxwarning("%s::grab: grab failed!\n",getClassName()); 
      if(getApp()->grabWindow==this){ fxwarning("%s::grab: was already grabbed to this window\n"); }
      else if(getApp()->grabWindow!=NULL){ fxwarning("%s::grab: was already grabbed to another window\n"); }
      }
    SetCursor((HCURSOR)dragCursor->id());
#endif
    getApp()->grabWindow=this;
    }
  }


// Release grab
// Ungrabs also switch back to the normal cursor
void FXWindow::ungrab(){
  if(xid){
    FXTRACE((150,"%s::ungrab\n",getClassName()));
#ifndef FX_NATIVE_WIN32
    XUngrabPointer(getApp()->display,CurrentTime);
    XFlush(getApp()->display);
#else
    ReleaseCapture();
    SetCursor((HCURSOR)defaultCursor->id());
#endif
    getApp()->grabWindow=NULL;
    }
  }


// Return true if active grab is in effect
FXbool FXWindow::grabbed() const {
  return getApp()->grabWindow==this;
  }


// The widget lost the grab for some reason [Windows].
// Subclasses should try to clean up the mess...
long FXWindow::onUngrabbed(FXObject*,FXSelector,void* ptr){
  FXTRACE((150,"%s::ungrab forced\n",getClassName()));
  FXASSERT(getApp()->grabWindow==this);
  getApp()->grabWindow=NULL;
#ifdef FX_NATIVE_WIN32
  SetCursor((HCURSOR)defaultCursor->id());
#endif
  if(isEnabled()){
    if(target && target->handle(this,MKUINT(message,SEL_UNGRABBED),ptr)) return 1;
    }
  return 0;
  }


/*******************************************************************************/

#ifndef FX_NATIVE_WIN32

// To pass arguments to matcher
struct XMatch {
  Window window;
  Atom   selection;
  int    gotit;
  };

#endif

// Enable widget as drop target
void FXWindow::dropEnable(){
  if(xid==0){ fxerror("%s::dropEnable: window has not yet been created.\n",getClassName()); }
  if(!(flags&FLAG_DROPTARGET)){
    FXID propdata=(FXID)XDND_PROTOCOL_VERSION;
    FXASSERT(getApp()->xdndAware);
#ifndef FX_NATIVE_WIN32
    XChangeProperty(getApp()->display,xid,getApp()->xdndAware,XA_ATOM,32,PropModeReplace,(FXuchar*)&propdata,1);
#else
    SetProp((HWND)xid,(LPCTSTR)MAKELONG(getApp()->xdndAware,0),(HANDLE)propdata);
#endif
    flags|=FLAG_DROPTARGET;
    }
  }


// Disable widget as drop target
void FXWindow::dropDisable(){
  if(xid==0){ fxerror("%s::dropDisable: window has not yet been created.\n",getClassName()); }
  if(flags&FLAG_DROPTARGET){
    FXASSERT(getApp()->xdndAware);
#ifndef FX_NATIVE_WIN32
    XDeleteProperty(getApp()->display,xid,getApp()->xdndAware);
#else
    RemoveProp((HWND)xid,(LPCTSTR)MAKELONG(getApp()->xdndAware,0));
#endif
    flags&=~FLAG_DROPTARGET;
    }
  }


// Is window drop enabled
FXbool FXWindow::isDropEnabled() const {
  return (flags&FLAG_DROPTARGET)!=0;
  }


// Set drag rectangle; we'll get no more DND_MOTION events while cursor is inside
void FXWindow::setDragRectangle(FXint x,FXint y,FXint w,FXint h,FXbool wantupdates){
  if(xid==0){ fxerror("%s::setDragRectangle: window has not yet been created.\n",getClassName()); }
  int tox,toy;
#ifndef FX_NATIVE_WIN32
  Window tmp;
  XTranslateCoordinates(getApp()->display,parent->id(),XDefaultRootWindow(getApp()->display),x,y,&tox,&toy,&tmp);
#else
  POINT pt;
  pt.x=x; 
  pt.y=y;
  ClientToScreen((HWND)parent->xid,&pt);
  tox=pt.x; 
  toy=pt.y;
#endif
  getApp()->xdndRect.x=tox;
  getApp()->xdndRect.y=toy;
  getApp()->xdndRect.w=w;
  getApp()->xdndRect.h=h;
  getApp()->xdndWantUpdates=wantupdates;
  }


// Clear drag rectangle
// Target always wants SEL_DND_MOTION messages
void FXWindow::clearDragRectangle(){
  if(xid==0){ fxerror("%s::clearDragRectangle: window has not yet been created.\n",getClassName()); }
  getApp()->xdndRect.x=0;
  getApp()->xdndRect.y=0;
  getApp()->xdndRect.w=0;
  getApp()->xdndRect.h=0;
  getApp()->xdndWantUpdates=TRUE;
  }


// Wait for arrival of dynamically exchanged data
#ifndef FX_NATIVE_WIN32
static Bool notifyevent(Display *,XEvent *event,XPointer ptr){
  XMatch *match=(XMatch*)ptr;
  if(event->type==DestroyNotify && event->xdestroywindow.window==match->window){
    match->gotit=0;
    return True;
    }
  if(event->type==SelectionNotify && event->xselection.selection==match->selection){
    return True;
    }
  return False;
  }
#endif


// Get dropped data; called in response to DND enter or DND drop
FXbool FXWindow::getDNDData(FXDNDOrigin origin,FXDragType targettype,FXuchar*& data,FXuint& size){
#ifndef FX_NATIVE_WIN32
  unsigned long  nitems,alloc_incr,bytes_after,transfersize,offset,toff;
  FXWindow      *ownerobject;
  Window         ownerwindow;
  XEvent         ev;
  XMatch         match;
  Bool           status;
  Atom           actualtype;
  Atom           selection;
  int            actualformat;
  FXuint         requesttype;
  FXuchar       *ptr;

  if(xid==0){ fxerror("%s::getDNDData: window has not yet been created.\n",getClassName()); }

  data=NULL;
  size=0;
  
  // Selection where data comes from
  if(origin==FROM_SELECTION){
    selection=XA_PRIMARY;
    requesttype=SEL_SELECTION_REQUEST;
    }
  else if(origin==FROM_CLIPBOARD){
    selection=getApp()->xcbSelection;
    requesttype=SEL_CLIPBOARD_REQUEST;
    }
  else if(origin==FROM_DRAGNDROP){
    selection=getApp()->xdndSelection;
    requesttype=SEL_DND_REQUEST;
    }
  else 
    return FALSE;
  
  // If no owner, we're done quickly
  ownerwindow=XGetSelectionOwner(getApp()->display,selection);
  if(ownerwindow == None) return FALSE;
  
  // Window is in same application; bypass protocol by returning data from the mailbox
  if(XFindContext(getApp()->display,ownerwindow,getApp()->wcontext,(XPointer*)&ownerobject)==0){
    FXASSERT(ownerobject);
    getApp()->event.type=requesttype;
    getApp()->event.window=ownerwindow;
    getApp()->event.requestor=xid;
    getApp()->event.target=targettype;
    getApp()->event.property=getApp()->ddeAtom;
    getApp()->event.origin=origin;
    ownerobject->handle(getApp(),MKUINT(0,requesttype),&getApp()->event);
    data=getApp()->ddeData;
    size=getApp()->ddeSize;
    getApp()->ddeData=NULL;
    getApp()->ddeSize=0;
    FXTRACE((100,"%s: getDNDData(target=%d,data=%08x,size=%d) from %d (%s)\n",getClassName(),targettype,data,size,ownerwindow,ownerobject->getClassName()));
    return data!=NULL;
    }
  
  // Send message to the selection owner
  XConvertSelection(getApp()->display,selection,targettype,getApp()->ddeAtom,xid,getApp()->event.time);
  
  // Wait for a response
  match.window=xid;
  match.selection=selection;
  match.gotit=1;
  status=XCheckIfEvent(getApp()->display,&ev,notifyevent,(XPointer)&match);

  // Broke out of event loop
  if(!match.gotit) return FALSE;
  if(!status){
    match.gotit=1;
    XIfEvent(getApp()->display,&ev,notifyevent,(XPointer)&match);
    if(!match.gotit) return FALSE;
    }
  
  // There was no selection or no data has been specified
  if(ev.xselection.property==None){
    return FALSE;
    }

  // Prepare for download of big property data
  transfersize=XMaxRequestSize(getApp()->display);
  if(!FXMALLOC(&getApp()->ddeData,FXuchar,8)) return FALSE;

  // Loop while we didn't get it all
  offset=0;
  toff=0;
  getApp()->ddeSize=0;
  bytes_after=1;
  while(bytes_after){
    if(XGetWindowProperty(getApp()->display,xid,getApp()->ddeAtom,toff,transfersize,False,AnyPropertyType,&actualtype,&actualformat,&nitems,&bytes_after,&ptr)!=Success){
      FXFREE(&getApp()->ddeData);
      getApp()->ddeData=NULL;
      getApp()->ddeSize=0;
      FXTRACE((100,"%s: getDNDData(target=%d) from %d (remote) failed\n",getClassName(),targettype,ownerwindow));
      return FALSE;
      }
    if(ptr == NULL || nitems == 0){
      XDeleteProperty(getApp()->display,xid,getApp()->ddeAtom);
      if(ptr) XFree(ptr);
      FXFREE(&getApp()->ddeData);
      getApp()->ddeData=NULL;
      getApp()->ddeSize=0;
      FXTRACE((100,"%s: getDNDData(target=%d,data=NULL,size=0) from %d (remote)\n",getClassName(),targettype,ownerwindow));
      return FALSE;
      }
    alloc_incr=nitems*(actualformat>>3);
    if(!FXRESIZE(&getApp()->ddeData,FXuchar,getApp()->ddeSize+alloc_incr+1)){
      XDeleteProperty(getApp()->display,xid,getApp()->ddeAtom);
      XFree(ptr);
      getApp()->ddeData=NULL;
      getApp()->ddeSize=0;
      FXTRACE((100,"%s: getDNDData(target=%d) from %d (remote) failed\n",getClassName(),targettype,ownerwindow));
      return FALSE;
      }
    memcpy(&getApp()->ddeData[offset],ptr,alloc_incr);
    getApp()->ddeSize+=alloc_incr;
    offset+=alloc_incr;
    toff+=(nitems*actualformat)/32;
    XFree(ptr);
    }
  XDeleteProperty(getApp()->display,xid,getApp()->ddeAtom);
  getApp()->ddeData[getApp()->ddeSize]=0;
  data=getApp()->ddeData;
  size=getApp()->ddeSize;
  if(targettype==stringType) size++;                // For XA_STRING, add the '\0'
  getApp()->ddeData=NULL;
  getApp()->ddeSize=0;
  FXTRACE((100,"%s: getDNDData(target=%d,data=%08x,size=%d) from %d (remote)\n",getClassName(),targettype,data,size,ownerwindow));
  return TRUE;

#else

  if(xid==0){ fxerror("%s::getDNDData: window has not yet been created.\n",getClassName()); }

  data=NULL;
  size=0;
  
  // Data from clip board
  if(origin==FROM_CLIPBOARD){
    if(IsClipboardFormatAvailable(targettype)){
      if(OpenClipboard((HWND)xid)){
	HANDLE hClipMemory=GetClipboardData(targettype);
	FXASSERT(hClipMemory);
	if(hClipMemory){
	  size=GlobalSize(hClipMemory);
	  if(FXMALLOC(&data,FXuchar,size)){
	    void *pClipMemory=GlobalLock(hClipMemory);
 	    FXASSERT(pClipMemory);
	    memcpy((void*)data,pClipMemory,size);
	    GlobalUnlock(hClipMemory);
	    CloseClipboard();
	    getApp()->ddeData=NULL;
	    getApp()->ddeSize=0;
	    return TRUE;
	    }
	  else{
	    data=NULL;
	    size=0;
	    }
	  }
	}
      }
    }
  
  // Data from drag and drop
  else if(origin==FROM_DRAGNDROP){
    // Window is in same application; bypass protocol by returning data from the mailbox
    HWND ownerwindow=getApp()->xdndSource;
    FXASSERT(ownerwindow);
    if(GetWindowThreadProcessId((HWND)xid,NULL)==GetWindowThreadProcessId(ownerwindow,NULL)){
      FXWindow *ownerobject=(FXWindow*)GetWindowLong(ownerwindow,0);
      FXASSERT(ownerobject);
      getApp()->event.type=SEL_DND_REQUEST;
      getApp()->event.window=ownerwindow;
      getApp()->event.requestor=xid;
      getApp()->event.target=targettype;
      getApp()->event.property=(FXID)getApp()->ddeAtom;
      getApp()->event.origin=origin;
      ownerobject->handle(getApp(),MKUINT(0,SEL_DND_REQUEST),&getApp()->event);
      data=getApp()->ddeData;
      size=getApp()->ddeSize;
      getApp()->ddeData=NULL;
      getApp()->ddeSize=0;
      FXTRACE((100,"%s: getDNDData(target=%d,data=%08x,size=%d) from %d (%s)\n",getClassName(),targettype,data,size,ownerwindow,ownerobject->getClassName()));
      return data!=NULL;
      }
    else{
      // Send a SEL_DND_REQUEST message to the owner
      SendMessage(ownerwindow,getApp()->xdndRequest,(WPARAM)targettype,(LPARAM)xid);

      // Open the memory mapped file & download the data
      HANDLE hMap=OpenFileMapping(FILE_MAP_READ,FALSE,"_FOX_DDE");
      if(hMap){
	LPVOID lpView=MapViewOfFile(hMap,FILE_MAP_READ,0,0,0);
	if(lpView){
	  getApp()->ddeSize=getApp()->dragContext->ddeSize; // How else to get the size?
	  FXMALLOC(&getApp()->ddeData,FXuchar,getApp()->ddeSize);
	  memcpy(getApp()->ddeData,lpView,getApp()->ddeSize);
	  UnmapViewOfFile(lpView);
	  }
	CloseHandle(hMap);
	}

      // Transfer ownership of the memory
      data=getApp()->ddeData;
      size=getApp()->ddeSize;
      getApp()->ddeData=NULL;
      getApp()->ddeSize=0;

      // We need to tell the drag source window that it's now
      // safe to close his handle to the file mapping object too.
      // Otherwise we've created a memory leak.
      FXASSERT(FALSE);
      FXTRACE((100,"%s: getDNDData(target=%d,data=%08x,size=%d) from %d (remote)\n",getClassName(),targettype,data,size,ownerwindow));
      return TRUE;
      }
    }
  
  // Data from selection not available for MS-Windows
  return FALSE;

#endif
  }



// Set drop data; data array will be deleted by the system automatically!
FXbool FXWindow::setDNDData(FXDNDOrigin origin,FXDragType targettype,FXuchar* data,FXuint size){
#ifndef FX_NATIVE_WIN32
  unsigned long  transferbytes,offset,tlen,mode;
  FXWindow      *reqobject;
  int            format;
  
  if(xid==0){ fxerror("%s::setDNDData: window has not yet been created.\n",getClassName()); }
  if(origin!=getApp()->event.origin){ fxerror("%s::setDNDData: DDE origin was not requested.\n",getClassName()); }

  
  // Window is in same application; bypass protocol by storing data into the mailbox
  if(XFindContext(getApp()->display,getApp()->event.requestor,getApp()->wcontext,(XPointer*)&reqobject)==0){
    FXTRACE((100,"%s: setDNDData(target=%d,data=%08x,size=%d) for %d (%s)\n",getClassName(),targettype,data,size,getApp()->event.requestor,reqobject->getClassName()));
    getApp()->ddeData=data;
    getApp()->ddeSize=size;
    }
  
  // Window in another application.  Stick the data into the indicated property,
  // then delete our local copy.  A new data array will be created in the remote
  // application.
  else{
    FXTRACE((100,"%s: setDNDData(target=%d,data=%08x,size=%d) for %d (remote)\n",getClassName(),targettype,data,size,getApp()->event.requestor));
    if(data!=NULL && size!=0){
      transferbytes=4*XMaxRequestSize(getApp()->display);    // Maximum number of bytes transferred in one shot
      mode=PropModeReplace;
      offset=0;
      format=8;
      FXASSERT(size!=0);
      if(targettype==stringType) size--;                // For XA_STRING, substract the '\0'
      do{
        if(size<transferbytes) tlen=size; else tlen=transferbytes;
        XChangeProperty(getApp()->display,getApp()->event.requestor,getApp()->event.property,getApp()->event.target,format,mode,&data[offset],tlen);
        mode=PropModeAppend;
        size-=tlen;
        offset+=tlen;
        }
      while(size!=0);
      FXFREE(&data);
      }
    else{
      getApp()->event.property=None;
      }
    }
  return TRUE;
  
#else
  
  if(xid==0){ fxerror("%s::setDNDData: window has not yet been created.\n",getClassName()); }
  if(origin!=getApp()->event.origin){ fxerror("%s::setDNDData: DDE source was not requested.\n",getClassName()); }

  FXTRACE((100,"%s: setDNDData()\n",getClassName()));

  // Was called in response to a WM_RENDERFORMAT
  if(origin==FROM_CLIPBOARD){
    HGLOBAL hGlobalMemory=GlobalAlloc(GHND,size);
    FXASSERT(hGlobalMemory);
    if(hGlobalMemory){
      void *pGlobalMemory=GlobalLock(hGlobalMemory);
      FXASSERT(pGlobalMemory);
      memcpy((FXchar*)pGlobalMemory,data,size);
      GlobalUnlock(hGlobalMemory);
      SetClipboardData(targettype,hGlobalMemory);
      FXFREE(&getApp()->ddeData);
      getApp()->ddeData=NULL;
      getApp()->ddeSize=0;
      return TRUE;
      }
    return FALSE;
    }
  else if(origin==FROM_DRAGNDROP){
    // Window is in same application; bypass protocol by storing data into the mailbox
    if(GetWindowThreadProcessId((HWND)xid,NULL)==GetWindowThreadProcessId((HWND)getApp()->event.requestor,NULL)){
      FXWindow *reqobject=(FXWindow*)GetWindowLong((HWND)getApp()->event.requestor,0);
      FXASSERT(reqobject!=0);
      FXTRACE((100,"%s: setDNDData(target=%d,data=%08x,size=%d) for %d (%s)\n",getClassName(),targettype,data,size,getApp()->event.requestor,reqobject->getClassName()));
      getApp()->ddeData=data;
      getApp()->ddeSize=size;
      }
    else{
      // Requesting window is in another application, so create a shared memory
      // section and stick the data there.
      FXTRACE((100,"%s: setDNDData(target=%d,data=%08x,size=%d) for %d (remote)\n",getClassName(),targettype,data,size,getApp()->event.requestor));
      if(data!=NULL && size!=0){
	HANDLE hMap=CreateFileMapping((HANDLE)0xFFFFFFFF,NULL,PAGE_READWRITE,0,size,"_FOX_DDE");
	FXASSERT(hMap!=0);
	if(hMap!=0){
	  FXASSERT(GetLastError()!=ERROR_ALREADY_EXISTS);
	  LPVOID lpView=MapViewOfFile(hMap,FILE_MAP_WRITE,0,0,0);
	  FXASSERT(lpView!=0);
	  if(lpView){
	    getApp()->dragContext->ddeSize=size; // Not sure how else to send the size of the shared memory block...
	    memcpy(lpView,data,size);
	    UnmapViewOfFile(lpView);
	    }
	  SetProp((HWND)xid,"_FOX_DDE_hMap",hMap);
	  }
	FXFREE(&data);
	}
      }
    return TRUE;
    }
  else {
    // Win32 doesn't have a selection!
    FXASSERT(FALSE);
    }
  return FALSE;
#endif
  }


// Target says accepts the drop
void FXWindow::acceptDrop(FXDragAction action){
  if(xid==0){ fxerror("%s::acceptDrop: window has not yet been created.\n",getClassName()); }
  if(getApp()->xdndTarget!=xid){ fxerror("%s::acceptDrop: window is not the drop target.\n",getClassName()); }

  // Accept anything or suggest a specific action?
  if(action!=DRAG_ACCEPT){
    if(action==DRAG_COPY) getApp()->xdndAction=getApp()->xdndActionCopy;
    if(action==DRAG_MOVE) getApp()->xdndAction=getApp()->xdndActionMove;
    if(action==DRAG_LINK) getApp()->xdndAction=getApp()->xdndActionLink;
    }
  
  // Accept a drop?
  getApp()->xdndAccepts=(action!=DRAG_REJECT);
  }


// Inquire about types being dragged or available on the clipboard or selection
FXbool FXWindow::inquireDNDTypes(FXDNDOrigin origin,FXDragType*& types,FXuint& numtypes) const {
  if(xid==0){ fxerror("%s::inquireDNDTypes: window has not yet been created.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
  unsigned long ni,b; int fmt; Atom typ;
  unsigned char* data=NULL;
  FXWindow *ownerobject;
  Window ownerwindow;
  XEvent ev;
  XMatch match;
  Bool status;
  
  types=NULL;
  numtypes=0;
  switch(origin){
    
    case FROM_DRAGNDROP:
      
      // Find window owning XDND selection
      ownerwindow=XGetSelectionOwner(getApp()->display,getApp()->xdndSelection);
      if(ownerwindow==None) return FALSE;
  
      // Window is in same application; bypass protocol by returning data from the mailbox
      if(XFindContext(getApp()->display,ownerwindow,getApp()->wcontext,(XPointer*)&ownerobject)==0){
        FXASSERT(ownerobject);
        FXMALLOC(&types,FXDragType,getApp()->xdndNumTypes);
        memcpy(types,getApp()->xdndTypeList,sizeof(FXDragType)*getApp()->xdndNumTypes);
        numtypes=getApp()->xdndNumTypes;
        FXTRACE((100,"%s::inquireDNDTypes(FROM_DRAGNDROP) received %d types from %d (%s)\n",getClassName(),numtypes,ownerwindow,ownerobject->getClassName()));
        return TRUE;
        }
      
      // We have to get it the hard way, by getting the property
      if(XGetWindowProperty(getApp()->display,ownerwindow,getApp()->xdndTypes,0,1024,False,XA_ATOM,&typ,&fmt,&ni,&b,&data)==Success){
        if(data){
          if(typ==XA_ATOM && fmt==32 && ni>0){
            FXMALLOC(&types,FXDragType,ni);
            memcpy(types,data,sizeof(FXDragType)*ni);
            numtypes=ni;
            XFree(data);
            FXTRACE((100,"%s::inquireDNDTypes(FROM_DRAGNDROP) received %d types from %d (remote)\n",getClassName(),numtypes,ownerwindow));
            return TRUE;
            }
          XFree(data);
          }
        }
      break;
      
    case FROM_CLIPBOARD:

      // Find window owning CLIPBOARD selection
      ownerwindow=XGetSelectionOwner(getApp()->display,getApp()->xcbSelection);
      if(ownerwindow==None) return FALSE;
  
      // Window is in same application; bypass protocol by returning data from the mailbox
      if(XFindContext(getApp()->display,ownerwindow,getApp()->wcontext,(XPointer*)&ownerobject)==0){
        FXASSERT(ownerobject);
        FXMALLOC(&types,FXDragType,getApp()->xcbNumTypes);
        memcpy(types,getApp()->xcbTypeList,sizeof(FXDragType)*getApp()->xcbNumTypes);
        numtypes=getApp()->xcbNumTypes;
        FXTRACE((100,"%s::inquireDNDTypes(FROM_CLIPBOARD) received %d types from %d (%s)\n",getClassName(),numtypes,ownerwindow,ownerobject->getClassName()));
        return TRUE;
        }

      // We have to get it the hard way, ask for TARGETS as per ICCCM
      XConvertSelection(getApp()->display,getApp()->xcbSelection,getApp()->reqTargets,getApp()->ddeAtom,xid,getApp()->event.time);

      // Wait for a response
      match.window=xid;
      match.selection=getApp()->xcbSelection;
      match.gotit=1;
      status=XCheckIfEvent(getApp()->display,&ev,notifyevent,(XPointer)&match);

      // Broke out of event loop
      if(!match.gotit) return FALSE;
      if(!status){
        match.gotit=1;
        XIfEvent(getApp()->display,&ev,notifyevent,(XPointer)&match);
        if(!match.gotit) return FALSE;
        }

      // There was no selection or no data has been specified
      if(ev.xselection.property==None){
        return FALSE;
        }

      // We have to get it the hard way, by getting the property
      if(XGetWindowProperty(getApp()->display,xid,getApp()->ddeAtom,0,1024,True,XA_ATOM,&typ,&fmt,&ni,&b,&data)==Success){
        if(data){
          if(typ==XA_ATOM && fmt==32 && ni>0){
            FXMALLOC(&types,FXDragType,ni);
            memcpy(types,data,sizeof(FXDragType)*ni);
            numtypes=ni;
            XFree(data);
            FXTRACE((100,"%s::inquireDNDTypes(FROM_CLIPBOARD) received %d types from %d (remote)\n",getClassName(),numtypes,ownerwindow));
            return TRUE;
            }
          XFree(data);
          }
        }
      break;
      
    case FROM_SELECTION:
      
      // Find window owning CLIPBOARD selection
      ownerwindow=XGetSelectionOwner(getApp()->display,XA_PRIMARY);
      if(ownerwindow==None) return FALSE;
  
      // Window is in same application; bypass protocol by returning data from the mailbox
      if(XFindContext(getApp()->display,ownerwindow,getApp()->wcontext,(XPointer*)&ownerobject)==0){
        FXASSERT(ownerobject);
        FXMALLOC(&types,FXDragType,getApp()->xselNumTypes);
        memcpy(types,getApp()->xselTypeList,sizeof(FXDragType)*getApp()->xselNumTypes);
        numtypes=getApp()->xselNumTypes;
        FXTRACE((100,"%s::inquireDNDTypes(FROM_SELECTION) received %d types from %d (%s)\n",getClassName(),numtypes,ownerwindow,ownerobject->getClassName()));
        return TRUE;
        }

      // We have to get it the hard way, ask for TARGETS as per ICCCM
      XConvertSelection(getApp()->display,XA_PRIMARY,getApp()->reqTargets,getApp()->ddeAtom,xid,getApp()->event.time);

      // Wait for a response
      match.window=xid;
      match.selection=XA_PRIMARY;
      match.gotit=1;
      status=XCheckIfEvent(getApp()->display,&ev,notifyevent,(XPointer)&match);

      // Broke out of event loop
      if(!match.gotit) return FALSE;
      if(!status){
        match.gotit=1;
        XIfEvent(getApp()->display,&ev,notifyevent,(XPointer)&match);
        if(!match.gotit) return FALSE;
        }

      // There was no selection or no data has been specified
      if(ev.xselection.property==None){
        return FALSE;
        }

      // We have to get it the hard way, by getting the property
      if(XGetWindowProperty(getApp()->display,xid,getApp()->ddeAtom,0,128,True,XA_ATOM,&typ,&fmt,&ni,&b,&data)==Success){
        if(data){
          if(typ==XA_ATOM && fmt==32 && ni>0){
            FXMALLOC(&types,FXDragType,ni);
            memcpy(types,data,sizeof(FXDragType)*ni);
            numtypes=ni;
            XFree(data);
            FXTRACE((100,"%s::inquireDNDTypes(FROM_SELECTION) received %d types from %d (remote)\n",getClassName(),numtypes,ownerwindow));
            return TRUE;
            }
          XFree(data);
          }
        }
    case FROM_UNKNOWN:
      break;
    }

#else

  FXWindow *ownerobject;
  HWND ownerwindow;

  types=NULL;
  numtypes=0;

  switch(origin){
    case FROM_DRAGNDROP:
      // Get the drag source window handle
      ownerwindow=getApp()->xdndSource;
      if(ownerwindow==0)
	return FALSE;

      // Window is in same application; bypass protocol by returning data from the mailbox
      if(GetWindowThreadProcessId((HWND)xid,NULL)==GetWindowThreadProcessId(ownerwindow,NULL)){
	ownerobject=(FXWindow*)GetWindowLong(ownerwindow,0);
        FXASSERT(ownerobject);
        FXMALLOC(&types,FXDragType,getApp()->xdndNumTypes);
        memcpy(types,getApp()->xdndTypeList,sizeof(FXDragType)*getApp()->xdndNumTypes);
        numtypes=getApp()->xdndNumTypes;
        FXTRACE((100,"%s::inquireDNDTypes(FROM_DRAGNDROP) received %d types from %d (%s)\n",getClassName(),numtypes,ownerwindow,ownerobject->getClassName()));
        return TRUE;
        }
      
      // Copy it out of the drag context
      numtypes=getApp()->dragContext->numTypes;
      if(numtypes>0){
	FXMALLOC(&types,FXDragType,numtypes);
	if(types){
	  HANDLE hMap=OpenFileMapping(FILE_MAP_READ,FALSE,"XdndTypeList");
	  FXASSERT(hMap);
	  if(hMap){
	    LPVOID lpView=MapViewOfFile(hMap,FILE_MAP_READ,0,0,0);
	    FXASSERT(lpView);
	    if(lpView){
	      memcpy(types,lpView,numtypes*sizeof(FXDragType));
	      UnmapViewOfFile(lpView);
	      }
	    CloseHandle(hMap);
	    }
	  }
	FXTRACE((100,"%s::inquireDNDTypes(FROM_DRAGNDROP) received %d types from %d (remote)\n",getClassName(),numtypes,ownerwindow));
	return TRUE;
	}
      break;
    case FROM_CLIPBOARD:
      if(OpenClipboard((HWND)xid)){
	numtypes=CountClipboardFormats();
	if(numtypes>0){
	  FXMALLOC(&types,FXDragType,numtypes);
	  numtypes=0;
	  UINT format=0;
	  while(0!=(format=EnumClipboardFormats(format))){
	    types[numtypes++]=format;
	    }
	  }
	CloseClipboard();
	return TRUE;
	}
      break;
    case FROM_SELECTION:
      FXASSERT(FALSE);
      break;
    }

#endif

  return FALSE;
  }


// Is a certain type being offered via drag and drop, clipboard, or selection?
FXbool FXWindow::offeredDNDType(FXDNDOrigin origin,FXDragType type) const {
  if(xid==0){ fxerror("%s::offeredDNDType: window has not yet been created.\n",getClassName()); }
  FXbool offered=FALSE;
  FXDragType *types;
  FXuint i,ntypes;
  if(inquireDNDTypes(origin,types,ntypes)){
    for(i=0; i<ntypes; i++){
      if(type==types[i]){ offered=TRUE; break; }
      }
    FXFREE(&types);
    }
  return offered;
  }


// What is the drag action being performed?
FXDragAction FXWindow::inquireDNDAction() const {
  FXDragAction action=DRAG_COPY;
  if(xid==0){ fxerror("%s::inquireDNDAction: window has not yet been created.\n",getClassName()); }
  if(getApp()->xdndTarget!=xid){ fxerror("%s::inquireDNDAction: window is not the drop target.\n",getClassName()); }
  if(getApp()->xdndAction==getApp()->xdndActionCopy) action=DRAG_COPY;
  if(getApp()->xdndAction==getApp()->xdndActionMove) action=DRAG_MOVE;
  if(getApp()->xdndAction==getApp()->xdndActionLink) action=DRAG_LINK;
  return action;
  }


// Return non-zero if we can drop on window
FXuint FXWindow::mayDropOn(FXID dropwindow) const {
  FXuint version=0;
#ifndef FX_NATIVE_WIN32
  unsigned long itemCount,remainingBytes;
  unsigned char* rawData=NULL;
  Atom actualType;
  FXuint i,j;
  int actualFormat;
  if(XGetWindowProperty(getApp()->display,dropwindow,getApp()->xdndAware,0,1024,False,XA_ATOM,&actualType,&actualFormat,&itemCount,&remainingBytes,&rawData)==Success){
    if(rawData){
      if(actualType==XA_ATOM && actualFormat==32 && itemCount>0){
        
        // Version of the protocol
        version=*((FXuint*)rawData) | 0x80000000;
        
        // Did target specify specific types?
        if(itemCount>1){
          
          // Try match up the type lists
          for(i=0; i<getApp()->xdndNumTypes; i++){
            for(j=1; j<itemCount; j++){
              if(getApp()->xdndTypeList[i]==((FXuint*)rawData)[j]) goto ok;
              }
            }
          
          // Too bad, didn't find it
          version=0;
          }
        }
ok:   XFree(rawData);
      }
    }
#else
  HANDLE hData=GetProp((HWND)dropwindow,(LPCTSTR)MAKELONG(getApp()->xdndAware,0));
  if(hData){
    // Version of the protocol
    version=(FXuint)hData;

    // JLJ: The X implementation (above) also checks to see if any types
    // were listed as a part of the property data for XdndAware; but
    // I don't see any place in the FOX code where we store any data
    // in XdndAware other than the protocol version number. Does this
    // maybe account for some previous version of XDND which did store
    // part of the types list in its XdndAware property???
    }
#endif
  return version;
  }



// Source wants to find out if target accepted
FXDragAction FXWindow::didAccept() const {
  FXDragAction action=DRAG_REJECT;
  if(xid==0){ fxerror("%s::didAccept: window has not yet been created.\n",getClassName()); }
  if(getApp()->xdndSource!=xid){ fxerror("%s::didAccept: window is not a drag source.\n",getClassName()); }
  if(getApp()->xdndAccepts){
    if(getApp()->xdndAction==getApp()->xdndActionCopy) action=DRAG_COPY;
    if(getApp()->xdndAction==getApp()->xdndActionMove) action=DRAG_MOVE;
    if(getApp()->xdndAction==getApp()->xdndActionLink) action=DRAG_LINK;
    }
  return action;
  }


// True if we're in a drag operation
FXbool FXWindow::isDragging() const { 
  return (getApp()->dragWindow==this);
  }


// True if this window is drop target
FXbool FXWindow::isDropTarget() const {
  if(xid==0){ fxerror("%s::isDropTarget: window has not yet been created.\n",getClassName()); }
  return xid==getApp()->xdndTarget;
  }
  

// Start a drag on the types mentioned
FXbool FXWindow::beginDrag(const FXDragType *types,FXuint numtypes){
  if(xid==0){ fxerror("%s::beginDrag: window has not yet been created.\n",getClassName()); }
  if(getApp()->dragWindow){ fxerror("%s::beginDrag: trying a second drag while first still in effect.\n",getClassName()); }
  if(getApp()->dragWindow==NULL){
    if(types==NULL || numtypes<1){ fxerror("%s::beginDrag: should have at least one type to drag.\n",getClassName()); }
#ifndef FX_NATIVE_WIN32
    XSetSelectionOwner(getApp()->display,getApp()->xdndSelection,xid,getApp()->event.time);
    if(XGetSelectionOwner(getApp()->display,getApp()->xdndSelection)!=xid){
      fxwarning("%s::beginDrag: failed to acquire DND selection.\n",getClassName());
      return FALSE;
      }
    getApp()->xdndTypeList=(FXID*)malloc(sizeof(FXID)*numtypes);
#else
    FXASSERT(getApp()->xdndTypeList==0);
    FXMALLOC(&getApp()->xdndTypeList,FXDragType,numtypes);
#endif
    if(!getApp()->xdndTypeList) return FALSE;
    getApp()->xdndNumTypes=numtypes;
    for(FXuint num=0; num<numtypes; num++){
      getApp()->xdndTypeList[num]=types[num];
      }
#ifndef FX_NATIVE_WIN32
    XChangeProperty(getApp()->display,xid,getApp()->xdndTypes,XA_ATOM,32,PropModeReplace,(const FXuchar*)getApp()->xdndTypeList,getApp()->xdndNumTypes);
    getApp()->xdndSource=xid;
    getApp()->xdndTarget=xid;
#else
    getApp()->dragContext->numTypes=getApp()->xdndNumTypes;
    FXASSERT(getApp()->xdndTypes==0);
    getApp()->xdndTypes=CreateFileMapping((HANDLE)0xFFFFFFFF,NULL,PAGE_READWRITE,0,getApp()->xdndNumTypes*sizeof(FXDragType),"XdndTypeList");
    FXASSERT(getApp()->xdndTypes!=0);
    if(getApp()->xdndTypes){
      FXASSERT(GetLastError()!=ERROR_ALREADY_EXISTS);
      LPVOID lpView=MapViewOfFile(getApp()->xdndTypes,FILE_MAP_WRITE,0,0,0);
      FXASSERT(lpView!=0);
      if(lpView){
	memcpy(lpView,getApp()->xdndTypeList,numtypes*sizeof(FXDragType));
	UnmapViewOfFile(lpView);
	}
      }
    getApp()->xdndSource=(HWND)xid;
    getApp()->xdndTarget=(HWND)xid;
#endif
    getApp()->xdndAccepts=FALSE;
    getApp()->xdndSendPosition=FALSE;
    getApp()->xdndStatusPending=FALSE;
    getApp()->xdndStatusReceived=FALSE;
    getApp()->xdndWantUpdates=FALSE;
#ifndef FX_NATIVE_WIN32
    getApp()->xdndVersion=XDND_PROTOCOL_VERSION;
#endif
    getApp()->xdndAction=getApp()->xdndActionCopy;
    getApp()->xdndRect.x=0;
    getApp()->xdndRect.y=0;
    getApp()->xdndRect.w=0;
    getApp()->xdndRect.h=0;
    getApp()->xdndXPos=0;
    getApp()->xdndYPos=0;
    getApp()->dragWindow=this;
    return TRUE;
    }
  return FALSE;
  }


// Drag to new position
FXbool FXWindow::handleDrag(FXint x,FXint y,FXDragAction action){
  if(xid==0){ fxerror("%s::handleDrag: window has not yet been created.\n",getClassName()); }
  if(action<DRAG_COPY || DRAG_LINK<action){ fxerror("%s::handleDrag: illegal drag action.\n",getClassName()); }
    
#ifndef FX_NATIVE_WIN32
    
  if(getApp()->dragWindow==this){
    FXuint version=XDND_PROTOCOL_VERSION;
    FXuint forcepos=0;
    FXWindow *tgt;
    Window child,tmp;
    XEvent se;
    
    if(getApp()->xdndSource!=xid){ fxerror("%s::handleDrag: window is not a drag source.\n",getClassName()); }
  
    // Get drop window
    child=getWindowAt(x,y);
    
    // Got proper drop window?
    if(child!=XDefaultRootWindow(getApp()->display)){
      
      // Test for XDND awareness
      version=mayDropOn(child);
      if(!version){
        child=XDefaultRootWindow(getApp()->display);
        }
      }
    
    version&=0xff;
    
    // Moved to different window?
    if(child!=getApp()->xdndTarget){
      
      // Moving OUT of XDND aware window?
      if(getApp()->xdndTarget!=XDefaultRootWindow(getApp()->display)){
        
        // If local, bypass protocol
        if(XFindContext(getApp()->display,getApp()->xdndTarget,getApp()->wcontext,(XPointer*)&tgt)==0){
          FXASSERT(tgt);
          getApp()->event.type=SEL_DND_LEAVE;
          getApp()->event.window=getApp()->xdndTarget;
          tgt->handle(getApp(),MKUINT(0,SEL_DND_LEAVE),&getApp()->event);
          }
        
        // Else send through protocol
        else{
          se.xclient.type=ClientMessage;
          se.xclient.display=getApp()->display;
          se.xclient.message_type=getApp()->xdndLeave;
          se.xclient.format=32;
          se.xclient.window=getApp()->xdndTarget;
          se.xclient.data.l[0]=getApp()->xdndSource;
          se.xclient.data.l[1]=0;
          se.xclient.data.l[2]=0;
          se.xclient.data.l[3]=0;
          se.xclient.data.l[4]=0;
          XSendEvent(getApp()->display,getApp()->xdndTarget,True,NoEventMask,&se);
          }
        }
      
      // Reset XDND variables
      getApp()->xdndAccepts=FALSE;
      getApp()->xdndSendPosition=FALSE;
      getApp()->xdndStatusPending=FALSE;
      getApp()->xdndStatusReceived=FALSE;
      getApp()->xdndVersion=FXMIN(version,XDND_PROTOCOL_VERSION);
      getApp()->xdndAction=getApp()->xdndActionCopy;
      getApp()->xdndWantUpdates=TRUE;
      getApp()->xdndRect.x=0;
      getApp()->xdndRect.y=0;
      getApp()->xdndRect.w=0;
      getApp()->xdndRect.h=0;
      getApp()->xdndXPos=0;
      getApp()->xdndYPos=0;
      getApp()->xdndTarget=child;
      
      // Moving INTO XDND aware window?
      if(getApp()->xdndTarget!=XDefaultRootWindow(getApp()->display)){
        
        // If local, bypass protocol
        if(XFindContext(getApp()->display,getApp()->xdndTarget,getApp()->wcontext,(XPointer*)&tgt)==0){
          FXASSERT(tgt);
          getApp()->event.type=SEL_DND_ENTER;
          getApp()->event.window=getApp()->xdndTarget;
          tgt->handle(getApp(),MKUINT(0,SEL_DND_ENTER),&getApp()->event);
          }
        
        // Else send through protocol
        else{
          se.xclient.type=ClientMessage;
          se.xclient.display=getApp()->display;
          se.xclient.message_type=getApp()->xdndEnter;
          se.xclient.format=32;
          se.xclient.window=getApp()->xdndTarget;
          se.xclient.data.l[0]=getApp()->xdndSource;
          se.xclient.data.l[1]=getApp()->xdndVersion<<24;
          se.xclient.data.l[2]=getApp()->xdndNumTypes>=1 ? getApp()->xdndTypeList[0] : None;
          se.xclient.data.l[3]=getApp()->xdndNumTypes>=2 ? getApp()->xdndTypeList[1] : None;
          se.xclient.data.l[4]=getApp()->xdndNumTypes>=3 ? getApp()->xdndTypeList[2] : None;
          if(getApp()->xdndNumTypes>3) se.xclient.data.l[1]|=1;
          XSendEvent(getApp()->display,getApp()->xdndTarget,True,NoEventMask,&se);
          }
        forcepos=1;
        }
      }

    // Send target a position update
    if(getApp()->xdndTarget!=XDefaultRootWindow(getApp()->display)){
      
      // Send position if we're outside the mouse box or ignoring it
      if(forcepos || getApp()->xdndRect.w==0 || getApp()->xdndRect.h==0 || (getApp()->xdndWantUpdates && getApp()->xdndRect.x<=x && getApp()->xdndRect.y<=y && x<getApp()->xdndRect.x+getApp()->xdndRect.w && y<getApp()->xdndRect.y+getApp()->xdndRect.h)){

        // No outstanding status message, so send new pos right away      
        if(!getApp()->xdndStatusPending){
          
          // Update drag action being performed
          if(action==DRAG_COPY) getApp()->xdndAction=getApp()->xdndActionCopy;
          if(action==DRAG_MOVE) getApp()->xdndAction=getApp()->xdndActionMove;
          if(action==DRAG_LINK) getApp()->xdndAction=getApp()->xdndActionLink;
          
          // New position
          getApp()->xdndXPos=x;
          getApp()->xdndYPos=y;
          
          // If local, bypass protocol
          if(XFindContext(getApp()->display,getApp()->xdndTarget,getApp()->wcontext,(XPointer*)&tgt)==0){
            FXASSERT(tgt);
            getApp()->event.type=SEL_DND_MOTION;
            getApp()->event.window=getApp()->xdndTarget;
            getApp()->event.root_x=x;
            getApp()->event.root_y=y;
            XTranslateCoordinates(getApp()->display,XDefaultRootWindow(getApp()->display),getApp()->xdndTarget,x,y,&getApp()->event.win_x,&getApp()->event.win_y,&tmp);
            getApp()->xdndAccepts=FALSE;
            tgt->handle(getApp(),MKUINT(0,SEL_DND_MOTION),&getApp()->event);
            getApp()->xdndStatusReceived=TRUE;
            getApp()->xdndStatusPending=FALSE;      // We know it got there, as its local
            }
          
          // Else send through protocol
          else{
            se.xclient.type=ClientMessage;
            se.xclient.display=getApp()->display;
            se.xclient.message_type=getApp()->xdndPosition;
            se.xclient.format=32;
            se.xclient.window=getApp()->xdndTarget;
            se.xclient.data.l[0]=getApp()->xdndSource;
            se.xclient.data.l[1]=0;
            se.xclient.data.l[2]=MKUINT(y,x);                               // Coordinates
            se.xclient.data.l[3]=getApp()->event.time;                      // Time stamp
            se.xclient.data.l[4]=getApp()->xdndAction;                      // Drag and Drop action
            XSendEvent(getApp()->display,getApp()->xdndTarget,True,NoEventMask,&se);
            getApp()->xdndStatusPending=TRUE;       // Waiting for the other app to respond
            }
          }

        // Send it later
        else{
          getApp()->xdndSendPosition=TRUE;
          }
        }
      }
    return TRUE;
    }
  return FALSE;
  
#else
  
  if(getApp()->dragWindow==this){
  
    FXuint version=XDND_PROTOCOL_VERSION;
    FXWindow *tgt;
    FXbool forcepos=FALSE;

    if(getApp()->xdndSource!=xid){ fxerror("%s::handleDrag: window is not a drag source.\n",getClassName()); }

    // Get drop window
    HWND child=(HWND)getWindowAt(x,y);
    
    // Got proper drop window?
    if(child!=GetDesktopWindow()){
      // Test for XDND awareness
      version=mayDropOn(child);
      if(!version){
        child=GetDesktopWindow();
        }
      }

    version&=0xff;
    
    // Moved to different window?
    if(child!=getApp()->xdndTarget){
      
      // Moving OUT of XDND aware window?
      if(getApp()->xdndTarget!=GetDesktopWindow()){
        
        // If local, bypass protocol
	if(GetWindowThreadProcessId((HWND)xid,NULL)==GetWindowThreadProcessId(getApp()->xdndTarget,NULL)){
	  tgt=(FXWindow*)GetWindowLong(getApp()->xdndTarget,0);
          FXASSERT(tgt);
          getApp()->event.type=SEL_DND_LEAVE;
          getApp()->event.window=getApp()->xdndTarget;
          tgt->handle(getApp(),MKUINT(0,SEL_DND_LEAVE),&getApp()->event);
          }
        
        // Else send through protocol
        else{
	  SendMessage(getApp()->xdndTarget,getApp()->xdndLeave,0,(LPARAM)xid);
          }
        }
      
      // Reset XDND variables
      getApp()->xdndAccepts=FALSE;
      getApp()->xdndSendPosition=FALSE;
      getApp()->xdndStatusPending=FALSE;
      getApp()->xdndStatusReceived=FALSE;
      getApp()->xdndAction=getApp()->xdndActionCopy;
      getApp()->xdndWantUpdates=TRUE;
      getApp()->xdndRect.x=0;
      getApp()->xdndRect.y=0;
      getApp()->xdndRect.w=0;
      getApp()->xdndRect.h=0;
      getApp()->xdndXPos=0;
      getApp()->xdndYPos=0;
      getApp()->xdndTarget=child;
      
      // Moving INTO XDND aware window?
      if(getApp()->xdndTarget!=GetDesktopWindow()){
        
        // If local, bypass protocol
	if(GetWindowThreadProcessId((HWND)xid,NULL)==GetWindowThreadProcessId(getApp()->xdndTarget,NULL)){
	  tgt=(FXWindow*)GetWindowLong(getApp()->xdndTarget,0);
          FXASSERT(tgt);
          getApp()->event.type=SEL_DND_ENTER;
          getApp()->event.window=getApp()->xdndTarget;
          tgt->handle(getApp(),MKUINT(0,SEL_DND_ENTER),&getApp()->event);
          }
        
        // Else send through protocol
        else{
	  getApp()->dragContext->numTypes=getApp()->xdndNumTypes;
	  SendMessage(getApp()->xdndTarget,getApp()->xdndEnter,getApp()->xdndNumTypes,(LPARAM)xid);
          }
        forcepos=TRUE;
        }
      }

    // Send target a position update
    if(getApp()->xdndTarget!=GetDesktopWindow()){
      
      // Send position if we're outside the mouse box or ignoring it
      if(forcepos || getApp()->xdndRect.w==0 || getApp()->xdndRect.h==0 || (getApp()->xdndWantUpdates && getApp()->xdndRect.x<=x && getApp()->xdndRect.y<=y && x<getApp()->xdndRect.x+getApp()->xdndRect.w && y<getApp()->xdndRect.y+getApp()->xdndRect.h)){

        // No outstanding status message, so send new pos right away
        if(!getApp()->xdndStatusPending){
          
          // Update drag action being performed
          if(action==DRAG_COPY) getApp()->xdndAction=getApp()->xdndActionCopy;
          if(action==DRAG_MOVE) getApp()->xdndAction=getApp()->xdndActionMove;
          if(action==DRAG_LINK) getApp()->xdndAction=getApp()->xdndActionLink;
          
          // New position
          getApp()->xdndXPos=x;
          getApp()->xdndYPos=y;
          
          // If local, bypass protocol
	  if(GetWindowThreadProcessId((HWND)xid,NULL)==GetWindowThreadProcessId(getApp()->xdndTarget,NULL)){
	    tgt=(FXWindow*)GetWindowLong(getApp()->xdndTarget,0);
            FXASSERT(tgt);
            getApp()->event.type=SEL_DND_MOTION;
            getApp()->event.window=getApp()->xdndTarget;
            getApp()->event.root_x=x;
            getApp()->event.root_y=y;
	    POINT pt;
	    pt.x=x;
	    pt.y=y;
	    ScreenToClient(getApp()->xdndTarget,&pt);
	    getApp()->event.win_x=pt.x;
	    getApp()->event.win_y=pt.y;
            getApp()->xdndAccepts=FALSE;
            tgt->handle(getApp(),MKUINT(0,SEL_DND_MOTION),&getApp()->event);
            getApp()->xdndStatusReceived=TRUE;
            getApp()->xdndStatusPending=FALSE;      // We know it got there, as its local
            }
          
          // Else send through protocol
          else{
	    getApp()->dragContext->time=getApp()->event.time;
	    getApp()->dragContext->action=getApp()->xdndAction;
	    SendMessage(getApp()->xdndTarget,getApp()->xdndPosition,MAKELONG(x,y),(LPARAM)xid);
            getApp()->xdndStatusPending=TRUE;       // Waiting for the other app to respond
            }
          }

        // Send it later
        else{
          getApp()->xdndSendPosition=TRUE;
          }
        }
      }

    return TRUE;
    }
  return FALSE;
  
#endif
  }


#ifndef FX_NATIVE_WIN32

// Scan for status and selection requests...
static Bool matchevent(Display*,XEvent* event,XPointer match){
  if(event->type == ClientMessage && event->xclient.message_type == *((Atom*)match)) return True;
  if(event->type == SelectionRequest) return True;
  return False;
  }

#endif

#ifndef FX_NATIVE_WIN32

// Compare times
static inline int operator<(const struct timeval& a,const struct timeval& b){
  return (a.tv_sec<b.tv_sec) || (a.tv_sec==b.tv_sec && a.tv_usec<b.tv_usec);
  }

#endif

// Terminate the drag; if drop flag is false, don't drop even if accepted.
FXbool FXWindow::endDrag(FXbool drop){
  FXbool dropped=FALSE;
#ifndef FX_NATIVE_WIN32
  FXbool nodrop=TRUE;
  FXWindow *tgt;
  XEvent se;
  FXuint loops;
  if(xid==0){ fxerror("%s::endDrag: window has not yet been created.\n",getClassName()); }
  if(getApp()->dragWindow==this){
    if(getApp()->xdndSource!=xid){ fxerror("%s::endDrag: window is not a drag source.\n",getClassName()); }

    // Ever received a status
    if(getApp()->xdndStatusReceived && drop){
      
      // If a status message is still pending, wait for it
      if(getApp()->xdndStatusPending){
        getApp()->xdndSendPosition=FALSE;          // No more position updates sent to this target
        loops=1000;
        while(loops){
          if(XCheckIfEvent(getApp()->display,&se,matchevent,(char*)&getApp()->xdndStatus)){
            getApp()->dispatchEvent(se);
            if(se.xany.type==ClientMessage && se.xclient.data.l[0]==getApp()->xdndTarget){
              getApp()->xdndStatusPending=FALSE;
              break;
              }
 
            // We either got a selection request or a status message for a stale target.
            // Since there is thus still hope for a message, lets wait a bit longer...
            loops=1000;
            }
          fxsleep(20000);
          loops--;
          }
        }
      
      // Got our status message
      if(!getApp()->xdndStatusPending && getApp()->xdndAccepts){
        
        // If local, bypass protocol
        if(XFindContext(getApp()->display,getApp()->xdndTarget,getApp()->wcontext,(XPointer*)&tgt)==0){
          FXASSERT(tgt);
          getApp()->event.type=SEL_DND_DROP;
          getApp()->event.window=getApp()->xdndTarget;
          tgt->handle(getApp(),MKUINT(0,SEL_DND_DROP),&getApp()->event);
          getApp()->xdndFinishPending=FALSE;        // We just handled the callback
          dropped=TRUE;                             // We're sure, as its handled directly!
          }
 
        // Else send through protocol
        else{
          se.xclient.type=ClientMessage;
          se.xclient.display=getApp()->display;
          se.xclient.message_type=getApp()->xdndDrop;
          se.xclient.format=32;
          se.xclient.window=getApp()->xdndTarget;
          se.xclient.data.l[0]=getApp()->xdndSource;
          se.xclient.data.l[1]=0;
          se.xclient.data.l[2]=getApp()->event.time;
          se.xclient.data.l[3]=0;
          se.xclient.data.l[4]=0;
          XSendEvent(getApp()->display,getApp()->xdndTarget,True,NoEventMask,&se);
          getApp()->xdndFinishPending=TRUE;         // Waiting for the drop confirmation!
          }
        
        // We wait till the drop has been processed...
        // The target should request the data at some point;
        // We should wait here till the target has processed the drop message
        // or some amount of timeout has expired....
        // Some error code should be returned if the timer expired...
        if(getApp()->xdndFinishPending){
          loops=1000;
          while(loops){
            if(XCheckIfEvent(getApp()->display,&se,matchevent,(char*)&getApp()->xdndFinished)){
              getApp()->dispatchEvent(se);
              if(se.xany.type==ClientMessage && se.xclient.data.l[0]==getApp()->xdndTarget){
                getApp()->xdndFinishPending=FALSE;
                dropped=TRUE;
                break;
                }
 
              // We either got a selection request or a status message for a stale target.
              // Since there is thus still hope for a message, lets wait a bit longer...
              loops=1000;
              }
            fxsleep(20000);
            loops--;
            }
          }
        
        // We tried a drop
        nodrop=FALSE;
        }
      }
    
    // Didn't drop, or didn't get any response, so just send a leave
    if(nodrop){
      
      // If local, bypass protocol
      if(XFindContext(getApp()->display,getApp()->xdndTarget,getApp()->wcontext,(XPointer*)&tgt)==0){
        FXASSERT(tgt);
        getApp()->event.type=SEL_DND_LEAVE;
        getApp()->event.window=getApp()->xdndTarget;
        tgt->handle(getApp(),MKUINT(0,SEL_DND_LEAVE),&getApp()->event);
        }
      
      // Else send through protocol
      else{
        se.xclient.type=ClientMessage;
        se.xclient.display=getApp()->display;
        se.xclient.message_type=getApp()->xdndLeave;
        se.xclient.format=32;
        se.xclient.window=getApp()->xdndTarget;
        se.xclient.data.l[0]=getApp()->xdndSource;                  // Source window
        se.xclient.data.l[1]=0;
        se.xclient.data.l[2]=0;
        se.xclient.data.l[3]=0;
        se.xclient.data.l[4]=0;
        XSendEvent(getApp()->display,getApp()->xdndTarget,True,NoEventMask,&se);
        }
      }
      
    // Clean up
    XSetSelectionOwner(getApp()->display,getApp()->xdndSelection,None,CurrentTime); 
    if(getApp()->xdndTypeList) XFree(getApp()->xdndTypeList);
    getApp()->xdndTypeList=NULL;
    getApp()->xdndNumTypes=0;
    XDeleteProperty(getApp()->display,xid,getApp()->xdndTypes);
    getApp()->xdndSource=0;
    getApp()->xdndTarget=0;
    getApp()->xdndAccepts=FALSE;
    getApp()->xdndSendPosition=FALSE;
    getApp()->xdndStatusPending=FALSE;
    getApp()->xdndFinishPending=FALSE;
    getApp()->xdndStatusReceived=FALSE;
    getApp()->xdndWantUpdates=TRUE;
    getApp()->xdndVersion=XDND_PROTOCOL_VERSION;
    getApp()->xdndRect.x=0;
    getApp()->xdndRect.y=0;
    getApp()->xdndRect.w=0;
    getApp()->xdndRect.h=0;
    getApp()->xdndXPos=0;
    getApp()->xdndYPos=0;
    getApp()->dragWindow=NULL;
    }
  
#else
  
  FXbool nodrop=TRUE;
  FXWindow *tgt;
  if(xid==0){ fxerror("%s::endDrag: window has not yet been created.\n",getClassName()); }
  if(getApp()->dragWindow==this){
    if(getApp()->xdndSource!=(HWND)xid){ fxerror("%s::endDrag: window is not a drag source.\n",getClassName()); }

    // Ever received a status
    if(getApp()->xdndStatusReceived && drop){
      
      // If a status message is still pending, wait for it
      if(getApp()->xdndStatusPending){
	FXASSERT(FALSE);
	getApp()->xdndSendPosition=FALSE;
	getApp()->xdndStatusPending=FALSE;
	/*
        getApp()->xdndSendPosition=FALSE;          // No more position updates sent to this target
        loops=1000;
        while(loops){
	  MSG msg;
	  if(PeekMessage(&msg,NULL,0,0,PM_REMOVE)){
            getApp()->dispatchEvent(msg);
            if(msg.message==getApp()->xdndStatus && (HWND)msg.lParam==getApp()->xdndTarget){
              getApp()->xdndStatusPending=FALSE;
              break;
              }
 
            // We either got a selection request or a status message for a stale target.
            // Since there is thus still hope for a message, lets wait a bit longer...
            loops=1000;
            }
          fxsleep(20000);
          loops--;
          }
	*/
        }
      
      // Got our status message
      if(!getApp()->xdndStatusPending && getApp()->xdndAccepts){
        
        // If local, bypass protocol
	if(GetWindowThreadProcessId((HWND)xid,NULL)==GetWindowThreadProcessId(getApp()->xdndTarget,NULL)){
	  tgt=(FXWindow*)GetWindowLong(getApp()->xdndTarget,0);
          FXASSERT(tgt);
          getApp()->event.type=SEL_DND_DROP;
          getApp()->event.window=getApp()->xdndTarget;
          tgt->handle(getApp(),MKUINT(0,SEL_DND_DROP),&getApp()->event);
          getApp()->xdndFinishPending=FALSE;        // We just handled the callback
          dropped=TRUE;                             // We're sure, as its handled directly!
          }
 
        // Else send through protocol
        else{
	  SendMessage(getApp()->xdndTarget,getApp()->xdndDrop,0,(LPARAM)xid);
          getApp()->xdndFinishPending=TRUE;         // Waiting for the drop confirmation!
          }
        
        // We wait till the drop has been processed...
        // The target should request the data at some point;
        // We should wait here till the target has processed the drop message
        // or some amount of timeout has expired....
        // Some error code should be returned if the timer expired...
        if(getApp()->xdndFinishPending){
	  FXASSERT(FALSE);
	  getApp()->xdndFinishPending=FALSE;
	  dropped=TRUE;
	  /*
          loops=1000;
          while(loops){
	    MSG msg;
	    if(PeekMessage(&msg,NULL,0,0,PM_REMOVE)){
              getApp()->dispatchEvent(msg);
              if(msg.message==getApp()->xdndFinished && (HWND)msg.lParam==getApp()->xdndTarget){
                getApp()->xdndFinishPending=FALSE;
                dropped=TRUE;
                break;
                }
 
              // We either got a selection request or a status message for a stale target.
              // Since there is thus still hope for a message, let's wait a bit longer...
              loops=1000;
              }
            fxsleep(20000);
            loops--;
            }
	  */
          }

        // We tried a drop
        nodrop=FALSE;
        }
      }
    
    // Didn't drop, or didn't get any response, so just send a leave
    if(nodrop){
      
      // If local, bypass protocol
      if(GetWindowThreadProcessId((HWND)xid,NULL)==GetWindowThreadProcessId(getApp()->xdndTarget,NULL)){
	tgt=(FXWindow*)GetWindowLong(getApp()->xdndTarget,0);
        FXASSERT(tgt);
        getApp()->event.type=SEL_DND_LEAVE;
        getApp()->event.window=getApp()->xdndTarget;
        tgt->handle(getApp(),MKUINT(0,SEL_DND_LEAVE),&getApp()->event);
        }
      
      // Else send through protocol
      else{
	SendMessage(getApp()->xdndTarget,getApp()->xdndLeave,0,(LPARAM)xid);
        }
      }
      
    // Clean up
    if(getApp()->xdndTypeList){
      FXFREE(&getApp()->xdndTypeList);
      getApp()->xdndTypeList=NULL;
      }
    getApp()->xdndNumTypes=0;
    getApp()->xdndSource=0;
    getApp()->xdndTarget=0;
    getApp()->xdndAccepts=FALSE;
    getApp()->xdndSendPosition=FALSE;
    getApp()->xdndStatusPending=FALSE;
    getApp()->xdndFinishPending=FALSE;
    getApp()->xdndStatusReceived=FALSE;
    getApp()->xdndWantUpdates=TRUE;
    getApp()->xdndRect.x=0;
    getApp()->xdndRect.y=0;
    getApp()->xdndRect.w=0;
    getApp()->xdndRect.h=0;
    getApp()->xdndXPos=0;
    getApp()->xdndYPos=0;
    getApp()->dragWindow=NULL;

    // Get rid of the shared memory block containing the types list
    if(getApp()->xdndTypes){
      CloseHandle(getApp()->xdndTypes);
      getApp()->xdndTypes=0;
      }

    // Get rid of the shared memory block containing the data (if remote)
    HANDLE hMap=RemoveProp((HWND)xid,"_FOX_DDE_hMap");
    if(hMap!=0)
      CloseHandle(hMap);
    }
#endif
  return dropped;
  }


/*******************************************************************************/


// Delete window
FXWindow::~FXWindow(){
  delete accelTable;
  if(prev) prev->next=next; else if(parent) parent->first=next;
  if(next) next->prev=prev; else if(parent) parent->last=prev; 
  if(parent && parent->focus==this) parent->focus=NULL;
  if(getApp()->focusWindow==this) getApp()->focusWindow=NULL;
  if(getApp()->cursorWindow==this) getApp()->cursorWindow=parent;
  if(getApp()->grabWindow==this) getApp()->grabWindow=NULL;
  if(getApp()->keyWindow==this) getApp()->keyWindow=NULL;
  if(getApp()->dragWindow==this) getApp()->dragWindow=NULL;
  if(getApp()->selectionWindow==this) getApp()->selectionWindow=NULL;
  if(getApp()->clipboardWindow==this) getApp()->clipboardWindow=NULL;
  if(getApp()->refresher==this) getApp()->refresher=parent;
  if(parent) parent->recalc();
  destroy(); 
  parent=(FXWindow*)-1;
  owner=(FXWindow*)-1;
  shell=(FXWindow*)-1;
  first=last=(FXWindow*)-1;
  next=prev=(FXWindow*)-1;
  focus=(FXWindow*)-1;
  defaultCursor=(FXCursor*)-1;
  dragCursor=(FXCursor*)-1;
  accelTable=(FXAccelTable*)-1;
  target=(FXObject*)-1;
  userData=(void*)-1;
  }

