/*
 * subdemo.c - written in 1998 by Andreas Beck   becka@ggi-project.org
 *                        1997 by Jason McMullan jmcc@ggi-project.org
 *
 * This is a demonstration of LibGGI's functions and can be used as a
 * reference programming example.
 *
 *   This software is placed in the public domain and can be used freely
 *   for any purpose. It comes without any kind of warranty, either
 *   expressed or implied, including, but not limited to the implied
 *   warranties of merchantability or fitness for a particular purpose.
 *   Use it at your own risk. the author is not responsible for any damage
 *   or consequences raised by use or inability to use this program.
 */

/* Include the necessary headers used for e.g. error-reporting.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Include the LibGGI declarations.
 */
#include <ggi/ggi.h>

/* We are (not:-) running on a single primary visual which is made accessible
 * globally so subroutines like "waitabit" can access it.
 * See its allocation in main() for details.
 */
ggi_visual_t root,vis,sub2,sub3;

#include <dlfcn.h>
#define __C_ASM_H /* Workaround for retarded Digital Unix headers */
#include <pthread.h>
       
/* In case we were called with wrong parameters, give an explanation.
 */
void usage(const char *prog)
{
	fprintf(stderr,"Usage:\n\n"
		       "%s <bpp> <xsize> <ysize> [<virtx> <virty>]\n\n"
		       "Default: %s 8 320 200 320 200\n",prog,prog);
	exit(1);
}

/* Wait for a keypress. Shut down everything, if "q" is pressed.
 */
void waitabit(void)
{
	int key;

	/* Make sure the display is in sync with the drawing command queue. 
	 */
	ggiFlush(root);

#if 0	/* activate this to test waiting for max times. 
	 * It will cause the wait to not block indefinitely but only for
	 * 5 seconds. However it will loose the 'q' functionality.
         */
	{
		struct timeval tv={5,0};
		key = ggiEventPoll(root, emKeyPress,&tv);
		if (key&emKeyPress) key = ggiGetc(root);
	}

#else
	/* ggiGetc will blocking-wait for a keypress. 
	 */
	key = ggiGetc(root);

	if (key == 'q') /* Q pressed */
	{	ggiExit();
		exit(0);
	}
	
#endif
}

/* Pixel value for black and white. See main() on how to get at it.
 */
static ggi_uint black;
static ggi_uint white;

/* Print the name of the current test in the top left corner.
 */
void TestName(const char *name)
{
	ggiSetGCForeground(vis,white);
	ggiPuts(vis,0,0,name);
}

/* The main routine.
 * It will set up a graphics mode as requested on the commandline and 
 * do an extensive test using about all graphics primitives LibGGI
 * knows.
 */
int main(int argc,char **argv)
{
	/* First we define a bunch of variables we will access throughout the
	 * main() function. Most of them are pretty meaningless loop-counters
	 * and helper-variables.
	 */

	const char *prog=argv[0];	/* Make an alias for the program name */

	ggi_graphtype type;		/* This is an enum holding the requested
					 * type of graphics mode. See the mode setting
					 * code below for details.
					 */

	int depth;			/* The depth is used for chosing a graphics type.
					 * it is mapped to the right GT_* by a switch-statement.
					 */

	int sx,sy,vx,vy;		/* These holde the visible screen size (sx,sy) and
					 * the virtual size (vx,vy). On most targets you can 
					 * request a certain larger area on which the visible
					 * area can be placed.
					 */

	int c,r,x,y,dx,dy,w,h,i,err;	/* Pretty meaningless helpers used everywhere. */

	ggi_color map[256];		/* Yet another array of colors used for the color-
					 * mapping test
					 */

	unsigned int pack[256];		/* This will hold the mapped colors. */

	int textmode=0;			/* Flag to indicate we want a textmode. */

	/* This is a struct containing visual and virtual screen size as
	 * well as the bpp/textmode information 
	 */

	ggi_mode mode = { /* This will cause the default mode to be set */
		1,                      /* 1 frame [???] */
		{GGI_AUTO,GGI_AUTO},    /* Default size */
		{GGI_AUTO,GGI_AUTO},    /* Virtual */
		{0,0},                  /* size in mm don't care */
		GT_AUTO,               /* Mode */
		{GGI_AUTO,GGI_AUTO}     /* Font size */
	};

	/* Now for the real fun. We will start two bouncing boxes inside two subvisuals later.
	 * The program doing the rendering is external. This is the dlopen handle to it.
	 */
	void *spinner_dlhand;

	/* And that's the function we import from there.
	 */
	void *(*modulemain)(void *vis);

	/* Now for the two threads we will spawn:
	 */
	pthread_t newthread1,newthread2;


	/* Set up the random number generator. */
	srandom(time(NULL));

	/* Open the module silently fail if missing.
	 */
	modulemain=NULL;
	spinner_dlhand=dlopen("./spinmodule.so",RTLD_NOW);
	if (spinner_dlhand)
		modulemain=dlsym(spinner_dlhand,"module_main");
	else
		modulemain=NULL;

	/* Initialize the GGI library. This must be called before any other 
	 * GGI function. Otherwise behaviour is undefined.
	 */
	if (ggiInit() != 0) {
		fprintf(stderr, "%s: unable to initialize libggi, exiting.\n",
			argv[0]);
		exit(1);
	}

	/* Open the default visual. This is automatically selected depending
	 * on the calling environment (Linux console, X, ...). You can 
	 * explicitly set this in the LIBGGI_DISPLAY environment variable.
	 */
	if ((vis=ggiOpen(NULL)) == NULL) {
		fprintf(stderr,
			"%s: unable to open default visual, exiting.\n",
			argv[0]);
		exit(1);
	}
	ggiSetFlags(vis,GGIFLAG_ASYNC);

	if (argc > 1) { /* A mode-string was given - so we parse it */
		if (ggiParseMode(argv[1], &mode) != 0) {
			/* Error parsing mode */
			fprintf(stderr,
				"Error parsing mode: %s\n"
				"Valid modes are:\n"
"  X x Y # XVIRT x YVIRT D dpp.x x dpp.y F frames (T) [bpp] + XOFF + YOFF\n"
"  X x Y # XVIRT x YVIRT D dpp.x x dpp.y F frames [graphtype] + XOFF + YOFF\n\n"
"  Everything (!) can be omitted, all ommitted values default to GGI_AUTO,\n"
"  with the exception of a textmode with unspecified size which defaults\n"
"  to TEXT16.\n  Whitespace is ignored.\n"
				, *argv);
			exit(1);
		}
		/* mode now holds the requested visible 
		 * and virtual size of the screen.
		 */
	}

	/* that's what we try. See what we get ... */
	fprintf(stderr,"Trying mode ");
	ggiFPrintMode(stderr,&mode);
	fprintf(stderr,"\n");

	/* Is the mode possible ? If not, a better one will be
	 * suggested. 
	 */

	ggiCheckMode(vis,&mode);

	fprintf(stderr,"Suggested mode ");
	ggiFPrintMode(stderr,&mode);
	fprintf(stderr,"\n");

	err=ggiSetMode(vis,&mode);   /* now try it. it *should* work! */
	
	/* Check if there were errors. Almost all LibGGI functions
	 * return 0 on success and an error code otherwise. No messing
	 * with errno, as this is not thread- friendly. The functions
	 * that are supposed to return a pointer normally return NULL on
	 * error.
	 */

	if (err) {
		fprintf(stderr,"Can't set mode\n");
		ggiClose(vis);
		ggiExit();
		return 2;
	}


	/* Now we read back the set mode, as it might have been
	 * autoselected or changed.
	 */

	type=mode.graphtype;
	vx=mode.virt.x;    vy=mode.virt.y;
	sx=mode.visible.x; sy=mode.visible.y;
	depth=GT_DEPTH(mode.graphtype);

	/* Set a colorful palette for the tests.
	   Please note that GGI always uses 16 bit color components,
	   so stretch the values accordingly when porting from DOS 
	   or other libs that make assumptions about palette size.

	   On some fixed-palette modes the ggiSetColorfulPalette()
	   call will fail.  We silently ignore that.
	 */
	 
	if (GT_SCHEME(mode.graphtype) == GT_PALETTE) {
		ggiSetColorfulPalette(vis);
	}
	
	/* Find the color "white".
	 * LibGGI makes no assumptions on how a color value (which is a ggi_pixel) is 
	 * actually mapped to a color on the screen. If you want to write strictly
	 * compatible programs, you need to ask LibGGI for the color value associated
	 * with a given color. We will do this now:
	 */
	map[0].r = 0xFFFF;
	map[0].g = 0xFFFF;
	map[0].b = 0xFFFF;
	white = ggiMapColor(vis, &map[0]);
	printf("white=%d\n",white);

	map[0].r = map[0].g = map[0].b = 0x0;
	black = ggiMapColor(vis, &map[0]);
	fprintf(stderr,"black=%d\n", black);

	/* Clear screen. */
	ggiSetGCForeground(vis, black);
	ggiFillscreen(vis);

	/* Set the drawing color to white and display some text at 0/0. 
	 */
	ggiSetGCForeground(vis,white);
	ggiPuts(vis,0,0,"Press any key to begin tests...");
	ggiFlush(vis);

	{
	  root=vis; /* Save root visual */
	  sx=vx=400;
	  sy=vy=300;
	  if ((vis=ggiOpen("display-sub",root)) == NULL
		  || ggiSetGraphMode(vis,10,175,sx,sy,GT_8BIT)
		  || (sub2=ggiOpen("display-sub",root)) == NULL
		  || ggiSetGraphMode(sub2,200,10,sx,sy,GT_8BIT)
		  || (sub3=ggiOpen("display-sub",sub2)) == NULL
		  || ggiSetGraphMode(sub3,50,50,200,150,GT_8BIT))
		  ggiPanic("Unable to open sub-visuals\n");
	}

	ggiSetGCForeground(vis,white);
	ggiSetGCForeground(sub2,white);
	ggiSetGCForeground(sub3,white);
	ggiSetGCClipping(vis,0,0,sx,sy);
#if 1	/* Activate this for clipping tests ... */
	ggiDrawHLine(vis,0,39,sx);
	ggiDrawHLine(vis,0,sy-40,sx);
	ggiDrawVLine(vis,39,0,sy);
	ggiDrawVLine(vis,sx-40,0,sy);
	ggiSetGCClipping(vis,40,40,sx-40,sy-40);
	ggiDrawHLine(sub2,0,39,sx);
	ggiDrawHLine(sub2,0,sy-40,sx);
	ggiDrawVLine(sub2,39,0,sy);
	ggiDrawVLine(sub2,sx-40,0,sy);
	ggiSetGCClipping(sub2,45,45,sx-45,sy-45);
	ggiDrawHLine(sub3,0,39,sx);
	ggiDrawHLine(sub3,0,sy-40,sx);
	ggiDrawVLine(sub3,39,0,sy);
	ggiDrawVLine(sub3,sx-40,0,sy);
#endif

	/* Wait for any keypress. 
	 * This is a blocking call which returns a 16 bit wide unicode character.
	 * See the comments at waitabit() for details. Do not just do things like
	 * if (ggiGetc=='y') yes();.
	 */
	ggiGetc(root);
	ggiFlush(root);

	/* Now for the first test. Put some text at random places. 
	 * ggiPuts and friends are very simple and always use a small pixmap font. 
	 * Scalable and outline fonts are found in libggi-2D.
	 */
	for (i=0;i<1024;i++) {

		/* Set the foreground color to some arbitrary values. 
		 * Please note that there is no guarantee what colors will appear.
		 * GT_8BIT modes will normally give repeatedly the 256 colors in the
		 * Palette, but there is no guarantee anything will happen.
		 * If we were on a "sparse" 16 bit mode which only uses the upper
		 * 16 bits from a 32 bit value, nothing visible would be drawn.
		 */

		ggiSetGCForeground(vis,i);
		ggiSetGCForeground(sub2,i);

		x=random()%vx;	/* Get some random starting point on the virtual screen. */
		y=random()%vy;
		ggiPuts(vis,x,y,"ggiPuts test!");
		ggiPuts(sub2,x,y,"other sub test!");

		ggiFlushRegion(vis,x,y,8+13,8);	
			/* Calling ggiFlush() makes sure the drawing on the screen
			 * is synced with what the programs thinks it should look like.
			 * This is not strictly necessary here, as LibGGI normally
			 * comes up in SYNC_MODE which does this automatically if
			 * necessary. But it is a good idea to get used to ASYNC mode,
			 * as it brings big performance advantages on some targets,
			 * as e.g. X.
			 */
		ggiFlushRegion(sub2,x,y,8*15,8);

		/* If the user hits a key abort this demo. The key will be available
		 * for fetching with ggiGetc() what will happen in waitabit().
		 */
		if (ggiKbhit(root)) break;
	}

waitabit();

	/* Now we start up the threads ...
	 */

	if (modulemain) {
		pthread_create(&newthread1, NULL, modulemain, sub2);
		pthread_create(&newthread2, NULL, modulemain, sub3);
	}

	
	/* Check the colormapping code. Draw four horizontal bars in red/green
	 * blue and grey.
	 */
	TestName("MapColor");

	for (i=0;i<sx;i++) {

		ggi_color col;

		/* make x so it varies from 0-0xffff over the visible length
		 */
		x= 0xffff*i/(sx-1);
		
		/* Now first make a color descriptor for the red bar. It will
		 * holf the R/G/B triplet x/0/0 describing the shades of red.
		 */
		col.r=x;col.g=0;col.b=0;

		/* ggiMapColor will return the color value that is closest
		 * to the requested color. We set this a ctive drawing color.
		 */
		ggiSetGCForeground(vis,ggiMapColor(vis, &col) );
		ggiSetGCForeground(sub3,ggiMapColor(vis, &col) );
		
		/* Then we draw a small vertical line at x=i, y=20 and height=20.
		 */
		ggiDrawVLine(vis,i,20,20);
		ggiDrawVLine(sub3,i,20,20);
		
		/* Now we do the same with green. 
		 */
		col.r=0;col.g=x;col.b=0;
		ggiSetGCForeground(vis,ggiMapColor(vis, &col) );
		ggiDrawVLine(vis,i,40,20);	/* x=i, y=40, height=20 */
		ggiSetGCForeground(sub3,ggiMapColor(vis, &col) );
		ggiDrawVLine(sub3,i,40,20);	/* x=i, y=40, height=20 */

		/* blue */
		col.r=0;col.g=0;col.b=x;
		ggiSetGCForeground(vis,ggiMapColor(vis, &col) );
		ggiDrawVLine(vis,i,60,20);
		ggiSetGCForeground(sub3,ggiMapColor(vis, &col) );
		ggiDrawVLine(sub3,i,60,20);

		/* grey */
		col.r=x;col.g=x;col.b=x;
		ggiSetGCForeground(vis,ggiMapColor(vis, &col) );
		ggiDrawVLine(vis,i,80,20);
		ggiSetGCForeground(sub3,ggiMapColor(vis, &col) );
		ggiDrawVLine(sub3,i,80,20);
	}
	ggiFlush(vis);ggiFlush(sub3);

waitabit();


	/* Now the memvisual and crossblitting code.
	 * This is an advanced topic which you should ignore if you are a beginner.
	 */
	{ 	ggi_visual_t memvis;
		TestName("MemoryVisual");

		/* Open a "memory-visual" which is simply a simulated display
		 * in memory. You can draw on it as you would on a screen.
		 * Nifty for preparing things in the background.
		 */
		memvis=ggiOpen("display-memory",NULL);

		/* Set it to a small 32 bit mode. 
		 */
		if (textmode) 
			err=ggiSetTextMode(memvis,sx,sy,vx,vy,GGI_AUTO,GGI_AUTO,type);
		else
			err=ggiSetGraphMode(memvis,160,20,160,20,GT_32BIT);
			
		/* Check for errors
		 */
		if (err) fprintf(stderr,"memvisual error ...\n");

		/* Now draw some string in the truecolor visual and blit it across. 
		 */
		for (i=0;i<3*256;i++) {

			/* Note that this normally does not guarantee any given color. 
			 * Use ggiMapColor, if you want that.
			 * However as the author knows the organization of the memvisual,
			 * so he's sure this always gives a blue->green->red sequence.
			 */
			ggiSetGCForeground(memvis,x=( (i&255) << ( (i&0x300) >> 5) ));
			x=random()%(sx-80);
			y=random()%(sy-8);
			ggiPuts(memvis,0,0,"ggiXBLit !");

			/* Note : This automatically converts colorspace. 
			 * Take care, this might be expensive ...
			 */
			ggiCrossBlit(memvis,0,0,80,8,
			             vis   ,x,y);
			if (ggiKbhit(vis)) break;
		}
		ggiFlush(vis);

		ggiClose(memvis);
	}

	waitabit();

	/* Set a colorful palette for the other tests.
	 * Please note that GGI always uses 16 bit palettes, so
	 * stretch the values accordingly when porting from DOS 
	 * or other libs that make assumptions about palette size.
	 *
	 * On paletteless modes the ggiSetPalette() call will fail.
	 * We silently ignore that.
	 */
        ggiSetColorfulPalette(vis);

	/* Blit rectangular solid boxes in random places.
	 */
	TestName("DrawBox");
	x=random()%(vx-32);
	y=random()%(vy-32);
	dx=1;
	dy=1;
	for (i=0;i<64*1024;i++) {
		ggiSetGCForeground(vis,i);
		ggiDrawBox(vis,x,y,32,32);
		x+=dx;
		y+=dy;
		if (x<0) {x=0;dx=-dx;}
		if (y<0) {y=0;dy=-dy;}
		if (x>=(vx-32)) {x=vx-33;dx=-dx;}
		if (y>=(vy-32)) {y=vy-33;dy=-dy;}
		if (ggiKbhit(vis)) break;
	}
	ggiFlush(vis);

waitabit();

	/* Hline tests. First get a clean screen back. */
	ggiSetGCForeground(vis, black);
	ggiFillscreen(vis);

	/* Draw horizontal lines.
	 */
	TestName("HLine");
	for (c=0;c<256;c++) {
		for (i=0;i<sy;i++) {
			ggiSetGCForeground(vis,c+i);

			/* Draw Horizontal lines starting at x=y=i of length vx-256.
			 */
			ggiDrawHLine(vis,i,i,vx-256);
		}
		ggiFlush(vis);
		if (ggiKbhit(vis)) break;
	}
	ggiFlush(vis);

waitabit();

	/* VLine Tests. First clear screen. */
	ggiSetGCForeground(vis, black);
	ggiFillscreen(vis);

	/* Draw vertical lines.
	 */
	TestName("VLine");
	for (c=0;c<256;c++) {
		for (i=0;i<256;i++) {
			ggiSetGCForeground(vis,c+i);

			/* Draw vertical lines at x=vx-i,y=i,height=20.
			 */
			ggiDrawVLine(vis,vx-i,i,20);
		}
		ggiFlush(vis);
		if (ggiKbhit(vis)) break;
	}
	ggiFlush(vis);

waitabit();

	/* Put horizontal lines. "Putting" means blitting the contents
	 * of a memory buffer.
	 */
	TestName("Hline Put");
	{ char buffer[64*1024];	/* Hope this is enough ... */
	  for(x=0;x<sizeof(buffer);x++) buffer[x]=x&0xff;
	  for(y=10;y<vy;y++)
	   ggiPutHLine(vis,0,y,vx,buffer);
	}
	ggiFlush(vis);

waitabit();

	/* The same with boxes.
	 */
	TestName("PutBox");
	{ char buffer[64*1024];	/* Hope this is enough ... */
	  for(x=0;x<sizeof(buffer);x++) buffer[x]=x&0xff;
	  for(y=10;y<vy;y+=10)
	   ggiPutBox(vis,y,y,20,20,buffer);
	}
	ggiFlush(vis);

waitabit();

	/* check if "getting" a box works.
	 */
	TestName("GetBox");
	{ char buffer[64*1024];	/* Hope this is enough ... */
	  ggiGetBox(vis,0,0,40,40,buffer);
	  for(y=20;y<vy;y+=20)
	   ggiPutBox(vis,y,y,40,40,buffer);
	}
	ggiFlush(vis);

waitabit();

	/* Linedrawing tests */
	ggiSetGCForeground(vis, black);
	ggiFillscreen(vis);

	TestName("Line");
	
	for (i=0;i<vy;i++) {
		ggiSetGCForeground(vis,i);
		ggiDrawLine(vis,0,i,vx-1,i);
		if (ggiKbhit(vis)) break;
	}
	ggiFlush(vis);
	waitabit();

	for (i=0;i<vx;i++) {
		ggiSetGCForeground(vis,i);
		ggiDrawLine(vis,i,0,i,vy-1);
		if (ggiKbhit(vis)) break;
	}
	ggiFlush(vis);
	waitabit();
	
	x=random()%vx;
	y=random()%vy;
	w=random()%vx;	/* Not really width & height, actually x2,y2 */
	h=random()%vy;
	dx=-2;
	dy=2;
	for (i=0;i<8*1024;i++) {
		ggiSetGCForeground(vis,i);
		ggiDrawLine(vis,x,y,w,h);
		ggiSetGCForeground(sub3,i);
		ggiDrawLine(sub3,x,y,w,h);
		x+=dx;
		y+=dy;
		w-=dx;
		h-=dy;
		if (x>vx) {x=vx-1; dx=-dx;}
		if (y>vy) {y=vy-1; dy=-dy;}
		if (w>vx) {w=vx-1; dx=-dx;}
		if (h>vy) {h=vy-1; dy=-dy;}
		if (x<0) {x=0; dx=-dx;}
		if (y<0) {y=0; dy=-dy;}
		if (w<0) {w=0; dx=-dx;}
		if (h<0) {h=0; dy=-dy;}
		if (ggiKbhit(vis)) break;
	}
		
	/* We will now cancel one thread, to see if that works.
	 */
	if (modulemain) {
		pthread_cancel(newthread2);
	}

	ggiFlush(vis);
waitabit();


	/* The usual "draw random sized box" test */
	ggiSetGCForeground(vis, black);
	ggiFillscreen(vis);
	TestName("Box");

	for (i=0;i<8*1024;i++) {
		ggiSetGCForeground(vis,i);
		x=random()%vx;
		y=random()%vy;
		w=random()%(vx-x)+1;
		h=random()%(vy-y)+1;
		ggiDrawBox(vis,x,y,w,h);
		if (ggiKbhit(vis)) break;
	}

	ggiFlush(vis);
waitabit();

#define MAGIC_6 1040

	/* Check color packing i.e. multi-"color-to-index" conversion 
	 */
	ggiSetGCForeground(vis, black);
	ggiFillscreen(vis);
	TestName("ColorPack");

	/* Make up a grayscale map.
	 */
	for (i=0;i<64;i++) {
	   map[i].r=(i*MAGIC_6);
	   map[i].g=(i*MAGIC_6);
	   map[i].b=(i*MAGIC_6);
	}

	/* Convert it to a list of color values.
	 */
	ggiPackColors(vis,pack,map,64);

	for (i=0;i<64;i++) {	
		ggiSetGCForeground(vis,pack[i]);
		ggiDrawBox(vis,100-i,100-i,100+i,100+i);
		if (ggiKbhit(vis)) break;
	}

	ggiFlush(vis);
waitabit();

	/* Circles */
	TestName("Circle");
	for (i=0;i<8*1024;i++) {
		r=random()%100;
		x=random()%(vx-r);
		y=random()%(vy-r);
		ggiSetGCForeground(vis,i);
		ggiDrawBox(vis,x-r,y-r,x+r,y+r);
		if (ggiKbhit(vis)) break;
	}

	/* Asynchronous mode. From now on LibGGI is no longer responsible
	 * for keeping display and command in sync. You need to explicitly
	 * call ggiFlush() to make sure the screen gets updated.
	 *
	 * All applications are recommended to use async mode as much as
     * possible, as it's much faster on most targets.
     * There are very few apps that really need sync mode anyway.
     *
     * Just remember only to call ggiFlush() when it's neccesary,
     * which in most cases equals once every frame.
	 */
	ggiSetFlags(vis,GGIFLAG_ASYNC);
	ggiFlush(vis);
waitabit();

	TestName("Async mode");
	for (i=0;i<8*1024;i++) {
		r=random()%100;
		x=random()%(vx-r);
		y=random()%(vy-r);
		ggiSetGCForeground(vis,i);
		ggiDrawBox(vis,x-r,y-r,x+r,y+r);

		/* This will only _force_ an update every 1024 circles.
		 */
		if ((i&0x3ff)==0x3ff) ggiFlush(vis);
		if (ggiKbhit(vis)) break;
	}

	/* Back to sync mode. One should ggiFlush() here. This
	 * happens in waitabit(), so no explicit call here.
	 */
	ggiRemoveFlags(vis,GGIFLAG_ASYNC);

waitabit();

	TestName("Palette ");

	/* Palette Test. We do some colorcycling here.
	 */
	if (depth <= 8) {
		int size=(1<<depth);
		ggi_color cols[257];
		ggiGetPalette(vis,0,size,cols);
		for (i=0;i<10000;i++) {
			cols[size]=cols[0];
			memcpy(cols,cols+1,size*sizeof(ggi_color));
			ggiSetPalette(vis,0,size,cols);
			ggiFlush(vis);
			if (ggiKbhit(vis)) break;
		}
	}
	ggiFlush(vis);

waitabit();
	
	/* Cancel second thread. We want to give up the dl.
	 */
	if (modulemain) {
		pthread_cancel(newthread1);
		pthread_join(newthread1,NULL);
		pthread_join(newthread2,NULL);
	}

	if (spinner_dlhand)
		dlclose(spinner_dlhand);

	/* O.K. - all done. we will now cleanly shut down LibGGI.
	 */

	/* First close down the visual we requested. Any further calls
	 * using vis have undefined behaviour. LibGGI is still up and 
	 * you could open another visual
	 */
	ggiClose(sub3);
	ggiClose(sub2);
	ggiClose(vis);
	ggiClose(root);

	/* Now close down LibGGI. Every LibGGI call except ggiInit has 
	 * undefined behaviour now. It is not recommended to needlessly
	 * deinit-reinit LibGGI, but it should work.
	 */
	ggiExit();	

	/* Terminate the program.
	 */
	return 0;
}
