/* File "malaga.c":
 * The interactive morphology/syntax analysis program. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* includes =================================================================*/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <setjmp.h>
#include <time.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "files.h"
#include "display_process.h"
#include "instr_type.h"
#include "rule_type.h"
#include "rules.h"
#include "tries.h"
#include "analysis.h"
#include "input.h"
#include "commands.h"
#include "options.h"
#include "malaga_lib.h"
#include "generation.h"
#include "debugger.h"
#include "breakpoints.h"
#include "cache.h"

#ifdef HANGUL
#include "hangul.h"
#endif

#undef GLOBAL
#define GLOBAL

/* macros ===================================================================*/

#define SAFE_STRING(s) ((s) != NULL ? (s) : "")
/* return an empty string if <s> == NULL */

/* variables ================================================================*/

LOCAL string_t program_name; /* the name the program is called by */

/* location where file analysis currently stands at */
LOCAL long_t input_line_number = -1; /* line number of input */
LOCAL string_t input_file_name = NULL; /* file name of input */

LOCAL FILE *error_stream;

LOCAL long_t debug_analysis_node_index; /* index of state to debug or -1 */

/* error handling ===========================================================*/

GLOBAL void error (string_t format, ...)
/* Print an error message and exit or return to command loop. */
{
  long_t line;
  string_t file_name, rule_name;
  va_list arg;

  fflush (stdout);

  if (error_jump_point != NULL)
    fprintf (error_stream, "error: ");
  else
    fprintf (error_stream, "%s: ", program_name);

  if (executing_rule) 
  {
    /* Get source file position.
     * Subtract 1 from <pc> because when an instruction is executed,
     * <pc> usually already points to next instruction */
    source_of_instr (executed_rule_sys, pc - 1, &line, NULL, &file_name,
		     &rule_name);
    
    if (line != -1)
      fprintf (error_stream, "file \"%s\", line %ld, rule \"%s\": ", 
	       name_in_path (file_name), line, rule_name);
    
    /* Do not reset rule state, so we can do post-mortem debugging. */
    executing_rule = FALSE;
  }
  else 
    line = -1;
  
  if (input_line_number != -1)
    fprintf (error_stream, "(file \"%s\", line %ld): ", 
	     name_in_path (input_file_name), input_line_number);

  va_start (arg, format);
  vfprintf (error_stream, format, arg);
  va_end (arg);

  fprintf (error_stream, "\n");

  if (getenv ("MALAGA_MODE") != NULL && line != -1 &&
      (error_stream == stdout || error_stream == stderr))
    printf ("SHOW \"%s\":%ld:0\n", file_name, line);

  input_line_number = -1;
  input_file_name = NULL;

  if (error_jump_point != NULL)
    longjmp (*error_jump_point, 1);
  else
    exit (1);
}

/* analysis result output ===================================================*/

LOCAL void print_output (long_t input_line_number, FILE *output)
/* Print the result of the last analysis on <output>. */
{
  string_t input_line = new_string_readable (analysis_input, NULL);
  
  if (! reset_analysis_results ())
  {
    if (*unknown_format != EOS)
    {
      string_t s;
      
      for (s = unknown_format; *s != EOS; s++)
      {
	if (*s == '%')
	{
	  s++;
	  switch (*s)
	  {
	  case 's':
	    fprintf (output, "%s", DECODED_STRING (input_line));
	    break;
	  case 'n':
	    fprintf (output, "%ld", state_count);
	    break;
	  case 'l':
	    fprintf (output, "%ld", input_line_number);
	    break;
	  default:
	    fprintf (output, "%%%c", *s);
	  }
	}
	else
	  fprintf (output, "%c", *s);
      }
      
      fprintf (output, "\n");
    }
  }
  else
  {
    long_t result_count;
    value_t cat;

    result_count = 0;
    for (cat = get_next_analysis_result ();	
	 cat != NULL; 
	 cat = get_next_analysis_result ())
    {
      string_t s;
      
      result_count++;
      if (*output_format != EOS)
      {
	for (s = output_format; *s != EOS; s++)
	{
	  if (*s == '%')
	  {
	    s++;
	    switch (*s)
	    {
	    case 's':
	      fprintf (output, "%s", DECODED_STRING (input_line));
	      break;
	    case 'n':
	      fprintf (output, "%ld", state_count);
	      break;
	    case 'l':
	      fprintf (output, "%ld", input_line_number);
	      break;
	    case 'r':
	      fprintf (output, "%ld", result_count);
	      break;
	    case 'c':
	      fprint_value (output, cat);
	      break;
	    default:
	      fprintf (output, "%%%c", *s);
	    }
	  }
	  else
	    fprintf (output, "%c", *s);
	}
	fprintf (output, "\n");
      }
    }
  }
  free (input_line);
  
  if (ferror (output))
    error ("can't write result: %s", strerror (errno));
}

/*---------------------------------------------------------------------------*/

LOCAL void display_result (void)
/* Generate result file and start TCL program to display result. */
{
  string_t input;
  value_t cat;
  long_t result_number;

  if (! reset_analysis_nodes ())
    error ("no analysis started");

  start_display_process ();

  fprintf (display_stream, "result\n");

  input = new_string_readable (analysis_input, NULL);
  fprintf (display_stream, "\"%s\"\n", DECODED_STRING (input));
  free (input);

  result_number = 0;
  reset_analysis_results ();
  for (cat = get_next_analysis_result (); 
       cat != NULL; 
       cat = get_next_analysis_result ())
  {
    result_number++;
    
    fprintf (display_stream, "%ld {", result_number);
    fprint_value (display_stream, cat);
    fprintf (display_stream, "}\n");
  }

  fprintf (display_stream, "end\n");
  fflush (display_stream);
}

/*---------------------------------------------------------------------------*/

LOCAL void display_tree (void)
/* Display analysis tree. */
{
  string_t input;
  analysis_node_t *node;

  if (! reset_analysis_nodes())
    error ("no analysis started");
  
  start_display_process ();
  
  fprintf (display_stream, "tree\n");

  /* Print sentence that has been analysed. */
  input = new_string_readable (analysis_input, NULL);
  fprintf (display_stream, "\"%s\"\n", DECODED_STRING (input));
  free (input);
  
  for (node = get_next_analysis_node (); 
       node != NULL; 
       node = get_next_analysis_node ())
  {
    string_t node_type;
    
    /* Determine node type */
    switch (node->type)
    {
    case INTER_NODE: node_type = "inter"; break;
    case BREAK_NODE: node_type = "break"; break;
    case FINAL_NODE: node_type = "final"; break;
    case PRUNED_NODE: node_type = "pruned"; break;
    default: node_type = "error"; break;
    }

    fprintf (display_stream, "%ld \"%s\" %ld \"%s\" ", 
	     node->index, node_type, node->mother_index, 
	     SAFE_STRING (node->rule_name));
  
    /* Print right surface and category. */
    fprintf (display_stream, "\"%s\" ", 
	     DECODED_STRING (SAFE_STRING (node->right_surf)));
    fprintf (display_stream, "{");
    fprint_value (display_stream, node->right_cat);
    fprintf (display_stream, "} ");
  
    /* Print result surface and category. */
    fprintf (display_stream, "\"%s\" ", 
	     DECODED_STRING (SAFE_STRING (node->result_surf)));
    fprintf (display_stream, "{");
    fprint_value (display_stream, node->result_cat);
    fprintf (display_stream, "} ");
    
    fprintf (display_stream, "\"%s\"", SAFE_STRING (node->rule_set));
    fprintf (display_stream, "\n");
  }
  
  fprintf (display_stream, "end\n");
  fflush (display_stream);
}

/*---------------------------------------------------------------------------*/

LOCAL void display_analysis_result (void)
/* Display result in the modes that have been switched on after analysis. */
{
  if (show_output) 
    print_output (0, stdout);

  if (show_result)
    display_result ();
 
  if (show_tree)
    display_tree ();
}

/*---------------------------------------------------------------------------*/

LOCAL void do_output (string_t arguments)
/* Print output of last analysis. */
{
  parse_end (arguments);

  if (! reset_analysis_nodes ())
    error ("no previous analysis");

  print_output (0, stdout);
}

LOCAL command_t output_command = 
{
  "output o", do_output,
  "Show output of last analysis.\n"
  "Arguments: (none)\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_result (string_t arguments)
/* Show result of last analysis. */
{
  parse_end (arguments);

  if (! reset_analysis_nodes ())
    error ("no previous analysis");

  display_result ();
}

LOCAL command_t result_command = 
{
  "result res", do_result,
  "Show result of last analysis.\n"
  "Arguments: (none)\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_tree (string_t arguments)
/* Generate analysis tree file and start TCL program to display tree. */
{
  parse_end (arguments);

  if (! reset_analysis_nodes ())
    error ("no analysis started");

  display_tree ();
}

LOCAL command_t tree_command =
{
  "tree t", do_tree,
  "Display the analysis tree.\n"
  "Arguments: (none)\n"
  "In debug mode or after a rule execution error, the tree may be"
  " incomplete.\n"
};

/* analysis functions =======================================================*/

LOCAL void analyse_argument (grammar_t grammar, string_t arguments)
/* Analyse <arguments> (or last analysis, if *<arguments> == EOS)
 * using <grammar> (SYNTAX or MORPHOLOGY). */
{
  if (in_debugger)
    error ("in debug mode");

  if (*arguments == EOS)
  {
    if (! reset_analysis_nodes ())
      error ("no previous analysis");
  }
  else
  {
    copy_string (analysis_input, ENCODED_STRING (arguments), 
		 analysis_input + ANALYSIS_INPUT_LENGTH);
    preprocess_input (analysis_input);
  }

  debug_state = NULL;
  analyse (grammar, analysis_input, TRUE, TRUE);
}

/*---------------------------------------------------------------------------*/

LOCAL void do_ma (string_t arguments)
/* Analyse <arguments> morphologically. */
{
  set_debug_mode (RUN_MODE, NULL);
  analyse_argument (MORPHOLOGY, arguments);
  display_analysis_result ();
}

LOCAL command_t ma_command =
{
  "ma", do_ma,
  "Analyse morphologically.\n"
  "Arguments:\n"
  "  <input> -- analyse <input>\n"
  "  (none) -- re-analyse last input\n"
  "\"ma\" can't be used in debug mode.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_sa (string_t arguments)
/* Analyse <arguments> syntactically. */
{
  set_debug_mode (RUN_MODE, NULL);
  analyse_argument (SYNTAX, arguments);
  display_analysis_result ();
}

LOCAL command_t sa_command =
{
  "sa", do_sa,
  "Analyse the argument syntactically.\n"
  "Arguments:\n"
  "  <input> -- analyse <input>\n"
  "  (none) -- re-analyse last input\n"
  "\"sa\" can't be used in debug mode.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void analysis_mode (grammar_t grammar)
/* Read input lines and analyse them using <grammar>. */
{
  if (in_debugger)
    error ("in debug mode");

  set_debug_mode (RUN_MODE, NULL);
  while (TRUE)
  {
    char input_line[ANALYSIS_INPUT_LENGTH];
    
    printf ("%s> ", grammar == MORPHOLOGY ? "ma" : "sa");
    read_line (stdin, input_line, ANALYSIS_INPUT_LENGTH);
    if (input_line[0] == EOS)
      break;
    
    analyse_argument (grammar, input_line);
    display_analysis_result ();
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void do_ma_mode (string_t arguments)
{
  parse_end (arguments);
  analysis_mode (MORPHOLOGY);
}

LOCAL command_t ma_mode_command =
{
  "ma-mode mam", do_ma_mode,
  "Enter ma mode: Read input lines and analyse them morphologically.\n"
  "Enter a blank line to leave ma mode.\n"
  "Arguments: (none)\n"
  "\"ma-mode\" can't be used in debug mode.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_sa_mode (string_t arguments)
{
  parse_end (arguments);
  analysis_mode (SYNTAX);
}

LOCAL command_t sa_mode_command =
{
  "sa-mode sam", do_sa_mode,
  "Enter sa mode: Read input lines and analyse them syntactically.\n"
  "Enter a blank line to leave sa mode.\n"
  "Arguments: (none)\n"
  "\"sa-mode\" can't be used in debug mode.\n"
};

/* debug support ============================================================*/

LOCAL void print_rule (void)
/* Print rule name, left and right surface. */
{
  rule_t *rule = executed_rule_sys->rules + executed_rule_number;
  
  printf ("at rule \"%s\", ", executed_rule_sys->strings + rule->name);

  if (in_analysis)
  {
    string_t surf, left_end;
    
    if (right_start > left_start && right_start[-1] == ' ')
      left_end = right_start - 1;
    else
      left_end = right_start;
    
    surf = new_string_readable (left_start, left_end);
    printf ("start: \"%s\"", DECODED_STRING (surf));
    free (surf);
    
    if (right_end != right_start)
    {
      surf = new_string_readable (right_start, right_end);
      printf (", next: \"%s\"", DECODED_STRING (surf));
      free (surf);
    }
  }
  else /* Generation rules are currently executed. */
  {
    printf ("start: ");
    print_surface (LEFT_SURFACE);
    printf (", next: ");
    print_surface (RIGHT_SURFACE);
  }

  printf ("\n");
}

/*---------------------------------------------------------------------------*/

LOCAL bool_t analysis_node_exists (long_t node_index)
/* Return TRUE iff an analysis node with index <node_index> has been
 * generated in the last analysis. */
{
  analysis_node_t *node;

  reset_analysis_nodes ();
  for (node = get_next_analysis_node (); 
       node != NULL; 
       node = get_next_analysis_node ())
  {
    if (node->index == node_index)
      return TRUE;
  }
  return FALSE;
}

/*---------------------------------------------------------------------------*/

LOCAL void malaga_debug_state (long_t index)
/* Callback function for "analyse". 
 * Changes to debug mode if <state> is a state to be debugged. */
{
  if (index == debug_analysis_node_index)
    set_debug_mode (WALK_MODE, rule_system[top_grammar]);
  else
    set_debug_mode (RUN_MODE, NULL);
}

/*---------------------------------------------------------------------------*/

LOCAL void do_debug_node (string_t arguments)
/* Analyse the last argument again and stop before executing the rules for a
 * state whose tree node index is specified in <arguments>. */
{
  if (in_debugger)
    error ("in debug mode");

  debug_analysis_node_index = parse_integer (&arguments);
  parse_end (arguments);

  if (! analysis_node_exists (debug_analysis_node_index))
    error ("tree node not found");

  debug_state = malaga_debug_state;
  /* debug mode is set by "malaga_debug_state" before each rule application. */

  analyse (top_grammar, analysis_input, TRUE, TRUE);
}

LOCAL command_t debug_node_command =
{
  "debug-node dn", do_debug_node,
  "Re-analyse the last analysis input.\n"
  "Execute successor rule for given node in debug mode.\n"
  "Arguments: <node-index>\n"
  "Analysis is restarted for last input and switches to debug mode\n"
  "when executing successor rules for state <node-index>.\n"
  "\"debug-node\" can't be used in debug mode.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_debug_mor (string_t arguments)
/* Analyse <arguments> morphologically.
 * Execute morphology combination rules in debug mode. */
{
  if (in_debugger)
    error ("in debug mode");

  set_debug_mode (STEP_MODE, rule_system[MORPHOLOGY]);
  analyse_argument (MORPHOLOGY, arguments);
}

LOCAL command_t debug_mor_command =
{
  "debug-mor dm ma-debug mad", do_debug_mor,
  "Analyse morphologically. "
  "Execute morphology combination rules in debug mode.\n"
  "Arguments:\n"
  "  <input> -- analyse <input>\n"
  "  (none) -- re-analyse the last analysis argument\n"
  "Rule execution stops at the first statement.\n"
  "\"debug-mor\" can't be used in debug mode.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_debug_syn (string_t arguments)
/* Analyse <arguments> syntactically.
 * Execute syntax combination rules in debug mode. */
{
  if (in_debugger)
    error ("in debug mode");

  set_debug_mode (STEP_MODE, rule_system[SYNTAX]);
  analyse_argument (SYNTAX, arguments);
}

LOCAL command_t debug_syn_command =
{
  "debug-syn ds sa-debug sad", do_debug_syn,
  "Analyse syntactically. Execute syntax combination rules in debug mode.\n"
  "Arguments:\n"
  "  <input> -- analyse <input>\n"
  "  (none) -- re-analyse the last analysis argument\n"
  "Rule execution stops at the first statement.\n"
  "\"debug-syn\" can't be used in debug mode.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_rule (string_t arguments)
/* Print rule name, left and right surface. */
{
  if (pc == -1)
    error ("no rule executed");
  parse_end (arguments);
  print_rule ();
}

LOCAL command_t rule_command =
{
  "rule", do_rule,
  "Show the name of the executed rule and the surfaces of start and next.\n"
  "Arguments: (none)\n"
  "\"rule\" can only be used in debug mode or after a rule execution error.\n"
};

/* file analysis ============================================================*/

LOCAL void analyse_stream (grammar_t grammar, 
			   FILE *input, 
			   FILE *output,
			   FILE *statistics,
			   string_t list_file_name)
/* Analyse words or sentences in <input>, write result to <output>.
 * Analyse as said in <grammar>, <list_file_name> is name of <input> stream.
 * Write statistic information to <statistics>. */
{
  time_t analysis_start_time, analysis_stop_time;
  volatile long_t analyses, recognised, results; /* statistic information */
  volatile long_t errors, line_number;

  set_debug_mode (RUN_MODE, NULL);
  debug_state = NULL;
  recognised = results = analyses = errors = 0;
  line_number = 0;

  cache_hits = cache_accesses = 0;
  time (&analysis_start_time);
  
  while (! feof (input)) 
  {
    check_user_break ();
    line_number++;
    read_line (input, analysis_input, ANALYSIS_INPUT_LENGTH);
    preprocess_input (analysis_input);
#ifdef HANGUL
    copy_string (analysis_input, ENCODED_STRING (analysis_input), 
		 analysis_input + ANALYSIS_INPUT_LENGTH);
#endif
    if (*analysis_input != EOS) 
    {
      value_t cat;
      FILE *old_error_stream;
      jmp_buf local_error_jump_point;
      jmp_buf *old_error_jump_point;

      analyses++;

      old_error_stream = error_stream;
      error_stream = output;
      old_error_jump_point = error_jump_point;
      if (setjmp (local_error_jump_point) == 0)
      {
	error_jump_point = &local_error_jump_point;
	input_line_number = line_number;
	input_file_name = list_file_name;
	analyse (grammar, analysis_input, FALSE, TRUE);
	error_stream = old_error_stream;
	error_jump_point = old_error_jump_point;
      
	if (reset_analysis_results ())
	  recognised++;

	for (cat = get_next_analysis_result (); 
	     cat != NULL; 
	     cat = get_next_analysis_result ())
	  results++;
	
	print_output (line_number, output);
      }
      else
      {
	errors++;
	error_stream = old_error_stream;
	error_jump_point = old_error_jump_point;
      }
    }
  }

  input_line_number = -1;
  input_file_name = NULL;

  time (&analysis_stop_time);
  
  if (analyses == 0)
    fprintf (statistics, "no items analysed\n");
  else 
  {
    double time_diff = difftime (analysis_stop_time, analysis_start_time);
    
    fprintf (statistics, "items analysed:              %ld\n", analyses);
    fprintf (statistics, "items recognised:            %ld (%ld%%)\n", 
	     recognised, (100 * recognised) / analyses);
    if (errors > 0)
      fprintf (statistics, "items that created errors:   %ld (%ld%%)\n", 
	       errors, (100 * errors) / analyses);
    if (results > 0)
      fprintf (statistics, "results per recognised item: %.4G\n",
	       ((double) results / (double) recognised));
    if (time_diff > 0)
    {
      fprintf (statistics, "analysis run time:           %ld sec\n", 
	       (long_t) time_diff);
      fprintf (statistics, "items per second:            %ld\n", 
	       (long_t) (analyses / time_diff));
    }
    if (cache_accesses > 0)
    {
      fprintf (statistics, "cache accesses:              %ld\n", 
	       cache_accesses);
      fprintf (statistics, "cache hits:                  %ld (%ld%%)\n",
	       cache_hits, (100 * cache_hits) / cache_accesses);
    }
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void analyse_file (string_t arguments, grammar_t grammar)
/* Open the file with name in <arguments>, which must contain a word list
 * or sentence list, analyse all its lines according to <grammar>,
 * and write the results to a file with extension ".cat". */
{
  static FILE *list_stream, *result_stream; /* input and output streams */
  string_t list_file_name, result_file_name, file_name;

  file_name = parse_word (&arguments);
  list_file_name = new_string (absolute_path (file_name, NULL));
  free (file_name);
  
  if (*arguments != EOS)
  {
    file_name = parse_word (&arguments);
    result_file_name = new_string (absolute_path (file_name, NULL));
    free (file_name);
  }
  else
    result_file_name = concat_strings (list_file_name, ".cat", NULL);

  parse_end (arguments);

  /* Close <list_stream> and <result_stream> if they are still open. */
  if (list_stream != NULL)
  {
    fclose (list_stream);
    list_stream = NULL;
  }
  if (result_stream != NULL)
  {
    fclose (result_stream);
    result_stream = NULL;
  }

  list_stream = fopen_save (list_file_name, "r");
  result_stream = fopen_save (result_file_name, "w");

  analyse_stream (grammar, list_stream, result_stream, stdout, list_file_name);

  fclose_save (list_stream, list_file_name);
  list_stream = NULL;
  fclose_save (result_stream, result_file_name);
  result_stream = NULL;

  free (list_file_name);
  free (result_file_name);
}

/*---------------------------------------------------------------------------*/

LOCAL void do_ma_file (string_t arguments)
/* Analyse file in <arguments> morphologically. */
{
  if (in_debugger)
    error ("in debug mode");

  analyse_file (arguments, MORPHOLOGY);
}

LOCAL command_t ma_file_command =
{
  "ma-file maf", do_ma_file,
  "Analyse a word list file.\n"
  "Arguments: <input-file> [<output-file>]\n"
  "<input-file> must contain one word form on each line.\n"
  "The results are written to \"<output-file>\".\n"
  "If <output-file> is missing, they are written to  \"<input-file>.cat\".\n"
  "\"ma-file\" can't be used in debug mode.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_sa_file (string_t arguments)
/* Analyse file in <arguments> syntactically. */
{
  if (in_debugger)
    error ("in debug mode");

  if (rule_system[SYNTAX] == NULL)
    error ("rule file not loaded");

  analyse_file (arguments, SYNTAX);
}

LOCAL command_t sa_file_command =
{
  "sa-file saf", do_sa_file,
  "Analyse a sentence list file.\n"
  "Arguments: <input-file> [<output-file>]\n"
  "<input-file> must contain one sentence on each line.\n"
  "The results are written to \"<output-file>\".\n"
  "If <output-file> is missing, they are written to  \"<input-file>.cat\".\n"
  "\"sa-file\" can't be used in debug mode.\n"
};

/*===========================================================================*/

LOCAL void do_clear_cache (string_t arguments)
/* Clear the wordform analysis cache. */
{
  parse_end (arguments);
  clear_cache ();
}

LOCAL command_t clear_cache_command =
{
  "clear-cache", do_clear_cache,
  "Clear the wordform analysis cache.\n"
  "Arguments: (none)\n"

};

/*---------------------------------------------------------------------------*/

LOCAL void do_info (string_t arguments)
/* Show information about morphology and syntax. */
{
  string_t grammar;

  grammar = parse_word (&arguments);

  if (strcmp_no_case (grammar, "mor") == 0)
  {
    if (info[MORPHOLOGY] != NULL)
      printf ("%s", info[MORPHOLOGY]);
  }
  else if (strcmp_no_case (grammar, "syn") == 0)
  {
    if (info[SYNTAX] != NULL)
      printf ("%s", info[SYNTAX]);
  }
  else
    error ("\"mor\" or \"syn\" expected, not \"%s\"", grammar);

  free (grammar);
  parse_end (arguments);
}

LOCAL command_t info_command =
{
  "info", do_info,
  "Arguments:\n"
  "mor -- show information about morphology\n"
  "syn -- show information about syntax\n"
};

/*---------------------------------------------------------------------------*/

LOCAL command_t *malaga_commands[] = 
/* the commands that can be called interactively, in alphabetical order */
{
  &break_command, &clear_cache_command, &debug_mor_command, 
  &debug_node_command, &debug_syn_command, &delete_command, &get_command, 
  &go_command, &help_command, &info_command, &list_command, &ma_command, 
  &ma_file_command, &ma_mode_command, &mg_command, &next_command, 
  &output_command, &print_command, &quit_command, &result_command, 
  &rule_command, &run_command, &sa_command, &sa_file_command, &sa_mode_command,
  &set_command, &sg_command, &step_command, &trace_command, &tree_command, 
  &variables_command, &walk_command, 
  NULL
};

/*---------------------------------------------------------------------------*/

LOCAL void program_message (void)
/* Print some information about the program. */
{
  printf ("%s (%s) - Copyright (C) 1995-1998 Bjoern Beutel\n",
	  program_name, MALAGA_VERSION);
  printf ("This program comes with ABSOLUTELY NO WARRANTY.\n");
  printf ("This is free software which you may redistribute "
	  "under certain conditions.\n");
  printf ("For details, refer to the GNU General Public License.\n");
}

/*---------------------------------------------------------------------------*/

GLOBAL int main (int argc, string_t argv[])
/* The main function of "malaga". */
{
  enum {INTERACTIVE_MODE, MORPHOLOGY_MODE, SYNTAX_MODE} malaga_mode;
  short_t i;
  string_t project_file;

  malaga_mode = INTERACTIVE_MODE;
  error_stream = stderr;
  program_name = name_in_path (argv[0]);

  /* Parse arguments. */
  if (argc == 2 && (strcmp_no_case (argv[1], "-version") == 0
		    || strcmp_no_case (argv[1], "-v") == 0))
  {
    program_message ();
    exit (0);
  }

  project_file = NULL;
  for (i = 1; i < argc; i++) 
  {
    string_t argument = ((*argv[i] == '-') 
			 ? argv[i] : absolute_path (argv[i], NULL));
    
    if (has_extension (argument, "pro"))
      set_file_name (&project_file, argument, NULL);
    else if (strcmp_no_case (argument, "-morphology") == 0
	     || strcmp_no_case (argument, "-m") == 0)
      malaga_mode = MORPHOLOGY_MODE;
    else if (strcmp_no_case (argument, "-syntax") == 0
	     || strcmp_no_case (argument, "-s") == 0)
      malaga_mode = SYNTAX_MODE;
    else
      error ("illegal argument \"%s\"", argument);
  }

  if (project_file == NULL)
    error ("missing project file name");
  
  init_malaga (project_file);

  if (malaga_mode == INTERACTIVE_MODE)
  {
    rule_sys_name_t rule_systems[2];
    
    init_debugger (print_rule, malaga_commands);
    
    rule_systems[0].rule_sys = rule_system[MORPHOLOGY];
    rule_systems[0].name = "mor";
    rule_systems[1].rule_sys = rule_system[SYNTAX];
    rule_systems[1].name = "syn";
    init_breakpoints (2, rule_systems);

    program_message ();
    error_stream = stdout;

    /* Enter command loop. */
    command_loop (program_name, malaga_commands);
    
    terminate_breakpoints ();
    terminate_debugger ();
    stop_display_process ();
  }
  else
  {
    grammar_t grammar = ((malaga_mode == MORPHOLOGY_MODE) ?
			 MORPHOLOGY : SYNTAX);
    
    if (rule_system[grammar] == NULL)
      error ("rule file not loaded");
    
    analyse_stream (grammar, stdin, stdout, stderr, "stdin");
  }

  terminate_malaga ();
  free (project_file);

  return 0;
}

/*---------------------------------------------------------------------------*/
