/* File "pools.c":
 * A new data type, "pool_t", for growing vectors of items 
 * of an arbitrary type. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "basic.h"
#include "files.h"

#undef GLOBAL
#define GLOBAL

#include "pools.h"

/* A pool is a linked list of chunks, 
 *which are of fixed size once allocated. */

typedef struct CHUNK_T
{
  long_t chunk_size;    /* the maximum number of items in this chunk */
  long_t num_items;     /* the actual number of items in this chunk */
  struct CHUNK_T *next; /* the next chunk in the pool */
  long_t pad;           /* pad to the next 8-byte boundary */
  /* items follow here */
} chunk_t;

/* the pool implementation type
 * (use "new_pool" before you use any other functions on a pool variable) */
typedef struct
{
  long_t item_size;       /* size of the items that are stored in this pool */
  long_t num_items;       /* the overall number of items stored in this pool */
  long_t chunk_size;      /* the size of new chunks */
  chunk_t *first_chunk;   /* first chunk of this pool */
  chunk_t *current_chunk; /* the chunk where items will currently be stored */
                          /* (note this isn't necessarily the last chunk) */
} pool_it;

#define MIN_CHUNK_SIZE 400
/* the minimum number of the data table in a chunk */

#define CHUNK_DATA(chunk_ptr) ((void *) ((chunk_ptr) + 1))
/* Use this macro to access the data in a chunk. */

/*---------------------------------------------------------------------------*/

GLOBAL pool_t new_pool (long_t item_size)
/* Create a new pool that records items of size <item_size>. */
{
  pool_it *pool = (pool_it *) new_mem (sizeof (pool_it));

  pool->item_size = item_size;
  pool->num_items = 0;
  pool->chunk_size = MIN_CHUNK_SIZE / item_size;
  pool->first_chunk = NULL;
  pool->current_chunk = NULL;

  return (pool_t) pool;
}

/*---------------------------------------------------------------------------*/

GLOBAL void clear_pool (pool_t pool)
/* Clear <pool> (do not free any memory used by the pool). */
{
  chunk_t *current_chunk;
  pool_it *pool_i = (pool_it *) pool; /* the implementation type of <pool> */

  for (current_chunk = pool_i->first_chunk; 
       current_chunk != NULL; 
       current_chunk = current_chunk->next)
    current_chunk->num_items = 0;

  pool_i->num_items = 0;
  pool_i->current_chunk = pool_i->first_chunk;
}

/*---------------------------------------------------------------------------*/

GLOBAL void *pool_to_vector (pool_t pool)
/* Return <pool> as a C vector (contiguous memory).
 * The vector can be freed with "free" after use. */
{
  pool_it *pool_i = (pool_it *) pool; /* the implementation type of <pool> */
  void *vector = new_vector (pool_i->item_size, pool_i->num_items);
  u_byte_t *chunk; /* position where next chunk is to be copied */
  chunk_t *current_chunk;
  
  /* Collect all chunks in a new vector */
  chunk = (u_byte_t *) vector;
  for (current_chunk = pool_i->first_chunk; 
       current_chunk != NULL; 
       current_chunk = current_chunk->next) 
    {
      COPY_MEM ((void *) chunk, CHUNK_DATA (current_chunk), pool_i->item_size, 
		current_chunk->num_items);
      chunk += pool_i->item_size * current_chunk->num_items;
      if (current_chunk == pool_i->current_chunk)
	/* Only chunks up to <pool_i>->current_chunk have items in them. */
	break;
    }

  return vector;
}

/*---------------------------------------------------------------------------*/

GLOBAL void write_pool (pool_t pool, FILE *stream, string_t file_name)
/* Write <pool> to <stream> (<file_name> is given for error messages). */
{
  chunk_t *current_chunk;
  pool_it *pool_i = (pool_it *) pool; /* the implementation type of <pool> */

  /* Collect all chunks to the new vector. */
  for (current_chunk = pool_i->first_chunk; 
       current_chunk != NULL; 
       current_chunk = current_chunk->next)
  {
    fwrite_save (CHUNK_DATA (current_chunk), pool_i->item_size, 
		 current_chunk->num_items, stream, file_name);
    if (current_chunk == pool_i->current_chunk)
      /* Only chunks up to <pool_i>->current_chunk have items in them. */
      break;
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void *get_pool_space (pool_t pool, long_t num_items, long_t *index)
/* Get space for <num_items> contiguous items in pool <pool>,
 * return its address as the function's result and the index in *<index>,
 * if <index> != NULL. */
{
  void *new_ptr; /* pointer to the pool space */
  chunk_t **chunk_ptr; /* address of a chunk pointer */
  pool_it *pool_i = (pool_it *) pool; /* the implementation type of <pool> */

  /* Find the first chunk that is big enough. */
  for (chunk_ptr = &pool_i->current_chunk;
       *chunk_ptr != NULL 
       && (*chunk_ptr)->num_items + num_items > (*chunk_ptr)->chunk_size;
       chunk_ptr = &(*chunk_ptr)->next)
  {
  }

  /* If we haven't found a chunk that fits, allocate a new one. */
  if (*chunk_ptr == NULL)
  {
    if (pool_i->chunk_size < num_items)
      pool_i->chunk_size = num_items;
    
    *chunk_ptr = (chunk_t *) new_mem (sizeof (chunk_t) 
				      + pool_i->item_size 
				      * pool_i->chunk_size);
    (*chunk_ptr)->chunk_size = pool_i->chunk_size;
    (*chunk_ptr)->num_items = 0;
    (*chunk_ptr)->next = NULL;
    
    /* If this is the first chunk allocated, save its address. */
    if (pool_i->first_chunk == NULL)
      pool_i->first_chunk = *chunk_ptr;
  }
  
  /* Remember address and index of current position in pool. */
  new_ptr = (void *) ((u_byte_t *) CHUNK_DATA (*chunk_ptr) 
		      + pool_i->item_size * (*chunk_ptr)->num_items);
  if (index != NULL)
    *index = pool_i->num_items;
  
  /* Adjust indices. */
  (*chunk_ptr)->num_items += num_items;
  pool_i->num_items += num_items;
  
  pool_i->current_chunk = *chunk_ptr;

  return new_ptr;
}

/*---------------------------------------------------------------------------*/

GLOBAL long_t pool_items (pool_t pool)
/* Return the number of the items in <pool>. */
{
  return ((pool_it *) pool)->num_items;
}

/*---------------------------------------------------------------------------*/

GLOBAL void *pool_item (pool_t pool, long_t index)
/* Return the address of item with <index> in pool <pool>,
 * or NULL if there is no such item. */
{
  chunk_t *chunk;
  pool_it *pool_i = (pool_it *) pool; /* the implementation type of <pool> */

  if (index < 0 || index >= pool_i->num_items)
    return NULL;

  /* Find the right chunk. */
  chunk = pool_i->first_chunk;
  while (index >= chunk->num_items)
  {
    index -= chunk->num_items;
    chunk = chunk->next;
  }

  /* Return the address of the item. */
  return (u_byte_t *) CHUNK_DATA (chunk) + pool_i->item_size * index;
}

/*---------------------------------------------------------------------------*/

GLOBAL void *copy_to_pool (pool_t pool, 
			   void *ptr, 
			   long_t num_items, 
			   long_t *index)
/* Copy the vector *<ptr> of length <num_items> into <pool>.
 * The items of *<ptr> must be of same size as the items in <pool>.
 * Return the address of the copy as the function's result and the
 * index in *<index>, if <index> != NULL. */
{
  void *new_ptr = get_pool_space (pool, num_items, index);

  /* Copy the data into the pool. */
  COPY_MEM (new_ptr, ptr, ((pool_it *) pool)->item_size, num_items);

  return new_ptr;
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t copy_string_to_pool (pool_t pool, 
				     string_t string, 
				     long_t *index)
/* Copy the string <string> into <pool>, which must be a string pool.
 * Return the copy of the string as the function's result as well as
 * its index in *<index>, if <index> != NULL. */
{
  return (string_t) (copy_to_pool (pool, (void *) string, 
				   strlen (string) + 1, index));
}

/*---------------------------------------------------------------------------*/

GLOBAL void free_pool (pool_t pool)
/* Free all memory used by <pool>. */
{
  chunk_t *chunk;
  chunk_t *next_chunk;
  pool_it *pool_i = (pool_it *) pool; /* the implementation type of <pool> */

  if (pool == NULL)
    return;

  /* Free all chunks of this pool. */
  chunk = pool_i->first_chunk;
  while (chunk != NULL) 
  {
    next_chunk = chunk->next;
    free (chunk);
    chunk = next_chunk;
  }

  free (pool_i);
}

/*---------------------------------------------------------------------------*/
