/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysvideo.h,v 1.5 1999/12/09 22:31:53 hercules Exp $";
#endif

#ifndef _SDL_sysvideo_h
#define _SDL_sysvideo_h

/* This file prototypes the video driver implementation.
   This is designed to be easily converted to C++ in the future.
 */

#include "SDL_mouse.h"
#include "SDL_syswm.h"

/* The SDL video driver */
typedef struct SDL_VideoDevice SDL_VideoDevice;

/* Define the SDL video driver structure */
#define _THIS	SDL_VideoDevice *_this
#ifndef _STATUS
#define _STATUS	SDL_status *status
#endif
struct SDL_VideoDevice {
	/* * * */
	/* Initialization/Query functions */

	/* Initialize the native video subsystem, filling 'vformat' with the 
	   "best" display pixel format, returning 0 or -1 if there's an error.
	 */
	int (*VideoInit)(_THIS, SDL_PixelFormat *vformat);

	/* List the available video modes for the given pixel format, sorted
	   from largest to smallest.
	 */
	SDL_Rect **(*ListModes)(_THIS, SDL_PixelFormat *format, Uint32 flags);

	/* Set the requested video mode, returning a surface which will be
	   set to the SDL_VideoSurface.  The width and height will already
	   be verified by ListModes(), and the video subsystem is free to
	   set the mode to a supported bit depth different from the one
	   specified -- the desired bpp will be emulated with a shadow
	   surface if necessary.  If a new mode is returned, this function
	   should take care of cleaning up the current mode.
	 */
	SDL_Surface *(*SetVideoMode)(_THIS, SDL_Surface *current,
				int width, int height, int bpp, Uint32 flags);

	/* Sets the system palette with 
	   SDL_Palette *palette = screen->format->palette;
		'palette->colors[firstcolor] ... palette->colors[ncolors-1]'
	   and returns 1 if all entries were set properly, or 0 otherwise, with
	   the palette colors set to the actual colors that were set.
	*/
	int (*SetColors)(_THIS, int firstcolor, int ncolors);

	/* This pointer should exist in the native video subsystem and should
	   point to an appropriate update function for the current video mode
	 */
	void (*UpdateRects)(_THIS, int numrects, SDL_Rect *rects);

	/* Reverse the effects VideoInit() -- called if VideoInit() fails
	   or if the application is shutting down the video subsystem.
	*/
	void (*VideoQuit)(_THIS);

	/* * * */
	/* Hardware acceleration functions */

	/* Information about the video hardware */
	SDL_VideoInfo info;

	/* Allocates a surface in video memory */
	int (*AllocHWSurface)(_THIS, SDL_Surface *surface);

	/* Sets the hardware accelerated blit function, if any, based
	   on the current flags of the surface (colorkey, alpha, etc.)
	 */
	int (*CheckHWBlit)(_THIS, SDL_Surface *src, SDL_Surface *dst);

	/* Fills a surface rectangle with the given color */
	int (*FillHWRect)(_THIS, SDL_Surface *dst, SDL_Rect *rect, Uint32 color);

	/* Sets video mem colorkey and accelerated blit function */
	int (*SetHWColorKey)(_THIS, SDL_Surface *surface, Uint32 key);

	/* Sets per surface hardware alpha value */
	int (*SetHWAlpha)(_THIS, SDL_Surface *surface, Uint8 value);

	/* Returns a readable/writable surface */
	int (*LockHWSurface)(_THIS, SDL_Surface *surface);
	void (*UnlockHWSurface)(_THIS, SDL_Surface *surface);

	/* Performs hardware flipping */
	int (*FlipHWSurface)(_THIS, SDL_Surface *surface);

	/* Frees a previously allocated video surface */
	void (*FreeHWSurface)(_THIS, SDL_Surface *surface);

	/* * * */
	/* Window manager functions */

	/* Set the window icon image */
	void (*SetIcon)(_THIS, SDL_Surface *icon, Uint8 *mask);

	/* Set the title and icon text */
	void (*SetCaption)(_THIS, const char *title, const char *icon);

	/* Get some platform dependent window information */
	int (*GetWMInfo)(_THIS, SDL_SysWMinfo *info);

	/* * * */
	/* Cursor manager functions */

	/* Free a window manager cursor */
	void (*FreeWMCursor)(_THIS, WMcursor *cursor);

	/* Create a black/white window manager cursor */
	WMcursor *(*CreateWMCursor)(_THIS,
		Uint8 *data, Uint8 *mask, int w, int h, int hot_x, int hot_y);

	/* Show or the specified cursor, or hide if NULL */
	int (*ShowWMCursor)(_THIS, WMcursor *cursor);

	/* Warp the window manager cursor to (x,y) */
	void (*WarpWMCursor)(_THIS, Uint16 x, Uint16 y);

	/* * * */
	/* Event manager functions */

	/* Initialize keyboard mapping for this driver */
	void (*InitOSKeymap)(_THIS);

	/* Handle any queued OS events */
	void (*PumpEvents)(_THIS);

	/* * * */
	/* Data common to all drivers */
	SDL_Surface *screen;
	SDL_Surface *shadow;
	SDL_Surface *visible;
	char *wm_title;
	char *wm_icon;
	int offset_x;
	int offset_y;

	/* * * */
	/* Data private to this driver */
	struct SDL_PrivateVideoData *hidden;

	/* * * */
	/* The function used to dispose of this structure */
	void (*free)(_THIS);
};
#undef _THIS

typedef struct VideoBootStrap {
	const char *name;
	int (*available)(void);
	SDL_VideoDevice *(*create)(int devindex);
} VideoBootStrap;

#ifdef ENABLE_X11
extern VideoBootStrap X11_bootstrap;
#endif
#ifdef ENABLE_FBCON
extern VideoBootStrap FBCON_bootstrap;
#endif
#ifdef ENABLE_GGI
extern VideoBootStrap GGI_bootstrap;
#endif
#ifdef ENABLE_SVGALIB
extern VideoBootStrap SVGALIB_bootstrap;
#endif
#ifdef ENABLE_WINDIB
extern VideoBootStrap WINDIB_bootstrap;
#endif
#ifdef ENABLE_DIRECTX
extern VideoBootStrap DIRECTX_bootstrap;
#endif
#ifdef ENABLE_BWINDOW
extern VideoBootStrap BWINDOW_bootstrap;
#endif
#ifdef macintosh
#define ENABLE_TOOLBOX
#endif
#ifdef ENABLE_TOOLBOX
extern VideoBootStrap TOOLBOX_bootstrap;
#endif

/* This is the current video device */
extern SDL_VideoDevice *current_video;

#define SDL_VideoSurface	(current_video->screen)
#define SDL_ShadowSurface	(current_video->shadow)
#define SDL_PublicSurface	(current_video->visible)

#endif /* _SDL_sysvideo_h */
