/*
              YIFF Client side to Server Communications

                    (YLib functions are here)

	Functions:

	YConnection *YOpenConnection(
		char *start_arg,
		char *con_arg
	)

	int YSetAudioModeValues(  
	        YConnection *con,
	        int sample_size,
	        int channels,
	        int sample_rate,
		int direction,
	        int allow_fragmenting,
	        int num_fragments,
	        int fragment_size         In bytes
	)
	int YChangeAudioModePreset(
	        YConnection *con,
	        char *name
	)
	YAudioModeValuesStruct **YGetAudioModes(
		YConnection *con,
		int *count
	)
	void YFreeAudioModesList(
		YAudioModeValuesStruct **list,
		int count
	)

	int YGetServerStats(
		YConnection *con,
		YEventServerStats *buf
	)

	long YCalculateCycle(
	        YConnection *con,
	        int sample_rate, int channels,
	        int sample_size, int fragment_size
	)
	int YSetCycle(YConnection *con, long us)
	void YSyncAll(YConnection *con, Boolean block)

        int YGetAudioStats(YConnection *con, YEventAudioStats *buf)

	int YAddHost(YConnection *con, char *address)
        int YRemoveHost(YConnection *con, char *address)

	int YSetMixerChannel(
		YConnection *con,
		int mixer_channel_code,
		Coefficient value1,
		Coefficient value2
	)
        int YGetMixerChannel(
                YConnection *con,
                int mixer_channel_code,
                Coefficient *value1,
                Coefficient *value2
        )

	YID YStartPlaySoundObject(
	        YConnection *con,
	        char *path,
	        YDataPosition pos,
	        Coefficient left_volume,
	        Coefficient right_volume,
		int repeats
	)
	int YGetSoundObjectAttributes(
		YConnection *con,
		char *path,
		YEventSoundObjectAttributes *buf
	)
	void YDestroyPlaySoundObject(YConnection *con, YID yid)
	void YCloseConnection(YConnection *con, Boolean no_shutdown)
	void YShutdownServer(YConnection *con)
	int YGetNextEvent(
	        YConnection *con,
	        YEvent *event,
	        Boolean block
	)
	void YPutBackEvent(
	        YConnection *con,
	        YEvent *event
	)

	---

	These functions are prototyped in Ylib.h (not yclientio.h).

 */

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <db.h>
#include <errno.h>
extern int h_errno;		/* For netdb.h */
#include <time.h>
#include <sys/time.h>
#include <signal.h>
#include <unistd.h>
#include <ctype.h>

#define _USE_BSD
#include <sys/types.h>
#include <sys/resource.h>
#include <sys/wait.h>

#include <netdb.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>          /* For inet_ntoa() */
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "../include/Y.h"
#include "../include/Ylib.h"
#include "../include/Yclientnet.h"


/*
 *	Connect retries:
 */
#define YIFF_CONNECT_RETRIES	5

/*
 *	Default connection argument:
 */
#define DEF_CONNECT_ARG		"127.0.0.1:9433"

#define DEF_PORT		9433


/* From yclientnet.c */
extern void YSetEventToBeDisconnect(YEvent *event);


/*
 *	Local functions:
 */
void YIFF_CHILD_KILL(int s);
char *YIFF_GET_PARENT(char *path);
char **YIFF_EXPLODE_CMD(char *cmd, int *strc);
int YIFF_START_SERVER(char *start_arg);
int YIFF_CONNECT_TO(char *address, int port);
void YIFF_COPY_EVENTS(YEvent *tar, YEvent *src);
void YIFF_FREE_CON(YConnection *con);



/* ******************************************************************* */


/*
 *	SIGCHLD signal handler.
 */
void YIFF_CHILD_KILL(int s)
{
        int status;

        while(wait3(&status, WNOHANG, (struct rusage *)0) > 0);

	return;
}

/*
 *	Get parent directory.
 */
char *YIFF_GET_PARENT(char *path)
{
        int i, len;
        char *strptr;
        static char rtnstr[YPathMax];


        if(path == NULL)
        {
            /* Must return a valid path, so toplevel will have to do. */
            return("/");
        }

        /* path must be absolute. */
        if(path[0] != '/')
        {
            /* Must return a valid path, so toplevel will have to do. */
            return("/");
        }


        strncpy(rtnstr, path, YPathMax);
        rtnstr[YPathMax - 1] = '\0';

        len = strlen(rtnstr);
        i = len - 1;   
        strptr = &(rtnstr[i]);

        /* Skip tailing '/' characters. */
        while(i >= 0)
        {
            if(*strptr == '/')
            {
                i--;
                strptr--;
            }
            else
            {
                break;
            }
        }

        while(i >= 0)
        {
            if(*strptr == '/')
            {
                *strptr = '\0';
                break;
            }

            i--;
            strptr--;
        }   

        if(rtnstr[0] == '\0')
            return("/");


        return(rtnstr);
}


/*
 *	Returns a dynamically allocated array of strings
 *	`exploded' from the cmd string.
 *
 *	Quotation grouping is parsed properly.
 *
 *	Calling function must free returned strings.
 */
char **YIFF_EXPLODE_CMD(char *cmd, int *strc)
{
        int x, y, z;
        int str_num;
        int src_pos;
        int len;
        char **strv;

        char skip_quotes;

char c = ' ';


        /* Error checks. */
        if(strc == NULL)
            return(NULL);
        if(cmd == NULL)
        {
            *strc = 0;  
            return(NULL);
        }


        /* ********************************************************** */
        /* Reset values. */
        src_pos = 0;
        str_num = 0;

        strv = NULL;
        *strc = 0;
             
        len = strlen(cmd);
        skip_quotes = 0;

        /* Skip initial characters. */
        while(src_pos < len)
        {
            if(cmd[src_pos] != c)
                break;

            src_pos++;
        }


        /* Begin exploding strings. */
        while(src_pos < len)
        {
            skip_quotes = 0;

            /* Get length y of this segment x. */
            x = src_pos;
            y = 0;
            while((cmd[x] != c) && (x < len))
            {
                /* Quote grouping skip. */
                if(cmd[x] == '"')
                {
                    skip_quotes = 1;
                    x++;

                    /* Seek to end quote. */
                    while(x < len)
                    {
                        if(cmd[x] == '"')
                            break;

                        x++; y++;
                    }

                    break;
                }

                x++;
                y++;
            }
            /* y is now length of this segment. */

            /* Allocate new string for this segment. */
            str_num = *strc;
            *strc += 1;
            strv = (char **)realloc(strv, *strc * sizeof(char *));
	    if(strv == NULL)
	    {
		*strc = 0;
		return(NULL);
	    }
            strv[str_num] = (char *)calloc(1, (y + 1) * sizeof(char));
            if(strv[str_num] == NULL)
            {
                return(strv);
            }

            /* Copy segment. */
            x = src_pos;
            z = 0;
            while((x < len) && (z < y))
            {
                /* Skip quotes. */
                if(cmd[x] == '"')
                {
                    x++;
                    continue;
                }

                strv[str_num][z] = cmd[x];
                z++; x++;
            }
            strv[str_num][y] = '\0';    /* Null terminate. */

            /* Seek next src_pos. */
            if(y < 1)
                src_pos += 1;
            else
                src_pos += y;
            if(skip_quotes)
            {
                while(src_pos < len)
                {
                    if(cmd[src_pos] == '"')
                    {
                        src_pos++;
                        break;
                    }
                    src_pos++;
                }
            }
            while(src_pos < len)
            {
                if(cmd[src_pos] != c)
                    break;

                src_pos++;
            }
        }


        return(strv);
}


/*
 *	Attempts to start server.
 */
int YIFF_START_SERVER(char *start_arg)
{
        int i;
        char *strptr;
        int argc;
        char **argv;
        char new_cd[YPathMax];

        pid_t p;

	char *cmd;


	if(start_arg == NULL)
	    return(-1);
        cmd = start_arg;


        /* Explode cmd string, argv and its strings are allocated. */
        argv = YIFF_EXPLODE_CMD(cmd, &argc);
        if((argv == NULL) || (argc <= 0))
            return(-1);

        /* Put NULL on last pointer. */
        argv = (char **)realloc(argv, (argc + 1) * sizeof(char *));
        if(argv == NULL)
            return(-1);   
        argv[argc] = NULL;


        /* Set new current dir. */
        if(argc > 0)
        {
            strptr = YIFF_GET_PARENT(argv[0]);
            if(strptr == NULL)
                strncpy(new_cd, "/", YPathMax);
            else
                strncpy(new_cd, strptr, YPathMax);
        }
        else
        {
            strncpy(new_cd, "/", YPathMax);
        }
        new_cd[YPathMax - 1] = '\0';


        /* ********************************************************* */

        /* Set SIGCHLD to signal handler to eliminate zombies. */
        signal(SIGCHLD, YIFF_CHILD_KILL);

        /* Fork off a process. */
        p = fork();
        switch(p)   
        {
          /* Forking error. */
          case -1:
            perror("fork");
            exit(1);
            break;

          /* We are the child: run the command then exit. */
          case 0:
            /* Change dir. */
            if(chdir(new_cd))
                chdir("/");

            /* Execute command and arguments. */
            execvp(argv[0], argv);

            exit(0);
            break;

          /* We are the parent: Do nothing. */
          default:
            break;
        }


        /* ********************************************************* */

        /* Free allocated string pointers. */
        for(i = 0; i < argc; i++)
        {
            free(argv[i]);
        }
        free(argv); argv = NULL;
        argc = 0;


	return(0);
}


/*
 *	Returns socket fd or -1 on error.
 */
int YIFF_CONNECT_TO(char *address, int port)
{
	int s;
        struct hostent *host;		/* Host pointer. */
        struct sockaddr_in haddr;


	if(address == NULL)
	    return(-1);


	/* Get host address data. */
	host = gethostbyname(address);
	if(host == NULL)
	    return(-1);

	/* Get a socket. */
	s = socket(AF_INET, SOCK_STREAM, 0);
	if(s < 0)
	    return(-1);

        haddr.sin_family = AF_INET;		/* In hbo. */   
        haddr.sin_port = htons(port);		/* In nbo. */
        haddr.sin_addr = *((struct in_addr *)host->h_addr);
        memset(&haddr.sin_zero, 0, 8);	/* Zero the rest of struct. */

	/* Connect. */
	if(
	    connect(
		s,
		(struct sockaddr *)&haddr,
                sizeof(struct sockaddr)
	    ) == -1
	)
            return(-1);

        /* Set socket to non-blocking. */
/*
        fcntl(s, F_SETFL, O_NONBLOCK);
 */

	return(s);
}


/*
 *	Coppies the information from YEvent src to tar.
 *
 *	Only the information pertainant to what type of
 *	event src is will be coppied.
 */
void YIFF_COPY_EVENTS(YEvent *tar, YEvent *src)
{
	if((tar == NULL) ||
           (src == NULL)
	)
	    return;

	if(tar == src)
	    return;

	memcpy(tar, src, sizeof(YEvent));

	return;
}

/*
 *	Deallocates connection structure and its resources.
 *
 *	Warning, does NOT close the connection!
 */
void YIFF_FREE_CON(YConnection *con)
{
	if(con == NULL)
	    return;

	free(con->queued_event);
	free(con->buf);
	free(con);

	return;
}

/*
 *	Attempts to connect to the sound server at the address
 *	and port contained in con_arg.
 *
 *	con_arg is a null terminated string of the format
 *	"address:port", ie "127.0.0.1:1234".
 *
 *	If the connect failed and start_arg is not NULL, then
 *	start_arg will be executed as the command to start the
 *	server.  After which a connect attempt will be made again.
 *
 *	Returns a pointer to a YConnection structure or NULL on
 *	error.  To disconnect and deallocate the YConnection structure,
 *	pass it to YIFFDisconnect().
 */
YConnection *YOpenConnection(char *start_arg, char *con_arg)
{
	int fd;
	char addr[1024];
	int port;

	int retry_count;
	int total_retries = YIFF_CONNECT_RETRIES;
	int yiff_we_started_server = 0;

	char *strptr;
	YConnection *con;



	/* The con_arg must contain something. */
	if(con_arg == NULL)
	    con_arg = DEF_CONNECT_ARG;


	/* Parse con_arg to get addr and port. */
	strncpy(addr, con_arg, 1024);
	addr[1024 - 1] = '\0';

	strptr = strchr(addr, ':');
	if(strptr != NULL)
	{
	    *strptr = '\0';

	    port = atoi(strptr + 1);
	}
	else
	{
	    port = DEF_PORT;
	}


	/* ********************************************************* */

	/* Connect. */	
	fd = YIFF_CONNECT_TO(addr, port);
	if(fd < 0)
	{
	    /*   Connect failed, implying that the server may not
             *   be running.  Run the server and try to connect again.
	     */

	    /* Check if start_arg is NULL. */
	    if(start_arg == NULL)
		return(NULL);	/* Can't connect, can't start, give up. */
	    else
	        YIFF_START_SERVER(start_arg);

	    /* Try reconnecting again for total_retries times. */
	    for(retry_count = 0; retry_count < total_retries; retry_count++)
	    {
	        fd = YIFF_CONNECT_TO(addr, port);
                if(fd > -1)
	        {
		    /* Server was restarted and we are able to connect. */
		    yiff_we_started_server = 1;
		    break;
		}
		sleep(1);
	    }

	    /* Could not establish connection. */
	    if(fd < 0)
		return(NULL);
	}


        /* Allocate YConnection structure. */
        con = (YConnection *)calloc(1, sizeof(YConnection));
        if(con == NULL)
            return(NULL);


	/* Set up connection values. */
	con->fd = fd;
	con->we_started_server = yiff_we_started_server;
	con->prev_generated_yid = YIDNULL;


	/* Reset queued events array. */
	con->total_queued_events = 0;
	con->queued_event = NULL;


	/* Allocate recieve buffer. */
	con->buf_len = YNetRecvBufLen;
	con->buf_cont = 0;
	con->buf = (u_int8_t *)calloc(
	    con->buf_len,
	    sizeof(u_int8_t)
	);
	if(con->buf == NULL)
	    con->buf_len = 0;


	return(con);
}


/*
 *	Changes the audio values.
 *
 *	Caution: You are strongly encuraged to NOT use this function if
 *	at all possible, as there is no way to predict if the values
 *	will work. Instead use YSetModePreset() whenever possible.
 *
 *	Returns -1 on failure, and 0 on success.
 */
int YSetAudioModeValues(
        YConnection *con,
        int sample_size,	/* 8 or 16. */
        int channels,		/* 1 or 2. */
        int sample_rate,	/* In hz. */
	int direction,		/* 0 for play, 1 for record. */
        int allow_fragmenting,
        int num_fragments,
        int fragment_size	/* In bytes. */
)
{
	int i;
	YEvent event;


	if(con == NULL)
	    return(-1);
	if(con->fd < 0)
	    return(-1);

	/* Send mode change values. */
	if(YNetSendAudioChangeValues(
            con,
            sample_size,
            channels,
            sample_rate,
            direction,
            allow_fragmenting,
            num_fragments,
            fragment_size
	) <= 0)
	    return(-1);

        /* Wait for response. */
        for(i = 0; i < 300000; i++)
        {
            if(YGetNextEvent(con, &event, False) > 0)
            {
                if(event.type == YAudioChange)
                {
                    /* No need to be picky about Preset or Values
                     * type mode change, the mode *did* change is
                     * what we care about.
                     */
                    break;
                } 
                else if((event.type == YDisconnect) ||
                        (event.type == YShutdown)
                )
		{
                    return(-1);
		}
		else
		{
                    /* Put back event. */
                    YPutBackEvent(con, &event);
		}
            }

            usleep(100);
        }

	return(0);
}

/*
 *      Changes the audio mode by preset name.
 *
 *      Returns -1 on failure, and 0 on success.
 */
int YChangeAudioModePreset(
        YConnection *con,
        char *name
)
{
	int i;
        YEvent event;


        if((con == NULL) ||
           (name == NULL)
	)
            return(-1);

        if(con->fd < 0)
            return(-1);

	/* Send change audio mode. */
	if(YNetSendAudioChangePreset(con, name) <= 0)
	    return(-1);

        /* Wait for response. */
        for(i = 0; i < 300000; i++)
        {
            if(YGetNextEvent(con, &event, False) > 0)
            {
                if(event.type == YAudioChange)
                {
		    /* Mode name different implies failure. */
		    if(strcasecmp(event.audio.mode_name, name))
			return(-1);

                    break;
                }
                else if((event.type == YDisconnect) ||
                        (event.type == YShutdown)
                )
                    return(-1);

                /* Put back event. */
                YPutBackEvent(con, &event);
            }  

            usleep(100);
        }

        return(0);
}

/*
 *	Gets list of preset audio modes and their values.
 */
YAudioModeValuesStruct **YGetAudioModes(
	YConnection *con,
	int *count
)
{
	int i, n;
	YEvent event;
	YAudioModeValuesStruct *amv_ptr;
	YAudioModeValuesStruct **ptr = NULL;


	if(count == NULL)
	    return(NULL);
	else
	    *count = 0;

        if(con == NULL)
            return(NULL);

        if(con->fd < 0)   
            return(NULL);

        /* Send get audio modes listing. */
        if(YNetSendListAudioModes(con) <= 0)
            return(NULL);

        /* Wait for response to get listing. */
        for(i = 0; i < 300000; i++)
        {
            if(YGetNextEvent(con, &event, False) > 0)
            {
                if(event.type == YListAudioModes)
                {
		    if(event.audio.mode_name[0] == '\0')
		    {
			/* An audio mode listing with no name marks the
			 * end of the listing.
			 */
			break;
		    }
		    else
		    {
			n = *count;
			*count = n + 1;

			ptr = (YAudioModeValuesStruct **)realloc(
			    ptr,
			    (*count) * sizeof(YAudioModeValuesStruct *)
			);
			if(ptr == NULL)
			{
			    *count = 0;
			    return(NULL);
			}

			ptr[n] = (YAudioModeValuesStruct *)malloc(
			    sizeof(YAudioModeValuesStruct)
			);
			if(ptr[n] == NULL)
			{
			    *count = n;
			    return(ptr);
			}

			amv_ptr = ptr[n];


			strncpy(
			    amv_ptr->name,
			    event.audio.mode_name,
			    YAudioNameMax
			);
			amv_ptr->name[YAudioNameMax - 1] = '\0';

			amv_ptr->sample_rate = event.audio.sample_rate;
                        amv_ptr->channels = event.audio.channels;
                        amv_ptr->sample_size = event.audio.sample_size;
                        amv_ptr->fragment_size_bytes = event.audio.fragment_size_bytes;
                        amv_ptr->direction = event.audio.direction;
			amv_ptr->allow_fragmenting = event.audio.allow_fragmenting;
                        amv_ptr->num_fragments = event.audio.num_fragments;

			i = 0;	/* Reset loop count to 0. */
			continue;
		    }
                }
                else if((event.type == YDisconnect) ||
                        (event.type == YShutdown)
                )
                    return(ptr);
        
                /* Put back event. */
                YPutBackEvent(con, &event);
            }
        
            usleep(100);
        }


	return(ptr);
}

/*
 *	Frees a list of audio mode values.
 */
void YFreeAudioModesList(YAudioModeValuesStruct **list, int count)
{
	int i;


	for(i = 0; i < count; i++)
	    free(list[i]);

	free(list);

	return;
}

/*
 *	Gets server stats.
 */
int YGetServerStats(YConnection *con, YEventServerStats *buf)
{
        int i;
        YEvent event;


        if((con == NULL) ||
           (buf == NULL)
        )
            return(-1);

        if(con->fd < 0)
            return(-1);


        /* Send request for server stats. */
        if(YNetSendGetServerStats(con) <= 0)
            return(-1);

        /* Wait for server stats event to come in. */
        for(i = 0; i < 300000; i++)
        {
            if(YGetNextEvent(con, &event, False) > 0)
            {
                if(event.type == YServerStats)
                {
                    memcpy(
                        buf,
                        &event.serverstats,
                        sizeof(YEventServerStats)
                    );
		    break;
                }
                else if((event.type == YDisconnect) ||
                        (event.type == YShutdown)
                )
                    return(-1);

                /* Put back event. */
                YPutBackEvent(con, &event);
            }

            usleep(100);
        }


        return(0);
}

/*
 *      Calculates the theoretical cycle (in microseconds) for
 *      the Y server given the Audio values.  The Audio values need
 *      not be the current Audio values of the recorder.
 *
 *      If the connection pointer con is NULL, then a general
 *      recommendation answer is returned. This value may not
 *      work with the recorder that you are connected to.
 */
long YCalculateCycle(
        YConnection *con,
        int sample_rate, int channels,
        int sample_size, int fragment_size
)
{
	if((sample_rate <= 0) ||
           (channels <= 0) ||
           (sample_size <= 0) ||
           (fragment_size <= 0)
	)
	    return(0);

	return((long)(
	    (Coefficient)1000000 / (Coefficient)(
                sample_rate * channels *
                ((sample_size == 16) ? 2 : 1)
	    ) * (Coefficient)fragment_size / 1.5 
	));
}

/*
 *	Sets the cycle.
 */
int YSetCycle(
	YConnection *con,
	long us
)
{
	int i;
        YEvent event;


        if(con == NULL)
            return(-1);
        if(con->fd < 0)
            return(-1);

	if(us < 1)
	    us = 1;

	/* Send set cycle. */
	if(YNetSendCycleChange(con, us) <= 0)
	    return(-1);

        /* Wait for response. */
        for(i = 0; i < 300000; i++)
        {
            if(YGetNextEvent(con, &event, False) > 0)
            {
                if(event.type == YCycleChange)
                {
                    break;
                }
                else if((event.type == YDisconnect) ||
                        (event.type == YShutdown)
                )
                    return(-1);

                /* Put back event. */ 
                YPutBackEvent(con, &event);
            }

            usleep(100);
        }

	return(0);
}


/*
 *	Syncs all of the Y server's resources.
 *
 *	Server and client will also be synced.
 */
void YSyncAll(YConnection *con, Boolean block)
{
        int i;
	YEvent event;


        if(con == NULL)
            return;
        if(con->fd < 0)
            return;

	if(YNetSendSync(con, 0) <= 0)
	    return;


	if(!block)
	    return;

        /* Wait for server sync response. */
        for(i = 0; i < 300000; i++)
        {
            if(YGetNextEvent(con, &event, False) > 0)
            {
                if(event.type == YSync)
                {
                    break;
                }
                else if((event.type == YDisconnect) ||
                        (event.type == YShutdown)
                )
                    return;

                /* Put back event. */
                YPutBackEvent(con, &event);
            }
 
            usleep(100);
        }


	return;
}


/*
 *	Gets audio device statistics and stores them in
 *	the YEventAudioStats buffer.
 */
int YGetAudioStats(YConnection *con, YEventAudioStats *buf)
{
	int i;
	YEvent event;


        if((con == NULL) ||
	   (buf == NULL)
	)
            return(-1);

        if(con->fd < 0)
            return(-1);


	/* Send request for audio stats. */
	if(YNetSendGetAudioStats(con) <= 0)
	    return(-1);

	/* Wait for audio stats event to come in. */
	for(i = 0; i < 300000; i++)
	{
	    if(YGetNextEvent(con, &event, False) > 0)
	    {
		if(event.type == YAudioStats)
		{
		    memcpy(
			buf,
			&event.audiostats,
			sizeof(YEventAudioStats)
		    );
		    break;
		}
		else if((event.type == YDisconnect) ||
		        (event.type == YShutdown)
		)
		    return(-1);

		/* Put back event. */
		YPutBackEvent(con, &event);
	    }

	    usleep(100);
	}


        return(0);
}


/*
 *	Add or remove host.  If op is set to 0, then address
 *	is removed from the hosts list. If op is set to 1, then
 *	address is added to the hosts list.
 */
int YAddHost(
	YConnection *con,
	YIPUnion *ip
)
{
	int i;
	YEvent event;


        if(con == NULL)
            return(-1);
        if(con->fd < 0)
            return(-1);
	if(ip == NULL)
	    return(-1);


	/* Send add host. */
	if(YNetSendSetHost(con, YSetHostAdd, ip) <= 0)
	    return(-1);

        /* Wait for response. */
        for(i = 0; i < 300000; i++)
        {
            if(YGetNextEvent(con, &event, False) > 0)
            {
                if(event.type == YSetHost)
                {
                    break;
                }
                else if((event.type == YDisconnect) ||
                        (event.type == YShutdown)
                )
                    return(-1);

                /* Put back event. */
                YPutBackEvent(con, &event);
            }

            usleep(100);
        }

	return(0);
}

int YRemoveHost(
        YConnection *con,
        YIPUnion *ip
)
{
	int i;
	YEvent event;


        if(con == NULL)
            return(-1);
        if(con->fd < 0)
            return(-1);
        if(ip == NULL)
            return(-1);


	/* Send remove host. */
        if(YNetSendSetHost(con, YSetHostRemove, ip) <= 0)
	    return(-1);

        /* Wait for response. */
        for(i = 0; i < 300000; i++)
        {
            if(YGetNextEvent(con, &event, False) > 0)
            {
                if(event.type == YSetHost)
                {
                    break;
                }
                else if((event.type == YDisconnect) ||
                        (event.type == YShutdown)
                )
                    return(-1);

                /* Put back event. */
                YPutBackEvent(con, &event);
            }

            usleep(100);
        }

        return(0);
}


/*
 *	Sets specified mixer channel values.
 */
int YSetMixerChannel(
	YConnection *con,
	int mixer_channel_code,
	Coefficient value1,
	Coefficient value2
) 
{
        if(con == NULL)
            return(-1);
        if(con->fd < 0)
            return(-1);

	/* Sanitize values. */
	if(mixer_channel_code < 0)
	    return(-1);

	if(value1 < 0)
	    value1 = 0;
	if(value1 > 1)
	    value1 = 1;

        if(value2 < 0)
            value2 = 0;
        if(value2 > 1) 
            value2 = 1;

	/* Send set mixer device. */
	if(YNetSendSetMixerChannel(
	    con,
	    mixer_channel_code,
	    value1,
	    value2
	) <= 0)
	    return(-1);

        /* Do not catch response, calling function may want to have it. */


        return(0);
}


/*
 *      Gets specified mixer channel values.
 */
int YGetMixerChannel(
	YConnection *con,
	int mixer_channel_code,
	Coefficient *value1,
	Coefficient *value2
)
{
        int i;
        YEvent event;
        
            
        if(con == NULL)
            return(-1);
        if(con->fd < 0)
            return(-1);


	/* Send request for mixer device values. */
	if(YNetSendGetMixerChannel(con, mixer_channel_code) <= 0)
	    return(-1);


        /* Wait for response. */
        for(i = 0; i < 300000; i++)
        {
            if(YGetNextEvent(con, &event, False) > 0)
            {
                if(event.type == YMixerChannel)
                {
                    if(event.mixer.code != mixer_channel_code)
                        return(-1);

		    if(YMixerValues >= 2)
		    {
			if(value1 != NULL)
			    *value1 = event.mixer.value[0];

                        if(value2 != NULL)
			    *value2 = event.mixer.value[1];
		    }

                    break;
                }
                else if((event.type == YDisconnect) ||
                        (event.type == YShutdown)
                )
                    return(-1);
 
                /* Put back event. */
                YPutBackEvent(con, &event);
            }

            usleep(100);
        }

	return(0);
}


/*
 *	Starts playing the sound object specified by path, returns
 *	the YID indentifying the sound object being played or YIDNULL on
 *	failure.
 *
 *	The only criteria for a success here is that the sound object
 *	exist.
 */
YID YStartPlaySoundObject(
        YConnection *con,
        char *path,	/* Path on disk on server's computer. */
        YDataPosition pos,
        Coefficient left_volume,
        Coefficient right_volume,
	int repeats
)
{
	int i;
	YVolumeStruct volume;
	YID yid;
	YEvent event;


	if((con == NULL) ||
           (path == NULL)
	)
	    return(YIDNULL);

	if(con->fd < 0)
	    return(YIDNULL);


	/* Sanitize volume. */
	if(left_volume < 0)
	    left_volume = 0;
	if(left_volume > 1)
	    left_volume = 1;

	if(right_volume < 0)
	    right_volume = 0;
	if(right_volume > 1)
	    right_volume = 1;

	/* Set volume structure. */
	volume.left = (Coefficient)left_volume *
	    (Coefficient)((u_int16_t)-1);
        volume.right = (Coefficient)right_volume *
            (Coefficient)((u_int16_t)-1);


	/* Sanitize position. */
	if(pos < 0)
	    pos = 0;

	/* Increment yid dispense number and get new yid. */
	con->prev_generated_yid++;
	yid = con->prev_generated_yid;

	/* Send request to start playing. */
	if(YNetSendSoundPlay(
	    con, yid, path,
	    pos, &volume, repeats
	) <= 0)
	    return(YIDNULL);

	/* Wait for response. */
        for(i = 0; i < 300000; i++)
        {
            if(YGetNextEvent(con, &event, False) > 0)
            {
                if(event.type == YSoundObjectPlay)
                {
		    if(event.play.yid == YIDNULL)
		    {
			/* Comfermation recieved but indicates play failed. */
			return(YIDNULL);
		    }
                    break;
                }
                else if((event.type == YDisconnect) ||
                        (event.type == YShutdown)
                )
                    return(YIDNULL);
 
                /* Put back event. */
                YPutBackEvent(con, &event);
            }

            usleep(100);
        }

	/* Return with the yid of the play object. */
	return(yid);
}

/*
 *	Gets sound object attributes, returns -1 on error or
 *	0 on success.
 *
 *	Note, the sound object need not be playing in order to
 *	fetch its attributes.
 */
int YGetSoundObjectAttributes(
	YConnection *con,
	char *path,
	YEventSoundObjectAttributes *buf 
)
{
        int i;
        YEvent event;


        if((con == NULL) ||
           (path == NULL)
        )
            return(-1);

        if(con->fd < 0)
            return(-1);


        /* Send request to start playing. */
        if(YNetSendGetSoundObjectAttributes(con, path) <= 0)
            return(-1);

        /* Wait for response. */
        for(i = 0; i < 300000; i++)
        {
            if(YGetNextEvent(con, &event, False) > 0)
            {
                if(event.type == YSoundObjectAttributes)
                {
                    if(*event.sndobjattributes.path == '\0')
                    {
                        /* Empth path implies failed. */
                        return(-1);
                    }
		    /* Returned path stored in the event needs to
                     * match what we sent or else we should
		     * interprite this as an error.
		     */
		    else if(!strcmp(event.sndobjattributes.path, path))
		    {
			memcpy(
			    buf,
			    &event.sndobjattributes,
			    sizeof(YEventSoundObjectAttributes)
			);
		    }
		    else
		    {
			/* Returned path in event does mot match the path
			 * that was sent, so consider this an error.
			 */
			return(-1);
		    }
                    break;
                }
                else if((event.type == YDisconnect) ||
                        (event.type == YShutdown)
                )
                    return(YIDNULL);

                /* Put back event. */
                YPutBackEvent(con, &event);
            }

            usleep(100);
        }


        return(0);
}

/*
 *	Destroys (stops/kills/terminates) the sound object
 *	being played, indetified by yid. If the yid does not
 *	match any sound object currently being played, no
 *	operation will be performed.
 */
void YDestroyPlaySoundObject(YConnection *con, YID yid)
{
	if((con == NULL) ||
           (yid == YIDNULL)
	)
	    return;

        if(con->fd < 0)
            return;

	if(YNetSendSoundKill(con, yid) <= 0)
	    return;

	/*  Do not catch event response for this, calling function
	 *  may want to have it.
	 */

	return;
}


/*
 *	Disconnects from YIFF server and frees all allocated
 *	resources.
 *
 *      The pointer con may no longer be used after a call to this
 *      function.
 */
void YCloseConnection(YConnection *con, Boolean no_shutdown)
{
	int i;
	YEvent event;


	if(con == NULL)
	    return;


        /* If we started the server, then we should shut it down. */
        if(con->we_started_server &&
           !no_shutdown
        )
        {
	    if(con->fd > -1)
	    {
		YNetSendShutdown(con, 0);

	        /* Wait for shutdown event (for 30 seconds). */
	        for(i = 0; i < 3750; i++)
	        {
		    if(YGetNextEvent(
                        con,
                        &event,
                        False
		    ) > 0)
		    {
		        if(event.type == YShutdown)
			    break;
		    }

                    /* Resend shutdown command. */
                    YNetSendShutdown(con, 0);

		    usleep(8000);
		}

                /* Close connection. */
                close(con->fd);
                con->fd = -1;
	    }
        }
	else
        {
	    if(con->fd > -1)
            {
		YNetSendDisconnect(con, 0);

		/* Close connection. */
                close(con->fd);
                con->fd = -1;
	    }
        }


	/* Free connection structure and its resources. */
	YIFF_FREE_CON(con);


	return;
}


/*
 *	Shuts down the YIFF server, thus disconnecting and
 *	freeing all allocated resources.
 *
 *	The pointer con may no longer be used after a call to this
 *	function.
 */
void YShutdownServer(YConnection *con)
{
	int i;
	YEvent event;


        if(con == NULL)
            return;

	if(con->fd > -1)
	{
	    YNetSendShutdown(con, 0);

            /* Wait for shutdown event (for 30 seconds). */
            for(i = 0; i < 3750; i++)
            {
                if(YGetNextEvent(
                    con,
                    &event,
                    False
                ) > 0)
                {
                    if((event.type == YShutdown) ||
                       (event.type == YDisconnect)
		    )
                        break;
                }

	        /* Resend shutdown command. */
                YNetSendShutdown(con, 0);

                usleep(8000);
            }

            /* Close connection. */
            close(con->fd);
            con->fd = -1;
	}


        /* Free connection structure and its resources. */
        YIFF_FREE_CON(con);


	return;
}



/*
 *	Get next (oldest) event from the server and stores it on
 *	the pointer to the YEvent event.
 *
 *	If block is set to True, then execution will be
 *	blocked untill next event is recieved.
 */
int YGetNextEvent(
	YConnection *con,
	YEvent *event,
	Boolean block
)
{
	int i, status;
	YEvent *src_event_ptr, *tar_event_ptr;


        if((con == NULL) ||
           (event == NULL)
        )
	    return(0);

        if(con->fd < 0)
        {
            YSetEventToBeDisconnect(event);
            return(1);
	}

	/* Sanitize total queued events. */
	if(con->total_queued_events < 0)
	    con->total_queued_events = 0;


	do
	{
	    /* Get new events and put them into the queue. */
	    status = YNetRecv(con);
	    if((status < 0) ||
               (con->fd < 0)
	    )
	    {
		/* A serious recieve error occured, the connection
		 * has probably been lost so report a disconnect
		 * event. The connection's  socket is now set to -1.
		 */
	        YSetEventToBeDisconnect(event);
	        return(1);
	    }

	    /* Are there any queued events to be reported? */
	    if(con->total_queued_events > 0)
	    {
	        /* Copy OLDEST queued event to event structure. */
	        YIFF_COPY_EVENTS(event, con->queued_event);


	        /* Decrement number of and shift queued events. */
	        con->total_queued_events--;

		tar_event_ptr = con->queued_event;
		/* src_event_ptr may be invalid but if so we not access it. */
		src_event_ptr = &(con->queued_event[1]);

	        for(i = 0;
		    i < con->total_queued_events;
		    i++, tar_event_ptr++, src_event_ptr++
		)
		    YIFF_COPY_EVENTS(
		        tar_event_ptr,
		        src_event_ptr
		    );

		/* Reallocate queued event pointers. */
		if(con->total_queued_events > 0)
		{
		    con->queued_event = (YEvent *)realloc(
			con->queued_event,
			con->total_queued_events * sizeof(YEvent)
		    );
		    if(con->queued_event == NULL)
		    {
			con->total_queued_events = 0;
		    }
		}
		else
		{
		    free(con->queued_event);
		    con->queued_event = NULL;

		    con->total_queued_events = 0;
		}

		/* Report that we got event. */
	        return(1);
	    }
	    else
	    {
		if(block)
		    usleep(100);
	    }

	} while(block);


	return(0);
}


/*
 *	Puts back the event onto the queue as the newest event.
 */
void YPutBackEvent(
        YConnection *con,
        YEvent *event
)
{
	int n;


        if((con == NULL) ||
           (event == NULL)
        )
            return;

        if(con->fd < 0)
            return;


	/* Sanitize total queued events. */
        if(con->total_queued_events < 0)
	    con->total_queued_events = 0;

	/* Increment total. */
	n = con->total_queued_events;
	con->total_queued_events++;

	/* Check if exceeded maximum allowed queued events. */
        if(con->total_queued_events > YQueuedEventsMax)
	{
	    fprintf(stderr,
 "YPutBackEvent(): Connection 0x%.8x: Limit of %i queued events exceeded.\n",
		(u_int32_t)con,
		YQueuedEventsMax
	    );
	    con->total_queued_events = YQueuedEventsMax;

	    return;
	}

        /* Allocate a new queued event. */
	con->queued_event = (YEvent *)realloc(
	    con->queued_event,
	    con->total_queued_events * sizeof(YEvent)
	);
	if(con->queued_event == NULL)
	{
	    con->total_queued_events = 0;
	    return;
	}


	/* Copy input event values to new queued event position. */
        YIFF_COPY_EVENTS(&con->queued_event[n], event);


	return;
}
