// bbmail.cc for bbmail - an tool for display the mail in X11.
//
//  Copyright (c) 1998-1999 by John Kennis, jkennis@chello.nl
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// (See the included file COPYING / GPL-2.0)
//

#include "bbmail.hh"
#include "version.h"
#include <utime.h>

ToolWindow::ToolWindow(int argc,char **argv,struct CMDOPTIONS *options):
  Basewindow(argc,argv,options) {

  spooldir=options->spooldir;
  frame.x=0;
	frame.y=0;
			
	/* initialize image */
	envelopePushed=False;
	close_app=False;
	broken=False;
	
	last_totalmail=0;
	last_newmail=0;

  /* initialize xrm database */
	XrmInitialize();
 	resource = new Resource(this);

  
	pixmaplist = new PIXMAPLIST[resource->report.number_of_spoolfiles];
	spoolfile = new LOCALSPOOLFILE[resource->report.number_of_spoolfiles];
	memset(spoolfile,0,
       	sizeof(LOCALSPOOLFILE)*resource->report.number_of_spoolfiles);

  number_of_checks=resource->report.force_check_delay;

  MakeWindow(False);
	CheckMail();

	Redraw(False);
  EventLoop();
}

ToolWindow::~ToolWindow() {
	int i;
	XUnmapWindow(getDisplay(),framewin);

	/* destroy pixmaps */
	if (pixmap.frame) getImageControl()->removeImage(pixmap.frame);
	if (pixmap.label) getImageControl()->removeImage(pixmap.label);
	if (pixmap.envelope) getImageControl()->removeImage(pixmap.envelope);
	/* destroy windows */
	for (i=0;i<resource[i].report.number_of_spoolfiles;i++) {
		if (pixmaplist[i].envelope_newmail)
			getImageControl()->removeImage(pixmaplist[i].envelope_newmail);
		if (pixmaplist[i].envelope_newmail_pressed)
			getImageControl()->removeImage(pixmaplist[i].envelope_newmail_pressed);
	}


	if (frameGC) XFreeGC(getDisplay(),frameGC);
  if (labelGC) XFreeGC(getDisplay(),labelGC);
  if (menuGC) XFreeGC(getDisplay(),menuGC);
	if (menuHiBGGC) XFreeGC(getDisplay(),menuHiBGGC);
	if (menuHiGC) XFreeGC(getDisplay(),menuHiGC);
	if (menuFrameGC)   XFreeGC(getDisplay(),menuFrameGC);
  if (labelwin)	XDestroyWindow(getDisplay(),labelwin);
	if (envelopewin) XDestroyWindow(getDisplay(),envelopewin);
	XDestroyWindow(getDisplay(),framewin);
	delete resource;
}


void ToolWindow::Reconfigure(void) {
	int i;

	/* destroy pixmaps */
	getImageControl()->removeImage(pixmap.frame);
	getImageControl()->removeImage(pixmap.label);
	getImageControl()->removeImage(pixmap.envelope);
	getImageControl()->removeImage(pixmap.envelope_pressed);
	for (i=0;i<resource->report.number_of_spoolfiles;i++) {
		getImageControl()->removeImage(pixmaplist[i].envelope_newmail);
		getImageControl()->removeImage(pixmaplist[i].envelope_newmail_pressed);
	}
	mbmenu->ClearMenu();
	delete [] pixmaplist;
  delete [] spoolfile;

	resource->Reload();

	pixmaplist = new PIXMAPLIST[resource->report.number_of_spoolfiles];

	spoolfile = new LOCALSPOOLFILE[resource->report.number_of_spoolfiles];
	memset(spoolfile,0,
       	sizeof(LOCALSPOOLFILE)*resource->report.number_of_spoolfiles);

	
	
	MakeWindow(True);

  if (resource->show.label) XClearWindow(getDisplay(), labelwin);
	XClearWindow(getDisplay(), envelopewin);
	XClearWindow(getDisplay(), framewin);
  number_of_checks=resource->report.force_check_delay;	
	CheckMail();
	Redraw(False);
}


void ToolWindow::saveMenuSearch(Window window,Basemenu *menu)
{
	menuwin=window;
}

void ToolWindow::removeMenuSearch(Window window)
{
	menuwin= (Window )NULL;
}

Resource *ToolWindow::getResource()
{
  return resource;
}

void ToolWindow::MakeWindow(bool reconfigure)
{
	XSetWindowAttributes attrib;
	XWMHints wmhints;
	int i;

  max_value=1;
  for (i=0;i<resource->report.number_of_digits;i++)
    max_value*=10;

  max_value-=1;
  
	unsigned long create_mask = CWBackPixmap|CWOverrideRedirect|
                              CWCursor|CWEventMask;

	frame.height=resource->frame.font->ascent +
             	resource->frame.font->descent+4*resource->frame.bevelWidth;

	if (resource->show.envelope) {
		envelope.height=frame.height-2*resource->frame.bevelWidth -
                		2*resource->envelope.bevelWidth;
		envelope.width=(envelope.height*3)/2;

		if (!(fmod((double)envelope.width,2.0) > 0.0)) envelope.width++;
	} else
		envelope.width=0;

	label.width=label.width0=label.width1=label.width2=label.width3=0;
	int extra_space=0;

	if (resource->show.label) {

    label.width0 = XTextWidth(resource->label.font,"0",strlen("0"));
    if (resource->show.newmail_counter)
			label.width1 = resource->report.number_of_digits*label.width0;
		if (resource->show.totalmail_counter)
			label.width3 =  resource->report.number_of_digits*label.width0;
		if ((resource->show.totalmail_counter) & (resource->show.newmail_counter))
			label.width2 = XTextWidth(resource->label.font,"/",strlen("/"));
		label.height=frame.height-2*resource->frame.bevelWidth;
		extra_space=resource->frame.bevelWidth;
  	label.width=label.width1 + label.width2 + label.width3 + 
                2*resource->frame.bevelWidth;
  }

	frame.width =label.width + extra_space + envelope.width
             	+ resource->frame.bevelWidth*2+resource->envelope.bevelWidth*2;

	if (resource->position.mask & XNegative)
		resource->position.x = getCurrentScreenInfo()->getWidth()+
                           resource->position.x-frame.width;
	if (resource->position.mask & YNegative)
		resource->position.y = getCurrentScreenInfo()->getHeight()+
                           resource->position.y-frame.height;

	if (!withdrawn) {
		frame.x=resource->position.x;
		frame.y=resource->position.y;
	}

	attrib.background_pixmap = ParentRelative;
	pixmap.frame = getImageControl()->renderImage(frame.width, 
                 frame.height, &resource->frame.texture);

	if (withdrawn) {
		attrib.override_redirect = False;
   	wmhints.initial_state = WithdrawnState;
  } else if (decorated) {
  	attrib.override_redirect = False;
	 	wmhints.initial_state = NormalState;
  }	else {
		attrib.override_redirect = True;
	 	wmhints.initial_state = NormalState;
	}
		
	attrib.cursor = getSessionCursor();
	attrib.event_mask = ButtonPressMask | ButtonReleaseMask | ExposureMask |
                    	FocusChangeMask | KeyPressMask | StructureNotifyMask;

                    	
	if	(!reconfigure)
		framewin = XCreateWindow(getDisplay(), 
                          getCurrentScreenInfo()->getRootWindow(), frame.x,
                         	frame.y, frame.width,
                         	frame.height, 0, getCurrentScreenInfo()->getDepth(),
                          InputOutput,getCurrentScreenInfo()->getVisual(),
                          create_mask, &attrib);
  else if (!withdrawn)
    XMoveResizeWindow(getDisplay(),framewin,frame.x,
                         	frame.y, frame.width,
                         	frame.height);
  else
   	XResizeWindow(getDisplay(),framewin,frame.width,
                      frame.height);

	wmhints.flags = IconWindowHint | StateHint;
	wmhints.icon_window = framewin;

//  XSetStandardProperties(getDisplay(),framewin,BBTOOL,
//			 BBTOOL, None,getArgv(),getArgc(),NULL);
  XTextProperty windowname;
  XClassHint classhints;
  
  char *name=BBTOOL;
  XSizeHints sizehints;

  classhints.res_name=BBTOOL;
  classhints.res_class="bbtools";
          
  sizehints.x=getResource()->position.x;
  sizehints.y=getResource()->position.y;
  sizehints.max_width=sizehints.min_width=frame.width;
  sizehints.max_height=sizehints.min_height=frame.height;
  sizehints.flags=USPosition|PMinSize|PMaxSize;

  XStringListToTextProperty(&name,1,&windowname);
  XSetWMProperties(getDisplay(),framewin,&windowname,NULL,getArgv(),getArgc(),
                  &sizehints,&wmhints,&classhints);
  XSetWMProtocols (getDisplay(), framewin,&wm_delete_window, 1);

	if (!shape)
		XSetWindowBackgroundPixmap(getDisplay(), framewin, pixmap.frame);

	if (resource->show.label) {
		if (!resource->label.transparent)
			pixmap.label =
	    			getImageControl()->renderImage(label.width, label.height,
    	    	            	           		&resource->label.texture);
    if (!reconfigure)
			    labelwin = XCreateWindow(getDisplay(), framewin,
                         		2*resource->frame.bevelWidth
                         		+envelope.width+2*resource->envelope.bevelWidth,
                         		resource->frame.bevelWidth, 
                            label.width, label.height, 0,
                         		getCurrentScreenInfo()->getDepth(), InputOutput,
                            getCurrentScreenInfo()->getVisual(),
                            create_mask, &attrib);
    else
      XMoveResizeWindow(getDisplay(),labelwin,
			      		     	2*resource->frame.bevelWidth+
                   		envelope.width+2*resource->envelope.bevelWidth,
                   		resource->frame.bevelWidth, label.width, label.height);
		if (!resource->label.transparent)
			XSetWindowBackgroundPixmap(getDisplay(), labelwin, pixmap.label);
	}

	if (resource->show.envelope) {
		pixmap.envelope =
    		getImageControl()->renderImage(envelope.width, envelope.height,
                               			&resource->envelope.texture);

		pixmap.envelope_pressed =
    		getImageControl()->renderImage(envelope.width, envelope.height,
                               			&resource->envelope.texture_pressed);

		for (i=0;i<resource->report.number_of_spoolfiles;i++) {
			pixmaplist[i].envelope_newmail =
    			getImageControl()->renderImage(envelope.width, envelope.height,
                               				&resource->spoolfile[i].newmail_texture);

			pixmaplist[i].envelope_newmail_pressed =
    			getImageControl()->renderImage(envelope.width, envelope.height,
                       			&resource->spoolfile[i].newmail_pressed_texture);
		}

		if (!reconfigure) 
			envelopewin = XCreateWindow(getDisplay(), framewin,
              		resource->frame.bevelWidth+resource->envelope.bevelWidth,
               		resource->frame.bevelWidth+resource->envelope.bevelWidth,
               		envelope.width, envelope.height, 0,
               		getCurrentScreenInfo()->getDepth(),
                  InputOutput,getCurrentScreenInfo()->getVisual(),
                  create_mask, &attrib);
    else
      XMoveResizeWindow(getDisplay(),envelopewin,
	     						resource->frame.bevelWidth+resource->envelope.bevelWidth,
                 	resource->frame.bevelWidth+resource->envelope.bevelWidth,
                 	envelope.width, envelope.height);

		
		points[0].x =  2;
		points[0].y =  2;
		points[1].x = envelope.width/2;
		points[1].y = (envelope.height*2)/3;
		points[2].x = envelope.width-3;
		points[2].y = 2;

		points[3].x=2;
		points[3].y=envelope.height-3;
		points[4].x=envelope.width/3;
		points[4].y=envelope.height/2-1;

		points[5].x=envelope.width-3;
		points[5].y=envelope.height-3;
		points[6].x=(envelope.width*2)/3;
		points[6].y=envelope.height/2;

		points[7].x=envelope.width/3;
		points[7].y=2;
		points[8].x=(envelope.width*2)/3-1;
		points[8].y=envelope.height/2;

		points[9].x=(envelope.width*2)/3-1;
		points[9].y=2;
		points[10].x=envelope.width/3;
		points[10].y=envelope.height/2;

	}

	
	

	if (!reconfigure) {

    if (resource->show.label) {
  	  gcv.foreground = resource->label.newmail_textColor.getPixel();
	  	gcv.font = resource->label.font->fid;
		  labelGC = XCreateGC(getDisplay(), labelwin,GCFont, &gcv);
    }

    gcv.foreground = resource->envelope.textColor.getPixel();
	 	gcv.font = resource->frame.font->fid;
	  frameGC = XCreateGC(getDisplay(), framewin,GCFont, &gcv);

		gcv.font = resource->menu.font->fid;                    	
   	gcv.foreground = resource->menu.texture.getColor()->getPixel();
	  menuGC = XCreateGC(getDisplay(), framewin,GCFont|GCForeground, &gcv);

	  gcv.foreground = resource->menu.highlightColor.getPixel();
    gcv.arc_mode = ArcChord;
 		gcv.fill_style = FillSolid;
	 	menuHiBGGC = XCreateGC(getDisplay(), framewin,GCForeground|
                           GCFillStyle|GCArcMode, &gcv);

	  gcv.foreground = resource->menu.hitextColor.getPixel();	                    	
	  menuHiGC = XCreateGC(getDisplay(), framewin,	GCFont|GCForeground, &gcv);

	  gcv.foreground = resource->menu.textColor.getPixel();                    	
		menuFrameGC = XCreateGC(getDisplay(), framewin,GCFont|GCForeground, &gcv);
                    	
   	mbmenu = new Mailboxmenu(this);
		mbmenu->Update();
  } else {
    if (resource->show.label) {
  		gcv.foreground = resource->label.newmail_textColor.getPixel();   	
	  	gcv.font = resource->label.font->fid;
		  XChangeGC(getDisplay(),labelGC,GCFont, &gcv);
    }

    gcv.foreground = resource->envelope.textColor.getPixel(); 
	  gcv.font = resource->frame.font->fid;
		XChangeGC(getDisplay(),frameGC,GCFont, &gcv);

    
		gcv.font = resource->menu.font->fid;                    	
   	gcv.foreground = resource->menu.texture.getColor()->getPixel();
	  XChangeGC(getDisplay(), menuGC,GCFont|GCForeground, &gcv);
	
	  gcv.foreground = resource->menu.highlightColor.getPixel();	
  	XChangeGC(getDisplay(), menuHiBGGC,GCFont|GCForeground, &gcv);

	  gcv.foreground = resource->menu.hitextColor.getPixel();	                    	
	  XChangeGC(getDisplay(), menuHiGC, GCFont|GCForeground, &gcv);

	  gcv.foreground = resource->menu.textColor.getPixel();                    	
		XChangeGC(getDisplay(), menuFrameGC,GCFont|GCForeground, &gcv);
		
	  mbmenu->Reconfigure();
  }

	if (!reconfigure) {
		XClearWindow(getDisplay(), framewin);
		XMapWindow(getDisplay(), framewin);
		XMapSubwindows(getDisplay(), framewin);	
		mapped=true;
	}
}

int ToolWindow::ReadPipe(FILE *fp,int num[2],int number_of_values)
{
  int i=0;
  bool found=False;
  bool newline=False;
  int x=0;
  char ch[6]={0,0,0,0,0,0};

  num[0]=num[1]=0;
  while (!(feof(fp))&(x!=number_of_values)&(ch[i]!='\n')&(!newline)) {
    fread(&ch[i],1,1,fp);
    while ((ch[i]>='0' && ch[i] <='9' && ch[1]!='\n')&(!feof(fp))) {
      i++;
      if (i==7) return(0);
      fread(&ch[i],1,1,fp);
      found=True;
    }
    if (ch[i]=='\n') newline=True;
    if (found) {
      ch[i]='\0';
      num[x]=atoi(ch);
      i=0;
      x++;
      found=False;
		}
  }
	/* skip to next line */
	while (!(feof(fp))&(ch[0]!='\n')&(!newline))
		fread(&ch[0],1,1,fp);	
	return(x);
}

bool ToolWindow::CheckMHBox(int spoolno,bool force)
{
  struct stat file_status;
  FILE *fp;
  int num[2]={0,0};
  int number_of_values;

  if (resource->spoolfile[spoolno].filename!=NULL) {
		if (stat(resource->spoolfile[spoolno].filename,&file_status)!=0) {
      fprintf(stderr,"Could not read status spool file %s\n",
      resource->spoolfile[spoolno].filename);
      return(false);
    }
  }
	
  if  ((file_status.st_mtime != spoolfile[spoolno].last_mtime)&&
      (resource->spoolfile[spoolno].filename!=NULL)||
      (force && resource->spoolfile[spoolno].force_check)) {

    if (resource->spoolfile[spoolno].second_ignore)
      number_of_values=1;
    else
      number_of_values=2;
    
    spoolfile[spoolno].last_mtime = file_status.st_mtime;
    int old_newmail=spoolfile[spoolno].newmail;
    spoolfile[spoolno].newmail=0;
    spoolfile[spoolno].totalmail=0;

    if (resource->spoolfile[spoolno].proc!=NULL) {
      if ((fp=(FILE*)popen(resource->spoolfile[spoolno].proc, "r")) == NULL) {
        fprintf(stderr,"Could not run command %s\n",
                resource->spoolfile[spoolno].proc);
        return(false);
      } else {
        while (!(feof(fp))) {
          if (ReadPipe(fp,num,number_of_values)!=0) {
            if (num[0]>old_newmail)
              new_mailbox=spoolno;
              spoolfile[spoolno].newmail+=num[0];
              spoolfile[spoolno].totalmail+=num[1];
          }
        }
        pclose(fp);
      }
    }
    if  (resource->spoolfile[spoolno].reversedProc!=NULL) {
      if ((fp=(FILE*)popen(resource->spoolfile[spoolno].reversedProc, "r"))
              == NULL) {
        fprintf(stderr,"Could not run command %s\n",
                resource->spoolfile[spoolno].reversedProc);
        return(false);
      } else {
        while (!(feof(fp))) {
          if (ReadPipe(fp,num,number_of_values)!=0) {
            if (num[1]>old_newmail) new_mailbox=spoolno;
            spoolfile[spoolno].newmail+=num[1];
            spoolfile[spoolno].totalmail+=num[0];
          }
        }
        pclose(fp);
      }
    }
    if (!resource->spoolfile[spoolno].first_is_new)
      spoolfile[spoolno].newmail=spoolfile[spoolno].totalmail-
                                 spoolfile[spoolno].newmail;
    if (!resource->spoolfile[spoolno].second_is_total)
      spoolfile[spoolno].totalmail=spoolfile[spoolno].newmail+
                                   spoolfile[spoolno].totalmail;

  }

  return(true);
}

bool ToolWindow::CheckMailbox(int spoolno)
{
  int readmail=0;
  int mailcount=0;
  FILE *spool_id;
  bool read_status=True;
  char mailbuf[128];
  struct stat file_status;
  int len;
  struct utimbuf ut;

  if (!(spool_id = fopen(resource->spoolfile[spoolno].filename,"r"))) {
    fprintf(stderr,"Could not open mail spool file %s\n",
            resource->spoolfile[spoolno].filename);
    return(false);
  }
  fstat(fileno(spool_id),&file_status);
  if(file_status.st_mtime != spoolfile[spoolno].last_mtime ||
       file_status.st_size != spoolfile[spoolno].last_size) {
    spoolfile[spoolno].last_mtime = file_status.st_mtime;
    spoolfile[spoolno].last_size = file_status.st_size;
    len = strlen(resource->spoolfile[spoolno].statusFlag);
    while(fgets(mailbuf, 127, spool_id)) {
      if(!strncmp(mailbuf, "From ", 5)) {
				mailcount++;
        read_status=False;
      } else 
        if (!strncmp(mailbuf, resource->spoolfile[spoolno].statusFlag,len)) {
          if(!read_status) {
            readmail++;
            read_status=True;
          }
        }
		}

    if ((mailcount-readmail)>spoolfile[spoolno].newmail)
      new_mailbox=spoolno;
    spoolfile[spoolno].newmail=mailcount-readmail;
    spoolfile[spoolno].totalmail=mailcount;
#ifdef DEBUG
		fprintf(stderr,"mailcount=%d, readmail=%d, spoolno=%d\n",mailcount,
        		readmail,spoolno);
#endif
  }
  fclose(spool_id);
  ut.actime = file_status.st_atime;
  ut.modtime = file_status.st_mtime;
  utime(resource->spoolfile[spoolno].filename, &ut);
  return(true);
}

void ToolWindow::CheckMail() {
  int totalmail=0;
  bool change_raise=False;
  bool force=False;
  int i;
  bool redraw=False;

  newmail=0;
  broken=False;
  if (resource->report.force_check) {
    if (resource->report.force_check_delay==number_of_checks) {
      force=True;
      number_of_checks=0;
    } else number_of_checks++;
  }
  for (i=0;i<resource->report.number_of_spoolfiles;i++) {
    if (resource->spoolfile[i].procDefined)	{
      if (!(CheckMHBox(i,force)))
        broken=True;
			else {
        newmail+=spoolfile[i].newmail;
        totalmail+=spoolfile[i].totalmail;
			
        mbmenu->UpdateNumbers(spoolfile[i].newmail,spoolfile[i].totalmail,i);
      }
    } else {
      if (!(CheckMailbox(i))) 
        broken=True;
      else {
        totalmail+=spoolfile[i].totalmail;
        newmail+=spoolfile[i].newmail;
        mbmenu->UpdateNumbers(spoolfile[i].newmail,spoolfile[i].totalmail,i);
      }
    }
  }

  if (mbmenu->isVisible() && mbmenu->WaitForUpdate()) mbmenu->Update();
  if (newmail>last_newmail)
		change_raise = True;
  else {
    /* are there  ny not empty mailboxes left */
    for (i=0;i<resource->report.number_of_spoolfiles;i++) {
      if (spoolfile[i].newmail!=0) {
        new_mailbox=i;
//        change_raise = True;
      }
    }
  }
  if ((newmail!=last_newmail)|(totalmail!=last_totalmail)|(broken) )
    redraw=True;

  last_newmail=newmail;
  last_totalmail=totalmail;
  if (redraw) Redraw(change_raise);
}

void ToolWindow::Redraw(bool change_raise) {
  if (resource->show.label) {
    char t[6];
    if (resource->show.label) XClearWindow(getDisplay(),labelwin);
    if (resource->show.newmail_counter) {
      if (newmail==0)
        XSetForeground(getDisplay(),labelGC,
                    resource->label.newmail_boring_textColor.getPixel());
      else
        XSetForeground(getDisplay(),labelGC,
                    resource->label.newmail_textColor.getPixel());

      if (newmail<max_value) {
        sprintf(t, "%02d", newmail);
        int xposition=(resource->report.number_of_digits-strlen(t))
                       *label.width0 + resource->frame.bevelWidth;
        XDrawString(getDisplay(), labelwin, labelGC, xposition,
                    (label.height+resource->label.font->ascent-
                    resource->label.font->descent) / 2,
                    t, strlen(t));
      } else {
        sprintf(t, ">");
        int xposition=(resource->report.number_of_digits-strlen(t))*
                      label.width0 + resource->frame.bevelWidth;
        XDrawString(getDisplay(), labelwin, labelGC, xposition,
                   (label.height+resource->label.font->ascent-
                    resource->label.font->descent) / 2,
           					t, strlen(t));
      }
    }
    if (resource->show.totalmail_counter) {
      int totalmail_xposition=label.width0;
      if (resource->show.newmail_counter) {
        if ((last_totalmail==0)|(newmail==0))
          XSetForeground(getDisplay(),labelGC,
                    resource->label.seperator_boring_textColor.getPixel());
        else
          XSetForeground(getDisplay(),labelGC,
            resource->label.seperator_textColor.getPixel());
        sprintf(t, "/");

        XDrawString(getDisplay(), labelwin, labelGC, label.width1 + 
                    resource->frame.bevelWidth,
                    (label.height+resource->label.font->ascent-
                    resource->label.font->descent) / 2,t, strlen(t));
				totalmail_xposition=label.width1+label.width2 + 
                            resource->frame.bevelWidth;
      }
			if (last_totalmail>max_value) {
        sprintf(t, ">");
        XDrawString(getDisplay(), labelwin, labelGC, totalmail_xposition,
                    (label.height+resource->label.font->ascent-
                     resource->label.font->descent) / 2,t, strlen(t));
      }	else {
        if (last_totalmail==0)
          XSetForeground(getDisplay(),labelGC,
          						resource->label.totalmail_boring_textColor.getPixel());
        else
          XSetForeground(getDisplay(),labelGC,
                         resource->label.totalmail_textColor.getPixel());

        sprintf(t, "%02d", last_totalmail);
        if (!(resource->show.newmail_counter)) 
          totalmail_xposition-=label.width0/2-1;
        XDrawString(getDisplay(), labelwin, labelGC, totalmail_xposition,
                    (label.height+resource->label.font->ascent-
                    resource->label.font->descent) / 2,t, strlen(t));
      }
    }
  }

  if (newmail==0) {
    mbmenu->setHighlight(-1);
    if (resource->show.envelope) {
      if (envelopePushed)
        XSetWindowBackgroundPixmap(getDisplay(), envelopewin,
                          					pixmap.envelope_pressed);
      else
				XSetWindowBackgroundPixmap(getDisplay(), envelopewin, 
                                   pixmap.envelope);
      XSetForeground(getDisplay(),frameGC,
                     resource->envelope.textColor.getPixel());
      XClearWindow(getDisplay(),envelopewin);
      XDrawLines(getDisplay(),envelopewin,frameGC,points,3,CoordModeOrigin);
      XDrawLine(getDisplay(), envelopewin, frameGC, points[3].x, points[3].y,
                points[4].x, points[4].y);
      XDrawLine(getDisplay(), envelopewin, frameGC, points[5].x, points[5].y,
                points[6].x, points[6].y);
      if (broken) {
        XSetForeground(getDisplay(),frameGC,
                       resource->envelope.error_textColor.getPixel());
        XSetLineAttributes(getDisplay(),frameGC,2,LineSolid,CapButt,JoinMiter);
        XDrawLine(getDisplay(), envelopewin, frameGC, points[7].x, points[7].y,
                  points[8].x, points[8].y);
        XDrawLine(getDisplay(), envelopewin, frameGC, points[9].x, points[9].y,
                  points[10].x, points[10].y);
        XSetLineAttributes(getDisplay(),frameGC,0,LineSolid,CapButt,JoinMiter);
      }
    }

    if (resource->show.onlyAtNewMail&mapped) {
      XUnmapSubwindows(getDisplay(),framewin);
      XUnmapWindow(getDisplay(),framewin);
      mapped=false;
    }
  } else {
    if (resource->show.envelope) {
      mbmenu->setHighlight(new_mailbox);
      if (envelopePushed)
        XSetWindowBackgroundPixmap(getDisplay(), envelopewin,
                          pixmaplist[new_mailbox].envelope_newmail_pressed);
      else
        XSetWindowBackgroundPixmap(getDisplay(), envelopewin,
                          pixmaplist[new_mailbox].envelope_newmail);

      XSetForeground(getDisplay(),frameGC,
            resource->spoolfile[new_mailbox].newmail_textColor.getPixel());

      XClearWindow(getDisplay(),envelopewin);
  		XDrawLines(getDisplay(),envelopewin,frameGC,points,3,CoordModeOrigin);
      XDrawLine(getDisplay(), envelopewin, frameGC, points[3].x,
                points[3].y, points[4].x, points[4].y);
      XDrawLine(getDisplay(), envelopewin, frameGC,points[5].x, points[5].y,
                points[6].x, points[6].y);

      if (broken) {
        XSetForeground(getDisplay(),frameGC,
          resource->spoolfile[new_mailbox].newmail_error_textColor.getPixel());
        XSetLineAttributes(getDisplay(),frameGC,2,LineSolid,CapButt,JoinMiter);
        XDrawLine(getDisplay(), envelopewin, frameGC, points[7].x, points[7].y,
                  points[8].x, points[8].y);
        XDrawLine(getDisplay(), envelopewin, frameGC, points[9].x, points[9].y,
                  points[10].x, points[10].y);
        XSetLineAttributes(getDisplay(),frameGC,0,LineSolid,CapButt,JoinMiter);
      }
    }
    if ((resource->show.onlyAtNewMail) & (!(mapped))) {
      XMapSubwindows(getDisplay(),framewin);
      XMapWindow(getDisplay(),framewin);
      mapped=true;
    }
    if (change_raise) {
      if (resource->spoolfile[new_mailbox].bell)
				XBell(getDisplay(),100);				
      if ((resource->spoolfile[new_mailbox].newmailProc!=NULL));
				system(resource->spoolfile[new_mailbox].newmailProc);
      if ((resource->report.auto_raise)&(!(raised))) {
        XRaiseWindow(getDisplay(),framewin);
        raised=True;
      }
    }
  }
}

void ToolWindow::CheckConfig() {
  struct stat file_status;

  if (stat(config_filename,&file_status)!=0)
		fprintf(stderr,"Could not open config file %s\n",
        		resource->style.conf_filename);
  else if (file_status.st_mtime != resource->style.mtime) {
    resource->style.mtime=file_status.st_mtime;
    Reconfigure();
  }
}

void ToolWindow::EventLoop(void) {
  int xfd = ConnectionNumber(getDisplay());
  time_t lastTime = time(NULL);

  while (! close_app) {
    if (XPending(getDisplay())) {
      XEvent Event;
      XNextEvent(getDisplay(), &Event);
      /* process events */
      switch (Event.type) {
        case ClientMessage:
        {
          if ((unsigned)Event.xclient.data.l[0]==wm_delete_window) return;
        }
        break;
        case Expose:
        {
          if (Event.xexpose.window==menuwin)
            mbmenu->exposeEvent(&Event.xexpose);
          else
            Redraw(False);
        }
        break;
        case ButtonPress:
        {
          if (Event.xbutton.window == menuwin)
					   mbmenu->buttonPressEvent(&Event.xbutton);
          else {
            if (Event.xbutton.button == LEFT_BUTTON) {
              if (!(raised)) {
                XRaiseWindow(getDisplay(),framewin);
                raised=True;
              }
              if (Event.xbutton.window==envelopewin) {
                envelopePushed=True;
                Redraw(False);
              }
            } else if (Event.xbutton.button == MIDDLE_BUTTON) {
              if (raised) {
                XLowerWindow(getDisplay(),framewin);
                raised=False;
              }
            } else if (Event.xbutton.button == RIGHT_BUTTON) {
              if (Event.xbutton.window==envelopewin) {
                if (!envelopePushed) {
                  envelopePushed=True;
                  Redraw(False);
								
                  if (mbmenu->isVisible()) 
                    mbmenu->hide();
                  else {
                    if (mbmenu->WaitForUpdate()) mbmenu->Update();
                    mbmenu->Move(frame.x,frame.y,withdrawn);
                    mbmenu->show();
                  }
                }
                else {
                  envelopePushed=False;
                  Redraw(False);
                  mbmenu->hide();
                }
              }
            }
          }
        }
        break;
        case ButtonRelease:
        {
          if (Event.xbutton.window == menuwin)
					   mbmenu->buttonReleaseEvent(&Event.xbutton);
          else {
            if (Event.xbutton.button == LEFT_BUTTON) {
              if (Event.xbutton.window==envelopewin) {
                envelopePushed=False;
                Redraw(False);
                if (Event.xbutton.x > 0 && Event.xbutton.x < envelope.width &&
                    Event.xbutton.y >0 && Event.xbutton.y < envelope.height ) {
                  if (newmail!=0)
                    system(resource->spoolfile[new_mailbox].runCommand);
                  else
                    system(resource->report.runCommand);
                }
              }
						}
/*            if (Event.xbutton.button == RIGHT_BUTTON) {
              if (Event.xbutton.window==envelopewin)
							{
							}
						}*/
					}
				}
        break;
        case MotionNotify:
        {
          if (Event.xmotion.window==menuwin)
            mbmenu->motionNotifyEvent(&Event.xmotion);
        }
    		break;
        case EnterNotify:
        {
          if (Event.xcrossing.window==menuwin)
  		 		 	mbmenu->enterNotifyEvent(&Event.xcrossing);
        }
        break;
        case LeaveNotify:
        {
          if (Event.xcrossing.window==menuwin)
		 	  	 	mbmenu->leaveNotifyEvent(&Event.xcrossing);
        }
        break;
        case ConfigureNotify:
        {
          if (Event.xconfigure.window==framewin && 
                          Event.xconfigure.send_event) {
  
            if (!decorated)
              Reconfigure();
            int parent_x,parent_y;
            Window parent_root;
            unsigned int parent_width;
            unsigned int parent_height;
            unsigned int parent_border_width;
            unsigned int parent_depth;
            if (!decorated) {
              XGetGeometry(getDisplay(),Event.xconfigure.above,&parent_root,
                         &parent_x,&parent_y,&parent_width,&parent_height,
                         &parent_border_width,&parent_depth);
              frame.x=Event.xconfigure.x+parent_x;
              frame.y=Event.xconfigure.y+parent_y;
              }
              else {
              frame.x=Event.xconfigure.x;
              frame.y=Event.xconfigure.y;
              if (position==NULL) position= new char [13];
              sprintf(position,"+%i+%i",frame.x,frame.y);
            }

            if (mbmenu->isVisible())
  						mbmenu->Move(frame.x,frame.y,withdrawn);
          }
        }
	  	}  /* case */

      if (time(NULL) - lastTime > resource->report.check_delay-1) {
        CheckMail();
        if (resource->style.auto_config && !withdrawn)
          CheckConfig();
        lastTime = time(NULL);
      }
    } else {
      if (time(NULL) - lastTime < resource->report.check_delay) {
        fd_set rfds;
        FD_ZERO(&rfds);
        FD_SET(xfd, &rfds);

        struct timeval tv;
        tv.tv_sec = resource->report.check_delay - (time(NULL) - lastTime);
        tv.tv_usec = 0;

        select(xfd + 1, &rfds, 0, 0, &tv);
      } else {
        CheckMail();
        if (resource->style.auto_config && !withdrawn)
					CheckConfig();
        lastTime = time(NULL);
      }
    }
  }
}
