/*  Spruce
 *  Copyright (C) 1999 Susixware
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "parse.h"

gint find_string(gchar *haystack, gchar *needle)
{           
   /* determines if one string exists inside another string.
    * returns the location of the string if found, -1 if it is not found. */
   gint index, i;
   gint len;
   gboolean same, found = FALSE;

   len = strlen(haystack) - strlen(needle);
   index = -1;
   do
   {
      index++;
      if (haystack[index] == needle[0])
      {
         i = 0;
         do
         {
            same = FALSE;
            if (haystack[index + i] == needle[i])
               same = TRUE;
            i++;
         } while(same && (i < strlen(needle)));
         if (same)
            found = TRUE;
      }
   } while (!found && (index < len));
             
   if (found)
      return (index);
   else
      return (-1);
}

gchar *strstrcase(gchar *haystack, gchar *needle)
{
   /* find the needle in the haystack neglecting case */
   gchar *ptr;
   guint len;

   if (needle == (gchar*) NULL || haystack == (gchar*) NULL)
      return NULL;

   len = strlen(needle);
   if (len > strlen(haystack))
      return NULL;

   for (ptr = haystack; *(ptr + len - 1) != '\0'; ptr++)
      if (!g_strncasecmp(ptr, needle, len))
         return ptr;

   return NULL;
}

void strninsert (gchar *string, gchar *insertstr, guint pos, size_t strmax)
{
   /* inserts one string into another string starting at position pos. */
   gchar tempstr[strmax], *tstrptr;    /* temp-string and it's pointer */
   gchar *strptr, *istrptr;            /* string pointer, insert-string pointer */
   guint i = 0;                        /* our counter */

   /* is there a string to insert? */
   if (insertstr == (gchar*) NULL)
      return;

   /* lets make sure our string isn't NULL */
   if (string == (gchar*) NULL && pos != 0)
      return;
   else
      if (string == (gchar*) NULL && pos == 0)
      {
         /* we're basically creating a new string... */
         strncpy(string, insertstr, strmax);
      }

   /* out-of-bounds condition */
   if (pos >= strmax || pos > strlen(string))
      return;

   /* We are now left with the condition that we have a string and we have
    * an insert string (non-NULL) and the insertion point is within bounds */

   memset(tempstr, 0, strmax);

   /* copy the string to to the point of insertion */
   for (strptr = string, tstrptr = tempstr; i < pos; strptr++, tstrptr++, i++)
      *tstrptr = *strptr;

   /* copy the instert-string into place */
   for (istrptr = insertstr; *istrptr != '\0' && i < strmax-1; istrptr++, tstrptr++, i++)
      *tstrptr = *istrptr;

   /* copy the remainder of the original string into the temp string */
   for ( ; *strptr != '\0' && i < strmax-1; strptr++, tstrptr++, i++)
      *tstrptr = *strptr;
   *tstrptr = '\0';  /* NULL-terminate the temp string */

   memset(string, 0, strmax);
   strncpy(string, tempstr, strmax);
}

void strcut (gchar *string, guint index, guint numchars)
{
   /* cuts a specific number of chars from a string starting at the position index */
   gchar *ptr, *strptr;
   guint i = 0;

   /* we can't cut from a NULL string */
   if (string == (gchar*) NULL)
      return;

   /* out-of-bounds condition */
   if (index >= strlen(string) || index + numchars > strlen(string))
      return;

   /* advance to the cutting-point */
   for (strptr = string; *strptr != '\0' && i < index; strptr++, i++);

   ptr = strptr;    /* strptr points to the cutting point */
   ptr += numchars; /* ptr points to the end of the cut */

   /* copy the portion after the cut */
   for ( ; *ptr != '\0'; ptr++, strptr++)
      *strptr = *ptr;
   *strptr = '\0';
}

gboolean replace_string(gchar *line, gchar *remove, gchar *replace)
{
    /* replaces the first occurence of remove with replace in line.
     * Returns TRUE on success and FALSE on failure (meaning that
     * `remove` could not be found within the string */
    gint index;
	
    index = find_string(line, remove);
    if (index == -1)
        return FALSE;
    strcut(line, index, strlen(remove));
    strninsert(line, replace, index, sizeof(line));

    return TRUE;
}

void trim_leadspc (gchar *string)
{
   /* delete the leading white spaces from the string */
   gchar *strt;
   gchar *ptr1, *ptr2;

   if (string == (gchar*) NULL)
      return;
      
   /* find the first non-space and point to it with 'strt' */
   for (strt = string; *strt != '\0'; strt++)
	   if (*strt != ' ' && *strt != '\t')
	      break;
      
   /* copy the string beginning at 'strt' into 'string' */
   for (ptr1 = string, ptr2 = strt; *ptr2 != '\0'; ptr1++, ptr2++)
	   *ptr1 = *ptr2;
   *ptr1 = '\0';
}

void trim_tailspc(gchar *string)
{
   /* delete the trailing white spaces from the string */
   gchar *end;

   if (string == (gchar *)NULL)
      return;         /* oops, someone sent us a NULL string */

   /* find the end of the string... */
   for (end = string; *end != '\0'; end++);
   /* 'end' should now be pointing to the end of the string */

   end--;

   /* move backwards until we hit the first non-space char */
   for ( ; end != string; end--)
	   if (*end != ' ' && *end != '\t' && *end != '\n')
	      break;

   /* 'end' should now be pointing to the last char in the string */

   end++;        /* move forward one space */
   *end = '\0';  /* end the string */
}

void trim_whtspc(gchar *string)
{
   /* delete the leading/trailing white space from the string */
   trim_tailspc(string);
   trim_leadspc(string);
}

void strip(gchar *string, gchar c)
{
   /* strip all occurances of c from the string */
   gchar *src, *ptr;

   for (src = string; *src != '\0'; src++)
      if (*src == c)
         for (ptr = src; *ptr != '\0'; ptr++)
            *ptr = *(ptr + 1);
   *src = '\0';
}

guint get_num_fields(gchar *string, gchar delim)
{
   /* returns the number of fields in a string
    * assumes that the delimeter does not appear
    * at the beginning nor the end of the string. */
   guint count = 1;
   gchar *ptr;

   trim_whtspc(string);

   if (string == (gchar *)NULL || *string == '\0')
      return 0;

   for (ptr = string; *ptr != '\0'; ptr++)
	   if (*ptr == delim)
	      count++;

   return count;
}

gchar *get_field(gchar *string, guint field, gchar delim)
{
   gchar *buffer = (gchar *)NULL;
   gchar *start, *ptr1;
   guint cfield = 1;

   /* Validate function inputs. */
   if (string == (gchar *)NULL || *string == '\0')
      return NULL;
      
   if (field <= 0)
      return NULL;
      
   /* set the starting position to the beginng of the string */
   start = string;

   /* Already on the field we want...? */
   if (cfield != field)
   {
	   for ( ; *start != '\0'; start++)   /* keep advancing chars... */
	      if (*start == delim)            /* when we hit a delimiter... */
         {
		      cfield++;                    /* increment the current field */
            if (cfield >= field)
               break;                    /* at the right field */
         }
   }
      
   if (field != 1)
      start++;  /* advace to the char after the delimiting char */
      
   /* copy the beginning of the new string into a buffer so we
    * won't mess up the original string when we capture the end
    * of the string in this field. */
   buffer = g_malloc0(strlen(start) + 1);
   strcpy(buffer, start);

   /* keept advancing chars until we get to the next delimiter */
   for(ptr1 = buffer; (*ptr1 != '\0') && (*ptr1 != delim); ptr1++);
      
   *ptr1 = '\0';    /* end the string */

   return buffer;
}

/* rfc1522 says that if you want 8bit chars in the headers
 * you have to encode it in a special way:
 * =?charset?encoding?text?=
 * the text goes into ?text? with the 8bit characters escaped
 * to =<hexcode> , for example =F6 would be .
 */
/* this whole function is ugly =) /kristoffer */
  
gint parse_8bit(gchar *text)
{
   gchar buffer[128], *decoded = NULL;
   gint len, wordlen = 0, index, pos, i, q = 0;
   gint equal = 0;

   memset(buffer, 0, sizeof(buffer));
   len = strlen(text);
   i = 0;
   while (i < len)
   {
      if (equal)
      {
         if(text[i] == '?')
         {
            index = i - 1;
            i++;
            pos = 0;
            wordlen = 0;
            q = 0;
            while(i < len)
            {
               buffer[pos] = text[i];
               if(buffer[pos] == '?')
                  q++;
               pos++;
               wordlen++;
               i++;
               if(q == 2)
                  break;
            }
            while(i < len && (text[i] != '?' || text[i+1] != '='))
            {
               buffer[pos] = text[i];
               pos++;
               wordlen++;
               i++;
            }
            buffer[pos] = '\0';
            pos = 0;
            /* ignore the charset and continue to the encoding */
            while(buffer[pos] != '\0' && buffer[pos] != '?')
            {
               pos++;
            }
            pos++;

            /* base64 */
            if(buffer[pos] == 'B' || buffer[pos] == 'b')
            {
            	strcut(buffer, 0, pos + 2);
            	pos = strlen(buffer);
            	decoded = decode_base64(buffer, &pos);
            }

            /* quoted-printable */
            else if(buffer[pos] == 'Q' || buffer[pos] == 'q')
            {
               strcut(buffer, 0, pos + 2);
               pos = strlen(buffer);
               decoded = decode_quoted_printable(buffer, &pos);
               pos = 0;
               while(decoded[pos] != '\0')
               {
                  if(decoded[pos] == '_')
                  {
				         decoded[pos] = ' ';
                  }
                  pos++;
               }
            }

            strcut(text, index, wordlen + 4);
            strninsert(text, decoded, index, sizeof(buffer) - 1);

            /* ok, i must be changed because it points at the end of the encoded data
             * which is has now been replaced with the decoded data */
            i = index + strlen(decoded) - 1;
            g_free(decoded);
         }
         equal = 0;
      }
      else if(text[i] == '=')
      {
      	 equal = 1;
      }
      i++;
   }
   return 1;
}

/* if there are 8bit chars in the string, encode it */
gchar *make_8bit(unsigned char *text)
{
   gint len;
   gint i = 0,pos = 15;
   gint eightbit = 0;
   unsigned char *buffer;
   gchar temp[6];

   len = strlen(text);

   while(i < len)
   {
      if(text[i] > 127)
         eightbit++;
      i++;
   }

   if(eightbit > 0)
   {
      i = 0;
      buffer = g_malloc(len + (eightbit * 3) + 40);
      memset(buffer, 0, len + (eightbit * 3) + 40);
      strcpy(buffer, "=?iso-8859-1?Q?");
      while(i < len)
      {
         if(text[i] > 127 || text[i] == '=')
         {
            sprintf(temp, "=%X", text[i]);
            strcat(buffer, temp);
            pos = pos + 3;
         }
         else if(text[i] == ' ')
         {
            buffer[pos] = '_';
            pos++;
         }
         else
         {
            buffer[pos] = text[i];
            pos++;
         }
         i++;
      }
      strcat(buffer, "?=");

      return buffer;
   }

   return g_strdup(text);
}

gchar* word_wrap(gchar *text, guint len)
{
   /* word-wrap a body of text to `len` chars per line */
   /* coded by Ben Canning <bdc@bdc.cx> */
   guint32 curpos = 0, last_break = 0, last_space = 0;

   for(curpos = 0; text[curpos] != '\0'; curpos++)
   {
      if(text[curpos] == ' ')
         last_space = curpos;
      else
         if(text[curpos] == '\n')
            last_break = curpos;

      if( (curpos - last_break >= len) && (last_space != last_break) )
      {
         text[last_space] = '\n';
         last_space = last_break = curpos;
      }
   }
   return (gchar*)text;
}

gchar *quote_body (gchar *body, gchar *rcpt, gchar *date)
{
   gchar *new_body;
   gchar *src, *dst, *new_date, *name;
   gint lines = 0;
   gint length;

   /* Count the number of lines in the body. */
   for (src = body; *src; src++)
      if (*src == '\n')
         lines++;

   new_date = (gchar *) g_malloc(strlen(date) + 1);
   dst = new_date;
   /* keep copying over until we hit the first ':' in the time stamp... */
   for (src = date; *src; src++)
   {
      if (*src == ':')
         break;
      *dst = *src;
      dst++;
   }
   /* ...now back-track... */
   for ( ; dst != new_date; dst--)
      if (*dst == ' ')
         break;
   *dst = '\0';

   /* if avail, get the user's name or else just use his email address */
   name = (gchar *) g_malloc0(strlen(rcpt) + 1);
   dst = name;
   for (src = rcpt; *src; src++)
   {
      if (*src == '<' && src != rcpt)
         break;
      *dst = *src;
      dst++;
   }
   *dst = ' ';
   for ( ; dst != name; dst--)
      if (*dst == ' ')
         *dst = '\0';
      else
         break;

   /* Allocate space for the new body. */
   length = (2 * (lines + 1)) + strlen(body) + strlen(name) + strlen(new_date) + 16;
   new_body = (gchar *)g_malloc(length);
   if (new_body == 0)
      return 0;

   /* On $new_date, $recpt wrote: */
   strcpy (new_body, "\n\nOn ");
   strcat (new_body, new_date);
   strcat (new_body, ", ");
   strcat (new_body, name);
   strcat (new_body, " wrote:\n"); 

   /* Reformat the text with leading "> ". */
   dst = new_body + strlen(new_body);
   *dst = '>';
   dst++;
   *dst = ' ';
   dst++;
   for (src = body; *src; src++)
   {
      *dst = *src;
      dst++;
      if (*src == '\n')
      {
         /* Don't quote an empty line. */
         /*if (*(src + 1) == '\n' || *(src + 1) == 0)
            break;*/

         /* Stop quoting at a sig. */
         /*if (*(src + 1) == '-' && *(src + 2) == '-' && *(src + 3) == ' ')
            break;*/

         *dst = '>';
         dst++;
         *dst = ' ';
         dst++;
      }
   }
   *(dst - 2) = '\0';

   return new_body;
}

gint getopts_short (gint argc, gchar *argv[], gchar *optstring)
{
   /* The getopts_short() function parses the command line arguments.
    * Its arguments argc and argv are the argument count and
    * array as passed to the main() function on program invocation.
    * An element of argv that starts with `-' (and is not exactly
    * "-" or "--") is an option element. The characters of this
    * element (aside from the initial `-') are option characters.
    * The third argument, optstring, holds the valid list of
    * option characters. If a character in an option element is
    * found not to be included in optstring, then 0 is returned.
    * If getopts_short() is called repeatedly, it returns
    * successively each of the option characters from each of
    * the option elements. If there are no more option characters,
    * getopts_short() returns EOF (-1). */

   gint n, i;
   gchar *s_opt, *arg_ptr;
   static guint nextchar = 0;
   static guint shortindex = 0;

   for (n = shortindex + 1; n < argc; n++)
   {
      if (*argv[n] == '-' && *(argv[n]+1) != '-')
      {
         /* we have ourselves a short option */
         arg_ptr = argv[n];
         arg_ptr++;  /* advance to the first char after the '-' */
         /* advance to where we left off */
         for (i = 0; i < nextchar; arg_ptr++, i++);

         for (s_opt = optstring; *s_opt != '\0'; s_opt++)
            if (*s_opt == *arg_ptr)
            {
               if ( *(s_opt+1) == '\0') /* if the next char is NULL... */
               {
                  shortindex = n;  /* up the index... */
                  nextchar = 0;    /* and reset nextchar */
               }
               else
                  nextchar++;      /* else remember the next char */

               return (gint) *s_opt;
            }
         if (*s_opt == '\0' && *arg_ptr != '\0')
         {
            /* Huston, we have a problem... This is an invalid arg */
            fprintf (stderr, "%s: invalid option [-%c]\n", argv[0], *arg_ptr);
            fflush(stderr);
            nextchar++;
            return 0;
         }
      }
      nextchar = 0;
      shortindex++;
   }

   return -1;
}

gint getopts_long (gint argc, gchar *argv[], gchar *optstring,
                   longopts_t *longopts, gint *longindex, gchar *param)
{
   /* The getopts_long() function works like getopts_short()
    * except that it also accepts long options, started out 
    * by two dashes ("--"). A long option may take a
    * parameter. This must be of the form "--arg=param" or 
    * "--arg param". If the long option takes a parameter,
    * param will point to the beginning of the parameter
    * inside of argv. Longindex holds the current position
    * in argv. You may use this to skip around inside of argv
    * or to skip the processing of any short options by
    * setting it to a non-zero value. In most cases, this should
    * be set to 0 on the first call fo this function so that
    * no options are missed. */

   gint n;
   gchar l_name[256], *l_ptr, *ptr;
   longopts_t *l_opt;
   gboolean used_equal = FALSE;

   param = NULL;

   /* get short options first (if we haven't started to get the long options yet) */
   if (*longindex == 0)
      if ( (n = getopts_short(argc, argv, optstring)) != -1)
         return n;

   /* okay, we made it here...so getopt() must have gotten all the short opts */
   for (n = *longindex + 1; n < argc; n++)
   {
      if (!strncmp("--", argv[n], 2))
      {
         /* looks like we found a long option */
         memset(l_name, 0, sizeof(l_name));
         l_ptr = argv[n];
         l_ptr += 2; /* advance to just beyond the "--" */
         strncpy(l_name, l_ptr, sizeof(l_name)-1);
         
         used_equal = FALSE;
         /* search and end the string at an '=' assuming there is one */
         for (ptr = l_name; *ptr != '\0'; ptr++)
            if (*ptr == '=')
            {
               *ptr = '\0';
               used_equal = TRUE;
            }

         for (l_opt = longopts; l_opt->name != NULL; l_opt++)
         {
            if (!strcmp(l_opt->name, l_name))
            {
               /* we found a match */
               *longindex = n;

               /* if NULL, then return the return val */
               /*if (l_opt->flag == NULL)
                  return l_opt->val;*/

               switch (l_opt->has_arg)
               {
                  case no_arg:
                     return l_opt->val;          /* return val */
                     break;
                  case required_arg:
                     if (used_equal)
                     {
                        for (ptr = argv[n]; *ptr != '='; ptr++);
                        ptr++;
                        param = ptr; /* set the param */
                     }
                     else
                     {
                        *longindex = n + 1;
                        if (n+1 < argc)
                           /* the param is the next element */
                           param = argv[n+1];
                        else
                        {
                           fprintf (stderr, "Warning: %s: incomplete option.\n", l_name);
                           param = NULL;    /* nothing to point to */
                        }
                     }
                     return l_opt->val;
                     break;
                  case optional_arg:
                     if (used_equal)
                     {
                        /* yep, we had an arg */
                        for (ptr = argv[n]; *ptr != '='; ptr++);
                        ptr++;
                        param = ptr; /* param is after the '=' */
                     }
                     else
                     {
                        /* maybe we have an arg, or maybe not */
                        if (n+1 < argc) /* if this fails, we can't have an arg */
                        {
                           if (argv[n+1][0] != '-') /* if this fails, no arg */
                           {
                              /* ladies and gents, we have an arg */
                              *longindex = n + 1;
                              param = argv[n+1];
                           }
                        }
                     }
                     return l_opt->val;
               }
            }
         }

         fprintf (stderr, "%s: invalid option [--%s]\n", argv[0], l_name);
         fflush(stderr);
         *longindex = n;
         return 0;
         break;
      }
   }

   return -1;
}
