/*  Spruce
 *  Copyright (C) 1999 Susixware
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "pop3io.h"

static gchar *pop3_errlist[] = { "",
                               ": Hostname failed to resolve", 
                               ": Socket failed to open",
                               ": No route to host?",
                               ": Server failed to acknowledge",
                               ": USER send()",
                               ": USER recv()",
                               ": PASS send()",
                               ": PASS recv()",
                               ": LIST send()",
                               ": LIST recv()",
                               ": STAT send()",
                               ": STAT recv()",
                               ": UIDL send()",
                               ": UIDL recv()",
                               ": RETR send()",
                               ": RETR recv()",
                               ": RSET send()",
                               ": RSET recv()",
                               ": QUIT send()",
                               ": QUIT recv()",
                               ": DELE send()",
                               ": DELE recv()" };

extern gint errno;
pop3errnos pop3errno;
extern GtkWidget *frmStatus;

extern guint timeout;


void pop3error(gchar *reason, gint len)
{
   strncpy(reason, pop3_errlist[pop3errno], len);
}

void emesg_destroy(Emesg *msg)
{
  /* destroy the mem space this email message is taking up */
  g_free(msg->header);
  g_free(msg->body);
}

gint p_user(gint sock, gchar *login)
{
   /* login to the server - step 1 (send login name) */
   gchar buffer[256];
   gchar command[256];

   /* lets send our user login to the server */
   g_snprintf(command, 255, "USER %s\r\n", login);
	
   if (send (sock, command, strlen(command), 0) < 0)
   {
      pop3errno = P_ERR_USER_SEND;
      return ERROR;
   }

   if ( (recvline(sock, buffer, 255) < 0) 
        || (find_string(buffer, "+OK") == -1) )
   {
      /* I'm guessing the pop3 server didn't like our USER command... */
      pop3errno = P_ERR_USER_RECV;
      return ERROR;
   }
	
   return SUCCESS;
}

gint p_pass(gint sock, gchar *passwd)
{
   /* login to the server - step 2 (send passwd) */
   gchar buffer[256];
   gchar command[256];

   /* lets send our user passwd to the server */
   g_snprintf(command, 255, "PASS %s\r\n", passwd);

   if (send (sock, command, strlen(command), 0) < 0)
   {
      pop3errno = P_ERR_PASS_SEND;
      return ERROR;
   }

   if ( (recvline(sock, buffer, 255) < 0) 
        || (find_string(buffer, "+OK") == -1) )
   {
      /* I'm guessing the pop3 server didn't like our PASS command... */
      pop3errno = P_ERR_USER_RECV;
      return ERROR;
   }

   return SUCCESS;
}

gint p_stat(gint sock)
{
	/* returns the number of messages on the server */
   gchar buffer[256];
   gchar command[256];
	gint num;

   /* lets STAT the pop3 server... */
   strncpy(command, "STAT\r\n", 255);

   if (send (sock, command, strlen(command), 0) < 0)
   {
      pop3errno = P_ERR_STAT_SEND;
      return ERROR;
   }

   if ( (recvline(sock, buffer, 255) < 0) 
        || (find_string(buffer, "+OK") == -1) )
   {
      /* something went wrong... */
      pop3errno = P_ERR_STAT_RECV;
      return ERROR;
   }
	
	/* get the number of messages on the server */
	sscanf(buffer, "+OK %d messages\r\n", &num);
	
	return num;
}

gint p_list(gint sock)
{
	/* list the messages on the server 
	 * returns the number of messages */
   gchar buffer[256];
   gchar command[256];
	gint num = 0;

   /* prepare the LIST command */
   strncpy(command, "LIST\r\n", 255);
	
   if (send (sock, command, strlen(command), 0) < 0)
   {
      pop3errno = P_ERR_LIST_SEND;
      return ERROR;
   }

   if ( (recvline(sock, buffer, 255) < 0) 
        || (find_string(buffer, "+OK") == -1) )
   {
      /* something went wrong */
      pop3errno = P_ERR_LIST_RECV;
      return ERROR;
   }
	
	while (recvline(sock, buffer, 255) > 0)
	{
		num++;
	}
	
	num--;
	
	return num;
}

gint p_uidl(gint sock, guint id, gchar *uid)
{
	/* get the uid of the message */
   gchar buffer[256];
   gchar command[256];
   gint id2;
   gint n;

   /* lets tell the server which message to retrieve */
   g_snprintf(command, 255, "UIDL %d\r\n", id);

   if (send (sock, command, strlen(command), 0) < 0)
   {
      pop3errno = P_ERR_UIDL_SEND;
      return ERROR;
   }

   if ( (n = recvline(sock, buffer, 255) < 0) 
        || (find_string(buffer, "+OK") == -1) )
   {
      if (n < 0)
         fprintf(stderr, "n < 0\n");
      if (find_string(buffer, "+OK") == -1)
         fprintf(stderr, "couldn't find \"+OK\" -> %s\n", buffer);
      /* Maybe that message doesn't exist? Or maybe something else got fux0red */
      pop3errno = P_ERR_UIDL_RECV;
      return ERROR;
   }

   sscanf(buffer, "+OK %d %s", &id2, uid);

   return SUCCESS;
}

gint p_retr(gint sock, guint id, FILE *fp, gboolean del_yn)
{
	/* retrieve a specific email message off the server */
   gchar buffer[256];
   gchar command[256];
   gchar uid[100];
   gchar *tmbx = get_mailbox("_Temp3");
   gint n;
   FILE *tmpfile;

   /* lets find out if we already have this message */
   if (p_uidl(sock, id, uid))
   {
      if (mailbox_have_uid(uid))
         return SUCCESS;
   }

   /* lets tell the server which message to retrieve */
   g_snprintf(command, 255, "RETR %d\r\n", id);

   if (send (sock, command, strlen(command), 0) < 0)
   {
      pop3errno = P_ERR_RETR_SEND;
      return ERROR;
   }

   if ( (n = recvline(sock, buffer, 255) < 0) 
        || (find_string(buffer, "+OK") == -1) )
   {
      if (n < 0)
         fprintf(stderr, "n < 0\n");
      if (find_string(buffer, "+OK") == -1)
         fprintf(stderr, "couldn't find \"+OK\" -> %s\n", buffer);
      /* Maybe that message doesn't exist? Or maybe something else got fux0red */
      pop3errno = P_ERR_RETR_RECV;
      g_free(tmbx);
      return ERROR;
   }

   /* first we get the mail to a temp file, then run a filter check on it
    * if it's not affected by any filters it goes to the normal mailbox. */
   tmpfile = fopen(tmbx, "w+t");
   if(tmpfile < 0)
   {
      g_free(tmbx);
      return ERROR;
   }
   chmod(tmbx, 00600);   /* set secure permissions */

   n = recvline(sock, buffer, 255);

   while ( !(n <= 0) && strncmp(buffer, ".\r\n", 3) )
   {
      strip(buffer, '\r');  /* strip all the \r's */
      fprintf(tmpfile, "%s", buffer);
      fflush(tmpfile);
      memset(buffer, 0, sizeof(buffer));
      n = recvline(sock, buffer, 255);
   }

   fprintf(tmpfile, ".\r\n");

   if(check_filters(tmpfile) == 0)
   {
      fseek(tmpfile, 0, SEEK_SET);
      while((fgets(buffer, 255, tmpfile)) != 0)
      {
         fprintf(fp, "%s", buffer);
         /*fflush(fp);*/
      }
      fflush(fp);
   }
   fclose(tmpfile);
   unlink(tmbx);

   g_free(tmbx);

   /* since we finished getting this mail message
    * lets add it to the uid list so that we know to
    * not download it next time */
   if (!del_yn)  /* if we're deleting this mesg, no use saving it's uid */
      mailbox_append_uid(uid);

   return SUCCESS;
}

gint p_dele(gint sock, guint id)
{
   /* delete message #id off the pop3 server */
   gchar buffer[256];
   gchar command[256];

   /* lets tell the server which message to delete */
   g_snprintf(command, 255, "DELE %d\r\n", id);
	
   if (send (sock, command, strlen(command), 0) < 0)
   {
      pop3errno = P_ERR_DELE_SEND;
      return ERROR;
   }

   if ( (recvline(sock, buffer, 255) < 0) 
        || (find_string(buffer, "+OK") == -1) )
   {
      /* Maybe that message doesn't exist? Or maybe something else got fux0red */
      pop3errno = P_ERR_DELE_RECV;
      return ERROR;
   }
	
   return SUCCESS;
}

gint p_rset(gint sock)
{
   /* we are going to reset the smtp server (just to be nice) */
   gchar buffer[256];
   gchar command[256];

   strncpy(command, "RSET\r\n", 255);
	
   if (send (sock, command, strlen(command), 0) < 0)
   {
      pop3errno = P_ERR_RSET_SEND;
      return ERROR;
   }
	
   if ( (recvline(sock, buffer, 255) < 0)
       || (find_string(buffer, "+OK") == -1) )
   {
      /* the server should have told us that it's been reset */
      pop3errno = P_ERR_RSET_RECV;
      return ERROR;
   }

   return SUCCESS;
}

gint p_quit(gint sock)
{
   /* lets tell the server we want to quit now */
   gchar buffer[256];
   gchar command[256];

   strncpy(command, "QUIT\r\n", 255);

   if (send (sock, command, strlen(command), 0) < 0)
   {
      pop3errno = P_ERR_QUIT_SEND;
      return ERROR;
   }
	
   if ( (recvline(sock, buffer, 255) < 0)
       || (find_string(buffer, "+OK") == -1) )
   {
      /* mr server didn't say bye-bye :( */
      pop3errno = P_ERR_QUIT_RECV;
      return ERROR;
   }
	
   return SUCCESS;
}

gint fetchmail(Server *server, gchar *login, gchar *passwd, gchar *finbox, 
               gboolean del_yn, gboolean show_dialog)
{
  /* this will interface with a pop3 daemon on a foreign host
   * and retrieve any new messages on the server.
   * "finbox" is the file to append new messages to (your Inbox)
   * "del_yn" specifies whether or not to delete the messages on
   * server once they are retrieved.
   */

   GtkWidget *progressbar = (GtkWidget *)NULL;
   GtkWidget *txtStatus   = (GtkWidget *)NULL;
   gint sock;
   gint num;
   guint i;
   gchar buffer[256];
   FILE *Inbox;

   if (show_dialog)
   {
      progressbar = get_widget(frmStatus, "progressbar");
      txtStatus   = get_widget(frmStatus, "txtStatus");
   }
	
   /* make sure we aren't claiming an error */
   pop3errno = P_NO_ERROR;
   
   /* lets open the inbox for appending */
   Inbox = fopen(finbox, "at");
   if (Inbox == NULL)
   {
      /* file failed to open */
      fprintf(stderr, "%s failed to open for appending.\n", finbox);
      return ERROR;
   }
   
   /* resolve the hostname */
   if ( !Resolve(server) )
   {
      pop3errno = P_ERR_RESOLVE;
      return ERROR;
   }

   /* assign and open a socket */
   sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
   if (sock < 0)
   {
     /* socket failed to open */
      pop3errno = P_ERR_SOCKET;
      return ERROR;
   }

   /* lets set up information about the server we will be connecting to... */
   server->sin.sin_family = AF_INET;
   server->sin.sin_port = htons(server->port);

   if (show_dialog)
   {
      gtk_entry_set_text(GTK_ENTRY(txtStatus), "Connecting to server...");
      update_display();
   }
	
   /* call the server */
   if (connect_timeo(sock, (struct sockaddr*)&server->sin, sizeof(server->sin), timeout) < 0)
   {
      /* we failed to connect */
      pop3errno = P_ERR_CONNECT;
      close(sock);
      return ERROR;
   }
	
   if ( (recvline(sock, buffer, 255) < 0)
       || (find_string(buffer, "+OK") == -1) )
   {
      /* the pop3 server failed to send a response (usually version info) */
      pop3errno = P_ERR_SERV_ACK;
      close(sock);
      return ERROR;
   }

   /* In case the daemon wants to give us more info than we expect...
    * basically what we do here is wait for recv() to return an error code
    * (EWOULDBLOCK) which would tell us that the wait timed out. To make
    * sure that recv() will send us that error code, we tell it not to
    * use blocking I/O */
   /*while ( recvline_timeo(sock, buffer, 255, 3) > 0);*/

   if (show_dialog)
   {
      gtk_entry_set_text(GTK_ENTRY(txtStatus), "Sending login name...");
      update_display();
   }
   
   /* lets login to the server... */
   if (!p_user(sock, login))
   {
      /* bad login name maybe? I dunno... */
      close(sock);
      return ERROR;
   }
   
   if (show_dialog)
   {
      gtk_entry_set_text(GTK_ENTRY(txtStatus), "Sending password...");
      update_display();
   }

   if (!p_pass(sock, passwd))
   {
      /* bad passwd maybe? I dunno... */
      close(sock);
      return ERROR;
   }
   
   if (show_dialog)
   {
      gtk_entry_set_text(GTK_ENTRY(txtStatus), "Sending STAT command...");
      update_display();
   }
   
   /* lets find out how many messages we got on the server */
   if ((num = p_stat(sock)) < 0)
   {
      /* something went wrong... */
      close(sock);
      return ERROR;
   }

   g_snprintf(buffer, 255, "%d messages on the server.", num);

   if (show_dialog)
   {
      gtk_entry_set_text(GTK_ENTRY(txtStatus), (gchar*) buffer);
       
      if (num > 0)
	      gtk_progress_bar_update(GTK_PROGRESS_BAR(progressbar), (gfloat) (0 / num) );
      else
	      gtk_progress_bar_update(GTK_PROGRESS_BAR(progressbar), (gfloat) 1);
      update_display();
   }
   
   for (i = 1; i <= num; i++)
   {
      g_snprintf(buffer, 255, "Downloading message %d of %d...", i, num);
#ifdef DEBUG
      fprintf(stderr, "%s\n", buffer);
      fflush(stderr);
#endif
      if (show_dialog)
      {
         gtk_entry_set_text(GTK_ENTRY(txtStatus), (gchar*) buffer);
         gtk_progress_bar_update(GTK_PROGRESS_BAR(progressbar), (gfloat) ((gfloat)i / (gfloat)num) );
         gtk_main_iteration();
      }

      /* quick fix for GUI locking up */
      update_display();

      if (!p_retr(sock, i, Inbox, del_yn))
      {
	      /* eek! an error occured!! */
	      fclose(Inbox);
	      close(sock);
	      return ERROR;
      }

      /* delete the message if we are told to do so */
      if (del_yn)
      {
         if (!p_dele(sock, i))
         {
	         /* eek! couldn't clickety-click to empty the recycle bin!!! */
            fclose(Inbox);
            close(sock);
            return ERROR;
         }
      }
   }
   
   /* we can close the inbox now, cuz we're done with it */
   fclose(Inbox);
   
   /*if ( !p_rset(sock) )
   {
      close(sock);
      return ERROR;
   }*/

   if ( !p_quit(sock) )
   {
      close(sock);
      return ERROR;
   }
   
   /* close the socket */
   close(sock);
   return SUCCESS;
}
