#include "../common/drawer.h"
#include "../common/ui.h"
#include "../common/comm.h"
#include "../common/space.h"
#include "../common/event.h"
#include "../common/motion.h"
#include "../common/worldio.h"
#include "../common/lang.h"
#include "clang.h"

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/signal.h>
#include <sys/types.h>
#include <sys/time.h>

#include <X11/X.h>
#include <X11/keysym.h>
#include "mibload.h"
#include "gvui.h"

/* gvevent.c global vars */
/*****************************************************************************/

#define ESC	'\033'

extern Display *dpy;
static void perftick();
static struct perf {            /* Performance metering */
    int interval;               /* Interval between auto-reports */
    int mindt, maxdt, meandt;   /* Integer milliseconds */
    int cycles;                 /* # cycles where we actually did something */
    struct timeval then;
} perf;

static int lastx;
static int lasty;
static int dragx;
static int dragy;
static int nseen;
static int dragstop, dragging, deldrag, numdrags;
int    justdragged;

static Time lastt;
Time last_event_time;
static Event  gev; /* geomview event structure */

/*****************************************************************************/

void main_loop()
{

  fd_set thesefds;
  int nwatch = 0;
  int queuefd;
  XEvent xev;
  XtInputMask xim;

  queuefd = ConnectionNumber(dpy);
  dragging = 0;
  dragstop = 1;
  justdragged = 0;
  deldrag = 0;
  numdrags = 0;

  while (1) {
    struct timeval await;
#define	BRIEF	0.1
    static struct timeval brief = { 0, 100000 };
    float timelimit;

    if(drawerstate.pause)
    {
	if(!dragging && !drawer_moving())
	    select(0, NULL, NULL, NULL, &brief);
	nseen = 0;
    }
    else
    {
      timelimit = PoolInputFDs( &thesefds, &nwatch );

      if(timelimit > 0 && drawer_moving())
	    timelimit = 0;	/* "Is anything moving?" */

      if (queuefd >= 0 )
	FD_SET(queuefd, &thesefds);

      if(queuefd >= nwatch)
	  nwatch = queuefd+1;

      if(timelimit > BRIEF) timelimit = BRIEF;

      await.tv_sec = floor(timelimit);
	await.tv_usec = 1000000*(timelimit - await.tv_sec);

	nseen = select(nwatch, &thesefds, NULL, NULL, &await);
    }

    gettimeofday(&perf.then, NULL);

    if(!drawerstate.pause)
	PoolInAll( &thesefds, &nseen );

    ui_update();

    justdragged = 0;
    deldrag = gev.t;


    while ((xim = XtAppPending(App)) != 0) {
      XtAppProcessEvent(App, xim);
    }

    if (dragging && !justdragged && !dragstop) {
      gv_rawevent( gev.dev, -1, gev.x, gev.y, gev.t);
      dragstop = 1;
    }

    if (dragging) {
      deldrag = gev.t - deldrag;
    } else {
      deldrag = 0;
    }

    XSync(dpy, False);

    gv_update_draw( ALLCAMS, 0.001 * (float)deldrag );
    mg_textureclock();

    if(perf.interval > 0)
        perftick();

    numdrags = 0;
 }

}

/*****************************************************************************/

static void
perftick()
{
    int dt;
    struct timeval now;
    gettimeofday(&now, NULL);
    dt = (now.tv_sec - perf.then.tv_sec)*1000 +
         (now.tv_usec - perf.then.tv_usec)/1000;
    if(dt > 0) {
        if(dt < perf.mindt) perf.mindt = dt;
        if(dt > perf.maxdt) perf.maxdt = dt;
        perf.meandt += dt;
        if(++perf.cycles == perf.interval)
            timing(perf.interval);
    }
}

/*****************************************************************************/

void
timing(int interval)
{
    if(perf.cycles > 0) {
        printf("%d..%d ms/cycle, mean %d ms over %d cycles\n",
                perf.mindt, perf.maxdt,
                perf.cycles ? perf.meandt/perf.cycles : 0,
                perf.cycles);
        fflush(stdout);
    }
    perf.mindt = 9999999, perf.maxdt = -1, perf.meandt = perf.cycles = 0;
    perf.interval = interval;
}

/*****************************************************************************/
void cam_input(Widget w, XtPointer data, XmDrawingAreaCallbackStruct *cbs)
{
  Boolean junk;
  panel_input(w, data, cbs->event, &junk);
}

void key_action(Widget w, XEvent *event, String *argv, Cardinal *argcp)
{
  Boolean junk;
  panel_input(w, NULL, event, &junk);
}
  
void panel_input(Widget w, XtPointer data, XEvent *event, Boolean *cont)
{
  char          str[1];
  KeySym	keysym;
  int		newstate, nc;

  switch (event->type)
  {
    case UnmapNotify:		/* Window iconified or otherwise temporarily gone */
      gv_freeze( (int)data, 1 );	/* Hard-frozen; prevent gv_redraw from working until we say so */
      break;

    case MapNotify:
      gv_freeze( (int)data, 0 );	/* Permit thawing. */
      gv_redraw( (int)data );		/* Thaw it now, in case Expose arrived before MapNotify */
      break;

    case MotionNotify:
      last_event_time = event->xmotion.time;
      cam_mouse(w, data, &event->xmotion, cont);
      return;

    case ButtonPress:
    case ButtonRelease:
      newstate = (event->type == ButtonPress);

      button.left = ((event->xbutton.state & Button1Mask) ? 1 : 0);
      button.middle = ((event->xbutton.state & Button2Mask) ? 1 : 0);
      button.right = ((event->xbutton.state & Button3Mask) ? 1 : 0);
      button.shift = ((event->xbutton.state & ShiftMask) ? 1 : 0);
      button.ctrl = ((event->xbutton.state & ControlMask) ? 1 : 0);
      dragging = newstate;

      lastt = last_event_time = event->xbutton.time;
      dragx = event->xbutton.x_root;
      dragy = YScrnSize - event->xbutton.y_root;

      if (dragstop && !newstate)
        gv_rawevent( gev.dev, -1, dragx, dragy, lastt);

      switch (event->xbutton.button) {
	 case 1:
           gev.dev = ELEFTMOUSE;
           button.left = newstate;
	   break;
	 case 2:
           gev.dev = EMIDDLEMOUSE;
           button.middle = newstate;
	   break;
	 case 3:
           gev.dev = ERIGHTMOUSE;
           button.right = newstate;
	   break;
      }

      if(gev.dev == ELEFTMOUSE && (event->xbutton.state & Mod1Mask)) {
	gev.dev = EMIDDLEMOUSE;
	button.middle = button.left;
	button.left = 0;
      }

      gev.x = dragx; gev.y = dragy; gev.t = lastt;
      deldrag = gev.t;
      gv_rawevent( gev.dev, newstate, gev.x, gev.y, gev.t);
      dragstop = 1;
      break;

    case KeyPress:
      nc = XLookupString((XKeyEvent *) event, str, 1, &keysym, NULL);
      last_event_time = event->xkey.time;

      if ((int)keysym == (int)'Q')
	str[0] = ESC;

      if (keysym == XK_Escape)
	str[0] = ESC;

      if (nc > 0) {
	  gev.dev = str[0];
	  gev.x = event->xkey.x_root;
	  gev.y = YScrnSize - event->xkey.y_root;
	  gev.t = event->xkey.time;
          gv_rawevent(str[0], 1, gev.x, gev.y, gev.t);
      }

      break;

    default:
      break;
  }
  
}

/*****************************************************************************/

void cam_mouse(Widget w, XtPointer data, XPointerMovedEvent *event,
	Boolean *cont)
{
  static lasthere = 0;
  dragging = 1;
  dragstop = 0;
  justdragged = 1;

  numdrags++;
  if (numdrags > 2)
    return;

  gev.x = event->x_root; gev.y = YScrnSize - event->y_root; gev.t = event->time;

  if ((gev.t - lastt) > uistate.cursor_still)
  {
    if (abs(gev.x - lastx) < uistate.cursor_twitch &&
	abs(gev.y - lasty) < uistate.cursor_twitch)
      {
	gev.x = dragx;
	gev.y = dragy;
      }
    lastx = gev.x;
    lasty = gev.y;
    lastt = gev.t;
  }
  dragx = gev.x;
  dragy = gev.y;
  if (lasthere != gev.t) {
    gv_rawevent( gev.dev, -1, gev.x, gev.y, gev.t);
  } else {
    gv_rawevent( gev.dev, -1, gev.x, gev.y, gev.t+1);
    dragstop = 1;
  }

  lasthere = gev.t;
}

/*****************************************************************************/

void cam_mousecross(Widget w, XtPointer data, XEnterWindowEvent *event,
	Boolean *cont)
{
  int id = (int)data;

  gv_winenter(id);
}

