/** \file lispatom.h
 *  implementation of the standard lisp elements: atom and sublist.
 *
 * class LispAtom. This class implements one atom, which is a
 * reference to a string it represents, and a pointer to the next
 * lisp atom if it is in a list.
 * The local class LispPtr implements automatic garbage collection
 * through reference counting.
 */

#ifndef __lispatom_h__
#define __lispatom_h__

#include "lispobject.h"
#include "lispstring.h"

// Flags used for atom types. These are not strictly necessary, but
// do speed up certain evaluations by avoiding a lot of overhead.
#define KFlagIsNumeric 0x01  // Quoted (after executing the args for
                          // a LispLambda, or for a LispSetQuoted)

class LispAtom : public LispObject
{
public:
    static LispAtom* New(LispStringPtr aString);
    virtual ~LispAtom();
    virtual LispStringPtr String() const;
    virtual LispObject* Copy(LispInt aRecursed);
private:
    LispAtom(LispStringPtr aString);

private:
    LispStringPtr iString;
};


class LispSubList : public LispObject
{
public:
    static LispSubList* New(LispObject* aSubList);
    virtual ~LispSubList();
    virtual LispPtr* SubList();
    virtual LispObject* Copy(LispInt aRecursed);
    virtual EvalFuncBase* EvalFunc();
    virtual void SetEvalFunc(EvalFuncBase* aEvalFunc);
private:
    LispSubList(LispObject* aSubList);
private:
    LispPtr iSubList;
    EvalFuncBase* iEvalFunc;
};


class LispGenericClass : public LispObject
{
public:
    static LispGenericClass* New(GenericClass* aClass);
    virtual ~LispGenericClass();
    virtual GenericClass* Generic();
    virtual LispObject* Copy(LispInt aRecursed);
private:
    LispGenericClass(GenericClass* aClass);
private:
    GenericClass* iClass;
};




#endif


