

#include "matrix.h"
#include "triagetypes.h"
#include "triagestubs.h"


CTrig::CTrig()
{
    PlatInt i;
    PlatDouble twopi = 2*PlatPi();
    
    for (i=0;i<NrTrigBins;i++)
    {
        bin[i] = (Fixed)(FixedOne*PlatSin((twopi*i)/TrigMask));
    }
}

Fixed CTrig::Sin(PlatInt x)
{
    return bin[((PlatInt)x)&TrigMask];
}

Fixed CTrig::Cos(PlatInt x)
{
    return bin[((PlatInt)(x+HalfPi))&TrigMask];
}

CVector::CVector(Fixed x, Fixed y, Fixed z)
{
    xf[0] = x;
    xf[1] = y;
    xf[2] = z;
    xf[3] = FixedOne;
}

void ZeroVector(CVector& aTarget)
{
    aTarget.xf[0] = 0;
    aTarget.xf[1] = 0;
    aTarget.xf[2] = 0;
    aTarget.xf[3] = FixedOne;
}

void MatrixMul(CMatrix& aTarget, CMatrix& m1, CMatrix& m2)
{
    PlatInt i,j,k;
    for (i=0;i<4;i++)
        for (j=0;j<4;j++)
            for (k=0;k<4;k++)
                aTarget.xf[i][j]+=FixedMul(m1.xf[i][k],m2.xf[k][j]);
}

void IdentityMatrix(CMatrix& aTarget)
{
    PlatInt i,j;
    for (i=0;i<4;i++)
    {
        for (j=0;j<4;j++)
        {
            if (i!=j)
                aTarget.xf[i][j]=0;
            else
                aTarget.xf[i][j]=FixedOne;
        }
    }
}

void ZeroMatrix(CMatrix& aTarget)
{
    PlatInt i,j;
    for (i=0;i<4;i++)
        for (j=0;j<4;j++)
            aTarget.xf[i][j]=0;
}

void Translate(CMatrix& aTarget, Fixed x, Fixed y, Fixed z)
{
    aTarget.xf[0][3] = x;
    aTarget.xf[1][3] = y;
    aTarget.xf[2][3] = z;
    aTarget.xf[3][3] = FixedOne;
}

void RotateZ(CMatrix& aTarget, CTrig& aTrig, PlatInt aAngle)
{
    aTarget.xf[0][0] = aTrig.Cos(aAngle);
    aTarget.xf[1][1] = aTrig.Cos(aAngle);

    aTarget.xf[0][1] = -aTrig.Sin(aAngle);
    aTarget.xf[1][0] =  aTrig.Sin(aAngle);

    aTarget.xf[2][2] = FixedOne;
}

void RotateY(CMatrix& aTarget, CTrig& aTrig, PlatInt aAngle)
{
    aTarget.xf[0][0] = aTrig.Cos(aAngle);
    aTarget.xf[2][2] = aTrig.Cos(aAngle);

    aTarget.xf[0][2] = -aTrig.Sin(aAngle);
    aTarget.xf[2][0] =  aTrig.Sin(aAngle);

    aTarget.xf[1][1] = FixedOne;
}

void RotateX(CMatrix& aTarget, CTrig& aTrig, PlatInt aAngle)
{
    aTarget.xf[1][1] = aTrig.Cos(aAngle);
    aTarget.xf[2][2] = aTrig.Cos(aAngle);

    aTarget.xf[1][2] = -aTrig.Sin(aAngle);
    aTarget.xf[2][1] =  aTrig.Sin(aAngle);

    aTarget.xf[0][0] = FixedOne;
}


void Scale(CMatrix& aTarget, Fixed aScaleX, Fixed aScaleY, Fixed aScaleZ)
{
    aTarget.xf[0][0] = FixedMul(aTarget.xf[0][0],aScaleX);
    aTarget.xf[1][1] = FixedMul(aTarget.xf[1][1],aScaleY);
    aTarget.xf[2][2] = FixedMul(aTarget.xf[2][2],aScaleZ);
}

void MatMulVec(CVector& aTarget, CMatrix& m, const CVector& v)
{
    PlatInt i,j;
    for (i=0;i<4;i++)
    {
        for (j=0;j<4;j++)
        {
            aTarget.xf[i]+=FixedMul(m.xf[i][j],v.xf[j]);
        }
    }
}




GlobalTransform::GlobalTransform()
{
    GrowTo(1);
    IdentityMatrix(Item(0));
}

void GlobalTransform::PushMatrix()
{
    CMatrix m =  Item(NrItems()-1);
    Append(m);
}

void GlobalTransform::PopMatrix()
{
    SetNrItems(NrItems()-1);
}

void GlobalTransform::Apply(CMatrix& aMatrix)
{
    CMatrix result;
    ZeroMatrix(result);
    MatrixMul(result, aMatrix, Item(NrItems()-1));
    Item(NrItems()-1) = result;
}


void GlobalTransform::Transform(CVector& aTarget, const CVector& aSource)
{
    ZeroVector(aTarget);
    MatMulVec(aTarget, Item(NrItems()-1),aSource);
}

void GlobalTransform::TransformNormal(CVector& aTarget, const CVector& aSource)
{
    CVector begin;
    CVector end;

    MatMulVec(begin, Item(NrItems()-1),CVector(0,0,0));
    MatMulVec(end, Item(NrItems()-1),aSource);

    PlatInt i;
    Fixed sum=0;
    for (i=0;i<3;i++)
    {
        aTarget.xf[i]=end.xf[i]-begin.xf[i];
        sum += FixedMul(aTarget.xf[i],aTarget.xf[i]);
    }
    Fixed length = FixedSqrt(sum);
    if (!FixedIsZero(length))
        for (i=0;i<3;i++)
        {
            aTarget.xf[i] = FixedDivide(aTarget.xf[i],length);
        }
}

void GlobalTransform::Transform(CVector& aTarget, PlatInt x, PlatInt y, PlatInt z)
{
    Transform(aTarget, CVector(IntToFixed(x),IntToFixed(y),IntToFixed(z)));
}

void UnFixVector(PlatInt& x, PlatInt& y, PlatInt& z, CVector& aSource)
{
    x=FixedToInt(aSource.xf[0]);
    y=FixedToInt(aSource.xf[1]);
    z=FixedToInt(aSource.xf[2]);
}

void GlobalTransform::Transform(PlatInt& x, PlatInt& y, PlatInt& z)
{
    CVector result(0,0,0);
    Transform(result, x, y, z);
    UnFixVector(x, y, z, result);
}



