/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module defines a new data type, "pool_t", for growing vectors of items 
 * of an arbitrary type.
 * A pool is a linked list of chunks which are of fixed size once allocated. */

/* includes =================================================================*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "basic.h"
#include "files.h"

#undef GLOBAL
#define GLOBAL

#include "pools.h"

/* constants ================================================================*/

#define MIN_CHUNK_SIZE 400
/* the minimum number of the data table in a chunk */

/* macros ===================================================================*/

#define CHUNK_DATA(chunk_ptr) ((u_byte_t *) ((chunk_ptr) + 1))
/* Use this macro to access the data in a chunk. */

/* types ====================================================================*/

typedef struct CHUNK_T
{
  int_t chunk_size; /* the maximum number of items in this chunk */
  int_t num_items; /* the actual number of items in this chunk */
  struct CHUNK_T *next; /* the next chunk in the pool */
  int_t pad; /* pad to the next 8-byte boundary */
  /* items follow here */
} chunk_t;

/* the pool implementation type
 * (use "new_pool" before you use any other functions on a pool variable) */
struct POOL_T
{
  int_t item_size; /* size of the items that are stored in this pool */
  int_t num_items; /* the overall number of items stored in this pool */
  int_t chunk_size; /* the size of new chunks */
  chunk_t *first_chunk; /* first chunk of this pool */
  chunk_t *current_chunk; /* the chunk where items will currently be stored
			   * (note this isn't necessarily the last chunk) */
};

/* functions ================================================================*/

GLOBAL pool_t new_pool (int_t item_size)
/* Create a new pool that records items of size <item_size>. */
{
  pool_t pool = new_mem (sizeof (struct POOL_T));

  pool->item_size = item_size;
  pool->num_items = 0;
  pool->chunk_size = MIN_CHUNK_SIZE / item_size;
  pool->first_chunk = NULL;
  pool->current_chunk = NULL;

  return pool;
}

/*---------------------------------------------------------------------------*/

GLOBAL void clear_pool (pool_t pool)
/* Clear <pool> (do not free any memory used by the pool). */
{
  chunk_t *current_chunk;

  for (current_chunk = pool->first_chunk; 
       current_chunk != NULL; 
       current_chunk = current_chunk->next)
    current_chunk->num_items = 0;

  pool->num_items = 0;
  pool->current_chunk = pool->first_chunk;
}

/*---------------------------------------------------------------------------*/

GLOBAL void *pool_to_vector (pool_t pool)
/* Return <pool> as a C vector (contiguous memory).
 * The vector can be freed with "free" after use. */
{
  void *vector = new_vector (pool->item_size, pool->num_items);
  u_byte_t *chunk; /* position where next chunk is to be copied */
  chunk_t *current_chunk;
  
  /* Collect all chunks in a new vector */
  chunk = (u_byte_t *) vector;
  for (current_chunk = pool->first_chunk; 
       current_chunk != NULL; 
       current_chunk = current_chunk->next) 
    {
      memcpy (chunk, CHUNK_DATA (current_chunk), 
	      pool->item_size * current_chunk->num_items);
      chunk += pool->item_size * current_chunk->num_items;
      if (current_chunk == pool->current_chunk)
	/* Only chunks up to <pool_i>->current_chunk have items in them. */
	break;
    }

  return vector;
}

/*---------------------------------------------------------------------------*/

GLOBAL void write_pool (pool_t pool, FILE *stream, string_t file_name)
/* Write <pool> to <stream> (<file_name> is given for error messages). */
{
  chunk_t *current_chunk;

  /* Collect all chunks to the new vector. */
  for (current_chunk = pool->first_chunk; 
       current_chunk != NULL; 
       current_chunk = current_chunk->next)
  {
    write_vector (CHUNK_DATA (current_chunk), pool->item_size, 
		  current_chunk->num_items, stream, file_name);
    if (current_chunk == pool->current_chunk)
      /* Only chunks up to <pool_i>->current_chunk have items in them. */
      break;
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void *get_pool_space (pool_t pool, int_t num_items, int_t *index)
/* Get space for <num_items> contiguous items in pool <pool>,
 * return its address as the function's result and the index in *<index>,
 * if <index> != NULL. */
{
  void *new_ptr; /* pointer to the pool space */
  chunk_t **chunk_ptr; /* address of a chunk pointer */

  /* Find the first chunk that is big enough. */
  for (chunk_ptr = &pool->current_chunk;
       *chunk_ptr != NULL 
       && (*chunk_ptr)->num_items + num_items > (*chunk_ptr)->chunk_size;
       chunk_ptr = &(*chunk_ptr)->next)
  {
  }

  /* If we haven't found a chunk that fits, allocate a new one. */
  if (*chunk_ptr == NULL)
  {
    if (pool->chunk_size < num_items)
      pool->chunk_size = num_items;
    
    *chunk_ptr = new_mem (sizeof (chunk_t) 
			  + pool->item_size * pool->chunk_size);
    (*chunk_ptr)->chunk_size = pool->chunk_size;
    (*chunk_ptr)->num_items = 0;
    (*chunk_ptr)->next = NULL;
    
    /* If this is the first chunk allocated, save its address. */
    if (pool->first_chunk == NULL)
      pool->first_chunk = *chunk_ptr;
  }
  
  /* Remember address and index of current position in pool. */
  new_ptr = (void *) (CHUNK_DATA (*chunk_ptr) 
		      + pool->item_size * (*chunk_ptr)->num_items);
  if (index != NULL)
    *index = pool->num_items;
  
  /* Adjust indices. */
  (*chunk_ptr)->num_items += num_items;
  pool->num_items += num_items;
  
  pool->current_chunk = *chunk_ptr;

  return new_ptr;
}

/*---------------------------------------------------------------------------*/

GLOBAL int_t pool_items (pool_t pool)
/* Return the number of the items in <pool>. */
{
  return pool->num_items;
}

/*---------------------------------------------------------------------------*/

GLOBAL void *pool_item (pool_t pool, int_t index)
/* Return the address of item with <index> in pool <pool>,
 * or NULL if there is no such item. */
{
  chunk_t *chunk;

  if (index < 0 || index >= pool->num_items)
    return NULL;

  /* Find the right chunk. */
  chunk = pool->first_chunk;
  while (index >= chunk->num_items)
  {
    index -= chunk->num_items;
    chunk = chunk->next;
  }

  /* Return the address of the item. */
  return CHUNK_DATA (chunk) + pool->item_size * index;
}

/*---------------------------------------------------------------------------*/

GLOBAL int_t pool_index (pool_t pool, void *item)
/* Return the index of <item> in <pool>.
 * Report an error if <item> doesn't exist in <pool>. */
{
  int_t index;
  chunk_t *chunk;
  u_byte_t *item_i = (u_byte_t *) item;

  index = 0;
  for (chunk = pool->first_chunk; chunk != NULL; chunk = chunk->next)
  {
    if (item_i >= CHUNK_DATA (chunk) 
	&& item_i < CHUNK_DATA (chunk) + pool->item_size * chunk->num_items)
      return index + (item_i - CHUNK_DATA (chunk)) / pool->item_size;
	
    index += chunk->num_items;
  }
  error ("item doesn't exist in chunk");
}

/*---------------------------------------------------------------------------*/

GLOBAL void *copy_to_pool (pool_t pool, 
			   void *ptr, 
			   int_t num_items, 
			   int_t *index)
/* Copy the vector *<ptr> of length <num_items> into <pool>.
 * The items of *<ptr> must be of same size as the items in <pool>.
 * Return the address of the copy as the function's result and the
 * index in *<index>, if <index> != NULL. */
{
  void *new_ptr = get_pool_space (pool, num_items, index);

  /* Copy the data into the pool. */
  memcpy (new_ptr, ptr, pool->item_size * num_items);

  return new_ptr;
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t copy_string_to_pool (pool_t pool, 
				     string_t string, 
				     int_t *index)
/* Copy the string <string> into <pool>, which must be a string pool.
 * Return the copy of the string as the function's result as well as
 * its index in *<index>, if <index> != NULL. */
{
  return (string_t) (copy_to_pool (pool, (void *) string, 
				   strlen (string) + 1, index));
}

/*---------------------------------------------------------------------------*/

GLOBAL void free_pool (pool_t *pool)
/* Free all memory used by <*pool>. */
{
  chunk_t *chunk, *next_chunk;

  if (*pool == NULL)
    return;

  /* Free all chunks of this pool. */
  for (chunk = (*pool)->first_chunk; chunk != NULL; chunk = next_chunk) 
  {
    next_chunk = chunk->next;
    free_mem (&chunk);
  }

  free_mem (pool);
}

/* end of file ==============================================================*/
