/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module contains the Malaga rule interpreter. */

/* includes =================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "patterns.h"
#include "files.h"
#include "malaga_files.h"
#include "rule_type.h"

#undef GLOBAL
#define GLOBAL
#include "rules.h"

/* constants ================================================================*/

#define BACKUP_STACK_SIZE 50 /* maximum number of backtracking points */

/* types ====================================================================*/

typedef struct SWITCH_T /* used to hold the value for a "switch". */
{ 
  struct SWITCH_T *next; 
  symbol_t key; 
  value_t value; 
} switch_t;

/* variables ================================================================*/

/* the internal state of rule execution */
GLOBAL bool_t executing_rule = FALSE;
GLOBAL int_t pc = -1;
GLOBAL int_t nested_subrules = 0;
GLOBAL int_t executed_rule_number = -1;
GLOBAL rule_sys_t *executed_rule_sys = NULL;
GLOBAL rule_sys_t *debug_rule_sys = NULL;

LOCAL int_t base; /* index of first stack element used in this rule */
LOCAL int_t bottom; /* index of first stack element used in this branch */

LOCAL struct 
{ 
  int_t pc;
  int_t nested_subrules;
  int_t base; 
  int_t bottom;
} backup_stack[BACKUP_STACK_SIZE];

GLOBAL int_t backup_top = 0; /* index of first free item in <backup_stack> */

switch_t *switches; /* the list of switches. */
switch_t *current_switch; /* used for "start_switches" and "get_next_switch" */

/* functions ================================================================*/

GLOBAL void set_switch (symbol_t key, value_t value)
/* Set the switch <key> to <value>. */
{
  switch_t **my_switch;

  my_switch = &switches;
  while (*my_switch != NULL && (*my_switch)->key != key)
    my_switch = &(*my_switch)->next;

  if (*my_switch != NULL)
  {
    free_mem (&(*my_switch)->value);
    (*my_switch)->value = new_value (value);
  }
  else
  {
    (*my_switch) = new_mem (sizeof (switch_t));
    (*my_switch)->key = key;
    (*my_switch)->value = new_value (value);
    (*my_switch)->next = NULL;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL value_t get_switch (symbol_t key)
/* Return the value of the switch <key>. 
 * Report an error if this switch doesn't exist. */
{
  switch_t *my_switch;

  for (my_switch = switches; my_switch != NULL; my_switch = my_switch->next)
  {
    if (my_switch->key == key)
      return my_switch->value;
  }
  error ("switch \"%s\" is not defined", get_symbol_name (key));
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t start_switches (void)
/* Must be called before "get_next_switch" is called.
 * Returns TRUE iff there are any switches */
{
  current_switch = switches;
  return (current_switch != NULL);
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t get_next_switch (symbol_t *key)
/* Return the value of the next switch. Return its key in <*key>.
 * If there is no more switch, return NULL. */
{
  value_t value;

  if (current_switch == NULL)
    return NULL;

  *key = current_switch->key;
  value = current_switch->value;

  current_switch = current_switch->next;
  return value;
}

/*---------------------------------------------------------------------------*/

GLOBAL void free_switches (void)
/* Free all settings. */
{
  switch_t *my_switch, *next_switch;

  for (my_switch = switches; my_switch != NULL; my_switch = next_switch)
  {
    next_switch = my_switch->next;
    free_mem (&my_switch->value);
    free_mem (&my_switch);
  }
  switches = NULL;
}

/* rule execution ===========================================================*/

LOCAL void standard_function (int_t function)
/* STACK EFFECT: <value> -> <new_value>.
 * Perform function <function> on <value> yielding <new_value>. */
{
  switch (function)
  {
  case FUNC_TO_ATOMS:
    push_value (get_atoms (value_to_symbol (value_stack[--top])));
    break;
  case FUNC_TO_MULTI:
    push_symbol_value (find_multi_symbol (value_stack[--top]));
    break;
  case FUNC_TO_SET:
    convert_list_to_set ();
    break;
  case FUNC_IS_CAPITAL:
  {
    string_t string = value_to_string (value_stack[--top]);

    if (TO_LOWER (*string) != *string)
      push_symbol_value (YES_SYMBOL);
    else
      push_symbol_value (NO_SYMBOL);
    break;
  }
  case FUNC_GET_SWITCH:
    push_value (get_switch (value_to_symbol (value_stack[--top])));
    break;
  case FUNC_GET_LENGTH:
    push_number_value (get_list_length (value_stack[--top]));
    break;
  case FUNC_GET_VALUE_TYPE:
    push_symbol_value (get_value_type (value_stack[--top]));
    break;
  case FUNC_GET_VALUE_STRING:
  {
    string_t value_string = value_to_readable (value_stack[--top], TRUE);

    push_string_value (value_string, NULL);
    free_mem (&value_string);
    break;
  }
  case FUNC_TRANSMIT:
    if (transmit == NULL)
      error ("no transmit function available");
    transmit ();
  case FUNC_FLOOR:
    push_number_value (floor (value_to_double (value_stack[--top])));
    break;
  default:
    error ("internal (unknown standard function)");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void execute_rule (rule_sys_t *rule_sys, int_t rule_number)
/* Execute rule <rule_number> in the rule system <rule_sys>.
 * Any parameters must be on the value stack. */
{
  static symbol_t nil = NIL_SYMBOL; /* the "nil" symbol */
  bool_t terminate; /* shall we terminate the current rule internal path? */
  int_t i;

  /* Initialise the value stack. */
  top = rule_sys->rules[rule_number].num_params;
  base = bottom = 0;
  backup_top = 0;
  nested_subrules = 0;

  /* See if the rule exists at all. */
  DB_ASSERT (rule_number < rule_sys->rules_size);

  /* Copy for debug purposes and error messages. */
  executed_rule_sys = rule_sys;
  executed_rule_number = rule_number;

  pc = rule_sys->rules[rule_number].first_instr;
  executing_rule = TRUE;

  rule_successful = FALSE;
  terminate = FALSE;
  while (! terminate) 
  {
    instr_t instruction;
    int_t info;

    if (debug_rule_sys == rule_sys)
      debug_rule ();
      
    instruction = rule_sys->instrs[pc];
    pc++;
    info = INSTR_INFO (instruction);
    switch (OPCODE (instruction)) 
    {
    case INS_ERROR:
      switch (info)
      {
      case ASSERTION_ERROR:
	error ("assertion failed");
      case NO_RETURN_ERROR:
	error ("missing return statement");
      default:
	error ("%s", rule_sys->strings + info);
      }

    case INS_TERMINATE:
      terminate = TRUE;
      break;

    case INS_NOP:
      break;

    case INS_TERMINATE_IF_NULL:
      DB_ASSERT (top >= base + 1);
      if (value_stack[--top] == NULL)
	terminate = TRUE;
      break;

    case INS_ADD_END_STATE:
      DB_ASSERT (top >= base + 1);
      add_end_state (value_stack[top-1]);
      top--;
      rule_successful = TRUE;
      break;

    case INS_ADD_STATE:
      DB_ASSERT (top >= base + 1);
      add_running_state (value_stack[top-1], info);
      top--;
      rule_successful = TRUE;
      break;

    case INS_ADD_ALLO:
      DB_ASSERT (top >= base + 2);
      add_allo (value_stack[top-2], value_stack[top-1]);
      top -= 2;
      rule_successful = TRUE;
      break;

    case INS_ACCEPT:
      DB_ASSERT (top >= base + 1);
      backup_top = 0;
      terminate = TRUE;
      break;

    case INS_PUSH_NULL:
      for (i = 0; i < info; i++)
	push_value (NULL);
      break;

    case INS_PUSH_VAR:
      DB_ASSERT (base + info < top);
      push_value (value_stack[base + info]);
      break;

    case INS_PUSH_CONST:
      DB_ASSERT (info < rule_sys->values_size);
      push_value (rule_sys->values + info);
      break;

    case INS_PUSH_SYMBOL:
      push_symbol_value (info);
      break;

    case INS_PUSH_PATTERN_VAR:
      DB_ASSERT (info < PATTERN_VAR_MAX);
      push_string_value (pattern_var[info], NULL);
      break;

    case INS_POP:
      DB_ASSERT (top >= base + info);
      top -= info;
      break;

    case INS_BUILD_LIST:
      DB_ASSERT (top >= base + info);
      build_list (info);
      break;

    case INS_BUILD_RECORD:
      DB_ASSERT (top >= base + 2*info);
      build_record (info);
      break;

    case INS_BUILD_PATH:
      DB_ASSERT (top >= base + info);
      build_path (info);
      break;

    case INS_DOT_OPERATION:
      DB_ASSERT (top >= base + 2);
      dot_operation ();
      if (value_stack[top-1] == NULL)
	value_stack[top-1] = &nil;
      break;
      
    case INS_PLUS_OPERATION:
      DB_ASSERT (top >= base + 2);
      plus_operation ();
      break;
      
    case INS_MINUS_OPERATION:
      DB_ASSERT (top >= base + 2);
      minus_operation ();
      break;

    case INS_ASTERISK_OPERATION:
      DB_ASSERT (top >= base + 2);
      asterisk_operation ();
      break;

    case INS_SLASH_OPERATION:
      DB_ASSERT (top >= base + 2);
      slash_operation ();
      break;

    case INS_UNARY_MINUS_OP:
      DB_ASSERT (top >= base + 1);
      unary_minus_operation ();
      break;

    case INS_GET_ATTRIBUTE:
      DB_ASSERT (top >= base + 1);
      value_stack[top-1] = get_attribute (value_stack[top-1], (symbol_t) info);
      if (value_stack[top-1] == NULL)
	value_stack[top-1] = &nil;
      break;

    case INS_REMOVE_ATTRIBUTE:
      DB_ASSERT (top >= base + 1);
      remove_attribute ((symbol_t) info);
      break;

    case INS_STD_FUNCTION:
      DB_ASSERT (top >= base + 1);
      standard_function (info);
      break;

    case INS_MATCH:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (info < rule_sys->strings_size);
      if (match_pattern (value_to_string (value_stack[--top]), 
			 rule_sys->strings + info))
	push_symbol_value (YES_SYMBOL);
      else
	push_symbol_value (NO_SYMBOL);
      break;

    case INS_SET_VAR:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (base + info < top);
      value_stack[base + info] = value_stack[--top];
      break;

    case INS_PLUS_VAR:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (base + info < top);
      insert_value (1, value_stack[base + info]);
      plus_operation ();
      value_stack[base + info] = value_stack[--top];
      break;

    case INS_MINUS_VAR:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (base + info < top);
      insert_value (1, value_stack[base + info]);
      minus_operation ();
      value_stack[base + info] = value_stack[--top];
      break;

    case INS_ASTERISK_VAR:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (base + info < top);
      insert_value (1, value_stack[base + info]);
      asterisk_operation ();
      value_stack[base + info] = value_stack[--top];
      break;

    case INS_SLASH_VAR:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (base + info < top);
      insert_value (1, value_stack[base + info]);
      slash_operation ();
      value_stack[base + info] = value_stack[--top];
      break;

    case INS_SET_VAR_PATH:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (base + info < top);
      insert_value (2, value_stack[base + info]);
      modify_value_part (right_value);
      value_stack[base + info] = value_stack[--top];
      break;

    case INS_PLUS_VAR_PATH:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (base + info < top);
      insert_value (2, value_stack[base + info]);
      modify_value_part (plus_operation);
      value_stack[base + info] = value_stack[--top];
      break;

    case INS_MINUS_VAR_PATH:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (base + info < top);
      insert_value (2, value_stack[base + info]);
      modify_value_part (minus_operation);
      value_stack[base + info] = value_stack[--top];
      break;

    case INS_ASTERISK_VAR_PATH:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (base + info < top);
      insert_value (2, value_stack[base + info]);
      modify_value_part (asterisk_operation);
      value_stack[base + info] = value_stack[--top];
      break;

    case INS_SLASH_VAR_PATH:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (base + info < top);
      insert_value (2, value_stack[base + info]);
      modify_value_part (slash_operation);
      value_stack[base + info] = value_stack[--top];
      break;

    case INS_GET_1ST_ELEMENT:
      DB_ASSERT (top >= base + 1);
      get_first_element ();
      break;

    case INS_ITERATE:
      DB_ASSERT (info >= 1 && base + info < top);
      get_next_element (base + info);
      break;

    case INS_JUMP:
      DB_ASSERT (info < rule_sys->instrs_size);
      pc = info;
      break;

    case INS_JUMP_IF_EQUAL:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (values_equal (value_stack[top-2], value_stack[top-1]))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_NOT_EQUAL:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (! values_equal (value_stack[top-2], value_stack[top-1]))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_CONGR:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (values_congruent (value_stack[top-2], value_stack[top-1]))
	pc = info;
      top -= 2;
      break;
	  
    case INS_JUMP_IF_NOT_CONGR:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (! values_congruent (value_stack[top-2], value_stack[top-1]))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_IN:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (value_in_value (value_stack[top-2], value_stack[top-1]))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_NOT_IN:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (! value_in_value (value_stack[top-2], value_stack[top-1]))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_LESS:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (value_to_double (value_stack[top-2])
	  < value_to_double (value_stack[top-1]))
	pc = info;
      top -= 2;
      break;
	  
    case INS_JUMP_IF_NOT_LESS:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (! (value_to_double (value_stack[top-2])
	     < value_to_double (value_stack[top-1])))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_GREATER:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (value_to_double (value_stack[top-2])
	  > value_to_double (value_stack[top-1]))
	pc = info;
      top -= 2;
      break;
	  
    case INS_JUMP_IF_NOT_GREATER:
      DB_ASSERT (top >= base + 2);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (! (value_to_double (value_stack[top-2])
	     > value_to_double (value_stack[top-1])))
	pc = info;
      top -= 2;
      break;

    case INS_JUMP_IF_NULL:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (value_stack[--top] == NULL)
	pc = info;
      break;

    case INS_JUMP_IF_NOT_NULL:
      DB_ASSERT (top >= base + 1);
      DB_ASSERT (info < rule_sys->instrs_size);
      if (value_stack[--top] != NULL)
	pc = info;
      break;

    case INS_JUMP_IF_YES:
    {
      symbol_t symbol;

      DB_ASSERT (top >= base + 1);
      DB_ASSERT (info < rule_sys->instrs_size);
	  
      symbol = value_to_symbol (value_stack[--top]);
      if (symbol != YES_SYMBOL && symbol != NO_SYMBOL)
	error ("boolean value expected");
      if (symbol == YES_SYMBOL)
	pc = info;
      break;
    }

    case INS_JUMP_IF_NO:
    {
      symbol_t symbol;

      DB_ASSERT (top >= base + 1);
      DB_ASSERT (info < rule_sys->instrs_size);
	  
      symbol = value_to_symbol (value_stack[--top]);
      if (symbol != YES_SYMBOL && symbol != NO_SYMBOL)
	error ("boolean value expected");
      if (symbol == NO_SYMBOL)
	pc = info;
      break;
    }
	  
    case INS_JUMP_NOW:
    {
      int_t old_top = top;

      DB_ASSERT (info < rule_sys->instrs_size);
      if (backup_top >= BACKUP_STACK_SIZE)
	error ("too many branches in a rule");

      backup_stack[backup_top].pc = pc;
      backup_stack[backup_top].nested_subrules = nested_subrules;
      backup_stack[backup_top].base = base;
      backup_stack[backup_top].bottom = bottom;
      while (bottom < old_top)
	push_value (value_stack[bottom++]);
      base += (top - old_top);
      backup_top++;
      pc = info;
      break;
    }

    case INS_JUMP_LATER:
    {
      int_t old_top = top;

      DB_ASSERT (info < rule_sys->instrs_size);
      if (backup_top >= BACKUP_STACK_SIZE)
	error ("too many branches in a rule");
 
      backup_stack[backup_top].pc = info;
      backup_stack[backup_top].nested_subrules = nested_subrules;
      backup_stack[backup_top].base = base;
      backup_stack[backup_top].bottom = bottom;
      while (bottom < old_top)
	push_value (value_stack[bottom++]);
      base += (top - old_top);
      backup_top++;
      break;
    }

    case INS_JUMP_SUBRULE:
      push_number_value (base - bottom);
      push_number_value (pc);
      base = top;
      pc = rule_sys->rules[info].first_instr;
      nested_subrules++;
      break;

    case INS_RETURN:
    {
      int_t old_base;

      DB_ASSERT (base - bottom >= 2 + info);
      old_base = bottom + value_to_int (value_stack[base - 2]);
      pc = value_to_int (value_stack[base - 1]);
      value_stack[base - info - 2] = value_stack[top-1]; /* Copy result. */
      top = base - (info + 1);
      base = old_base;
      nested_subrules--;
      break;
    }

    default:
      error ("internal (unknown instruction %d)", OPCODE (instruction));
      break;
    }

    if (terminate && backup_top > 0) 
    { /* Load a previously saved rule-internal state and continue. */
      backup_top--;
      top = bottom;
      base = backup_stack[backup_top].base;
      bottom = backup_stack[backup_top].bottom;
      pc = backup_stack[backup_top].pc;
      nested_subrules = backup_stack[backup_top].nested_subrules;
      terminate = FALSE;
    }
  }

  executing_rule = FALSE;
  pc = -1;
  executed_rule_number = -1;
  executed_rule_sys = NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL rule_sys_t *read_rule_sys (string_t file_name)
/* Read rule system from file <file_name>.
 * A symbol file must have already been loaded. */
{
  FILE *stream;
  rule_header_t header;
  rule_sys_t *rule_sys = new_mem (sizeof (rule_sys_t));

  stream = open_stream (file_name, "rb");

  read_vector (&header, sizeof (header), 1, stream, file_name);

  check_header (&header.common_header, file_name, 
		RULE_FILE, RULE_CODE_VERSION);

  rule_sys->initial_rule_set = header.initial_rule_set;
  rule_sys->initial_cat = header.initial_cat;
  rule_sys->robust_rule = header.robust_rule;
  rule_sys->allo_rule = header.allo_rule;
  rule_sys->pruning_rule = header.pruning_rule;
  rule_sys->input_filter = header.input_filter;
  rule_sys->output_filter = header.output_filter;
  rule_sys->rules_size = header.rules_size;
  rule_sys->rules = read_new_vector (sizeof (rule_t), header.rules_size, 
				     stream, file_name);
  rule_sys->rule_sets_size = header.rule_sets_size;
  rule_sys->rule_sets = read_new_vector (sizeof (int_t), header.rule_sets_size,
					 stream, file_name);
  rule_sys->instrs_size = header.instrs_size;
  rule_sys->instrs = read_new_vector (sizeof (instr_t), header.instrs_size, 
				      stream, file_name);
  rule_sys->values_size = header.values_size;
  rule_sys->values = read_new_vector (sizeof (cell_t), header.values_size, 
				      stream, file_name);
  rule_sys->src_lines_size = header.src_lines_size;
  rule_sys->src_lines = read_new_vector (sizeof (src_line_t), 
					 header.src_lines_size, 
					 stream, file_name);
  rule_sys->vars_size = header.vars_size;
  rule_sys->vars = read_new_vector (sizeof (var_t), header.vars_size,
				    stream, file_name);
  rule_sys->var_scopes_size = header.var_scopes_size;
  rule_sys->var_scopes = read_new_vector (sizeof (var_scope_t), 
					  header.var_scopes_size,
					  stream, file_name);
  rule_sys->strings_size = header.strings_size;
  rule_sys->strings = read_new_vector (sizeof (char), header.strings_size, 
				       stream, file_name);

  close_stream (&stream, file_name);
  
  return rule_sys;
}

/*---------------------------------------------------------------------------*/

GLOBAL void free_rule_sys (rule_sys_t **rule_sys)
/* Free all memory used by <*rule_sys>. */
{ 
  if (*rule_sys != NULL)
  {
    free_mem (&(*rule_sys)->rules);
    free_mem (&(*rule_sys)->rule_sets);
    free_mem (&(*rule_sys)->instrs);
    free_mem (&(*rule_sys)->values);
    free_mem (&(*rule_sys)->src_lines);
    free_mem (&(*rule_sys)->vars);
    free_mem (&(*rule_sys)->var_scopes);
    free_mem (&(*rule_sys)->strings);
    free_mem (rule_sys);
  }
}

/* debug support functions ==================================================*/

GLOBAL value_t inspect_stack (int_t index)
/* Return S[index] or NULL if it is not defined. */
{
  if (base + index >= top)
    return NULL;
  else
    return value_stack[base + index];
}

/*---------------------------------------------------------------------------*/

GLOBAL void get_base_and_pc_indexes (int_t index, 
				     int_t *base_index, 
				     int_t *pc_index)
/* If <index> == 0, return the current <pc_index> and the <base_index> of the
 * stack when this rule was called as a subrule. */
{
  if (index == 0)
  {
    *pc_index = pc;
    *base_index = base - bottom;
  }
  else
  {
    *pc_index = value_to_int (value_stack[bottom + index - 1]);
    *base_index = value_to_int (value_stack[bottom + index - 2]);
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL int_t inspect_stack_pointer (void)
/* Return value of rule stack pointer. */
{
  return top - base;
}

/*---------------------------------------------------------------------------*/

GLOBAL int_t first_variable_index (rule_sys_t *rule_sys,
				   int_t instr_index)
/* Return the stack index of the first variable that is visible
 * when pc is at <instr_index> in <rule_sys>. */
{
  rule_t *rule;
  int_t i, first_instr;

  /* Find the rule/subrule we're in. */
  rule = NULL;
  first_instr = -1;
  for (i = 0; i < rule_sys->rules_size; i++)
  {
    rule_t *rule2 = rule_sys->rules + i;

    if (rule2->first_instr <= instr_index 
	&& rule2->first_instr > first_instr)
    {
      rule = rule2;
      first_instr = rule->first_instr;
    }
  }

  DB_ASSERT (rule != NULL);

  if (rule->type == SUBRULE)
    return - (2 + rule->num_params);
  else
    return 0;
}

/*---------------------------------------------------------------------------*/

GLOBAL void source_of_instr (rule_sys_t *rule_sys,
			     int_t instr_index, 
			     int_t *first_line, 
			     int_t *last_line, 
			     string_t *file_name,
			     string_t *rule_name)
/* Set *<first_line>, *<last_line> and *<file_name> to appropriate values
 * for the statement that has generated the instruction at <instr_index>. */
{
  int_t lower, upper, middle;
  src_line_t *src_line;

  if (rule_sys->src_lines_size == 0 
      || rule_sys->src_lines[0].instr > instr_index)
  {
    if (first_line != NULL)
      *first_line = -1;
    if (last_line != NULL)
      *last_line = -1;
    if (file_name != NULL)
      *file_name = NULL;
    if (rule_name != NULL)
      *rule_name = NULL;
    return;
  }

  /* Find the last src_line entry with <instr> <= <instr_index>. */
  lower = 0;
  upper = rule_sys->src_lines_size - 1;
  while (lower < upper) 
  {
    middle = (lower + upper + 1) / 2;
    src_line = rule_sys->src_lines + middle;

    if (src_line->instr <= instr_index)
      lower = middle;
    else
      upper = middle - 1;
  }
  
  src_line = rule_sys->src_lines + lower;

  if (first_line != NULL)
    *first_line = src_line->line;

  if (file_name != NULL)
  {
    if (src_line->file != -1)
      *file_name = rule_sys->strings + src_line->file;
    else
      *file_name = NULL;
  }

  /* Find the last line of the statement. */
  if (last_line != NULL) 
  {
    do
    {
      src_line++;
    } while (src_line < rule_sys->src_lines + rule_sys->src_lines_size
	     && src_line->line == -1);

    if (src_line < rule_sys->src_lines + rule_sys->src_lines_size)
      *last_line = src_line->line - 1;
    else
    *last_line = -1;
  }

  /* Find the rule of the statement */
  if (rule_name != NULL)
  {
    int_t rule_number;
    int_t i;
    int_t first_instr;
    
    rule_number = 0;
    first_instr = -1;
    for (i = 0; i < rule_sys->rules_size; i++)
    {
      rule_t *rule = rule_sys->rules + i;

      if (rule->first_instr <= instr_index && rule->first_instr > first_instr)
      {
	rule_number = i;
	first_instr = rule->first_instr;
      }
    }
    *rule_name = rule_sys->strings + rule_sys->rules[rule_number].name;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL int_t local_variable_index (rule_sys_t *rule_sys, 
				  var_t *var, 
				  int_t instr_index)
/* Return the stack index of variable <var> at <instr_index>.
 * Return -1 if it is not currently defined. */
{
  int_t lower, upper, middle;
  var_scope_t *var_scope;

  /* Find last scope whose <first_instr> is not higher than <instr_index>. */ 
  lower = var->first_scope;
  upper = lower + var->number_of_scopes - 1;

  while (lower < upper) 
  {
    middle = (lower + upper + 1) / 2;
    var_scope = rule_sys->var_scopes + middle;

    if (var_scope->first_instr <= instr_index)
      lower = middle;
    else
      upper = middle - 1;
  }

  /* <lower> is the index of the highest line
   * with an instruction index not more than <instr_index>. */
  if (lower == upper) 
  {
    /* We found a scope. */
    var_scope = rule_sys->var_scopes + lower;
    if (instr_index >= var_scope->first_instr 
	&& instr_index <= var_scope->last_instr)
      /* Variable is defined. */
      return var_scope->stack_index;
  }
  
  return -1;
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t variable_at_index (rule_sys_t *rule_sys,
				   int_t stack_index, 
				   int_t instr_index)
/* Return the name of the variable that is defined at <stack_index>
 * when instruction <instr_index> is executed or NULL if there is none. */
{
  int_t i;

  /* There is never a variable at stack index -2 or -1. */
  if (stack_index == -2 || stack_index == -1)
    return NULL;

  /* For each variable name, test if it is the right one. */
  for (i = 0; i < rule_sys->vars_size; i++) 
  {
    var_t *var = rule_sys->vars + i;

    if (stack_index == local_variable_index (rule_sys, var, instr_index))
      return rule_sys->strings + var->name;
  }

  return NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL int_t variable_index (rule_sys_t *rule_sys,
			     string_t var_name, 
			     int_t instr_index)
/* Return the stack index of variable <var_name> at <instr_index>. */
{
  /* Search for the right variable name (binary search). */
  int_t lower = 0;
  int_t upper = rule_sys->vars_size - 1;

  while (lower <= upper) 
  {
    int_t middle = (lower + upper) / 2;
    var_t *var = rule_sys->vars + middle;
    int_t result = strcmp_no_case (var_name, rule_sys->strings + var->name);

    if (result < 0)
      upper = middle - 1;
    else if (result > 0)
      lower = middle + 1;
    else
      return local_variable_index (rule_sys, var, instr_index);
  }
  return -1;
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t rule_set_readable (rule_sys_t *rule_sys, int_t rule_set)
/* Return <rule_set> in <rule_sys> as a readable string.
 * The string must be freed with "free" after use. */
{
  text_t text = new_text ();

  if (rule_set == -1)
    add_to_text (text, "(end state)");
  else
  { 
    bool_t name_has_been_printed;
    int_t *rule;
      
    add_to_text (text, "rules ");

    rule = rule_sys->rule_sets + rule_set;
    while (TRUE)
    {
      name_has_been_printed = FALSE;
      while (*rule >= 0)
      {
	if (name_has_been_printed)
	  add_to_text (text, ", ");
	else
	  name_has_been_printed = TRUE;
	
	add_to_text (text, rule_sys->strings + rule_sys->rules[*rule++].name);
      }
      
      if (*rule == -1)
	break;

      add_to_text (text, " else ");
      rule++;
    }
  }

  return text_to_string (&text);
}

/* end of file ==============================================================*/
