/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module contains the Malaga rule interpreter. */

/* types ====================================================================*/

typedef struct
/* contains a rule system of a compiled rule file.
 * A "..._size" variable contains the number of elements
 * in the following table. */
{
  int_t initial_rule_set; /* rules index of the initial combi_rule set */
  int_t initial_cat; /* values index of initial combi_rule category */

  int_t robust_rule; /* number of robust_rule or -1 */
  int_t pruning_rule; /* number of pruning_rule or -1 */
  int_t allo_rule; /* number of allo_rule or -1 */
  int_t input_filter; /* number of input filter rule or -1 */
  int_t output_filter; /* number of output filter rule or -1 */

  rule_t *rules; /* name and code of every rule */
  int_t rules_size;

  int_t *rule_sets; /* a collection of lists; each list is a series of rules,
		     * followed by -1. A list may be subdivided into sublists,
		     * which are separated by -2. The rules of a sublist are
		     * executed if none of the rules of the preceding sublist
		     * has been successful. */
  int_t rule_sets_size;

  instr_t *instrs; /* the actual rule instructions */
  int_t instrs_size;

  cell_t *values; /* all constant Malaga values */
  int_t values_size;

  char *strings; /* names of files, variables, rules, patterns */
  int_t strings_size;
  
  src_line_t *src_lines; /* correspondence between source lines
			  * and rule instructions */
  int_t src_lines_size;

  var_t *vars; /* variable names */
  int_t vars_size;

  var_scope_t *var_scopes; /* variable scopes */
  int_t var_scopes_size;
} rule_sys_t;

/* variables ================================================================*/

GLOBAL void (*add_end_state) (value_t cat);
/* Add a state, consisting of <cat>, as an end state.
 * This is a callback function called from "execute_rule". */

GLOBAL void (*add_running_state) (value_t cat, int_t rule_set);
/* Add a running state, consisting of <cat> and <rule_set>.
 * This is a callback function called from "execute_rule". */

GLOBAL void (*add_allo) (value_t surf, value_t cat);
/* Add an allomorph, consisting of <surf> and <cat>, to the lexicon.
 * This is a callback function called from "execute_rule". */

GLOBAL void (*debug_rule) (void);
/* Called from "execute_rule" before instruction at <pc> is executed.
 * This is a callback function called from "execute_rule". */

GLOBAL void (*transmit) (void);
/* STACK EFFECTS: <value> -> <new_value>.
 * Called when "FUNC_TRANSMIT" is executed.
 * This is a callback function called from "execute_rule". */

/* These values are used by "execute_rule".
 * but they are global to support debuggers and error messages. */
extern bool_t executing_rule;
extern int_t pc; /* READ ONLY! */
extern int_t nested_subrules; /* READ ONLY! */
extern int_t executed_rule_number; /* READ ONLY! */
extern rule_sys_t *executed_rule_sys; /* READ ONLY! */
extern int_t backup_top; /* READ ONLY! */

extern rule_sys_t *debug_rule_sys; /* rule_sys to debug or NULL */

GLOBAL bool_t rule_successful; /* READ ONLY! */
/* indicator for execution of result, accept, or allo statement */

/* functions ================================================================*/

extern void execute_rule (rule_sys_t *rule_sys, 
			  int_t rule_number);
/* Execute rule <rule_number> in the rule system <rule_sys>.
 * Any parameters must be on the value stack. */

extern rule_sys_t *read_rule_sys (string_t file_name);
/* Read rule system from file <file_name>.
 * A symbol file must have already been loaded. */

extern void free_rule_sys (rule_sys_t **rule_sys);
/* Free all memory used by <*rule_sys>. */

/* debug support functions ==================================================*/

extern value_t inspect_stack (int_t index);
/* Return S[index] or NULL if it is not defined. */

extern int_t inspect_stack_pointer (void);
/* Return value of rule stack pointer. */

extern int_t first_variable_index (rule_sys_t *rule_sys,
				   int_t instr_index);
/* Return the stack index of the first variable that is visible
 * when pc is at <instr_index> in <rule_sys>. */

extern void get_base_and_pc_indexes (int_t index, 
				     int_t *base_index, 
				     int_t *pc_index);
/* If <index> == 0, return the current <pc_index> and the <base_index> of the
 * stack when this rule was called as a subrule. */

extern void source_of_instr (rule_sys_t *rule_sys, 
			     int_t instr_index, 
			     int_t *first_line, 
			     int_t *last_line, 
			     string_t *file_name,
			     string_t *rule_name);
/* Set *<first_line>, *<last_line> and *<file_name> to appropriate values
 * for the statement that has generated the instruction at <instr_index>. */

extern int_t rule_of_instr (rule_sys_t *rule_sys,
			    int_t instr);
/* Return the rule index in <rule_sys> where <instr> is located. */

extern int_t variable_index (rule_sys_t *rule_sys, 
			     string_t var_name, 
			     int_t instr_index);
/* Return the stack index of variable <var_name> at <instr_index>. */

extern string_t variable_at_index (rule_sys_t *rule_sys, 
				   int_t stack_index, 
				   int_t instr_index);
/* Return the name of the variable that is defined at <stack_index>
 * when instruction <instr_index> is executed or NULL if there is none. */

extern string_t rule_set_readable (rule_sys_t *rule_sys, int_t rule_set);
/* Return <rule_set> in <rule_sys> as a readable string.
 * The string must be freed with "free" after use. */

/* switch support ===========================================================*/

extern void set_switch (symbol_t key, value_t value);
/* Set the switch <key> to <value>. */

extern bool_t start_switches (void);
/* Must be called before "get_next_switch" is called.
 * Returns TRUE iff there are any switches */

extern value_t get_next_switch (symbol_t *key);
/* Return the value of the next switch. Return its key in <*key>.
 * If there is no more switch, return NULL. */

extern void free_switches (void);
/* Free all settings. */

/* end of file ==============================================================*/
