/*
 * puzzle1.c
 *
 *
 * Part of PUZZLE 4.0.2 (April 1999)
 *
 * (c) 1995-1999 by Korbinian Strimmer and Arndt von Haeseler
 *
 * All parts of the source except where indicated are distributed under
 * the GNU public licence.  See http://www.opensource.org for details.
 */


#define EXTERN

#include "puzzle.h"
#include "gamma.h"



/******************************************************************************/
/* options - file I/O - output                                                */
/******************************************************************************/

/* compute TN parameters according to F84 Ts/Tv ratio */
void makeF84model()
{
	double rho, piA, piC, piG, piT, piR, piY, ts, yr;
	
	piA = Freqtpm[0];
	piC = Freqtpm[1];
	piG = Freqtpm[2];
	piT = Freqtpm[3];
	piR = piA + piG;
	piY = piC + piT;
	if (piC*piT*piR + piA*piG*piY == 0.0) {		
		printf("\n\n\nF84 model not possible ");
		printf("(bad combination of base frequencies)\n");
		tstvf84 = 0.0;
		return;
	}
	rho = (piR*piY*(piR*piY*tstvf84 - (piA*piG + piC*piT)))/
		(piC*piT*piR + piA*piG*piY);
	
	if(piR == 0.0 || piY == 0.0 || (piR + rho) == 0.0) {
		printf("\n\n\nF84 model not possible ");
		printf("(bad combination of base frequencies)\n");
		tstvf84 = 0.0;
		return;
	}
	ts = 0.5 + 0.25*rho*(1.0/piR + 1.0/piY);
	yr = (piY + rho)/piY * piR/(piR + rho);
	if (ts < MINTS || ts > MAXTS) {
		printf("\n\n\nF84 model not possible ");
		printf("(bad Ts/Tv parameter)\n");
		tstvf84 = 0.0;
		return;
	}
	if (yr < MINYR || yr > MAXYR) {
		printf("\n\n\nF84 model not possible ");
		printf("(bad Y/R transition parameter)\n");
		tstvf84 = 0.0;
		return;
	}
	TSparam = ts;
	YRparam = yr;
	optim_optn = FALSE;
}

/* compute number of quartets used in LM analysis */
void compnumqts()
{
	if (lmqts == 0) {
		if (numclust == 4)
			Numquartets = (uli) clustA*clustB*clustC*clustD;
		if (numclust == 3)
			Numquartets = (uli) clustA*clustB*clustC*(clustC-1)/2;
		if (numclust == 2)
			Numquartets = (uli) clustA*(clustA-1)/2 * clustB*(clustB-1)/2;
		if (numclust == 1)	
			Numquartets = (uli) Maxspc*(Maxspc-1)*(Maxspc-2)*(Maxspc-3)/24;
	} else {
		Numquartets = lmqts;
	}
}

/* set options interactively */
void setoptions()
{	
	int i, valid;
	double sumfreq;
	char ch;

	/* defaults */
	rhetmode = 0;          /* assume rate homogeneity               */
	numcats = 1;
	Geta = 0.05;
	grate_optim = FALSE;
	fracinv = 0.0;
	fracinv_optim = FALSE;

	compclock = FALSE;     /* compute clocklike branch lengths      */
	locroot = -1;          /* search for optimal place of root      */ 
	qcalg_optn = FALSE;    /* don't use sampling of quartets        */
	approxp_optn = TRUE;   /* approximate parameter estimates       */
	listqptrees = FALSE;   /* list puzzling step trees              */
	
	/* approximate QP quartets? */
	if (Maxspc <= 6) approxqp = FALSE;
	else approxqp = TRUE;
	
	codon_optn = 0;        /* use all positions in a codon          */
	
	/* number of puzzling steps */
	if (Maxspc <= 25) Numtrial = 1000;
	else if (Maxspc <= 50) Numtrial = 10000;
	else if (Maxspc <= 75) Numtrial = 25000;
	else Numtrial = 50000;
	    
	utree_optn = TRUE;     /* use first user tree for estimation    */
	outgroup = 0;          /* use first taxon as outgroup           */
	sym_optn = FALSE;      /* symmetrize doublet frequencies        */
	tstvf84 = 0.0;         /* disable F84 model                     */
	show_optn = FALSE;     /* show unresolved quartets              */
	typ_optn = 0;          /* tree reconstruction                   */
	numclust = 1;          /* one clusters in LM analysis           */
	lmqts = 0;             /* all quartets in LM analysis           */
	compnumqts();
	if (Numquartets > 10000) {
		lmqts = 10000;     /* 10000 quartets in LM analysis         */
		compnumqts();
	}
	
 	do {
		printf("\n\n\nGENERAL OPTIONS\n");
		printf(" b                     Type of analysis?  ");
		if (typ_optn == 0) printf("Tree reconstruction\n");
		if (typ_optn == 1) printf("Likelihood mapping\n");
		if (typ_optn == 0) {
			printf(" k                Tree search procedure?  ");
			if (puzzlemode == 0) printf("Quartet puzzling\n");
			if (puzzlemode == 1) printf("User defined trees\n");
			if (puzzlemode == 2) printf("Pairwise distances only (no tree)\n");
			if (puzzlemode == 0) {
				printf(" v       Approximate quartet likelihood?  %s\n",
					(approxqp ? "Yes" : "No"));
				printf(" u             List unresolved quartets?  %s\n",
					(show_optn ? "Yes" : "No"));
				printf(" n             Number of puzzling steps?  %lu\n",
						Numtrial);
				printf(" j             List puzzling step trees?  %s\n",
					(listqptrees ? "Yes" : "No"));						
				printf(" o                  Display as outgroup?  ");
				fputid(stdout, outgroup);
				printf("\n");
			}
			if (puzzlemode == 0 || puzzlemode == 1) {
				printf(" z     Compute clocklike branch lengths?  ");
				if (compclock) printf("Yes\n");
				else printf("No\n");
			}
			if (compclock)
				if (puzzlemode == 0 || puzzlemode == 1) {
					printf(" l                     Location of root?  ");
					if (locroot < 0) printf("Best place (automatic search)\n");
					else if (locroot < Maxspc) {
						printf("Branch %d (", locroot + 1);
						fputid(stdout, locroot);
						printf(")\n");
					} else printf("Branch %d (internal branch)\n", locroot + 1);
				}
		}
		if (typ_optn == 1) {
			  printf(" g          Group sequences in clusters?  ");
			  if (numclust == 1) printf("No\n");
			  else printf("Yes (%d clusters as specified)\n", numclust);
			  printf(" n                   Number of quartets?  ");
			  if (lmqts == 0) printf("%lu (all possible)\n", Numquartets);
			  else printf("%lu (random choice)\n", lmqts);
		}
		printf(" e                  Parameter estimates?  ");
		if (approxp_optn) printf("Approximate (faster)\n");
		else printf("Exact (slow)\n");
		if (!(puzzlemode == 1 && typ_optn == 0)) {	
			printf(" x            Parameter estimation uses?  ");
			if (qcalg_optn) printf("Quartet sampling + NJ tree\n");
			else printf("Neighbor-joining tree\n");
			
		} else {
			printf(" x            Parameter estimation uses?  ");
			if (utree_optn)
				printf("1st input tree\n");
			else if (qcalg_optn) printf("Quartet sampling + NJ tree\n");
			else printf("Neighbor-joining tree\n");
		}
		printf("SUBSTITUTION PROCESS\n");
		printf(" d          Type of sequence input data?  ");
		if (data_optn == 0) printf("Nucleotides\n");
		if (data_optn == 1) printf("Amino acids\n");
		if (data_optn == 2) printf("Binary states\n");
		if (data_optn == 0 && (Maxseqc % 3) == 0  && !SH_optn) {
			printf(" h             Codon positions selected?  ");
			if (codon_optn == 0) printf("Use all positions\n");
			if (codon_optn == 1) printf("Use only 1st positions\n");
			if (codon_optn == 2) printf("Use only 2nd positions\n");
			if (codon_optn == 3) printf("Use only 3rd positions\n");
			if (codon_optn == 4) printf("Use 1st and 2nd positions\n");
		}
		printf(" m                Model of substitution?  ");
		if (data_optn == 0) { /* nucleotides */
			if (nuc_optn) {
				if(HKY_optn)
					printf("HKY (Hasegawa et al. 1985)\n");	
				else {
					printf("TN (Tamura-Nei 1993)\n");
					printf(" p      Constrain TN model to F84 model?  ");
					if (tstvf84 == 0.0)
						printf("No\n");
					else printf("Yes (Ts/Tv ratio = %.2f)\n", tstvf84);
				}
				printf(" t    Transition/transversion parameter?  ");
				if (optim_optn)
					printf("Estimate from data set\n");
				else
					printf("%.2f\n", TSparam);	
				if (TN_optn) {
					printf(" r             Y/R transition parameter?  ");
					if (optim_optn)
						printf("Estimate from data set\n");
					else
						printf("%.2f\n", YRparam);		
 				}			
			}
			if (SH_optn) {
					printf("SH (Schoeniger-von Haeseler 1994)\n");
					printf(" t    Transition/transversion parameter?  ");
					if (optim_optn)
						printf("Estimate from data set\n");
					else
						printf("%.2f\n", TSparam);
			}	
		}
		if (data_optn == 0 && SH_optn) {
			printf(" h                  Doublets defined by?  ");
			if (SHcodon)
				printf("1st and 2nd codon positions\n");
			else
				printf("1st+2nd, 3rd+4th, etc. site\n");
		}
		if (data_optn == 1) { /* amino acids */
			if (Dayhf_optn) printf("Dayhoff (Dayhoff et al. 1978)\n");	
			if (Jtt_optn) printf("JTT (Jones et al. 1992)\n");
			if (mtrev_optn) printf("mtREV24 (Adachi-Hasegawa 1996)\n");
			if (blosum62_optn) printf("BLOSUM 62 (Henikoff-Henikoff 1992)\n");
		}
		if (data_optn == 2) { /* binary states */
			printf("Two-state model (Felsenstein 1981)\n");
		}
		if (data_optn == 1)
			printf(" f               Amino acid frequencies?  ");
		else if (data_optn == 0 && SH_optn)
			printf(" f                  Doublet frequencies?  ");
		else if (data_optn == 0 && nuc_optn)
			printf(" f               Nucleotide frequencies?  ");
		else if (data_optn == 2)
			printf(" f             Binary state frequencies?  ");
		printf("%s\n", (Frequ_optn ? "Estimate from data set" :
			"Use specified values"));
		if (data_optn == 0 && SH_optn)
			printf(" s       Symmetrize doublet frequencies?  %s\n",
				(sym_optn ? "Yes" : "No"));
				
		printf("RATE HETEROGENEITY\n");
		printf(" w          Model of rate heterogeneity?  ");
		if (rhetmode == 0) printf("Uniform rate\n");
		if (rhetmode == 1) printf("Gamma distributed rates\n");
		if (rhetmode == 2) printf("Two rates (1 invariable + 1 variable)\n");
		if (rhetmode == 3) printf("Mixed (1 invariable + %d Gamma rates)\n", numcats);
		
		if (rhetmode == 2 || rhetmode == 3) {
			printf(" i         Fraction of invariable sites?  ");
			if (fracinv_optim) printf("Estimate from data set");
			else printf("%.2f", fracinv);
			if (fracinv == 0.0 && !fracinv_optim) printf(" (all sites variable)");
			printf("\n");
		}
		if (rhetmode == 1 || rhetmode == 3) {
			printf(" a   Gamma distribution parameter alpha?  ");
			if (grate_optim)
				printf("Estimate from data set\n");
			else if (Geta > 0.5)
				printf("%.2f (strong rate heterogeneity)\n", (1.0-Geta)/Geta);
			else printf("%.2f (weak rate heterogeneity)\n", (1.0-Geta)/Geta);
			printf(" c      Number of Gamma rate categories?  %d\n", numcats);
		}
	
		printf("\nQuit [q], confirm [y], or change [menu] settings: ");
		
		/* read one char */
		ch = getchar();
		if (ch != '\n') {
			do ;
			while (getchar() != '\n');
		}
		ch = (char) tolower((int) ch);
		
		/* letters in use: a b c d e f g h i j k l m n o p q r s t u v w y x z */
		/* letters not in use:  */

		switch (ch) {

			case '\n':	break;
			
			case 'z':	if (typ_optn == 0 && (puzzlemode == 0 || puzzlemode == 1)) {
							compclock = compclock + 1;
							if (compclock == 2) compclock = 0;
						} else {
							printf("\n\n\nThis is not a possible option!\n");
						}
						break;

			case 'l':	if (compclock && typ_optn == 0 && (puzzlemode == 0 || puzzlemode == 1)) {
							printf("\n\n\nEnter an invalid branch number to search ");
							printf("for the best location!\n");
							printf("\nPlace root at branch (1-%d): ",
								2*Maxspc-3);
							scanf("%d", &locroot);
							do ;
							while (getchar() != '\n');
							if (locroot < 1 || locroot > 2*Maxspc-3) locroot = 0;
							locroot = locroot - 1; 
						} else {
							printf("\n\n\nThis is not a possible option!\n");
						}
						break;
			
			case 'e':	if ((rhetmode == 2 || rhetmode == 3) && fracinv_optim) {
							printf("\n\n\nInvariable sites estimation needs to be exact!\n");
						} else {
							approxp_optn = approxp_optn + 1;
							if (approxp_optn == 2) approxp_optn = 0;						
						}
						break;
						
			case 'w':	rhetmode = rhetmode + 1;
						if (rhetmode == 4) rhetmode = 0;
						if (rhetmode == 0) { /* uniform rate */
								numcats = 1;
								Geta = 0.05;
								grate_optim = FALSE;
								fracinv = 0.0;
								fracinv_optim = FALSE;
						}
						if (rhetmode == 1) { /* Gamma distributed rates */
								numcats = 8;
								Geta = 0.05;
								grate_optim = TRUE;
								fracinv = 0.0;
								fracinv_optim = FALSE;
						}
						if (rhetmode == 2) { /* two rates (1 invariable + 1 variable) */
								approxp_optn = FALSE;
								numcats = 1;
								Geta = 0.05;
								grate_optim = FALSE;
								fracinv = 0.0;
								fracinv_optim = TRUE;
						}
						if (rhetmode == 3) { /* mixed (1 invariable + Gamma rates) */
								approxp_optn = FALSE;
								numcats = 8;
								Geta = 0.05;
								grate_optim = TRUE;
								fracinv = 0.0;
								fracinv_optim = TRUE;
						}
						break;

			case 'i':	if (rhetmode == 2 || rhetmode == 3) {
							printf("\n\n\nEnter an invalid value for ");
							printf("estimation from data set!\n");
							printf("\nFraction of invariable sites among all sites (%.2f-%.2f): ",
								MINFI, MAXFI);
							scanf("%lf", &fracinv);
							do ;
							while (getchar() != '\n');
							if (fracinv < MINFI || fracinv > MAXFI) {
									fracinv_optim = TRUE;
									fracinv = 0.0;
							} else {
								fracinv_optim = FALSE;
							}
						} else {
							printf("\n\n\nThis is not a possible option!\n");
						}
						break;

			case 'a':	if (rhetmode == 1 || rhetmode == 3) {
							printf("\n\n\nEnter an invalid value for estimation from data set!\n");
							printf("\nGamma distribution parameter alpha (%.2f-%.2f): ",
								(1.0-MAXGE)/MAXGE, (1.0-MINGE)/MINGE);
							scanf("%lf", &Geta);
							do ;
							while (getchar() != '\n');
							if (Geta < (1.0-MAXGE)/MAXGE || Geta > (1.0-MINGE)/MINGE) {
								grate_optim = TRUE;
								Geta = 0.05;
							} else {
								grate_optim = FALSE;
								Geta = 1.0/(1.0 + Geta);
							}
						} else 
							printf("\n\n\nThis is not a possible option!\n");
						break;
						
			case 'c':	if (rhetmode == 1 || rhetmode == 3) {
							printf("\n\n\nNumber of Gamma rate categories (%d-%d): ",
								MINCAT, MAXCAT);
							scanf("%d", &numcats);
							do ;
							while (getchar() != '\n');
							if (numcats < MINCAT || numcats > MAXCAT) {
								printf("\n\n\nThis number of categories is not available!\n");
								numcats = 4;
							}
						} else {
							printf("\n\n\nThis is not a possible option!\n");
						}	
						break;
			
			case 'h':	if (data_optn == 0 && (Maxseqc % 3) == 0  && !SH_optn) {
							codon_optn = codon_optn + 1;
							if (codon_optn == 5) codon_optn = 0;
							translatedataset();
							/* reestimate nucleotide frequencies only 
							   if user did not specify other values */
							if (Frequ_optn) estimatebasefreqs();

						} else if (data_optn == 0 && SH_optn) { 
							if (Maxseqc % 2 != 0 && Maxseqc % 3 == 0) {
								SHcodon = TRUE;
								printf("\n\n\nThis is the only possible option for the data set!\n");
							}
							if (Maxseqc % 3 != 0 && Maxseqc % 2 == 0) {
								SHcodon = FALSE;
								printf("\n\n\nThis is the only possible option for the data set!\n");
							}
							if (Maxseqc % 2 == 0 && Maxseqc % 3 == 0) {
								if (SHcodon)
									SHcodon = FALSE;
								else
									SHcodon = TRUE;	
								translatedataset();	
								/* reestimate nucleotide frequencies only 
							   	   if user did not specify other values */
								if (Frequ_optn) estimatebasefreqs();
							}
						} else {
							printf("\n\n\nThis is not a possible option!\n");
						}
						break;

			case 'x':	if (typ_optn == 0 && puzzlemode == 1) {
							if (utree_optn) {
								utree_optn = FALSE;
								qcalg_optn = FALSE;
							} else {
								qcalg_optn = qcalg_optn + 1;
								if (qcalg_optn == 2) {
									qcalg_optn = 0;
									utree_optn = TRUE;
								}
							}
						} else {
							qcalg_optn = qcalg_optn + 1;
							if (qcalg_optn == 2) qcalg_optn = 0;
						}
						break;
						
			case 'k':	if (typ_optn == 0) {
							puzzlemode = puzzlemode + 1;
							if (puzzlemode == 3) puzzlemode = 0;
						} else {
							printf("\n\n\nThis is not a possible option!\n");
						}
						break;

			case 'b':	typ_optn = typ_optn + 1;
						if (typ_optn == 2) typ_optn = 0;
						break;

			case 'g':	if (typ_optn == 1) {
							clustA = clustB = clustC = clustD = 0;
							if (numclust != 1) {
								numclust = 1;
							} else {
								printf("\n\n\nNumber of clusters (2-4): ");
								scanf("%d", &numclust);
								do ;
								while (getchar() != '\n');
								if (numclust < 2 || numclust > 4) {
									numclust = 1;
									printf("\n\n\nOnly 2, 3, or 4 ");
									printf("clusters possible\n");
								} else {
									printf("\nDistribute all sequences over the ");
									if (numclust == 2) {
										printf("two clusters a and b (At least two\n");
										printf("sequences per cluster are necessary), ");
									}
									if (numclust == 3) {
										printf("three clusters a, b, and c\n");
										printf("(At least one sequence in cluster a and b, and at least two\n");
										printf("sequences in c are necessary), ");
									}
									if (numclust == 4) {
										printf("four clusters a, b, c, and d\n");
										printf("(At least one sequence per cluster is necessary),\n");
									}
									printf("type x to exclude a sequence:\n\n");
																
									for (i = 0; i < Maxspc; i++) {
										valid = FALSE;
										do {
											fputid10(stdout, i);
											printf(": ");
											/* read one char */
											ch = getchar();
											if (ch != '\n') {
											do ;
											while (getchar() != '\n');
											}	
											ch = (char) tolower((int) ch);
											if (ch == 'a' || ch == 'b' || ch == 'x')
												valid = TRUE;
											if (numclust == 3 || numclust == 4)
												if (ch == 'c') valid = TRUE;
											if (numclust == 4)
												if (ch == 'd') valid = TRUE;
										} while (!valid);
										if (ch == 'a') {
											clusterA[clustA] = i;
											clustA++;
										}
										if (ch == 'b') {
											clusterB[clustB] = i;
											clustB++;
										}
										if (ch == 'c') {
											clusterC[clustC] = i;
											clustC++;
										}
										if (ch == 'd') {
											clusterD[clustD] = i;
											clustD++;
										}
									}
									/* check clusters */
									valid = TRUE;
									if (numclust == 4) {
										if (clustA == 0) {
											valid = FALSE;
											numclust = 1;
											printf("\n\n\nNo sequence in cluster a\n");
										}
										if (clustB == 0) {
											valid = FALSE;
											numclust = 1;
											printf("\n\n\nNo sequence in cluster b\n");
										}
										if (clustC == 0) {
											valid = FALSE;
											numclust = 1;
											printf("\n\n\nNo sequence in cluster c\n");
										}
										if (clustD == 0) {
											valid = FALSE;
											numclust = 1;
											printf("\n\n\nNo sequence in cluster d\n");
										}
									}
									if (numclust == 3) {
										if (clustA == 0) {
											valid = FALSE;
											numclust = 1;
											printf("\n\n\nNo sequence in cluster a\n");
										}
										if (clustB == 0) {
											valid = FALSE;
											numclust = 1;
											printf("\n\n\nNo sequence in cluster b\n");
										}
										if (clustC < 2) {
											valid = FALSE;
											numclust = 1;
											if (clustC == 0)
												printf("\n\n\nNo sequence in cluster c\n");
											else
												printf("\n\n\nOnly one sequence in cluster c\n");
										}
									}
									if (numclust == 2) {
										if (clustA < 2) {
											valid = FALSE;
											numclust = 1;
											if (clustA == 0)
												printf("\n\n\nNo sequence in cluster a\n");
											else
												printf("\n\n\nOnly one sequence in cluster a\n");
										}
										if (clustB < 2) {
											valid = FALSE;
											numclust = 1;
											if (clustB == 0)
												printf("\n\n\nNo sequence in cluster b\n");
											else
												printf("\n\n\nOnly one sequence in cluster b\n");
										}	
									}
									if (valid) {
										printf("\nNumber of sequences in each cluster:\n\n");
										printf("Cluster a: %d\n", clustA);
										printf("Cluster b: %d\n", clustB);
										if (numclust > 2)
											printf("Cluster c: %d\n", clustC);
										if (numclust == 4)
											printf("Cluster d: %d\n", clustD);
										printf("\nExcluded sequences: ");
										if (numclust == 2) printf("%d\n",
											Maxspc-clustA-clustB);
										if (numclust == 3) printf("%d\n",
											Maxspc-clustA-clustB-clustC);
										if (numclust == 4) printf("%d\n",
											Maxspc-clustA-clustB-clustC-clustD);
									
									}
								}
							}
							/* number of resulting quartets */
							compnumqts();
							
						} else {
							printf("\n\n\nThis is not a possible option!\n");
						}
						break;

			case 'd':	data_optn = data_optn + 1;
						if (data_optn == 3) data_optn = 0;
						/* translate characters into format used by ML engine */
	                    translatedataset();
	                    estimatebasefreqs();
						break;

			case 'u':	if (puzzlemode == 0 && typ_optn == 0)
							show_optn = 1 - show_optn;
						else
							printf("\n\n\nThis is not a possible option!\n");
						break;

			case 'j':	if (puzzlemode == 0 && typ_optn == 0)
							listqptrees = 1 - listqptrees;
						else
							printf("\n\n\nThis is not a possible option!\n");
						break;
							
			case 'v':	if (puzzlemode == 0 && typ_optn == 0)
							approxqp = 1 - approxqp;
						else
							printf("\n\n\nThis is not a possible option!\n");
						break;

			case 'f':	if (Frequ_optn) {
							tstvf84 = 0.0;
							Frequ_optn = FALSE;
							sumfreq = 0.0;
							if (data_optn == 1)
								printf("\n\n\nAmino acid");
							else if (data_optn == 0 && SH_optn)
								printf("\n\n\nDoublet");
							else if (data_optn == 0 && nuc_optn)
								printf("\n\n\nNucleotide");
							else if (data_optn == 2)
								printf("\n\n\nBinary state");
							printf(" frequencies (in %%):\n\n");
							for (i = 0; i < gettpmradix() - 1; i++) {
								printf("pi(%s) = ", int2code(i));
								scanf("%lf", &(Freqtpm[i]));
								do ;
								while (getchar() != '\n');
								Freqtpm[i] = Freqtpm[i]/100.0;
								if (Freqtpm[i] < 0.0) {
									printf("\n\n\nNegative frequency not possible\n");
									estimatebasefreqs();
									break;
								}
								sumfreq = sumfreq + Freqtpm[i];
								if (sumfreq > 1.0) {
									printf("\n\n\nThe sum of ");
									printf("all frequencies exceeds");
									printf(" 100%%\n");
									estimatebasefreqs();
									break;
								}
								if (i == gettpmradix() - 2)
									Freqtpm[i+1] = 1.0 - sumfreq;
							}
						} else estimatebasefreqs();
						break;
			
			case 's':	if (data_optn == 0 && SH_optn) {
							sym_optn = 1 - sym_optn;
						} else {
							printf("\n\n\nThis is not a possible option!\n");
						}
						break;

			case 'n':	if (puzzlemode == 0 && typ_optn == 0)
						{
							printf("\n\n\nNumber of puzzling steps: ");
							scanf("%lu", &Numtrial);
							do ;
							while (getchar() != '\n');
							if (Numtrial < 1) {
								printf("\n\n\nThe number of puzzling");
								printf(" steps can't be smaller than one\n");
								Numtrial = 1000;
							}
						}
						else if (typ_optn == 1)
						{
							printf("\n\nEnter zero to use all possible");
							printf(" quartets in the analysis!\n");
							printf("\nNumber of random quartets: ");
							scanf("%lu", &lmqts);
							do ;
							while (getchar() != '\n');
							
							/* compute number of quartets used */
							compnumqts();
						}
						else
						{
							printf("\n\n\nThis is not a possible option!\n");
						}
						break;
						
			case 'o':	if (puzzlemode == 0 && typ_optn == 0) {
							printf("\n\n\nSequence to be displayed as outgroup (1-%d): ",
									Maxspc);
							scanf("%d", &outgroup);
							do ;
							while (getchar() != '\n');
							if (outgroup < 1 || outgroup > Maxspc) {
								printf("\n\n\nSequences are numbered ");
								printf("from 1 to %d\n",
										Maxspc);
								outgroup = 1;
							}
							outgroup = outgroup - 1;
						} else {
							printf("\n\n\nThis is not a possible option!\n");
						}
						break;
	
			case 'm':	if (data_optn == 0) { /* nucleotide data */
							if(HKY_optn && nuc_optn) {
								/* HKY -> TN */
								tstvf84 = 0.0;
								TSparam = 2.0;
								YRparam = 0.9;
								HKY_optn = FALSE;
								TN_optn = TRUE;
								optim_optn = TRUE;
								nuc_optn = TRUE;
								SH_optn = FALSE;
								break;
							}
							if(TN_optn && nuc_optn) {
								if (Maxseqc % 2 == 0 || Maxseqc % 3 == 0) {
									/* number of chars needs to be a multiple 2 or 3 */
									/* TN -> SH */		
									if (Maxseqc % 2 != 0 && Maxseqc % 3 == 0)
										SHcodon = TRUE;
									else
										SHcodon = FALSE;								
									tstvf84 = 0.0;
									TSparam = 2.0;
									YRparam = 1.0;
									HKY_optn = TRUE;
									TN_optn = FALSE;
									optim_optn = TRUE;
									nuc_optn = FALSE;
									SH_optn = TRUE;
									/* translate characters into format */
									/* used by ML engine */
									translatedataset();
									estimatebasefreqs();
								} else {
									printf("\n\n\nSH model not ");
									printf("available for the data set!\n");
									/* TN -> HKY */
									tstvf84 = 0.0;
									TSparam = 2.0;
									YRparam = 1.0;
									HKY_optn = TRUE;
									TN_optn = FALSE;
									optim_optn = TRUE;
									nuc_optn = TRUE;
									SH_optn = FALSE;
								}
								break;
							}
							if(SH_optn) {
								/* SH -> HKY */
								tstvf84 = 0.0;
								TSparam = 2.0;
								YRparam = 1.0;
								HKY_optn = TRUE;
								TN_optn = FALSE;
								optim_optn = TRUE;
								nuc_optn = TRUE;
								SH_optn = FALSE;
								/* translate characters into format */
								/* used by ML engine */
								translatedataset();
								estimatebasefreqs();
								break;
							}
							break;
						}
						if (data_optn == 1) { /* amino acid data */
							if (Dayhf_optn) {
								/* Dayhoff -> JTT */
								Dayhf_optn = FALSE;
								Jtt_optn = TRUE;
								mtrev_optn = FALSE;
								blosum62_optn = FALSE;
								break;
							}
							if (Jtt_optn) {
								/* JTT -> mtREV */
								Dayhf_optn = FALSE;
								Jtt_optn = FALSE;
								mtrev_optn = TRUE;
								blosum62_optn = FALSE;
								break;
							}
							if (mtrev_optn) {
								/* mtREV -> BLOSUM 62 */
								Dayhf_optn = FALSE;
								Jtt_optn = FALSE;
								mtrev_optn = FALSE;
								blosum62_optn = TRUE;
								break;
							}
							if (blosum62_optn) {
								/* BLOSUM 62 -> Dayhoff */
								Dayhf_optn = TRUE;
								Jtt_optn = FALSE;
								mtrev_optn = FALSE;
								blosum62_optn = FALSE;
								break;
							}
							break;
						}
						if (data_optn == 2) {
							printf("\n\n\nNo other model available!\n");
						}
						break;
						
			case 't':	if (data_optn != 0) {
							printf("\n\n\nThis is not a possible option!\n");
						} else {
							tstvf84 = 0.0;
							printf("\n\n\nEnter an invalid value for ");
							printf("estimation from data set!\n");
							printf("\nTransition/transversion parameter (%.2f-%.2f): ",
									MINTS, MAXTS);
							scanf("%lf", &TSparam);
							do ;
							while (getchar() != '\n');
							if (TSparam < MINTS || TSparam > MAXTS) {
								optim_optn = TRUE;
								TSparam = 2.0;
							} else {
								optim_optn = FALSE;
							}
						}
						break;

			case 'q':	printf("\n\n\n");
						exit(0);
						
						break;

			case 'r':	if (!(TN_optn && nuc_optn)){
							printf("\n\n\nThis is not a possible option!\n");
						} else {
							tstvf84 = 0.0;
							printf("\n\n\nEnter an invalid value ");
							printf("for estimation from data set!\n");
							printf("\nY/R transition parameter (%.2f-%.2f): ", MINYR, MAXYR);
							scanf("%lf", &YRparam);
							do ;
							while (getchar() != '\n');
							if (YRparam < MINYR || YRparam > MAXYR) {
								optim_optn = TRUE;
								YRparam = 0.9;
							} else if (YRparam == 1.0) {
								TN_optn = FALSE;
								HKY_optn = TRUE;
								if (optim_optn) TSparam = 2.0;
							} else {
								optim_optn = FALSE;
							}
						}
						break;
						
			case 'p':	if (!(TN_optn && nuc_optn)){
							printf("\n\n\nThis is not a possible option!\n");
						} else {
							printf("\n\n\nThe F84 model (Felsenstein 1984) is a restricted");
							printf(" TN model, and the one\nF84 parameter uniquely");
							printf(" determines the two corresponding TN parameters!\n\n");
							printf("F84 expected transition/transversion ratio: ");
							scanf("%lf", &tstvf84);
							do ;
							while (getchar() != '\n');
							if (tstvf84 <= 0.0) tstvf84 = 0.0;
							else makeF84model();
						}
						break;

			case 'y':	break;

			default:	printf("\n\n\nThis is not a possible option!\n");
						break;
		}
	} while (ch != 'y');

	printf("\n\n\n");
}

/* open file for reading */
void openfiletoread(FILE **fp, char name[], char descr[])
{
	int count = 0;
	cvector str;

	if ((*fp = fopen(name, "r")) == NULL) {
		printf("\n\n\nPlease enter a file name for the %s: ", descr);
		str = mygets();
		while ((*fp = fopen(str, "r")) == NULL)
		{
			count++;
			if (count > 10)
			{
				printf("\n\n\nToo many trials - quitting ...\n");
				exit(1);
			}
   			printf("File '%s' not found, ", str);
			printf("please enter alternative name: ");
			free_cvector(str);
			str = mygets();
		}
		free_cvector(str);
		printf("\n");
	}
}

/* open file for writing */
void openfiletowrite(FILE **fp, char name[], char descr[])
{
	int count = 0;
	cvector str;

	if ((*fp = fopen(name, "w")) == NULL) {
   		printf("\n\n\nPlease enter a file name for the %s: ", descr);
		str = mygets();
		while ((*fp = fopen(str, "w")) == NULL)
		{
			count++;
			if (count > 10)
			{
				printf("\n\n\nToo many trials - quitting ...\n");
				exit(1);
			}
   			printf("File '%s' not created, ", str);
			printf("please enter other name: ");
			free_cvector(str);
			str = mygets();
		}
		free_cvector(str);
		printf("\n");
	}
}

/* close file */
void closefile(FILE *fp)
{	
	fclose(fp);
}

/* symmetrize doublet frequencies */
void symdoublets()
{
	int i, imean;
	double mean;
	
	if (data_optn == 0 && SH_optn && sym_optn) {
		/* ML frequencies */
		mean = (Freqtpm[1] + Freqtpm[4])/2.0; /* AC CA */
		Freqtpm[1] = mean;
		Freqtpm[4] = mean;
		mean = (Freqtpm[2] + Freqtpm[8])/2.0; /* AG GA */
		Freqtpm[2] = mean;
		Freqtpm[8] = mean;
		mean = (Freqtpm[3] + Freqtpm[12])/2.0; /* AT TA */
		Freqtpm[3] = mean;
		Freqtpm[12] = mean;
		mean = (Freqtpm[6] + Freqtpm[9])/2.0; /* CG GC */
		Freqtpm[6] = mean;
		Freqtpm[9] = mean;
		mean = (Freqtpm[7] + Freqtpm[13])/2.0; /* CT TC */
		Freqtpm[7] = mean;
		Freqtpm[13] = mean;
		mean = (Freqtpm[11] + Freqtpm[14])/2.0; /* GT TG */
		Freqtpm[11] = mean;
		Freqtpm[14] = mean;
		
		/* base composition of each taxon */
		for (i = 0; i < Maxspc; i++) {
			imean = (Basecomp[i][1] + Basecomp[i][4])/2; /* AC CA */
			Basecomp[i][1] = imean;
			Basecomp[i][4] = imean;
			imean = (Basecomp[i][2] + Basecomp[i][8])/2; /* AG GA */
			Basecomp[i][2] = imean;
			Basecomp[i][8] = imean;
			imean = (Basecomp[i][3] + Basecomp[i][12])/2; /* AT TA */
			Basecomp[i][3] = imean;
			Basecomp[i][12] = imean;
			imean = (Basecomp[i][6] + Basecomp[i][9])/2; /* CG GC */
			Basecomp[i][6] = imean;
			Basecomp[i][9] = imean;
			imean = (Basecomp[i][7] + Basecomp[i][13])/2; /* CT TC */
			Basecomp[i][7] = imean;
			Basecomp[i][13] = imean;
			imean = (Basecomp[i][11] + Basecomp[i][14])/2; /* GT TG */
			Basecomp[i][11] = imean;
			Basecomp[i][14] = imean;
		}
	}
}

/* show Ts/Tv ratio and Ts Y/R ratio */
void computeexpectations()
{
	double AlphaYBeta, AlphaRBeta, piR, piY, num, denom, pyr, pur;
	
	if (nuc_optn == TRUE) { /* 4x4 nucs */
		piR = Freqtpm[0] + Freqtpm[2];
		piY = Freqtpm[1] + Freqtpm[3];
		AlphaRBeta = 4.0*TSparam / (1 + YRparam);
		AlphaYBeta = AlphaRBeta * YRparam;
		tstvratio = (AlphaRBeta*Freqtpm[0]*Freqtpm[2] +
					 AlphaYBeta*Freqtpm[1]*Freqtpm[3])/(piR * piY);
		yrtsratio = (AlphaYBeta*Freqtpm[1]*Freqtpm[3]) /
					(AlphaRBeta*Freqtpm[0]*Freqtpm[2]);
	} else { /* 16x16 nucs */
		pyr = Freqtpm[1]*Freqtpm[3] + Freqtpm[5]*Freqtpm[7] +
			Freqtpm[9]*Freqtpm[11] + Freqtpm[4]*Freqtpm[12] +
			Freqtpm[5]*Freqtpm[13] + Freqtpm[6]*Freqtpm[14] +
			Freqtpm[7]*Freqtpm[15] + Freqtpm[13]*Freqtpm[15];
		pur = Freqtpm[0]*Freqtpm[2] + Freqtpm[4]*Freqtpm[6] +
			Freqtpm[0]*Freqtpm[8] + Freqtpm[1]*Freqtpm[9] +
			Freqtpm[2]*Freqtpm[10] + Freqtpm[8]*Freqtpm[10] +
			Freqtpm[3]*Freqtpm[11] + Freqtpm[12]*Freqtpm[14];
		num = pyr + pur;
		denom = Freqtpm[0]*Freqtpm[1] + Freqtpm[1]*Freqtpm[2] +
			Freqtpm[0]*Freqtpm[3] + Freqtpm[2]*Freqtpm[3] +
			Freqtpm[0]*Freqtpm[4] + Freqtpm[1]*Freqtpm[5] +
			Freqtpm[4]*Freqtpm[5] + Freqtpm[2]*Freqtpm[6] +
			Freqtpm[5]*Freqtpm[6] + Freqtpm[3]*Freqtpm[7] +
			Freqtpm[4]*Freqtpm[7] + Freqtpm[6]*Freqtpm[7] +
			Freqtpm[4]*Freqtpm[8] + Freqtpm[5]*Freqtpm[9] +
			Freqtpm[8]*Freqtpm[9] + Freqtpm[6]*Freqtpm[10] +
			Freqtpm[9]*Freqtpm[10] + Freqtpm[7]*Freqtpm[11] +
			Freqtpm[8]*Freqtpm[11] + Freqtpm[10]*Freqtpm[11] +
			Freqtpm[0]*Freqtpm[12] + Freqtpm[8]*Freqtpm[12] +
			Freqtpm[1]*Freqtpm[13] + Freqtpm[9]*Freqtpm[13] +
			Freqtpm[12]*Freqtpm[13] + Freqtpm[2]*Freqtpm[14] +
			Freqtpm[10]*Freqtpm[14] + Freqtpm[13]*Freqtpm[14] +
			Freqtpm[3]*Freqtpm[15] + Freqtpm[11]*Freqtpm[15] +
			Freqtpm[12]*Freqtpm[15] + Freqtpm[14]*Freqtpm[15];
		tstvratio = 2.0*TSparam * num/denom;
		yrtsratio = pyr/pur;
	}
}

/* write ML distance matrix to file */
void putdistance(FILE *fp)
{
	int i, j;
	
	fprintf(fp, "  %d\n", Maxspc);
	for (i = 0; i < Maxspc; i++) {
		fputid10(fp, i);
		for (j = 0; j < Maxspc; j++) {
			fprintf(fp, "  %.5f", Distanmat[i][j]/100.0);
			/* seven in one row */
			if ((j + 1) % 7 == 0 && j+1 != Maxspc)
				fprintf(fp, "\n          ");
		}
		fprintf(fp, "\n");
	}
}


/* find identical sequences */
void findidenticals(FILE *fp)
{
	int i, j, noids;
	cvector useqs;
	
	useqs = new_cvector(Maxspc);
	
	for (i = 0; i < Maxspc; i++)
		useqs[i] = 0;
	
	noids = TRUE;
	for (i = 0; i < Maxspc && noids; i++)
		for (j = i + 1; j < Maxspc && noids; j++)
			if (Distanmat[i][j] == 0.0) noids = FALSE;
	
	if (noids)
		fprintf(fp, " All sequences are unique.\n");
	else {
		for (i = 0; i < Maxspc; i++) {	
			noids = TRUE;
			for (j = i + 1; j < Maxspc && noids; j++)
				if (Distanmat[i][j] == 0.0) noids = FALSE;
				
			if (!noids && useqs[i] == 0) {
				fputid(fp, i);
				useqs[i] = 1;	
				for (j = i + 1; j < Maxspc; j++)
					if (Distanmat[i][j] == 0.0) {
						fprintf(fp, ", ");
						fputid(fp, j);
						useqs[j] = 1;
					}				
				fprintf(fp, ".\n");
			}
		}
	}
	free_cvector(useqs);
}

/* compute average distance */
double averagedist()
{	
	int i, j;
	double sum;
	
	sum = 0.0;
	for (i = 0; i < Maxspc; i++)
		for (j = i + 1; j < Maxspc; j++)
			sum = sum + Distanmat[i][j];
	
	sum = sum / (double) Maxspc / ((double) Maxspc - 1.0) * 2.0;
	
	return sum;
}

/* first lines of EPSF likelihood mapping file */
void initps(FILE *ofp)
{
	fprintf(ofp, "%%!PS-Adobe-3.0 EPSF-3.0\n");
	fprintf(ofp, "%%%%BoundingBox: 60 210 550 650\n");
	fprintf(ofp, "%%%%Pages: 1\n");
	fprintf(ofp, "%%%%Creator: PUZZLE (version %s)\n", VERSION);
	fprintf(ofp, "%%%%Title: Likelihood Mapping Analysis\n");
	fprintf(ofp, "%%%%CreationDate: %s", asctime(localtime(&Starttime)) );
	fprintf(ofp, "%%%%DocumentFonts: Helvetica\n");
	fprintf(ofp, "%%%%DocumentNeededFonts: Helvetica\n");
	fprintf(ofp, "%%%%EndComments\n");
	fprintf(ofp, "%% use inch as unit\n");
	fprintf(ofp, "/inch {72 mul} def\n");
	fprintf(ofp, "%% triangle side length (3 inch)\n");
	fprintf(ofp, "/tl {3 inch mul} def\n");
	fprintf(ofp, "%% plot one dot (x-y coordinates on stack)\n");
	fprintf(ofp, "/dot {\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, "0.002 tl 0 360 arc  %% radius is 0.002 of the triangle length\n");
	fprintf(ofp, "closepath\n");
	fprintf(ofp, "fill\n");
	fprintf(ofp, "} def\n");
	fprintf(ofp, "%% preamble\n");
	fprintf(ofp, "/Helvetica findfont\n");
	fprintf(ofp, "12 scalefont\n");
	fprintf(ofp, "setfont\n");
	fprintf(ofp, "%% 0/0 for triangle of triangles\n");
	fprintf(ofp, "0.9 inch 3 inch translate\n");
	fprintf(ofp, "%% first triangle (the one with dots)\n");
	fprintf(ofp, "0.6 tl 1.2 tl 0.8660254038 mul translate\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.0 tl 0.0 tl moveto\n");
	fprintf(ofp, " 1.0 tl 0.0 tl lineto\n");
	fprintf(ofp, " 0.5 tl 0.8660254038 tl lineto\n");
	fprintf(ofp, "closepath\n");
	fprintf(ofp, "stroke\n");
}

/* plot one point of likelihood mapping analysis */
void plotlmpoint(FILE *ofp, double w1, double w2)
{
	fprintf(ofp,"%.10f tl %.10f tl dot\n",
		0.5*w1 + w2, w1*0.8660254038);
}

/* last lines of EPSF likelihood mapping file */
void finishps(FILE *ofp)
{
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "%% second triangle (the one with 3 basins)\n");
	fprintf(ofp, "/secondtriangle {\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.0 tl 0.0 tl moveto\n");
	fprintf(ofp, " 1.0 tl 0.0 tl lineto\n");
	fprintf(ofp, " 0.5 tl 0.8660254038 tl lineto\n");
	fprintf(ofp, "closepath\n");
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.50 tl 0.2886751346 tl moveto\n");
	fprintf(ofp, " 0.50 tl 0.0000000000 tl lineto\n");
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.50 tl 0.2886751346 tl moveto\n");
	fprintf(ofp, " 0.25 tl 0.4330127019 tl lineto\n");
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.50 tl 0.2886751346 tl moveto\n");
	fprintf(ofp, " 0.75 tl 0.4330127019 tl lineto\n");
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "0.44 tl 0.5 tl moveto %% up\n");
	fprintf(ofp, "(%.1f%%) show\n", (double) ar1*100.0/Numquartets);
	fprintf(ofp, "0.25 tl 0.15 tl moveto %% down left\n");
	fprintf(ofp, "(%.1f%%) show\n", (double) ar3*100.0/Numquartets);
	fprintf(ofp, "0.63 tl 0.15 tl moveto %% down right\n");
	fprintf(ofp, "(%.1f%%) show\n", (double) ar2*100.0/Numquartets);
	fprintf(ofp, "} def\n");
	fprintf(ofp, "%% third triangle (the one with 7 basins)\n");
	fprintf(ofp, "/thirdtriangle {\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.0 tl 0.0 tl moveto\n");
	fprintf(ofp, " 1.0 tl 0.0 tl lineto\n");
	fprintf(ofp, " 0.5 tl 0.8660254038 tl lineto\n");
	fprintf(ofp, "closepath\n");
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.25 tl 0.1443375673 tl moveto\n");
	fprintf(ofp, " 0.75 tl 0.1443375673 tl lineto\n");
	fprintf(ofp, " 0.50 tl 0.5773502692 tl lineto\n");
	fprintf(ofp, "closepath\n");
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.125 tl 0.2165063509 tl moveto\n");
	fprintf(ofp, " 0.250 tl 0.1443375673 tl lineto\n");
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.375 tl 0.6495190528 tl moveto\n");
	fprintf(ofp, " 0.500 tl 0.5773502692 tl lineto\n");
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.625 tl 0.6495190528 tl moveto\n");
	fprintf(ofp, " 0.500 tl 0.5773502692 tl lineto\n");
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.875 tl 0.2165063509 tl moveto\n");
	fprintf(ofp, " 0.750 tl 0.1443375673 tl lineto\n");
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.750 tl 0.00 tl moveto\n");
	fprintf(ofp, " 0.750 tl 0.1443375673 tl lineto\n");
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "newpath\n");
	fprintf(ofp, " 0.250 tl 0.00 tl moveto\n");
	fprintf(ofp, " 0.250 tl 0.1443375673 tl lineto\n");
	fprintf(ofp, "stroke\n");
	fprintf(ofp, "0.42 tl 0.66 tl moveto %% up\n");
	fprintf(ofp, "(%.1f%%) show\n", (double) reg1*100.0/Numquartets);
	fprintf(ofp, "0.07 tl 0.05 tl moveto %% down left\n");
	fprintf(ofp, "(%.1f%%) show\n", (double) reg3*100.0/Numquartets);
	fprintf(ofp, "0.77 tl 0.05 tl moveto %% down right\n");
	fprintf(ofp, "(%.1f%%) show\n", (double) reg2*100.0/Numquartets);
	fprintf(ofp, "0.43 tl 0.05 tl moveto %% down side\n");
	fprintf(ofp, "(%.1f%%) show\n", (double) reg5*100.0/Numquartets);
	fprintf(ofp, "0.43 tl 0.28 tl moveto %% center\n");
	fprintf(ofp, "(%.1f%%) show\n", (double) reg7*100.0/Numquartets);
	fprintf(ofp, "gsave\n");
	fprintf(ofp, "-60 rotate\n");
	fprintf(ofp, "-0.07 tl 0.77 tl moveto %% right side\n");
	fprintf(ofp, "(%.1f%%) show\n", (double) reg4*100.0/Numquartets);
	fprintf(ofp, "grestore\n");
	fprintf(ofp, "gsave\n");
	fprintf(ofp, "60 rotate\n");
	fprintf(ofp, "0.4 tl -0.09 tl moveto %% left side\n");
	fprintf(ofp, "(%.1f%%) show\n", (double) reg6*100.0/Numquartets);
	fprintf(ofp, "grestore\n");
	fprintf(ofp, "} def\n");
	fprintf(ofp, "%% print the other two triangles\n");
	fprintf(ofp, "-0.6 tl -1.2 tl 0.8660254038 mul translate\n");
	fprintf(ofp, "secondtriangle\n");
	fprintf(ofp, "1.2 tl 0 translate\n");
	fprintf(ofp, "thirdtriangle\n");	
	if (numclust == 4) { /* four cluster analysis */
		fprintf(ofp, "%% label corners\n");
		fprintf(ofp, "0.375 tl 0.9 tl moveto\n");
		fprintf(ofp, "((a,b)-(c,d)) show %% CHANGE HERE IF NECESSARY\n");
		fprintf(ofp, "-0.16 tl -0.08 tl moveto\n");
		fprintf(ofp, "((a,d)-(b,c)) show %% CHANGE HERE IF NECESSARY\n");
		fprintf(ofp, "0.92 tl -0.08 tl moveto\n");
		fprintf(ofp, "((a,c)-(b,d)) show %% CHANGE HERE IF NECESSARY\n");
	}
	if (numclust == 3) { /* three cluster analysis */
		fprintf(ofp, "%% label corners\n");
		fprintf(ofp, "0.375 tl 0.9 tl moveto\n");
		fprintf(ofp, "((a,b)-(c,c)) show %% CHANGE HERE IF NECESSARY\n");
		fprintf(ofp, "-0.16 tl -0.08 tl moveto\n");
		fprintf(ofp, "((a,c)-(b,c)) show %% CHANGE HERE IF NECESSARY\n");
		fprintf(ofp, "0.92 tl -0.08 tl moveto\n");
		fprintf(ofp, "((a,c)-(b,c)) show %% CHANGE HERE IF NECESSARY\n");
	}
	if (numclust == 2) { /* two cluster analysis */
		fprintf(ofp, "%% label corners\n");
		fprintf(ofp, "0.375 tl 0.9 tl moveto\n");
		fprintf(ofp, "((a,a)-(b,b)) show %% CHANGE HERE IF NECESSARY\n");
		fprintf(ofp, "-0.16 tl -0.08 tl moveto\n");
		fprintf(ofp, "((a,b)-(a,b)) show %% CHANGE HERE IF NECESSARY\n");
		fprintf(ofp, "0.92 tl -0.08 tl moveto\n");
		fprintf(ofp, "((a,b)-(a,b)) show %% CHANGE HERE IF NECESSARY\n");
	}
	fprintf(ofp, "showpage\n");
	fprintf(ofp, "%%%%EOF\n");
}

/* computes LM point from the three log-likelihood values,
   plots the point, and does some statistics */
void makelmpoint(FILE *fp, double b1, double b2, double b3)
{
	double w1, w2, w3, temp;
	unsigned char qpbranching;
	double temp1, temp2, temp3, onethird;
	unsigned char discreteweight[3], treebits[3];
	
	onethird = 1.0/3.0;
	treebits[0] = (unsigned char) 1;
	treebits[1] = (unsigned char) 2;
	treebits[2] = (unsigned char) 4;

	/* sort in descending order */
	qweight[0] = b1;
	qweight[1] = b2;
	qweight[2] = b3;
	sort3doubles(qweight, qworder);

	/* compute Bayesian weights */
	qweight[qworder[1]] = exp(qweight[qworder[1]]-qweight[qworder[0]]);
	qweight[qworder[2]] = exp(qweight[qworder[2]]-qweight[qworder[0]]);
	qweight[qworder[0]] = 1.0;
	temp = qweight[0] + qweight[1] + qweight[2];
	qweight[0] = qweight[0]/temp;
	qweight[1] = qweight[1]/temp;
	qweight[2] = qweight[2]/temp;

	/* plot one point in likelihood mapping triangle */
	w1 = qweight[0];
	w2 = qweight[1];
	w3 = qweight[2];
	plotlmpoint(fp, w1, w2);
	
	/* check areas 1,2,3 */	
	if (treebits[qworder[0]] == 1) ar1++;
	else if (treebits[qworder[0]] == 2) ar2++;
	else ar3++;				

	/* check out regions 1,2,3,4,5,6,7 */

	/* 100 distribution */
	temp1 = 1.0 - qweight[qworder[0]];
	sqdiff[0] = temp1*temp1 +
		qweight[qworder[1]]*qweight[qworder[1]] +
		qweight[qworder[2]]*qweight[qworder[2]];
	discreteweight[0] = treebits[qworder[0]];

	/* 110 distribution */
	temp1 = 0.5 - qweight[qworder[0]];
	temp2 = 0.5 - qweight[qworder[1]];
	sqdiff[1] = temp1*temp1 + temp2*temp2 +
		qweight[qworder[2]]*qweight[qworder[2]];
	discreteweight[1] = treebits[qworder[0]] + treebits[qworder[1]];

	/* 111 distribution */
	temp1 = onethird - qweight[qworder[0]];
	temp2 = onethird - qweight[qworder[1]];
	temp3 = onethird - qweight[qworder[2]];
	sqdiff[2] = temp1 * temp1 + temp2 * temp2 + temp3 * temp3;
	discreteweight[2] = (unsigned char) 7;

	/* sort in descending order */
	sort3doubles(sqdiff, sqorder);
			
	qpbranching = (unsigned char) discreteweight[sqorder[2]];
							
	if (qpbranching == 1) {
		reg1++;
		if (w2 < w3) reg1l++;
		else reg1r++;
	}
	if (qpbranching == 2) {
		reg2++;
		if (w1 < w3) reg2d++;
		else reg2u++;
	}
	if (qpbranching == 4) {
		reg3++;
		if (w1 < w2) reg3d++;
		else reg3u++;
	}
	if (qpbranching == 3) {
		reg4++;
		if (w1 < w2) reg4d++;
		else reg4u++;
	}
	if (qpbranching == 6) {
		reg5++;
		if (w2 < w3) reg5l++;
		else reg5r++;
	}
	if (qpbranching == 5) {
		reg6++;
		if (w1 < w3) reg6d++;
		else reg6u++;
	}
	if (qpbranching == 7) reg7++;
}

/* print tree statistics */
void printtreestats(FILE *ofp)
{
	int i, j, besttree;
	double bestlkl, difflkl, difflklps, temp, sum;
	
	/* find best tree */
	besttree = 0;
	bestlkl = ulkl[0];
	for (i = 1; i < numutrees; i++)
		if (ulkl[i] > bestlkl) {
			besttree = i;
			bestlkl = ulkl[i];
		}
	
	fprintf(ofp, "\n\nCOMPARISON OF USER TREES (NO CLOCK)\n\n");
	fprintf(ofp, "Tree   log L   difference     S.E.   Significantly worse\n");
	fprintf(ofp, "--------------------------------------------------------\n");
	for (i = 0; i < numutrees; i++) {
		difflkl = ulkl[besttree]-ulkl[i];
		fprintf(ofp, "%2d %10.2f %8.2f ", i+1, ulkl[i], difflkl);
		if (i == besttree) {
			fprintf(ofp, " <----------------- best tree");
		} else {
			/* compute variance of Log L differences over sites */
			difflklps = difflkl/(double)Maxsite;
			sum = 0.0;
			for (j = 0; j < Numptrn; j++) {
				temp = allsites[besttree][j] - allsites[i][j] - difflklps;
				sum += temp*temp*Weight[j];
			}
			sum = sqrt(fabs(sum/(Maxsite-1.0)*Maxsite));
			fprintf(ofp, "%11.2f         ", sum);
			if (difflkl > 1.96*sum)
				fprintf(ofp, "yes");
			else
				fprintf(ofp, "no");
		}
		fprintf(ofp, "\n");
	}
	fprintf(ofp, "\nThis test (5%% significance) follows Kishino and Hasegawa (1989).\n");
	
	if (compclock) {
	
		/* find best tree */
		besttree = 0;
		bestlkl = ulklc[0];
		for (i = 1; i < numutrees; i++)
			if (ulklc[i] > bestlkl) {
				besttree = i;
				bestlkl = ulklc[i];
			}
	
		fprintf(ofp, "\n\nCOMPARISON OF USER TREES (WITH CLOCK)\n\n");
		fprintf(ofp, "Tree   log L   difference     S.E.   Significantly worse\n");
		fprintf(ofp, "--------------------------------------------------------\n");
		for (i = 0; i < numutrees; i++) {
			difflkl = ulklc[besttree]-ulklc[i];
			fprintf(ofp, "%2d %10.2f %8.2f ", i+1, ulklc[i], difflkl);
			if (i == besttree) {
				fprintf(ofp, " <----------------- best tree");
			} else {
				/* compute variance of Log L differences over sites */
				difflklps = difflkl/(double)Maxsite;
				sum = 0.0;
				for (j = 0; j < Numptrn; j++) {
					temp = allsitesc[besttree][j] - allsitesc[i][j] - difflklps;
					sum += temp*temp*Weight[j];
				}
				sum = sqrt(fabs(sum/(Maxsite-1.0)*Maxsite));
				fprintf(ofp, "%11.2f         ", sum);
				if (difflkl > 1.96*sum)
					fprintf(ofp, "yes");
				else
					fprintf(ofp, "no");
			}
			fprintf(ofp, "\n");
		}
		fprintf(ofp, "\nThis test (5%% significance) follows Kishino and Hasegawa (1989).\n");
	}
}

/* time stamp */
void timestamp(FILE* ofp)
{
	fprintf(ofp, "\n\nTIME STAMP\n\n");
	fprintf(ofp, "Date and time: %s", asctime(localtime(&Starttime)) );
	fprintf(ofp, "Runtime: %lu seconds (= %.1f minutes = %.1f hours)\n\n",
		(uli) (Stoptime - Starttime), (Stoptime - Starttime)/60.,
		(Stoptime - Starttime)/3600.);
}


/* write output file */
void writeoutputfile(FILE *ofp)
{
	int i, fail, df;
	uli li;
	double pval, delta;

	fprintf(ofp, "PUZZLE %s\n\n", VERSION);
	fprintf(ofp, "Type of analysis: ");
	if (typ_optn == 0) fprintf(ofp, "tree reconstruction\n");
	if (typ_optn == 1) fprintf(ofp, "likelihood mapping\n");
	fprintf(ofp, "Parameter estimation: ");
	if (approxp_optn) fprintf(ofp, "approximate (faster)\n");
	else fprintf(ofp, "accurate (slow)\n");
	if (!(puzzlemode == 1 && typ_optn == 0)) {
		fprintf(ofp, "Parameter estimation uses: ");
		if (qcalg_optn)
			fprintf(ofp, "quartet sampling (for substitution process) + NJ tree (for rate variation)\n");
		else
			fprintf(ofp, "neighbor-joining tree (for substitution process and rate variation)\n");
	} else {
		fprintf(ofp, "Parameter estimation uses: ");
		if (utree_optn)
			fprintf(ofp, "1st user tree (for substitution process and rate variation)\n");
		else if (qcalg_optn)
			fprintf(ofp, "quartet sampling (for substitution process) + NJ tree (for rate variation)\n");
		else
			fprintf(ofp, "neighbor-joining tree (for substitution process and rate variation)\n");
	}
	fprintf(ofp, "\nStandard errors (S.E.) are obtained by the curvature method.\n");
	fprintf(ofp, "The upper and lower bounds of an approximate 95%% confidence interval\n");
	fprintf(ofp, "for parameter or branch length x are x-1.96*S.E. and x+1.96*S.E.\n");
	fprintf(ofp, "\n\n");
	fprintf(ofp, "SEQUENCE ALIGNMENT\n\n");
	fprintf(ofp, "Input data: %d sequences with %d ", Maxspc, Maxsite);
	if (data_optn == 1)
		fprintf(ofp, "amino acid");
	else if (data_optn == 0 && SH_optn)
		fprintf(ofp, "doublet (%d nucleotide)", Maxsite*2);
	else if (data_optn == 0 && nuc_optn)
		fprintf(ofp, "nucleotide");
	else if (data_optn == 2)
		fprintf(ofp, "binary state");
	fprintf(ofp, " sites");
	if (data_optn == 0 && (Maxseqc % 3) == 0  && !SH_optn) {
		if (codon_optn == 1) fprintf(ofp, " (1st codon positions)");
		if (codon_optn == 2) fprintf(ofp, " (2nd codon positions)");
		if (codon_optn == 3) fprintf(ofp, " (3rd codon positions)");
		if (codon_optn == 4) fprintf(ofp, " (1st and 2nd codon positions)");
	}
	if (data_optn == 0 && SH_optn) {
		if (SHcodon)
			fprintf(ofp, " (1st and 2nd codon positions)");
		else
			fprintf(ofp, " (1st+2nd, 3rd+4th, etc. site)");
	}	
	fprintf(ofp, "\n");
	fprintf(ofp, "Number of constant sites: %d (= %.1f%% of all sites)\n\n\n",
		Numconst, 100.0*fracconst);
	fprintf(ofp, "SUBSTITUTION PROCESS\n\n");
	fprintf(ofp, "Model of substitution: ");
	if (data_optn == 0) { /* nucleotides */
		if (nuc_optn) {
			if(HKY_optn) fprintf(ofp, "HKY (Hasegawa et al. 1985)\n");	
			else fprintf(ofp, "TN (Tamura-Nei 1993)\n");
			fprintf(ofp, "Transition/transversion parameter");
			if (optim_optn)
				fprintf(ofp, " (estimated from data set)");
			fprintf(ofp, ": %.2f", TSparam);
			if (optim_optn)
				fprintf(ofp, " (S.E. %.2f)", tserr);
			fprintf(ofp, "\n");
			
			if (optim_optn && TSparam > MAXTS - 1.0)
				fprintf(ofp, "WARNING --- parameter estimate close to internal upper bound!\n");
			if (optim_optn && TSparam < MINTS + 0.1)
				fprintf(ofp, "WARNING --- parameter estimate close to internal lower bound!\n");		
			
			if (TN_optn) {
				fprintf(ofp, "Y/R transition parameter");
				if (optim_optn)
					fprintf(ofp, " (estimated from data set)");
				fprintf(ofp, ": %.2f", YRparam);
				if (optim_optn)
					fprintf(ofp, " (S.E. %.2f)", yrerr);
				fprintf(ofp, "\n");
				
				if (optim_optn && YRparam > MAXYR - 0.5)
					fprintf(ofp, "WARNING --- parameter estimate close to internal upper bound!\n");
				if (optim_optn && YRparam < MINYR + 0.1)
					fprintf(ofp, "WARNING --- parameter estimate close to internal lower bound!\n");		

			}
		}
		if (SH_optn) {
			fprintf(ofp, "SH (Schoeniger-von Haeseler 1994)\n");
			fprintf(ofp, "Transition/transversion parameter");
			if (optim_optn) fprintf(ofp, " (estimated from data set)");
			fprintf(ofp, ": %.2f\n", TSparam);
			if (optim_optn)
				fprintf(ofp, " (S.E. %.2f)", tserr);
			fprintf(ofp, "\n");
						
			if (optim_optn && TSparam > MAXTS - 1.0)
				fprintf(ofp, "WARNING --- parameter estimate close to internal upper bound!\n");
			if (optim_optn && TSparam < MINTS + 0.1)
				fprintf(ofp, "WARNING --- parameter estimate close to internal lower bound!\n");		

		}	
	}
	if (data_optn == 1) { /* amino acids */
		if (Dayhf_optn) fprintf(ofp, "Dayhoff (Dayhoff et al. 1978)\n");	
		if (Jtt_optn) fprintf(ofp, "JTT (Jones et al. 1992)\n");
		if (mtrev_optn) fprintf(ofp, "mtREV24 (Adachi-Hasegawa 1996)\n");
		if (blosum62_optn) fprintf(ofp, "BLOSUM 62 (Henikoff-Henikoff 1992)\n");
	}
	if (data_optn == 2) { /* binary states */
		fprintf(ofp, "Two-state model (Felsenstein 1981)\n");
	}
	if (data_optn == 1)
			fprintf(ofp, "Amino acid ");
		else if (data_optn == 0 && SH_optn)
			fprintf(ofp, "Doublet ");
		else if (data_optn == 0 && nuc_optn)
			fprintf(ofp, "Nucleotide ");
		else if (data_optn == 2)
			fprintf(ofp, "Binary state ");
	fprintf(ofp, "frequencies (");
	if (Frequ_optn) fprintf(ofp, "estimated from data set");
	else fprintf(ofp, "user specified");
	if (data_optn == 0 && SH_optn && sym_optn)
		fprintf(ofp, " and symmetrized");
	fprintf(ofp, "):\n\n");
	for (i = 0; i < gettpmradix(); i++)
		fprintf(ofp, " pi(%s) = %.1f%%\n",
			int2code(i), Freqtpm[i]*100);
	if (data_optn == 0) {
		fprintf(ofp, "\nExpected transition/transversion ratio: %.2f",
			tstvratio);
		if (tstvf84 == 0.0) fprintf(ofp, "\n");
		else fprintf(ofp, " (= F84 parameter)\n");
		fprintf(ofp, "Expected pyrimidine transition/purine transition");
		fprintf(ofp, " ratio: %.2f\n", yrtsratio);
		if (tstvf84 != 0.0 && TN_optn)
			fprintf(ofp,
				"This TN model is equivalent to a F84 model (Felsenstein 1984).\n");
	}
	fprintf(ofp, "\n\nRATE HETEROGENEITY\n\n");
	fprintf(ofp, "Model of rate heterogeneity: ");
	if (rhetmode == 0) fprintf(ofp, "uniform rate\n");
	if (rhetmode == 1) fprintf(ofp, "Gamma distributed rates\n");
	if (rhetmode == 2) fprintf(ofp, "two rates (1 invariable + 1 variable)\n");
	if (rhetmode == 3) fprintf(ofp, "mixed (1 invariable + %d Gamma rates)\n", numcats);
	if (rhetmode == 2 || rhetmode == 3) {
		fprintf(ofp, "Fraction of invariable sites");
		if (fracinv_optim) fprintf(ofp, " (estimated from data set)");
		fprintf(ofp, ": %.2f", fracinv);
		if (fracinv_optim) fprintf(ofp, " (S.E. %.2f)", fierr);
		fprintf(ofp, "\n");
			
		if (fracinv_optim && fracinv > MAXFI - 0.05)
			fprintf(ofp, "WARNING --- parameter estimate close to internal upper bound!\n");
		
		fprintf(ofp, "Number of invariable sites: %.0f\n", floor(fracinv*Maxsite));
	}
	if (rhetmode == 1 || rhetmode == 3) {
		fprintf(ofp, "Gamma distribution parameter alpha");
		if (grate_optim) fprintf(ofp, " (estimated from data set)");
		fprintf(ofp, ": %.2f", (1.0-Geta)/Geta);
		if (grate_optim) fprintf(ofp, " (S.E. %.2f)",
			geerr/(Geta*Geta)); /* first order approximation */
		fprintf(ofp, "\n");
		
		if (grate_optim && Geta > MAXGE - 0.02)
			fprintf(ofp, "WARNING --- parameter estimate close to internal upper bound!\n");
		if (grate_optim && Geta < MINGE + 0.01)
			fprintf(ofp, "WARNING --- parameter estimate close to internal lower bound!\n");		

		fprintf(ofp, "Number of Gamma rate categories: %d\n", numcats);
	}
	if (rhetmode == 3) {
		fprintf(ofp, "Total rate heterogeneity (invariable sites + Gamma model): ");
		fprintf(ofp, "%.2f", fracinv + Geta - fracinv*Geta);
		if (grate_optim && fracinv_optim)
			fprintf(ofp, " (S.E. %.2f)", geerr + fierr); /* first order approximation */
		else if (grate_optim && !fracinv_optim)
			fprintf(ofp, " (S.E. %.2f)", geerr);
		else if (!grate_optim && fracinv_optim)
			fprintf(ofp, " (S.E. %.2f)", fierr);
		fprintf(ofp, "\n");
	}
	if (rhetmode != 0) {
		fprintf(ofp, "\nRates and their respective probabilities used in the likelihood function:\n");
		fprintf(ofp, "\n Category  Relative rate  Probability\n");
		if (rhetmode == 2 || rhetmode == 3)
			fprintf(ofp, "  0         0.0000         %.4f\n", fracinv);
		for (i = 0; i < numcats; i++)
			fprintf(ofp, "  %d         %.4f         %.4f\n",
				i+1, Rates[i], (1.0-fracinv)/(double) numcats);	
	}
	if (rhetmode == 1 || rhetmode == 3) {
		fprintf(ofp, "\nCategories 1-%d approximate a continous ", numcats);
		fprintf(ofp, "Gamma-distribution with expectation 1\n"); 
		fprintf(ofp, "and variance ");
		if (Geta == 1.0) fprintf(ofp, "infinity");
		else fprintf(ofp, "%.2f", Geta/(1.0-Geta));
		fprintf(ofp, ".\n");
	}

	if (typ_optn == 0 && (puzzlemode == 0 || puzzlemode == 1))
		if (rhetmode != 0) {
			fprintf(ofp, "\nCombination of categories that contributes");
			fprintf(ofp, " the most to the likelihood\n");
			fprintf(ofp, "(computation done without clock assumption assuming ");
			if (puzzlemode == 0) fprintf(ofp, "quartet-puzzling tree");
			if (puzzlemode == 1) fprintf(ofp, "1st user tree");
			fprintf(ofp, "):\n\n");
			printbestratecombination(ofp);
		}
		
	fprintf(ofp, "\n\nSEQUENCES IN INPUT ORDER\n\n");
	fail = FALSE;
	fprintf(ofp, "              5%% chi-square test  p-value\n");
	for (i = 0; i < Maxspc; i++) {
		fprintf(ofp, " ");
		fputid10(ofp, i);
		pval = homogentest(i);
		if ( pval < 0.05 ) fprintf(ofp, "        failed       ");
		else fprintf(ofp, "        passed       ");
		if (chi2fail) fail = TRUE;
		fprintf(ofp, "  %2.2f%%  ", pval*100.0);
		if (puzzlemode == 0 && typ_optn == 0)
			fprintf(ofp, "  [%lu] ", badtaxon[i]);
		fprintf(ofp, "\n");	
	}
	fprintf(ofp, "\n");
	fprintf(ofp, "The chi-square tests compares the ");
	if (data_optn == 1)
		fprintf(ofp, "amino acid");
	else if (data_optn == 0 && SH_optn)
		fprintf(ofp, "doublet");
	else if (data_optn == 0 && nuc_optn)
		fprintf(ofp, "nucleotide");
	else if (data_optn == 2)
		fprintf(ofp, "binary state");
	fprintf(ofp," composition of each sequence\n");
	fprintf(ofp, "to the frequency distribution assumed in the maximum likelihood model.\n");	
	if (fail) {
		fprintf(ofp, "\nWarning: Result of chi-square test may not be valid");
		fprintf(ofp, " because of small\nmaximum likelihood frequencies and");
		fprintf(ofp, " short sequence length!\n");
	}
	if (puzzlemode == 0 &&typ_optn == 0) {
		fprintf(ofp, "The number in square brackets indicates how often each sequence is\n");
		fprintf(ofp, "involved in one of the %lu completely unresolved quartets of the\n", badqs);
		fprintf(ofp, "quartet puzzling tree search.");
	}
	fprintf(ofp, "\n\nIDENTICAL SEQUENCES\n\n");
	fprintf(ofp, "The sequences in each of the following groups are all identical. To speed\n");
	fprintf(ofp, "up computation please remove all but one of each group from the data set.\n\n");
	findidenticals(ofp);
	fprintf(ofp, "\n\nMAXIMUM LIKELIHOOD DISTANCES\n\n");
	fprintf(ofp, "Maximum likelihood distances are computed using the ");
	fprintf(ofp, "selected model of\nsubstitution and rate heterogeneity.\n\n");
	putdistance(ofp);
	fprintf(ofp, "\nAverage distance (over all possible pairs of sequences):  %.5f\n",
		averagedist() / 100.0);

	if (typ_optn == 0) {
	
		fprintf(ofp, "\n\nTREE SEARCH\n\n");
		if (puzzlemode == 0) {
			fprintf(ofp, "Quartet puzzling is used to choose from the possible tree topologies\n");
			fprintf(ofp, "and to simultaneously infer support values for internal branches.\n\n");
			fprintf(ofp, "Number of puzzling steps: %lu\n", Numtrial);
			fprintf(ofp, "Analysed quartets: %lu\n", Numquartets);
			fprintf(ofp, "Unresolved quartets: %lu (= %.1f%%)\n",
				badqs, (double) badqs / (double) Numquartets * 100);	
			fprintf(ofp, "\nQuartet trees are based on %s maximum likelihood values\n",
				(approxqp ? "approximate" : "exact"));
			fprintf(ofp, "using the selected model of substitution and rate heterogeneity.\n\n\n");
		}
		if (puzzlemode == 1) {
			fprintf(ofp, "%d tree topologies were specified by the user.\n", numutrees);		
		}
		if (puzzlemode == 2) {
			fprintf(ofp, "No tree search performed (maximum likelihood distances only).\n");
		}

		if (puzzlemode == 0) {
			fprintf(ofp, "QUARTET PUZZLING TREE\n\n");
			fprintf(ofp, "Support for the internal branches of the unrooted quartet puzzling\n");
			fprintf(ofp, "tree topology is shown in percent.\n");
			if (consincluded == Maxspc - 3)
				fprintf(ofp,"\nThis quartet puzzling tree is completely resolved.\n");
			else
				fprintf(ofp,"\nThis quartet puzzling tree is not completely resolved!\n");
			fprintf(ofp, "\n\n");
			plotconsensustree(ofp);
			fprintf(ofp, "\n\nQuartet puzzling tree (in CLUSTAL W notation):\n\n");
			writeconsensustree(ofp);
			fprintf(ofp, "\n\nBIPARTITIONS\n\n");
			fprintf(ofp, "The following bipartitions occured at least once");
			fprintf(ofp, " in all intermediate\ntrees that have been generated ");
			fprintf(ofp, "in the %lu puzzling steps:\n\n", Numtrial);
			fprintf(ofp, "Bipartitions included in the quartet puzzling tree:\n");
			fprintf(ofp,
				"(bipartition with sequences in input order : number of times seen)\n\n");
			for (li = 0; li < consincluded; li++) {
				fprintf(ofp, " ");
				printsplit(ofp, splitfreqs[2*li+1]);
				fprintf(ofp, "  :  %lu\n", splitfreqs[2*li]);
			}
			if (consincluded == 0) fprintf(ofp, " None (no bipartition included)\n");
			fprintf(ofp, "\nBipartitions not included in the quartet puzzling tree:\n");
			fprintf(ofp,
				"(bipartition with sequences in input order : number of times seen)\n\n");
						
			if (consincluded == numbiparts) {
				fprintf(ofp, " None (all bipartitions are included)\n");
			} else {
				/* print first 20 bipartions not included */				
				for (li = consincluded; (li < numbiparts) && (li < consincluded + 20UL); li++) {
					fprintf(ofp, " ");
					printsplit(ofp, splitfreqs[2*li+1]);
					fprintf(ofp, "  :  %lu\n", splitfreqs[2*li]);
				}
				if ((li == consincluded + 20UL) && (li != numbiparts)) 
					fprintf(ofp, "\n(%lu other less frequent bipartitions not shown)\n",
						numbiparts - consincluded - 20UL);			
			}	
		}
	
		if (puzzlemode == 0) {
			fprintf(ofp, "\n\nMAXIMUM LIKELIHOOD BRANCH LENGTHS ON QUARTET");
			fprintf(ofp, " PUZZLING TREE (NO CLOCK)\n\nBranch lengths are computed using");
			fprintf(ofp, " the selected model of\nsubstitution and rate heterogeneity.\n\n\n");
			clockmode = 0; /* nonclocklike branch lengths */
			prtopology(ofp);
			fprintf(ofp, "\n");
			resulttree(ofp);
			fprintf(ofp, "\n\nQuartet puzzling tree with maximum likelihood branch lengths");
			fprintf(ofp, "\n(in CLUSTAL W notation):\n\n");
			fputphylogeny(ofp);			
			if (compclock) {
				fprintf(ofp, "\n\nMAXIMUM LIKELIHOOD BRANCH LENGTHS OF QUARTET");
				fprintf(ofp, " PUZZLING TREE (WITH CLOCK)\n\nBranch lengths are computed using");
				fprintf(ofp, " the selected model of\nsubstitution and rate heterogeneity.\n");
				fprintf(ofp, "\nRoot located at branch: %d  ", locroot+1);
				if (rootsearch == 0) fprintf(ofp, "(user specified)\n\n\n");
				if (rootsearch == 1) {
					fprintf(ofp, "(automatic search)");
					if (numbestroot > 1) fprintf(ofp, "- WARNING: %d best locations found! -", numbestroot);
					fprintf(ofp, "\n\n");	
					fprintf(ofp, "If the automatic search misplaces the root please rerun the analysis\n");
					fprintf(ofp, "(rename \"outtree\" to \"intree\") and select location of root manually!");
					fprintf(ofp, "\n\n\n");
				}
				if (rootsearch == 2) fprintf(ofp, "(displayed outgroup)\n\n\n");
				clockmode = 1; /* clocklike branch lengths */
				prtopology(ofp);
				fprintf(ofp, "\n");
				fprintf(ofp, "\nTree drawn as unrooted tree for better ");
				fprintf(ofp, "comparison with non-clock tree!\n");
				resulttree(ofp);
				fprintf(ofp, "\n");
				resultheights(ofp);
				fprintf(ofp, "\n\nRooted quartet puzzling tree with clocklike");
				fprintf(ofp, " maximum likelihood branch lengths\n");
				fprintf(ofp, "(in CLUSTAL W notation):\n\n");
				fputrooted(ofp, locroot);
			}
			
			if (compclock) {
				fprintf(ofp, "\n\nMOLECULAR CLOCK LIKELIHOOD RATIO TEST\n\n");
				fprintf(ofp, "log L without clock: %.2f (independent branch parameters: %d)\n",
					Ctree->lklhd, Numspc + Numibrnch);
				fprintf(ofp, "log L with clock:    %.2f (independent branch parameters: %d)\n\n",
					Ctree->lklhdc, Numhts + 1);
				delta = 2.0*((Ctree->lklhd) - (Ctree->lklhdc));
				fprintf(ofp, "Likelihood ratio test statistic delta: %.2f\n", delta);	
				df = Numspc + Numibrnch - Numhts - 1;
				fprintf(ofp, "Degress of freedom of chi-square distribution: %d\n", df);
				
				pval = IncompleteGammaQ(df*0.5, delta*0.5);
				
				fprintf(ofp, "Critical significance level: %.2f%%\n\n", pval*100.0);
				if (pval >= 0.05) {
					fprintf(ofp, "The simpler (clocklike) tree can not be rejected on a significance\n");
					fprintf(ofp, "level of 5%%. The log-likelihood of the more complex (no clock) tree\n");
					fprintf(ofp, "is not significantly increased.\n");
				} else {
					fprintf(ofp, "The simpler (clocklike) tree is rejected on a significance level\n");
					fprintf(ofp, "of 5%%. The log-likelihood of the more complex (no clock) tree is\n");
					fprintf(ofp, "significantly increased.\n");
				}
				fprintf(ofp, "\nPlease take care that the correct root is used!\n");
			}
				
		}
	}
	
	if (typ_optn == 1) {
	
			fprintf(ofp, "\n\nLIKELIHOOD MAPPING ANALYSIS\n\n");
			fprintf(ofp, "Number of quartets: %lu", Numquartets);
			if (lmqts == 0) fprintf(ofp, " (all possible)\n");
			else fprintf(ofp, " (random choice)\n");
			fprintf(ofp, "\nQuartet trees are based on approximate maximum likelihood values\n");
			fprintf(ofp, "using the selected model of substitution and rate heterogeneity.\n\n\n");
			if (numclust == 1) {
				fprintf(ofp, "Sequences are not grouped in clusters.\n");
			} else {
				fprintf(ofp, "Sequences are grouped in %d clusters.\n", numclust);
				fprintf(ofp, "\nCluster a: %d sequences\n\n", clustA);
				for (i = 0; i < clustA; i++) {
					fprintf(ofp, " ");
					fputid(ofp, clusterA[i]);
					fprintf(ofp, "\n");
				}
				fprintf(ofp, "\nCluster b: %d sequences\n\n", clustB);
				for (i = 0; i < clustB; i++) {
					fprintf(ofp, " ");
					fputid(ofp, clusterB[i]);
					fprintf(ofp, "\n");
				}
				if (numclust > 2) {
					fprintf(ofp, "\nCluster c: %d sequences\n\n", clustC);
					for (i = 0; i < clustC; i++) {
						fprintf(ofp, " ");
						fputid(ofp, clusterC[i]);
						fprintf(ofp, "\n");
					}
				}
				if (numclust == 4) {
					fprintf(ofp, "\nCluster d: %d sequences\n\n", clustD);
					for (i = 0; i < clustD; i++) {
						fprintf(ofp, " ");
						fputid(ofp, clusterD[i]);
						fprintf(ofp, "\n");
					}
				}
				fprintf(ofp, "\nQuartets of sequences used in the likelihood");
				fprintf(ofp, " mapping analysis are generated\n");
				if (numclust == 2)
					fprintf(ofp, "by drawing two sequences from cluster a and two from cluster b.");
				if (numclust == 3)
					fprintf(ofp, "by drawing one sequence from clusters a and b and two from cluster c.");
				if (numclust == 4)
					fprintf(ofp, "by drawing one sequence from each of the clusters a, b, c, and d.");
			}

			fprintf(ofp, "\n\nLIKELIHOOD MAPPING STATISTICS\n\n");
			fprintf(ofp, "Occupancies of the three areas 1, 2, 3:\n\n");
			if (numclust == 4)
				fprintf(ofp, "                    (a,b)-(c,d)\n");
			if (numclust == 3)
				fprintf(ofp, "                    (a,b)-(c,c)\n");
			if (numclust == 2)
				fprintf(ofp, "                    (a,a)-(b,b)\n");
			fprintf(ofp, "                        /\\\n");
			fprintf(ofp, "                       /  \\\n");
			fprintf(ofp, "                      /    \\\n");
			fprintf(ofp, "                     /   1  \\\n");
			fprintf(ofp, "                    / \\    / \\\n");
			fprintf(ofp, "                   /   \\  /   \\\n");
			fprintf(ofp, "                  /     \\/     \\\n");
			fprintf(ofp, "                 /  3    :   2  \\\n");
			fprintf(ofp, "                /        :       \\\n");
			fprintf(ofp, "               /__________________\\\n");
			if (numclust == 4)
				fprintf(ofp, "     (a,d)-(b,c)                  (a,c)-(b,d)\n");
			if (numclust == 3)
				fprintf(ofp, "     (a,c)-(b,c)                  (a,c)-(b,c)\n");
			if (numclust == 2)
				fprintf(ofp, "     (a,b)-(a,b)                  (a,b)-(a,b)\n");
			fprintf(ofp, "\n");
			fprintf(ofp, "Number of quartets in region 1: %lu (= %.1f%%)\n",
			 	ar1, (double) ar1*100.0/Numquartets);
			fprintf(ofp, "Number of quartets in region 2: %lu (= %.1f%%)\n",
				ar2, (double) ar2*100.0/Numquartets);
			fprintf(ofp, "Number of quartets in region 3: %lu (= %.1f%%)\n\n",
				ar3, (double) ar3*100.0/Numquartets);
			fprintf(ofp, "Occupancies of the seven areas 1, 2, 3, 4, 5, 6, 7:\n\n");
			if (numclust == 4)
				fprintf(ofp, "                    (a,b)-(c,d)\n");
			if (numclust == 3)
				fprintf(ofp, "                    (a,b)-(c,c)\n");
			if (numclust == 2)
				fprintf(ofp, "                    (a,a)-(b,b)\n");
			fprintf(ofp, "                        /\\\n");
			fprintf(ofp, "                       /  \\\n");
			fprintf(ofp, "                      /  1 \\\n");
			fprintf(ofp, "                     / \\  / \\\n");
			fprintf(ofp, "                    /   /\\   \\\n");
			fprintf(ofp, "                   / 6 /  \\ 4 \\\n");
			fprintf(ofp, "                  /   /  7 \\   \\\n");
			fprintf(ofp, "                 / \\ /______\\ / \\\n");
			fprintf(ofp, "                / 3  :   5  :  2 \\\n");
			fprintf(ofp, "               /__________________\\\n");
			if (numclust == 4)
				fprintf(ofp, "     (a,d)-(b,c)                  (a,c)-(b,d)\n");
			if (numclust == 3)
				fprintf(ofp, "     (a,c)-(b,c)                  (a,c)-(b,c)\n");
			if (numclust == 2)
				fprintf(ofp, "     (a,b)-(a,b)                  (a,b)-(a,b)\n");
			fprintf(ofp, "\n");
			fprintf(ofp, "Number of quartets in region 1: %lu (= %.1f%%)    left:   %lu   right: %lu\n",
				reg1, (double) reg1*100.0/Numquartets, reg1l, reg1r);
			fprintf(ofp, "Number of quartets in region 2: %lu (= %.1f%%)    bottom: %lu   top:   %lu\n",
				reg2, (double) reg2*100.0/Numquartets, reg2d, reg2u);
			fprintf(ofp, "Number of quartets in region 3: %lu (= %.1f%%)    bottom: %lu   top:   %lu\n",
				reg3, (double) reg3*100.0/Numquartets, reg3d, reg3u);
			fprintf(ofp, "Number of quartets in region 4: %lu (= %.1f%%)    bottom: %lu   top:   %lu\n",
				reg4, (double) reg4*100.0/Numquartets, reg4d, reg4u);
			fprintf(ofp, "Number of quartets in region 5: %lu (= %.1f%%)    left:   %lu   right: %lu\n",
				reg5, (double) reg5*100.0/Numquartets, reg5l, reg5r);
			fprintf(ofp, "Number of quartets in region 6: %lu (= %.1f%%)    bottom: %lu   top:   %lu\n",
				reg6, (double) reg6*100.0/Numquartets, reg6d, reg6u);
			fprintf(ofp, "Number of quartets in region 7: %lu (= %.1f%%)\n",
				reg7, (double) reg7*100.0/Numquartets);
	}
}


/* write current user tree to file */
void writecutree(FILE *ofp, int num)
{
	int df;
	double pval, delta;


	if (typ_optn == 0) {
	
		if (puzzlemode == 1) {
			fprintf(ofp, "\n\nMAXIMUM LIKELIHOOD BRANCH LENGTHS OF USER");
			fprintf(ofp, " DEFINED TREE # %d (NO CLOCK)\n\nBranch lengths are computed using", num);
			fprintf(ofp, " the selected model of\nsubstitution and rate heterogeneity.\n\n\n");
			clockmode = 0; /* nonclocklike branch lengths */
			prtopology(ofp);
			fprintf(ofp, "\n");
			resulttree(ofp);
			fprintf(ofp, "\n\nUnrooted user defined tree with maximum likelihood branch lengths");
			fprintf(ofp, "\n(in CLUSTAL W notation):\n\n");
			fputphylogeny(ofp);
			if (compclock) {
				fprintf(ofp, "\n\nMAXIMUM LIKELIHOOD BRANCH LENGTHS OF USER");
				fprintf(ofp, " DEFINED TREE # %d (WITH CLOCK)\n\nBranch lengths are computed using", num);
				fprintf(ofp, " the selected model of\nsubstitution and rate heterogeneity.\n");
				fprintf(ofp, "\nRoot located at branch: %d  ", locroot+1);
				if (rootsearch == 0) fprintf(ofp, "(user specified)\n\n\n");
				if (rootsearch == 1) {
					fprintf(ofp, "(automatic search)");
					if (numbestroot > 1) fprintf(ofp, "- WARNING: %d best locations found! -", numbestroot);
					fprintf(ofp, "\n\n");
					fprintf(ofp, "If the automatic search misplaces the root please rerun the analysis\n");
					fprintf(ofp, "and select location of root manually!");
					fprintf(ofp, "\n\n\n");

				}
				if (rootsearch == 2) fprintf(ofp, "(displayed outgroup)\n\n\n");
				clockmode = 1; /* clocklike branch lengths */
				prtopology(ofp);
				fprintf(ofp, "\n");
				resulttree(ofp);
				fprintf(ofp, "\n");
				resultheights(ofp);
				fprintf(ofp, "\n\nRooted user defined tree with clocklike ");
				fprintf(ofp, "maximum likelihood branch lengths\n");
				fprintf(ofp, "(in CLUSTAL W notation):\n\n");
				fputrooted(ofp, locroot);
			}
			
			if (compclock) {
				fprintf(ofp, "\n\nMOLECULAR CLOCK LIKELIHOOD RATIO TEST FOR USER TREE # %d\n\n", num);
				fprintf(ofp, "log L without clock: %.2f (independent branch parameters: %d)\n",
					Ctree->lklhd, Numspc + Numibrnch);
				fprintf(ofp, "log L with clock:    %.2f (independent branch parameters: %d)\n\n",
					Ctree->lklhdc, Numhts + 1);
				delta = 2.0*((Ctree->lklhd) - (Ctree->lklhdc));
				fprintf(ofp, "Likelihood ratio test statistic delta: %.2f\n", delta);	
				df = Numspc + Numibrnch - Numhts - 1;
				fprintf(ofp, "Degress of freedom of chi-square distribution: %d\n", df);
				
				pval = IncompleteGammaQ (df*0.5, delta*0.5);
				
				fprintf(ofp, "Critical significance level: %.2f%%\n\n", pval*100.0);
				if (pval >= 0.05) {
					fprintf(ofp, "The simpler (clocklike) tree can not be rejected on a significance\n");
					fprintf(ofp, "level of 5%%. The log-likelihood of the more complex (no clock) tree\n");
					fprintf(ofp, "is not significantly increased.\n");
				} else {
					fprintf(ofp, "The simpler (clocklike) tree is rejected on a significance level\n");
					fprintf(ofp, "of 5%%. The log-likelihood of the more complex (no clock) tree is\n");
					fprintf(ofp, "significantly increased.\n");
				}
				fprintf(ofp, "\nPlease take care that the correct root is used!\n");
			}			
		}	
	}
}



/* start timer */
void starttimer()
{
	time(&time0);
	time1 = time0;
}

/* check remaining time and print message if necessary */
void checktimer(uli numqts)
{
	double tc2, mintogo, minutes, hours;

	time(&time2);
	if ( (time2 - time1) > 900) { /* generate message every 15 minutes */
		/* every 900 seconds */
		/* percentage of completed quartets */
		if (mflag == 0) {
			mflag = 1;
			printf("\n");
		}
		tc2 = 100.*numqts/Numquartets;
		mintogo = (100.0-tc2) *
			(double) (time2-time0)/60.0/tc2;
		hours = floor(mintogo/60.0);
		minutes = mintogo - 60.0*hours;
		printf("%.2f%%", tc2);
		printf(" completed (remaining");
		printf(" time: %.0f", hours);
		printf(" hours %.0f", minutes);
		printf(" minutes)\n");
		time1 = time2;
	}

}


/* estimate parameters of substitution process and rate heterogeneity - no tree
   n-taxon tree is not needed because of quartet method or NJ tree topology */
void estimateparametersnotree()
{
	int it, nump, change;
	double TSold, YRold, FIold, GEold;

	it = 0;
	nump = 0;

	/* count number of parameters */
	if (data_optn == 0 && optim_optn) nump++;
	if (fracinv_optim || grate_optim) nump++;

	do { /* repeat until nothing changes any more */
		it++;
		change = FALSE;

		/* optimize substitution parameters */
		if (data_optn == 0 && optim_optn) {
		
			TSold = TSparam;
			YRold = YRparam;
			

			/*
			 * optimize
			 */
			 
			printf("Optimizing missing substitution process parameters\n");

			if (qcalg_optn) { /* quartet sampling */
				optimseqevolparamsq();
			} else { /* NJ tree */
				tmpfp = tmpfile();
				njtree(tmpfp);				
				rewind(tmpfp);
				readusertree(tmpfp);
				closefile(tmpfp);
				optimseqevolparamst();				
			}
			
			computedistan(); /* update ML distances */
		
			/* same tolerance as 1D minimization */
			if ((fabs(TSparam - TSold) > 3.3*PEPS1) ||
				(fabs(YRparam - YRold) > 3.3*PEPS1)
			) change = TRUE;

		}

		/* optimize rate heterogeneity variables */
		if (fracinv_optim || grate_optim) {

			FIold = fracinv;
			GEold = Geta;


			/* 
			 * optimize 
			 */

			printf("Optimizing missing rate heterogeneity parameters\n");
			/* compute NJ tree */
			tmpfp = tmpfile();
			njtree(tmpfp);
			/* use NJ tree topology to estimate parameters */
			rewind(tmpfp);
			readusertree(tmpfp);
			closefile(tmpfp);

			optimrateparams();				
			computedistan(); /* update ML distances */


			/* same tolerance as 1D minimization */
			if ((fabs(fracinv - FIold) > 3.3*PEPS2) ||
				(fabs(Geta - GEold) > 3.3*PEPS2)
			) change = TRUE;
			
		}

		if (nump == 1) return;
		
	} while (it != MAXITS && change);

	return;
}


/* estimate parameters of substitution process and rate heterogeneity - tree
   same as above but here the n-taxon tree is already in memory */
void estimateparameterstree()
{
	int it, nump, change;
	double TSold, YRold, FIold, GEold;

	it = 0;
	nump = 0;

	/* count number of parameters */
	if (data_optn == 0 && optim_optn) nump++;
	if (fracinv_optim || grate_optim) nump++;

	do { /* repeat until nothing changes any more */
		it++;
		change = FALSE;

		/* optimize substitution process parameters */
		if (data_optn == 0 && optim_optn) {
		
			TSold = TSparam;
			YRold = YRparam;
			

			/*
			 * optimize
			 */
			 
			printf("Optimizing missing substitution process parameters\n");
			optimseqevolparamst();
			computedistan(); /* update ML distances */

		
			/* same tolerance as 1D minimization */
			if ((fabs(TSparam - TSold) > 3.3*PEPS1) ||
				(fabs(YRparam - YRold) > 3.3*PEPS1)
			) change = TRUE;

		}

		/* optimize rate heterogeneity variables */
		if (fracinv_optim || grate_optim) {

			FIold = fracinv;
			GEold = Geta;


			/* 
			 * optimize 
			 */

			printf("Optimizing missing rate heterogeneity parameters\n");
			optimrateparams();				
			computedistan(); /* update ML distances */


			/* same tolerance as 1D minimization */
			if ((fabs(fracinv - FIold) > 3.3*PEPS2) ||
				(fabs(Geta - GEold) > 3.3*PEPS2)
			) change = TRUE;
			
		}

		if (nump == 1) return;
		
	} while (it != MAXITS && change);

	return;
}


/******************************************************************************/
/* main part                                                                  */
/******************************************************************************/

int main()
{	
	int i, a, a1, a2, b, b1, b2, c, c1, c2, d, ci, oldlocroot=0;
	uli nq;
	double tc2, mintogo, minutes, hours, logs[3], d1, d2, d3, temp;
	ivector qts, mlorder, gettwo;

	
	/* vectors used in QP and LM analysis */
	qweight = new_dvector(3);
	sqdiff = new_dvector(3);
	qworder = new_ivector(3);
	sqorder = new_ivector(3);
	
	printf("\n\n\nWELCOME TO PUZZLE %s!\n", VERSION);

	/* initialize random numbers generator */
	initrandom();

	/* get sequences */
	openfiletoread(&seqfp, INFILE, "sequence data");
	getsizesites(seqfp);
	printf("\n\n\nInput data set contains %d sequences of length %d\n", Maxspc, Maxseqc);
	getdataset(seqfp);
	closefile(seqfp);		
	data_optn = guessdatatype();
	
	/* translate characters into format used by ML engine */
	nuc_optn = TRUE; 
	SH_optn = FALSE;
	Seqchar = NULL;
	translatedataset();
	
	/* estimate base frequencies from data set */
	Freqtpm = NULL;
	Basecomp = NULL;
	estimatebasefreqs();
	
	/* guess model of substitution */
	guessmodel();

	/* check for user specified tree */
	if ((utfp = fopen(INTREE, "r")) != NULL) {
		fclose(utfp);
		puzzlemode = 1;
	} else {
		puzzlemode = 0;
	}

	/* reserve memory for cluster LM analysis */
	clusterA = new_ivector(Maxspc);
	clusterB = new_ivector(Maxspc);
	clusterC = new_ivector(Maxspc);
	clusterD = new_ivector(Maxspc);

	/* set options interactively */
	setoptions();

	/* open usertree file right after start */
	if (typ_optn == 0 && puzzlemode == 1) {
		openfiletoread(&utfp, INTREE, "user trees");
	}

	/* start main timer */
	time(&Starttime);

	/* symmetrize doublet frequencies if specified */
	symdoublets();

	/* initialise ML */
	mlstart();

	/* determine how many usertrees */
	if (typ_optn == 0 && puzzlemode == 1) {
		numutrees = 0;		
		do {
			ci = fgetc(utfp);
			if ((char) ci == ';') numutrees++;
		} while (ci != EOF);
		rewind(utfp);
		if (numutrees < 1) {
			printf("Unable to proceed (no tree in input tree file)\n\n\n");
			exit(1);
		}		
	}
	
	/* check fraction of invariable sites */
	if ((rhetmode == 2 || rhetmode == 3) && !fracinv_optim)
		/* fraction of invariable site was specified manually */
		if (fracinv > fracconst)
			fracinv = fracconst;

	/* estimate parameters */
	if (!(typ_optn == 0 && puzzlemode == 1)) {
		/* no tree present */
		estimateparametersnotree();
	} else {
		if (utree_optn) {
			/* use 1st user tree */
			readusertree(utfp);
			rewind(utfp);
			estimateparameterstree();
		} else {
			/* don't use first user tree */
			estimateparametersnotree();
		}
	}

	/* compute expected Ts/Tv ratio */
	if (data_optn == 0) computeexpectations();
	
	if (typ_optn == 0) { /* tree reconstruction */

		if (puzzlemode == 0) { /* quartet puzzling */
	
			/* allocate memory for taxon list of bad quartets */
			badtaxon = new_ulivector(Maxspc);
			for (i = 0; i < Maxspc; i++) badtaxon[i] = 0;

			/* allocate variable used for randomizing input order */
			trueID = new_ivector(Maxspc);
	
			/* allocate memory for quartets */
			mallocquartets(Maxspc);
	
			/* prepare for consensus tree analysis */
			initconsensus();
		
			/* compute quartets */
			printf("Computing quartet maximum likelihood trees\n");
			computeallquartets();
	
			printf("Computing quartet puzzling tree\n");
	
			/* start timer - percentage of completed trees */
			time(&time0);
			time1 = time0;
			mflag = 0;
			
			/* open file for list of puzzling step trees */
			if(listqptrees)
				openfiletowrite(&qptlist, OUTPTLIST, "puzzling step trees");
									
			for (Currtrial = 0; Currtrial < Numtrial; Currtrial++) {
			
				/* randomize input order */
				chooser(Maxspc, Maxspc, trueID);
				
				/* initialize tree */
				inittree();

				/* adding all other leafs */
				for (i = 3; i < Maxspc; i++) { 
					
					/* clear all edgeinfos */
					resetedgeinfo();
				
					/* clear counter of quartets */
					nq = 0;

					/*
					 * core of quartet puzzling algorithm
				 	*/

					for (a = 0; a < nextleaf - 2; a++)
						for (b = a + 1; b < nextleaf - 1; b++)
							for (c = b + 1; c < nextleaf; c++) {

								/*	check which two _leaves_ out of a, b, c
								    are closer related to each other than
								    to leaf i according to a least squares
								    fit of the continous Baysian weights to the
								    seven trivial "attractive regions". We assign 
								    a score of 1 to all edges between these two leaves
								    chooseA and chooseB */

								checkquartet(a, b, c, i);
								incrementedgeinfo(chooseA, chooseB);
								
								nq++;
							
								/* generate message every 15 minutes */
								
								/* check timer */
								time(&time2);
								if ( (time2 - time1) > 900) {
									/* every 900 seconds */
									/* percentage of completed trees */
									if (mflag == 0) {
										printf("\n");
										mflag = 1;
									}
									tc2 = 100.0*Currtrial/Numtrial + 
											100.0*nq/Numquartets/Numtrial;
									mintogo = (100.0-tc2) *
										(double) (time2-time0)/60.0/tc2;
									hours = floor(mintogo/60.0);
									minutes = mintogo - 60.0*hours;
									printf("%2.2f%%", tc2);
									printf(" completed (remaining");
									printf(" time: %.0f", hours);
									printf(" hours %.0f", minutes);
									printf(" minutes)\n");
									time1 = time2;
								}
							}

					/* find out which edge has the lowest edgeinfo */
					minimumedgeinfo();
				
					/* add the next leaf on minedge */
					addnextleaf(minedge);
				}
	
				/* compute bipartitions of current tree */
				computebiparts();
				makenewsplitentries();

				/* write tree */
				if (listqptrees)
					writetree(qptlist);

				/* free tree before building the next tree */
				freetree();
			}
			
			/* close file for list of puzzling step trees */
			if(listqptrees)
				closefile(qptlist);
			
			if (mflag == 1) printf("\n");

			/* garbage collection */
			free(splitcomp);
			free_ivector(trueID);
			free_cmatrix(biparts);
			freequartets();

			/* compute majority rule consensus tree */
			makeconsensus();
			
			/* write consensus tree to tmp file */
			tmpfp = tmpfile();
			writeconsensustree(tmpfp);
	
		} /* quartet puzzling */

	} /* tree reconstruction */
	
	if (typ_optn == 1) { /* likelihood mapping */
	
		/* reset variables */
		ar1 = ar2 = ar3 = 0;
		reg1 = reg2 = reg3 = reg4 = reg5 = reg6 = reg7 = 0;
		reg1l = reg1r = reg2u = reg2d = reg3u = reg3d = reg4u =
			reg4d = reg5l = reg5r = reg6u = reg6d = 0;
			
		/* place for random quartet */
			qts = new_ivector(4);

		/* initialize output file */
		openfiletowrite(&trifp, TRIANGLE, "Postscript output");
		initps(trifp);
		printf("Performing likelihood mapping analysis\n");
			
		/* start timer */
		starttimer();
		nq = 0;
		mflag = 0;

		if (lmqts == 0) { /* all possible quartets */
			
			if (numclust == 4) { /* four-cluster analysis */

				for (a = 0; a < clustA; a++)
					for (b = 0; b < clustB; b++)
						for (c = 0; c < clustC; c++)
							for (d = 0; d < clustD; d++) {
									
								nq++;
									
								/* check timer */
								checktimer(nq);

								/* maximum likelihood values */
								/* approximate ML is sufficient */
								d1 = quartet_alklhd(clusterA[a],clusterB[b],clusterC[c],clusterD[d]); /* (a,b)-(c,d) */
								d2 = quartet_alklhd(clusterA[a],clusterC[c],clusterB[b],clusterD[d]); /* (a,c)-(b,d) */
								d3 = quartet_alklhd(clusterA[a],clusterD[d],clusterB[b],clusterC[c]); /* (a,d)-(b,c) */
								
								/* draw point for LM analysis */
								makelmpoint(trifp, d1, d2, d3);

							}					
			}
				
			if (numclust == 3) { /* three-cluster analysis */

				gettwo = new_ivector(2);

				for (a = 0; a < clustA; a++)
					for (b = 0; b < clustB; b++)
						for (c1 = 0; c1 < clustC-1; c1++)
							for (c2 = c1+1; c2 < clustC; c2++) {

								nq++;

								/* check timer */
								checktimer(nq);
								
								/* maximum likelihood values */
								/* approximate ML is sufficient */
								d1 = quartet_alklhd(clusterA[a],clusterB[b],clusterC[c1],clusterC[c2]); /* (a,b)-(c,d) */
								d2 = quartet_alklhd(clusterA[a],clusterC[c1],clusterB[b],clusterC[c2]); /* (a,c)-(b,d) */
								d3 = quartet_alklhd(clusterA[a],clusterC[c2],clusterB[b],clusterC[c1]); /* (a,d)-(b,c) */
								
								/* randomize order of d2 and d3 */
								if (randominteger(2) == 1) {
									temp = d3;
									d3 = d2;
									d2 = temp;
								}
						
								/* draw point for LM analysis */
								makelmpoint(trifp, d1, d2, d3);

				}				
				free_ivector(gettwo);
			}
				
			if (numclust == 2) { /* two-cluster analysis */
					
				gettwo = new_ivector(2);

				for (a1 = 0; a1 < clustA-1; a1++)
					for (a2 = a1+1; a2 < clustA; a2++)
						for (b1 = 0; b1 < clustB-1; b1++)
							for (b2 = b1+1; b2 < clustB; b2++) {

								nq++;

								/* check timer */
								checktimer(nq);
								
								/* maximum likelihood values */
								/* approximate ML is sufficient */
								d1 = quartet_alklhd(clusterA[a1],clusterA[a2],clusterB[b1],clusterB[b2]); /* (a,b)-(c,d) */
								d2 = quartet_alklhd(clusterA[a1],clusterB[b1],clusterA[a2],clusterB[b2]); /* (a,c)-(b,d) */
								d3 = quartet_alklhd(clusterA[a1],clusterB[b2],clusterA[a2],clusterB[b1]); /* (a,d)-(b,c) */
								
								/* randomize order of d2 and d3 */
								if (randominteger(2) == 1) {
									temp = d3;
									d3 = d2;
									d2 = temp;
								}
						
								/* draw point for LM analysis */
								makelmpoint(trifp, d1, d2, d3);

				}
					
				free_ivector(gettwo);
			}
			
			if (numclust == 1) { /* normal likelihood mapping (one cluster) */
			
				mlorder = new_ivector(3);

				for (i = 3; i < Maxspc; i++) 
					for (a = 0; a < i - 2; a++) 
						for (b = a + 1; b < i - 1; b++) 
							for (c = b + 1; c < i; c++) {
							
								nq++;

								/* check timer */
								checktimer(nq);

								/* maximum likelihood values */
								/* approximate ML is sufficient */
								logs[0] = quartet_alklhd(a,b,c,i); /* (a,b)-(c,d) */
								logs[1] = quartet_alklhd(a,c,b,i); /* (a,c)-(b,d) */
								logs[2] = quartet_alklhd(a,i,b,c); /* (a,d)-(b,c) */

								/* randomize order */
								chooser(3,3,mlorder);
								d1 = logs[mlorder[0]];
								d2 = logs[mlorder[1]];
								d3 = logs[mlorder[2]];
								
								/* draw point for LM analysis */
								makelmpoint(trifp, d1, d2, d3);
				
				}
				free_ivector(mlorder);
			}
			
		} else { /* randomly selected quartets */
			
			if (numclust == 4) { /* four-cluster analysis */

				for (lmqts = 0; lmqts < Numquartets; lmqts++) {
				
					nq++;

					/* check timer */
					checktimer(nq);

					/* choose random quartet */	
					qts[0] = clusterA[ randominteger(clustA) ];
					qts[1] = clusterB[ randominteger(clustB) ];
					qts[2] = clusterC[ randominteger(clustC) ];
					qts[3] = clusterD[ randominteger(clustD) ];			

					/* maximum likelihood values */
					/* approximate ML is sufficient */
					d1 = quartet_alklhd(qts[0],qts[1],qts[2],qts[3]); /* (a,b)-(c,d) */
					d2 = quartet_alklhd(qts[0],qts[2],qts[1],qts[3]); /* (a,c)-(b,d) */
					d3 = quartet_alklhd(qts[0],qts[3],qts[1],qts[2]); /* (a,d)-(b,c) */
					
					/* draw point for LM analysis */
					makelmpoint(trifp, d1, d2, d3);

				}
			}
				
			if (numclust == 3) { /* three-cluster analysis */

				gettwo = new_ivector(2);

				for (lmqts = 0; lmqts < Numquartets; lmqts++) {
				
					nq++;

					/* check timer */
					checktimer(nq);

					/* choose random quartet */	
					qts[0] = clusterA[ randominteger(clustA) ];
					qts[1] = clusterB[ randominteger(clustB) ];
					chooser(clustC, 2, gettwo);
					qts[2] = clusterC[gettwo[0]];
					qts[3] = clusterC[gettwo[1]];		

					/* maximum likelihood values */
					/* approximate ML is sufficient */
					d1 = quartet_alklhd(qts[0],qts[1],qts[2],qts[3]); /* (a,b)-(c,d) */
					d2 = quartet_alklhd(qts[0],qts[2],qts[1],qts[3]); /* (a,c)-(b,d) */
					d3 = quartet_alklhd(qts[0],qts[3],qts[1],qts[2]); /* (a,d)-(b,c) */
					
					/* order of d2 and d3 is already randomized! */
						
					/* draw point for LM analysis */
					makelmpoint(trifp, d1, d2, d3);

				}
					
				free_ivector(gettwo);
			}
				
			if (numclust == 2) { /* two-cluster analysis */

				gettwo = new_ivector(2);

				for (lmqts = 0; lmqts < Numquartets; lmqts++) {
				
					nq++;

					/* check timer */
					checktimer(nq);

					/* choose random quartet */	
					chooser(clustA, 2, gettwo);
					qts[0] = clusterA[gettwo[0]];
					qts[1] = clusterA[gettwo[1]];		
					chooser(clustB, 2, gettwo);
					qts[2] = clusterB[gettwo[0]];
					qts[3] = clusterB[gettwo[1]];		

					/* maximum likelihood values */
					/* approximate ML is sufficient */
					d1 = quartet_alklhd(qts[0],qts[1],qts[2],qts[3]); /* (a,b)-(c,d) */
					d2 = quartet_alklhd(qts[0],qts[2],qts[1],qts[3]); /* (a,c)-(b,d) */
					d3 = quartet_alklhd(qts[0],qts[3],qts[1],qts[2]); /* (a,d)-(b,c) */
					
					/* order of d2 and d3 is already randomized! */
						
					/* draw point for LM analysis */
					makelmpoint(trifp, d1, d2, d3);

				}
				free_ivector(gettwo);
			}
				
			if (numclust == 1) { /* normal likelihood mapping (one cluster) */
			
				for (lmqts = 0; lmqts < Numquartets; lmqts++) {
				
					nq++;

					/* check timer */
					checktimer(nq);

					/* choose random quartet */	
					chooser(Maxspc, 4, qts);				

					/* maximum likelihood values */
					/* approximate ML is sufficient */
					d1 = quartet_alklhd(qts[0],qts[1],qts[2],qts[3]); /* (a,b)-(c,d) */
					d2 = quartet_alklhd(qts[0],qts[2],qts[1],qts[3]); /* (a,c)-(b,d) */
					d3 = quartet_alklhd(qts[0],qts[3],qts[1],qts[2]); /* (a,d)-(b,c) */
					
					/* order of d1, d2, and d3 is already randomized! */
				
					/* draw point for LM analysis */
					makelmpoint(trifp, d1, d2, d3);

				}
			}
		}

		finishps(trifp);
		closefile(trifp);
		free_ivector(qts);

	}  /* likelihood mapping */

	free_cmatrix(Seqchar);
	free_cmatrix(seqchars);

	/* reserve memory for tree statistics */
	if (typ_optn == 0 && puzzlemode == 1 && numutrees > 1) {
		ulkl = new_dvector(numutrees);
		allsites = new_dmatrix(numutrees,Numptrn);
		if (compclock) {
			ulklc = new_dvector(numutrees);
			allsitesc = new_dmatrix(numutrees,Numptrn);
		}
	}

	/* write distance matrix */
	openfiletowrite(&dfp, DISTANCES, "pairwise distances");
	putdistance(dfp);
	closefile(dfp);

	/* compute ML branch lengths for QP tree and for 1st user tree */
	if (typ_optn == 0 && (puzzlemode == 0 || puzzlemode == 1)) {

		/* open outtree file */
		openfiletowrite(&tfp, TREEFILE, "output tree(s)");

		/* read QP tree */
		if (puzzlemode == 0) {
			rewind(tmpfp);
			readusertree(tmpfp);			
			closefile(tmpfp);
			printf("Computing maximum likelihood branch lengths (without clock)\n");			
			usertree_lklhd();
		}
		
		/* read user tree */
		if (puzzlemode == 1) {
			readusertree(utfp);
			printf("Computing maximum likelihood branch lengths (without clock) for tree # 1\n");			
			usertree_lklhd();
		    if (numutrees > 1) {
		    	ulkl[0] = Ctree->lklhd;
		    	allsitelkl(Ctree->condlkl, allsites[0]);
		    }
		}

		/* find most probable combination of rate categories */
		findbestratecombination();

		if (compclock) { /* clocklike branch length */
		
			if (puzzlemode == 0)
				printf("Computing maximum likelihood branch lengths (with clock)\n");
			if (puzzlemode == 1)
				printf("Computing maximum likelihood branch lengths (with clock) for tree # 1\n");
							
			/* find best place for root */
			rootsearch = 0;
			oldlocroot = locroot;
			if (locroot < 0) {
				locroot = findrootedge();
				rootsearch = 1;
			}
			/* if user-specified edge for root does not exist use displayed outgroup */
			if (!checkedge(locroot)) {
				locroot = outgroup; 
				rootsearch = 2;
			}
			/* compute likelihood */
			clock_lklhd(locroot);
			if (numutrees > 1) {
		    	ulklc[0] = Ctree->lklhdc;
		    	allsitelkl(Ctree->condlkl, allsitesc[0]);
		    }

		}
		
		/* write ML branch length tree to outree file */
		clockmode = 0; /* nonclocklike branch lengths */
		fputphylogeny(tfp);
		
		/* clocklike branch lengths */
		if (compclock) {
			clockmode = 1;
			fputrooted(tfp, locroot);
		}
	}

	/* write first part of outfile */
	openfiletowrite(&ofp, OUTFILE, "general output");
	writeoutputfile(ofp);
	writecutree(ofp, 1);

	/* compute branch lengths for the other user trees */
	if (typ_optn == 0 && puzzlemode == 1) {
		for (i = 1; i < numutrees; i++) {
			readusertree(utfp);
			printf("Computing maximum likelihood branch lengths (without clock) for tree # %d\n", i+1);			
			usertree_lklhd();
			if (numutrees > 1) {
				ulkl[i] = Ctree->lklhd;
				allsitelkl(Ctree->condlkl, allsites[i]);
			}
			if (compclock) { /* clocklike branch length */
				printf("Computing maximum likelihood branch lengths (with clock) for tree # %d\n", i+1);
									
				/* find best place for root */
				rootsearch = 0;
				locroot = oldlocroot;
				if (locroot < 0) {
					locroot = findrootedge();
					rootsearch = 1;
				}
				/* if user-specified edge for root does not exist use displayed outgroup */
				if (!checkedge(locroot)) {
					locroot = outgroup; 
					rootsearch = 2;
				}
				/* compute likelihood */
				clock_lklhd(locroot);
				if (numutrees > 1) {
					ulklc[i] = Ctree->lklhdc;
					allsitelkl(Ctree->condlkl, allsitesc[i]);
				}

			}

			/* write ML branch length tree to outree file */
			clockmode = 0; /* nonclocklike branch lengths */
			fputphylogeny(tfp);
		
			/* clocklike branch lengths */
			if (compclock) {
				clockmode = 1;
				fputrooted(tfp, locroot);
			}
			
			writecutree(ofp, i+1);			
		}
	}

	/* close trees */
	if (typ_optn == 0 && (puzzlemode == 0 || puzzlemode == 1)) {
		closefile(tfp);
		if (puzzlemode == 1) closefile(utfp);
	}
	
	/* print tree statistics */
	if (typ_optn == 0 && puzzlemode == 1 && numutrees > 1)
		printtreestats(ofp);
	
	/* free memory for tree statistics */
	if (typ_optn && puzzlemode == 1 && numutrees > 1) {
		free_dvector(ulkl);
		free_dmatrix(allsites);
		if (compclock) {
			free_dvector(ulklc);
			free_dmatrix(allsitesc);
		}
	}
	
	/* stop timer */
	time(&Stoptime);
	timestamp(ofp);
	closefile(ofp);

	mlfinish();

	printf("\nAll results written to disk (%s, %s", OUTFILE, DISTANCES);
	
	if (typ_optn == 0 && puzzlemode != 2) printf(", %s", TREEFILE);
	if (listqptrees && typ_optn == 0 && puzzlemode == 0) printf(", %s", OUTPTLIST);	
	if (show_optn && typ_optn == 0 && puzzlemode == 0) printf(", %s", UNRESOLVED);
	if (typ_optn == 1) printf(", %s", TRIANGLE);
	printf(")\n");

	/* runtime message */
	printf(
		"This run took %lu seconds (= %.1f minutes = %.1f hours)\n\n",
			(uli) (Stoptime - Starttime), (Stoptime - Starttime)/60.,
			(Stoptime - Starttime)/3600.);

	/* free memory */
	if (puzzlemode == 0 && typ_optn == 0) {
		free(splitfreqs);
		free(splitpatterns);
		free(splitsizes);
		free_ivector(consconfid);
		free_ivector(conssizes);
		free_cmatrix(consbiparts);
		free_ulivector(badtaxon);
	}
	free_cmatrix(Identif);
	free_dvector(Freqtpm);
	free_imatrix(Basecomp);
	free_ivector(clusterA);
	free_ivector(clusterB);
	free_ivector(clusterC);
	free_ivector(clusterD);
	free_dvector(qweight);
	free_dvector(sqdiff);
	free_ivector(qworder);
	free_ivector(sqorder);

	return 0;
}

/* compare function for uli - sort largest numbers first */
int ulicmp(const void *ap, const void *bp)
{
	uli a, b;
	
	a = *((uli *) ap);
	b = *((uli *) bp);

	if (a > b) return -1;
	else if (a < b) return 1;
	else return 0;
}

/* compare function for int - sort smallest numbers first */
int intcmp(const void *ap, const void *bp)
{
	int a, b;
	
	a = *((int *) ap);
	b = *((int *) bp);

	if (a < b) return -1;
	else if (a > b) return 1;
	else return 0;
}
