/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdarg.h>

#include "dcc.h"
#include "send.h"
#include "dialogs.h"
#include "util.h"
#include "ch_utils.h"
#include "msg_utils.h"
#include "irc.h"
#include "bezerk.h"
#include "debug.h"

GSList *dcc_connections=NULL;

extern messageData irc_message;

extern GdkColor colour_red;
extern GdkColor colour_yellow;
extern GdkColor colour_blue;
extern GdkColor colour_grey;
extern GdkColor colour_green;
extern GdkColor colour_white;

/* ------------------------------------------------------------
 * create_connection
 *
 *  Create a new DCC_Connection object and set up it's
 *  default values
 *
 * Arguments:
 *	"type" - the connection type (e.g. DCC_SEND)
 *
 * Results:
 *	The object created is returned
 *
 * Side effects:
 *
 * ----------------------------------------------------------- */

DCC_Connection *create_connection(DCC_Type type)
{
  DCC_Connection *new_connection;

  bs_function_enter();

  new_connection = g_malloc(sizeof(DCC_Connection));
  new_connection->type = type;
  new_connection->sent = 0;
  new_connection->acked = 0;
  new_connection->file_name = NULL;
  new_connection->nick = NULL;
  new_connection->sd = -1;
  new_connection->fd = -1;
  new_connection->tag = -1;
  new_connection->progress_dialog = NULL;
  new_connection->buff = NULL;

  if (type == DCC_TYPE_CHAT) {
    new_connection->buff = g_malloc(BUFFLEN);
    new_connection->recv_count = 0;
    new_connection->message_window = NULL;
  }

  bs_function_leave();
  return(new_connection);
}

/* ------------------------------------------------------------
 * check_filetype
 *
 *  check filename is valid for DCC transfer
 *
 * Arguments:
 *	"dcc_connection" - the DCC connection object for this connection
 *	"filename" - the fully qualified name of the file to check
 *
 * Results:
 *	TRUE or FALSE value indicating suitability of the file
 *
 * Side effects:
 *	"dcc_connection" has it's file_stat component filled in
 *
 * ----------------------------------------------------------- */

int check_filetype(DCC_Connection *dcc_connection, char *filename)
{

  bs_function_enter();
  printf("check_filetype %s\n", filename);
  if (dcc_connection->type == DCC_TYPE_SEND) {
    printf("check_filetype %s\n", filename);
    if (stat(filename, &dcc_connection->file_stat) == -1) {
      printf("%s\n", filename);
      bs_function_leave();
      return(FALSE);
    }
    if ( !S_ISREG(dcc_connection->file_stat.st_mode) ) {
      printf("2\n");
      bs_function_leave();
      return(FALSE);
    }
    if ( dcc_connection->file_stat.st_size == 0L ) {
      printf("3\n");
      bs_function_leave();
      return(FALSE);
    }
    if ( !strncmp("/etc/", filename, 5) ) {
      printf("4\n");
      bs_function_leave();
      return(FALSE);
    }
  } else if (dcc_connection->type == DCC_TYPE_GET) {
    char cwd[BUFFLEN];
    char *file_ptr, *new_filename;
    int pre_count = 0;

    if ( getcwd(cwd, BUFFLEN) == NULL ) {
      bs_function_leave();
      return(FALSE);
    }

    /* strip out any path information */
    if ( (file_ptr = strrchr(filename, '/')) == NULL) {
      file_ptr = filename;
    } else {
      file_ptr++;
    }
    file_ptr = g_strdup(file_ptr);

    /* don't allow dot files */
    if ( *file_ptr == '.' ) {
      file_ptr[0] = 'a';
    }

    /* make sure the file doesn't exist */
    while (stat(file_ptr, &dcc_connection->file_stat) == 0) {
      /* file exists */
      new_filename = g_malloc(strlen(file_ptr)+2);
      sprintf(new_filename, "a%s", file_ptr);
      g_free(file_ptr);
      file_ptr = new_filename;
      if ( (++pre_count) > 20) {
	bs_function_leave();
	return(FALSE);
      }
    }
    new_filename = g_malloc(strlen(cwd)+strlen(file_ptr)+2);
    sprintf(new_filename, "%s/%s", cwd, file_ptr);
    if (dcc_connection->file_name) {
      g_free(dcc_connection->file_name);
      dcc_connection->file_name = new_filename;
    }
  }

  bs_function_leave();
  return(TRUE);
}

/* ------------------------------------------------------------
 * dcc_send_data
 *
 *  Send the next block of data in a DCC Send connection
 *
 * Arguments:
 *	"dcc_connection" - the DCC Connection object for the transfer
 *
 * Results:
 *
 * Side effects:
 *	The file pointer of the file being transfered changes
 *	dcc_connection->sent is changed
 *		or
 *	the connection is terminated on a file or socket error
 * ----------------------------------------------------------- */

void dcc_send_data(DCC_Connection *dcc_connection)
{
  char buff[DCC_BUFFLEN];
  int len, sent=0, n;

  bs_function_enter();

  if ( (len = read(dcc_connection->fd, buff, 1024)) < 0 ) {
    /* This will in turn call dcc_send_cancel */
    progress_cancel(NULL, dcc_connection->progress_dialog);
    bs_function_leave();
    return;
  }    

  do {
    if ( ( n = send(dcc_connection->sd, buff+sent, len-sent, 0) ) < 0) {
      if (errno == EWOULDBLOCK || errno == EAGAIN) {
	continue;
      }
      /* This will in turn call dcc_send_cancel */
      progress_cancel(NULL, dcc_connection->progress_dialog);
      bs_function_leave();
      return;
    }
    sent += n;
  } while (sent != len);

  dcc_connection->sent += len;

  bs_function_leave();
  return;
}

/* ------------------------------------------------------------
 * dcc_send_ready
 *
 *  Called when a DCC Send socket is ready for reading
 *
 * Arguments:
 *	"dcc_connection" - the DCC Connection object for the transfer
 *
 * Results:
 *
 * Side effects:
 *	connection is terminated on a socket error
 *		or
 *	dcc_connection->acked is changed
 *
 * ----------------------------------------------------------- */

gint dcc_send_ready(DCC_Connection *dcc_connection)
{
  guint32 bytes_acked;
  int count;

  bs_function_enter();

  if ( (count = recv(dcc_connection->sd, (char *)&bytes_acked, sizeof(guint32), 0)) < 4) {
    if (errno == EWOULDBLOCK || errno == EAGAIN) {
      bs_function_leave();
      return(TRUE);
    } else {
      /* This will in turn call dcc_send_cancel */
      progress_cancel(NULL, dcc_connection->progress_dialog);
      bs_function_leave();
      return(TRUE);
    }
  }

  dcc_connection->acked = ntohl(bytes_acked);

  progress_update( (float) dcc_connection->acked/dcc_connection->file_stat.st_size, dcc_connection->progress_dialog);

  if ( dcc_connection->acked == dcc_connection->file_stat.st_size ) {
    /* This will in turn call dcc_send_cancel */
    progress_cancel(NULL, dcc_connection->progress_dialog);
    bs_function_leave();
    return(TRUE);
  }

  if ( (dcc_connection->acked == dcc_connection->sent) && 
       (dcc_connection->sent != dcc_connection->file_stat.st_size)) {
    dcc_send_data(dcc_connection);
  }

  bs_function_leave();
  return(TRUE);
}

/* ------------------------------------------------------------
 * dcc_send_filesel_ok
 *
 *  Callback for the file selection dialog when the user
 *  selects the OK button. Checks the selected file is valid
 *
 * Arguments:
 *	defined by definition of FILESEL_OK_CALLBACK macro
 *
 * Results:
 *	TRUE or FALSE indicating if the user's selection is valid
 *
 * Side effects:
 *	
 * ----------------------------------------------------------- */

FILESEL_OK_CALLBACK(dcc_send_filesel_ok)
{
  char *new_args;
  int length;
  DCC_Connection *dcc_connection = data;

  bs_function_enter();

  printf("dcc_send_filesel_ok %s\n", filename);
  if (!check_filetype(dcc_connection, filename)) {
    bs_function_leave();
    return(FALSE);
  }

  length = strlen(filename)+strlen(dcc_connection->nick)+2;
  new_args = g_malloc(length);
  g_snprintf(new_args, length, "%s %s", dcc_connection->nick, filename);
  
  dcc_send_send(dcc_connection->connection, new_args);

  g_free(dcc_connection->nick);
  g_free(dcc_connection);
  g_free(new_args);

  bs_function_leave();
  return(TRUE);
}

/* ------------------------------------------------------------
 * dcc_send_filesel_cancel
 *
 *  Callback for the file selection dialog when the user
 *  selects the CANCEL button. Cleans up.
 *
 * Arguments:
 *	defined by definition of FILESEL_CANCEL_CALLBACK macro
 *
 * Results:
 *
 * Side effects:
 *	"dcc_connection" is free'd
 * 	
 * ----------------------------------------------------------- */

FILESEL_CANCEL_CALLBACK(dcc_send_filesel_cancel)
{
  DCC_Connection *dcc_connection = data;

  bs_function_enter();

  g_free(dcc_connection->nick);
  g_free(dcc_connection);

  bs_function_leave();
  return;
}

/* ------------------------------------------------------------
 * dcc_send_cancel
 *
 *  Callback for the progress dialog when the user
 *  selects the CANCEL button. Closes the connection and cleans up
 *
 * Arguments:
 *	defined by definition of PROGRESS_CANCEL_CALLBACK macro
 *
 * Results:
 *
 * Side effects:
 *	"dcc_connection->[fd,sd]" are closed
 *	"dcc_connection" is free'd	
 *
 * ----------------------------------------------------------- */

DIALOG_CALLBACK(dcc_send_cancel)
{
  DCC_Connection *dcc_connection = data;

  close(dcc_connection->fd);
  close(dcc_connection->sd);
  gdk_input_remove(dcc_connection->tag);
  if (dcc_connection->nick) {
     g_free(dcc_connection->nick);
  }
  if (dcc_connection->file_name) {
     g_free(dcc_connection->file_name);
  }
  g_free(dcc_connection);

  bs_function_leave();
  return;
}

/* ------------------------------------------------------------
 * dcc_send_accept
 *
 *  Callback for acceptance of DCC Send request by remote user.
 *
 * Arguments:
 *	"dcc_connection" - the DCC Connection object
 *
 * Results:
 *	TRUE or FALSE
 *
 * Side effects:
 *	socket connection is established	
 *	"dcc_connection->sd" is set to socket descriptor
 *	callback to dcc_send_ready established on socket
 *
 * ----------------------------------------------------------- */

gint dcc_send_accept(DCC_Connection *dcc_connection)
{
  struct sockaddr_in cli_addr;
  int clen, complete_sd;
  char buff[BUFFLEN];

  bs_function_enter();

  clen = sizeof(cli_addr);
  
  if ( (complete_sd = accept(dcc_connection->sd, 
			     (struct sockaddr *) &cli_addr, &clen)) < 0 ) {
    perror("accept");
    return(TRUE);
  }
  gdk_input_remove(dcc_connection->tag);

  close(dcc_connection->sd);
  dcc_connection->sd = complete_sd;
  dcc_connection->tag = gdk_input_add(complete_sd, GDK_INPUT_READ,
				      (GdkInputFunction) dcc_send_ready, dcc_connection);

  g_snprintf(buff, BUFFLEN, "To: %s\nFile: %s\nFile size: %d\n", dcc_connection->nick,
	     dcc_connection->file_name, (int) dcc_connection->file_stat.st_size);

  dcc_connection->progress_dialog = progress_dialog(dcc_connection, "DCC Send", buff, dcc_send_cancel);

  time(&dcc_connection->start);
/*   dcc_connections = g_slist_append(dcc_connections,  dcc_connection); */

  dcc_send_data(dcc_connection);

  bs_function_leave();
  return(FALSE);
}

/* ------------------------------------------------------------
 * dcc_send_send
 *
 *  Handle user request to estblish DCC Send connection
 *
 * Arguments:
 *      "connection" - Connection type pointer
 *	"args" - command arguments given by user
 *
 * Results:
 *	0 or 1 indicating the addition of a message
 *
 * Side effects:
 *	listening socket established
 *	DCC Connection object created
 *	
 * ----------------------------------------------------------- */

int dcc_send_send(Connection *connection, char *args)
{
  Message *message;
  DCC_Connection *dcc_connection;
  char *to_nick;
  char *file_name;
  char *clean;
  struct sockaddr_in serv_addr;
  int setopt = 1;
  int slen;

  bs_function_enter();

  to_nick = strtok(args, " ");
  file_name = strtok(NULL, "\0");

  if (!to_nick) {
      message = message_new(MT_CHANNEL, connection, NULL);
      message->parts = g_slist_append(message->parts, 
				      message_part_new(&colour_blue, NULL, NULL, 
					    "Not enough arguments: DCC SEND <nick> [<filename>]"));
      bs_function_leave();
      return(message->type);
  }

  dcc_connection = create_connection(DCC_TYPE_SEND);
  dcc_connection->nick = g_strdup(to_nick);
  dcc_connection->connection = connection;

  if (!file_name) {
    file_selection_dialog(dcc_connection, dcc_send_filesel_ok, dcc_send_filesel_cancel);
    bs_function_leave();
    return(0);
  } else {
    if ( (clean = strrchr(file_name, '/')) ) {
      dcc_connection->file_name = g_strdup(clean+1);
    } else {
      dcc_connection->file_name = g_strdup(file_name);
    }
  }

  if (!check_filetype(dcc_connection, file_name)) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "Not a regular file or file does not exist/is empty"));
    g_free(dcc_connection);
    bs_function_leave();
    return(message->type);
  }

  if ( (dcc_connection->fd = open(file_name, O_RDONLY)) < 0) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "Error opening file: "));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     strerror(errno) ));    
    bs_function_leave();
    g_free(dcc_connection);
    return(message->type);
  }

  if ( (dcc_connection->sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "Error opening socket: "));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     strerror(errno) ));    
    close(dcc_connection->fd);
    g_free(dcc_connection);
    bs_function_leave();
    return(message->type);
  }

  memset((char *) &serv_addr, 0, sizeof(serv_addr));

  slen = sizeof(serv_addr);
  if ( getsockname(connection->sd, (struct sockaddr *) &serv_addr, &slen) != 0) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "Error creating socket binding"));
    close(dcc_connection->fd);
    close(dcc_connection->sd);
    g_free(dcc_connection);
    bs_function_leave();
    return(message->type);
  }

  serv_addr.sin_family = AF_INET;
  serv_addr.sin_port = htons(0);

  setsockopt(dcc_connection->sd, SOL_SOCKET, SO_REUSEADDR, (char *) &setopt, sizeof(setopt));
  setsockopt(dcc_connection->sd, SOL_SOCKET, SO_KEEPALIVE, (char *) &setopt, sizeof(setopt));

  if (bind(dcc_connection->sd, (struct sockaddr *) &serv_addr, 
           sizeof(serv_addr)) < 0)
  {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "Error binding socket"));
    close(dcc_connection->fd);
    close(dcc_connection->sd);
    g_free(dcc_connection);
    bs_function_leave();
    return(message->type);
  }

  slen = sizeof(serv_addr);
  if ( getsockname(dcc_connection->sd, 
                   (struct sockaddr *) &serv_addr, &slen) != 0) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "Error getting socket binding"));
    close(dcc_connection->fd);
    close(dcc_connection->sd);
    g_free(dcc_connection);
    bs_function_leave();
    return(message->type);
  }

  dcc_connection->port = ntohs(serv_addr.sin_port);

  if (listen(dcc_connection->sd, 1) < 0) {
    close(dcc_connection->fd);
    close(dcc_connection->sd);
    g_free(dcc_connection);
    bs_function_leave();
    return -1;
  }
     
  fcntl(dcc_connection->sd, F_SETFL, O_NDELAY);

  dcc_connection->tag = gdk_input_add(dcc_connection->sd, GDK_INPUT_READ,
				      (GdkInputFunction) dcc_send_accept, dcc_connection);

  irc_send_format(connection->sd, "PRIVMSG %s :\001DCC SEND %s %lu %d %lu\001",
	     dcc_connection->nick, dcc_connection->file_name, htonl(serv_addr.sin_addr.s_addr), 
	     dcc_connection->port, dcc_connection->file_stat.st_size);

/*   printf("PRIVMSG %s :DCC SEND %s %lu %d %lu\n", */
/* 	     dcc_connection->nick, dcc_connection->file_name, htonl(serv_addr.sin_addr.s_addr),  */
/* 	     dcc_connection->port, dcc_connection->file_stat.st_size); */

  /*time(dcc_connection->start);*/

  bs_function_leave();
  return(0);
}

DIALOG_CALLBACK(dcc_get_cancel)
{
  DCC_Connection *dcc_connection = data;

  bs_function_enter();

  if (dcc_connection->nick) {
    g_free(dcc_connection->nick);
  }
  if (dcc_connection->file_name) {
    g_free(dcc_connection->file_name);
  }
  if (dcc_connection->sd >= 0) {
    close(dcc_connection->sd);
  }
  if (dcc_connection->fd >= 0) {
    unlink(dcc_connection->file_name);
    close(dcc_connection->fd);
  }
  if (dcc_connection->tag >= 0) {
    gdk_input_remove(dcc_connection->tag);
  }
  if (dcc_connection->progress_dialog) {
    progress_cancel(NULL, dcc_connection->progress_dialog);
  }
  g_free(dcc_connection);

  bs_function_leave();
  return;
}

gint dcc_get_receive(DCC_Connection *dcc_connection)
{
  char buff[BUFFLEN];
  int length;
  guint32 ack;

  if ( (length = recv(dcc_connection->sd, buff, BUFFLEN, 0)) < 0) {
    dcc_get_cancel(dcc_connection);
    bs_function_leave();
    return(TRUE);
  }

  if ( write(dcc_connection->fd, buff, length) < 0) {
    dcc_get_cancel(dcc_connection);
    bs_function_leave();
    return(TRUE);
  }

  dcc_connection->acked += length;

  if (dcc_connection->acked > dcc_connection->size) {
    dcc_get_cancel(dcc_connection);
    bs_function_leave();
    return(TRUE);
  }
  
  ack = htonl(dcc_connection->acked);
  if ( send(dcc_connection->sd, (char *) &ack, 4, 0) != 4) {
    dcc_get_cancel(dcc_connection);
    bs_function_leave();
    return(TRUE);
  }

  progress_update( (float) dcc_connection->acked/dcc_connection->size, dcc_connection->progress_dialog);

  if (dcc_connection->acked == dcc_connection->size) {
    /* We've received the whole file */
    close(dcc_connection->sd);
    close(dcc_connection->fd);
    progress_done(dcc_connection->progress_dialog);
    dcc_connection->progress_dialog = NULL;
    dcc_get_cancel(dcc_connection); /* Clean-up */
  }

  bs_function_leave();
  return(TRUE);
}

DIALOG_CALLBACK(dcc_get_ok)
{
  Message *message;
  DCC_Connection *dcc_connection = data;
  struct sockaddr_in serv_addr;
  int setopt = 1;
  char *old_filename;
  char buff[BUFFLEN];

  bs_function_enter();

  memset(&serv_addr, 0, sizeof(serv_addr));
  serv_addr.sin_family = AF_INET;
  serv_addr.sin_addr.s_addr = dcc_connection->address;
  serv_addr.sin_port = htons(dcc_connection->port);

  if ( (dcc_connection->sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    dcc_get_cancel(dcc_connection);
    bs_function_leave();
    return;
  }

  setsockopt(dcc_connection->sd, SOL_SOCKET, SO_REUSEADDR, (char *) &setopt, sizeof(setopt));
  setsockopt(dcc_connection->sd, SOL_SOCKET, SO_KEEPALIVE, (char *) &setopt, sizeof(setopt));

  if ( connect(dcc_connection->sd, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) < 0 ) {
    dcc_get_cancel(dcc_connection);
    bs_function_leave();
    return;
  }

  fcntl(dcc_connection->sd, F_SETFL, O_NDELAY);

  old_filename = g_strdup(dcc_connection->file_name);
  if ( check_filetype(dcc_connection, dcc_connection->file_name) == FALSE) {
    message = message_new(MT_CHANNEL, dcc_connection->connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "DCC Get: error checking file"));
    dcc_get_cancel(dcc_connection);
    bs_printf(3, "check_filetype failed");
    bs_function_leave();
    return;
  }

  if ( (dcc_connection->fd = open(dcc_connection->file_name, O_WRONLY|O_CREAT|O_EXCL, 0644)) < 0) {
    message = message_new(MT_CHANNEL, dcc_connection->connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "Error opening file: "));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     strerror(errno) ));    
    dcc_get_cancel(dcc_connection);
    bs_printf(3, "open file failed");    
    bs_function_leave();
    return;
  }

  g_snprintf(buff, BUFFLEN, "From: %s\nFile: %s\nAs: %s\nFile size: %lu\n",
	     dcc_connection->nick,
	     old_filename,
	     dcc_connection->file_name,
	     dcc_connection->size);

  g_free(old_filename);

  dcc_connection->progress_dialog = progress_dialog(dcc_connection, "DCC Get", buff, dcc_get_cancel);

  dcc_connection->tag = gdk_input_add(dcc_connection->sd, GDK_INPUT_READ,
				      (GdkInputFunction) dcc_get_receive, dcc_connection);
  time(&dcc_connection->start);

  bs_function_leave();
  return;
}

int dcc_get(char *args)
{
  char *file_name, *address, *port, *size;
  DCC_Connection *dcc_connection;
  char buff[BUFFLEN];

  bs_function_enter();

  if (!args) {
    bs_function_leave();
    return(0);
  }

  file_name = bs_strtok(args, " ");
  address = bs_strtok(NULL, " ");
  port = bs_strtok(NULL, " ");
  size = bs_strtok(NULL, " ");

  if (!file_name || !address || !port || !size) {
    bs_function_leave();
    return(0);
  }

  dcc_connection = create_connection(DCC_TYPE_GET);
  dcc_connection->nick = g_strdup(irc_message.nick);
  dcc_connection->file_name = g_strdup(file_name);
  dcc_connection->address = ntohl(strtoul(address, NULL, 10));
  dcc_connection->port = strtoul(port, NULL, 10);
  dcc_connection->size = atol(size);

/*   bs_printf(3, "DCC Get, address:%s, port:%d\n", */
/* 	    inet_ntoa( *((struct in_addr *) &dcc_connection->address)), */
/* 	    dcc_connection->port); */

  g_snprintf(buff, BUFFLEN, "File upload (DCC SEND) request from %s\nFile: %s\nSize: %lu",
	     dcc_connection->nick,
	     dcc_connection->file_name,
	     dcc_connection->size);

  choice_dialog(dcc_connection,
		"DCC Get Request",
		buff,
		"Get File",
		"Reject",
		dcc_get_ok,
		dcc_get_cancel);

  bs_function_leave();
  return(0);
}

void dcc_chat_close(DCC_Connection *dcc_connection)
{
  bs_function_enter();

  if ( dcc_connection->sd >= 0) {
    close(dcc_connection->sd);
    gdk_input_remove(dcc_connection->tag);
  }

  if ( dcc_connection->message_window ) {
    gtk_widget_destroy(dcc_connection->message_window->window);
  }

  if (dcc_connection->nick) {
     g_free(dcc_connection->nick);
  }

  if (dcc_connection->buff) {
    g_free(dcc_connection->buff);
  }

  dcc_connections = g_slist_remove(dcc_connections, dcc_connection);

  bs_function_leave();
  return;
}

gint dcc_chat_send_format(DCC_Connection *dcc_connection, char *format, ...)
{
  char buff[BUFFLEN];

#ifdef HAVE_VSNPRINTF
  va_list args;

  bs_function_enter();

  va_start (args, format);
  vsnprintf (buff, BUFFLEN, format, args);
  va_end (args);

#else
  gchar *printed;
  va_list args, args2;

  bs_function_enter();

  va_start (args, format);
  va_start (args2, format);

  printed = g_vsprintf (format, &args, &args2);
  strncpy (buff, printed, BUFFLEN);
  buff[BUFFLEN-1] = '\0';

  va_end (args2);
  va_end (args);

#endif             

#if 0
  va_start(args, format);
  vsnprintf(buff, BUFFLEN, format, args);
  va_end(args);
#endif

  if ( write(dcc_connection->sd, buff, strlen(buff)) < 0) {
    dcc_chat_close(dcc_connection);
  } else if ( write(dcc_connection->sd, "\r\n", 2) < 0) {
    dcc_chat_close(dcc_connection);
  }

  bs_function_leave();
  return(0);
}
  
gint dcc_chat_receive_data(DCC_Connection *dcc_connection)
{
  Message *message;
  int count;
  char *buff_pos, *last_start;

  bs_function_enter();

  buff_pos = dcc_connection->buff+dcc_connection->recv_count;
  if ( (count = recv(dcc_connection->sd, buff_pos, BUFFLEN-1-dcc_connection->recv_count, 0)) < 0) {
    if (errno == EWOULDBLOCK || errno == EAGAIN) {
      bs_function_leave();
      return(TRUE);
    } else {
      dcc_chat_close(dcc_connection);
      bs_function_leave();
      return(TRUE);
    }
  }

  if (count == 0) {
    dcc_chat_close(dcc_connection);
    bs_function_leave();
    return(TRUE);
  }

  buff_pos[count] = '\0';

  buff_pos = last_start = dcc_connection->buff;
  while (*buff_pos) {
    if ( *buff_pos == '\n') {
      buff_pos[0] = '\0';
      message = message_new(MT_NICK, dcc_connection->connection, dcc_connection->nick);
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "<"));
      message->parts = g_slist_append(message->parts, 
				      message_part_new(&colour_white, NULL, NULL, dcc_connection->nick));
      message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "> "));
      message->parts = message_part_parse_new(message->parts, &colour_white, 
					      last_start);
      last_start = ++buff_pos;
    } else {
      buff_pos++;
    }
  }
  messages_write();
  
  if (last_start == buff_pos) {
    dcc_connection->recv_count = 0;
  } else {
    dcc_connection->recv_count = strlen(last_start);
    memmove(dcc_connection->buff, last_start, dcc_connection->recv_count+1); /* +1 to get the '\0' */
  }

  bs_function_leave();
  return(TRUE);
}

/* ------------------------------------------------------------
 * dcc_chat_accept
 *
 *  Callback for acceptance of DCC Chat request by remote user.
 *
 * Arguments:
 *	"dcc_connection" - the DCC Connection object
 *
 * Results:
 *	TRUE or FALSE
 *
 * Side effects:
 *	socket connection is established	
 *	"dcc_connection->sd" is set to socket descriptor
 *	callback to dcc_chat_receive established on socket
 *
 * ----------------------------------------------------------- */

gint dcc_chat_accept(DCC_Connection *dcc_connection)
{
  struct sockaddr_in cli_addr;
  int clen, complete_sd;

  bs_function_enter();

  clen = sizeof(cli_addr);
  
  if ( (complete_sd = accept(dcc_connection->sd, 
			     (struct sockaddr *) &cli_addr, &clen)) < 0 ) {
    perror("accept");
    return(TRUE);
  }
  gdk_input_remove(dcc_connection->tag);

  close(dcc_connection->sd);
  dcc_connection->sd = complete_sd;

  dcc_connection->tag = gdk_input_add(complete_sd, GDK_INPUT_READ,
				      (GdkInputFunction) dcc_chat_receive_data, dcc_connection);

  dcc_connection->message_window = (BezMessageWindow *) create_message_window(DCC_CHAT, dcc_connection->nick);
  dcc_connection->message_window->connection = connection_create(NULL,
								 dcc_connection->sd,
								 dcc_connection->nick, 0,
								 (BezWindow *) dcc_connection->message_window);
  dcc_connection->message_window->connection->current_window = (BezWindow *) dcc_connection->message_window;
  dcc_connection->connection = dcc_connection->message_window->connection;
  dcc_connection->connection->messages =  g_slist_append(dcc_connection->connection->messages,
							 dcc_connection->message_window);
  time(&dcc_connection->start);
  dcc_connections = g_slist_append(dcc_connections,  dcc_connection);

  bs_function_leave();
  return(FALSE);
}

/* ------------------------------------------------------------
 * dcc_send_chat
 *
 *  Handle user request to estblish DCC Chat connection
 *
 * Arguments:
 *      "connection" - Connection type pointer
 *	"args" - command arguments given by user
 *
 * Results:
 *	0 or 1 indicating the addition of a message
 *
 * Side effects:
 *	listening socket established
 *	DCC Connection object created
 *	
 * ----------------------------------------------------------- */

int dcc_send_chat(Connection *connection, char *to_nick)
{
  Message *message;
  DCC_Connection *dcc_connection;
  struct sockaddr_in serv_addr;
  int setopt = 1;
  int slen;

  if (!to_nick) {
      message = message_new(MT_CHANNEL, connection, NULL);
      message->parts = g_slist_append(message->parts, 
				      message_part_new(&colour_blue, NULL, NULL, 
					    "Not enough arguments: DCC CHAT <nick>"));
      bs_function_leave();
      return(message->type);
  }

  dcc_connection = create_connection(DCC_TYPE_CHAT);
  dcc_connection->nick = g_strdup(to_nick);

  if ( (dcc_connection->sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "Error opening socket: "));
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     strerror(errno) ));    
    close(dcc_connection->fd);
    g_free(dcc_connection);
    bs_function_leave();
    return(message->type);
  }

  memset((char *) &serv_addr, 0, sizeof(serv_addr));
  slen = sizeof(serv_addr);
  if ( getsockname(connection->sd, (struct sockaddr *) &serv_addr, &slen) != 0) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "Error creating socket binding"));
    close(dcc_connection->fd);
    close(dcc_connection->sd);
    g_free(dcc_connection);
    bs_function_leave();
    return(message->type);
  }

  serv_addr.sin_family = AF_INET;
  serv_addr.sin_port = htons(0);

  setsockopt(dcc_connection->sd, SOL_SOCKET, SO_REUSEADDR, (char *) &setopt, sizeof(setopt));
  setsockopt(dcc_connection->sd, SOL_SOCKET, SO_KEEPALIVE, (char *) &setopt, sizeof(setopt));

  if (bind(dcc_connection->sd, (struct sockaddr *) &serv_addr, 
           sizeof(serv_addr)) < 0)
  {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "Error binding socket"));
    close(dcc_connection->fd);
    close(dcc_connection->sd);
    g_free(dcc_connection);
    bs_function_leave();
    return(message->type);
  }

  slen = sizeof(serv_addr);
  if ( getsockname(dcc_connection->sd, 
                   (struct sockaddr *) &serv_addr, &slen) != 0) {
    message = message_new(MT_CHANNEL, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "Error getting socket binding"));
    close(dcc_connection->fd);
    close(dcc_connection->sd);
    g_free(dcc_connection);
    bs_function_leave();
    return(message->type);
  }

  dcc_connection->port = ntohs(serv_addr.sin_port);

  if (listen(dcc_connection->sd, 1) < 0) {
    close(dcc_connection->fd);
    close(dcc_connection->sd);
    g_free(dcc_connection);
    bs_function_leave();
    return -1;
  }
     
  fcntl(dcc_connection->sd, F_SETFL, O_NDELAY);

  dcc_connection->tag = gdk_input_add(dcc_connection->sd, GDK_INPUT_READ,
				      (GdkInputFunction) dcc_chat_accept, dcc_connection);

  irc_send_format(connection->sd, "PRIVMSG %s :\001DCC CHAT CHAT %lu %d\001",
		  dcc_connection->nick,
		  htonl(serv_addr.sin_addr.s_addr),
		  dcc_connection->port);

  bs_printf(3, "PRIVMSG %s :DCC CHAT\n",
	    dcc_connection->nick,
	    htonl(serv_addr.sin_addr.s_addr),
	    dcc_connection->port);
  bs_function_leave();
  return(0);
}

DIALOG_CALLBACK(dcc_chat_cancel)   
{
  DCC_Connection *dcc_connection = data;

  bs_function_enter();

  dcc_chat_close(dcc_connection);

  bs_function_leave();
  return;
}

void dcc_chat_finish(char *nick)
{
  DCC_Connection *dcc_connection;
  GSList *glist_entry;

  bs_function_enter();

  glist_entry = dcc_connections;
  while (glist_entry) {
    dcc_connection = (DCC_Connection *) glist_entry->data;
    if ( !strcasecmp(nick, dcc_connection->nick) ) {
      dcc_chat_close(dcc_connection);
      bs_function_leave();
      return;
    }
    glist_entry = g_slist_next(glist_entry);    
  }

  bs_function_leave();
  return;
}

DIALOG_CALLBACK(dcc_chat_ok)
{
  DCC_Connection *dcc_connection = data;
  struct sockaddr_in serv_addr;
  int setopt = 1;

  bs_function_enter();

  memset(&serv_addr, 0, sizeof(serv_addr));
  serv_addr.sin_family = AF_INET;
  serv_addr.sin_addr.s_addr = dcc_connection->address;
  serv_addr.sin_port = htons(dcc_connection->port);

  if ( (dcc_connection->sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    dcc_chat_close(dcc_connection);
    bs_function_leave();
    return;
  }

  setsockopt(dcc_connection->sd, SOL_SOCKET, SO_REUSEADDR, (char *) &setopt, sizeof(setopt));
  setsockopt(dcc_connection->sd, SOL_SOCKET, SO_KEEPALIVE, (char *) &setopt, sizeof(setopt));

  if ( connect(dcc_connection->sd, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) < 0 ) {
    dcc_get_cancel(dcc_connection);
    bs_function_leave();
    return;
  }

  fcntl(dcc_connection->sd, F_SETFL, O_NDELAY);

  dcc_connection->message_window = (BezMessageWindow *) create_message_window(DCC_CHAT, dcc_connection->nick);
  dcc_connection->message_window->connection = connection_create(NULL,
								 dcc_connection->sd,
								 "DCC CHAT", 0,
								 (BezWindow *) dcc_connection->message_window);
  dcc_connection->message_window->connection->current_window = (BezWindow *) dcc_connection->message_window;
  dcc_connection->tag = gdk_input_add(dcc_connection->sd, GDK_INPUT_READ,
				      (GdkInputFunction) dcc_chat_receive_data, dcc_connection);
  time(&dcc_connection->start);
  dcc_connections = g_slist_append(dcc_connections,  dcc_connection);

  bs_function_leave();
  return;
}

int dcc_recv_chat(char *args)
{
  char *chat, *address, *port;
  DCC_Connection *dcc_connection;
  char buff[BUFFLEN];

  bs_function_enter();

  if (!args) {
    bs_function_leave();
    return(0);
  }

  chat = bs_strtok(args, " ");
  address = bs_strtok(NULL, " ");
  port = bs_strtok(NULL, " ");

  if (!chat || !address || !port) {
    bs_function_leave();
    return(0);
  }

  dcc_connection = create_connection(DCC_TYPE_CHAT);
  dcc_connection->nick = g_strdup(irc_message.nick);
  dcc_connection->address = ntohl(strtoul(address, NULL, 10));
  dcc_connection->port = strtoul(port, NULL, 10);
 
  g_snprintf(buff, BUFFLEN, "Message (DCC CHAT) request from %s",
	     dcc_connection->nick);

  choice_dialog(dcc_connection,
		"DCC Chat Request",
		buff,
		"Accept",
		"Reject",
		dcc_chat_ok,
		dcc_chat_cancel);

  bs_function_leave();
  return(0);
}

void dcc_chat_command(char *comm, BezWindow *window)
{
  char *command, *args;
  Message *message;

  if (!comm) {
    bs_function_leave();
    return;
  }

  command = bs_strtok(comm, " ");
  args = bs_strtok(NULL, "");
 
  if (!command) {
    bs_function_leave();
    return;
  }

  if (!strcasecmp("QUIT", command))       gtk_widget_destroy(BEZ_BASE_WINDOW(window)->window);
  else if (!strcasecmp("PART", command))  gtk_widget_destroy(BEZ_BASE_WINDOW(window)->window);
  else {
    message = message_new(MT_NICK, BEZ_BASE_WINDOW(window)->connection,
			  BEZ_BASE_WINDOW(window)->connection->server);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_blue, NULL, NULL, 
						     "Command not applicable to DCC Chat"));
  }

  bs_function_leave();
  return;
}

/* ------------------------------------------------------------
 * process_dcc_command
 *
 *  Parse user command and dispatch to appropriate function
 *
 * Arguments:
 *	defined by definition of USER_COMMAND macro
 *
 * Results:
 *	0 or return value from handler function
 *
 * Side effects:
 *	the command line string is altered
 *	
 * ----------------------------------------------------------- */

USER_COMMAND(process_dcc_command)
{
  char *comm, *rest;
  int retval=0;

  bs_function_enter();

  if (!args) {
    bs_function_leave();
    return(0);
  }
  comm = bs_strtok(args, " ");
  rest = bs_strtok(NULL, "");

  if (!comm) {
    bs_function_leave();
    return(0);
  }

  if (!strcasecmp("SEND", comm))         retval = dcc_send_send(BEZ_BASE_WINDOW(window)->connection, rest);
  else if (!strcasecmp("CHAT", comm))    retval = dcc_send_chat(BEZ_BASE_WINDOW(window)->connection, rest); 

  bs_function_leave();
  return(retval);
}

int process_dcc_request(char *request)
{
  char *command, *args; 
  int retval=0;

  bs_function_enter();

  command = bs_strtok(request, " ");
  if (command[strlen(command)-1] == '\001') {
    command[strlen(command)-1] = '\0';
  }
  args = bs_strtok(NULL, "\001");

  if (!strcmp("SEND", command))                retval = dcc_get(args);
  else if (!strcmp("CHAT", command))           retval = dcc_recv_chat(args); 

  bs_function_leave();
  return(retval);
}





