/*
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#define ACCOUNT_INTERNAL_ACCESS
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#include "compat/strtoul.h"
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include "compat/strdup.h"
#include <ctype.h>
#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <errno.h>
#include "compat/strerror.h"
#include <sys/types.h>
#ifdef HAVE_DIRENT_H
# include <dirent.h>
#else
# define dirent direct
# ifdef HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
#endif
#include "eventlog.h"
#include "prefs.h"
#include "util.h"
#include "field_sizes.h"
#include "bnethash.h"
#include "list.h"
#include "account.h"


static t_list * accountlist_head=NULL;

static t_account * default_acct=NULL;
static unsigned int maxuserid=0;


static char * escape_chars(char const * in, unsigned int len);
static char * unescape_chars(char const * in);
static unsigned int account_hash(char const * username);
static int account_insert_attr(t_account * account, char const * key, char const * val);
static t_account * account_load(char const * filename);
static int account_load_attrs(t_account * account);
static void account_unload_attrs(t_account * account);



static char * escape_chars(char const * in, unsigned int len)
{
    char *       out;
    unsigned int inpos;
    unsigned int outpos;
    
    if (!in)
    {
	eventlog(eventlog_level_error,"escape_chars","got NULL input string");
	return NULL;
    }
    if (!(out = malloc(len*4+1))) /* if all turn into \xxx */
    {
	eventlog(eventlog_level_error,"escape_chars","could not allocate memory for string");
	return NULL;
    }
    
    for (inpos=0,outpos=0; inpos<len; inpos++)
    {
	if (in[inpos]=='\\')
	{
	    out[outpos++] = '\\';
	    out[outpos++] = '\\';
	}
	else if (in[inpos]=='"')
	{
	    out[outpos++] = '\\';
	    out[outpos++] = '"';
	}
        else if (isprint((int)in[inpos]))
	    out[outpos++] = in[inpos];
        else if (in[inpos]=='\a')
	{
	    out[outpos++] = '\\';
	    out[outpos++] = 'a';
	}
        else if (in[inpos]=='\b')
	{
	    out[outpos++] = '\\';
	    out[outpos++] = 'b';
	}
        else if (in[inpos]=='\t')
	{
	    out[outpos++] = '\\';
	    out[outpos++] = 't';
	}
        else if (in[inpos]=='\n')
	{
	    out[outpos++] = '\\';
	    out[outpos++] = 'n';
	}
        else if (in[inpos]=='\v')
	{
	    out[outpos++] = '\\';
	    out[outpos++] = 'v';
	}
        else if (in[inpos]=='\f')
	{
	    out[outpos++] = '\\';
	    out[outpos++] = 'f';
	}
        else if (in[inpos]=='\r')
	{
	    out[outpos++] = '\\';
	    out[outpos++] = 'r';
	}
	else
	{
	    out[outpos++] = '\\';
	    /* always 001 through 377 octal */
	    sprintf(&out[outpos],"%03o",(unsigned int)(unsigned char)in[inpos]);
	    outpos += 3;
	}
    }
    if (outpos>=len*4+1)
        eventlog(eventlog_level_error,"escape_chars","overflowed buffer!!! (outpos=%u len=%u)",outpos,len*4+1);
    out[outpos] = '\0';
    
    return out;
}


static char * unescape_chars(char const * in)
{
    char *       out;
    unsigned int inpos;
    unsigned int outpos;
    
    if (!in)
    {
	eventlog(eventlog_level_error,"unescape_chars","got NULL input string");
	return NULL;
    }
    if (!(out = malloc(strlen(in)+1)))
    {
	eventlog(eventlog_level_error,"unescape_chars","could not allocate memory for string");
	return NULL;
    }
    
    for (inpos=0,outpos=0; inpos<strlen(in); inpos++)
    {
        if (in[inpos]!='\\')
	    out[outpos++] = in[inpos];
        else
	    switch (in[++inpos])
	    {
	    case '\\':
		out[outpos++] = '\\';
		break;
	    case '"':
		out[outpos++] = '"';
		break;
	    case 'a':
		out[outpos++] = '\a';
		break;
	    case 'b':
		out[outpos++] = '\b';
		break;
	    case 't':
		out[outpos++] = '\t';
		break;
	    case 'n':
		out[outpos++] = '\n';
		break;
	    case 'v':
		out[outpos++] = '\v';
		break;
	    case 'f':
		out[outpos++] = '\f';
		break;
	    case 'r':
		out[outpos++] = '\r';
		break;
	    default:
	    {
		char         temp[4];
		unsigned int i;
		unsigned int num;
		
		for (i=0; i<3; i++)
		{
		    if (in[inpos]!='0' &&
		        in[inpos]!='1' &&
		        in[inpos]!='2' &&
		        in[inpos]!='3' &&
		        in[inpos]!='4' &&
		        in[inpos]!='5' &&
		        in[inpos]!='6' &&
		        in[inpos]!='7')
			break;
		    temp[i] = in[inpos++];
		}
		temp[i] = '\0';
		inpos--;
		
		num = strtoul(temp,NULL,8);
		if (i<3 || num<1 || num>255) /* bad escape (including \000), leave it as-is */
		{
		    out[outpos++] = '\\';
		    strcpy(&out[outpos],temp);
		    outpos += strlen(temp);
		}
		else
		    out[outpos++] = (unsigned char)num;
	    }
	}
    }
    out[outpos] = '\0';
    
    return out;
}


#define BITS_PER_BYTE 8
#define ROL(x,n) (((x)<<(n)) | ((x)>>((sizeof(unsigned int)*BITS_PER_BYTE)-(n))))

static unsigned int account_hash(char const * username)
{
    unsigned int i;
    unsigned int pos;
    unsigned int hash;
    
    if (!username)
    {
	eventlog(eventlog_level_error,"account_hash","got NULL username");
	return 0;
    }
    
    for (hash=0,pos=0,i=0; i<strlen(username); i++)
    {
	hash ^= ROL(((unsigned int)tolower((int)username[i])),
		    (pos%(sizeof(unsigned int)*BITS_PER_BYTE)));
	pos += BITS_PER_BYTE-1;
    }
    
    return hash;
}


extern t_account * account_create(char const * username, char const * passhash1)
{
    t_account * account;
    
    if (!(account = malloc(sizeof(t_account))))
    {
	eventlog(eventlog_level_error,"account_create","could not allocate memory for account");
	return NULL;
    }
    
    account->attrs    = NULL;
    account->namehash = 0; /* hash it later */
    account->uid      = 0; /* hash it later */
    account->dirty    = 0;
    account->loaded   = 0;
    account->accessed = 0;
    account->age      = 0;
    account->filename = NULL;
    
    if (username) /* actually making a new account */
    {
	char * temp;
	
	if (username[0]=='\0')
	{
	    eventlog(eventlog_level_error,"account_create","got empty usrname");
	    account_destroy(account);
	    return NULL;
	}
	if (username[0]=='#')
	{
	    eventlog(eventlog_level_error,"account_create","username starts with '#'");
	    account_destroy(account);
	    return NULL;
	}
	if (strchr(username,' '))
	{
	    eventlog(eventlog_level_error,"account_create","username contains spaces");
	    account_destroy(account);
	    return NULL;
	}
	if (strlen(username)>=USER_NAME_LEN)
	{
	    eventlog(eventlog_level_error,"account_create","username is too long (%u chars)",strlen(username));
	    account_destroy(account);
	    return NULL;
	}
	
        if (!(temp = malloc(strlen(prefs_get_userdir())+1+8+1))) /* dir + / + uid  + NUL */
	{
	    eventlog(eventlog_level_error,"account_create","could not allocate memory for temp");
	    account_destroy(account);
	    return NULL;
	}
	++maxuserid;
	sprintf(temp,"%s/%06u",prefs_get_userdir(),maxuserid);
	account->filename = temp;
	
        account->loaded = 1;
	
	account_set_strattr(account,"BNET\\acct\\username",username);
	account_set_numattr(account,"BNET\\acct\\userid",maxuserid);
	account_set_strattr(account,"BNET\\acct\\passhash1",passhash1);
    }
    
    return account;
}


static void account_unload_attrs(t_account * account)
{
    t_attribute const * attr;
    t_attribute const * temp;
    
/*    eventlog(eventlog_level_debug,"account_unload_attrs","unloading \"%s\"",account->filename);*/
    for (attr=account->attrs; attr; attr=temp)
    {
	if (attr->key)
	    pfree((void *)attr->key,strlen(attr->key)+1); /* avoid warning */
	if (attr->val)
	    pfree((void *)attr->val,strlen(attr->val)+1); /* avoid warning */
        temp = attr->next;
	pfree((void *)attr,sizeof(t_attribute)); /* avoid warning */
    }
    account->attrs = NULL;
    account->loaded = 0;
}


extern void account_destroy(t_account * account)
{
    if (!account)
    {
	eventlog(eventlog_level_error,"account_destroy","got NULL account");
	return;
    }
    account_unload_attrs(account);
    if (account->filename)
	pfree((void *)account->filename,strlen(account->filename)+1); /* avoid warning */
    pfree(account,sizeof(t_account));
}


extern int account_match(t_account * account, char const * username)
{
    unsigned int userid=0;
    unsigned int namehash;
    char const * tname;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_match","got NULL account");
	return -1;
    }
    if (!username)
    {
	eventlog(eventlog_level_error,"account_match","got NULL username");
	return -1;
    }
    
    if (account->uid==0) /* set the cache, should not happen currently */
    {
	if (!account->loaded)
	    if (account_load_attrs(account)<0)
	    {
		eventlog(eventlog_level_error,"account_match","could not load attributes");
		return -1;
	    }
	
	account->uid      = account_get_uid(account);
	account->namehash = account_hash((tname = account_get_name(account)));
	account_unget_name(tname);
    }
    
    if (username[0]=='#')
        if (str_to_uint(&username[1],&userid)<0)
            userid = 0;
    
    if (userid)
    {
        if (account->uid==userid)
            return 1;
    }
    else
    {
	namehash = account_hash(username);
        if (account->namehash==namehash &&
	    (tname = account_get_name(account)))
	    if (strcasecmp(tname,username)==0)
	    {
		account_unget_name(tname);
		return 1;
	    }
	    else
		account_unget_name(tname);
    }
    
    return 0;
}


extern int account_save(t_account * account, unsigned int delta)
{
    FILE *        accountfile;
    t_attribute * attr;
    char const *  key;
    char const *  val;
    char *        tempname;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_save","got NULL account");
	return -1;
    }
    if (!account->filename)
    {
	eventlog(eventlog_level_error,"account_save","account has NULL filename");
	return -1;
    }
    
    /* account aging logic */
    if (account->accessed)
	account->age >>= 1;
    else
	account->age += delta;
    if (account->age>( (3*prefs_get_user_flush_timer()) >>1))
        account->age = ( (3*prefs_get_user_flush_timer()) >>1);
    account->accessed = 0;
    
    if (!account->loaded)
	return 0;
    
    if (!account->dirty)
    {
	if (account->age>=prefs_get_user_flush_timer())
	    account_unload_attrs(account);
	return 0;
    }
    
    if (!(tempname = malloc(strlen(prefs_get_userdir())+1+strlen(BNETD_ACCOUNT_TMP)+1)))
    {
	eventlog(eventlog_level_error,"account_save","uable to allocate memory for tempname");
	return -1;
    }
    
    sprintf(tempname,"%s/%s",prefs_get_userdir(),BNETD_ACCOUNT_TMP);
    
    if (!(accountfile = fopen(tempname,"w")))
    {
	eventlog(eventlog_level_error,"account_save","unable to open file \"%s\" for writing (fopen: %s)",tempname,strerror(errno));
	pfree(tempname,strlen(prefs_get_userdir())+1+strlen(BNETD_ACCOUNT_TMP)+1);
	return -1;
    }
    
    for (attr=account->attrs; attr; attr=attr->next)
    {
	if (attr->key)
	    key = escape_chars(attr->key,strlen(attr->key));
	else
	{
	    eventlog(eventlog_level_error,"account_save","attribute with NULL key in list");
	    key = NULL;
	}
	if (attr->val)
	    val = escape_chars(attr->val,strlen(attr->val));
	else
	{
	    eventlog(eventlog_level_error,"account_save","attribute with NULL val in list");
	    val = NULL;
	}
	if (key && val)
	    fprintf(accountfile,"\"%s\"=\"%s\"\n",key,val);
	else
	    eventlog(eventlog_level_error,"account_save","could not save attribute key=\"%s\"",attr->key);
	if (key)
	    pfree((void *)key,strlen(key)+1); /* avoid warning */
	if (val)
	    pfree((void *)val,strlen(val)+1); /* avoid warning */
    }
    
    fclose(accountfile);
    
    if (rename(tempname,account->filename)<0)
    {
	eventlog(eventlog_level_error,"account_save","could not rename account to \"%s\" (rename: \"%s\")",account->filename,strerror(errno));
	pfree(tempname,strlen(prefs_get_userdir())+1+strlen(BNETD_ACCOUNT_TMP)+1);
	return -1;
    }
    
    account->dirty = 0;
    
    pfree(tempname,strlen(prefs_get_userdir())+1+strlen(BNETD_ACCOUNT_TMP)+1);
    return 1;
}


static int account_insert_attr(t_account * account, char const * key, char const * val)
{
    t_attribute * nattr;
    char *        nkey;
    char *        nval;
    
    if (!(nattr = malloc(sizeof(t_attribute))))
    {
	eventlog(eventlog_level_error,"account_insert_attr","could not allocate attribute");
	return -1;
    }
    if (!(nkey = strdup(key)))
    {
	eventlog(eventlog_level_error,"account_insert_attr","could not allocate attribute key");
	pfree(nattr,sizeof(t_attribute));
	return -1;
    }
    if (!(nval = strdup(val)))
    {
	eventlog(eventlog_level_error,"account_insert_attr","could not allocate attribute value");
	pfree(nkey,strlen(nkey)+1);
	pfree(nattr,sizeof(t_attribute));
	return -1;
    }
    nattr->key  = nkey;
    nattr->val  = nval;
    nattr->next = account->attrs;
    
    account->attrs = nattr;
    
    return 0;
}


extern char const * account_get_strattr(t_account * account, char const * key)
{
    t_attribute const * curr;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_get_strattr","got NULL account");
	return NULL;
    }
    if (!key)
    {
	eventlog(eventlog_level_error,"account_get_strattr","got NULL key");
	return NULL;
    }
    
    account->accessed = 1;
    
    if (!account->loaded)
        if (account_load_attrs(account)<0)
	{
	    eventlog(eventlog_level_error,"account_get_strattr","could load attributes");
	    return NULL;
	}
    
    if (account->attrs)
	for (curr=account->attrs; curr; curr=curr->next)
	    if (strcasecmp(curr->key,key)==0)
	    {
#ifdef TESTUNGET
		return strdup(curr->val);
#else
                return curr->val;
#endif
	    }
    
    if (account==default_acct) /* don't recurse infinitely */
	return NULL;
    
    return account_get_strattr(default_acct,key); /* FIXME: this is sorta dangerous because this pointer can go away if we re-read the config files... verify that nobody caches non-username, userid strings */
}


extern void account_unget_strattr(char const * val)
{
    if (!val)
    {
	eventlog(eventlog_level_error,"account_unget_strattr","got NULL val");
	return;
    }
#ifdef TESTUNGET
    pfree((void *)val,strlen(val)+1); /* avoid warning */
#endif
}


extern int account_set_strattr(t_account * account, char const * key, char const * val)
{
    t_attribute * curr;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_set_strattr","got NULL account");
	return -1;
    }
    if (!key)
    {
	eventlog(eventlog_level_error,"account_set_strattr","got NULL key");
	return -1;
    }
    
    if (!account->loaded)
        if (account_load_attrs(account)<0)
	{
	    eventlog(eventlog_level_error,"account_set_strattr","could not load attributes");
	    return -1;
	}
    
    account->dirty = 1; /* will need to be saved */
    
    curr = account->attrs;
    if (!curr)
    {
	if (val)
	    return account_insert_attr(account,key,val);
	return 0;
    }
    
    if (strcasecmp(curr->key,key)==0) /* if key is already the first in the attr list */
    {
	if (val)
	{
	    char * temp;
	    
	    if (!(temp = strdup(val)))
	    {
		eventlog(eventlog_level_error,"account_set_strattr","could not allocate attribute value");
		return -1;
	    }
	    
	    pfree((void *)curr->val,strlen(curr->val)+1); /* avoid warning */
	    curr->val = temp;
	}
	else
	{
	    t_attribute * temp;
	    
	    temp = curr->next;
	    
	    pfree((void *)curr->key,strlen(curr->key)+1); /* avoid warning */
	    pfree((void *)curr->val,strlen(curr->val)+1); /* avoid warning */
	    pfree((void *)curr,sizeof(t_attribute)); /* avoid warning */
	    
	    account->attrs = temp;
	}
	return 0;
    }
    
    for (; curr->next; curr=curr->next)
	if (strcasecmp(curr->next->key,key)==0)
	    break;
    
    if (curr->next) /* if key is already in the attr list */
    {
	if (val)
	{
	    char * temp;
	    
	    if (!(temp = strdup(val)))
	    {
		eventlog(eventlog_level_error,"account_set_strattr","could not allocate attribute value");
		return -1;
	    }
	    
	    pfree((void *)curr->next->val,strlen(curr->next->val)+1); /* avoid warning */
	    curr->next->val = temp;
	}
	else
	{
	    t_attribute * temp;
	    
	    temp = curr->next->next;
	    
	    pfree((void *)curr->next->key,strlen(curr->next->key)+1); /* avoid warning */
	    pfree((void *)curr->next->val,strlen(curr->next->val)+1); /* avoid warning */
	    pfree(curr->next,sizeof(t_attribute));
	    
	    curr->next = temp;
	}
	return 0;
    }
    
    if (val)
	return account_insert_attr(account,key,val);
    return 0;
}


static int account_load_attrs(t_account * account)
{
    FILE *       accountfile;
    unsigned int line;
    char const * buff;
    unsigned int len;
    char *       esckey;
    char *       escval;
    char const * key;
    char const * val;
    
    if (!account)
    {
	eventlog(eventlog_level_error,"account_load_attrs","got NULL account");
	return -1;
    }
    if (!account->filename)
    {
	eventlog(eventlog_level_error,"account_load_attrs","account has NULL filename");
	return -1;
    }
    
    if (account->loaded) /* already done */
	return 0;
    if (account->dirty) /* if not loaded, how dirty? */
    {
	eventlog(eventlog_level_error,"account_load_attrs","can not load modified account");
	return -1;
    }
    
    eventlog(eventlog_level_debug,"account_load_attrs","loading \"%s\"",account->filename);
    if (!(accountfile = fopen(account->filename,"r")))
    {
	eventlog(eventlog_level_error,"account_load_attrs","could not open account file \"%s\" for reading (fopen: %s)",account->filename,strerror(errno));
	return -1;
    }
    
    account->loaded = 1; /* set now so set_strattr works */
    for (line=1; (buff=file_get_line(accountfile)); line++)
    {
	if (buff[0]=='#' || buff[0]=='\0')
	{
	    pfree((void *)buff,strlen(buff)+1); /* avoid warning */
	    continue;
	}
	
	len = strlen(buff)-5+1; /* - ""="" + NUL */
	if (!(esckey = malloc(len)))
	{
	    eventlog(eventlog_level_error,"account_load_attrs","could not allocate memory for esckey on line %d of account file \"%s\"",line,account->filename);
	    pfree((void *)buff,strlen(buff)+1); /* avoid warning */
	    continue;
	}
	if (!(escval = malloc(len)))
	{
	    eventlog(eventlog_level_error,"account_load_attrs","could not allocate memory for escval on line %d of account file \"%s\"",line,account->filename);
	    pfree((void *)buff,strlen(buff)+1); /* avoid warning */
	    pfree(esckey,len);
	    continue;
	}
	
	if (sscanf(buff,"\"%[^\"]\" = \"%[^\"]\"",esckey,escval)!=2)
	{
	    if (sscanf(buff,"\"%[^\"]\" = \"\"",esckey)!=1) /* hack for an empty value field */
	    {
		eventlog(eventlog_level_error,"account_load_attrs","malformed entry on line %d of account file \"%s\"",line,account->filename);
		pfree(escval,len);
		pfree(esckey,len);
		pfree((void *)buff,strlen(buff)+1); /* avoid warning */
		continue;
	    }
	    escval[0] = '\0';
	}
	pfree((void *)buff,strlen(buff)+1); /* avoid warning */
	
	key = unescape_chars(esckey);
	val = unescape_chars(escval);
	
/* eventlog(eventlog_level_debug,"account_load_attrs","strlen(esckey)=%u (%c), len=%u",strlen(esckey),esckey[0],len);*/
	pfree(esckey,len);
	pfree(escval,len);
	
	if (key && val)
	    account_set_strattr(account,key,val);
	if (key)
	    pfree((void *)key,strlen(key)+1); /* avoid warning */
	if (val)
	    pfree((void *)val,strlen(val)+1); /* avoid warning */
    }
    
    fclose(accountfile);
    account->dirty = 0;
    
    return 0;
}


static t_account * account_load(char const * filename)
{
    t_account * account;
    
    if (!filename)
    {
	eventlog(eventlog_level_error,"account_load","got NULL filename");
	return NULL;
    }
    
    if (!(account = account_create(NULL,NULL)))
    {
	eventlog(eventlog_level_error,"account_load","could not load account from file \"%s\"",filename);
	return NULL;
    }
    if (!(account->filename = strdup(filename)))
    {
	eventlog(eventlog_level_error,"account_load","could not allocate memory for account->filename");
	account_destroy(account);
	return NULL;
    }
    
    return account;
}


extern int accountlist_load_default(void)
{
    if (default_acct)
	account_destroy(default_acct);
    
    if (!(default_acct = account_load(prefs_get_defacct())))
    {
        eventlog(eventlog_level_error,"accountlist_load_default","could not load default account template from file \"%s\"",prefs_get_defacct());
	return -1;
    }
    if (account_load_attrs(default_acct)<0)
    {
	eventlog(eventlog_level_error,"accountlist_load_default","could not load default account template attributes");
	return -1;
    }
    
    eventlog(eventlog_level_info,"accountlist_load_default","loaded default account template");
    return 0;
}

    
extern int accountlist_load(void)
{
    DIR *           accountdir;
    struct dirent * dentry;
    char *          filename;
    t_account *     account;
    unsigned int    count;
    
    if (accountlist_head)
    {
        eventlog(eventlog_level_error,"accountlist_load","accountlist is not empty before load");
	return -1;
    }
    
    if (!(accountdir = opendir(prefs_get_userdir())))
    {
        eventlog(eventlog_level_error,"accountlist_load","unable to open directory \"%s\" for reading (opendir: %s)",prefs_get_userdir(),strerror(errno));
        return -1;
    }
    
    count = 0;
    while ((dentry = readdir(accountdir)))
    {
	if (dentry->d_name[0]=='.')
	    continue;
	if (!(filename = malloc(strlen(prefs_get_userdir())+1+strlen(dentry->d_name)+1))) /* dir + / + file + NUL */
	{
	    eventlog(eventlog_level_error,"accountlist_load","could not allocate memory for filename");
	    continue;
	}
	sprintf(filename,"%s/%s",prefs_get_userdir(),dentry->d_name);
	if (!(account = account_load(filename)))
	{
	    eventlog(eventlog_level_error,"accountlist_load","could not load account from file \"%s\"",filename);
	    pfree(filename,strlen(prefs_get_userdir())+1+strlen(dentry->d_name)+1);
	    continue;
	}
	
	if (!accountlist_add_account(account))
	{
	    eventlog(eventlog_level_error,"accountlist_load","could not add account from file \"%s\" to list",filename);
	    pfree(filename,strlen(prefs_get_userdir())+1+strlen(dentry->d_name)+1);
	    account_destroy(account);
	    continue;
	}
	
	pfree(filename,strlen(prefs_get_userdir())+1+strlen(dentry->d_name)+1);
	
	/* might as well free up the memory since we probably won't need it */
	account->accessed = 0; /* lie */
	account_save(account,1000); /* big delta to force unload */
	
        count++;
    }
    
    closedir(accountdir);
    eventlog(eventlog_level_info,"accountlist_load","loaded %u user accounts",count);
    
    return 0;
}


extern int accountlist_unload(void)
{
    t_account * account;
    
    while (accountlist_head)
    {
	if (!(account = list_get_item(accountlist_head)))
	    eventlog(eventlog_level_error,"accountlist_unload","found NULL account in list");
	else
	{
	    if (account_save(account,0)<0)
		eventlog(eventlog_level_error,"accountlist_unload","could not save account");
	    
	    account_destroy(account);
	}
	list_remove(&accountlist_head);
    }
    
    return 0;
}


extern void accountlist_unload_default(void)
{
    account_destroy(default_acct);
}


extern int accountlist_get_length(void)
{
    return list_get_length(accountlist_head);
}


extern int accountlist_save(unsigned int delta)
{
    t_list const * const * save;
    t_account *            account;
    unsigned int           scount;
    unsigned int           tcount;
    
    scount=tcount = 0;
    for (account=accountlist_get_first(&save); account; account=accountlist_get_next(&save),tcount++)
	switch (account_save(account,delta))
	{
	case -1:
	    eventlog(eventlog_level_error,"accountlist_save","could not save account");
	    break;
	case 1:
	    scount++;
	    break;
	case 0:
	default:
	    break;
	}
    
    if (scount>0)
	eventlog(eventlog_level_debug,"accountlist_save","saved %u of %u user accounts",scount,tcount);
    return 0;
}


extern t_account * accountlist_find_account(char const * username)
{
    unsigned int           userid=0;
    t_list const * const * save;
    t_account *            account;
    
    if (!username)
    {
	eventlog(eventlog_level_error,"accountlist_find_account","got NULL username");
	return NULL;
    }
    
    if (username[0]=='#')
        if (str_to_uint(&username[1],&userid)<0)
            userid = 0;
    
    /* all accounts in list must be hashed already, no need to check */
    if (userid)
    {
	for (account=accountlist_get_first(&save); account; account=accountlist_get_next(&save))
	    if (account->uid==userid)
		return account;
    }
    else
    {
	unsigned int namehash;
	char const * tname;
	
	namehash = account_hash(username);
	for (account=accountlist_get_first(&save); account; account=accountlist_get_next(&save))
            if (account->namehash==namehash &&
		(tname = account_get_name(account)))
		if (strcasecmp(tname,username)==0)
		{
		    account_unget_name(tname);
		    return account;
		}
		else
		    account_unget_name(tname);
    }
    
    return NULL;
}


extern t_account * accountlist_add_account(t_account * account)
{
    unsigned int    uid;
    char const *    username;
    
    if (!account)
    {
        eventlog(eventlog_level_error,"accountlist_add_account","got NULL account");
        return NULL;
    }
    if (!(username = account_get_name(account)) || username[0]=='\0')
    {
        eventlog(eventlog_level_error,"accountlist_add_account","got bad account (no username)");
        return NULL;
    }
    if ((uid = account_get_uid(account))<1)
    {
        eventlog(eventlog_level_error,"accountlist_add_account","got bad account (bad uid)");
	account_unget_name(username);
	return NULL;
    }
    
    /* hash it */
    account->uid      = uid;
    account->namehash = account_hash(username);
    
    /* mini version of accountlist_find_account() */
    {
	t_list const * const * save;
	t_account *            curraccount;
	char const *           tname;
	
	for (curraccount=accountlist_get_first(&save); curraccount; curraccount=accountlist_get_next(&save))
	{
	    if (curraccount->uid==uid)
	    {
		eventlog(eventlog_level_error,"accountlist_add_account","user \"%s\":#%06u already has an account (\"%s\":#%06u)",username,uid,(tname = account_get_name(curraccount)),account_get_uid(curraccount));
		account_unget_name(tname);
		account_unget_name(username);
		return NULL;
	    }
	    
	    if (curraccount->namehash==account->namehash &&
		(tname = account_get_name(curraccount)))
		if (strcasecmp(tname,username)==0)
		{
		    eventlog(eventlog_level_error,"accountlist_add_account","user \"%s\":#%06u already has an account (\"%s\":#%06u)",username,uid,tname,account_get_uid(curraccount));
		    account_unget_name(tname);
		    account_unget_name(username);
		    return NULL;
		}
		else
		    account_unget_name(tname);
	}
    }
    account_unget_name(username);
    
    if (list_prepend_item(&accountlist_head,account)<0)
    {
	eventlog(eventlog_level_error,"accountlist_add_account","could not add account to list");
	return NULL;
    }
    
    if (uid>maxuserid)
        maxuserid = uid;
    
    return account;
}


extern t_account * accountlist_get_first(t_list const * const * * save)
{
    void * account;
    
    if (!save)
    {
	eventlog(eventlog_level_error,"accountlist_get_first","got NULL save");
	return NULL;
    }
    
    *save = (t_list const * const *)&accountlist_head; /* avoid warning */
    
    if (!**save)
    {
	*save = NULL;
	return NULL;
    }
    account = list_get_item(**save);
    *save = list_get_next_const(**save);
    
    return account;
}


extern t_account * accountlist_get_next(t_list const * const * * save)
{
    void * account;
    
    if (!save)
    {
	eventlog(eventlog_level_error,"accountlist_get_next","got NULL save");
	return NULL;
    }
    
    if (!*save || !**save)
    {
	*save = NULL;
	return NULL;
    }
    account = list_get_item(**save);
    *save = list_get_next_const(**save);
    
    return account;
}
